\name{ACF}
\title{Autocorrelation Function}
\usage{
ACF(object, maxLag, \dots)
}
\alias{ACF}
\arguments{
 \item{object}{any object from which an autocorrelation function can be
   obtained. Generally an object resulting from a model fit, from which
   residuals can be extracted.}
 \item{maxLag}{maximum lag for which the autocorrelation should be
   calculated.} 
 \item{\dots}{some methods for this generic require additional
   arguments.} 
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include: \code{gls} and  \code{lme}.
}
\value{
  will depend on the method function used; see the appropriate documentation.
}
\references{
  Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994) "Time Series
  Analysis: Forecasting and Control", 3rd Edition, Holden-Day. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{Bates@stat.wisc.edu}}
\seealso{\code{\link{ACF.gls}}, \code{\link{ACF.lme}}}
\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: ACF.gls.Rd,v 1.7.2.1 2003/04/17 22:28:45 bates Exp $
\name{ACF.gls}
\title{Autocorrelation Function for gls Residuals}
\usage{
\method{ACF}{gls}(object, maxLag, resType, form, na.action, \dots)
}
\alias{ACF.gls}
\arguments{
  \item{object}{an object inheriting from class \code{gls}, representing
    a generalized least squares fitted  model.}
  \item{maxLag}{an optional integer giving the maximum lag for which the
    autocorrelation should be calculated. Defaults to maximum lag in the
    residuals.}  
  \item{resType}{an optional character string specifying the type of
    residuals to be used. If \code{"response"}, the "raw" residuals
    (observed - fitted) are used; else, if \code{"pearson"}, the
    standardized residuals (raw residuals divided by the corresponding
    standard errors) are used; else, if \code{"normalized"}, the
    normalized residuals (standardized residuals pre-multiplied by the
    inverse square-root factor of the estimated error correlation
    matrix) are used. Partial matching of arguments is used, so only the
    first character needs to be provided. Defaults to \code{"pearson"}.} 
  \item{form}{an optional one sided formula of the form \code{~ t}, or
    \code{~ t | g}, specifying a time covariate \code{t} and,  optionally, a
    grouping factor \code{g}. The time covariate must be integer
    valued. When a grouping factor is present in 
    \code{form}, the autocorrelations are calculated using residual pairs
    within the same group. Defaults to \code{~ 1}, which corresponds to
    using the order of the observations in the data as a covariate, and
    no groups.}   
  \item{na.action}{a function that indicates what should happen when the
    data contain \code{NA}s.  The default action (\code{na.fail}) causes
    \code{ACF.gls} to print an error message and terminate if there are any
    incomplete observations.}
  \item{\dots}{some methods for this generic require additional
    arguments.} 
}
\description{
  This method function calculates the empirical autocorrelation function
  for the residuals from an \code{gls} fit. If a grouping variable is
  specified in \code{form}, the autocorrelation values
  are calculated using pairs of residuals within the same group;
  otherwise all possible residual pairs are used. The autocorrelation
  function is useful for investigating serial correlation models for
  equally spaced data.  
}
\value{
  a data frame with columns \code{lag} and \code{ACF} representing,
  respectively, the lag between residuals within a pair and the corresponding
  empirical autocorrelation. The returned value inherits from class
  \code{ACF}.  
}
\references{
  Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994) "Time Series
  Analysis: Forecasting and Control", 3rd Edition, Holden-Day. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{ACF.gls}}, \code{\link{plot.ACF}}}
\examples{
data(Ovary)
fm1 <- gls(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), Ovary)
ACF(fm1, form = ~ 1 | Mare)
}
\keyword{models}

\eof
% $Id: ACF.lme.Rd,v 1.7.2.1 2003/04/17 22:28:45 bates Exp $
\name{ACF.lme}
\title{Autocorrelation Function for lme Residuals}
\usage{
\method{ACF}{lme}(object, maxLag, resType, \dots)
}
\alias{ACF.lme}
\arguments{
  \item{object}{an object inheriting from class \code{lme}, representing
    a fitted linear mixed-effects model.}
  \item{maxLag}{an optional integer giving the maximum lag for which the
    autocorrelation should be calculated. Defaults to maximum lag in the
    within-group residuals.}  
  \item{resType}{an optional character string specifying the type of
    residuals to be used. If \code{"response"}, the "raw" residuals
    (observed - fitted) are used; else, if \code{"pearson"}, the
    standardized residuals (raw residuals divided by the corresponding
    standard errors) are used; else, if \code{"normalized"}, the
    normalized residuals (standardized residuals pre-multiplied by the
    inverse square-root factor of the estimated error correlation
    matrix) are used. Partial matching of arguments is used, so only the
    first character needs to be provided. Defaults to \code{"pearson"}.} 
 \item{\dots}{some methods for this generic require additional
   arguments -- not used.} 
}
\description{
  This method function calculates the empirical autocorrelation function
  for the within-group residuals from an \code{lme} fit. The
  autocorrelation values are calculated using pairs of residuals within
  the innermost group level. The autocorrelation function is  useful for
  investigating serial correlation models for equally spaced data. 
}
\value{
  a data frame with columns \code{lag} and \code{ACF} representing,
  respectively, the lag between residuals within a pair and the corresponding
  empirical autocorrelation. The returned value inherits from class
  \code{ACF}.  
}
\references{
  Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994) "Time Series
  Analysis: Forecasting and Control", 3rd Edition, Holden-Day. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{ACF.gls}}, \code{\link{plot.ACF}}}
\examples{
data(Ovary)
fm1 <- lme(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), Ovary,
           random = ~ sin(2*pi*Time) | Mare)
ACF(fm1, maxLag = 11)
}
\keyword{models}

\eof
\name{Alfalfa}
\alias{Alfalfa}
\non_function{}
\title{Split-Plot Experiment on Varieties of Alfalfa}
\description{
  The \code{Alfalfa} data frame has 72 rows and 4 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Variety}{
      a factor with levels
      \code{Cossack},
      \code{Ladak}, and 
      \code{Ranger} 
    }
    \item{Date}{
      a factor with levels
      \code{None} 
      \code{S1} 
      \code{S20} 
      \code{O7} 
    }
    \item{Block}{
      a factor with levels
      \code{1} 
      \code{2} 
      \code{3} 
      \code{4} 
      \code{5} 
      \code{6} 
    }
    \item{Yield}{
      a numeric vector
    }
  }
}
\details{
  These data are described in Snedecor and Cochran (1980) as
  an example of a split-plot design. The treatment structure used in the
  experiment was a 3\eqn{ x }{\times}4 full factorial, with three varieties of
  alfalfa and four dates of third cutting in 1943. The experimental
  units were arranged into six blocks, each subdivided into four plots.
  The varieties of alfalfa (\emph{Cossac}, \emph{Ladak}, and
  \emph{Ranger}) were assigned randomly to the blocks and the dates of
  third cutting (\emph{None}, \emph{S1}---September 1,
  \emph{S20}---September 20, and \emph{O7}---October 7) were randomly
  assigned to the plots.  All four dates were used on each block.
}
\source{
  Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
    and S-PLUS}, Springer, New York.  (Appendix A.1)

  Snedecor, G. W. and Cochran, W. G. (1980), \emph{Statistical Methods (7th
    ed)}, Iowa State University Press, Ames, IA
}
\examples{
data(Alfalfa)
}
\keyword{datasets}

\eof
\name{Assay}
\alias{Assay}
\non_function{}
\title{Bioassay on Cell Culture Plate}
\description{
    The \code{Assay} data frame has 60 rows and 4 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{Block}{
	    an ordered factor with levels
	    \code{2} < \code{1} identifying the block where the wells
	    are measured.
	}
	\item{sample}{
	    a factor with levels \code{a} to \code{f} identifying the
	    sample corresponding to the well.
	}
	\item{dilut}{
	    a factor with levels
	    \code{1} to \code{5}
	    indicating the dilution applied to the well
	}
	\item{logDens}{
	    a numeric vector of the log-optical density
	}
    }
}
\details{
    These data, courtesy of Rich Wolfe and David Lansky from Searle,
    Inc., come from a bioassay run on a 96-well cell culture plate.  The
    assay is performed using a split-block design.  The 8 rows on the
    plate are labeled A--H from top to bottom and the 12 columns on the
    plate are labeled 1--12 from left to right.  Only the central 60 wells
    of the plate are used for the bioassay (the intersection of rows B--G
    and columns 2--11).  There are two blocks in the design: Block 1
    contains columns 2--6 and Block 2 contains columns 7--11. Within each
    block, six samples are assigned randomly to rows and five (serial)
    dilutions are assigned randomly to columns. The response variable is
    the logarithm of the optical density. The cells are treated with a
    compound that they metabolize to produce the stain.  Only live cells
    can make the stain, so the optical density is a measure of the number
    of cells that are alive and healthy.}
\source{
   Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
     and S-PLUS}, Springer, New York. (Appendix A.2)
}
\examples{
data(Assay)
}
\keyword{datasets}

\eof
% $Id: BIC.Rd,v 1.5.2.1 2003/04/17 22:28:45 bates Exp $
\name{BIC}
\title{Bayesian Information Criterion}
\usage{
BIC(object, \dots)
}
\alias{BIC}
\alias{BIC.lm}
\alias{BIC.lmList}
\alias{BIC.gls}
\alias{BIC.lme}
\alias{BIC.nls}
\alias{BIC.nlsList}
\alias{BIC.nlme}
\arguments{
 \item{object}{a fitted model object, for which there exists a
   \code{logLik} method to extract the corresponding log-likelihood, or
   an object inheriting from class \code{logLik}.}
 \item{\dots}{optional fitted model objects.}
}
\description{
  This generic function calculates the Bayesian information criterion,
  also known as Schwarz's Bayesian criterion (SBC), for one or several
  fitted model objects for which a log-likelihood value can be obtained,
  according to the formula \eqn{-2 \mbox{log-likelihood} + n_{par}
    \log(n_{obs})}{-2*log-likelihood + npar*log(nobs)}, where
  \eqn{n_{par}}{npar}  represents the
  number of parameters and \eqn{n_{obs}}{nobs} the number of
  observations in the fitted model. 
}
\value{
 if just one object is provided, returns a numeric value with the
 corresponding BIC; if more than one object are provided, returns a
 \code{data.frame} with rows corresponding to the objects and columns
 representing the number of parameters in the model (\code{df}) and the
 BIC.
}
\references{
  Schwarz, G. (1978) "Estimating the Dimension of a Model", Annals of
  Statistics, 6, 461-464.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{logLik}}, \code{\link{AIC}}, \code{\link{BIC.logLik}}}
\examples{
data(Orthodont)
fm1 <- lm(distance ~ age, data = Orthodont) # no random effects
BIC(fm1)
%\dontrun{
fm2 <- lme(distance ~ age, data = Orthodont) # random is ~age
BIC(fm1, fm2)
%}
}
\keyword{models}

\eof
% $Id: BIC.logLik.Rd,v 1.7.2.1 2003/04/17 22:28:45 bates Exp $
\name{BIC.logLik}
\title{BIC of a logLik Object}
\usage{
\method{BIC}{logLik}(object, \dots)
}
\alias{BIC.logLik}
\arguments{
 \item{object}{an object inheriting from class \code{logLik}, usually
   resulting from applying a \code{logLik} method to a fitted model
   object.}
 \item{\dots}{some methods for this generic use optional arguments.
   None are used in this method.}
}
\description{
  This function calculates the Bayesian information criterion,
  also known as Schwarz's Bayesian criterion (SBC) for an object
  inheriting from class \code{logLik}, according to the formula 
  \eqn{-2 \mbox{log-likelihood} + n_{par} \log(n_{obs})}{log-likelihood +
  npar*log(nobs)}, where \eqn{n_{par}}{npar} represents the number of
  parameters  and \eqn{n_{obs}}{nobs} the number of  observations in the
  fitted model. When comparing fitted objects, the smaller the BIC, the
  better the fit. 
}
\value{
  a numeric value with the corresponding BIC.
}
\references{
  Schwarz, G. (1978) "Estimating the Dimension of a Model", Annals of
  Statistics, 6, 461-464.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{BIC}}, \code{\link{logLik}}, \code{\link{AIC}}.}
\examples{
data(Orthodont)
fm1 <- lm(distance ~ age, data = Orthodont) 
BIC(logLik(fm1))
}
\keyword{models}

\eof
\name{BodyWeight}
\alias{BodyWeight}
\non_function{}
\title{Rat weight over time for different diets}
\description{
    The \code{BodyWeight} data frame has 176 rows and 4 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{weight}{
	    a numeric vector giving the body weight of the rat (grams).
	}
	\item{Time}{
	    a numeric vector giving the time at which the measurement is
	    made (days).
	}
	\item{Rat}{
	    an ordered factor with levels \code{2} < \code{3} < \code{4}
	    < \code{1} < \code{8} < \code{5} < \code{6} < \code{7} <
	    \code{11} < \code{9} < \code{10} < \code{12} < \code{13} <
	    \code{15} < \code{14} < \code{16} identifying the rat whose
	    weight is measured.
	}
	\item{Diet}{
	    a factor with levels
	    \code{1} to \code{3} indicating the diet that the rat receives.
	}
    }
}
\details{
    Hand and Crowder (1996) describe data on the body weights of rats
    measured over 64 days.  These data also appear in Table 2.4 of
    Crowder and Hand (1990).  The body weights of the rats (in grams)
    are measured on day 1 and every seven days thereafter until day 64,
    with an extra measurement on day 44. The experiment started several
    weeks before ``day 1.''  There are three groups of rats, each on a
    different diet.
}
\source{
   Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
       and S-PLUS}, Springer, New York. (Appendix A.3)

   Crowder, M. and Hand, D. (1990), \emph{Analysis of Repeated
       Measures}, Chapman and Hall, London.

   Hand, D. and Crowder, M. (1996), \emph{Practical Longitudinal Data
       Analysis}, Chapman and Hall, London.
}
\examples{
data(BodyWeight)
}
\keyword{datasets}

\eof
\name{Cefamandole}
\alias{Cefamandole}
\non_function{}
\title{Pharmacokinetics of Cefamandole}
\description{
  The \code{Cefamandole} data frame has 84 rows and 3 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Subject}{
      a factor giving the subject from which the sample was drawn.
    }
    \item{Time}{
      a numeric vector giving the time at which the sample was drawn
      (minutes post-injection).
    }
    \item{conc}{
      a numeric vector giving the observed plasma concentration of
      cefamandole (mcg/ml).
    }
  }
}
\details{
  Davidian and Giltinan (1995, 1.1, p. 2) describe data
  obtained during a pilot study to investigate the pharmacokinetics of
  the drug cefamandole. Plasma concentrations of the drug were measured
  on six healthy volunteers at 14 time points following an intraveneous
  dose of 15 mg/kg body weight of cefamandole.
}
\source{
  Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
    and S-PLUS}, Springer, New York.  (Appendix A.4)

  Davidian, M. and Giltinan, D. M. (1995), \emph{Nonlinear Models for
    Repeated Measurement Data},  Chapman and Hall, London.
}
\examples{
data(Cefamandole)
plot(Cefamandole)
fm1 <- nlsList(SSbiexp, data = Cefamandole)
summary(fm1)
}
\keyword{datasets}

\eof
% $Id: Coef.Rd,v 1.2.2.1 2003/04/17 22:28:45 bates Exp $
\name{Coef}
\title{Assign Values to Coefficients}
\usage{
coef(object, \dots) <-  value
}
\alias{coef<-}
\alias{coefficients<-}
\arguments{
 \item{object}{any object representing a fitted model, or, by default,
   any object with a \code{coef} component.}
 \item{\dots}{some methods for this generic function may require
   additional arguments.}
 \item{value}{a value to be assigned to the coefficients associated with
 \code{object}.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include all \code{pdMat}, \code{corStruct}, and 
  \code{varFunc} classes, \code{reStruct}, and \code{modelStruct}.
}
\value{
  will depend on the method function; see the appropriate documentation.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{coef}}}

\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: Covariate.Rd,v 1.2 2002/03/05 14:59:39 bates Exp $
\name{Covariate}
\title{Assign Covariate Values}
\usage{
covariate(object) <- value
}
\alias{covariate<-}
\arguments{
 \item{object}{any object with a \code{covariate} component.}
 \item{value}{a value to be assigned to the covariate associated with
   \code{object}.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include all \code{varFunc} classes.
}
\value{
  will depend on the method function; see the appropriate documentation.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{getCovariate}}}

\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: Covariate.varFunc.Rd,v 1.2.2.1 2003/08/09 22:45:17 bates Exp $
\name{Covariate.varFunc}
\title{Assign varFunc Covariate}
\usage{
\method{covariate}{varFunc}(object) <- value
}
\alias{covariate<-.varFunc}
\arguments{
 \item{object}{an object inheriting from class \code{varFunc},
   representing a variance function structure.} 
 \item{value}{a value to be assigned to the covariate associated with
   \code{object}.}
}
\description{
  The covariate(s) used in the calculation of the weights of the
  variance function represented by \code{object} is (are) replaced by
  \code{value}. If \code{object} has been initialized, \code{value} must
  have the same dimensions as \code{getCovariate(object)}.
}
\value{
  a \code{varFunc} object similar to \code{object}, but with its
  \code{covariate} attribute replaced by \code{value}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{getCovariate.varFunc}}}

\examples{
data(Orthodont)
vf1 <- varPower(1.1, form = ~age)
covariate(vf1) <- Orthodont[["age"]]
}
\keyword{models}

\eof
\name{Dialyzer}
\alias{Dialyzer}
\non_function{}
\title{High-Flux Hemodialyzer}
\description{
    The \code{Dialyzer} data frame has 140 rows and 5 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{Subject}{
	    an ordered factor with levels
	    \code{10} < \code{8} < \code{2} < \code{6} < \code{3} <
	    \code{5} < \code{9} < \code{7} < \code{1} < \code{4} <
	    \code{17} < \code{20} < \code{11} < \code{12} < \code{16} <
	    \code{13} < \code{14} < \code{18} < \code{15} < \code{19}
	    giving the unique identifier for each subject
	}
	\item{QB}{
	    a factor with levels
	    \code{200} and 
	    \code{300} giving the bovine blood flow rate (dL/min).
	}
	\item{pressure}{
	    a numeric vector giving the transmembrane pressure (dmHg).
	}
	\item{rate}{
	    the hemodialyzer ultrafiltration rate (mL/hr).
	}
	\item{index}{
	    index of observation within subject---1 through 7.
	}
    }
}
\details{
    Vonesh and Carter (1992) describe data measured on high-flux
    hemodialyzers to assess their \emph{in vivo} ultrafiltration
    characteristics. The ultrafiltration rates (in mL/hr) of 20 high-flux
    dialyzers were measured at seven different transmembrane pressures (in
    dmHg). The \emph{in vitro} evaluation of the dialyzers used bovine
    blood at flow rates of either 200~dl/min or 300~dl/min. The data,
    are also analyzed in Littell, Milliken, Stroup, and Wolfinger (1996).
}
\source{
    Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
	and S-PLUS}, Springer, New York. (Appendix A.6)

    Vonesh, E. F. and Carter, R. L. (1992), Mixed-effects nonlinear
    regression for unbalanced repeated measures, \emph{Biometrics},
    \bold{48}, 1-18.

    Littell, R. C., Milliken, G. A., Stroup, W. W. and Wolfinger,
    R. D. (1996), \emph{SAS System for Mixed Models}, SAS Institute,
    Cary, NC.
}
\examples{
data(Dialyzer)
}
\keyword{datasets}

\eof
% $Id: Dim.Rd,v 1.4.2.1 2003/04/17 22:28:45 bates Exp $
\name{Dim}
\title{Extract Dimensions from an Object}
\usage{
Dim(object, \dots)
}
\alias{Dim}
\alias{Dim.default}
\arguments{
 \item{object}{any object for which dimensions can be extracted.}
 \item{\dots}{some methods for this generic function require additional
   arguments.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include: \code{corSpatial}, \code{corStruct},
  \code{pdCompSymm}, \code{pdDiag}, \code{pdIdent}, \code{pdMat},
  and \code{pdSymm}.
}
\value{
  will depend on the method function used; see the appropriate documentation.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  If \code{dim} allowed more than one argument, there would be no
  need for this generic function.
}
\seealso{
  \code{\link{Dim.pdMat}}, \code{\link{Dim.corStruct}}
}
\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: Dim.corSpatial.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{Dim.corSpatial}
\title{Dimensions of a corSpatial Object}
\usage{
\method{Dim}{corSpatial}(object, groups, \dots)
}
\alias{Dim.corSpatial}
\arguments{
 \item{object}{an object inheriting from class \code{corSpatial},
   representing a spatial correlation structure.}
 \item{groups}{an optional factor defining the grouping of the
   observations; observations within a group are correlated and
   observations in different groups are uncorrelated.}
 \item{\dots}{further arguments to be passed to or from methods.}
}
\description{
  if \code{groups} is missing, it returns the \code{Dim} attribute of
  \code{object}; otherwise, calculates the dimensions associated with
  the grouping factor.
}
\value{
  a list with components:
  \item{N}{length of \code{groups}}
  \item{M}{number of groups}
  \item{spClass}{an integer representing the spatial correlation class;
    0 = user defined class, 1 = \code{corSpher}, 2 = \code{corExp}, 3 =
    \code{corGaus}, 4 = \code{corLin}}
  \item{sumLenSq}{sum of the squares of the number of observations per
    group} 
  \item{len}{an integer vector with the number of observations per
    group}
  \item{start}{an integer vector with the starting position for the
    distance vectors in each group, beginning from zero}
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{Dim}}, \code{\link{Dim.corStruct}}}

\examples{
data(Orthodont)
Dim(corGaus(), getGroups(Orthodont))
}
\keyword{models}

\eof
% $Id: Dim.corStruct.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{Dim.corStruct}
\title{Dimensions of a corStruct Object}
\usage{
\method{Dim}{corStruct}(object, groups, \dots)
}
\alias{Dim.corStruct}
\arguments{
 \item{object}{an object inheriting from class \code{corStruct},
   representing a correlation structure.}
 \item{groups}{an optional factor defining the grouping of the
   observations; observations within a group are correlated and
   observations in different groups are uncorrelated.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  if \code{groups} is missing, it returns the \code{Dim} attribute of
  \code{object}; otherwise, calculates the dimensions associated with
  the grouping factor.
}
\value{
  a list with components:
  \item{N}{length of \code{groups}}
  \item{M}{number of groups}
  \item{maxLen}{maximum number of observations in a group}
  \item{sumLenSq}{sum of the squares of the number of observations per
    group} 
  \item{len}{an integer vector with the number of observations per
    group}
  \item{start}{an integer vector with the starting position for the
    observations in each group, beginning from zero}
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{Dim}}, \code{\link{Dim.corSpatial}}}

\examples{
data(Orthodont)
Dim(corAR1(), getGroups(Orthodont))
}
\keyword{models}

\eof
% $Id: Dim.pdMat.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{Dim.pdMat}
\title{Dimensions of a pdMat Object}
\usage{
\method{Dim}{pdMat}(object, \dots)
}
\alias{Dim.pdMat}
\alias{Dim.pdCompSymm}
\alias{Dim.pdDiag}
\alias{Dim.pdIdent}
\alias{Dim.pdNatural}
\alias{Dim.pdSymm}
\arguments{
 \item{object}{an object inheriting from class \code{pdMat},
   representing a positive-definite matrix.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function returns the dimensions of the matrix represented
  by \code{object}. 
}
\value{
  an integer vector with the number of rows and columns of the
  matrix represented by \code{object}.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{Dim}}}
\examples{
Dim(pdSymm(diag(3)))
}
\keyword{models}

\eof
\name{Earthquake}
\alias{Earthquake}
\non_function{}
\title{Earthquake Intensity}
\description{
    The \code{Earthquake} data frame has 182 rows and 5 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{Quake}{
	    an ordered factor with levels
	    \code{20} < \code{16} < \code{14} < \code{10} < \code{3} <
	    \code{8} < \code{23} < \code{22} < \code{6} < \code{13} <
	    \code{7} < \code{21} < \code{18} < \code{15} < \code{4} <
	    \code{12} < \code{19} < \code{5} < \code{9} < \code{1} <
	    \code{2} < \code{17} < \code{11} indicating the earthquake
	    on which the measurements were made.
	}
	\item{Richter}{
	    a numeric vector giving the intensity of the earthquake on
	    the Richter scale.
	}
	\item{distance}{
	    the distance from the seismological measuring station to the
	    epicenter of the earthquake (km).
	}
	\item{soil}{
	    a factor with levels \code{0} and \code{1} 
	    giving the soil condition at the measuring station, either
	    soil or rock. 
	}
	\item{accel}{
	    maximum horizontal acceleration observed (g).
	}
    }
}
\details{
    Measurements recorded at available seismometer locations for 23
    large earthquakes in western North America between 1940 and 1980.
    They were originally given in Joyner and Boore (1981); are
    mentioned in Brillinger (1987); and are analyzed in Davidian
    and Giltinan (1995).
}
\source{
    Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
	and S-PLUS}, Springer, New York.  (Appendix A.8)

    Davidian, M. and Giltinan, D. M. (1995), \emph{Nonlinear Models for
	Repeated Measurement Data},  Chapman and Hall, London.

    Joyner and Boor (1981), Peak horizontal acceleration and velocity from
    strong-motion records including records from the 1979 Imperial
    Valley, California, earthquake, \emph{Bulletin of the Seismological
	Society of America}, \bold{71}, 2011-2038.

    Brillinger, D. (1987), Comment on a paper by C. R. Rao,
    \emph{Statistical Science}, \bold{2}, 448-450.
}
\examples{
data(Earthquake)
}
\keyword{datasets}

\eof
% $Id: Extract.pdMat.Rd,v 1.4 2002/03/05 14:59:39 bates Exp $
\name{[.pdMat}
\title{Subscript a pdMat Object}
\usage{
x[i, j, drop]
x[i, j] <- value
}
\alias{[.pdMat}
\alias{[.pdBlocked}
\alias{[<-.pdMat}
\arguments{
 \item{x}{an object inheriting from class \code{pdMat} representing a
   positive-definite matrix.}
 \item{i, j}{optional subscripts applying respectively to the rows
   and columns of the positive-definite matrix represented by
   \code{object}. When \code{i} (\code{j}) is omitted, all rows (columns)
   are extracted.}
 \item{drop}{a logical value. If \code{TRUE}, single rows or columns are
   converted to vectors. If \code{FALSE} the returned value retains its
   matrix representation.}
 \item{value}{a vector, or matrix, with the replacement values for the
   relevant piece of the matrix represented by \code{x}.}
}
\description{
  This method function extracts sub-matrices from the positive-definite
  matrix represented by \code{x}. 
}
\value{
  if \code{i} and \code{j} are identical, the returned value will be
  \code{pdMat} object with the same class as \code{x}. Otherwise, the
  returned value will be a matrix. In the case a single row (or column)
  is selected, the returned value may be converted to a vector,
  according to the rules above.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{[}}, \code{\link{pdMat}}}

\examples{
pd1 <- pdSymm(diag(3))
pd1[1, , drop = FALSE]
%\dontrun{
pd1[1:2, 1:2] <- 3 * diag(2)
%}
}
\keyword{models}

\eof
\name{Fatigue}
\alias{Fatigue}
\non_function{}
\title{Cracks caused by metal fatigue}
\description{
  The \code{Fatigue} data frame has 262 rows and 3 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Path}{
      an ordered factor with levels
      \code{1} < \code{2} < \code{3} < \code{4} < \code{5} <
      \code{6} < \code{7} < \code{8} < \code{9} < \code{10} <
      \code{11} < \code{12} < \code{13} < \code{14} < \code{15} <
      \code{16} < \code{17} < \code{18} < \code{19} < \code{20} <
      \code{21} giving the test path (or test unit) number.  The
      order is in terms of increasing failure time or decreasing
      terminal crack length.
    }
    \item{cycles}{
      number of test cycles at which the measurement is made
      (millions of cycles).
    }
    \item{relLength}{
      relative crack length (dimensionless).
    }
  }
}
\details{
  These data are given in Lu and Meeker (1993) where they state
  ``We obtained the data in Table 1 visually from figure 4.5.2 on page 242
  of Bogdanoff and Kozin (1985).''
  The data represent the growth of cracks in metal for 21 test units.
  An initial notch of length 0.90 inches was made on each unit which then
  was subjected to several thousand test cycles.  After every 10,000
  test cycles the crack length was measured.  Testing was stopped if the
  crack length exceeded 1.60 inches, defined as a failure, or at 120,000
  cycles.
}
\source{
  Lu, C. Joseph , and Meeker, William Q. (1993), Using degradation
  measures to estimate a time-to-failure distribution,
  \emph{Technometrics}, \bold{35}, 161-174 
}
\examples{
data(Fatigue)
}
\keyword{datasets}

\eof
\name{Gasoline}
\alias{Gasoline}
\non_function{}
\title{Refinery yield of gasoline}
\description{
  The \code{Gasoline} data frame has 32 rows and 6 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{yield}{
      a numeric vector giving the percentage of crude oil converted to
      gasoline after distillation and fractionation
    }
    \item{endpoint}{
        a numeric vector giving the temperature (degrees F) at which all
	the gasoline is vaporized
    }
    \item{Sample}{
      an ordered factor giving the inferred crude oil sample number
    }
    \item{API}{
      a numeric vector giving the crude oil gravity (degrees API)
    }
    \item{vapor}{
	a numeric vector giving the vapor pressure of the crude oil
	\eqn{(\mathrm{lbf}/\mathrm{in}^2)}{(lbf/in^2)}
    }
    \item{ASTM}{
      a numeric vector giving the crude oil 10\% point ASTM---the
      temperature at which 10\% of the crude oil has become vapor. 
    }
  }
}
\details{
  Prater (1955) provides data on crude oil properties and
  gasoline yields.  Atkinson (1985)
  uses these data to illustrate the use of diagnostics in multiple
  regression analysis.  Three of the covariates---\textsf{API},
  \textsf{vapor}, and \textsf{ASTM}---measure characteristics of the
  crude oil used to produce the gasoline.  The other covariate ---
  \textsf{endpoint}---is a characteristic of the refining process.
  Daniel and Wood (1980) notice that the covariates characterizing
  the crude oil occur in only ten distinct groups and conclude that the
  data represent responses measured on ten different crude oil samples.
}
\source{
  Prater, N. H. (1955), Estimate gasoline yields from crudes,
  \emph{Petroleum Refiner}, \bold{35} (5).

  Atkinson, A. C. (1985), \emph{Plots, Transformations, and
    Regression}, Oxford Press, New York.

  Daniel, C. and Wood, F. S. (1980), \emph{Fitting Equations to Data},
  Wiley, New York

  Venables, W. N. and Ripley, B. D. (1999) \emph{Modern Applied
    Statistics with S-PLUS (3rd ed)}, Springer, New York.
}
\examples{
data(Gasoline)
}
\keyword{datasets}

\eof
\name{Glucose}
\alias{Glucose}
\non_function{}
\title{Glucose levels over time}
\description{
    The \code{Glucose} data frame has 378 rows and 4 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{Subject}{
	    an ordered factor with levels
	    \code{6} < \code{2} < \code{3} < \code{5} < \code{1} < \code{4}
	}
	\item{Time}{
	    a numeric vector
	}
	\item{conc}{
	    a numeric vector of glucose levels
	}
	\item{Meal}{
	    an ordered factor with levels
	    \code{2am} < \code{6am} < \code{10am} < \code{2pm} < \code{6pm} < \code{10pm}
	}
    }
}
\source{
    Hand, D. and Crowder, M. (1996), \emph{Practical Longitudinal Data
	Analysis}, Chapman and Hall, London.
}
\examples{
data(Glucose)
}
\keyword{datasets}

\eof
\name{Glucose2}
\alias{Glucose2}
\non_function{}
\title{Glucose Levels Following Alcohol Ingestion}
\description{
  The \code{Glucose2} data frame has 196 rows and 4 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Subject}{
      a factor with levels
      \code{1} to \code{7} identifying the subject whose glucose
      level is measured. 
    }
    \item{Date}{
      a factor with levels
      \code{1} 
      \code{2}
      indicating the occasion in which the experiment was conducted.
    }
    \item{Time}{
      a numeric vector giving the time since alcohol ingestion (in min/10).
    }
    \item{glucose}{
      a numeric vector giving the blood glucose level (in mg/dl).
    }
  }
}
\details{
  Hand and Crowder (Table A.14, pp. 180-181, 1996) describe data on
  the blood glucose levels measured at 14 time points over 5 hours for 7
  volunteers who took alcohol at time 0. The same experiment was
  repeated on a second date with the same subjects but with a dietary
  additive used for all subjects.
}
\source{
  Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
    and S-PLUS}, Springer, New York.  (Appendix A.10)

  Hand, D. and Crowder, M. (1996), \emph{Practical Longitudinal Data
    Analysis}, Chapman and Hall, London.
}
\examples{
data(Glucose2)
}
\keyword{datasets}

\eof
\name{Gun}
\alias{Gun}
\non_function{}
\title{Methods for firing naval guns}
\description{
  The \code{Gun} data frame has 36 rows and 4 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{rounds}{
      a numeric vector
    }
    \item{Method}{
      a factor with levels
      \code{M1} 
      \code{M2} 
    }
    \item{Team}{
      an ordered factor with levels
      \code{T1S} < \code{T3S} < \code{T2S} < \code{T1A} <
      \code{T2A} < \code{T3A} < \code{T1H} < \code{T3H} <
      \code{T2H}
    }
    \item{Physique}{
      an ordered factor with levels
      \code{Slight} < \code{Average} < \code{Heavy}
    }
  }
}
\details{
  Hicks (p.180, 1993) reports data from an experiment on
  methods for firing naval guns. Gunners of three different physiques
  (slight, average, and heavy) tested two firing methods. Both methods
  were tested twice by each of nine teams of three gunners with
  identical physique. The response was the number of rounds fired per
  minute.}
\source{
  Hicks, C. R. (1993), \emph{Fundamental Concepts in the Design of
    Experiments (4th ed)}, Harcourt Brace, New York.
}
\examples{
data(Gun)
}
\keyword{datasets}

\eof
\name{IGF}
\alias{IGF}
\non_function{}
\title{Radioimmunoassay of IGF-I Protein}
\description{
    The \code{IGF} data frame has 237 rows and 3 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{Lot}{
	    an ordered factor giving the radioactive tracer lot.
	}
	\item{age}{
	    a numeric vector giving the age (in days) of the radioactive
	    tracer.
	}
	\item{conc}{
	    a numeric vector giving the estimated concentration of IGF-I
	    protein (ng/ml)
	}
    }
}
\details{
    Davidian and Giltinan (1995) describe data obtained during quality
    control radioimmunoassays for ten different lots of radioactive
    tracer used to calibrate the Insulin-like Growth Factor (IGF-I)
    protein concentration measurements. 
}
\source{
    Davidian, M. and Giltinan, D. M. (1995), \emph{Nonlinear Models for
	Repeated Measurement Data},  Chapman and Hall, London.
    
    Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
	and S-PLUS}, Springer, New York.  (Appendix A.11)
}
\examples{
data(IGF)
}
\keyword{datasets}

\eof
% $Id: Initialize.Rd,v 1.1.2.2 2003/04/17 22:28:45 bates Exp $
\name{Initialize}
\title{Initialize Object}
\usage{
Initialize(object, data, \dots)
}
\alias{Initialize}
\arguments{
 \item{object}{any object requiring initialization, e.g. "plug-in"
   structures such as \code{corStruct} and \code{varFunc} objects. }
 \item{data}{a data frame to be used in the initialization procedure.}
 \item{\dots}{some methods for this generic function require additional
   arguments.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include: \code{corStruct}, \code{lmeStruct},
  \code{reStruct}, and \code{varFunc}.
}
\value{
  an initialized object with the same class as \code{object}. Changes
  introduced by the initialization procedure will depend on the method
  function used; see the appropriate documentation. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: Initialize.corStruct.Rd,v 1.1.2.1 2002/08/09 19:45:29 bates Exp $
\name{Initialize.corStruct}
\title{Initialize corStruct Object}
\usage{
\method{Initialize}{corStruct}(object, data, \dots)
}
\alias{Initialize.corStruct}
\alias{Initialize.corAR1}
\alias{Initialize.corARMA}
\alias{Initialize.corCAR1}
\alias{Initialize.corCompSymm}
\alias{Initialize.corHF}
\alias{Initialize.corIdent}
\alias{Initialize.corLin}
\alias{Initialize.corNatural}
\alias{Initialize.corSpatial}
\alias{Initialize.corSpher}
\alias{Initialize.corSymm}
\arguments{
 \item{object}{an object inheriting from class \code{corStruct}
   representing a correlation structure.}
 \item{data}{a data frame in which to evaluate the variables defined in
   \code{formula(object)}.}
 \item{\dots}{this argument is included to make this method compatible
   with the generic.}
}
\description{
  This method initializes \code{object} by evaluating its associated
  covariate(s) and grouping factor, if any is present, in \code{data},
  calculating various dimensions and constants used by optimization
  algorithms involving \code{corStruct} objects (see the appropriate
  \code{Dim} method documentation), and assigning initial values for
  the coefficients in \code{object}, if none were present.
}
\value{
  an initialized object with the same class as \code{object}
  representing a correlation structure.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{Dim.corStruct}}}

\examples{
data(Orthodont)
cs1 <- corAR1(form = ~ 1 | Subject)
cs1 <- Initialize(cs1, data = Orthodont)
}
\keyword{models}

\eof
% $Id: Initialize.glsStruct.Rd,v 1.1.2.1 2002/08/09 19:45:29 bates Exp $
\name{Initialize.glsStruct}
\title{Initialize a glsStruct Object}
\usage{
\method{Initialize}{glsStruct}(object, data, control, \dots)
}
\alias{Initialize.glsStruct}
\arguments{
 \item{object}{an object inheriting from class \code{glsStruct},
   representing a list of linear model components, such as
   \code{corStruct} and \code{varFunc} objects.}
 \item{data}{a data frame in which to evaluate the variables defined in
   \code{formula(object)}.}
 \item{control}{an optional list with control parameters for the
   initialization and optimization algorithms used in
   \code{gls}. Defaults to \code{list(singular.ok = FALSE, qrTol =
     .Machine$single.eps)}, implying that linear dependencies are not
   allowed in the model and that the tolerance for detecting linear
   dependencies among the columns of the regression matrix is
   \code{.Machine$single.eps}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}

\description{
  The individual linear model components of the \code{glsStruct} list
  are initialized.  
}
\value{
  a \code{glsStruct} object similar to \code{object}, but with
  initialized model components. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gls}}, \code{\link{Initialize.corStruct}} ,
  \code{\link{Initialize.varFunc}}} 

\keyword{models}

\eof
% $Id: Initialize.lmeStruct.Rd,v 1.1.2.1 2002/08/09 19:45:29 bates Exp $
\name{Initialize.lmeStruct}
\title{Initialize an lmeStruct Object}
\usage{
\method{Initialize}{lmeStruct}(object, data, groups, conLin, control, \dots)
}
\alias{Initialize.lmeStruct}
\arguments{
 \item{object}{an object inheriting from class \code{lmeStruct},
   representing a list of linear mixed-effects model components, such as
   \code{reStruct}, \code{corStruct}, and \code{varFunc} objects.}
 \item{data}{a data frame in which to evaluate the variables defined in
   \code{formula(object)}.}
 \item{groups}{a data frame with the grouping factors corresponding to
   the lme model associated with \code{object} as columns, sorted from
   innermost to outermost grouping level.}
 \item{conLin}{an optional condensed linear model object, consisting of
   a list with components \code{"Xy"}, corresponding to a regression
   matrix (\code{X}) combined with a response vector (\code{y}), and 
   \code{"logLik"}, corresponding to the log-likelihood of the
   underlying lme model. Defaults to \code{attr(object, "conLin")}.}
 \item{control}{an optional list with control parameters for the
   initialization and optimization algorithms used in
   \code{lme}. Defaults to \code{list(niterEM=20, gradHess=TRUE)},
   implying that 20 EM iterations are to be used in the derivation of
   initial estimates for the coefficients of the \code{reStruct}
   component of \code{object} and, if possible, numerical gradient
   vectors and Hessian matrices for the log-likelihood function are to
   be used in the optimization algorithm.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The individual linear mixed-effects model components of the
  \code{lmeStruct} list are initialized. 
}
\value{
  an \code{lmeStruct} object similar to \code{object}, but with
  initialized model components. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lme}}, \code{\link{Initialize.reStruct}},
  \code{\link{Initialize.corStruct}} , \code{\link{Initialize.varFunc}}}

\keyword{models}

\eof
% $Id: Initialize.reStruct.Rd,v 1.1.2.1 2002/08/09 19:45:29 bates Exp $
\name{Initialize.reStruct}
\title{Initialize reStruct Object}
\usage{
\method{Initialize}{reStruct}(object, data, conLin, control, \dots)
}
\alias{Initialize.reStruct}
\arguments{
 \item{object}{an object inheriting from class \code{reStruct},
   representing a random effects structure and consisting of a list of
   \code{pdMat} objects.} 
 \item{data}{a data frame in which to evaluate the variables defined in
   \code{formula(object)}.}
 \item{conLin}{a condensed linear model object, consisting of a list
   with components \code{"Xy"}, corresponding to a regression matrix
   (\code{X}) combined with a response vector (\code{y}), and
   \code{"logLik"}, corresponding to the log-likelihood of the
   underlying model.} 
 \item{control}{an optional list with a single component \code{niterEM}
   controlling the number of iterations for the EM algorithm used to
   refine initial parameter estimates. It is given as a list for
   compatibility with other \code{Initialize} methods. Defaults to
   \code{list(niterEM = 20)}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  Initial estimates for the parameters in the \code{pdMat} objects
  forming \code{object}, which have not yet been initialized, are
  obtained using the methodology described in Bates and Pinheiro
  (1998). These estimates may be refined using a series of EM
  iterations, as described in Bates and Pinheiro (1998). The number of
  EM iterations to be used is defined in \code{control}. 
}
\value{
  an \code{reStruct} object similar to \code{object}, but with all
  \code{pdMat} components initialized.
}
\references{
  Bates, D.M. and Pinheiro, J.C. (1998) "Computational methods for
  multilevel models" available in PostScript or PDF formats at
  http://franz.stat.wisc.edu/pub/NLME/
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{reStruct}}, \code{\link{pdMat}}}

\keyword{models}

\eof
% $Id: Initialize.varFunc.Rd,v 1.1.2.2 2003/04/17 22:28:45 bates Exp $
\name{Initialize.varFunc}
\title{Initialize varFunc Object}
\usage{
\method{Initialize}{varFunc}(object, data, \dots)
}
\alias{Initialize.varFunc}
\alias{Initialize.varComb}
\alias{Initialize.varConstPower}
\alias{Initialize.varExp}
\alias{Initialize.varFixed}
\alias{Initialize.varIdent}
\alias{Initialize.varPower}
\arguments{
 \item{object}{an object inheriting from class \code{varFunc},
   representing a variance function structure.} 
 \item{data}{a data frame in which to evaluate the variables named in
   \code{formula(object)}. }
 \item{\dots}{this argument is included to make this method compatible
   with the generic.}
}
\description{
  This method initializes \code{object} by evaluating its associated
  covariate(s) and grouping factor, if any is present, in \code{data};
  determining if the covariate(s) need to be updated when the
  values of the coefficients associated with \code{object} change;
  initializing the log-likelihood and the weights associated with
  \code{object}; and assigning initial values for the coefficients in
  \code{object}, if none were present. The covariate(s) will only be
  initialized if no update is needed when \code{coef(object)} changes. 
}
\value{
  an initialized object with the same class as \code{object}
  representing a variance function structure.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\examples{
data(Orthodont)
vf1 <- varPower( form = ~ age | Sex )
vf1 <- Initialize( vf1, Orthodont )
}
\keyword{models}

\eof
\name{LDEsysMat}
\alias{LDEsysMat}
\title{Generate system matrix for LDEs}
\description{
  Generate the system matrix for the linear differential equations
  determined by a compartment model.
}
\usage{
LDEsysMat(pars, incidence)
}
\arguments{
  \item{pars}{a numeric vector of parameter values.}
  \item{incidence}{an integer matrix with columns named \code{From},
    \code{To}, and \code{Par}.  Values in the \code{Par} column must
    be in the range 1 to \code{length(pars)}.  
    Values in the \code{From}
    column must be between 1 and the number of compartments.
    Values in the \code{To}
    column must be between 0 and the number of compartments.}
}
\details{
  A compartment model describes material transfer between \code{k}
  in a system of \code{k} compartments to a linear system of differential
  equations. Given a description of the system and a vector of parameter
  values this function returns the system matrix.

  This function is intended for use in a general system for solving
  compartment models, as described in Bates and Watts (1988).
}
\value{
  A \code{k} by \code{k} numeric matrix.
}
\references{
  Bates, D. M. and Watts, D. G. (1988), \emph{Nonlinear Regression
    Analysis and Its Applications}, Wiley, New York.
}
\author{Douglas Bates \email{bates@stat.wisc.edu}}
\examples{
# incidence matrix for a two compartment open system
incidence <-
  matrix(c(1,1,2,2,2,1,3,2,0), ncol = 3, byrow = TRUE,
   dimnames = list(NULL, c("Par", "From", "To")))
incidence
LDEsysMat(c(1.2, 0.3, 0.4), incidence)
}
\keyword{models}

\eof
\name{Machines}
\alias{Machines}
\non_function{}
\title{Productivity Scores for Machines and Workers}
\description{
  The \code{Machines} data frame has 54 rows and 3 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Worker}{
      an ordered factor giving the unique identifier for the worker.
    }
    \item{Machine}{
      a factor with levels
      \code{A}, 
      \code{B}, and 
      \code{C} identifying the machine brand.
    }
    \item{score}{
      a productivity score.
    }
  }
}
\details{
  Data on an experiment to compare three brands of machines used in an
  industrial process are presented in Milliken and Johnson (p. 285,
  1992). Six workers were chosen randomly among the
  employees of a factory to operate each machine three times. The
  response is an overall productivity score taking into account the
  number and quality of components produced.}
\source{
  Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models
    in S and S-PLUS}, Springer, New York.  (Appendix A.14)

  Milliken, G. A. and Johnson, D. E. (1992), \emph{Analysis of Messy Data,
    Volume I: Designed Experiments}, Chapman and Hall, London.
}
\examples{
data(Machines)
}
\keyword{datasets}

\eof
\name{MathAchSchool}
\alias{MathAchSchool}
\non_function{}
\title{School demographic data for MathAchieve}
\description{
    The \code{MathAchSchool} data frame has 160 rows and 7 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{School}{
	    a factor giving the school on which the measurement is made.
	}
	\item{Size}{
	    a numeric vector giving the number of students in the school
	}
	\item{Sector}{
	    a factor with levels
	    \code{Public} 
	    \code{Catholic} 
	}
	\item{PRACAD}{
	    a numeric vector giving the percentage of students on the
	    academic track
	}
	\item{DISCLIM}{
	    a numeric vector measuring the discrimination climate
	}
	\item{HIMINTY}{
	    a factor with levels
	    \code{0} 
	    \code{1} 
	}
	\item{MEANSES}{
	    a numeric vector giving the mean SES score.
	}
    }
}
\details{
  These variables give the school-level demographic data to accompany
  the \code{MathAchieve} data.
}
%\source{}
\examples{
data(MathAchSchool)
}
\keyword{datasets}

\eof
\name{MathAchieve}
\alias{MathAchieve}
\non_function{}
\title{Mathematics achievement scores}
\description{
    The \code{MathAchieve} data frame has 7185 rows and 6 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{School}{
	    an ordered factor identifying the school that the student attends
	}
	\item{Minority}{
	    a factor with levels
	    \code{No} 
	    \code{Yes}
	    indicating if the student is a member of a minority racial group.
	}
	\item{Sex}{
	    a factor with levels
	    \code{Male} 
	    \code{Female} 
	}
	\item{SES}{
	    a numeric vector of socio-economic status.
	}
	\item{MathAch}{
	    a numeric vector of mathematics achievement scores.
	}
	\item{MEANSES}{
	    a numeric vector of the mean SES for the school.
	}
    }
}
\details{
  Each row in this data frame contains the data for one student.
}
%\source{}
\examples{
data(MathAchieve)
summary(MathAchieve)
}
\keyword{datasets}

\eof
% $Id: Matrix.Rd,v 1.2 2002/03/05 14:59:39 bates Exp $
\name{Matrix}
\title{Assign Matrix Values}
\usage{
matrix(object) <- value
}
\alias{matrix<-}
\arguments{
 \item{object}{any object to which \code{as.matrix} can be applied.}
 \item{value}{a matrix, or list of matrices, with the same dimensions as
   \code{as.matrix(object)} with the new values to be assigned to the
   matrix associated with \code{object}.} 
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include \code{pdMat}, \code{pdBlocked}, and 
  \code{reStruct}.
}
\value{
  will depend on the method function; see the appropriate documentation.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{as.matrix}}}

\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: Matrix.pdMat.Rd,v 1.2.2.1 2003/08/09 22:45:17 bates Exp $
\name{Matrix.pdMat}
\title{Assign Matrix to a pdMat Object}
\usage{
\method{matrix}{pdMat}(object) <- value
}
\alias{matrix<-.pdMat}
\alias{matrix<-.pdBlocked}
\arguments{
 \item{object}{an object inheriting from class \code{pdMat}, representing
   a positive definite matrix.} 
 \item{value}{a matrix with the new values to be assigned to the
   positive-definite matrix represented by \code{object}. Must have the
   same dimensions as \code{as.matrix(object)}.}
}
\description{
  The positive-definite matrix represented by \code{object} is replaced
  by \code{value}. If the original matrix had row and/or column names,
  the corresponding names for \code{value} can either be \code{NULL}, or
  a permutation of the original names. 
}
\value{
  a \code{pdMat} object similar to \code{object}, but with its
  coefficients modified to  produce the matrix in \code{value}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{pdMat}}}

\examples{
pd1 <- pdSymm(diag(3))
matrix(pd1) <- diag(1:3)
pd1
}
\keyword{models}

\eof
% $Id: Matrix.reStruct.Rd,v 1.2.2.1 2003/08/09 22:45:17 bates Exp $
\name{Matrix.reStruct}
\title{Assign reStruct Matrices}
\usage{
\method{matrix}{reStruct}(object) <-  value
}
\alias{matrix<-.reStruct}
\arguments{
 \item{object}{an object inheriting from class \code{reStruct},
   representing a random effects structure and consisting of a list of
   \code{pdMat} objects.} 
 \item{value}{a matrix, or list of matrices, with the new values to be
   assigned to the matrices associated with the \code{pdMat} components
   of \code{object}.}  
}
\description{
  The individual matrices in \code{value} are assigned to each
  \code{pdMat} component of \code{object}, in the order they are listed. The
  new matrices must have the same dimensions as the matrices they are
  meant to replace.
}
\value{
  an \code{reStruct} object similar to \code{object}, but with the
  coefficients of the individual \code{pdMat} components modified to
  produce the matrices listed in \code{value}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{reStruct}}, \code{\link{pdMat}}}

\examples{
data(Pixel)
rs1 <- reStruct(list(Dog = ~day, Side = ~1), data = Pixel)
matrix(rs1) <- list(diag(2), 3)
}
\keyword{models}

\eof
\name{Meat}
\alias{Meat}
\non_function{}
\title{Tenderness of meat}
\description{
  The \code{Meat} data frame has 30 rows and 4 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Storage}{
      an ordered factor specifying the storage treatment - 1 (0
      days), 2 (1 day),  3 (2 days), 4 (4 days), 5 (9 days), and 6
      (18 days)
    }
    \item{score}{
      a numeric vector giving the tenderness score of beef roast.
    }
    \item{Block}{
      an ordered factor identifying the muscle from which the
      roast was extracted with levels 
      \code{II} < \code{V} < \code{I} < \code{III} < \code{IV}
    }
    \item{Pair}{
      an ordered factor giving the unique identifier for each pair
      of beef roasts with levels \code{II-1} < \dots{} < \code{IV-1}
    }
  }
}
\details{
  Cochran and Cox (section 11.51, 1957) describe data from an
  experiment conducted at Iowa State College (Paul, 1943) to
  compare the effects of length of cold storage on the tenderness of
  beef roasts.  Six storage periods ranging from 0 to 18 days were used.
  Thirty roasts were scored by four judges on a scale from 0 to 10, with
  the score increasing with tenderness. The response was the sum of all
  four scores. Left and right roasts from the same animal were grouped
  into pairs, which were further grouped into five blocks, according to
  the muscle from which they were extracted. Different storage periods
  were applied to each roast within a pair according to a balanced
  incomplete block design.}
\source{
  Cochran, W. G. and Cox, G. M. (1957), \emph{Experimental Designs},
  Wiley, New York.
}
\examples{
data(Meat)
}
\keyword{datasets}

\eof
\name{Milk}
\alias{Milk}
\non_function{}
\title{Protein content of cows' milk}
\description{
    The \code{Milk} data frame has 1337 rows and 4 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{protein}{
	    a numeric vector giving the protein content of the milk.
	}
	\item{Time}{
	    a numeric vector giving the time since calving (weeks).
	}
	\item{Cow}{
	    an ordered factor giving a unique identifier for each cow.
	}
	\item{Diet}{
	    a factor with levels
	    \code{barley}, 
	    \code{barley+lupins}, and 
	    \code{lupins} identifying the diet for each cow.
	}
    }
}
\details{
    Diggle, Liang, and Zeger (1994) describe data on the protein content
    of cows' milk in the weeks following calving.  The cattle are grouped
    according to whether they are fed a diet with barley alone, with
    barley and lupins, or with lupins alone. 
}
\source{
     Diggle, Peter J., Liang, Kung-Yee and Zeger, Scott L. (1994),
     \emph{Analysis of longitudinal data}, Oxford University Press,
     Oxford.
}
\examples{
data(Milk)
}
\keyword{datasets}

\eof
\name{Muscle}
\alias{Muscle}
\non_function{}
\title{Contraction of heart muscle sections}
\description{
    The \code{Muscle} data frame has 60 rows and 3 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{Strip}{
	    an ordered factor indicating the strip of muscle being measured.
	}
	\item{conc}{
	    a numeric vector giving the concentration of CaCl2
	}
	\item{length}{
	    a numeric vector giving the shortening ofthe heart muscle strip.
	}
    }
}
\details{
    Baumann and Waldvogel (1963) describe data on the shortening of heart
    muscle strips dipped in a CaCl$_2$ solution.  The muscle strips are
    taken from the left auricle of a rat's heart.}
\source{
    Baumann, F. and Waldvogel, F. (1963), La restitution pastsystolique
    de la contraction de l'oreillette gauche du rat. Effets de divers
    ions et de l'acetylcholine, \emph{Helvetica Physiologica Acta},
    \bold{21}.
}
\examples{
data(Muscle)
}
\keyword{datasets}

\eof
% $Id: Names.Rd,v 1.5.2.1 2003/05/16 21:49:21 bates Exp $
\name{Names}
\title{Names Associated with an Object}
\usage{
Names(object, \dots)
Names(object, \dots) <- value
}
\alias{Names}
\alias{Names<-}
\arguments{
 \item{object}{any object for which names can be extracted and/or assigned.}
 \item{\dots}{some methods for this generic function require additional
   arguments.}
 \item{value}{names to be assigned to \code{object}.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include: \code{formula}, \code{modelStruct},
  \code{pdBlocked}, \code{pdMat}, and \code{reStruct}.
}
\value{
  will depend on the method function used; see the appropriate documentation.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\section{SIDE EFFECTS}{
  On the left side of an assignment, sets the names associated with
  \code{object} to \code{value}, which must have an appropriate length. 
}
\note{
  If \code{names} were generic, there would be no need for this generic
  function. 
}
\seealso{\code{\link{Names.formula}}, \code{\link{Names.pdMat}}}
\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: Names.formula.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{Names.formula}
\title{Extract Names from a formula}
\usage{
\method{Names}{formula}(object, data, exclude, \dots)
}
\alias{Names.formula}
\alias{Names.listForm}
\arguments{
 \item{object}{an object inheriting from class \code{formula}.}
 \item{data}{an optional data frame containing the variables specified
   in \code{object}. By default the variables are taken from the
   environment from which \code{Names.formula} is called.}
 \item{exclude}{an optional character vector with names to be excluded
   from the returned value. Default is \code{c("pi",".")}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function returns the names of the terms corresponding to
  the right hand side of \code{object} (treated as a linear formula),
  obtained as the column names of the corresponding
  \code{model.matrix}. 
}
\value{
  a character vector with the column names of the \code{model.matrix}
  corresponding to the right hand side of \code{object} which are not
  listed in \code{excluded}.  
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{model.matrix}}, \code{\link{terms}},
  \code{\link{Names}}
}

\examples{
data(Orthodont)
Names(distance ~ Sex * age, data = Orthodont)
}
\keyword{models}

\eof
% $Id: Names.pdBlocked.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{Names.pdBlocked}
\title{Names of a pdBlocked Object}
\usage{
\method{Names}{pdBlocked}(object, asList, \dots)
}
\alias{Names.pdBlocked}
\alias{Names<-.pdBlocked}
\arguments{
 \item{object}{an object inheriting from class \code{pdBlocked}
   representing a positive-definite matrix with block diagonal
   structure} 
 \item{asList}{a logical value. If \code{TRUE} a \code{list} with the
   names for each block diagonal element is returned. If \code{FALSE}
   a character vector with all column names is returned. Defaults to
   \code{FALSE}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts the first element of the \code{Dimnames}
  attribute, which contains the column names, for each block diagonal
  element in the matrix represented by \code{object}.
}
\value{
  if \code{asList} is \code{FALSE}, a character vector with column names
  of the matrix represented by \code{object}; otherwise, if
  \code{asList} is \code{TRUE}, a list with components given by the
  column names of the individual block diagonal elements in the matrix
  represented by \code{object}.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{Names}}, \code{\link{Names.pdMat}}}

\examples{
data(Orthodont)
pd1 <- pdBlocked(list(~Sex - 1, ~age - 1), data = Orthodont)
Names(pd1)
}
\keyword{models}

\eof
% $Id: Names.pdMat.Rd,v 1.9.2.1 2003/08/09 22:45:17 bates Exp $
\name{Names.pdMat}
\title{Names of a pdMat Object}
\usage{
\method{Names}{pdMat}(object, \dots)
\method{Names}{pdMat}(object, \dots) <- value
}
\alias{Names.pdMat}
\alias{Names<-.pdMat}
\arguments{
 \item{object}{an object inheriting from class \code{pdMat},
   representing a positive-definite matrix.}
 \item{value}{a character vector with the replacement values for the
   column and row names of the matrix represented by \code{object}. It
   must have length equal to the dimension of the matrix
   represented by \code{object} and, if names have been previously
   assigned to \code{object}, it must correspond to a permutation of the
   original names.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function returns the fist element of the \code{Dimnames}
  attribute of \code{object}, which contains the column names of the
  matrix represented by \code{object}. 
}
\value{
  if \code{object} has a \code{Dimnames} attribute then the first
  element of this attribute is returned; otherwise \code{NULL}. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\section{SIDE EFFECTS}{
  On the left side of an assignment, sets the \code{Dimnames} attribute
  of \code{object} to \code{list(value, value)}.
}
\seealso{\code{\link{Names}}, \code{\link{Names.pdBlocked}}}

\examples{
data(Orthodont)
pd1 <- pdSymm(~age, data = Orthodont)
Names(pd1)
}
\keyword{models}

\eof
% $Id: Names.reStruct.Rd,v 1.8.2.1 2003/08/09 22:45:17 bates Exp $
\name{Names.reStruct}
\title{Names of an reStruct Object}
\usage{
\method{Names}{reStruct}(object, \dots)
\method{Names}{reStruct}(object, \dots) <- value
}
\alias{Names.reStruct}
\alias{Names<-.reStruct}
\arguments{
 \item{object}{an object inheriting from class \code{reStruct},
   representing a random effects structure and consisting of a list of
   \code{pdMat} objects.} 
 \item{value}{a list of character vectors with the replacement values
   for the names of the individual \code{pdMat} objects that form
   \code{object}. It must have the same length as \code{object}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts the column names of each of the
  positive-definite matrices represented the \code{pdMat}
  elements of \code{object}.
}
\value{
  a list containing the column names of each of the positive-definite
  matrices represented by the \code{pdMat} elements of \code{object}. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\section{SIDE EFFECTS}{
  On the left side of an assignment, sets the \code{Names} of the
  \code{pdMat} elements of \code{object} to the corresponding element of
  \code{value}.  
}
\seealso{\code{\link{reStruct}}, \code{\link{pdMat}},
  \code{\link{Names.pdMat}}
} 
\examples{
data(Pixel)
rs1 <- reStruct(list(Dog = ~day, Side = ~1), data = Pixel)
Names(rs1)
}
\keyword{models}

\eof
\name{Nitrendipene}
\alias{Nitrendipene}
\non_function{}
\title{Assay of nitrendipene}
\description{
    The \code{Nitrendipene} data frame has 89 rows and 4 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{activity}{
	    a numeric vector
	}
	\item{NIF}{
	    a numeric vector
	}
	\item{Tissue}{
	    an ordered factor with levels
	    \code{2} < \code{1} < \code{3} < \code{4}
	}
	\item{log.NIF}{
	    a numeric vector
	}
    }
}
\details{

}
\source{
  Bates, D. M. and Watts, D. G. (1988), \emph{Nonlinear Regression
    Analysis and Its Applications}, Wiley, New York.
}
\examples{
data(Nitrendipene)
}
\keyword{datasets}

\eof
\name{Oats}
\alias{Oats}
\non_function{}
\title{Split-plot Experiment on Varieties of Oats}
\description{
  The \code{Oats} data frame has 72 rows and 4 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Block}{
      an ordered factor with levels
      \code{VI} < \code{V} < \code{III} < \code{IV} < \code{II} < \code{I}
    }
    \item{Variety}{
      a factor with levels
      \code{Golden Rain} 
      \code{Marvellous} 
      \code{Victory} 
    }
    \item{nitro}{
      a numeric vector
    }
    \item{yield}{
      a numeric vector
    }
  }
}
\details{
  These data have been introduced by Yates (1935) as an
  example of a split-plot design. The treatment structure used in the
  experiment was a 3$\times$4 full factorial, with three varieties of
  oats and four concentrations of nitrogen. The experimental units were
  arranged into six blocks, each with three whole-plots subdivided into
  four subplots. The varieties of oats were assigned randomly to the
  whole-plots and the concentrations of nitrogen to the subplots.  All
  four concentrations of nitrogen were used on each whole-plot.
}
\source{
  Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
    and S-PLUS}, Springer, New York.  (Appendix A.15)

  Venables, W. N. and Ripley, B. D. (1999) \emph{Modern Applied
    Statistics with S-PLUS (3rd ed)}, Springer, New York.
}
\examples{
data(Oats)
}
\keyword{datasets}

\eof
\name{Orthodont}
\alias{Orthodont}
\non_function{}
\title{Growth curve data on an orthdontic measurement}
\description{
  The \code{Orthodont} data frame has 108 rows and 4 columns of the
  change in an orthdontic measurement over time for several young subjects.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{distance}{
      a numeric vector of distances from the pituitary to the
      pterygomaxillary fissure (mm).  These distances are measured
      on x-ray images of the skull.
    }
    \item{age}{
      a numeric vector of ages of the subject (yr).
    }
    \item{Subject}{
      an ordered factor indicating the subject on which the
      measurement was made.  The levels are labelled \code{M01}
      to \code{M16} for the males and \code{F01} to \code{F13} for
      the females.  The ordering is by increasing average distance
      within sex.
    }
    \item{Sex}{
      a factor with levels
      \code{Male} and
      \code{Female} 
    }
  }
}
\details{
  Investigators at the University of North Carolina Dental School
  followed the growth of 27 children (16 males, 11 females) from age 8
  until age 14.  Every two years they measured the distance between the
  pituitary and the pterygomaxillary fissure, two points that are easily
  identified on x-ray exposures of the side of the head.
}
\source{
  Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
    and S-PLUS}, Springer, New York.  (Appendix A.17)

  Potthoff, R. F. and Roy,  S. N. (1964), ``A generalized multivariate
  analysis of variance model useful especially for growth curve
  problems'', \emph{Biometrika}, \bold{51}, 313--326.
}
\examples{
data(Orthodont)
formula(Orthodont)
plot(Orthodont)
}
\keyword{datasets}

\eof
\name{Ovary}
\alias{Ovary}
\non_function{}
\title{Counts of Ovarian Follicles}
\description{
  The \code{Ovary} data frame has 308 rows and 3 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Mare}{
      an ordered factor indicating the mare on which the
      measurement is made.
    }
    \item{Time}{
      time in the estrus cycle.  The data were recorded daily from 3
      days before ovulation until 3 days after the next ovulation. The
      measurement times for each mare are scaled so that the ovulations for
      each mare occur at times 0 and 1.
    }
    \item{follicles}{
      the number of ovarian follicles greater than 10 mm in diameter.
    }
  }
}
\details{
  Pierson and Ginther (1987) report on a study of the number of large
  ovarian follicles detected in different mares at several times in
  their estrus cycles.
}
\source{
  Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
    and S-PLUS}, Springer, New York.  (Appendix A.18)

  Pierson, R. A. and Ginther, O. J. (1987), Follicular population
  dynamics during the estrus cycle of the mare, \emph{Animal
    Reproduction Science}, \bold{14}, 219-231.
}
\examples{
data(Ovary)
}
\keyword{datasets}

\eof
\name{Oxboys}
\alias{Oxboys}
\non_function{}
\title{Heights of Boys in Oxford}
\description{
  The \code{Oxboys} data frame has 234 rows and 4 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Subject}{
      an ordered factor giving a unique identifier for each boy in
      the experiment
    }
    \item{age}{
      a numeric vector giving the standardized age (dimensionless)
    }
    \item{height}{
      a numeric vector giving the height of the boy (cm)
    }
    \item{Occasion}{
      an ordered factor - the result of converting \code{age} from a
      continuous variable to a count so these slightly unbalanced
      data can be analyzed as balanced.
    }
  }
}
\details{
  These data are described in Goldstein (1987) as data on the
  height of a selection of boys from Oxford, England versus a
  standardized age.
}
\source{
  Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in
    S and S-PLUS}, Springer, New York.  (Appendix A.19)
}
\examples{
data(Oxboys)
}
\keyword{datasets}

\eof
\name{Oxide}
\alias{Oxide}
\non_function{}
\title{Variability in Semiconductor Manufacturing}
\description{
  The \code{Oxide} data frame has 72 rows and 5 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Source}{
      a factor with levels
      \code{1} and
      \code{2} 
    }
    \item{Lot}{
      a factor giving a unique identifier for each lot.
    }
    \item{Wafer}{
      a factor giving a unique identifier for each wafer within a lot.
    }
    \item{Site}{
      a factor with levels
      \code{1}, 
      \code{2}, and 
      \code{3} 
    }
    \item{Thickness}{
      a numeric vector giving the thickness of the oxide layer.
    }
  }
}
\details{
  These data are described in Littell et al. (1996, p. 155) as coming
  ``from a passive data collection study in the semiconductor industry
  where the objective is to estimate the variance components to
  determine the assignable causes of the observed variability.'' The
  observed response is the thickness of the oxide layer on silicon
  wafers, measured at three different sites of each of three wafers
  selected from each of eight lots sampled from the population of
  lots.
}
\source{
  Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
    and S-PLUS}, Springer, New York.  (Appendix A.20)

  Littell, R. C., Milliken, G. A., Stroup, W. W. and Wolfinger,
  R. D. (1996), \emph{SAS System for Mixed Models}, SAS Institute,
  Cary, NC.
}
\examples{
data(Oxide)
}
\keyword{datasets}

\eof
\name{PBG}
\alias{PBG}
\non_function{}
\title{Effect of Phenylbiguanide on Blood Pressure}
\description{
    The \code{PBG} data frame has 60 rows and 5 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{deltaBP}{
	    a numeric vector
	}
	\item{dose}{
	    a numeric vector
	}
	\item{Run}{
	    an ordered factor with levels
	    \code{T5} < \code{T4} < \code{T3} < \code{T2} < \code{T1} < \code{P5} < \code{P3} < \code{P2} < \code{P4} < \code{P1}
	}
	\item{Treatment}{
	    a factor with levels
	    \code{MDL 72222} 
	    \code{Placebo} 
	}
	\item{Rabbit}{
	    an ordered factor with levels
	    \code{5} < \code{3} < \code{2} < \code{4} < \code{1}
	}
    }
}
\details{
Data on an experiment to examine the effect of a antagonist MDL 72222
on the change in blood pressure experienced with increasing dosage of
phenylbiguanide are described in Ludbrook (1994) and analyzed
in Venables and Ripley (1999, section 8.8).  Each of five rabbits was exposed to
increasing doses of phenylbiguanide after having either a placebo or
the HD5-antagonist MDL 72222 administered.
}
\source{
    Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
      and S-PLUS}, Springer, New York.  (Appendix A.21)

    Venables, W. N. and Ripley, B. D. (1999) \emph{Modern Applied
      Statistics with S-PLUS (3rd ed)}, Springer, New York.

    Ludbrook, J. (1994), Repeated measurements and multiple comparisons
    in cardiovascular research, \emph{Cardiovascular Research},
    \bold{28}, 303-311.
}
\examples{
data(PBG)
}
\keyword{datasets}

\eof
\name{Phenobarb}
\alias{Phenobarb}
\non_function{}
\title{Phenobarbitol Kinetics}
\description{
  The \code{Phenobarb} data frame has 744 rows and 7 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Subject}{
      an ordered factor identifying the infant.
    }
    \item{Wt}{
      a numeric vector giving the birth weight of the infant (kg).
    }
    \item{Apgar}{
      an ordered factor giving the the 5-minute Apgar score for
      the infant.  This is an indication of health of the newborn
      infant.
    }
    \item{ApgarInd}{
      a factor indicating whether the 5-minute Apgar score is \code{< 5}
      or \code{>= 5}. 
    }
    \item{time}{
      a numeric vector giving the time when the sample is drawn or
      drug administered (hr).
    }
    \item{dose}{
      a numeric vector giving the dose of drug administered
      (\eqn{u}{\mu}g/kg).
    }
    \item{conc}{
      a numeric vector giving the phenobarbital concentration in
      the serum (\eqn{u}{\mu}g/L).
    }
  }
}
\details{
  Data from a pharmacokinetics study of phenobarbital in neonatal
  infants.  During the first few days of life the infants receive
  multiple doses of phenobarbital for prevention of seizures.  At
  irregular intervals blood samples are drawn and serum phenobarbital
  concentrations are determined.  The data
  were originally given in Grasela and Donn(1985) and are analyzed
  in Boeckmann, Sheiner and Beal (1994), in
  Davidian and Giltinan (1995), and in Littell et al. (1996).
}
\source{
  Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
    and S-PLUS}, Springer, New York.  (Appendix A.23)

  Davidian, M. and Giltinan, D. M. (1995), \emph{Nonlinear Models for
    Repeated Measurement Data},  Chapman and Hall, London.  (section 6.6)

  Grasela and Donn (1985), Neonatal population pharmacokinetics of
  phenobarbital derived from routine clinical data,
  \emph{Developmental Pharmacology and Therapeutics}, \bold{8},
  374-383.

  Boeckmann, A. J., Sheiner, L. B., and Beal, S. L. (1994),
  \emph{NONMEM Users Guide: Part V}, University of California, San
  Francisco.

  Littell, R. C., Milliken, G. A., Stroup, W. W. and Wolfinger,
  R. D. (1996), \emph{SAS System for Mixed Models}, SAS Institute,
  Cary, NC.
}
\examples{
data(Phenobarb)
}
\keyword{datasets}

\eof
\name{Pixel}
\alias{Pixel}
\non_function{}
\title{X-ray pixel intensities over time}
\description{
    The \code{Pixel} data frame has 102 rows and 4 columns of data on the
    pixel intensities of CT scans of dogs over time
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{Dog}{
	    a factor with levels \code{1} to \code{10} designating the dog
	    on which the scan was made
	}
	\item{Side}{
	    a factor with levels \code{L} and \code{R} designating the side
	    of the dog being scanned
	}
	\item{day}{
	    a numeric vector giving the day post injection of the contrast
	    on which the scan was made
	}
	\item{pixel}{
	    a numeric vector of pixel intensities
	}
    }
}
\source{
    Pinheiro, J. C. and Bates, D. M. (2000) \emph{Mixed-effects Models in
	S and S-PLUS}, Springer.
}
\examples{
data(Pixel)
fm1 <- lme(pixel ~ day + I(day^2), data = Pixel,
           random = list(Dog = ~ day, Side = ~ 1))
summary(fm1)
VarCorr(fm1)
}
\keyword{datasets}

\eof
\name{Quinidine}
\alias{Quinidine}
\non_function{}
\title{Quinidine Kinetics}
\description{
  The \code{Quinidine} data frame has 1471 rows and 14 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Subject}{
      a factor identifying the patient on whom the data were collected.
    }
    \item{time}{
      a numeric vector giving the time (hr) at which the drug was
      administered or the blood sample drawn.  This is measured
      from the time the patient entered the study.
    }
    \item{conc}{
      a numeric vector giving the serum quinidine concentration (mg/L).
    }
    \item{dose}{
      a numeric vector giving the dose of drug administered (mg).
      Although there were two different forms of quinidine
      administered, the doses were adjusted for differences in
      salt content by conversion to milligrams of quinidine base.
    }
    \item{interval}{
      a numeric vector giving the when the drug has been given at
      regular intervals for a sufficiently long period of time to
      assume steady state behavior, the interval is recorded.
    }
    \item{Age}{
      a numeric vector giving the age of the subject on entry to
      the study (yr).
    }
    \item{Height}{
      a numeric vector giving the height of the subject on entry
      to the study (in.).
    }
    \item{Weight}{
      a numeric vector giving the body weight of the subject (kg).
    }
    \item{Race}{
      a factor with levels
      \code{Caucasian}, 
      \code{Latin}, and
      \code{Black} identifying the race of the subject.
    }
    \item{Smoke}{
      a factor with levels
      \code{no} and
      \code{yes} giving smoking status at the time of the
      measurement.
    }
    \item{Ethanol}{
      a factor with levels
      \code{none}, 
      \code{current},
      \code{former} giving ethanol (alcohol) abuse status at the
      time of the measurement.
    }
    \item{Heart}{
      a factor with levels
      \code{No/Mild}, 
      \code{Moderate}, and 
      \code{Severe} indicating congestive heart failure for the subject.
    }
    \item{Creatinine}{
      an ordered factor with levels
      \code{< 50} < \code{>= 50} indicating the creatine clearance (mg/min).
    }
    \item{glyco}{
      a numeric vector giving the alpha-1 acid glycoprotein
      concentration (mg/dL).  Often measured at the same time as
      the quinidine concentration.
    }
  }
}
\details{
  Verme et al. (1992) analyze routine clinical data on
  patients receiving the drug quinidine as a treatment for cardiac
  arrythmia (atrial fibrillation of ventricular arrythmias).  All
  patients were receiving oral quinidine doses.  At irregular intervals
  blood samples were drawn and serum concentrations of quinidine were
  determined.  These data are
  analyzed in several publications, including Davidian and Giltinan (1995,
  section 9.3).
}
\source{
  Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
    and S-PLUS}, Springer, New York.  (Appendix A.25)

  Davidian, M. and Giltinan, D. M. (1995), \emph{Nonlinear Models for
    Repeated Measurement Data},  Chapman and Hall, London.

  Verme, C. N., Ludden,  T. M., Clementi, W. A. and Harris,
  S. C. (1992), Pharmacokinetics of quinidine in male patients: A
  population analysis, \emph{Clinical Pharmacokinetics}, \bold{22},
  468-480.
}
\examples{
data(Quinidine)
}
\keyword{datasets}

\eof
\name{Rail}
\alias{Rail}
\non_function{}
\title{Evaluation of Stress in Railway Rails}
\description{
    The \code{Rail} data frame has 18 rows and 2 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{Rail}{
	    an ordered factor identifying the rail on which the
	    measurement was made.
	}
	\item{travel}{
	  a numeric vector giving the travel time for ultrasonic
	  head-waves in the rail (nanoseconds).  The value given is the
	  original travel time minus 36,100 nanoseconds.
	}
    }
}
\details{
  Devore (2000, Example 10.10, p. 427) cites data from an article in
  \emph{Materials Evaluation} on ``a study of travel time for a certain
  type of wave that results from longitudinal stress of rails used for
  railroad track.''
}
\source{
    Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
	and S-PLUS}, Springer, New York.  (Appendix A.26)

    Devore, J. L. (2000), \emph{Probability and Statistics for
      Engineering and the Sciences (5th ed)}, Duxbury, Boston, MA.
}
\examples{
data(Rail)
}
\keyword{datasets}

\eof
\name{RatPupWeight}
\alias{RatPupWeight}
\non_function{}
\title{The weight of rat pups}
\description{
    The \code{RatPupWeight} data frame has 322 rows and 5 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{weight}{
	    a numeric vector
	}
	\item{sex}{
	    a factor with levels
	    \code{Male} 
	    \code{Female} 
	}
	\item{Litter}{
	    an ordered factor with levels
	    \code{9} < \code{8} < \code{7} < \code{4} < \code{2} < \code{10} < \code{1} < \code{3} < \code{5} < \code{6} < \code{21} < \code{22} < \code{24} < \code{27} < \code{26} < \code{25} < \code{23} < \code{17} < \code{11} < \code{14} < \code{13} < \code{15} < \code{16} < \code{20} < \code{19} < \code{18} < \code{12}
	}
	\item{Lsize}{
	    a numeric vector
	}
	\item{Treatment}{
	    an ordered factor with levels
	    \code{Control} < \code{Low} < \code{High}
	}
    }
}
\details{
}
\source{
    Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
	and S-PLUS}, Springer, New York.
}
\examples{
data(RatPupWeight)
}
\keyword{datasets}

\eof
\name{Relaxin}
\alias{Relaxin}
\non_function{}
\title{Assay for Relaxin}
\description{
    The \code{Relaxin} data frame has 198 rows and 3 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{Run}{
	    an ordered factor with levels
	    \code{5} < \code{8} < \code{9} < \code{3} < \code{4} < \code{2} < \code{7} < \code{1} < \code{6}
	}
	\item{conc}{
	    a numeric vector
	}
	\item{cAMP}{
	    a numeric vector
	}
    }
}
\details{
}
\source{
    Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
	and S-PLUS}, Springer, New York.
}
\examples{
data(Relaxin)
}
\keyword{datasets}

\eof
\name{Remifentanil}
\alias{Remifentanil}
\non_function{}
\title{Pharmacokinetics of remifentanil}
\description{
    The \code{Remifentanil} data frame has 2107 rows and 12 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{ID}{
	    a numeric vector
	}
	\item{Subject}{
	    an ordered factor
	}
	\item{Time}{
	    a numeric vector
	}
	\item{conc}{
	    a numeric vector
	}
	\item{Rate}{
	    a numeric vector
	}
	\item{Amt}{
	    a numeric vector
	}
	\item{Age}{
	    a numeric vector
	}
	\item{Sex}{
	    a factor with levels
	    \code{Female} 
	    \code{Male} 
	}
	\item{Ht}{
	    a numeric vector
	}
	\item{Wt}{
	    a numeric vector
	}
	\item{BSA}{
	    a numeric vector
	}
	\item{LBM}{
	    a numeric vector
	}
    }
}
\details{
}
\source{
    Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
	and S-PLUS}, Springer, New York.
}
\examples{
data(Remifentanil)
}
\keyword{datasets}

\eof
\name{Soybean}
\alias{Soybean}
\non_function{}
\title{Growth of soybean plants}
\description{
  The \code{Soybean} data frame has 412 rows and 5 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Plot}{
      a factor giving a unique identifier for each plot.
    }
    \item{Variety}{
      a factor indicating the variety; Forrest (F) or Plant
      Introduction \#416937 (P).
    }
    \item{Year}{
      a factor indicating the year the plot was planted.
    }
    \item{Time}{
      a numeric vector giving the time the sample was taken (days
      after planting).
    }
    \item{weight}{
      a numeric vector giving the average leaf weight per plant (g).
    }
  }
}
\details{
  These data are described in
  Davidian and Giltinan (1995, 1.1.3, p.7) as ``Data from an
  experiment to compare growth patterns of two genotypes of soybeans:
  Plant Introduction \#416937 (P), an experimental strain, and Forrest
  (F), a commercial variety.''
}
\source{
  Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
    and S-PLUS}, Springer, New York.  (Appendix A.27)

  Davidian, M. and Giltinan, D. M. (1995), \emph{Nonlinear Models for
    Repeated Measurement Data},  Chapman and Hall, London.
}
\examples{
data(Soybean)
summary(fm1 <- nlsList(SSlogis, data = Soybean))
}
\keyword{datasets}

\eof
\name{Spruce}
\alias{Spruce}
\non_function{}
\title{Growth of Spruce Trees}
\description{
  The \code{Spruce} data frame has 1027 rows and 4 columns.
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Tree}{
      a factor giving a unique identifier for each tree.
    }
    \item{days}{
      a numeric vector giving the number of days since the
      beginning of the experiment.
    }
    \item{logSize}{
      a numeric vector giving the logarithm of an estimate of the
      volume of the tree trunk.
    }
    \item{plot}{
      a factor identifying the plot in which the tree was grown.
    }
  }
}
\details{
  Diggle, Liang, and Zeger (1994, Example 1.3, page 5)
  describe data on the growth of spruce trees that have been exposed to
  an ozone-rich atmosphere or to a normal atmosphere.
}
\source{
  Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
    and S-PLUS}, Springer, New York.  (Appendix A.28)

  Diggle, Peter J., Liang, Kung-Yee and Zeger, Scott L. (1994),
  \emph{Analysis of longitudinal data}, Oxford University Press,
  Oxford.
}
\examples{
data(Spruce)
}
\keyword{datasets}

\eof
\name{Tetracycline1}
\alias{Tetracycline1}
\non_function{}
\title{Pharmacokinetics of tetracycline}
\description{
    The \code{Tetracycline1} data frame has 40 rows and 4 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{conc}{
	    a numeric vector
	}
	\item{Time}{
	    a numeric vector
	}
	\item{Subject}{
	    an ordered factor with levels
	    \code{5} < \code{3} < \code{2} < \code{4} < \code{1}
	}
	\item{Formulation}{
	    a factor with levels
	    \code{tetrachel} 
	    \code{tetracyn} 
	}
    }
}
\details{
}
\source{
    Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
	and S-PLUS}, Springer, New York.
}
\examples{
data(Tetracycline1)
}
\keyword{datasets}

\eof
\name{Tetracycline2}
\alias{Tetracycline2}
\non_function{}
\title{Pharmacokinetics of tetracycline}
\description{
    The \code{Tetracycline2} data frame has 40 rows and 4 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{conc}{
	    a numeric vector
	}
	\item{Time}{
	    a numeric vector
	}
	\item{Subject}{
	    an ordered factor with levels
	    \code{4} < \code{5} < \code{2} < \code{1} < \code{3}
	}
	\item{Formulation}{
	    a factor with levels
	    \code{Berkmycin} 
	    \code{tetramycin} 
	}
    }
}
\details{
}
\source{
    Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
	and S-PLUS}, Springer, New York.
}
\examples{
data(Tetracycline2)
}
\keyword{datasets}

\eof
% $Id: VarCorr.Rd,v 1.8 2002/03/05 14:59:39 bates Exp $
\name{VarCorr}
\title{Extract variance and correlation components}
\usage{
VarCorr(x, sigma, rdig)
}
\alias{VarCorr}
\alias{VarCorr.lme}
\alias{VarCorr.pdMat}
\alias{VarCorr.pdBlocked}
\alias{summary.VarCorr.lme}
\alias{print.VarCorr.lme}
\arguments{
 \item{x}{a fitted model object, usually an object inheriting from
     class \code{lme}. 
 }
 \item{sigma}{an optional numeric value used as a multiplier for the
     standard deviations. Default is \code{1}.
 }
 \item{rdig}{an optional integer value specifying the number of digits
     used to represent correlation estimates. Default is \code{3}.
 }
}
\description{
  This function calculates the estimated variances, standard
  deviations, and correlations between the random-effects terms in a
  linear mixed-effects model, of class \code{lme}, or a nonlinear
  mixed-effects model, of class \code{nlme}. The within-group error
  variance and standard deviation are also calculated. 
}
\value{
    a matrix with the estimated variances, standard deviations, and
    correlations for the random effects. The first two columns, named
    \code{Variance} and \code{StdDev}, give, respectively, the variance
    and the standard deviations. If there are correlation components in
    the random effects model, the third column, named \code{Corr},
    and the remaining unnamed columns give the estimated correlations
    among random effects within the same level of grouping. The
    within-group error variance and standard deviation are included as
    the last row in the matrix. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{lme}}, \code{\link{nlme}}}
\examples{
data(Orthodont)
fm1 <- lme(distance ~ age, data = Orthodont, random = ~age)
VarCorr(fm1)
}
\keyword{models}

\eof
% $Id: Variogram.Rd,v 1.5.2.1 2003/04/17 22:28:45 bates Exp $
\name{Variogram}
\title{Calculate Semi-variogram}
\usage{
Variogram(object, distance, \dots)
}
\alias{Variogram}
\arguments{
  \item{object}{a numeric vector with the values to be used for
    calculating the semi-variogram, usually a residual vector from a
    fitted model.}
  \item{distance}{a numeric vector with the pairwise distances
    corresponding to the elements of \code{object}. The order of the
    elements in \code{distance} must correspond to the pairs
    \code{(1,2), (1,3), \dots, (n-1,n)}, with \code{n} representing the
    length of \code{object}, and must have length \code{n(n-1)/2}.}
  \item{\dots}{some methods for this generic function require additional
    arguments.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include \code{default}, \code{gls} and \code{lme}. See
  the appropriate method documentation for a description of the
  arguments. 
}
\value{
  will depend on the method function used; see the appropriate documentation.
}
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
}  

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{Variogram.default}},\code{\link{Variogram.gls}},
  \code{\link{Variogram.lme}}, \code{\link{plot.Variogram}}}
\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: Variogram.corExp.Rd,v 1.7.2.1 2002/08/09 19:45:29 bates Exp $
\name{Variogram.corExp}
\title{Calculate Semi-variogram for a corExp Object}
\usage{
\method{Variogram}{corExp}(object, distance, sig2, length.out, \dots)
}
\alias{Variogram.corExp}
\arguments{
  \item{object}{an object inheriting from class \code{corExp},
    representing an exponential spatial correlation structure.}
  \item{distance}{an optional numeric vector with the distances at
    which the semi-variogram is to be calculated. Defaults to
    \code{NULL}, in which case a sequence of length \code{length.out}
    between the minimum and maximum values of
    \code{getCovariate(object)} is used.}
  \item{sig2}{an optional numeric value representing the process
    variance. Defaults to \code{1}.}
  \item{length.out}{an optional integer specifying the length of the
    sequence of distances to be used for calculating the semi-variogram,
    when \code{distance = NULL}. Defaults to \code{50}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function calculates the semi-variogram values
  corresponding to the Exponential correlation model, using the estimated
  coefficients corresponding to \code{object}, at the distances defined
  by \code{distance}.
}
\value{
  a data frame with columns \code{variog} and \code{dist} representing,
  respectively, the semi-variogram values and the corresponding
  distances. The returned value inherits from class \code{Variogram}. 
}
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
}  
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{corExp}}, \code{\link{plot.Variogram}}}
\examples{
library(mva)
data(BodyWeight)
cs1 <- corExp(3, form = ~ Time | Rat)
cs1 <- Initialize(cs1, BodyWeight)
Variogram(cs1)[1:10,]
}
\keyword{models}

\eof
% $Id: Variogram.corGaus.Rd,v 1.7.2.1 2002/08/09 19:45:29 bates Exp $
\name{Variogram.corGaus}
\title{Calculate Semi-variogram for a corGaus Object}
\usage{
\method{Variogram}{corGaus}(object, distance, sig2, length.out, \dots)
}
\alias{Variogram.corGaus}
\arguments{
  \item{object}{an object inheriting from class \code{corGaus},
    representing an Gaussian spatial correlation structure.}
  \item{distance}{an optional numeric vector with the distances at
    which the semi-variogram is to be calculated. Defaults to
    \code{NULL}, in which case a sequence of length \code{length.out}
    between the minimum and maximum values of
    \code{getCovariate(object)} is used.}
  \item{sig2}{an optional numeric value representing the process
    variance. Defaults to \code{1}.}
  \item{length.out}{an optional integer specifying the length of the
    sequence of distances to be used for calculating the semi-variogram,
    when \code{distance = NULL}. Defaults to \code{50}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function calculates the semi-variogram values
  corresponding to the Gaussian correlation model, using the estimated
  coefficients corresponding to \code{object}, at the distances defined
  by \code{distance}.
}
\value{
  a data frame with columns \code{variog} and \code{dist} representing,
  respectively, the semi-variogram values and the corresponding
  distances. The returned value inherits from class \code{Variogram}. 
}
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
}  
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{corGaus}}, \code{\link{plot.Variogram}}}
\examples{
data(BodyWeight)
cs1 <- corGaus(3, form = ~ Time | Rat)
cs1 <- Initialize(cs1, BodyWeight)
Variogram(cs1)[1:10,]
}
\keyword{models}

\eof
% $Id: Variogram.corLin.Rd,v 1.7.2.1 2002/08/09 19:45:29 bates Exp $
\name{Variogram.corLin}
\title{Calculate Semi-variogram for a corLin Object}
\usage{
\method{Variogram}{corLin}(object, distance, sig2, length.out, \dots)
}
\alias{Variogram.corLin}
\arguments{
  \item{object}{an object inheriting from class \code{corLin},
    representing an Linear spatial correlation structure.}
  \item{distance}{an optional numeric vector with the distances at
    which the semi-variogram is to be calculated. Defaults to
    \code{NULL}, in which case a sequence of length \code{length.out}
    between the minimum and maximum values of
    \code{getCovariate(object)} is used.}
  \item{sig2}{an optional numeric value representing the process
    variance. Defaults to \code{1}.}
  \item{length.out}{an optional integer specifying the length of the
    sequence of distances to be used for calculating the semi-variogram,
    when \code{distance = NULL}. Defaults to \code{50}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function calculates the semi-variogram values
  corresponding to the Linear correlation model, using the estimated
  coefficients corresponding to \code{object}, at the distances defined
  by \code{distance}.
}
\value{
  a data frame with columns \code{variog} and \code{dist} representing,
  respectively, the semi-variogram values and the corresponding
  distances. The returned value inherits from class \code{Variogram}. 
}
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
}  
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{corLin}}, \code{\link{plot.Variogram}}}
\examples{
data(BodyWeight)
cs1 <- corLin(15, form = ~ Time | Rat)
cs1 <- Initialize(cs1, BodyWeight)
Variogram(cs1)[1:10,]
}
\keyword{models}

\eof
% $Id: Variogram.corRatio.Rd,v 1.7.2.1 2002/08/09 19:45:29 bates Exp $
\name{Variogram.corRatio}
\title{Calculate Semi-variogram for a corRatio Object}
\usage{
\method{Variogram}{corRatio}(object, distance, sig2, length.out, \dots)
}
\alias{Variogram.corRatio}
\arguments{
  \item{object}{an object inheriting from class \code{corRatio},
    representing an Rational Quadratic spatial correlation structure.}
  \item{distance}{an optional numeric vector with the distances at
    which the semi-variogram is to be calculated. Defaults to
    \code{NULL}, in which case a sequence of length \code{length.out}
    between the minimum and maximum values of
    \code{getCovariate(object)} is used.}
  \item{sig2}{an optional numeric value representing the process
    variance. Defaults to \code{1}.}
  \item{length.out}{an optional integer specifying the length of the
    sequence of distances to be used for calculating the semi-variogram,
    when \code{distance = NULL}. Defaults to \code{50}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function calculates the semi-variogram values
  corresponding to the Rational Quadratic correlation model, using the
  estimated coefficients corresponding to \code{object}, at the
  distances defined by \code{distance}.
}
\value{
  a data frame with columns \code{variog} and \code{dist} representing,
  respectively, the semi-variogram values and the corresponding
  distances. The returned value inherits from class \code{Variogram}. 
}
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
}  
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{corRatio}}, \code{\link{plot.Variogram}}}
\examples{
data(BodyWeight)
cs1 <- corRatio(7, form = ~ Time | Rat)
cs1 <- Initialize(cs1, BodyWeight)
Variogram(cs1)[1:10,]
}
\keyword{models}

\eof
% $Id: Variogram.corSpatial.Rd,v 1.7.2.2 2003/04/17 22:28:45 bates Exp $
\name{Variogram.corSpatial}
\title{Calculate Semi-variogram for a corSpatial Object}
\usage{
\method{Variogram}{corSpatial}(object, distance, sig2, length.out, FUN, \dots)
}
\alias{Variogram.corSpatial}
\arguments{
  \item{object}{an object inheriting from class \code{corSpatial},
    representing spatial correlation structure.}
  \item{distance}{an optional numeric vector with the distances at
    which the semi-variogram is to be calculated. Defaults to
    \code{NULL}, in which case a sequence of length \code{length.out}
    between the minimum and maximum values of
    \code{getCovariate(object)} is used.}
  \item{sig2}{an optional numeric value representing the process
    variance. Defaults to \code{1}.}
  \item{length.out}{an optional integer specifying the length of the
    sequence of distances to be used for calculating the semi-variogram,
    when \code{distance = NULL}. Defaults to \code{50}.}
  \item{FUN}{a function of two arguments, the distance and the range
    corresponding to \code{object}, specifying the semi-variogram
    model.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function calculates the semi-variogram values
  corresponding to the model defined in \code{FUN}, using the estimated
  coefficients corresponding to \code{object}, at the distances defined
  by \code{distance}.
}
\value{
  a data frame with columns \code{variog} and \code{dist} representing,
  respectively, the semi-variogram values and the corresponding
  distances. The returned value inherits from class \code{Variogram}. 
}
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
}  
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{corSpatial}}, \code{\link{Variogram.default}},
  \code{\link{Variogram.corExp}}, \code{\link{Variogram.corGaus}},
  \code{\link{Variogram.corLin}}, \code{\link{Variogram.corRatio}},
  \code{\link{Variogram.corSpher}}, \code{\link{plot.Variogram}}}
\examples{
data(BodyWeight)
cs1 <- corExp(3, form = ~ Time | Rat)
cs1 <- Initialize(cs1, BodyWeight)
Variogram(cs1, FUN = function(x, y) (1 - exp(-x/y)))[1:10,]
}
\keyword{models}

\eof
% $Id: Variogram.corSpher.Rd,v 1.7.2.1 2002/08/09 19:45:29 bates Exp $
\name{Variogram.corSpher}
\title{Calculate Semi-variogram for a corSpher Object}
\usage{
\method{Variogram}{corSpher}(object, distance, sig2, length.out, \dots)
}
\alias{Variogram.corSpher}
\arguments{
  \item{object}{an object inheriting from class \code{corSpher},
    representing an Spherical spatial correlation structure.}
  \item{distance}{an optional numeric vector with the distances at
    which the semi-variogram is to be calculated. Defaults to
    \code{NULL}, in which case a sequence of length \code{length.out}
    between the minimum and maximum values of
    \code{getCovariate(object)} is used.}
  \item{sig2}{an optional numeric value representing the process
    variance. Defaults to \code{1}.}
  \item{length.out}{an optional integer specifying the length of the
    sequence of distances to be used for calculating the semi-variogram,
    when \code{distance = NULL}. Defaults to \code{50}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function calculates the semi-variogram values
  corresponding to the Spherical correlation model, using the estimated
  coefficients corresponding to \code{object}, at the distances defined
  by \code{distance}.
}
\value{
  a data frame with columns \code{variog} and \code{dist} representing,
  respectively, the semi-variogram values and the corresponding
  distances. The returned value inherits from class \code{Variogram}. 
}
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
}  
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{corSpher}}, \code{\link{plot.Variogram}}}
\examples{
data(BodyWeight)
cs1 <- corSpher(15, form = ~ Time | Rat)
cs1 <- Initialize(cs1, BodyWeight)
Variogram(cs1)[1:10,]
}
\keyword{models}

\eof
% $Id: Variogram.default.Rd,v 1.8.2.1 2003/04/17 22:28:45 bates Exp $
\name{Variogram.default}
\title{Calculate Semi-variogram}
\usage{
\method{Variogram}{default}(object, distance, \dots)
}
\alias{Variogram.default}
\arguments{
  \item{object}{a numeric vector with the values to be used for
    calculating the semi-variogram, usually a residual vector from a
    fitted model.}
  \item{distance}{a numeric vector with the pairwise distances
    corresponding to the elements of \code{object}. The order of the
    elements in \code{distance} must correspond to the pairs
    \code{(1,2), (1,3), \dots, (n-1,n)}, with \code{n} representing the
    length of \code{object}, and must have length \code{n(n-1)/2}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function calculates the semi-variogram for an arbitrary
  vector \code{object}, according to the distances in \code{distance}.
  For each pair of elements \eqn{x,y} in \code{object}, the
  corresponding semi-variogram is \eqn{(x-y)^2/2}. The semi-variogram is
  useful for identifying and modeling spatial correlation structures in
  observations with constant expectation and constant variance.   
}
\value{
  a data frame with columns \code{variog} and \code{dist} representing,
  respectively, the semi-variogram values and the corresponding
  distances. The returned value inherits from class \code{Variogram}. 
}
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
}  

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{Variogram.gls}}, \code{\link{Variogram.lme}},
  \code{\link{plot.Variogram}}} 
\examples{
data(Ovary)
\dontrun{
fm1 <- lm(follicles ~ sin(2 * pi * Time) + cos(2 * pi * Time), Ovary,
          subset = Mare == 1)
Variogram(resid(fm1), dist(1:29))[1:10,]
}
}
\keyword{models}

\eof
% $Id: Variogram.gls.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{Variogram.gls}
\title{Calculate Semi-variogram for Residuals from a gls Object}
\usage{
\method{Variogram}{gls}(object, distance, form, resType, data,
      na.action, maxDist, length.out, collapse, nint, breaks,
      robust, metric, \dots)
}
\alias{Variogram.gls}
\arguments{
  \item{object}{an object inheriting from class \code{gls}, representing
    a generalized least squares fitted model.}
  \item{distance}{an optional numeric vector with the distances between
    residual pairs. If a grouping variable is present, only the
    distances between residual pairs within the same group should be
    given. If missing, the distances are calculated based on the
    values of the arguments \code{form}, \code{data}, and
    \code{metric}, unless \code{object} includes a \code{corSpatial}
    element, in which case the associated covariate (obtained with the
    \code{getCovariate} method) is used.}
  \item{form}{an optional one-sided formula specifying the covariate(s)
    to be used for calculating the distances between residual pairs and,
    optionally, a grouping factor for partitioning the residuals (which
    must appear to the right of a \code{|} operator in
    \code{form}). Default is \code{~1}, implying that the observation 
    order within the groups is used to obtain the distances.}
  \item{resType}{an optional character string specifying the type of
    residuals to be used. If \code{"response"}, the "raw" residuals
    (observed - fitted) are used; else, if \code{"pearson"}, the
    standardized residuals (raw residuals divided by the corresponding
    standard errors) are used; else, if \code{"normalized"}, the
    normalized residuals (standardized residuals pre-multiplied by the
    inverse square-root factor of the estimated error correlation
    matrix) are used. Partial matching of arguments is used, so only the
    first character needs to be provided. Defaults to \code{"pearson"}.} 
  \item{data}{an optional data frame in which to interpret the variables
    in \code{form}. By default, the same data used to fit \code{object}
    is used.}
  \item{na.action}{a function that indicates what should happen when the
    data contain \code{NA}s. The default action (\code{na.fail}) causes
    an error message to be printed and the function to terminate, if there
    are any incomplete observations.}
  \item{maxDist}{an optional numeric value for the maximum distance used
    for calculating the semi-variogram between two residuals. By default
    all residual pairs are included.}
  \item{length.out}{an optional integer value. When \code{object}
    includes a \code{corSpatial} element, its semi-variogram values are
    calculated and this argument is used as the \code{length.out}
    argument to the corresponding \code{Variogram} method. Defaults to
    \code{50}.}
  \item{collapse}{an optional character string specifying the type of
    collapsing to be applied to the individual semi-variogram values. If
    equal to \code{"quantiles"}, the semi-variogram values are split
    according to quantiles of the distance distribution, with equal
    number of observations per group, with possibly varying distance
    interval lengths. Else, if \code{"fixed"}, the semi-variogram values
    are divided according to distance intervals of equal lengths, with
    possibly different number of observations per interval. Else, if
    \code{"none"}, no collapsing is used and the individual
    semi-variogram values are returned. Defaults to \code{"quantiles"}.}
  \item{nint}{an optional integer with the number of intervals to be
    used when collapsing the semi-variogram values. Defaults to \code{20}.}
  \item{robust}{an optional logical value specifying if a robust
    semi-variogram estimator should be used when collapsing the
    individual values. If \code{TRUE} the robust estimator is
    used. Defaults to \code{FALSE}.}
  \item{breaks}{an optional numeric vector with the breakpoints for the
    distance intervals to be used in collapsing the semi-variogram
    values. If not missing, the option specified in \code{collapse} is
    ignored.}
  \item{metric}{an optional character string specifying the distance
    metric to be used. The currently available options are
    \code{"euclidean"} for the root sum-of-squares of distances;
    \code{"maximum"} for the maximum difference; and \code{"manhattan"}
    for the sum of the absolute differences. Partial matching of
    arguments is used, so only the first three characters need to be
    provided. Defaults to \code{"euclidean"}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function calculates the semi-variogram for the residuals
  from a \code{gls} fit. The semi-variogram values are calculated for
  pairs of residuals within the same group level, if a grouping factor
  is present. If \code{collapse} is different from \code{"none"}, the
  individual semi-variogram values are collapsed using either a robust
  estimator (\code{robust = TRUE}) defined in Cressie (1993), or the
  average of the values within the same distance interval. The
  semi-variogram is useful for modeling the error term correlation
  structure.  
}
\value{
  a data frame with columns \code{variog} and \code{dist} representing,
  respectively, the semi-variogram values and the corresponding
  distances. If the semi-variogram values are collapsed, an extra
  column, \code{n.pairs}, with the number of residual pairs used in each
  semi-variogram calculation, is included in the returned data frame. If
  \code{object} includes a \code{corSpatial} element, a data frame with
  its corresponding semi-variogram is included in the returned value, as
  an attribute \code{"modelVariog"}. The returned value inherits from
  class \code{Variogram}. 
} 
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
}  

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gls}}, \code{\link{Variogram.default}},
  \code{\link{Variogram.lme}}, \code{\link{plot.Variogram}}}
\examples{
data(BodyWeight)
\dontrun{
fm1 <- gls(weight ~ Time * Diet, BodyWeight)
Variogram(fm1, form = ~ Time | Rat)[1:10,]
}
}
\keyword{models}

\eof
% $Id: Variogram.lme.Rd,v 1.8 2002/03/05 14:59:39 bates Exp $
\name{Variogram.lme}
\title{Calculate Semi-variogram for Residuals from an lme Object}
\usage{
\method{Variogram}{lme}(object, distance, form, resType, data,
       na.action, maxDist, length.out, collapse, nint, breaks,
       robust, metric, \dots)
}
\alias{Variogram.lme}
\arguments{
  \item{object}{an object inheriting from class \code{lme}, representing
    a fitted linear mixed-effects model.}
  \item{distance}{an optional numeric vector with the distances between
    residual pairs. If a grouping variable is present, only the
    distances between residual pairs within the same group should be
    given. If missing, the distances are calculated based on the
    values of the arguments \code{form}, \code{data}, and
    \code{metric}, unless \code{object} includes a \code{corSpatial}
    element, in which case the associated covariate (obtained with the
    \code{getCovariate} method) is used.}
  \item{form}{an optional one-sided formula specifying the covariate(s)
    to be used for calculating the distances between residual pairs and,
    optionally, a grouping factor for partitioning the residuals (which
    must appear to the right of a \code{|} operator in
    \code{form}). Default is \code{~1}, implying that the observation 
    order within the groups is used to obtain the distances.}
  \item{resType}{an optional character string specifying the type of
    residuals to be used. If \code{"response"}, the "raw" residuals
    (observed - fitted) are used; else, if \code{"pearson"}, the
    standardized residuals (raw residuals divided by the corresponding
    standard errors) are used; else, if \code{"normalized"}, the
    normalized residuals (standardized residuals pre-multiplied by the
    inverse square-root factor of the estimated error correlation
    matrix) are used. Partial matching of arguments is used, so only the
    first character needs to be provided. Defaults to \code{"pearson"}.} 
  \item{data}{an optional data frame in which to interpret the variables
    in \code{form}. By default, the same data used to fit \code{object}
    is used.}
  \item{na.action}{a function that indicates what should happen when the
    data contain \code{NA}s. The default action (\code{na.fail}) causes
    an error message to be printed and the function to terminate, if there
    are any incomplete observations.}
  \item{maxDist}{an optional numeric value for the maximum distance used
    for calculating the semi-variogram between two residuals. By default
    all residual pairs are included.}
  \item{length.out}{an optional integer value. When \code{object}
    includes a \code{corSpatial} element, its semi-variogram values are
    calculated and this argument is used as the \code{length.out}
    argument to the corresponding \code{Variogram} method. Defaults to
    \code{50}.}
  \item{collapse}{an optional character string specifying the type of
    collapsing to be applied to the individual semi-variogram values. If
    equal to \code{"quantiles"}, the semi-variogram values are split
    according to quantiles of the distance distribution, with equal
    number of observations per group, with possibly varying distance
    interval lengths. Else, if \code{"fixed"}, the semi-variogram values
    are divided according to distance intervals of equal lengths, with
    possibly different number of observations per interval. Else, if
    \code{"none"}, no collapsing is used and the individual
    semi-variogram values are returned. Defaults to \code{"quantiles"}.}
  \item{nint}{an optional integer with the number of intervals to be
    used when collapsing the semi-variogram values. Defaults to \code{20}.}
  \item{robust}{an optional logical value specifying if a robust
    semi-variogram estimator should be used when collapsing the
    individual values. If \code{TRUE} the robust estimator is
    used. Defaults to \code{FALSE}.}
  \item{breaks}{an optional numeric vector with the breakpoints for the
    distance intervals to be used in collapsing the semi-variogram
    values. If not missing, the option specified in \code{collapse} is
    ignored.}
  \item{metric}{an optional character string specifying the distance
    metric to be used. The currently available options are
    \code{"euclidean"} for the root sum-of-squares of distances;
    \code{"maximum"} for the maximum difference; and \code{"manhattan"}
    for the sum of the absolute differences. Partial matching of
    arguments is used, so only the first three characters need to be
    provided. Defaults to \code{"euclidean"}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function calculates the semi-variogram for the
  within-group residuals from an \code{lme} fit. The semi-variogram
  values are calculated for pairs of residuals within the same group. If
  \code{collapse} is different from \code{"none"}, the individual
  semi-variogram values are collapsed using either a robust estimator
  (\code{robust = TRUE}) defined in Cressie (1993), or the average of
  the values within the same distance interval. The semi-variogram is
  useful for modeling the error term correlation structure.  
}
\value{
  a data frame with columns \code{variog} and \code{dist} representing,
  respectively, the semi-variogram values and the corresponding
  distances. If the semi-variogram values are collapsed, an extra
  column, \code{n.pairs}, with the number of residual pairs used in each
  semi-variogram calculation, is included in the returned data frame. If
  \code{object} includes a \code{corSpatial} element, a data frame with
  its corresponding semi-variogram is included in the returned value, as
  an attribute \code{"modelVariog"}. The returned value inherits from
  class \code{Variogram}. 
} 
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
}  

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lme}}, \code{\link{Variogram.default}},
  \code{\link{Variogram.gls}}, \code{\link{plot.Variogram}}}
\examples{
data(BodyWeight)
\dontrun{
fm1 <- lme(weight ~ Time * Diet, BodyWeight, ~ Time | Rat)
Variogram(fm1, form = ~ Time | Rat, nint = 10, robust = TRUE)
}
}
\keyword{models}

\eof
\name{Wafer}
\alias{Wafer}
\non_function{}
\title{Modeling of Analog MOS Circuits}
\description{
    The \code{Wafer} data frame has 400 rows and 4 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{Wafer}{
	    a factor with levels
	    \code{1} 
	    \code{2} 
	    \code{3} 
	    \code{4} 
	    \code{5} 
	    \code{6} 
	    \code{7} 
	    \code{8} 
	    \code{9} 
	    \code{10} 
	}
	\item{Site}{
	    a factor with levels
	    \code{1} 
	    \code{2} 
	    \code{3} 
	    \code{4} 
	    \code{5} 
	    \code{6} 
	    \code{7} 
	    \code{8} 
	}
	\item{voltage}{
	    a numeric vector
	}
	\item{current}{
	    a numeric vector
	}
    }
}
\details{
}
\source{
    Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
	and S-PLUS}, Springer, New York.
}
\examples{
data(Wafer)
}
\keyword{datasets}

\eof
\name{Wheat}
\alias{Wheat}
\non_function{}
\title{Yields by growing conditions}
\description{
    The \code{Wheat} data frame has 48 rows and 4 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{Tray}{
	    an ordered factor with levels
	    \code{3} < \code{1} < \code{2} < \code{4} < \code{5} < \code{6} < \code{8} < \code{9} < \code{7} < \code{12} < \code{11} < \code{10}
	}
	\item{Moisture}{
	    a numeric vector
	}
	\item{fertilizer}{
	    a numeric vector
	}
	\item{DryMatter}{
	    a numeric vector
	}
    }
}
\details{
}
\source{
    Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
	and S-PLUS}, Springer, New York.
}
\examples{
data(Wheat)
}
\keyword{datasets}

\eof
\name{Wheat2}
\alias{Wheat2}
\non_function{}
\title{Wheat Yield Trials}
\description{
    The \code{Wheat2} data frame has 224 rows and 5 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{Block}{
	    an ordered factor with levels
	    \code{4} < \code{2} < \code{3} < \code{1}
	}
	\item{variety}{
	    a factor with levels
	    \code{ARAPAHOE} 
	    \code{BRULE} 
	    \code{BUCKSKIN} 
	    \code{CENTURA} 
	    \code{CENTURK78} 
	    \code{CHEYENNE} 
	    \code{CODY} 
	    \code{COLT} 
	    \code{GAGE} 
	    \code{HOMESTEAD} 
	    \code{KS831374} 
	    \code{LANCER} 
	    \code{LANCOTA} 
	    \code{NE83404} 
	    \code{NE83406} 
	    \code{NE83407} 
	    \code{NE83432} 
	    \code{NE83498} 
	    \code{NE83T12} 
	    \code{NE84557} 
	    \code{NE85556} 
	    \code{NE85623} 
	    \code{NE86482} 
	    \code{NE86501} 
	    \code{NE86503} 
	    \code{NE86507} 
	    \code{NE86509} 
	    \code{NE86527} 
	    \code{NE86582} 
	    \code{NE86606} 
	    \code{NE86607} 
	    \code{NE86T666} 
	    \code{NE87403} 
	    \code{NE87408} 
	    \code{NE87409} 
	    \code{NE87446} 
	    \code{NE87451} 
	    \code{NE87457} 
	    \code{NE87463} 
	    \code{NE87499} 
	    \code{NE87512} 
	    \code{NE87513} 
	    \code{NE87522} 
	    \code{NE87612} 
	    \code{NE87613} 
	    \code{NE87615} 
	    \code{NE87619} 
	    \code{NE87627} 
	    \code{NORKAN} 
	    \code{REDLAND} 
	    \code{ROUGHRIDER} 
	    \code{SCOUT66} 
	    \code{SIOUXLAND} 
	    \code{TAM107} 
	    \code{TAM200} 
	    \code{VONA} 
	}
	\item{yield}{
	    a numeric vector
	}
	\item{latitude}{
	    a numeric vector
	}
	\item{longitude}{
	    a numeric vector
	}
    }
}
\details{
}
\source{
    Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
	and S-PLUS}, Springer, New York.
}
\examples{
data(Wheat2)
}
\keyword{datasets}

\eof
% $Id: allCoef.Rd,v 1.4.4.1 2003/04/17 22:28:45 bates Exp $
\name{allCoef}
\title{Extract Coefficients from a Set of Objects}
\usage{
allCoef(\dots, extract)
}
\alias{allCoef}
\arguments{
 \item{\dots}{objects to which \code{extract} will be applied. Generally
 these will be model components, such as \code{corStruct} and
 \code{varFunc} objects.}
 \item{extract}{an optional extractor function. Defaults to \code{coef}.}
}
\description{
  The extractor function is applied to each object in \code{\dots}, with
  the result being converted to a vector. A \code{map} attribute is
  included to indicate which pieces of the returned vector correspond to
  the original objects in \code{dots}.
}
\value{
  a vector with all elements, generally coefficients, obtained by
  applying \code{extract} to the objects in \code{\dots}. 
}
\author{Jose' Pinheiro and Douglas Bates}

\seealso{\code{\link{lmeStruct}},\code{\link{nlmeStruct}}}

\examples{
cs1 <- corAR1(0.1)
vf1 <- varPower(0.5)
allCoef(cs1, vf1)
}
\keyword{models}


\eof
% $Id: anova.gls.Rd,v 1.6.2.1 2003/04/17 22:28:45 bates Exp $
\name{anova.gls}
\title{Compare Likelihoods of Fitted Objects}
\usage{
\method{anova}{gls}(object, \dots, test, type, adjustSigma, Terms, L, verbose)
}
\alias{anova.gls}
\arguments{
 \item{object}{a fitted model object inheriting from class \code{gls},
   representing a generalized least squares fit.}
 \item{\dots}{other optional fitted model objects inheriting from
   classes \code{gls}, \code{gnls}, \code{lm}, \code{lme},
   \code{lmList}, \code{nlme}, \code{nlsList}, or \code{nls}.}
 \item{test}{an optional logical value controlling whether likelihood
   ratio tests should be used to compare the fitted models represented
   by \code{object} and the objects in \code{\dots}. Defaults to
   \code{TRUE}.}
 \item{type}{an optional character string specifying the type of sum of
   squares to be used in F-tests for the terms in the model. If 
   \code{"sequential"}, the sequential sum of squares obtained by
   including the terms in the order they appear in the model is used;
   else, if \code{"marginal"}, the marginal sum of squares
   obtained by deleting a term from the model at a time is used. This
   argument is only used when a single fitted object is passed to the
   function. Partial matching of arguments is used, so only the first
   character needs to be provided. Defaults to \code{"sequential"}.}
 \item{adjustSigma}{an optional logical value. If \code{TRUE} and the
   estimation method used to obtain \code{object} was maximum
   likelihood, the residual standard error is multiplied by
   \eqn{\sqrt{n_{obs}/(n_{obs} - n_{par})}}{sqrt(nobs/(nobs - npar))},
   converting it to a REML-like estimate. This argument is only used
   when a single fitted object is passed to the function. Default is
   \code{TRUE}.} 
 \item{Terms}{an optional integer or character vector specifying which
   terms in the model should be jointly tested to be zero using a Wald
   F-test. If given as a character vector, its elements must correspond
   to term names; else, if given as an integer vector, its elements must
   correspond to the order in which terms are included in the
   model. This argument is only used when a single fitted object is
   passed to the function. Default is \code{NULL}.} 
 \item{L}{an optional numeric vector or array specifying linear
   combinations of the coefficients in the model that should be tested
   to be zero. If given as an array, its rows define the linear
   combinations to be tested. If names are assigned to the vector
   elements (array columns), they must correspond to coefficients
   names and will be used to map the linear combination(s) to the
   coefficients; else, if no names are available, the vector elements
   (array columns) are assumed in the same order as the coefficients
   appear in the model. This argument is only used when a single fitted
   object is passed to the function. Default is \code{NULL}.}
 \item{verbose}{an optional logical value. If \code{TRUE}, the calling
   sequences for each fitted model object are printed with the rest of
   the output, being omitted if \code{verbose = FALSE}. Defaults to
   \code{FALSE}.}
}
\description{
  When only one fitted model object is present, a data frame with the
  sums of squares, numerator degrees of freedom, F-values, and P-values
  for Wald tests for the terms in the model (when \code{Terms} and
  \code{L} are \code{NULL}), a combination of model terms (when
  \code{Terms} in not \code{NULL}), or linear combinations of the model
  coefficients (when \code{L} is not \code{NULL}). Otherwise, when
  multiple fitted objects are being compared, a data frame with
  the degrees of freedom, the (restricted) log-likelihood, the 
  Akaike Information Criterion (AIC), and the Bayesian Information
  Criterion (BIC) of each object is returned. If \code{test=TRUE},
  whenever two consecutive  objects have different number of degrees of
  freedom, a likelihood ratio statistic, with the associated p-value is
  included in the returned data frame.   
}
\value{
  a data frame inheriting from class \code{anova.lme}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  Likelihood comparisons are not meaningful for objects fit using
  restricted maximum likelihood and with different fixed effects.
}

\seealso{\code{\link{gls}}, \code{\link{gnls}}, \code{\link{lme}},
  \code{\link{AIC}}, \code{\link{BIC}}, \code{\link{print.anova.lme}} }

\examples{
data(Ovary)
%\dontrun{
# AR(1) errors within each Mare
fm1 <- gls(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), Ovary,
           correlation = corAR1(form = ~ 1 | Mare))
anova(fm1)
# variance changes with a power of the absolute fitted values?
fm2 <- update(fm1, weights = varPower())
anova(fm1, fm2)
%}
}
\keyword{models}

\eof
% $Id: anova.lme.Rd,v 1.6.2.1 2003/04/17 22:28:45 bates Exp $
\name{anova.lme}
\alias{anova.lme}
\alias{print.anova.lme}
\title{Compare Likelihoods of Fitted Objects}
\usage{
\method{anova}{lme}(object, \dots, test, type, adjustSigma, Terms, L, verbose)
\method{print}{anova.lme}(x, verbose, \dots)
}
\arguments{
 \item{object}{a fitted model object inheriting from class \code{lme},
   representing a mixed-effects model.}
 \item{\dots}{other optional fitted model objects inheriting from
   classes \code{gls}, \code{gnls}, \code{lm}, \code{lme},
   \code{lmList}, \code{nlme}, \code{nlsList}, or \code{nls}.}
 \item{test}{an optional logical value controlling whether likelihood
   ratio tests should be used to compare the fitted models represented
   by \code{object} and the objects in \code{\dots}. Defaults to
   \code{TRUE}.}
 \item{type}{an optional character string specifying the type of sum of
   squares to be used in F-tests for the terms in the model. If 
   \code{"sequential"}, the sequential sum of squares obtained by
   including the terms in the order they appear in the model is used;
   else, if \code{"marginal"}, the marginal sum of squares
   obtained by deleting a term from the model at a time is used. This
   argument is only used when a single fitted object is passed to the
   function. Partial matching of arguments is used, so only the first
   character needs to be provided. Defaults to \code{"sequential"}.}
 \item{adjustSigma}{an optional logical value. If \code{TRUE} and the
   estimation method used to obtain \code{object} was maximum
   likelihood, the residual standard error is multiplied by
   \eqn{\sqrt{n_{obs}/(n_{obs} - n_{par})}}{sqrt(nobs/(nobs - npar))},
   converting it to a REML-like estimate. This argument is only used
   when a single fitted object is passed to the function. Default is
   \code{TRUE}.} 
 \item{Terms}{an optional integer or character vector specifying which
   terms in the model should be jointly tested to be zero using a Wald
   F-test. If given as a character vector, its elements must correspond
   to term names; else, if given as an integer vector, its elements must
   correspond to the order in which terms are included in the
   model. This argument is only used when a single fitted object is
   passed to the function. Default is \code{NULL}.} 
 \item{L}{an optional numeric vector or array specifying linear
   combinations of the coefficients in the model that should be tested
   to be zero. If given as an array, its rows define the linear
   combinations to be tested. If names are assigned to the vector
   elements (array columns), they must correspond to coefficients
   names and will be used to map the linear combination(s) to the
   coefficients; else, if no names are available, the vector elements
   (array columns) are assumed in the same order as the coefficients
   appear in the model. This argument is only used when a single fitted
   object is passed to the function. Default is \code{NULL}.}
 \item{x}{an object inheriting from class \code{anova.lme}}
 \item{verbose}{an optional logical value. If \code{TRUE}, the calling
   sequences for each fitted model object are printed with the rest of
   the output, being omitted if \code{verbose = FALSE}. Defaults to
   \code{FALSE}.}
}
\description{
  When only one fitted model object is present, a data frame with the
  sums of squares, numerator degrees of freedom, denominator degrees of
  freedom, F-values, and P-values for Wald tests for the terms in the
  model (when \code{Terms} and \code{L} are \code{NULL}), a combination
  of model terms (when \code{Terms} in not \code{NULL}), or linear
  combinations of the model coefficients (when \code{L} is not
  \code{NULL}). Otherwise, when multiple fitted objects are being
  compared, a data frame with the degrees of freedom, the (restricted)
  log-likelihood, the Akaike Information Criterion (AIC), and the
  Bayesian Information Criterion (BIC) of each object is returned. If
  \code{test=TRUE}, whenever two consecutive  objects have different
  number of degrees of freedom, a likelihood ratio statistic, with the
  associated p-value is included in the returned data frame.   
}
\value{
  a data frame inheriting from class \code{anova.lme}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  Likelihood comparisons are not meaningful for objects fit using
  restricted maximum likelihood and with different fixed effects.
}

\seealso{\code{\link{gls}}, \code{\link{gnls}}, \code{\link{nlme}},
  \code{\link{lme}}, \code{\link{AIC}}, \code{\link{BIC}},
  \code{\link{print.anova.lme}} }  

\examples{
data(Orthodont)
fm1 <- lme(distance ~ age, Orthodont, random = ~ age | Subject)
anova(fm1)
fm2 <- update(fm1, random = pdDiag(~age))
anova(fm1, fm2)
}
\keyword{models}

\eof
% $Id: as.matrix.corStruct.Rd,v 1.6.2.1 2002/08/09 19:45:29 bates Exp $
\name{as.matrix.corStruct}
\title{Matrix of a corStruct Object}
\usage{
\method{as.matrix}{corStruct}(x)
}
\alias{as.matrix.corStruct}
\arguments{
 \item{x}{an object inheriting from class \code{corStruct},
   representing a correlation structure.}
}
\description{
  This method function extracts the correlation matrix, or list of
  correlation matrices, associated with \code{object}.
}
\value{
  If the correlation structure includes a grouping factor, the returned
  value will be a list with components given by the correlation
  matrices for each group. Otherwise, the returned value will be a
  matrix representing the correlation structure associated with
  \code{object}. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{corClasses}}, \code{\link{corMatrix}}}
\examples{
data(Orthodont)
cst1 <- corAR1(form = ~1|Subject)
cst1 <- Initialize(cst1, data = Orthodont)
as.matrix(cst1)
}
\keyword{models}

\eof
% $Id: as.matrix.pdMat.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{as.matrix.pdMat}
\title{Matrix of a pdMat Object}
\usage{
\method{as.matrix}{pdMat}(x)
}
\alias{as.matrix.pdMat}
\arguments{
 \item{x}{an object inheriting from class \code{pdMat}, representing a
   positive-definite matrix.}
}
\description{
  This method function extracts the positive-definite matrix represented
  by \code{x}.
}
\value{
  a matrix corresponding to the positive-definite matrix represented by
  \code{x}.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{pdMat}}, \code{\link{corMatrix}}}

\examples{
as.matrix(pdSymm(diag(4)))
}

\keyword{models}

\eof
% $Id: as.matrix.reStruct.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{as.matrix.reStruct}
\title{Matrices of an reStruct Object}
\usage{
\method{as.matrix}{reStruct}(x)
}
\alias{as.matrix.reStruct}
\arguments{
 \item{x}{an object inheriting from class \code{reStruct}, representing
   a random effects structure and consisting of a list of \code{pdMat}
   objects.}
}
\description{
  This method function extracts the positive-definite matrices
  corresponding to the \code{pdMat} elements of \code{object}. 
}
\value{
  a list with components given by the positive-definite matrices
  corresponding to the elements of \code{object}.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{as.matrix.pdMat}}, \code{\link{reStruct}},
  \code{\link{pdMat}}}

\examples{
data(Orthodont)
rs1 <- reStruct(pdSymm(diag(3), ~age+Sex, data = Orthodont))
as.matrix(rs1)
}
\keyword{models}

\eof
% $Id: asOneFormula.Rd,v 1.5.2.1 2003/04/17 22:28:45 bates Exp $
\name{asOneFormula}
\title{Combine Formulas of a Set of Objects}
\usage{
asOneFormula(\dots, omit)
}
\alias{asOneFormula}
\arguments{
 \item{\dots}{objects, or lists of objects, from which a formula can be
   extracted.}
 \item{omit}{an optional character vector with the names of variables to
   be omitted from the returned formula. Defaults to c(".", "pi").}
}
\description{
  The names of all variables used in the formulas extracted from the
  objects defined in \code{\dots} are converted into a single linear
  formula, with the variables names separated by \code{+}. 
}
\value{
  a one-sided linear formula with all variables named in the formulas
  extracted from the objects in \code{\dots}, except the ones listed in
  \code{omit}. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{formula}}, \code{\link{all.vars}}}

\examples{
asOneFormula(y ~ x + z | g, list(~ w, ~ t * sin(2 * pi)))
}
\keyword{models}

\eof
% $Id: asTable.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{asTable}
\alias{asTable}
\alias{asTable.groupedData}
\title{Convert groupedData to a matrix}
\usage{
asTable(object)
}
\arguments{
 \item{object}{A balanced \code{groupedData} object}
}
\description{
  Create a tabular representation of the response in a balanced
  groupedData object.
}
\details{
  A balanced groupedData object can be represented as a matrix or table
  of response values corresponding to the values of a primary covariate
  for each level of a grouping factor.  This function creates such a
  matrix representation of the data in \code{object}.
}
\value{
  A matrix.  The data in the matrix are the values of the response.  The
  columns correspond to the distinct values of the primary covariate and
  are labelled as such.  The rows correspond to the distinct levels of
  the grouping factor and are labelled as such.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{groupedData}}, \code{\link{isBalanced}},
  \code{\link{balancedGrouped}}}
\examples{
data(Orthodont)
asTable(Orthodont)
}
\keyword{manip}

\eof
% $Id: augPred.Rd,v 1.7.2.1 2003/04/17 22:28:45 bates Exp $
\name{augPred}
\title{Augmented Predictions}
\usage{
augPred(object, primary, minimum, maximum, length.out, \dots)
}
\alias{augPred}
\alias{augPred.gls}
\alias{augPred.lme}
\alias{augPred.lmList}
\arguments{
 \item{object}{a fitted model object from which predictions can be
   extracted, using a \code{predict} method.}
 \item{primary}{an optional one-sided formula specifying the primary
   covariate to be used to generate the augmented predictions. By
   default, if a  covariate can be extracted from the data used to generate
   \code{object} (using \code{getCovariate}), it will be used as
   \code{primary}.}
 \item{minimum}{an optional lower limit for the primary
   covariate. Defaults to \code{min(primary)}.}
 \item{maximum}{an optional upper limit for the primary
   covariate. Defaults to \code{max(primary)}.}
 \item{length.out}{an optional integer with the number of primary
   covariate values at which to evaluate the predictions. Defaults to
   51.}
%   \item{level}{an optional integer vector specifying the desired
%     prediction levels. Levels increase from outermost to innermost
%     grouping, with level 0 representing the population (fixed effects)
%     predictions. Defaults to the innermost level.}
 \item{\dots}{some methods for the generic may require additional
   arguments.}
}
\description{
  Predicted values are obtained at the specified values of
  \code{primary}. If \code{object} has a grouping structure
  (i.e. \code{getGroups(object)} is not \code{NULL}), predicted values
  are obtained for each group. If \code{level} has more than one
  element, predictions are obtained for each level of the
  \code{max(level)} grouping factor. If other covariates besides
  \code{primary} are used in the prediction model, their average
  (numeric covariates) or most frequent value (categorical covariates)
  are used to obtain the predicted values. The original observations are
  also included in the returned object. 
}
\value{
  a data frame with four columns representing, respectively, the values
  of the primary covariate, the groups (if \code{object} does not have a
  grouping structure, all elements will be \code{1}), the predicted or
  observed values, and the type of value in the third column:
  \code{original} for the observed values and \code{predicted} (single
  or no grouping factor) or \code{predict.groupVar} (multiple levels of
  grouping), with \code{groupVar} replaced by the actual grouping
  variable name (\code{fixed} is used for population predictions). The
  returned object inherits from class \code{augPred}.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include: \code{gls}, \code{lme}, and \code{lmList}.
}
\seealso{\code{\link{plot.augPred}}, \code{\link{getGroups}},
  \code{\link{predict}}} 

\examples{
data(Orthodont)
fm1 <- lme(Orthodont, random = ~1)
augPred(fm1, length.out = 2, level = c(0,1))
}
\keyword{models}

\eof
% $Id: balancedGrouped.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{balancedGrouped}
\alias{balancedGrouped}
\title{Create a groupedData object from a matrix}
\usage{
balancedGrouped(form, data, labels=NULL, units=NULL)
}
\arguments{
 \item{form}{A formula of the form \code{y ~ x | g} giving the name of
   the response, the primary covariate, and the grouping factor.}
 \item{data}{A matrix or data frame containing the values of the
   response grouped according to the levels of the grouping factor
   (rows) and the distinct levels of the primary covariate (columns).
   The \code{dimnames} of the matrix are used to construct the levels of
   the grouping factor and the primary covariate.}
 \item{labels}{an optional list of character strings giving labels for
   the response and the primary covariate.  The label for the primary
   covariate is named \code{x} and that for the response is named
   \code{y}.  Either label can be omitted.}
 \item{units}{an optional list of character strings giving the units for
   the response and the primary covariate.  The units string for the
   primary covariate is named \code{x} and that for the response is
   named \code{y}.  Either units string can be omitted.}
}
\description{
  Create a \code{groupedData} object from a data matrix.  This function
  can be used only with balanced data.  The opposite conversion, from a
  \code{groupedData} object to a \code{matrix}, is done with \code{asTable}.
}
\value{
  A balanced \code{groupedData} object.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{groupedData}}, \code{\link{isBalanced}}, \code{\link{asTable}}}
\examples{
data(Orthodont)
OrthoMat <- asTable( Orthodont )
Orth2 <- balancedGrouped(distance ~ age | Subject, data = OrthoMat,
    labels = list(x = "Age",
                  y = "Distance from pituitary to pterygomaxillary fissure"),
    units = list(x = "(yr)", y = "(mm)"))
Orth2[ 1:10, ]        ## check the first few entries
}
\keyword{data}

\eof
\name{bdf}
\alias{bdf}
\non_function{}
\title{Language scores}
\usage{data(bdf)}
\description{
The \code{bdf} data frame has 2287 rows and 25 columns of language
scores from grade 8 pupils in elementary schools in The Netherlands.
}
\format{
  \describe{
    \item{schoolNR}{a factor denoting the school.}
    \item{pupilNR}{a factor denoting the pupil.}
    \item{IQ.verb}{a numeric vector of verbal IQ scores}
    \item{IQ.perf}{a numeric vector of IQ scores.}
    \item{sex}{Sex of the student.}
    \item{Minority}{a factor indicating if the student is a member of a
      minority group.}
    \item{repeatgr}{an ordered factor indicating if one or more grades
      have been repeated.}
    \item{aritPRET}{a numeric vector}
    \item{classNR}{a numeric vector}
    \item{aritPOST}{a numeric vector}
    \item{langPRET}{a numeric vector}
    \item{langPOST}{a numeric vector}
    \item{ses}{a numeric vector of socioeconomic status indicators.}
    \item{denomina}{a factor indicating of the school is a public
      school, a Protestant private school, a Catholic private school, or
      a non-denominational private school.}
    \item{schoolSES}{a numeric vector}
    \item{satiprin}{a numeric vector}
    \item{natitest}{a factor with levels \code{0} and \code{1}}
    \item{meetings}{a numeric vector}
    \item{currmeet}{a numeric vector}
    \item{mixedgra}{a factor indicating if the class is a mixed-grade class.}
    \item{percmino}{a numeric vector}
    \item{aritdiff}{a numeric vector}
    \item{homework}{a numeric vector}
    \item{classsiz}{a numeric vector}
    \item{groupsiz}{a numeric vector}
  }
}
\source{
  \url{http://stat.gamma.rug.nl/snijders/multilevel.htm}
}
\references{
  Snijders, Tom and Bosker, Roel (1999), \emph{Multilevel Analysis: An
    Introduction to Basic and Advanced Multilevel Modeling}, Sage.
}
\examples{
data(bdf)
summary(bdf)
}
\keyword{datasets}

\eof
% $Id: coef.corStruct.Rd,v 1.6.2.3 2003/08/09 22:45:17 bates Exp $
\name{coef.corStruct}
\title{Coefficients of a corStruct Object}
\usage{
\method{coef}{corStruct}(object, unconstrained, \dots)
\method{coef}{corStruct}(object, \dots) <- value
}
\alias{coef.corStruct}
\alias{coef.corAR1}
\alias{coef.corARMAd}   
\alias{coef.corCAR1}   
\alias{coef.corCompSymm}
\alias{coef.corHF}
\alias{coef.corIdent}  
\alias{coef.corLin}
\alias{coef.corNatural}
\alias{coef.corSpatial}
\alias{coef.corSpher}
\alias{coef.corSymm}
\alias{coef<-.corStruct}
\alias{coef<-.corAR1}
\alias{coef<-.corARMA}   
\alias{coef<-.corCAR1}   
\alias{coef<-.corCompSymm}
\alias{coef<-.corNatural}
\alias{coef<-.corHF}
\alias{coef<-.corIdent}  
\alias{coef<-.corLin}
\alias{coef<-.corSpatial}
\alias{coef<-.corSpher}
\alias{coef<-.corSymm}
\alias{coef.summary.nlsList}

\arguments{
 \item{object}{an object inheriting from class \code{corStruct},
   representing a correlation structure.}
 \item{unconstrained}{a logical value. If \code{TRUE} the coefficients
   are returned in unconstrained form (the same used in the optimization
   algorithm). If \code{FALSE} the coefficients are returned in
   "natural", possibly constrained, form. Defaults to \code{TRUE}.}
 \item{value}{a vector with the replacement values for the coefficients
   associated with \code{object}. It must be a vector with the same length
   of \code{coef{object}} and must be given in unconstrained form.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts the coefficients associated with the
  correlation structure represented by \code{object}. 
}
\value{
  a vector with the coefficients corresponding to \code{object}.
}
\author{Jose Pinheiro and Douglas Bates }
\section{SIDE EFFECTS}{
  On the left side of an assignment, sets the values of the coefficients
  of \code{object} to \code{value}. \code{Object} must be initialized (using
  \code{Initialize}) before new values can be assigned to its
  coefficients. 
}

\seealso{\code{\link{corAR1}}, \code{\link{corARMA}},
    \code{\link{corCAR1}}, \code{\link{corCompSymm}},
    \code{\link{corExp}}, \code{\link{corGaus}}, \code{\link{corLin}},
    \code{\link{corRatio}}, \code{\link{corSpatial}}, \code{\link{corSpher}},
    \code{\link{corSymm}},\code{\link{Initialize}}}
\examples{
cst1 <- corARMA(p = 1, q = 1)
coef(cst1)
}
\keyword{models}

\eof
% $Id: coef.gnls.Rd,v 1.8 2002/03/05 14:59:39 bates Exp $
\name{coef.gnls}
\title{Extract gnls Coefficients}
\usage{
\method{coef}{gnls}(object, \dots)
}
\alias{coef.gnls}
\arguments{
 \item{object}{an object inheriting from class \code{gnls}, representing
   a generalized nonlinear least squares fitted model.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The estimated coefficients for the nonlinear model represented by
  \code{object} are extracted. 
}
\value{
  a vector with the estimated coefficients for the nonlinear model
  represented by \code{object}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gnls}}} 

\examples{
data(Soybean)
fm1 <- gnls(weight ~ SSlogis(Time, Asym, xmid, scal), Soybean,
            weights = varPower())
coef(fm1)
}
\keyword{models}

\eof
% $Id: coef.lmList.Rd,v 1.7.2.1 2003/04/17 22:28:45 bates Exp $
\name{coef.lmList}
\title{Extract lmList Coefficients}
\usage{
\method{coef}{lmList}(object, augFrame, data, which, FUN,
   omitGroupingFactor, \dots)
}
\alias{coef.lmList}
\arguments{
 \item{object}{an object inheriting from class \code{lmList}, representing
   a list of \code{lm} objects with a common model.
 }
 \item{augFrame}{an optional logical value. If \code{TRUE}, the returned
   data frame is augmented with variables defined in the data frame used
   to produce \code{object}; else, if \code{FALSE}, only the coefficients
   are returned. Defaults to \code{FALSE}.
 }
 \item{data}{an optional data frame with the variables to be used for
   augmenting the returned data frame when \code{augFrame =
     TRUE}. Defaults to the data frame used to fit \code{object}.}
 \item{which}{an optional positive integer or character vector
   specifying which columns of the data frame used to produce \code{object}
   should be used in the augmentation of the returned data
   frame. Defaults to all variables in the data.
 }
 \item{FUN}{an optional summary function or a list of summary functions
   to be applied to group-varying variables, when collapsing the data
   by groups.  Group-invariant variables are always summarized by the
   unique value that they assume within that group. If \code{FUN} is a
   single function it will be applied to each non-invariant variable by
   group to produce the summary for that variable.  If \code{FUN} is a
   list of functions, the names in the list should designate classes of
   variables in the frame such as \code{ordered}, \code{factor}, or
   \code{numeric}.  The indicated function will be applied to any
   group-varying variables of that class.  The default functions to be
   used are \code{mean} for numeric factors, and \code{Mode} for both
   \code{factor} and \code{ordered}.  The \code{Mode} function, defined
   internally in \code{gsummary}, returns the modal or most popular
   value of the variable.  It is different from the \code{mode} function
   that returns the S-language mode of the variable.
 }
 \item{omitGroupingFactor}{an optional logical value.  When \code{TRUE}
   the grouping factor itself will be omitted from the group-wise
   summary of \code{data} but the levels of the grouping factor will
   continue to be used as the row names for the returned data frame.
   Defaults to \code{FALSE}.
 }
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The coefficients of each \code{lm} object in the \code{object} list are
  extracted and organized into a data frame, with rows corresponding to
  the \code{lm} components and columns corresponding to the
  coefficients.  Optionally, the returned data frame may be augmented
  with covariates summarized over the groups associated with the
  \code{lm} components.  
}
\value{
  a data frame inheriting from class \code{coef.lmList} with the estimated
  coefficients for each \code{lm} component of \code{object} and,
  optionally, other covariates summarized over the groups corresponding
  to the \code{lm} components. The returned object also inherits from
  classes \code{ranef.lmList} and \code{data.frame}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lmList}}, \code{\link{fixed.effects.lmList}},
  \code{\link{ranef.lmList}},
  \code{\link{plot.ranef.lmList}}, \code{\link{gsummary}}} 

\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age|Subject, data = Orthodont)
coef(fm1)
coef(fm1, augFrame = TRUE)
}
\keyword{models}

\eof
% $Id: coef.lme.Rd,v 1.9.2.1 2003/05/17 21:50:21 bates Exp $
\name{coef.lme}
\title{Extract lme Coefficients}
\usage{
\method{coef}{lme}(object, augFrame, level, data, which, FUN, 
       omitGroupingFactor, subset, \dots)
}
\alias{coef.lme}
\arguments{
 \item{object}{an object inheriting from class \code{lme}, representing
   a fitted linear mixed-effects model.}
 \item{augFrame}{an optional logical value. If \code{TRUE}, the returned
   data frame is augmented with variables defined in \code{data}; else,
   if \code{FALSE}, only the coefficients are returned. Defaults to
   \code{FALSE}.}
 \item{level}{an optional positive integer giving the level of grouping
   to be used in extracting the coefficients from an object with
   multiple nested grouping levels. Defaults to the highest or innermost
   level of grouping.} 
 \item{data}{an optional data frame with the variables to be used for
   augmenting the returned data frame when \code{augFrame =
     TRUE}. Defaults to the data frame used to fit \code{object}.}
 \item{which}{an optional positive integer or character vector
   specifying which columns of \code{data} should be used in the
   augmentation of the returned data frame. Defaults to all columns in
   \code{data}.} 
 \item{FUN}{an optional summary function or a list of summary functions
   to be applied to group-varying variables, when collapsing \code{data}
   by groups.  Group-invariant variables are always summarized by the
   unique value that they assume within that group. If \code{FUN} is a
   single function it will be applied to each non-invariant variable by
   group to produce the summary for that variable.  If \code{FUN} is a
   list of functions, the names in the list should designate classes of
   variables in the frame such as \code{ordered}, \code{factor}, or
   \code{numeric}.  The indicated function will be applied to any
   group-varying variables of that class.  The default functions to be
   used are \code{mean} for numeric factors, and \code{Mode} for both
   \code{factor} and \code{ordered}.  The \code{Mode} function, defined
   internally in \code{gsummary}, returns the modal or most popular
   value of the variable.  It is different from the \code{mode} function
   that returns the S-language mode of the variable.}
 \item{omitGroupingFactor}{an optional logical value.  When \code{TRUE}
   the grouping factor itself will be omitted from the group-wise
   summary of \code{data} but the levels of the grouping factor will
   continue to be used as the row names for the returned data frame.
   Defaults to \code{FALSE}.}
 \item{subset}{an optional expression specifying a subset}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The estimated coefficients at level \eqn{i} are obtained by adding
  together the fixed effects estimates and the corresponding random
  effects estimates at grouping levels less or equal to \eqn{i}. The
  resulting estimates are returned as a data frame, with rows
  corresponding to groups and columns to coefficients. Optionally, the
  returned data frame may be augmented with covariates summarized over
  groups.  
}
\value{
  a data frame inheriting from class \code{coef.lme} with the estimated
  coefficients at level \code{level} and, optionally, other covariates
  summarized over groups. The returned object also inherits from classes
  \code{ranef.lme} and \code{data.frame}.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{lme}}, %\code{\link{fixef.lme}},
  \code{\link{ranef.lme}},
  \code{\link{plot.ranef.lme}}, \code{\link{gsummary}}
}
\examples{
data(Orthodont)
fm1 <- lme(distance ~ age, Orthodont, random = ~ age | Subject)
coef(fm1)
coef(fm1, augFrame = TRUE)
}
\keyword{models}

\eof
% $Id: coef.modelStruct.Rd,v 1.7.2.3 2003/08/09 22:45:17 bates Exp $
\name{coef.modelStruct}
\title{Extract modelStruct Object Coefficients}
\usage{
\method{coef}{modelStruct}(object, unconstrained, \dots)
\method{coef}{modelStruct}(object, \dots) <- value
}
\alias{coef.modelStruct}
\alias{coef<-.modelStruct}
\arguments{
 \item{object}{an object inheriting from class \code{modelStruct},
   representing a list of model components, such as \code{corStruct} and
   \code{varFunc} objects.}
 \item{unconstrained}{a logical value. If \code{TRUE} the coefficients
   are returned in unconstrained form (the same used in the optimization
   algorithm). If \code{FALSE} the coefficients are returned in
   "natural", possibly constrained, form. Defaults to \code{TRUE}.}
 \item{value}{a vector with the replacement values for the coefficients
   associated with \code{object}. It must be a vector with the same length
   of \code{coef{object}} and must be given in unconstrained form.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts the coefficients associated with each
  component of the \code{modelStruct} list.
}
\value{
  a vector with all coefficients corresponding to the components of
  \code{object}.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\section{SIDE EFFECTS}{
  On the left side of an assignment, sets the values of the coefficients
  of \code{object} to \code{value}. \code{Object} must be initialized (using
  \code{Initialize}) before new values can be assigned to its
  coefficients. 
}

\seealso{\code{\link{Initialize}}}

\examples{
lms1 <- lmeStruct(reStruct = reStruct(pdDiag(diag(2), ~age)),
   corStruct = corAR1(0.3))
coef(lms1)
}
\keyword{models}

\eof
% $Id: coef.pdMat.Rd,v 1.6.2.3 2003/08/09 22:45:17 bates Exp $
\name{coef.pdMat}
\title{pdMat Object Coefficients}
\usage{
\method{coef}{pdMat}(object, unconstrained, \dots)
\method{coef}{pdMat}(object, \dots) <- value
}
\alias{coef.pdMat}
\alias{coef.pdBlocked}
\alias{coef.pdCompSymm}
\alias{coef.pdDiag}
\alias{coef.pdIdent}
\alias{coef.pdNatural}
\alias{coef.pdSymm}
\alias{coef<-.pdMat}
\alias{coef<-.pdBlocked}
\arguments{
 \item{object}{an object inheriting from class \code{pdMat},
   representing a positive-definite matrix.}
 \item{unconstrained}{a logical value. If \code{TRUE} the coefficients
   are returned in unconstrained form (the same used in the optimization
   algorithm). If \code{FALSE} the upper triangular elements of the
   positive-definite matrix represented by \code{object} are
   returned. Defaults to \code{TRUE}.}
 \item{value}{a vector with the replacement values for the coefficients
   associated with \code{object}. It must be a vector with the same length
   of \code{coef{object}} and must be given in unconstrained form.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts the coefficients associated with the
  positive-definite matrix represented by \code{object}. 
}
\value{
  a vector with the coefficients corresponding to \code{object}.
}
\references{
 Pinheiro, J.C. and Bates., D.M.  (1996) "Unconstrained Parametrizations
 for Variance-Covariance Matrices", Statistics and Computing, 6, 289-296.
}
\author{Jose Pinheiro and Douglas Bates }
\section{SIDE EFFECTS}{
  On the left side of an assignment, sets the values of the coefficients
  of \code{object} to \code{value}. 
}

\seealso{\code{\link{pdMat}}}

\examples{
coef(pdSymm(diag(3)))
}
\keyword{models}

\eof
% $Id: coef.reStruct.Rd,v 1.7.2.2 2003/08/09 22:45:17 bates Exp $
\name{coef.reStruct}
\title{reStruct Object Coefficients}
\usage{
\method{coef}{reStruct}(object, unconstrained, \dots)
\method{coef}{reStruct}(object, \dots) <- value
}
\alias{coef.reStruct}
\alias{coef<-.reStruct}
\arguments{
 \item{object}{an object inheriting from class \code{reStruct},
   representing a random effects structure and consisting of a list of
   \code{pdMat} objects.} 
 \item{unconstrained}{a logical value. If \code{TRUE} the coefficients
   are returned in unconstrained form (the same used in the optimization
   algorithm). If \code{FALSE} the coefficients are returned in
   "natural", possibly constrained, form. Defaults to \code{TRUE}.}
 \item{value}{a vector with the replacement values for the coefficients
   associated with \code{object}. It must be a vector with the same length
   of \code{coef(object)} and must be given in unconstrained form.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts the coefficients associated with the
  positive-definite matrix represented by \code{object}. 
}
\value{
  a vector with the coefficients corresponding to \code{object}.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\section{SIDE EFFECTS}{
  On the left side of an assignment, sets the values of the coefficients
  of \code{object} to \code{value}. 
}

\seealso{\code{\link{coef.pdMat}}, \code{\link{reStruct}},
  \code{\link{pdMat}}} 

\examples{
rs1 <- reStruct(list(A = pdSymm(diag(1:3), form = ~Score),
  B = pdDiag(2 * diag(4), form = ~Educ)))
coef(rs1)
}
\keyword{models}

\eof
% $Id: coef.varFunc.Rd,v 1.6.2.2 2003/08/09 22:45:17 bates Exp $
\name{coef.varFunc}
\title{varFunc Object Coefficients}
\usage{
\method{coef}{varFunc}(object, unconstrained, allCoef, \dots)
\method{coef}{varIdent}(object, \dots) <- value
}
\alias{coef.varFunc}
\alias{coef.varComb}
\alias{coef.varConstPower}
\alias{coef.varExp}
\alias{coef.varFixed}
\alias{coef.varIdent}
\alias{coef.varPower}
\alias{coef<-.varComb}
\alias{coef<-.varConstPower}
\alias{coef<-.varExp}
\alias{coef<-.varFixed}
\alias{coef<-.varIdent}
\alias{coef<-.varPower}
\arguments{
 \item{object}{an object inheriting from class \code{varFunc}
   representing a variance function structure.}
 \item{unconstrained}{a logical value. If \code{TRUE} the coefficients
   are returned in unconstrained form (the same used in the optimization
   algorithm). If \code{FALSE} the coefficients are returned in
   "natural", generally constrained form. Defaults to \code{TRUE}.}
 \item{allCoef}{a logical value. If \code{FALSE} only the coefficients
   which may vary during the optimization are returned. If \code{TRUE}
   all coefficients are returned. Defaults to \code{FALSE}.}
 \item{value}{a vector with the replacement values for the coefficients
   associated with \code{object}. It must be have the same length of
   \code{coef{object}} and must be given in unconstrained
   form. \code{Object} must be initialized before new values can be
   assigned to its coefficients.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts the coefficients associated with the
  variance function structure represented by \code{object}. 
}
\value{
  a vector with the coefficients corresponding to \code{object}.
}
\author{Jose Pinheiro and Douglas Bates }
\section{SIDE EFFECTS}{
  On the left side of an assignment, sets the values of the coefficients
  of \code{object} to \code{value}.
}
\seealso{\code{\link{varFunc}}}

\examples{
vf1 <- varPower(1)
coef(vf1)
%\dontrun{
coef(vf1) <- 2
%}
}
\keyword{models}

\eof
% $Id: collapse.Rd,v 1.4.2.1 2003/04/17 22:28:45 bates Exp $
\name{collapse}
\title{Collapse According to Groups}
\usage{
collapse(object, \dots)
}
\alias{collapse}
\arguments{
 \item{object}{an object to be collapsed, usually a data frame.}
 \item{\dots}{some methods for the generic may require additional
   arguments.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Currently, only a \code{groupedData}
  method is available. 
}
\value{
  will depend on the method function used; see the appropriate documentation.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{collapse.groupedData}}}

\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: collapse.groupedData.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{collapse.groupedData}
\title{Collapse a groupedData Object}
\usage{
\method{collapse}{groupedData}(object, collapseLevel, displayLevel,
       outer, inner, preserve, FUN, subset, \dots)
}
\alias{collapse.groupedData}
\arguments{
 \item{object}{an object inheriting from class \code{groupedData},
   generally with multiple grouping factors.
 }
 \item{collapseLevel}{an optional positive integer or character string
   indicating the grouping level to use when collapsing the data. Level
   values increase from outermost to innermost grouping. Default is the
   highest or innermost level of grouping.
 }
 \item{displayLevel}{an optional positive integer or character string
   indicating the grouping level to use as the grouping factor for the
   collapsed data. Default is \code{collapseLevel}.
 }
 \item{outer}{an optional logical value or one-sided formula,
   indicating covariates that are outer to the \code{displayLevel}
   grouping factor. If equal to \code{TRUE}, the \code{displayLevel}
   element \code{attr(object, "outer")} is used to indicate the 
   outer covariates. An outer covariate is invariant within the sets
   of rows defined by the grouping factor.  Ordering of the groups is
   done in such a way as to preserve adjacency of groups with the same
   value of the outer variables. Defaults to \code{NULL}, meaning that
   no outer covariates are to be used.
 }  
 \item{inner}{an optional logical value or one-sided formula, indicating
   a covariate that is inner to the \code{displayLevel} grouping
   factor. If equal to \code{TRUE}, \code{attr(object, "outer")} is used
   to indicate the inner covariate. An inner covariate can change within
   the sets of rows defined by the grouping  factor. Defaults to
   \code{NULL}, meaning that no inner covariate is present.  
 } 
 \item{preserve}{an optional one-sided formula indicating a covariate
   whose levels should be preserved when collapsing the data according
   to the \code{collapseLevel} grouping factor. The collapsing factor is
   obtained by pasting together the levels of the \code{collapseLevel}
   grouping factor and the values of the covariate to be
   preserved. Default is \code{NULL}, meaning that no covariates need to
   be preserved.
 }
  \item{FUN}{an optional summary function or a list of summary functions
   to be used for collapsing the data.  The function or functions are
   applied only to variables in \code{object} that vary within the
   groups defined by \code{collapseLevel}.  Invariant variables are 
   always summarized by group using the unique value that they assume
   within that group.  If \code{FUN} is a single
   function it will be applied to each non-invariant variable by group
   to produce the summary for that variable.  If \code{FUN} is a list of
   functions, the names in the list should designate classes of
   variables in the data such as \code{ordered}, \code{factor}, or
   \code{numeric}.  The indicated function will be applied to any
   non-invariant variables of that class.  The default functions to be
   used are \code{mean} for numeric factors, and \code{Mode} for both
   \code{factor} and \code{ordered}.  The \code{Mode} function, defined
   internally in \code{gsummary}, returns the modal or most popular
   value of the variable.  It is different from the \code{mode} function
   that returns the S-language mode of the variable.}
 \item{subset}{an optional named list. Names can be either positive
   integers representing grouping levels, or names of grouping
   factors. Each element in the list is a vector indicating the levels
   of the corresponding grouping factor to be preserved in the collapsed
   data. Default is \code{NULL}, meaning that all levels are
   used.
 } 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
 If \code{object} has a single grouping factor, it is returned
 unchanged. Else, it is summarized by the values of the
 \code{displayLevel} grouping factor (or the combination of its values
 and the values of the covariate indicated in \code{preserve}, if any is
 present). The collapsed data is used to produce a new
 \code{groupedData} object, with grouping factor given by the
 \code{displayLevel} factor. 
}
\value{
  a \code{groupedData} object with a single grouping factor given by the
  \code{displayLevel} grouping factor, resulting from collapsing
  \code{object} over the levels of the \code{collapseLevel} grouping
  factor. 
}

\references{
Pinheiro, J.C. and Bates, D.M. (1997) "Future Directions in
Mixed-Effects Software: Design of NLME 3.0" available at
http://franz.stat.wisc.edu/pub/NLME.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{groupedData}}, \code{\link{plot.nmGroupedData}}
}  

\examples{
data(Pixel)
# collapsing by Dog
collapse(Pixel, collapse = 1)  # same as collapse(Pixel, collapse = "Dog")
}
\keyword{models}

\eof
% $Id: compareFits.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{compareFits}
\title{Compare Fitted Objects}
\usage{
compareFits(object1, object2, which)
}
\alias{compareFits}
\alias{print.compareFits}
\arguments{
 \item{object1,object2}{data frames, or matrices, with the same
   row names, but possibly different column names. These will usually
   correspond to coefficients from fitted objects with a grouping
   structure (e.g. \code{lme} and \code{lmList} objects).}
 \item{which}{an optional integer or character vector indicating which
   columns in \code{object1} and \code{object2} are to be used in the
   returned object. Defaults to all columns.}
}
\description{
  The columns in \code{object1} and \code{object2} are put together in
  matrices which allow direct comparison of the individual elements for
  each object. Missing columns in either object are replaced by
  \code{NA}s.  
}
\value{
 a three-dimensional array, with the third dimension given by the number
 of unique column names in either \code{object1} or \code{object2}. To
 each column name there corresponds a matrix with as many rows as the
 rows in \code{object1} and two columns, corresponding to \code{object1}
 and \code{object2}. The returned object inherits from class
 \code{compareFits}. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{
  \code{\link{plot.compareFits}},
  \code{\link{pairs.compareFits}},
  \code{\link{comparePred}},
  \code{\link{coef}},
  \code{\link{random.effects}}
}
\examples{
data(Orthodont)
fm1 <- lmList(Orthodont)
fm2 <- lme(fm1)
compareFits(coef(fm1), coef(fm2))
}
\keyword{models}

\eof
% $Id: comparePred.Rd,v 1.5.2.1 2003/04/17 22:28:45 bates Exp $
\name{comparePred}
\title{Compare Predictions}
\usage{
comparePred(object1, object2, primary, minimum, maximum,
    length.out, level, \dots)
}
\alias{comparePred}
\alias{comparePred.gls}
\alias{comparePred.lme}
\alias{comparePred.lmList}
\arguments{
 \item{object1,object2}{fitted model objects, from which predictions can
   be extracted using the \code{predict} method.}
 \item{primary}{an optional one-sided formula specifying the primary
   covariate to be used to generate the augmented predictions. By
   default, if a  covariate can be extracted from the data used to generate
   the objects (using \code{getCovariate}), it will be used as
   \code{primary}.}
 \item{minimum}{an optional lower limit for the primary
   covariate. Defaults to \code{min(primary)}.}
 \item{maximum}{an optional upper limit for the primary
   covariate. Defaults to \code{max(primary)}.}
 \item{length.out}{an optional integer with the number of primary
   covariate values at which to evaluate the predictions. Defaults to
   51.}
 \item{level}{an optional integer specifying the desired
   prediction level. Levels increase from outermost to innermost
   grouping, with level 0 representing the population (fixed effects)
   predictions. Only one level can be specified. Defaults to the
   innermost level.}
 \item{\dots}{some methods for the generic may require additional
   arguments.}
}
\description{
  Predicted values are obtained at the specified values of
  \code{primary} for each object. If either \code{object1} or
  \code{object2} have a grouping structure
  (i.e. \code{getGroups(object)} is not \code{NULL}), predicted values
  are obtained for each group. When both objects determine groups, the
  group levels must be the same. If other covariates besides
  \code{primary} are used in the prediction model, their group-wise averages
  (numeric covariates) or most frequent values (categorical covariates)
  are used to obtain the predicted values. The original observations are
  also included in the returned object. 
}
\value{
  a data frame with four columns representing, respectively, the values
  of the primary covariate, the groups (if \code{object} does not have a
  grouping structure, all elements will be \code{1}), the predicted or
  observed values, and the type of value in the third column: the
  objects' names are used to classify the predicted values and
  \code{original} is used for the observed values. The returned object
  inherits from classes \code{comparePred} and \code{augPred}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include: \code{gls}, \code{lme}, and \code{lmList}.
}
\seealso{\code{\link{augPred}}, \code{\link{getGroups}}}
\examples{
data(Orthodont)
fm1 <- lme(distance ~ age * Sex, data = Orthodont, random = ~ age)
fm2 <- update(fm1, distance ~ age)
comparePred(fm1, fm2, length.out = 2)
}
\keyword{models}

\eof
\name{contr.SAS}
\alias{contr.SAS}
\title{Contrast Matrices}
\description{
  Return a matrix of contrasts corresponding to the parameterization for
  a factor used in SAS.
}
\usage{
contr.SAS(n, contrasts=TRUE)
}
\arguments{
  \item{n}{a vector of levels for a factor, or the number of levels.}
  \item{contrasts}{a logical indicating whether contrasts should be
    computed.}
}
\details{
  This function is used for creating contrast matrices for use in
  fitting analysis of variance and regression models.  The columns of
  the resulting matrix contain contrasts which can be used for coding
  a factor with \code{n} levels.  The returned value contains the
  computed contrasts.  If the argument \code{contrasts} is \code{FALSE}
  then a square indicator matrix is returned.

  These contrasts are similar to those from \code{contr.treatment}
  except that they represent shifts relative to the last level of the
  factor, not the first level.  This generates the same parameterization for a
  linear model as is used in SAS (see \url{http://www.sas.com}).
}
\value{
  A matrix with \code{n} rows and \code{k} columns, with \code{k=n-1} if
  \code{contrasts} is \code{TRUE} and \code{k=n} if \code{contrasts} is
  \code{FALSE}.
}
\author{Jose Pinheiro \email{jcp@research.bell-labs.com}, Douglas Bates
  \email{bates@stat.wisc.edu}} 
\seealso{\code{\link[base:contrast]{contr.treatment}}}
\examples{
contr.SAS(4)
}
\keyword{design}
\keyword{regression}
\keyword{array}

\eof
% $Id: corAR1.Rd,v 1.5.2.1 2002/08/09 19:45:29 bates Exp $
\name{corAR1}
\title{AR(1) Correlation Structure}
\usage{
corAR1(value, form, fixed)
}
\alias{corAR1}
\arguments{
 \item{value}{the value of the lag 1 autocorrelation, which must be
   between -1 and 1. Defaults to 0 (no autocorrelation).}
 \item{form}{a one sided formula of the form \code{~ t}, or \code{~ t |
     g}, specifying a time covariate \code{t} and,  optionally, a
   grouping factor \code{g}. A covariate for this correlation structure
   must be integer valued. When a grouping factor is present in
   \code{form}, the correlation structure is assumed to apply only
   to observations within the same grouping level; observations with
   different grouping levels are assumed to be uncorrelated. Defaults to
   \code{~ 1}, which corresponds to using the order of the observations
   in the data as a covariate, and no groups.}
 \item{fixed}{an optional logical value indicating whether the
   coefficients should be allowed to vary in the optimization, or kept
   fixed at their initial value. Defaults to \code{FALSE}, in which case
   the coefficients are allowed to vary.}
}
\description{
  This function is a constructor for the \code{corAR1} class,
  representing an autocorrelation structure of order 1. Objects
  created using this constructor must later be initialized using the
  appropriate \code{Initialize} method. 
}
\value{
  an object of class \code{corAR1}, representing an autocorrelation
   structure of order 1. 
}
\references{
  Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994) "Time Series
  Analysis: Forecasting and Control", 3rd Edition, Holden-Day. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{Initialize.corStruct}}}
\examples{
## covariate is observation order and grouping factor is Mare
cs1 <- corAR1(0.2, form = ~ 1 | Mare)
}
\keyword{models}

\eof
% $Id: corARMA.Rd,v 1.5.2.1 2002/08/09 19:45:29 bates Exp $
\name{corARMA}
\title{ARMA(p,q) Correlation Structure}
\usage{
corARMA(value, form, p, q, fixed)
}
\alias{corARMA}
\alias{coef.corARMA}
\arguments{
  \item{value}{a vector with the values of the autoregressive and moving
   average parameters, which must have length \code{p + q} and all
   elements between -1 and 1. Defaults to a vector of zeros,
   corresponding to uncorrelated observations.}
 \item{form}{a one sided formula of the form \code{~ t}, or \code{~ t |
     g}, specifying a time covariate \code{t} and,  optionally, a
   grouping factor \code{g}. A covariate for this correlation structure
   must be integer valued. When a grouping factor is present in
   \code{form}, the correlation structure is assumed to apply only
   to observations within the same grouping level; observations with
   different grouping levels are assumed to be uncorrelated. Defaults to
   \code{~ 1}, which corresponds to using the order of the observations
   in the data as a covariate, and no groups.}  
 \item{p, q}{non-negative integers specifying respectively the
   autoregressive order and the moving average order of the \code{ARMA}
   structure. Both default to 0.}
 \item{fixed}{an optional logical value indicating whether the
   coefficients should be allowed to vary in the optimization, or kept
   fixed at their initial value. Defaults to \code{FALSE}, in which case
   the coefficients are allowed to vary.}
} 
\description{
  This function is a constructor for the \code{corARMA} class,
  representing an autocorrelation-moving average correlation structure
  of order (p, q). Objects created using this constructor must later
  be initialized using the appropriate \code{Initialize} method. 
}
\value{
  an object of class \code{corARMA}, representing an
  autocorrelation-moving average correlation structure. 
}

\references{
  Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994) "Time Series
  Analysis: Forecasting and Control", 3rd Edition, Holden-Day. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{Initialize.corStruct}}}
\examples{
## ARMA(1,2) structure, with observation order as a covariate and
## Mare as grouping factor
cs1 <- corARMA(c(0.2, 0.3, -0.1), form = ~ 1 | Mare, p = 1, q = 2)
}
\keyword{models}

\eof
% $Id: corCAR1.Rd,v 1.5.2.1 2002/08/09 19:45:29 bates Exp $
\name{corCAR1}
\title{Continuous AR(1) Correlation Structure}
\usage{
corCAR1(value, form, fixed)
}
\alias{corCAR1}
\arguments{
 \item{value}{the correlation between two observations one unit of time
   apart. Must be between 0 and 1. Defaults to 0.2.}
 \item{form}{a one sided formula of the form \code{~ t}, or \code{~ t |
     g}, specifying a time covariate \code{t} and,  optionally, a
   grouping factor \code{g}. Covariates for this correlation structure
   need not be integer valued.  When a grouping factor is present in
   \code{form}, the correlation structure is assumed to apply only
   to observations within the same grouping level; observations with
   different grouping levels are assumed to be uncorrelated. Defaults to
   \code{~ 1}, which corresponds to using the order of the observations
   in the data as a covariate, and no groups.}  
 \item{fixed}{an optional logical value indicating whether the
   coefficients should be allowed to vary in the optimization, or kept
   fixed at their initial value. Defaults to \code{FALSE}, in which case
   the coefficients are allowed to vary.}
}
\description{
  This function is a constructor for the \code{corCAR1} class,
  representing an autocorrelation structure of order 1, with a
  continuous time covariate. Objects created using this constructor must
  be later initialized using the appropriate \code{Initialize}
  method.  
}
\value{
  an object of class \code{corCAR1}, representing an autocorrelation
   structure of order 1, with a continuous time covariate. 
}
\references{
  Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994) "Time Series
  Analysis: Forecasting and Control", 3rd Edition, Holden-Day.

  Jones, R.H. (1993) "Longitudinal Data with Serial Correlation: A
  State-space Approach", Chapman and Hall
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{Initialize.corStruct}}}
\examples{
## covariate is Time and grouping factor is Mare
cs1 <- corCAR1(0.2, form = ~ Time | Mare)
}
\keyword{models}

\eof
% $Id: corClasses.Rd,v 1.4 2002/03/05 14:59:39 bates Exp $
\name{corClasses}
\title{Correlation Structure Classes}
\alias{corClasses}
\alias{corIdent}
\description{
  Standard classes of correlation structures (\code{corStruct})
  available in the \code{nlme} library. 
}
\value{
  Available standard classes:
  \item{corAR1}{autoregressive process of order 1.}
  \item{corARMA}{autoregressive moving average process, with arbitrary
    orders for the autoregressive and moving average components.}
  \item{corCAR1}{continuous autoregressive process (AR(1) process for a
    continuous time covariate).}
  \item{corCompSymm}{compound symmetry structure corresponding to a
    constant correlation.}
  \item{corExp}{exponential spatial correlation.}
  \item{corGaus}{Gaussian spatial correlation.}
  \item{corLin}{linear spatial correlation.}
  \item{corRatio}{Rational quadratics spatial correlation.}
  \item{corSpher}{spherical spatial correlation.}
  \item{corSymm}{general correlation matrix, with no additional
    structure.} 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\note{
  Users may define their own \code{corStruct} classes by specifying a
  \code{constructor} function and, at a minimum, methods for the
  functions \code{corMatrix} and \code{coef}. For
  examples of these functions, see the methods for classes \code{corSymm}
  and \code{corAR1}. 
}
\seealso{\code{\link{corAR1}}, \code{\link{corARMA}},
  \code{\link{corCAR1}}, \code{\link{corCompSymm}},
  \code{\link{corExp}}, \code{\link{corGaus}}, \code{\link{corLin}},
  \code{\link{corRatio}}, \code{\link{corSpher}}, \code{\link{corSymm}}}
\keyword{models}

\eof
% $Id: corCompSymm.Rd,v 1.5.2.1 2002/08/09 19:45:29 bates Exp $
\name{corCompSymm}
\title{Compound Symmetry Correlation Structure}
\usage{
corCompSymm(value, form, fixed)
}
\alias{corCompSymm}
\arguments{
 \item{value}{the correlation between any two correlated
   observations. Defaults to 0.} 
 \item{form}{a one sided formula of the form \code{~ t}, or \code{~ t |
     g}, specifying a time covariate \code{t} and,  optionally, a
   grouping factor \code{g}. When a grouping factor is present in
   \code{form}, the correlation structure is assumed to apply only
   to observations within the same grouping level; observations with
   different grouping levels are assumed to be uncorrelated. Defaults to
   \code{~ 1}, which corresponds to using the order of the observations
   in the data as a covariate, and no groups.}  
 \item{fixed}{an optional logical value indicating whether the
   coefficients should be allowed to vary in the optimization, or kept
   fixed at their initial value. Defaults to \code{FALSE}, in which case
   the coefficients are allowed to vary.}
}
\description{
  This function is a constructor for the \code{corCompSymm} class,
  representing a compound symmetry structure corresponding to uniform
  correlation. Objects created using this constructor must later be
  initialized using the appropriate \code{Initialize} method. 
}
\value{
  an object of class \code{corCompSymm}, representing a compound
  symmetry correlation structure. 
}
\references{
  Milliken, G. A. and Johnson, D. E. (1992) "Analysis of Messy Data,
  Volume I: Designed Experiments", Van Nostrand Reinhold.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{Initialize.corStruct}}}
\examples{
## covariate is observation order and grouping factor is Subject
cs1 <- corCompSymm(0.5, form = ~ 1 | Subject)
}
\keyword{models}


\eof
% $Id: corExp.Rd,v 1.5.2.1 2002/08/09 19:45:29 bates Exp $
\name{corExp}
\title{Exponential Correlation Structure}
\usage{
corExp(value, form, nugget, metric, fixed)
}
\alias{corExp}
\arguments{
 \item{value}{an optional vector with the parameter values in
   constrained form. If \code{nugget} is \code{FALSE}, \code{value} can
   have only one element, corresponding to the "range" of the
   exponential correlation structure, which must be greater than
   zero. If \code{nugget} is \code{TRUE}, meaning that a nugget effect
   is present, \code{value} can contain one or two elements, the first
   being the "range" and the second the "nugget effect" (one minus the
   correlation between two observations taken arbitrarily close
   together); the first must be greater than zero and the second must be
   between zero and one. Defaults to \code{numeric(0)}, which results in
   a range of 90\% of the minimum distance and a nugget effect of 0.1
   being assigned to the parameters when \code{object} is initialized.}
 \item{form}{a one sided formula of the form \code{~ S1+...+Sp}, or
   \code{~ S1+...+Sp | g}, specifying spatial covariates \code{S1}
   through \code{Sp} and,  optionally, a grouping factor \code{g}. 
   When a grouping factor is present in \code{form}, the correlation
   structure is assumed to apply only to observations within the same
   grouping level; observations with different grouping levels are
   assumed to be uncorrelated. Defaults to \code{~ 1}, which corresponds
   to using the order of the observations in the data as a covariate,
   and no groups.}   
 \item{nugget}{an optional logical value indicating whether a nugget
   effect is present. Defaults to \code{FALSE}.}
 \item{metric}{an optional character string specifying the distance
   metric to be used. The currently available options are
   \code{"euclidean"} for the root sum-of-squares of distances;
   \code{"maximum"} for the maximum difference; and \code{"manhattan"}
   for the sum of the absolute differences. Partial matching of
   arguments is used, so only the first three characters need to be
   provided. Defaults to \code{"euclidean"}.}
 \item{fixed}{an optional logical value indicating whether the
   coefficients should be allowed to vary in the optimization, or kept
   fixed at their initial value. Defaults to \code{FALSE}, in which case
   the coefficients are allowed to vary.}
}
\description{
  This function is a constructor for the \code{corExp} class,
  representing an exponential spatial correlation structure. Letting
  \eqn{d} denote the range and \eqn{n} denote the nugget
  effect, the correlation between two observations a distance
  \eqn{r} apart is \eqn{\exp(-r/d)}{exp(-r/d)} when no nugget effect
  is present and \eqn{(1-n) \exp(-r/d)}{(1-n)*exp(-r/d)} when a nugget
  effect is assumed. Objects created using this constructor must later be
  initialized using the appropriate \code{Initialize} method.
}
\value{
  an object of class \code{corExp}, also inheriting from class
  \code{corSpatial}, representing an exponential spatial correlation
  structure. 
}
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
  Venables, W.N. and Ripley, B.D. (1997) "Modern Applied Statistics with
  S-plus", 2nd Edition, Springer-Verlag.
  Littel, Milliken, Stroup, and Wolfinger (1996) "SAS Systems for Mixed
  Models", SAS Institute.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{Initialize.corStruct}}, \code{\link[mva]{dist}}
}
\examples{
sp1 <- corExp(form = ~ x + y + z)
}
\keyword{models}

\eof
% $Id: corFactor.Rd,v 1.4.2.1 2003/04/17 22:28:45 bates Exp $
\name{corFactor}
\title{Factor of a Correlation Matrix}
\usage{
corFactor(object, \dots)
}
\alias{corFactor}
\arguments{
 \item{object}{an object from which a correlation matrix can be
   extracted.}
 \item{\dots}{some methods for this generic function require additional
   arguments.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include all \code{corStruct} classes.
}
\value{
  will depend on the method function used; see the appropriate
  documentation. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: corFactor.corStruct.Rd,v 1.8.2.1 2002/08/09 19:45:29 bates Exp $
\name{corFactor.corStruct}
\title{Factor of a corStruct Object Matrix}
\usage{
\method{corFactor}{corStruct}(object, \dots)
}
\alias{corFactor.compSymm}
\alias{corFactor.corAR1}
\alias{corFactor.corARMA}
\alias{corFactor.corCAR1}
\alias{corFactor.corNatural}
\alias{corFactor.corSpatial}
\alias{corFactor.corStruct}
\alias{corFactor.corSymm}
\arguments{
 \item{object}{an object inheriting from class \code{corStruct}
   representing a correlation structure, which must have been
   initialized (using \code{Initialize}).}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts a transpose inverse square-root factor,
  or a series of transpose inverse square-root factors, of the
  correlation matrix, or list of correlation matrices,  represented by
  \code{object}. Letting \eqn{\Sigma}{S} denote a correlation
  matrix, a square-root factor of \eqn{\Sigma}{S} is any square
  matrix \eqn{L}{L} such that \eqn{\Sigma = L'L}{S = L'L}. This method
  extracts \eqn{L^{-t}}{L^(-t)}.
}
\value{
  If the correlation structure does not include a grouping factor, the
  returned value will be a vector with a transpose inverse square-root
  factor of the correlation matrix associated with \code{object} stacked
  column-wise.  If the correlation structure includes a grouping factor,
  the returned value will be a vector with transpose inverse
  square-root factors of the correlation matrices for each group, stacked
  by group and stacked column-wise within each group.  
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and
  Douglas Bates \email{bates@stat.wisc.edu}
}
\note{
  This method function is used intensively in optimization
  algorithms and its value is returned as a vector for efficiency
  reasons. The \code{corMatrix} method function can be used to obtain
  transpose inverse square-root factors in matrix form.
}
\seealso{\code{\link{corMatrix.corStruct}},
  \code{\link{recalc.corStruct}}, \code{\link{Initialize.corStruct}}}

\examples{
data(Orthodont)
cs1 <- corAR1(form = ~1 | Subject)
cs1 <- Initialize(cs1, data = Orthodont)
corFactor(cs1)
}
\keyword{models}

\eof
% $Id: corGaus.Rd,v 1.5.2.1 2002/08/09 19:45:29 bates Exp $
\name{corGaus}
\title{Gaussian Correlation Structure}
\usage{
corGaus(value, form, nugget, metric, fixed)
}
\alias{corGaus}
\arguments{
 \item{value}{an optional vector with the parameter values in
   constrained form. If \code{nugget} is \code{FALSE}, \code{value} can
   have only one element, corresponding to the "range" of the
   Gaussian correlation structure, which must be greater than
   zero. If \code{nugget} is \code{TRUE}, meaning that a nugget effect
   is present, \code{value} can contain one or two elements, the first
   being the "range" and the second the "nugget effect" (one minus the
   correlation between two observations taken arbitrarily close
   together); the first must be greater than zero and the second must be
   between zero and one. Defaults to \code{numeric(0)}, which results in
   a range of 90\% of the minimum distance and a nugget effect of 0.1
   being assigned to the parameters when \code{object} is initialized.}
 \item{form}{a one sided formula of the form \code{~ S1+...+Sp}, or
   \code{~ S1+...+Sp | g}, specifying spatial covariates \code{S1}
   through \code{Sp} and,  optionally, a grouping factor \code{g}. 
   When a grouping factor is present in \code{form}, the correlation
   structure is assumed to apply only to observations within the same
   grouping level; observations with different grouping levels are
   assumed to be uncorrelated. Defaults to \code{~ 1}, which corresponds
   to using the order of the observations in the data as a covariate,
   and no groups.}   
 \item{nugget}{an optional logical value indicating whether a nugget
   effect is present. Defaults to \code{FALSE}.}
 \item{metric}{an optional character string specifying the distance
   metric to be used. The currently available options are
   \code{"euclidean"} for the root sum-of-squares of distances;
   \code{"maximum"} for the maximum difference; and \code{"manhattan"}
   for the sum of the absolute differences. Partial matching of
   arguments is used, so only the first three characters need to be
   provided. Defaults to \code{"euclidean"}.}
 \item{fixed}{an optional logical value indicating whether the
   coefficients should be allowed to vary in the optimization, or kept
   fixed at their initial value. Defaults to \code{FALSE}, in which case
   the coefficients are allowed to vary.}
}
\description{
  This function is a constructor for the \code{corGaus} class,
  representing a Gaussian spatial correlation structure. Letting
  \eqn{d} denote the range and \eqn{n} denote the nugget
  effect, the correlation between two observations a distance
  \eqn{r} apart is \eqn{\exp(-(r/d)^2)}{exp(-(r/d)^2)} when no nugget
  effect is present and \eqn{(1-n) \exp(-(r/d)^2)}{(1-n)*exp(-(r/d)^2)}
  when a nugget effect is assumed. Objects created using this
  constructor must later be initialized using the appropriate
`  \code{Initialize} method. 
}
\value{
  an object of class \code{corGaus}, also inheriting from class
  \code{corSpatial}, representing a Gaussian spatial correlation
  structure. 
}
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
  Venables, W.N. and Ripley, B.D. (1997) "Modern Applied Statistics with
  S-plus", 2nd Edition, Springer-Verlag.
  Littel, Milliken, Stroup, and Wolfinger (1996) "SAS Systems for Mixed
  Models", SAS Institute.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{Initialize.corStruct}}, \code{\link[mva]{dist}}
}
\examples{
sp1 <- corGaus(form = ~ x + y + z)
}
\keyword{models}

\eof
% $Id: corLin.Rd,v 1.5.2.1 2002/08/09 19:45:29 bates Exp $
\name{corLin}
\title{Linear Correlation Structure}
\usage{
corLin(value, form, nugget, metric, fixed)
}
\alias{corLin}
\arguments{
 \item{value}{an optional vector with the parameter values in
   constrained form. If \code{nugget} is \code{FALSE}, \code{value} can
   have only one element, corresponding to the "range" of the
   linear correlation structure, which must be greater than
   zero. If \code{nugget} is \code{TRUE}, meaning that a nugget effect
   is present, \code{value} can contain one or two elements, the first
   being the "range" and the second the "nugget effect" (one minus the
   correlation between two observations taken arbitrarily close
   together); the first must be greater than zero and the second must be
   between zero and one. Defaults to \code{numeric(0)}, which results in
   a range of 90\% of the minimum distance and a nugget effect of 0.1
   being assigned to the parameters when \code{object} is initialized.}
 \item{form}{a one sided formula of the form \code{~ S1+...+Sp}, or
   \code{~ S1+...+Sp | g}, specifying spatial covariates \code{S1}
   through \code{Sp} and,  optionally, a grouping factor \code{g}. 
   When a grouping factor is present in \code{form}, the correlation
   structure is assumed to apply only to observations within the same
   grouping level; observations with different grouping levels are
   assumed to be uncorrelated. Defaults to \code{~ 1}, which corresponds
   to using the order of the observations in the data as a covariate,
   and no groups.}   
 \item{nugget}{an optional logical value indicating whether a nugget
   effect is present. Defaults to \code{FALSE}.}
 \item{metric}{an optional character string specifying the distance
   metric to be used. The currently available options are
   \code{"euclidean"} for the root sum-of-squares of distances;
   \code{"maximum"} for the maximum difference; and \code{"manhattan"}
   for the sum of the absolute differences. Partial matching of
   arguments is used, so only the first three characters need to be
   provided. Defaults to \code{"euclidean"}.}
 \item{fixed}{an optional logical value indicating whether the
   coefficients should be allowed to vary in the optimization, or kept
   fixed at their initial value. Defaults to \code{FALSE}, in which case
   the coefficients are allowed to vary.}
}
\description{
  This function is a constructor for the \code{corLin} class,
  representing a linear spatial correlation structure. Letting
  \eqn{d} denote the range and \eqn{n} denote the nugget
  effect, the correlation between two observations a distance
  \eqn{r < d} apart is \eqn{1-(r/d)} when no nugget effect
  is present and \eqn{(1-n) (1 -(r/d))}{(1-n)*(1-(r/d))} when a nugget
  effect is assumed. If \eqn{r \geq d}{r >= d} the correlation is
  zero. Objects created using this constructor must later be
  initialized using the appropriate \code{Initialize} method. 
}
\value{
  an object of class \code{corLin}, also inheriting from class
  \code{corSpatial}, representing a linear spatial correlation
  structure. 
}
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
  Venables, W.N. and Ripley, B.D. (1997) "Modern Applied Statistics with
  S-plus", 2nd Edition, Springer-Verlag.
  Littel, Milliken, Stroup, and Wolfinger (1996) "SAS Systems for Mixed
  Models", SAS Institute.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{Initialize.corStruct}}, \code{\link[mva]{dist}}}
\examples{
sp1 <- corLin(form = ~ x + y)
}
\keyword{models}

\eof
% $Id: corMatrix.Rd,v 1.4.2.1 2003/04/17 22:28:45 bates Exp $
\name{corMatrix}
\title{Extract Correlation Matrix}
\usage{
corMatrix(object, \dots)
}
\alias{corMatrix}
\arguments{
 \item{object}{an object for which a correlation matrix can be
   extracted.} 
 \item{\dots}{some methods for this generic function require additional
   arguments.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include all \code{corStruct} classes.
}
\value{
  will depend on the method function used; see the appropriate
  documentation. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: corMatrix.corStruct.Rd,v 1.9.2.1 2002/08/09 19:45:29 bates Exp $
\name{corMatrix.corStruct}
\title{Matrix of a corStruct Object}
\usage{
\method{corMatrix}{corStruct}(object, covariate, corr, \dots)
}
\alias{corMatrix.corStruct}
\alias{corMatrix.compSymm}
\alias{corMatrix.corAR1}
\alias{corMatrix.corARMA}
\alias{corMatrix.corCAR1}
\alias{corMatrix.corCompSymm}
\alias{corMatrix.corIdent}
\alias{corMatrix.corNatural}
\alias{corMatrix.corSpatial}
\alias{corMatrix.corSymm}
\arguments{
 \item{object}{an object inheriting from class \code{corStruct}
   representing a correlation structure.}
 \item{covariate}{an optional covariate vector (matrix), or list of
   covariate vectors (matrices), at which values the correlation matrix,
   or list of correlation  matrices, are to be evaluated. Defaults to
   \code{getCovariate(object)}.} 
 \item{corr}{a logical value. If \code{TRUE} the function returns the
   correlation matrix, or list of correlation matrices, represented by
   \code{object}. If \code{FALSE} the function returns a transpose
   inverse square-root of the correlation matrix, or a list of transpose
   inverse square-root factors of the correlation matrices.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts the correlation matrix (or its transpose
  inverse square-root factor), or list of correlation matrices (or their
  transpose inverse square-root factors) corresponding to
  \code{covariate} and \code{object}. Letting \eqn{\Sigma}{S} denote
  a correlation matrix, a square-root factor of \eqn{\Sigma}{S} is
  any square matrix \eqn{L} such that \eqn{\Sigma =  L'L}{S=L'L}. When
  \code{corr = FALSE}, this method extracts \eqn{L^{-t}}{L^(-t)}. 
}
\value{
  If \code{covariate} is a vector (matrix), the returned value will be
  an array with the corresponding correlation matrix (or its transpose
  inverse square-root factor). If the \code{covariate} is a list of
  vectors (matrices), the returned value will be a list with the
  correlation matrices (or their transpose inverse square-root factors)
  corresponding to each component of \code{covariate}.  
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{corFactor.corStruct}}, \code{\link{Initialize.corStruct}}}

\examples{
cs1 <- corAR1(0.3)
corMatrix(cs1, covariate = 1:4)
corMatrix(cs1, covariate = 1:4, corr = FALSE)
}
\keyword{models}

\eof
% $Id: corMatrix.pdMat.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{corMatrix.pdMat}
\title{Extract Correlation Matrix from a pdMat Object}
\usage{
\method{corMatrix}{pdMat}(object, \dots)
}
\alias{corMatrix.pdBlocked}
\alias{corMatrix.pdCompSymm}
\alias{corMatrix.pdDiag}
\alias{corMatrix.pdIdent}
\alias{corMatrix.pdMat}
\alias{corMatrix.pdSymm}

\arguments{
 \item{object}{an object inheriting from class \code{pdMat}, representing
   a positive definite matrix.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}

\description{
  The correlation matrix corresponding to the positive-definite matrix
  represented by \code{object} is obtained. 
}
\value{
  the correlation matrix corresponding to the positive-definite matrix
  represented by \code{object}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{as.matrix.pdMat}}, \code{\link{pdMatrix}}}

\examples{
pd1 <- pdSymm(diag(1:4))
corMatrix(pd1)
}
\keyword{models}

\eof
% $Id: corMatrix.reStruct.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{corMatrix.reStruct}
\title{Extract Correlation Matrix from Components of an reStruct Object}
\usage{
\method{corMatrix}{reStruct}(object, \dots)
}
\alias{corMatrix.reStruct}

\arguments{
 \item{object}{an object inheriting from class \code{reStruct},
   representing a random effects structure and consisting of a list of
   \code{pdMat} objects.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts the correlation matrices
  corresponding to the \code{pdMat} elements of \code{object}. 
}
\value{
  a list with components given by the correlation matrices
  corresponding to the elements of \code{object}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{as.matrix.reStruct}}, \code{\link{reStruct}},
  \code{\link{pdMat}}}

\examples{
data(Orthodont)
rs1 <- reStruct(pdSymm(diag(3), ~age+Sex, data = Orthodont))
corMatrix(rs1)
}
\keyword{models}

\eof
% $Id: corNatural.Rd,v 1.2.2.1 2002/08/09 19:45:29 bates Exp $
\name{corNatural}
\title{General correlation in natural parameterization}
\usage{
corNatural(value, form, fixed)
}
\alias{corNatural}
\alias{print.corNatural}
\arguments{
 \item{value}{an optional vector with the parameter values. Default is
   \code{numeric(0)}, which results in a vector of zeros of appropriate
   dimension being assigned to the parameters when \code{object} is
   initialized (corresponding to an identity correlation structure).} 
 \item{form}{a one sided formula of the form \code{~ t}, or \code{~ t |
     g}, specifying a time covariate \code{t} and,  optionally, a
   grouping factor \code{g}. A covariate for this correlation structure
   must be integer valued. When a grouping factor is present in
   \code{form}, the correlation structure is assumed to apply only
   to observations within the same grouping level; observations with
   different grouping levels are assumed to be uncorrelated. Defaults to
   \code{~ 1}, which corresponds to using the order of the observations
   in the data as a covariate, and no groups.}  
 \item{fixed}{an optional logical value indicating whether the
   coefficients should be allowed to vary in the optimization, or kept
   fixed at their initial value. Defaults to \code{FALSE}, in which case
   the coefficients are allowed to vary.}
}
\description{
  This function is a constructor for the \code{corNatural} class,
  representing a general correlation structure in the ``natural''
  parameterization, which is described under \code{\link{pdNatural}}.
  Objects created using this constructor must later be
  initialized using the  appropriate \code{Initialize} method. 
}
\value{
  an object of class \code{corNatural} representing a general correlation
  structure.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{Initialize.corNatural}}, \code{\link{pdNatural}}}
\examples{
## covariate is observation order and grouping factor is Subject
cs1 <- corNatural(form = ~ 1 | Subject)
}
\keyword{models}

\eof
% $Id: corRatio.Rd,v 1.5.2.1 2002/08/09 19:45:29 bates Exp $
\name{corRatio}
\title{Rational Quadratic Correlation Structure}
\usage{
corRatio(value, form, nugget, metric, fixed)
}
\alias{corRatio}
\arguments{
 \item{value}{an optional vector with the parameter values in
   constrained form. If \code{nugget} is \code{FALSE}, \code{value} can
   have only one element, corresponding to the "range" of the
   rational quadratic correlation structure, which must be greater than
   zero. If \code{nugget} is \code{TRUE}, meaning that a nugget effect
   is present, \code{value} can contain one or two elements, the first
   being the "range" and the second the "nugget effect" (one minus the
   correlation between two observations taken arbitrarily close
   together); the first must be greater than zero and the second must be
   between zero and one. Defaults to \code{numeric(0)}, which results in
   a range of 90\% of the minimum distance and a nugget effect of 0.1
   being assigned to the parameters when \code{object} is initialized.}
 \item{form}{a one sided formula of the form \code{~ S1+...+Sp}, or
   \code{~ S1+...+Sp | g}, specifying spatial covariates \code{S1}
   through \code{Sp} and,  optionally, a grouping factor \code{g}. 
   When a grouping factor is present in \code{form}, the correlation
   structure is assumed to apply only to observations within the same
   grouping level; observations with different grouping levels are
   assumed to be uncorrelated. Defaults to \code{~ 1}, which corresponds
   to using the order of the observations in the data as a covariate,
   and no groups.}   
 \item{nugget}{an optional logical value indicating whether a nugget
   effect is present. Defaults to \code{FALSE}.}
 \item{metric}{an optional character string specifying the distance
   metric to be used. The currently available options are
   \code{"euclidean"} for the root sum-of-squares of distances;
   \code{"maximum"} for the maximum difference; and \code{"manhattan"}
   for the sum of the absolute differences. Partial matching of
   arguments is used, so only the first three characters need to be
   provided. Defaults to \code{"euclidean"}.}
 \item{fixed}{an optional logical value indicating whether the
   coefficients should be allowed to vary in the optimization, or kept
   fixed at their initial value. Defaults to \code{FALSE}, in which case
   the coefficients are allowed to vary.}
}
\description{
  This function is a constructor for the \code{corRatio} class,
  representing a rational quadratic spatial correlation structure. Letting
  \eqn{d} denote the range and \eqn{n} denote the nugget
  effect, the correlation between two observations a distance
  \eqn{r} apart is \eqn{(r/d)^2/(1+(r/d)^2)} when no nugget effect
  is present and
  \eqn{(1-n)(r/d)^2/(1+(r/d)^2)}{(1-n)*(r/d)^2/(1+(r/d)^2)} when a
  nugget effect is  assumed. Objects created using this constructor need
  to be later initialized using the appropriate \code{Initialize}
  method. 
}
\value{
  an object of class \code{corRatio}, also inheriting from class
  \code{corSpatial}, representing a rational quadratic spatial correlation
  structure. 
}
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
  Venables, W.N. and Ripley, B.D. (1997) "Modern Applied Statistics with
  S-plus", 2nd Edition, Springer-Verlag.
  Littel, Milliken, Stroup, and Wolfinger (1996) "SAS Systems for Mixed
  Models", SAS Institute.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{Initialize.corStruct}}, \code{\link[mva]{dist}}
}
\examples{
sp1 <- corRatio(form = ~ x + y + z)
}
\keyword{models}

\eof
% $Id: corSpatial.Rd,v 1.5.2.1 2002/08/09 19:45:29 bates Exp $
\name{corSpatial}
\title{Spatial Correlation Structure}
\usage{
corSpatial(value, form, nugget, type, metric, fixed)
}
\alias{corSpatial}
\arguments{
 \item{value}{an optional vector with the parameter values in
   constrained form. If \code{nugget} is \code{FALSE}, \code{value} can
   have only one element, corresponding to the "range" of the
   spatial correlation structure, which must be greater than
   zero. If \code{nugget} is \code{TRUE}, meaning that a nugget effect
   is present, \code{value} can contain one or two elements, the first
   being the "range" and the second the "nugget effect" (one minus the
   correlation between two observations taken arbitrarily close
   together); the first must be greater than zero and the second must be
   between zero and one. Defaults to \code{numeric(0)}, which results in
   a range of 90\% of the minimum distance and a nugget effect of 0.1
   being assigned to the parameters when \code{object} is initialized.}
 \item{form}{a one sided formula of the form \code{~ S1+...+Sp}, or
   \code{~ S1+...+Sp | g}, specifying spatial covariates \code{S1}
   through \code{Sp} and,  optionally, a grouping factor \code{g}. 
   When a grouping factor is present in \code{form}, the correlation
   structure is assumed to apply only to observations within the same
   grouping level; observations with different grouping levels are
   assumed to be uncorrelated. Defaults to \code{~ 1}, which corresponds
   to using the order of the observations in the data as a covariate,
   and no groups.}   
 \item{nugget}{an optional logical value indicating whether a nugget
   effect is present. Defaults to \code{FALSE}.}
 \item{type}{an optional character string specifying the desired type of
   correlation structure. Available types include \code{"spherical"},
   \code{"exponential"}, \code{"gaussian"}, \code{"linear"}, and
   \code{"rational"}. See the documentation on the functions
   \code{corSpher}, \code{corExp}, \code{corGaus}, \code{corLin}, and
   \code{corRatio} for a description of these correlation
   structures. Partial matching of arguments is used, so only the first
   character needs to be provided.Defaults to \code{"spherical"}.} 
 \item{metric}{an optional character string specifying the distance
   metric to be used. The currently available options are
   \code{"euclidean"} for the root sum-of-squares of distances;
   \code{"maximum"} for the maximum difference; and \code{"manhattan"}
   for the sum of the absolute differences. Partial matching of
   arguments is used, so only the first three characters need to be
   provided. Defaults to \code{"euclidean"}.}
 \item{fixed}{an optional logical value indicating whether the
   coefficients should be allowed to vary in the optimization, or kept
   fixed at their initial value. Defaults to \code{FALSE}, in which case
   the coefficients are allowed to vary.}
}
\description{
  This function is a constructor for the \code{corSpatial} class,
  representing a spatial correlation structure. This class is "virtual",
  having four "real" classes, corresponding to specific spatial
  correlation structures, associated with it: \code{corExp},
  \code{corGaus}, \code{corLin}, \code{corRatio}, and
  \code{corSpher}. The returned object will inherit from one of these
  "real" classes, determined by the \code{type} argument, and from the
  "virtual" \code{corSpatial} class. Objects created using this
  constructor must later be initialized using the appropriate
  \code{Initialize} method. 
}
\value{
  an object of class determined by the \code{type} argument and also
  inheriting from class \code{corSpatial}, representing a spatial
  correlation structure.
}
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
  Venables, W.N. and Ripley, B.D. (1997) "Modern Applied Statistics with
  S-plus", 2nd Edition, Springer-Verlag.
  Littel, Milliken, Stroup, and Wolfinger (1996) "SAS Systems for Mixed
  Models", SAS Institute.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{corExp}}, \code{\link{corGaus}},
    \code{\link{corLin}}, \code{\link{corRatio}}, \code{\link{corSpher}},
    \code{\link{Initialize.corStruct}}, \code{\link[mva]{dist}}
}
\examples{
sp1 <- corSpatial(form = ~ x + y + z, type = "g", metric = "man")
}
\keyword{models}


\eof
% $Id: corSpher.Rd,v 1.5.2.1 2002/08/09 19:45:29 bates Exp $
\name{corSpher}
\title{Spherical Correlation Structure}
\usage{
corSpher(value, form, nugget, metric, fixed)
}
\alias{corSpher}
\arguments{
 \item{value}{an optional vector with the parameter values in
   constrained form. If \code{nugget} is \code{FALSE}, \code{value} can
   have only one element, corresponding to the "range" of the
   spherical correlation structure, which must be greater than
   zero. If \code{nugget} is \code{TRUE}, meaning that a nugget effect
   is present, \code{value} can contain one or two elements, the first
   being the "range" and the second the "nugget effect" (one minus the
   correlation between two observations taken arbitrarily close
   together); the first must be greater than zero and the second must be
   between zero and one. Defaults to \code{numeric(0)}, which results in
   a range of 90\% of the minimum distance and a nugget effect of 0.1
   being assigned to the parameters when \code{object} is initialized.}
 \item{form}{a one sided formula of the form \code{~ S1+...+Sp}, or
   \code{~ S1+...+Sp | g}, specifying spatial covariates \code{S1}
   through \code{Sp} and,  optionally, a grouping factor \code{g}. 
   When a grouping factor is present in \code{form}, the correlation
   structure is assumed to apply only to observations within the same
   grouping level; observations with different grouping levels are
   assumed to be uncorrelated. Defaults to \code{~ 1}, which corresponds
   to using the order of the observations in the data as a covariate,
   and no groups.}   
 \item{nugget}{an optional logical value indicating whether a nugget
   effect is present. Defaults to \code{FALSE}.}
 \item{metric}{an optional character string specifying the distance
   metric to be used. The currently available options are
   \code{"euclidean"} for the root sum-of-squares of distances;
   \code{"maximum"} for the maximum difference; and \code{"manhattan"}
   for the sum of the absolute differences. Partial matching of
   arguments is used, so only the first three characters need to be
   provided. Defaults to \code{"euclidean"}.}
 \item{fixed}{an optional logical value indicating whether the
   coefficients should be allowed to vary in the optimization, or kept
   fixed at their initial value. Defaults to \code{FALSE}, in which case
   the coefficients are allowed to vary.}
}
\description{
  This function is a constructor for the \code{corSpher} class,
  representing a spherical spatial correlation structure. Letting
  \eqn{d} denote the range and \eqn{n} denote the nugget
  effect, the correlation between two observations a distance
  \eqn{r < d} apart is \eqn{1-1.5(r/d)+0.5(r/d)^3} when no
  nugget effect is present and \eqn{(1-n)
    (1-1.5(r/d)+0.5(r/d)^3)}{(1-n)*(1-1.5(r/d)+0.5(r/d)^3)}  
  when a nugget effect is assumed. If \eqn{r \geq d}{r >= d} the
  correlation is zero. Objects created using this constructor must later
  be initialized using the appropriate \code{Initialize} method. 
}
\value{
  an object of class \code{corSpher}, also inheriting from class
  \code{corSpatial}, representing a spherical spatial correlation
  structure. 
}
\references{
  Cressie, N.A.C. (1993), "Statistics for Spatial Data", J. Wiley & Sons.
  Venables, W.N. and Ripley, B.D. (1997) "Modern Applied Statistics with
  S-plus", 2nd Edition, Springer-Verlag.
  Littel, Milliken, Stroup, and Wolfinger (1996) "SAS Systems for Mixed
  Models", SAS Institute.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{Initialize.corStruct}}, \code{\link[mva]{dist}}}
\examples{
sp1 <- corSpher(form = ~ x + y)
}
\keyword{models}

\eof
% $Id: corSymm.Rd,v 1.5.2.1 2002/08/09 19:45:29 bates Exp $
\name{corSymm}
\title{General Correlation Structure}
\usage{
corSymm(value, form, fixed)
}
\alias{corSymm}
\arguments{
 \item{value}{an optional vector with the parameter values. Default is
   \code{numeric(0)}, which results in a vector of zeros of appropriate
   dimension being assigned to the parameters when \code{object} is
   initialized (corresponding to an identity correlation structure).} 
 \item{form}{a one sided formula of the form \code{~ t}, or \code{~ t |
     g}, specifying a time covariate \code{t} and,  optionally, a
   grouping factor \code{g}. A covariate for this correlation structure
   must be integer valued. When a grouping factor is present in
   \code{form}, the correlation structure is assumed to apply only
   to observations within the same grouping level; observations with
   different grouping levels are assumed to be uncorrelated. Defaults to
   \code{~ 1}, which corresponds to using the order of the observations
   in the data as a covariate, and no groups.}  
 \item{fixed}{an optional logical value indicating whether the
   coefficients should be allowed to vary in the optimization, or kept
   fixed at their initial value. Defaults to \code{FALSE}, in which case
   the coefficients are allowed to vary.}
}
\description{
  This function is a constructor for the \code{corSymm} class,
  representing a general correlation structure. The internal
  representation of this structure, in terms of unconstrained
  parameters, uses the spherical parametrization defined in Pinheiro and
  Bates (1996).  Objects created using this constructor must later be
  initialized using the  appropriate \code{Initialize} method. 
}
\value{
  an object of class \code{corSymm} representing a general correlation
  structure.
}
\references{ 
Pinheiro, J.C. and Bates., D.M.  (1996) "Unconstrained
Parametrizations for Variance-Covariance Matrices", Statistics and
Computing, 6, 289-296.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{Initialize.corSymm}}}

\examples{
## covariate is observation order and grouping factor is Subject
cs1 <- corSymm(form = ~ 1 | Subject)
}
\keyword{models}

\eof
\name{ergoStool}
\alias{ergoStool}
\non_function{}
\title{Ergometrics experiment with stool types}
\description{
    The \code{ergoStool} data frame has 36 rows and 3 columns.
}
\format{
    This data frame contains the following columns:
    \describe{
	\item{effort}{
	    a numeric vector giving the effort (Borg scale) required to
	    arise from a stool.
	}
	\item{Type}{
	    a factor with levels
	    \code{T1}, 
	    \code{T2}, 
	    \code{T3}, and 
	    \code{T4} giving the stool type.
	}
	\item{Subject}{
	    an ordered factor giving a unique identifier for the subject
	    in the experiment.
	}
    }
}
\details{
  Devore (2000) cites data from an article in \emph{Ergometrics} (1993,
  pp. 519-535) on ``The Effects of a Pneumatic Stool and a One-Legged
  Stool on Lower Limb Joint Load and Muscular Activity.''
}
\source{
    Pinheiro, J. C. and Bates, D. M. (2000), \emph{Mixed-Effects Models in S
	and S-PLUS}, Springer, New York.  (Appendix A.9)

    Devore, J. L. (2000), \emph{Probability and Statistics for
      Engineering and the Sciences (5th ed)}, Duxbury, Boston, MA.
}
\examples{
data(ergoStool)
fm1 <-
   lme(effort ~ Type, data = ergoStool, random = ~ 1 | Subject)
anova( fm1 )
}
\keyword{datasets}

\eof
\name{fdHess}
\alias{fdHess}
\title{Finite difference Hessian}
\description{
  Evaluate an approximate Hessian and gradient of a scalar function
  using finite differences.
}
\usage{
fdHess(pars, fun, \dots, .relStep=(.Machine$double.eps)^(1/3), minAbsPar=0)
}
\arguments{
  \item{pars}{the numeric values of the parameters at which to evaluate the
    function \code{fun} and its derivatives.}
  \item{fun}{a function depending on the parameters \code{pars} that
    returns a numeric scalar.}
  \item{\dots}{Optional additional arguments to \code{fun}}
  \item{.relStep}{The relative step size to use in the finite
    differences.  It defaults to the cube root of \code{.Machine$double.eps}}
  \item{minAbsPar}{The minimum magnitude of a parameter value that is
    considered non-zero.  It defaults to zero meaning that any non-zero
    value will be considered different from zero.}
}
\details{
  This function uses a second-order response surface design known as a
  Koschal design to determine the parameter values at which the function
  is evaluated.
}
\value{
  A list with components
  \item{mean}{the value of function \code{fun} evaluated at the
    parameter values \code{pars}}
  \item{gradient}{an approximate gradient}
  \item{Hessian}{a matrix whose upper triangle containst an approximate Hessian.}
}
\author{Jose Pinheiro \email{jcp@research.bell-labs.com}, Douglas Bates
  \email{bates@stat.wisc.edu}} 
\examples{
fdHess(c(12.3, 2.34), function(x) x[1]*(1-exp(-0.4*x[2])))
}
\keyword{models}

\eof
% $Id: fitted.glsStruct.Rd,v 1.6.2.2 2003/05/17 21:51:26 bates Exp $
\name{fitted.glsStruct}
\title{Calculate glsStruct Fitted Values}
\usage{
\method{fitted}{glsStruct}(object, glsFit, \dots)
}
\alias{fitted.glsStruct}
\arguments{
 \item{object}{an object inheriting from class \code{glsStruct},
   representing a list of linear model components, such as
   \code{corStruct} and \code{varFunc} objects.}
 \item{glsFit}{an optional list with components \code{logLik}
   (log-likelihood), \code{beta} (coefficients), \code{sigma} (standard
   deviation for error term), \code{varBeta} (coefficients' covariance
   matrix), \code{fitted} (fitted values), and \code{residuals}
   (residuals). Defaults to \code{attr(object, "glsFit")}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The fitted values for the linear model represented by \code{object}
  are extracted.
}
\value{
  a vector with the fitted values for the linear model represented by
  \code{object}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  This method function is generally only used inside \code{gls} and 
  \code{fitted.gls}.
}

\seealso{\code{\link{gls}}, %\code{\link{fitted.gls}},
  \code{\link{residuals.glsStruct}} } 

\keyword{models}

\eof
% $Id: fitted.gnlsStruct.Rd,v 1.6.2.1 2003/05/17 21:52:25 bates Exp $
\name{fitted.gnlsStruct}
\title{Calculate gnlsStruct Fitted Values}
\usage{
\method{fitted}{gnlsStruct}(object, \dots)
}
\alias{fitted.gnlsStruct}
\arguments{
 \item{object}{an object inheriting from class \code{gnlsStruct},
   representing a list of model components, such as
   \code{corStruct} and \code{varFunc} objects, and attributes
   specifying the underlying nonlinear model and the response variable.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The fitted values for the nonlinear model represented by \code{object}
  are extracted.
}
\value{
  a vector with the fitted values for the nonlinear model represented by
  \code{object}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  This method function is generally only used inside \code{gnls} and 
  \code{fitted.gnls}.
}

\seealso{\code{\link{gnls}}, %\code{\link{fitted.gnls}},
  \code{\link{residuals.gnlsStruct}} } 

\keyword{models}

\eof
% $Id: fitted.lmList.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{fitted.lmList}
\title{Extract lmList Fitted Values}
\usage{
\method{fitted}{lmList}(object, subset, asList, \dots)
}
\alias{fitted.lmList}
\arguments{
 \item{object}{an object inheriting from class \code{lmList}, representing
   a list of \code{lm} objects with a common model.
 }
 \item{subset}{an optional character or integer vector naming the
   \code{lm} components of \code{object} from which the fitted values
   are to be extracted. Default is \code{NULL}, in which case all
   components are used.
 }
 \item{asList}{an optional logical value. If \code{TRUE}, the returned
   object is a list with the fitted values split by groups; else the
   returned value is a vector. Defaults to \code{FALSE}.
 }
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The fitted values are extracted from each \code{lm} component of
  \code{object} and arranged into a list with as many components as
  \code{object}, or combined into a single vector.
}
\value{
  a list with components given by the fitted values of each \code{lm}
  component of \code{object}, or a vector with the fitted values for all
  \code{lm} components of \code{object}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lmList}}, \code{\link{residuals.lmList}}}

\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age | Subject, Orthodont)
fitted(fm1)
}
\keyword{models}

\eof
% $Id: fitted.lme.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{fitted.lme}
\title{Extract lme Fitted Values}
\usage{
\method{fitted}{lme}(object, level, asList, \dots)
}
\alias{fitted.lme}
\arguments{
 \item{object}{an object inheriting from class \code{lme}, representing
   a fitted linear mixed-effects model.}
 \item{level}{an optional integer vector giving the level(s) of grouping
   to be used in extracting the fitted values from \code{object}. Level
   values increase from outermost to innermost grouping, with
   level zero corresponding to the population fitted values. Defaults to
   the highest or innermost level of grouping.} 
 \item{asList}{an optional logical value. If \code{TRUE} and a single
   value is given in \code{level}, the returned object is a list with
   the fitted values split by groups; else the returned value is
   either a vector or a data frame, according to the length of
   \code{level}. Defaults to \code{FALSE}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The fitted values at level \eqn{i} are obtained by adding together the
  population fitted values (based only on the fixed effects estimates)
  and the estimated contributions of the random effects to the fitted
  values at grouping levels less or equal to \eqn{i}. The resulting
  values estimate the best linear unbiased predictions (BLUPs) at level
  \eqn{i}.  
}
\value{
  If a single level of grouping is specified in \code{level}, the
  returned value is either a list with the fitted values split by groups
  (\code{asList = TRUE}) or a vector with the fitted values
  (\code{asList = FALSE}); else, when multiple grouping levels are
  specified in \code{level}, the returned object is a data frame with
  columns given by the fitted values at different levels and the
  grouping factors. 
}
\references{
  Bates, D.M. and Pinheiro, J.C. (1998) "Computational methods for
  multilevel models" available in PostScript or PDF formats at
  http://nlme.stat.wisc.edu/pub/NLME/
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lme}}, \code{\link{residuals.lme}} }

\examples{
data(Orthodont)
fm1 <- lme(distance ~ age + Sex, data = Orthodont, random = ~ 1)
fitted(fm1, level = 0:1)
}
\keyword{models}

\eof
% $Id: fitted.lmeStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{fitted.lmeStruct}
\title{Calculate lmeStruct Fitted Values}
\usage{
\method{fitted}{lmeStruct}(object, level, conLin, lmeFit, \dots)
}
\alias{fitted.lmeStruct}
\arguments{
 \item{object}{an object inheriting from class \code{lmeStruct},
   representing a list of linear mixed-effects model components, such as
   \code{reStruct}, \code{corStruct}, and \code{varFunc} objects.}
 \item{level}{an optional integer vector giving the level(s) of grouping
   to be used in extracting the fitted values from \code{object}. Level
   values increase from outermost to innermost grouping, with
   level zero corresponding to the population fitted values. Defaults to
   the highest or innermost level of grouping.} 
 \item{conLin}{an optional condensed linear model object, consisting of
   a list with components \code{"Xy"}, corresponding to a regression
   matrix (\code{X}) combined with a response vector (\code{y}), and 
   \code{"logLik"}, corresponding to the log-likelihood of the
   underlying lme model. Defaults to \code{attr(object, "conLin")}.}
 \item{lmeFit}{an optional list with components \code{beta} and \code{b}
   containing respectively the fixed effects estimates and the random
   effects estimates to be used to calculate the fitted values. Defaults
   to \code{attr(object, "lmeFit")}.}
 \item{\dots}{some methods for this generic accept other optional arguments.}
}
\description{
  The fitted values at level \eqn{i} are obtained by adding together the
  population fitted values (based only on the fixed effects estimates)
  and the estimated contributions of the random effects to the fitted
  values at grouping levels less or equal to \eqn{i}. The resulting
  values estimate the best linear unbiased predictions (BLUPs) at level
  \eqn{i}. 
}
\value{
  if a single level of grouping is specified in \code{level},
  the returned value is a vector with the fitted values at the desired
  level; else, when multiple grouping levels are specified in
  \code{level}, the returned object is a matrix with 
  columns given by the fitted values at different levels. 
}
\references{
  Bates, D.M. and Pinheiro, J.C. (1998) "Computational methods for
  multilevel models" available in PostScript or PDF formats at
  http://franz.stat.wisc.edu/pub/NLME/
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  This method function is generally only used inside \code{lme} and 
  \code{fitted.lme}.
}

\seealso{\code{\link{lme}}, \code{\link{fitted.lme}},
  \code{\link{residuals.lmeStruct}} } 

\keyword{models}

\eof
% $Id: fitted.nlmeStruct.Rd,v 1.5.2.1 2003/04/17 22:28:45 bates Exp $
\name{fitted.nlmeStruct}
\title{Calculate nlmeStruct Fitted Values}
\usage{
\method{fitted}{nlmeStruct}(object, level, conLin, \dots)
}
\alias{fitted.nlmeStruct}
\arguments{
 \item{object}{an object inheriting from class \code{nlmeStruct},
   representing a list of mixed-effects model components, such as
   \code{reStruct}, \code{corStruct}, and \code{varFunc} objects, plus
   attributes  specifying the underlying nonlinear model and the
   response variable.} 
 \item{level}{an optional integer vector giving the level(s) of grouping
   to be used in extracting the fitted values from \code{object}. Level
   values increase from outermost to innermost grouping, with
   level zero corresponding to the population fitted values. Defaults to
   the highest or innermost level of grouping.} 
 \item{conLin}{an optional condensed linear model object, consisting of
   a list with components \code{"Xy"}, corresponding to a regression
   matrix (\code{X}) combined with a response vector (\code{y}), and 
   \code{"logLik"}, corresponding to the log-likelihood of the
   underlying nlme model. Defaults to \code{attr(object, "conLin")}.}
 \item{\dots}{additional arguments that could be given to this method.
   None are used.}
}
\description{
  The fitted values at level \eqn{i} are obtained by adding together the
  contributions from the estimated fixed effects and the estimated
  random effects at levels less or equal to \eqn{i} and evaluating the
  model function at the resulting estimated parameters. The resulting
  values estimate the predictions at level \eqn{i}. 
}
\value{
  if a single level of grouping is specified in \code{level},
  the returned value is a vector with the fitted values at the desired
  level; else, when multiple grouping levels are specified in
  \code{level}, the returned object is a matrix with 
  columns given by the fitted values at different levels. 
}
\references{
  Bates, D.M. and Pinheiro, J.C. (1998) "Computational methods for
  multilevel models" available in PostScript or PDF formats at
  http://nlme.stat.wisc.edu/pub/NLME/
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  This method function is generally only used inside \code{nlme} and 
  \code{fitted.nlme}.
}
\seealso{\code{\link{nlme}}, \code{\link{residuals.nlmeStruct}}} 
\keyword{models}

\eof
% $Id: fixed.effects.Rd,v 1.3.4.2 2003/05/17 21:57:49 bates Exp $
\name{fixed.effects}
\title{Extract Fixed Effects}
\usage{
fixed.effects(object, \dots)
fixef(object, \dots)
}
\alias{fixed.effects}
\alias{fixef}
\arguments{
 \item{object}{any fitted model object from which fixed effects
   estimates can be extracted.}
 \item{\dots}{some methods for this generic function require additional
   arguments.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include \code{lmList} and \code{lme}.
}
\value{
  will depend on the method function used; see the appropriate documentation.
}
\seealso{\code{\link{fixef.lmList}}} 
\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: fixef.lmList.Rd,v 1.4 2002/03/05 14:59:39 bates Exp $
\name{fixef.lmList}
\title{Extract lmList Fixed Effects}
\usage{
\method{fixef}{lmList}(object, \dots)
}
\alias{fixed.effects.lmList}
\alias{fixef.lmList}
\arguments{
 \item{object}{an object inheriting from class \code{lmList}, representing
   a list of \code{lm} objects with a common model.
 }
 \item{\dots}{some methods for this generic require additional
   arguments.  None are used in this method.} 
}
\description{
  The average of the coefficients corresponding to the \code{lm}
  components of \code{object} is calculated.
}
\value{
  a vector with the average of the individual \code{lm} coefficients in
  \code{object}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lmList}}, \code{\link{random.effects.lmList}}}

\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age | Subject, Orthodont)
fixed.effects(fm1)
}
\keyword{models}

\eof
% $Id: formula.pdBlocked.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{formula.pdBlocked}
\title{Extract pdBlocked Formula}
\usage{
\method{formula}{pdBlocked}(x, asList, \dots)
}
\alias{formula.pdBlocked}
\arguments{
 \item{x}{an object inheriting from class \code{pdBlocked},
   representing a positive definite block diagonal matrix.} 
 \item{asList}{an optional logical value. If \code{TRUE}, a list with
   the formulas for the individual block diagonal elements of
   \code{x} is returned; else, if \code{FALSE}, a one-sided formula
   combining all individual formulas is returned. Defaults to
   \code{FALSE}.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The \code{formula} attributes of the \code{pdMat} elements of
  \code{x} are extracted and returned as a list, in case
  \code{asList=TRUE}, or converted to a single one-sided formula when
  \code{asList=FALSE}. If the \code{pdMat} elements do not have a
  \code{formula} attribute, a \code{NULL} value is returned.
}
\value{
  a list of one-sided formulas, or a single one-sided formula, or
  \code{NULL}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}


\seealso{\code{\link{pdBlocked}}, \code{\link{pdMat}}}

\examples{
pd1 <- pdBlocked(list(~ age, ~ Sex - 1))
formula(pd1)
formula(pd1, asList = TRUE)
}
\keyword{models}

\eof
% $Id: formula.pdMat.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{formula.pdMat}
\title{Extract pdMat Formula}
\usage{
\method{formula}{pdMat}(x, asList, \dots)
}
\alias{formula.pdMat}
\arguments{
 \item{x}{an object inheriting from class \code{pdMat}, representing
   a positive definite matrix.}
 \item{asList}{logical.  Should the asList argument be applied to each
   of the components?  Never used.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts the formula associated with a
  \code{pdMat} object, in which the column and row names are specified.
}
\value{
  if \code{x} has a \code{formula} attribute, its value is
  returned, else \code{NULL} is returned.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\note{
  Because factors may be present in \code{formula(x)}, the
  \code{pdMat} object needs to have access to a data frame where the
  variables named in the formula can be evaluated, before it can resolve
  its row and column names from the formula.
}

\seealso{\code{\link{pdMat}}}

\examples{
pd1 <- pdSymm(~Sex*age)
formula(pd1)
}
\keyword{models}

\eof
% $Id: formula.reStruct.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{formula.reStruct}
\title{Extract reStruct Object Formula}
\usage{
\method{formula}{reStruct}(x, asList, \dots)
}
\alias{formula.reStruct}
\arguments{
 \item{x}{an object inheriting from class \code{reStruct},
   representing a random effects structure and consisting of a list of
   \code{pdMat} objects.}
 \item{asList}{logical.  Should the asList argument be applied to each
   of the components?}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts a formula from each of the 
  components of \code{x}, returning a list of formulas.
}
\value{
  a list with the formulas of each component of \code{x}.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{formula}}}

\examples{
rs1 <- reStruct(list(A = pdDiag(diag(2), ~age), B = ~1))
formula(rs1)
}
\keyword{models}

\eof
% $Id: gapply.Rd,v 1.7.2.1 2003/04/17 22:28:45 bates Exp $
\name{gapply}
\title{Apply a Function by Groups}
\usage{
gapply(object, which, FUN, form, level, groups, \dots)
}
\alias{gapply}
\arguments{
 \item{object}{an object to which the function will be applied - usually
   a \code{groupedData} object or a \code{data.frame}. Must inherit from
   class \code{data.frame}.
 }
 \item{which}{an optional character or positive integer vector
   specifying which columns of \code{object} should be used with
   \code{FUN}. Defaults to all columns in \code{object}.
 }
 \item{FUN}{function to apply to the distinct sets of rows
   of the data frame \code{object} defined by the values of
   \code{groups}.
 } 
 \item{form}{an optional one-sided formula that defines the groups.
   When this formula is given the right-hand side is evaluated in
   \code{object}, converted to a factor if necessary, and the unique
   levels are used to define the groups.  Defaults to
   \code{formula(object)}.
 }
 \item{level}{an optional positive integer giving the level of grouping
   to be used in an object with multiple nested grouping levels.
   Defaults to the highest or innermost level of grouping.
 }
 \item{groups}{an optional factor that will be used to split the 
   rows into groups.  Defaults to \code{getGroups(object, form, level)}.
 }
 \item{\dots}{optional additional arguments to the summary function
   \code{FUN}.  Often it is helpful to specify \code{na.rm = TRUE}.
 }
}

\description{
  Applies the function to the distinct sets of rows of the data frame
  defined by \code{groups}.
}

\value{
  Returns a data frame with as many rows as there are levels in the
  \code{groups} argument.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gsummary}}}

\examples{
data( Phenobarb )
## Find number of non-missing "conc" observations for each Subject
gapply( Phenobarb, FUN = function(x) sum(!is.na(x$conc)) )
}
\keyword{data}

\eof
% $Id: getCovariate.Rd,v 1.4 2002/03/05 14:59:39 bates Exp $
\name{getCovariate}
\title{Extract Covariate from an Object}
\usage{
getCovariate(object, form, data)
}
\alias{getCovariate}
\arguments{
 \item{object}{any object with a \code{covariate} component}
 \item{form}{an optional one-sided formula specifying the covariate(s)
   to be extracted. Defaults to \code{formula(object)}.}
 \item{data}{a data frame in which to evaluate the variables defined in
   \code{form}. }
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include \code{corStruct}, \code{corSpatial},
  \code{data.frame}, and \code{varFunc}.
}
\value{
  will depend on the method function used; see the appropriate documentation.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{
    \code{\link{getCovariate.corStruct}},
    \code{\link{getCovariate.data.frame}},
    \code{\link{getCovariate.varFunc}},
    \code{\link{getCovariateFormula}}
}
\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: getCovariate.corStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{getCovariate.corStruct}
\title{Extract corStruct Object Covariate}
\usage{
\method{getCovariate}{corStruct}(object, form, data)
}
\alias{getCovariate.corStruct}
\alias{getCovariate.corSpatial}
\arguments{
 \item{object}{an object inheriting from class \code{corStruct}
   representing a correlation structure.}
 \item{form}{this argument is included to make the method function
   compatible with the generic. It will be assigned the value of
   \code{formula(object)} and should not be modified.}
 \item{data}{an optional data frame in which to evaluate the variables
   defined in \code{form}, in case \code{object} is not initialized and
   the covariate needs to be evaluated.}
}
\description{
  This method function extracts the covariate(s) associated with
  \code{object}.
}
\value{
  when the correlation structure does not include a grouping factor, the
  returned value will be a vector or a matrix with the covariate(s)
  associated with \code{object}. If a grouping factor is present, the
  returned value will be a list of vectors or matrices with the
  covariate(s) corresponding to each grouping level.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{getCovariate}}}

\examples{
data(Orthodont)
cs1 <- corAR1(form = ~ 1 | Subject)
getCovariate(cs1, data = Orthodont)
}
\keyword{models}

\eof
% $Id: getCovariate.data.frame.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{getCovariate.data.frame}
\title{Extract Data Frame Covariate}
\usage{
\method{getCovariate}{data.frame}(object, form, data)
}
\alias{getCovariate.data.frame}
\arguments{
 \item{object}{an object inheriting from class \code{data.frame}.}
 \item{form}{an optional formula specifying the covariate to be
   evaluated in \code{object}. Defaults to \code{formula(object)}.}
 \item{data}{some methods for this generic require a separate data
   frame. Not used in this method.}
}
\description{
  The right hand side of \code{form}, stripped of any conditioning
  expression (i.e. an expression following a \code{|} operator), is
  evaluated in \code{object}.  
}
\value{
  the value of the right hand side of \code{form}, stripped of
  any conditional expression, evaluated in \code{object}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{getCovariateFormula}}}

\examples{
data(Orthodont)
getCovariate(Orthodont)
}
\keyword{models}

\eof
% $Id: getCovariate.varFunc.Rd,v 1.7.2.1 2004/01/14 21:12:17 bates Exp $
\name{getCovariate.varFunc}
\title{Extract varFunc Covariate}
\usage{
\method{getCovariate}{varFunc}(object, form, data)
}
\alias{getCovariate.varFunc}
\arguments{
 \item{object}{an object inheriting from class \code{varFunc},
   representing a variance function structure.}
 \item{form}{an optional formula specifying the covariate to be
   evaluated in \code{object}. Defaults to \code{formula(object)}.}
 \item{data}{some methods for this generic require a \code{data}
   object.  Not used in this method.} 
}
\description{
  This method function extracts the covariate(s) associated with the
  variance function represented by \code{object}, if any is present.
}
\value{
  if \code{object} has a \code{covariate} attribute, its value is
  returned; else \code{NULL} is returned.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{covariate<-.varFunc}}}

\examples{
data(Orthodont)
vf1 <- varPower(1.1, form = ~age)
covariate(vf1) <- Orthodont[["age"]]
getCovariate(vf1)
}
\keyword{models}

\eof
\name{getCovariateFormula}
\title{Extract Covariates Formula}
\usage{
getCovariateFormula(object)
}
\alias{getCovariateFormula}
\arguments{
 \item{object}{any object from which a formula can be extracted.}
}
\description{
  The right hand side of \code{formula(object)}, without any
  conditioning expressions (i.e. any expressions after a \code{|}
  operator) is returned as a one-sided formula.
}
\value{
  a one-sided formula describing the covariates associated with
  \code{formula(object)}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{getCovariate}}}

\examples{
getCovariateFormula(y ~ x | g)
getCovariateFormula(y ~ x)
}
\keyword{models}


\eof
% $Id: getData.Rd,v 1.4 2002/03/05 14:59:39 bates Exp $
\name{getData}
\title{Extract Data from an Object}
\usage{
getData(object)
}
\alias{getData}
\arguments{
 \item{object}{an object from which a data.frame can be extracted,
   generally a fitted model object.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include \code{gls}, \code{lme},
  and \code{lmList}.
}
\value{
  will depend on the method function used; see the appropriate documentation.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: getData.gls.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{getData.gls}
\title{Extract gls Object Data}
\usage{
\method{getData}{gls}(object)
}
\alias{getData.gls}
\alias{getData.gnls}
\arguments{
 \item{object}{an object inheriting from class \code{gls}, representing
   a generalized least squares fitted linear model.}
}
\description{
  If present in the calling sequence used to produce \code{object}, the
  data frame used to fit the model is obtained. 
}
\value{
  if a \code{data} argument is present in the calling sequence that
  produced \code{object}, the corresponding data frame (with
  \code{na.action} and \code{subset} applied to it, if also present in
  the call that produced \code{object}) is returned;
  else, \code{NULL} is returned.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gls}}}

\examples{
data(Ovary)
fm1 <- gls(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), data = Ovary,
           correlation = corAR1(form = ~ 1 | Mare))
getData(fm1)
}
\keyword{models}

\eof
% $Id: getData.lmList.Rd,v 1.5.2.1 2003/08/09 22:45:17 bates Exp $
\name{getData.lmList}
\title{Extract lmList Object Data}
\usage{
\method{getData}{lmList}(object)
}
\alias{getData.lmList}
\arguments{
 \item{object}{an object inheriting from class \code{lmList}, representing
   a list of \code{lm} objects with a common model.
 }
}
\description{
  If present in the calling sequence used to produce \code{object}, the
  data frame used to fit the model is obtained. 
}
\value{
  if a \code{data} argument is present in the calling sequence that
  produced \code{object}, the corresponding data frame (with
  \code{na.action} and \code{subset} applied to it, if also present in
  the call that produced \code{object}) is returned;
  else, \code{NULL} is returned.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lmList}}}

\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age | Subject, Orthodont)
getData(fm1)
}
\keyword{models}

\eof
% $Id: getData.lme.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{getData.lme}
\title{Extract lme Object Data}
\usage{
\method{getData}{lme}(object)
}
\alias{getData.lme}
\alias{getData.nlme}
\alias{getData.nls}
\arguments{
 \item{object}{an object inheriting from class \code{lme}, representing
   a linear mixed-effects fitted model.}
}
\description{
  If present in the calling sequence used to produce \code{object}, the
  data frame used to fit the model is obtained. 
}
\value{
  if a \code{data} argument is present in the calling sequence that
  produced \code{object}, the corresponding data frame (with
  \code{na.action} and \code{subset} applied to it, if also present in
  the call that produced \code{object}) is returned;
  else, \code{NULL} is returned.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lme}}}

\examples{
data(Ovary)
fm1 <- lme(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), data = Ovary,
           random = ~ sin(2*pi*Time))
getData(fm1)
}
\keyword{models}

\eof
% $Id: getGroups.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{getGroups}
\title{Extract Grouping Factors from an Object}
\usage{
getGroups(object, form, level, data, sep)
}
\alias{getGroups}
\arguments{
 \item{object}{any object}
 \item{form}{an optional formula with a conditioning expression on its
   right hand side (i.e. an expression involving the \code{|}
   operator). Defaults to \code{formula(object)}.}
 \item{level}{a positive integer vector with the level(s) of grouping to
   be used when multiple nested levels of grouping are present. This
   argument is optional for most methods of this generic function and
   defaults to all levels of nesting.}
 \item{data}{a data frame in which to interpret the variables named in
   \code{form}. Optional for most methods.}
 \item{sep}{character, the separator to use between group levels when
   multiple levels are collapsed.  The default is \code{'/'}.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include \code{corStruct}, \code{data.frame},
  \code{gls}, \code{lme}, \code{lmList}, and \code{varFunc}.
}
\value{
  will depend on the method function used; see the appropriate documentation.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{getGroupsFormula}}, \code{\link{getGroups.data.frame}},
\code{\link{getGroups.gls}}, \code{\link{getGroups.lmList}},
\code{\link{getGroups.lme}}}
\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: getGroups.corStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{getGroups.corStruct}
\title{Extract corStruct Groups}
\usage{
\method{getGroups}{corStruct}(object, form, level, data, sep)
}
\alias{getGroups.corStruct}
\arguments{
 \item{object}{an object inheriting from class \code{corStruct}
   representing a correlation structure.}
 \item{form}{this argument is included to make the method function
   compatible with the generic. It will be assigned the value of
   \code{formula(object)} and should not be modified.}
 \item{level}{this argument is included to make the method function
   compatible with the generic and is not used.}
 \item{data}{an optional data frame in which to evaluate the variables
   defined in \code{form}, in case \code{object} is not initialized and
   the grouping factor needs to be evaluated.}
 \item{sep}{character, the separator to use between group levels when
   multiple levels are collapsed.  The default is \code{'/'}.}
}
\description{
  This method function extracts the grouping factor associated with
  \code{object}, if any is present.
}
\value{
  if a grouping factor is present in the correlation structure
  represented by \code{object}, the function returns the corresponding
  factor vector; else the function returns \code{NULL}.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{getGroups}}}

\examples{
data(Orthodont)
cs1 <- corAR1(form = ~ 1 | Subject)
getGroups(cs1, data = Orthodont)
}
\keyword{models}

\eof
% $Id: getGroups.data.frame.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{getGroups.data.frame}
\title{Extract Groups from a Data Frame}
\usage{
\method{getGroups}{data.frame}(object, form, level, data, sep)
}
\alias{getGroups.data.frame}
\arguments{
 \item{object}{an object inheriting from class \code{data.frame}.}
 \item{form}{an optional formula with a conditioning expression on its
   right hand side (i.e. an expression involving the \code{|}
   operator). Defaults to \code{formula(object)}.}  
 \item{level}{a positive integer vector with the level(s) of grouping to
   be used when multiple nested levels of grouping are present. Defaults
   to all levels of nesting.}
 \item{data}{unused}
 \item{sep}{character, the separator to use between group levels when
   multiple levels are collapsed.  The default is \code{'/'}.}
}
\description{
  Each variable named in the expression after the \code{|} operator on
  the right hand side of \code{form} is evaluated in \code{object}. If
  more than one variable is indicated in \code{level} they are combined
  into a data frame; else the selected variable is returned as a vector.
  When multiple grouping levels are defined in \code{form} and
  \code{level > 1}, the levels of the returned factor are obtained by
  pasting together the levels of the grouping factors of level greater
  or equal to \code{level}, to ensure their uniqueness. 
}
\value{
  either a data frame with columns given by the grouping factors
  indicated in \code{level}, from outer to inner, or, when a single
  level is requested, a factor representing the selected grouping
  factor. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{getGroupsFormula}}}
\examples{
data(Pixel)
getGroups(Pixel)
getGroups(Pixel, level = 2)
}
\keyword{models}

\eof
% $Id: getGroups.gls.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{getGroups.gls}
\title{Extract gls Object Groups}
\usage{
\method{getGroups}{gls}(object, form, level, data, sep)
}
\alias{getGroups.gls}
\arguments{
 \item{object}{an object inheriting from class \code{gls}, representing
   a generalized least squares fitted linear model.}
 \item{form}{an optional formula with a conditioning expression on its
   right hand side (i.e. an expression involving the \code{|}
   operator). Defaults to \code{formula(object)}.  Not used.}
 \item{level}{a positive integer vector with the level(s) of grouping to
   be used when multiple nested levels of grouping are present. This
   argument is optional for most methods of this generic function and
   defaults to all levels of nesting.  Not used.}
 \item{data}{a data frame in which to interpret the variables named in
   \code{form}. Optional for most methods.  Not used.}
 \item{sep}{character, the separator to use between group levels when
   multiple levels are collapsed.  The default is \code{'/'}.  Not used.}
}
\description{
  If present, the grouping factor associated to the correlation
  structure for the linear model represented by \code{object} is extracted. 
}
\value{
  if the linear model represented by \code{object} incorporates a
  correlation structure and the corresponding \code{corStruct} object
  has a grouping factor, a vector with the group values is returned;
  else, \code{NULL} is returned.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gls}}, \code{\link{corClasses}}}

\examples{
data(Ovary)
fm1 <- gls(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), Ovary,
           correlation = corAR1(form = ~ 1 | Mare))
getGroups(fm1)
}
\keyword{models}

\eof
% $Id: getGroups.lmList.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{getGroups.lmList}
\title{Extract lmList Object Groups}
\usage{
\method{getGroups}{lmList}(object, form, level, data, sep)
}
\alias{getGroups.lmList}
\arguments{
 \item{object}{an object inheriting from class \code{lmList}, representing
   a list of \code{lm} objects with a common model.}
 \item{form}{an optional formula with a conditioning expression on its
   right hand side (i.e. an expression involving the \code{|}
   operator). Defaults to \code{formula(object)}.  Not used.}
 \item{level}{a positive integer vector with the level(s) of grouping to
   be used when multiple nested levels of grouping are present. This
   argument is optional for most methods of this generic function and
   defaults to all levels of nesting.  Not used.}
 \item{data}{a data frame in which to interpret the variables named in
   \code{form}. Optional for most methods.  Not used.}
 \item{sep}{character, the separator to use between group levels when
   multiple levels are collapsed.  The default is \code{'/'}.  Not used.}
}
\description{
  The grouping factor determining the partitioning of the observations
  used to produce the \code{lm} components of \code{object} is
  extracted.  
}
\value{
  a vector with the grouping factor corresponding to the \code{lm}
  components of \code{object}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lmList}}}

\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age | Subject, Orthodont)
getGroups(fm1)
}
\keyword{models}

\eof
% $Id: getGroups.lme.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{getGroups.lme}
\title{Extract lme Object Groups}
\usage{
\method{getGroups}{lme}(object, form, level, data, sep)
}
\alias{getGroups.lme}
\arguments{
 \item{object}{an object inheriting from class \code{lme}, representing
   a fitted linear mixed-effects model.}
 \item{form}{this argument is included to make the method function
   compatible with the generic and is ignored in this method.}
 \item{level}{an optional integer vector giving the level(s) of grouping
   to be extracted from \code{object}. Defaults to the highest or
   innermost level of grouping.}  
 \item{data}{unused}
 \item{sep}{character, the separator to use between group levels when
   multiple levels are collapsed.  The default is \code{'/'}.}
}
\description{
  The grouping factors corresponding to the linear mixed-effects model
  represented by \code{object} are extracted. If more than one level is
  indicated in \code{level}, the corresponding grouping factors are
  combined  into a data frame; else the selected grouping factor is
  returned as a vector. 
}
\value{
  either a data frame with columns given by the grouping factors
  indicated in \code{level}, or, when a single level is requested, a
  factor representing the selected grouping factor. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lme}}}

\examples{
data(Pixel)
fm1 <- lme(pixel ~ day + day^2, Pixel,
  random = list(Dog = ~day, Side = ~1))
getGroups(fm1, level = 1:2)
}
\keyword{models}

\eof
% $Id: getGroups.varFunc.Rd,v 1.6.2.1 2002/08/09 19:45:29 bates Exp $
\name{getGroups.varFunc}
\title{Extract varFunc Groups}
\usage{
\method{getGroups}{varFunc}(object, form, level, data, sep)
}
\alias{getGroups.varFunc}
\arguments{
 \item{object}{an object inheriting from class \code{varFunc},
   representing a variance function structure.}
 \item{form}{an optional formula with a conditioning expression on its
   right hand side (i.e. an expression involving the \code{|}
   operator). Defaults to \code{formula(object)}.  Not used.}
 \item{level}{a positive integer vector with the level(s) of grouping to
   be used when multiple nested levels of grouping are present. This
   argument is optional for most methods of this generic function and
   defaults to all levels of nesting.  Not used.}
 \item{data}{a data frame in which to interpret the variables named in
   \code{form}. Optional for most methods.  Not used.}
 \item{sep}{character, the separator to use between group levels when
   multiple levels are collapsed.  The default is \code{'/'}.  Not used.}
}
\description{
  This method function extracts the grouping factor associated with the
  variance function represented by \code{object}, if any is present.
}
\value{
  if \code{object} has a \code{groups} attribute, its value is
  returned; else \code{NULL} is returned.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\examples{
data(Orthodont)
vf1 <- varPower(form = ~ age | Sex)
vf1 <- Initialize(vf1, Orthodont)
getGroups(vf1)
}
\keyword{models}

\eof
% $Id: getGroupsFormula.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{getGroupsFormula}
\title{Extract Grouping Formula}
\usage{
getGroupsFormula(object, asList, sep)
}
\alias{getGroupsFormula}
\alias{getGroupsFormula.default}
\alias{getGroupsFormula.gls}
\alias{getGroupsFormula.lmList}
\alias{getGroupsFormula.lme}
\alias{getGroupsFormula.reStruct}
\arguments{
 \item{object}{any object from which a formula can be extracted.}
 \item{asList}{an optional logical value. If \code{TRUE} the returned
   value with be a list of formulas; else, if \code{FALSE} the returned
   value will be a one-sided formula. Defaults to \code{FALSE}.}
 \item{sep}{character, the separator to use between group levels when
   multiple levels are collapsed.  The default is \code{'/'}.}
}
\description{
  The conditioning expression associated with \code{formula(object)}
  (i.e. the expression after the \code{|} operator) is returned either as
  a named list of one-sided formulas, or a single one-sided formula,
  depending on the value of \code{asList}. The components of the
  returned list are ordered from outermost to innermost level and are
  named after the grouping factor expression.
}
\value{
  a one-sided formula, or a list of one-sided formulas, with the
  grouping structure associated with \code{formula(object)}. If no
  conditioning expression is present in \code{formula(object)} a
  \code{NULL} value is returned.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{
    \code{\link{getGroupsFormula.gls}},
    \code{\link{getGroupsFormula.lmList}},
    \code{\link{getGroupsFormula.lme}},
    \code{\link{getGroupsFormula.reStruct}},
    \code{\link{getGroups}}
}
\examples{
getGroupsFormula(y ~ x | g1/g2)
}
\keyword{models}

\eof
% $Id: getResponse.Rd,v 1.5.2.1 2003/04/17 22:28:45 bates Exp $
\name{getResponse}
\title{Extract Response Variable from an Object}
\usage{
getResponse(object, form)
}
\alias{getResponse}
\alias{getResponse.data.frame}
\arguments{
 \item{object}{any object}
 \item{form}{an optional two-sided formula. Defaults to
   \code{formula(object)}.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include \code{data.frame}, \code{gls}, \code{lme},
  and \code{lmList}.
}
\value{
  will depend on the method function used; see the appropriate documentation.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{getResponseFormula}}}
\examples{
data(Orthodont)
getResponse(Orthodont)
}
\keyword{models}

\eof
\name{getResponseFormula}
\title{Extract Formula Specifying Response Variable}
\usage{
getResponseFormula(object)
}
\alias{getResponseFormula}
\arguments{
 \item{object}{any object from which a formula can be extracted.}
}
\description{
  The left hand side of \code{formula{object}} is returned as a
  one-sided formula. 
}
\value{
  a one-sided formula with the response variable associated with
  \code{formula{object}}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{getResponse}}}

\examples{
getResponseFormula(y ~ x | g)
}
\keyword{models}


\eof
\name{getVarCov}
\alias{getVarCov}
\alias{getVarCov.lme}
\alias{getVarCov.gls}
\alias{print.VarCov}
\title{Extract variance-covariance matrix}
\description{
  Extract the variance-covariance matrix from a fitted model, such as a
  mixed-effects model.
}
\usage{
getVarCov(obj, \dots)
\method{getVarCov}{lme}(obj, individuals,
    type = c("random.effects", "conditional", "marginal"), \dots)
\method{getVarCov}{gls}(obj, individual = 1, \dots)
}
\arguments{
  \item{obj}{A fitted model.  Methods are available for models fit by
    \code{\link{lme}} and by \code{\link{gls}}}
  \item{individuals}{For models fit by \code{\link{lme}} a vector of
    levels of the grouping factor can be specified for the conditional
    or marginal variance-covariance matrices.}
  \item{individual}{For models fit by \code{\link{gls}} the only type of
    variance-covariance matrix provided is the marginal
    variance-covariance of the responses by group.  The
    optional argument \code{individual} specifies the group of responses.}
  \item{type}{For models fit by \code{\link{lme}} the \code{type}
    argument specifies the type of variance-covariance matrix, either
    \code{"random.effects"} for the random-effects variance-covariance
    (the default), or \code{"conditional"} for the conditional.
    variance-covariance of the responses or \code{"marginal"} for the
    the marginal variance-covariance of the responses.}
  \item{\dots}{Optional arguments for some methods, as described above}
}
\value{
  A variance-covariance matrix or a list of variance-covariance matrices.
}
\author{Mary Lindstrom \email{lindstro@biostat.wisc.edu}}
\seealso{\code{\link{lme}}, \code{\link{gls}}}
\examples{
data(Orthodont)
fm1 <- lme(distance ~ age, data = Orthodont, subset = Sex == "Female")
getVarCov(fm1)
getVarCov(fm1, individual = "F01", type = "marginal")
getVarCov(fm1, type = "conditional")
data(Ovary)
fm2 <- gls(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), Ovary,
           correlation = corAR1(form = ~ 1 | Mare))
getVarCov(fm2)
}
\keyword{models}

\eof
% $Id: gls.Rd,v 1.6 2002/01/28 03:44:17 bates Exp $
\name{gls}
\title{Fit Linear Model Using Generalized Least Squares}
\usage{
gls(model, data, correlation, weights, subset, method, na.action,
    control, verbose)
\method{update}{gls}(object, model, data, correlation, weights, subset,
        method, na.action, control, verbose, \dots)
}
\alias{gls}
\alias{update.gls}
\arguments{
 \item{object}{an object inheriting from class \code{gls}, representing
   a generalized least squares fitted linear model.}
  \item{model}{a two-sided linear formula object describing the
    model, with the response on the left of a \code{~} operator and the
    terms, separated by \code{+} operators, on the right.}
 \item{data}{an optional data frame containing the variables named in
   \code{model}, \code{correlation}, \code{weights}, and
   \code{subset}. By default the variables are taken from the
   environment from which \code{gls} is called.}
 \item{correlation}{an optional \code{corStruct} object describing the
   within-group correlation structure. See the documentation of
   \code{corClasses} for a description of the available \code{corStruct}
   classes. If a grouping variable is to be used, it must be specified in
   the \code{form} argument to the \code{corStruct}
   constructor. Defaults to \code{NULL}, corresponding to uncorrelated 
   errors.}  
 \item{weights}{an optional \code{varFunc} object or one-sided formula
   describing the within-group heteroscedasticity structure. If given as
   a formula, it is used as the argument to \code{varFixed},
   corresponding to fixed variance weights. See the documentation on
   \code{varClasses} for a description of the available \code{varFunc}
   classes. Defaults to \code{NULL}, corresponding to homoscesdatic
   errors.} 
 \item{subset}{an optional expression indicating which subset of the rows of
   \code{data} should  be  used in the fit. This can be a logical
   vector, or a numeric vector indicating which observation numbers are
   to be included, or a  character  vector of the row names to be
   included.  All observations are included by default.}
 \item{method}{a character string.  If \code{"REML"} the model is fit by
   maximizing the restricted log-likelihood.  If \code{"ML"} the
   log-likelihood is maximized.  Defaults to \code{"REML"}.}
 \item{na.action}{a function that indicates what should happen when the
   data contain \code{NA}s.  The default action (\code{na.fail}) causes
   \code{gls} to print an error message and terminate if there are any
   incomplete observations.}
 \item{control}{a list of control values for the estimation algorithm to
   replace the default values returned by the function \code{glsControl}.
   Defaults to an empty list.}
 \item{verbose}{an optional logical value. If \code{TRUE} information on
   the evolution of the iterative algorithm is printed. Default is
   \code{FALSE}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This function fits a linear model using generalized least
  squares. The errors are allowed to be correlated and/or have unequal
  variances.  
}
\value{
  an object of class \code{gls} representing the linear model
  fit. Generic functions such as \code{print}, \code{plot}, and 
  \code{summary} have methods to show the results of the fit. See
  \code{glsObject} for the components of the fit. The functions
  \code{resid}, \code{coef}, and \code{fitted} can be used to extract
  some of its components. 
}
\references{
 The different correlation structures available for the
 \code{correlation} argument are described in Box, G.E.P., Jenkins,
 G.M., and Reinsel G.C. (1994), Littel, R.C., Milliken, G.A., Stroup,
 W.W., and Wolfinger, R.D. (1996), and Venables, W.N. and Ripley,
 B.D. (1997). The use of variance functions for linear 
 and nonlinear models is presented in detail in Carroll, R.J. and Ruppert,
 D. (1988) and Davidian, M. and Giltinan, D.M. (1995).  

 Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994) "Time Series
 Analysis: Forecasting and Control", 3rd Edition, Holden-Day. 

 Carroll, R.J. and Ruppert, D. (1988) "Transformation and Weighting in
 Regression", Chapman and Hall.

 Davidian, M. and Giltinan, D.M. (1995) "Nonlinear Mixed Effects Models
 for Repeated Measurement Data", Chapman and Hall.

 Littel, R.C., Milliken, G.A., Stroup, W.W., and Wolfinger, R.D. (1996)
 "SAS Systems for Mixed Models", SAS Institute.

 Venables, W.N. and Ripley, B.D. (1997) "Modern Applied Statistics with
 S-PLUS", 2nd Edition, Springer-Verlag.
}
\author{Jose Pinheiro \email{jcp@research.bell-labs.com}, Douglas Bates
  \email{bates@stat.wisc.edu}} 
\seealso{
  \code{\link{glsControl}}, \code{\link{glsObject}},
  \code{\link{varFunc}}, \code{\link{corClasses}}, \code{\link{varClasses}}
}
\examples{
data(Ovary)
# AR(1) errors within each Mare
fm1 <- gls(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), Ovary,
           correlation = corAR1(form = ~ 1 | Mare))
# variance increases as a power of the absolute fitted values
fm2 <- update(fm1, weights = varPower())
}
\keyword{models}

\eof
% $Id: glsControl.Rd,v 1.8 2002/03/05 14:59:39 bates Exp $
\name{glsControl}
\title{Control Values for gls Fit}
\usage{
glsControl(maxIter, msMaxIter, tolerance, msTol, msScale, msVerbose,
           singular.ok, qrTol, returnObject, apVar, .relStep,
           nlmStepMax, minAbsParApVar, natural)
}
\alias{glsControl}
\arguments{
 \item{maxIter}{maximum number of iterations for the \code{gls}
   optimization algorithm. Default is 50.} 
 \item{msMaxIter}{maximum number of iterations
   for the optimization step inside the \code{gls}
   optimization. Default is 50.}
 \item{tolerance}{tolerance for the convergence criterion in the
   \code{gls} algorithm. Default is 1e-6.}
 \item{msTol}{tolerance for the convergence criterion in \code{ms},
   passed as the \code{rel.tolerance} argument to the function (see
   documentation on \code{ms}). Default is 1e-7.}
 \item{msScale}{scale function passed as the \code{scale} argument to
   the \code{ms} function (see documentation on that function). Default
   is \code{lmeScale}.}
 \item{msVerbose}{a logical value passed as the \code{trace} argument to
   \code{ms} (see documentation on that function). Default is
   \code{FALSE}.}
 \item{singular.ok}{a logical value indicating whether non-estimable
   coefficients (resulting from linear dependencies among the columns of
   the regression matrix) should be allowed. Default is \code{FALSE}.}
 \item{qrTol}{a tolerance for detecting linear dependencies among the
   columns of the regression matrix in its QR decomposition. Default is
   \code{.Machine$single.eps}.}
 \item{returnObject}{a logical value indicating whether the fitted
   object should be returned when the maximum number of iterations is
   reached without convergence of the algorithm. Default is
   \code{FALSE}.} 
 \item{apVar}{a logical value indicating whether the approximate
   covariance matrix of the variance-covariance parameters should be
   calculated. Default is \code{TRUE}.}
 \item{.relStep}{relative step for numerical derivatives
   calculations. Default is \code{.Machine$double.eps^(1/3)}.}
 \item{nlmStepMax}{stepmax value to be passed to nlm. See
   \code{\link{nlm}} for details. Default is 100.0}
 \item{minAbsParApVar}{numeric value - minimum absolute parameter value
   in the approximate variance calculation.  The default is \code{0.05}.}
 \item{natural}{logical.  Should the natural parameterization be used
   for the approximate variance calculations?  Default is \code{TRUE}.}
}
\description{
  The values supplied in the function call replace the defaults and a
  list with all possible arguments is returned. The returned list is
  used as the \code{control} argument to the \code{gls} function.
}
\value{
  a list with components for each of the possible arguments.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gls}}, \code{\link{lmeScale}}}

\examples{
# decrease the maximum number iterations in the optimization call and
# request that information on the evolution of the ms iterations be printed
glsControl(msMaxIter = 20, msVerbose = TRUE)

}
\keyword{models}

\eof
% $Id: glsObject.Rd,v 1.4 2002/03/05 14:59:39 bates Exp $
\name{glsObject}
\title{Fitted gls Object}
\alias{glsObject}
\description{
  An object returned by the \code{gls} function, inheriting from class
  \code{gls} and representing a generalized least squares fitted linear 
  model. Objects of this class have methods for the generic functions 
  \code{anova}, \code{coef}, \code{fitted}, \code{formula},
  \code{getGroups}, \code{getResponse}, \code{intervals}, \code{logLik},
  \code{plot}, \code{predict}, \code{print}, \code{residuals},
  \code{summary}, and \code{update}.   

}
\value{
  The following components must be included in a legitimate \code{gls}
  object. 
  \item{apVar}{an approximate covariance matrix for the
    variance-covariance coefficients. If \code{apVar = FALSE} in the list
    of control values used in the call to \code{gls}, this
    component is equal to \code{NULL}.}
  \item{call}{a list containing an image of the \code{gls} call that
    produced the object.}
  \item{coefficients}{a vector with the estimated linear model
    coefficients.} 
  \item{contrasts}{a list with the contrasts used to represent factors
    in the model formula. This information is important for making
    predictions from a new data frame in which not all levels of the
    original factors are observed. If no factors are used in the model,
    this component will be an empty list.}
  \item{dims}{a list with basic dimensions used in the model fit,
    including the components \code{N} - the number of observations in
    the data and \code{p} - the number of coefficients in the linear
    model.}
  \item{fitted}{a vector with the fitted values..}
  \item{glsStruct}{an object inheriting from class \code{glsStruct},
   representing a list of linear model components, such as
   \code{corStruct} and \code{varFunc} objects.}
  \item{groups}{a vector with the correlation structure grouping factor,
    if any is present.}
  \item{logLik}{the log-likelihood at convergence.}
  \item{method}{the estimation method: either \code{"ML"} for maximum
    likelihood, or \code{"REML"} for restricted maximum likelihood.}
  \item{numIter}{the number of iterations used in the iterative
    algorithm.} 
  \item{residuals}{a vector with the residuals.} 
  \item{sigma}{the estimated residual standard error.}
  \item{varBeta}{an approximate covariance matrix of the
    coefficients estimates.}
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gls}}, \code{glsStruct}}

\keyword{models}

\eof
% $Id: glsStruct.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{glsStruct}
\title{Generalized Least Squares Structure}
\usage{
glsStruct(corStruct, varStruct)
}
\alias{glsStruct}
\arguments{
 \item{corStruct}{an optional \code{corStruct} object, representing a
   correlation structure. Default is \code{NULL}.}
 \item{varStruct}{an optional \code{varFunc} object, representing a
   variance function structure. Default is \code{NULL}.}
}
\description{
  A generalized least squares structure is a list of model components
  representing different sets of parameters in the linear 
  model. A \code{glsStruct}  may contain \code{corStruct} and
  \code{varFunc} objects. \code{NULL} arguments are not included in the
  \code{glsStruct} list. 
}
\value{
  a list of model variance-covariance components determining the
  parameters to be estimated for the associated linear model.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gls}}, \code{\link{corClasses}},
  \code{\link{varFunc}} } 

\examples{
gls1 <- glsStruct(corAR1(), varPower())
}
\keyword{models}

\eof
% $Id: gnls.Rd,v 1.7.2.4 2003/08/09 22:45:17 bates Exp $
\name{gnls}
\title{Fit Nonlinear Model Using Generalized Least Squares}
\usage{
gnls(model, data, params, start, correlation, weights, subset,
     na.action, naPattern, control, verbose)
%\method{update}{gnls}(object, model, data, params, start, correlation, 
%       weights, subset, na.action, naPattern, control, verbose, \dots) 
}
\alias{gnls}
%\alias{update.gnls}
\arguments{
  \item{model}{a two-sided formula object describing the
    model, with the response on the left of a \code{~} operator and 
    a nonlinear expression involving parameters and covariates on the
    right. If \code{data} is given, all names used in the formula should
    be defined as parameters or variables in the data frame.} 
 \item{data}{an optional data frame containing the variables named in
   \code{model}, \code{correlation}, \code{weights}, 
   \code{subset}, and \code{naPattern}. By default the variables are 
   taken from the environment from which \code{gnls} is called.}
 \item{params}{an optional two-sided linear formula of the form
   \code{p1+...+pn~x1+...+xm}, or list of two-sided formulas of the form
   \code{p1~x1+...+xm}, with possibly different models for each
   parameter. The \code{p1,\dots,pn} represent parameters included on the
   right hand side of \code{model} and \code{x1+...+xm} define a linear
   model for the parameters (when the left hand side of the formula
   contains several parameters, they are all assumed to follow the same
   linear model described by the right hand side expression). A \code{1}
   on the right hand side of the formula(s) indicates a single fixed
   effects for the corresponding parameter(s). By default, the
   parameters are obtained from the names of \code{start}.} 
 \item{start}{an optional named list, or numeric vector, with the
   initial values for the parameters in \code{model}. It can be omitted
   when a \code{selfStarting} function is used in \code{model}, in which
   case the starting estimates will be obtained from a single call to the
   \code{nls} function.}
 \item{correlation}{an optional \code{corStruct} object describing the
   within-group correlation structure. See the documentation of
   \code{corClasses} for a description of the available \code{corStruct}
   classes. If a grouping variable is to be used, it must be specified
   in the \code{form} argument to the \code{corStruct}
   constructor. Defaults to \code{NULL}, corresponding to uncorrelated 
   errors.}  
 \item{weights}{an optional \code{varFunc} object or one-sided formula
   describing the within-group heteroscedasticity structure. If given as
   a formula, it is used as the argument to \code{varFixed},
   corresponding to fixed variance weights. See the documentation on
   \code{varClasses} for a description of the available \code{varFunc}
   classes. Defaults to \code{NULL}, corresponding to homoscesdatic
   errors.} 
 \item{subset}{an optional expression indicating which subset of the rows of
   \code{data} should  be  used in the fit. This can be a logical
   vector, or a numeric vector indicating which observation numbers are
   to be included, or a  character  vector of the row names to be
   included.  All observations are included by default.}
 \item{na.action}{a function that indicates what should happen when the
   data contain \code{NA}s.  The default action (\code{na.fail}) causes
   \code{gnls} to print an error message and terminate if there are any
   incomplete observations.}
 \item{naPattern}{an expression or formula object, specifying which returned
   values are to be regarded as missing.}
 \item{control}{a list of control values for the estimation algorithm to
   replace the default values returned by the function \code{gnlsControl}.
   Defaults to an empty list.}
 \item{verbose}{an optional logical value. If \code{TRUE} information on
   the evolution of the iterative algorithm is printed. Default is
   \code{FALSE}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This function fits a nonlinear model using generalized least
  squares. The errors are allowed to be correlated and/or have unequal
  variances.  
}
\value{
  an object of class \code{gnls}, also inheriting from class \code{gls},
  representing the nonlinear model fit. Generic functions such as
  \code{print}, \code{plot} and  \code{summary} have methods to show the
  results of the fit. See \code{gnlsObject} for the components of the
  fit. The functions \code{resid}, \code{coef}, and \code{fitted} can be
  used to extract some of its components.  
}
\references{
 The different correlation structures available for the
 \code{correlation} argument are described in Box, G.E.P., Jenkins,
 G.M., and Reinsel G.C. (1994), Littel, R.C., Milliken, G.A., Stroup,
 W.W., and Wolfinger, R.D. (1996), and Venables, W.N. and Ripley,
 B.D. (1997). The use of variance functions for linear 
 and nonlinear models is presented in detail in Carrol, R.J. and Rupert,
 D. (1988) and Davidian, M. and Giltinan, D.M. (1995).  

 Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994) "Time Series
 Analysis: Forecasting and Control", 3rd Edition, Holden-Day. 

 Carrol, R.J. and Rupert, D. (1988) "Transformation and Weighting in
 Regression", Chapman and Hall.

 Davidian, M. and Giltinan, D.M. (1995) "Nonlinear Mixed Effects Models
 for Repeated Measurement Data", Chapman and Hall.

 Littel, R.C., Milliken, G.A., Stroup, W.W., and Wolfinger, R.D. (1996)
 "SAS Systems for Mixed Models", SAS Institute.

 Venables, W.N. and Ripley, B.D. (1997) "Modern Applied Statistics with
 S-plus", 2nd Edition, Springer-Verlag.
}
\author{Jose Pinheiro \email{jose.pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{
  \code{\link{gnlsControl}}, \code{\link{gnlsObject}},
  \code{\link{varFunc}}, \code{\link{corClasses}}, \code{\link{varClasses}}
}
\examples{
data(Soybean)
# variance increases with a power of the absolute fitted values
fm1 <- gnls(weight ~ SSlogis(Time, Asym, xmid, scal), Soybean,
            weights = varPower())
summary(fm1)
}
\keyword{models}

\eof
% $Id: gnlsControl.Rd,v 1.8 2002/03/05 14:59:39 bates Exp $
\name{gnlsControl}
\title{Control Values for gnls Fit}
\usage{
gnlsControl(maxIter, nlsMaxIter, msMaxIter, minScale, tolerance,
            nlsTol, msTol, msScale, returnObject, msVerbose,
            apVar, .relStep, nlmStepMax, minAbsParApVar)
}
\alias{gnlsControl}
\arguments{
 \item{maxIter}{maximum number of iterations for the \code{gnls}
   optimization algorithm. Default is 50.} 
 \item{nlsMaxIter}{maximum number of iterations
   for the \code{nls} optimization step inside the \code{gnls}
   optimization. Default is 7.}
 \item{msMaxIter}{maximum number of iterations
   for the \code{ms} optimization step inside the \code{gnls}
   optimization. Default is 50.}
 \item{minScale}{minimum factor by which to shrink the default step size
   in an attempt to decrease the sum of squares in the \code{nls} step.
   Default 0.001.}
 \item{tolerance}{tolerance for the convergence criterion in the
   \code{gnls} algorithm. Default is 1e-6.}
 \item{nlsTol}{tolerance for the convergence criterion in \code{nls}
   step. Default is 1e-3.} 
 \item{msTol}{tolerance for the convergence criterion in \code{ms},
   passed as the \code{rel.tolerance} argument to the function (see
   documentation on \code{ms}). Default is 1e-7.}
 \item{msScale}{scale function passed as the \code{scale} argument to
   the \code{ms} function (see documentation on that function). Default
   is \code{lmeScale}.}
 \item{returnObject}{a logical value indicating whether the fitted
   object should be returned when the maximum number of iterations is
   reached without convergence of the algorithm. Default is
   \code{FALSE}.} 
 \item{msVerbose}{a logical value passed as the \code{trace} argument to
   \code{ms} (see documentation on that function). Default is
   \code{FALSE}.}
 \item{apVar}{a logical value indicating whether the approximate
   covariance matrix of the variance-covariance parameters should be
   calculated. Default is \code{TRUE}.}
 \item{.relStep}{relative step for numerical derivatives
   calculations. Default is \code{.Machine$double.eps^(1/3)}.}
 \item{nlmStepMax}{stepmax value to be passed to nlm. See
     \code{\link{nlm}} for details. Default is 100.0}
 \item{minAbsParApVar}{numeric value - minimum absolute parameter value
   in the approximate variance calculation.  The default is \code{0.05}.}
}
\description{
  The values supplied in the function call replace the defaults and a
  list with all possible arguments is returned. The returned list is
  used as the \code{control} argument to the \code{gnls} function.
}
\value{
  a list with components for each of the possible arguments.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gnls}}, \code{\link{lmeScale}}}

\examples{
# decrease the maximum number iterations in the ms call and
# request that information on the evolution of the ms iterations be printed
gnlsControl(msMaxIter = 20, msVerbose = TRUE)
}
\keyword{models}

\eof
% $Id: gnlsObject.Rd,v 1.4 2002/03/05 14:59:39 bates Exp $
\name{gnlsObject}
\title{Fitted gnls Object}
\alias{gnlsObject}
\description{
  An object returned by the \code{gnls} function, inheriting from class
  \code{gnls} and also from class \code{gls}, and representing a
  generalized nonlinear least squares fitted model. Objects of this
  class have methods for the generic functions  \code{anova},
  \code{coef}, \code{fitted}, \code{formula}, \code{getGroups},
  \code{getResponse}, \code{intervals}, \code{logLik}, \code{plot},
  \code{predict}, \code{print}, \code{residuals}, \code{summary}, and
  \code{update}.    

}
\value{
  The following components must be included in a legitimate \code{gnls}
  object. 
  \item{apVar}{an approximate covariance matrix for the
    variance-covariance coefficients. If \code{apVar = FALSE} in the list
    of control values used in the call to \code{gnls}, this
    component is equal to \code{NULL}.}
  \item{call}{a list containing an image of the \code{gnls} call that
    produced the object.}
  \item{coefficients}{a vector with the estimated nonlinear model
    coefficients.} 
  \item{contrasts}{a list with the contrasts used to represent factors
    in the model formula. This information is important for making
    predictions from a new data frame in which not all levels of the
    original factors are observed. If no factors are used in the model,
    this component will be an empty list.}
  \item{dims}{a list with basic dimensions used in the model fit,
    including the components \code{N} - the number of observations used in
    the fit and \code{p} - the number of coefficients in the nonlinear
    model.}
  \item{fitted}{a vector with the fitted values.}
  \item{modelStruct}{an object inheriting from class \code{gnlsStruct},
    representing a list of model components, such as \code{corStruct} and
    \code{varFunc} objects.} 
  \item{groups}{a vector with the correlation structure grouping factor,
    if any is present.}
  \item{logLik}{the log-likelihood at convergence.}
  \item{numIter}{the number of iterations used in the iterative
    algorithm.}
  \item{plist}{}
  \item{pmap}{}
  \item{residuals}{a vector with the residuals.} 
  \item{sigma}{the estimated residual standard error.}
  \item{varBeta}{an approximate covariance matrix of the
    coefficients estimates.}
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gnls}}, \code{gnlsStruct}}

\keyword{models}

\eof
% $Id: gnlsStruct.Rd,v 1.5.2.1 2002/08/09 19:45:29 bates Exp $
\name{gnlsStruct}
\title{Generalized Nonlinear Least Squares Structure}
\usage{
gnlsStruct(corStruct, varStruct)
}
\alias{gnlsStruct}
\alias{Initialize.gnlsStruct}
\arguments{
 \item{corStruct}{an optional \code{corStruct} object, representing a
   correlation structure. Default is \code{NULL}.}
 \item{varStruct}{an optional \code{varFunc} object, representing a
   variance function structure. Default is \code{NULL}.}
}
\description{
  A generalized nonlinear least squares structure is a list of model
  components representing different sets of parameters in the nonlinear 
  model. A \code{gnlsStruct}  may contain \code{corStruct} and
  \code{varFunc} objects. \code{NULL} arguments are not included in the
  \code{gnlsStruct} list. 
}
\value{
  a list of model variance-covariance components determining the
  parameters to be estimated for the associated nonlinear model.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gnls}}, \code{\link{corClasses}},
  \code{\link{varFunc}} } 

\examples{
gnls1 <- gnlsStruct(corAR1(), varPower())
}
\keyword{models}

\eof
% $Id: groupedData.Rd,v 1.8 2002/01/28 03:44:17 bates Exp $
\name{groupedData}
\title{Construct a groupedData Object}
\usage{
groupedData(formula, data, order.groups, FUN, outer, inner,
 labels, units)
\method{update}{groupedData}(object, formula, data, order.groups, FUN,
outer, inner, labels, units, \dots)
}
\alias{groupedData}
\alias{nfGroupedData}
\alias{nmGroupedData}
\alias{[.groupedData}
\alias{as.data.frame.groupedData}
\alias{update.groupedData}
\arguments{
 \item{object}{an object inheriting from class \code{groupedData}.}
 \item{formula}{a formula of the form \code{resp ~ cov | group} where
   \code{resp} is the response, \code{cov} is the primary covariate, and
   \code{group} is the grouping factor.  The expression \code{1} can be
   used for the primary covariate when there is no other suitable
   candidate.  Multiple nested grouping factors can be listed separated
   by the \code{/} symbol as in \code{fact1/fact2}.  In an expression
   like this the \code{fact2} factor is nested within the \code{fact1}
   factor.}
 \item{data}{a data frame in which the expressions in \code{formula} can
   be evaluated.  The resulting \code{groupedData} object will consist
   of the same data values in the same order but with additional
   attributes.}
 \item{order.groups}{an optional logical value, or list of logical
   values, indicating if the grouping factors should be converted to
   ordered factors according to the function \code{FUN} applied to the
   response from each group. If multiple levels of grouping are present,
   this argument can be either a single logical value (which will be
   repeated for all grouping levels) or a list of logical values. If no
   names are assigned to the list elements, they are assumed in the same
   order as the group levels (outermost to innermost grouping). Ordering
   within a level of grouping is done within the levels of the grouping
   factors which are outer to it. Changing the grouping factor to an
   ordered factor does not affect the ordering of the rows in the data
   frame but it does affect the order of the panels in a trellis display
   of the data or models fitted to the data.  Defaults to \code{TRUE}.}  
 \item{FUN}{an optional summary function that will be applied to the
   values of the response for each level of the grouping factor, when
   \code{order.groups = TRUE}, to determine the ordering.  Defaults to
   the \code{max} function.}
 \item{outer}{an optional one-sided formula, or list of one-sided
   formulas, indicating covariates that are outer to the grouping
   factor(s).  If multiple levels of grouping are present,
   this argument can be either a single one-sided formula, or a list of
   one-sided formulas. If no names are assigned to the list elements,
   they are assumed in the same order as the group levels (outermost to
   innermost grouping). An outer covariate is invariant within the sets
   of rows defined by the grouping factor.  Ordering of the groups is
   done in such a way as to preserve adjacency of groups with the same
   value of the outer variables.  When plotting a  groupedData object,
   the argument \code{outer = TRUE} causes the panels to be determined
   by the \code{outer} formula.  The points within the panels are 
   associated by level of the grouping factor. Defaults to \code{NULL},
   meaning that no outer covariates are present.} 
 \item{inner}{an optional one-sided formula, or list of one-sided
   formulas, indicating covariates that are inner to the grouping
   factor(s). If multiple levels of grouping are present,
   this argument can be either a single one-sided formula, or a list of
   one-sided formulas. If no names are assigned to the list elements,
   they are assumed in the same order as the group levels (outermost to
   innermost grouping). An inner covariate can change 
   within the sets of rows defined by the grouping factor.  An inner
   formula can be used to associate points in a plot of a groupedData
   object.  Defaults to \code{NULL}, meaning that no inner covariates
   are present.} 
 \item{labels}{an optional list of character strings giving labels for
   the response and the primary covariate.  The label for the primary
   covariate is named \code{x} and that for the response is named
   \code{y}.  Either label can be omitted.}
 \item{units}{an optional list of character strings giving the units for
   the response and the primary covariate.  The units string for the
   primary covariate is named \code{x} and that for the response is
   named \code{y}.  Either units string can be omitted.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  An object of the \code{groupedData} class is constructed from the
  \code{formula} and \code{data} by attaching the \code{formula} as an
  attribute of the data, along with any of \code{outer}, \code{inner},
  \code{labels}, and \code{units} that are given.  If
  \code{order.groups} is \code{TRUE} the grouping factor is converted to
  an ordered factor with the ordering determined by
  \code{FUN}. Depending on the number of grouping levels and the type of
  primary covariate, the returned object will be of one of three
  classes: \code{nfnGroupedData} - numeric covariate, single level of
  nesting; \code{nffGroupedData} - factor covariate, single level of
  nesting; and \code{nmGroupedData} - multiple levels of
  nesting. Several modeling and plotting functions can use the formula
  stored with a \code{groupedData} object to construct default plots and
  models. 
}
\value{
  an object of one of the classes \code{nfnGroupedData},
  \code{nffGroupedData}, or \code{nmGroupedData}, and also inheriting
  from  classes \code{groupedData} and \code{data.frame}. 
}
\references{
Bates, D.M. and Pinheiro, J.C. (1997), "Software Design for Longitudinal
Data", in "Modelling Longitudinal and Spatially Correlated Data:
Methods, Applications and Future Directions", T.G. Gregoire (ed.),
Springer-Verlag, New York.

Pinheiro, J.C. and Bates, D.M. (1997) "Future Directions in
Mixed-Effects Software: Design of NLME 3.0" available at
http://nlme.stat.wisc.edu/
}

\author{Douglas Bates and Jose Pinheiro}

\seealso{\code{\link{formula}}, \code{\link{gapply}},
  \code{\link{gsummary}}, \code{\link{lme}}
}
\examples{
data(Orthodont)
Orth.new <-  # create a new copy of the groupedData object
  groupedData( distance ~ age | Subject,
              data = as.data.frame( Orthodont ),
              FUN = mean,
              outer = ~ Sex,
              labels = list( x = "Age",
                y = "Distance from pituitary to pterygomaxillary fissure" ),
              units = list( x = "(yr)", y = "(mm)") )
\dontrun{
plot( Orth.new )         # trellis plot by Subject
}
formula( Orth.new )      # extractor for the formula
gsummary( Orth.new )     # apply summary by Subject
fm1 <- lme( Orth.new )   # fixed and groups formulae extracted from object
Orthodont2 <- update(Orthodont, FUN = mean)
}
\keyword{manip}
\keyword{attribute}

\eof
% $Id: gsummary.Rd,v 1.7.2.1 2003/04/17 22:28:45 bates Exp $
\name{gsummary}
\title{Summarize by Groups}
\usage{
gsummary(object, FUN, omitGroupingFactor, form, level,
   groups, invariantsOnly, \dots)
}
\alias{gsummary}
\arguments{
 \item{object}{an object to be summarized - usually a \code{groupedData}
   object or a \code{data.frame}.
 }
 \item{FUN}{an optional summary function or a list of summary functions
   to be applied to each variable in the frame.  The function or
   functions are applied only to variables in \code{object} that vary
   within the groups defined by \code{groups}.  Invariant variables are
   always summarized by group using the unique value that they assume
   within that group.  If \code{FUN} is a single
   function it will be applied to each non-invariant variable by group
   to produce the summary for that variable.  If \code{FUN} is a list of
   functions, the names in the list should designate classes of
   variables in the frame such as \code{ordered}, \code{factor}, or
   \code{numeric}.  The indicated function will be applied to any
   non-invariant variables of that class.  The default functions to be
   used are \code{mean} for numeric factors, and \code{Mode} for both
   \code{factor} and \code{ordered}.  The \code{Mode} function, defined
   internally in \code{gsummary}, returns the modal or most popular
   value of the variable.  It is different from the \code{mode} function
   that returns the S-language mode of the variable.
 }
 \item{omitGroupingFactor}{an optional logical value.  When \code{TRUE}
   the grouping factor itself will be omitted from the group-wise
   summary but the levels of the grouping factor will continue to be
   used as the row names for the data frame that is produced by the
   summary. Defaults to \code{FALSE}.
 }
 \item{form}{an optional one-sided formula that defines the groups.
   When this formula is given, the right-hand side is evaluated in
   \code{object}, converted to a factor if necessary, and the unique
   levels are used to define the groups.  Defaults to
   \code{formula(object)}.
 } 
 \item{level}{an optional positive integer giving the level of grouping
   to be used in an object with multiple nested grouping levels.
   Defaults to the highest or innermost level of grouping.}
 \item{groups}{an optional factor that will be used to split the 
   rows into groups.  Defaults to \code{getGroups(object, form, level)}.
 }
 \item{invariantsOnly}{an optional logical value.  When \code{TRUE} only 
   those covariates that are invariant within each group will be
   summarized.  The summary value for the group is always the unique
   value taken on by that covariate within the group.  The columns in
   the summary are of the same class as the corresponding columns in
   \code{object}. By definition, the grouping factor itself must be an
   invariant.   When combined with \code{omitGroupingFactor = TRUE},
   this option can be used to discover is there are invariant covariates 
   in the data frame.  Defaults to \code{FALSE}.
 }
 \item{\dots}{optional additional arguments to the summary functions
   that are invoked on the variables by group.  Often it is helpful to
   specify \code{na.rm = TRUE}.
 }
}

\description{
  Provide a summary of the variables in a data frame by groups of rows.
  This is most useful with a \code{groupedData} object to examine the
  variables by group.
}

\value{
  A \code{data.frame} with one row for each level of the grouping
  factor.  The number of columns is at most the number of columns in
  \code{object}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{summary}}, \code{\link{groupedData}},
  \code{\link{getGroups}}} 

\examples{
data(Orthodont)
gsummary(Orthodont)  # default summary by Subject
## gsummary with invariantsOnly = TRUE and omitGroupingFactor = TRUE
## determines whether there are covariates like Sex that are invariant
## within the repeated observations on the same Subject.
gsummary(Orthodont, inv = TRUE, omit = TRUE)
}
\keyword{manip}

\eof
% $Id: intervals.Rd,v 1.4.2.1 2003/04/17 22:28:45 bates Exp $
\name{intervals}
\title{Confidence Intervals on Coefficients}
\usage{
intervals(object, level, \dots)
}
\alias{intervals}
\arguments{
 \item{object}{a fitted model object from which parameter estimates can
   be extracted.}
 \item{level}{an optional numeric value for the interval confidence
   level. Defaults to 0.95.}
 \item{\dots}{some methods for the generic may require additional
   arguments.}
}
\description{
  Confidence intervals on the parameters associated with the model
  represented by \code{object} are obtained. This function is generic;
  method functions can be written to handle specific classes of
  objects. Classes which already have methods for this function include:
  \code{gls}, \code{lme}, and \code{lmList}. 
}
\value{
  will depend on the method function used; see the appropriate documentation.
}
\seealso{\code{\link{intervals.gls}}, \code{\link{intervals.lme}},
    \code{\link{intervals.lmList}}
} 

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\examples{
## see the method documentation
}
\keyword{models}

\eof
% $Id: intervals.gls.Rd,v 1.9 2002/03/05 14:59:39 bates Exp $
\name{intervals.gls}
\title{Confidence Intervals on gls Parameters}
\usage{
\method{intervals}{gls}(object, level, which, \dots)
}
\alias{intervals.gls}
\alias{print.intervals.gls}
\arguments{
 \item{object}{an object inheriting from class \code{gls}, representing
   a generalized least squares fitted linear model.}
 \item{level}{an optional numeric value for the interval confidence
   level. Defaults to 0.95.}
 \item{which}{an optional character string specifying the subset
   of parameters for which to construct the confidence
   intervals. Possible values are \code{"all"} for all parameters,
   \code{"var-cov"} for the variance-covariance parameters only, and
   \code{"coef"} for the linear model coefficients  only. Defaults to
   \code{"all"}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  Approximate confidence intervals for the parameters in the linear
  model represented by \code{object} are obtained, using
  a normal approximation to the distribution of the (restricted)
  maximum likelihood estimators (the estimators are assumed to have a
  normal distribution centered at the true parameter values and with
  covariance matrix equal to the negative inverse Hessian matrix of the
  (restricted) log-likelihood evaluated at the estimated parameters).
  Confidence intervals are obtained in an unconstrained scale first,
  using the normal approximation, and, if necessary, transformed to the
  constrained scale.
}
\value{
  a list with components given by data frames with rows corresponding to
  parameters and columns \code{lower}, \code{est.}, and \code{upper}
  representing respectively lower confidence limits, the estimated
  values, and upper confidence limits for the parameters. Possible
  components are:
  \item{coef}{linear model coefficients, only present when \code{which}
    is not equal to \code{"var-cov"}.}
  \item{corStruct}{correlation parameters, only present when
    \code{which} is not equal to \code{"coef"} and a 
    correlation structure is used in \code{object}.}
  \item{varFunc}{variance function parameters, only present when
    \code{which} is not equal to \code{"coef"} and a variance function
    structure is used in \code{object}.} 
  \item{sigma}{residual standard error.}
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gls}}, \code{\link{print.intervals.gls}}}

\examples{
data(Ovary)
fm1 <- gls(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), Ovary,
           correlation = corAR1(form = ~ 1 | Mare))
intervals(fm1)
}
\keyword{models}

\eof
% $Id: intervals.lmList.Rd,v 1.8 2002/03/05 14:59:39 bates Exp $
\name{intervals.lmList}
\title{Confidence Intervals on lmList Coefficients}
\usage{
\method{intervals}{lmList}(object, level, pool, \dots)
}
\alias{intervals.lmList}
\alias{print.intervals.lmList}
\arguments{
 \item{object}{an object inheriting from class \code{lmList}, representing
   a list of \code{lm} objects with a common model.
 }
 \item{level}{an optional numeric value with the confidence level for
   the intervals. Defaults to 0.95.
 }
 \item{pool}{an optional logical value indicating whether a pooled
   estimate of the residual standard error should be used. Default is
   \code{attr(object, "pool")}.
 }
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  Confidence intervals on the linear model coefficients are obtained for
  each \code{lm} component of \code{object} and organized into a three
  dimensional array. The first dimension corresponding to the names
  of the \code{object} components. The second dimension is given by
  \code{lower}, \code{est.}, and \code{upper} corresponding,
  respectively, to the lower confidence limit, estimated coefficient,
  and upper confidence limit. The third dimension is given by the
  coefficients names. 
}
\value{
  a three dimensional array with the confidence intervals and estimates
  for the coefficients of each \code{lm} component of \code{object}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lmList}}, \code{\link{plot.intervals.lmList}}}

\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age | Subject, Orthodont)
intervals(fm1)
}
\keyword{models}

\eof
% $Id: intervals.lme.Rd,v 1.8 2002/03/05 14:59:39 bates Exp $
\name{intervals.lme}
\title{Confidence Intervals on lme Parameters}
\usage{
\method{intervals}{lme}(object, level, which, \dots)
}
\alias{intervals.lme}
\alias{print.intervals.lme}
\arguments{
 \item{object}{an object inheriting from class \code{lme}, representing
   a fitted linear mixed-effects model.}
 \item{level}{an optional numeric value with the confidence level for
   the intervals. Defaults to 0.95.} 
 \item{which}{an optional character string specifying the  subset
   of parameters for which to construct the confidence
   intervals. Possible values are \code{"all"} for all parameters,
   \code{"var-cov"} for the variance-covariance parameters only, and
   \code{"fixed"} for the fixed effects only. Defaults to \code{"all"}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  Approximate confidence intervals for the parameters in the linear
  mixed-effects model represented by \code{object} are obtained, using
  a normal approximation to the distribution of the (restricted)
  maximum likelihood estimators (the estimators are assumed to have a
  normal distribution centered at the true parameter values and with
  covariance matrix equal to the negative inverse Hessian matrix of the
  (restricted) log-likelihood evaluated at the estimated parameters).
  Confidence intervals are obtained in an unconstrained scale first,
  using the normal approximation, and, if necessary, transformed to the
  constrained scale. The \code{pdNatural} parametrization is used for
  general positive-definite matrices.
}
\value{
  a list with components given by data frames with rows corresponding to
  parameters and columns \code{lower}, \code{est.}, and \code{upper}
  representing respectively lower confidence limits, the estimated
  values, and upper confidence limits for the parameters. Possible
  components are:
  \item{fixed}{fixed effects, only present when \code{which} is not
    equal to \code{"var-cov"}.}
  \item{reStruct}{random effects variance-covariance parameters, only
    present when \code{which} is not equal to \code{"fixed"}.}
  \item{corStruct}{within-group correlation parameters, only
    present when \code{which} is not equal to \code{"fixed"} and a
    correlation structure is used in \code{object}.}
  \item{varFunc}{within-group variance function parameters, only
    present when \code{which} is not equal to \code{"fixed"} and a
    variance function structure is used in \code{object}.}
  \item{sigma}{within-group standard deviation.}
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lme}}, \code{\link{print.intervals.lme}},
    \code{\link{pdNatural}}}

\examples{
data(Orthodont)
fm1 <- lme(distance ~ age, Orthodont, random = ~ age | Subject)
intervals(fm1)
}
\keyword{models}


\eof
% $Id: isBalanced.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{isBalanced}
\alias{isBalanced}
\alias{isBalanced.groupedData}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Check a Design for Balance}
\usage{
isBalanced(object, countOnly, level)
}
\arguments{
 \item{object}{A \code{groupedData} object containing a data frame and a
   formula that describes the roles of variables in the data frame.  The
   object will have one or more nested grouping factors and a primary
   covariate.}
 \item{countOnly}{A logical value indicating if the check for balance
   should only consider the number of observations at each level of the
   grouping factor(s).  Defaults to \code{FALSE}.}
 \item{level}{an optional integer vector specifying the desired
   prediction levels. Levels increase from outermost to innermost
   grouping, with level 0 representing the population (fixed effects)
   predictions. Defaults to the innermost level.}
}
\description{
Check the design of the experiment or study for balance.
}
\details{
A design is
balanced with respect to the grouping factor(s) if there are the same
number of observations at each distinct value of the grouping factor or
each combination of distinct levels of the nested grouping factors.  If
\code{countOnly} is \code{FALSE} the design is also checked for balance
with respect to the primary covariate, which is often the time of the
observation.  A design is balanced with respect to the grouping factor
and the covariate if the number of observations at each distinct level
(or combination of levels for nested factors) is constant and the times
at which the observations are taken (in general, the values of the
primary covariates) also are constant.
}
\value{
\code{TRUE} or \code{FALSE} according to whether the data are balanced
or not
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{table}}, \code{\link{groupedData}}}
\examples{
data(Orthodont)
isBalanced(Orthodont)                    # should return TRUE
isBalanced(Orthodont, countOnly = TRUE)  # should return TRUE
data(Pixel)
isBalanced(Pixel)                        # should return FALSE
isBalanced(Pixel, level = 1)             # should return FALSE
}
\keyword{data}

\eof
% $Id: isInitialized.Rd,v 1.4.4.1 2002/08/09 19:45:29 bates Exp $
\name{isInitialized}
\title{Check if Object is Initialized}
\usage{
isInitialized(object)
}
\alias{isInitialized}
\alias{isInitialized.pdMat}
\alias{isInitialized.pdBlocked}
\arguments{
 \item{object}{any object requiring initialization.}
}
\description{
  Checks if \code{object} has been initialized (generally through a call
  to \code{Initialize}), by searching for components and attributes
  which are modified during initialization.
}
\value{
  a logical value indicating whether \code{object} has been
  initialized. 
}
\author{Jose Pinheiro and Douglas Bates }

\seealso{\code{\link{Initialize}}}

\examples{
pd1 <- pdDiag(~age)
isInitialized(pd1)
}
\keyword{models}

\eof
% $Id: lmList.Rd,v 1.7.2.1 2003/04/17 22:28:45 bates Exp $
\name{lmList}
\title{List of lm Objects with a Common Model}
\usage{
lmList(object, data, level, subset, na.action, pool)
\method{update}{lmList}(object, formula, data, level, subset,
   na.action, pool, \dots)
\method{print}{lmList}(x, pool, \dots)
}
\alias{lmList}
\alias{lmList.formula}
\alias{print.lmList}
\alias{update.lmList}
\arguments{
  \item{object}{For \code{lmList},
    either a linear formula object of the form \code{y ~ x1+...+xn | g}
    or a \code{groupedData} object. In the formula object, \code{y}
    represents the response, \code{x1,...,xn} the covariates, and
    \code{g} the grouping factor specifying the partitioning of the data
    according to which different \code{lm} fits should be performed. The
    grouping factor \code{g} may be omitted from the formula, in which
    case the grouping structure will be obtained from \code{data}, which
    must inherit from class \code{groupedData}. The method function 
    \code{lmList.groupedData} is documented separately.
    For the method \code{update.lmList}, \code{object} is an object
    inheriting from class \code{lmList}.
  }
  \item{formula}{(used in \code{update.lmList} only)
    a two-sided linear formula with the common model for the individuals
    \code{lm} fits.
    }
  \item{data}{
    a data frame in which to interpret the variables named in
    \code{object}. 
  }
  \item{level}{
    an optional integer specifying the level of grouping to be used when 
    multiple nested levels of grouping are present.
  }
 \item{subset}{an optional expression indicating which subset of the rows of
   \code{data} should  be  used in the fit. This can be a logical
   vector, or a numeric vector indicating which observation numbers are
   to be included, or a  character  vector of the row names to be
   included.  All observations are included by default.}
 \item{na.action}{a function that indicates what should happen when the
   data contain \code{NA}s.  The default action (\code{na.fail}) causes
   \code{lmList} to print an error message and terminate if there are any
   incomplete observations.
 }
 \item{pool}{
   an optional logical value indicating whether a pooled
   estimate of the residual standard error should be used 
   in calculations of standard deviations or standard errors for
   summaries.
 }
 \item{x}{an object inheriting from class \code{lmList} to be printed.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  \code{Data} is partitioned according to the levels of the grouping
  factor \code{g} and individual \code{lm} fits are obtained for each
  \code{data} partition, using the model defined in \code{object}.
}

\value{
  a list of \code{lm} objects with as many components as the number of
  groups defined by the grouping factor. Generic functions such as
  \code{coef}, \code{fixed.effects}, \code{lme}, \code{pairs},
  \code{plot}, \code{predict}, \code{random.effects}, \code{summary},
  and \code{update} have methods that can be applied to an \code{lmList}
  object. 
}

\seealso{
\code{\link{lm}}, \code{\link{lme.lmList}}.
}
\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age | Subject, Orthodont)
summary(fm1)
}
\keyword{models}


\eof
% $Id: lmList.groupedData.Rd,v 1.7.2.1 2003/04/17 22:28:45 bates Exp $
\name{lmList.groupedData}
\title{lmList Fit from a groupedData Object}
\usage{
\method{lmList}{groupedData}(object, data, level, subset, na.action, pool)
}
\alias{lmList.groupedData}
\arguments{
  \item{object}{a data frame inheriting from class \code{groupedData}.}
  \item{data}{this argument is included for consistency with the generic
    function. It is ignored in this method function.}
  \item{level}{
    an optional integer specifying the level of grouping to be used when 
    multiple nested levels of grouping are present.
  }
  \item{subset}{an optional expression indicating which subset of the rows of
    \code{data} should  be  used in the fit. This can be a logical
    vector, or a numeric vector indicating which observation numbers are
    to be included, or a  character  vector of the row names to be
    included.  All observations are included by default.}
  \item{na.action}{a function that indicates what should happen when the
    data contain \code{NA}s.  The default action (\code{na.fail}) causes
    \code{lmList} to print an error message and terminate if there are any
    incomplete observations.
  }
  \item{pool}{
    an optional logical value that is preserved as an attribute of the
    returned value.  This will be used as the default for \code{pool} in
    calculations of standard deviations or standard errors for summaries.
  }
}
\description{
  The response variable and primary covariate in \code{formula(object)}
  are used to construct the linear model formula. This formula
  and the \code{groupedData} object are passed as the \code{object} and
  \code{data} arguments to \code{lmList.formula}, together with any other
  additional arguments in the function call. See the documentation on
  \code{lmList.formula} for a description of that function.
}
\value{
  a list of \code{lm} objects with as many components as the number of
  groups defined by the grouping factor. Generic functions such as
  \code{coef}, \code{fixed.effects}, \code{lme}, \code{pairs},
  \code{plot}, \code{predict}, \code{random.effects}, \code{summary},
  and \code{update} have methods that can be applied to an \code{lmList}
  object. 
}

\seealso{
  \code{\link{groupedData}}, \code{\link{lm}}, \code{\link{lme.lmList}},
  \code{\link{lmList.formula}}
}
\examples{
data(Orthodont)
fm1 <- lmList(Orthodont)
summary(fm1)
}
\keyword{models}

\eof
% $Id: lme.Rd,v 1.6.2.1 2003/01/30 18:34:52 bates Exp $
\name{lme}
\title{Linear Mixed-Effects Models}
\usage{
lme(fixed, data, random, correlation, weights, subset, method,
    na.action, control, contrasts = NULL)
\method{update}{lme}(object, fixed, data, random, correlation, weights,
      subset, method, na.action, control, contrasts, \dots)
}
\alias{lme}
\alias{lme.formula}
\alias{update.lme}
\arguments{
 \item{object}{an object inheriting from class \code{lme}, representing
   a fitted linear mixed-effects model.}
  \item{fixed}{a two-sided linear formula object describing the
    fixed-effects part of the model, with the response on the left of a
    \code{~} operator and the terms, separated by \code{+} operators, on
    the right, an \code{lmList} object, or a \code{groupedData}
    object. The method functions \code{lme.lmList} and
    \code{lme.groupedData} are documented separately.}
 \item{data}{an optional data frame containing the variables named in
   \code{fixed}, \code{random}, \code{correlation}, \code{weights}, and
   \code{subset}.  By default the variables are taken from the
   environment from which \code{lme} is called.}
 \item{random}{optionally, any of the following: (i) a one-sided formula
   of the form \code{~x1+...+xn | g1/.../gm}, with \code{x1+...+xn}
   specifying the model for the random effects and \code{g1/.../gm} the
   grouping structure (\code{m} may be equal to 1, in which case no
   \code{/} is required). The random effects formula will be repeated
   for all levels of grouping, in the case of multiple levels of
   grouping; (ii) a list of one-sided formulas of the form
   \code{~x1+...+xn | g}, with possibly different random effects models
   for each grouping level. The order of nesting will be assumed the
   same as the order of the elements in the list; (iii) a one-sided
   formula of the form \code{~x1+...+xn}, or a \code{pdMat} object with
   a formula (i.e. a non-\code{NULL} value for \code{formula(object)}),
   or a list of such formulas or \code{pdMat} objects. In this case, the
   grouping structure formula will be derived from the data used to
   fit the linear mixed-effects model, which should inherit from class
   \code{groupedData}; (iv) a named list of formulas or \code{pdMat}
   objects as in (iii), with the grouping factors as names. The order of
   nesting will be assumed the same as the order of the order of the
   elements in the list; (v) an \code{reStruct} object. See the
   documentation on \code{pdClasses} for a description of the available
   \code{pdMat} classes. Defaults to a formula consisting of the right
   hand side of \code{fixed}.}  
 \item{correlation}{an optional \code{corStruct} object describing the
   within-group correlation structure. See the documentation of
   \code{corClasses} for a description of the available \code{corStruct}
   classes. Defaults to \code{NULL},
   corresponding to no within-group correlations.}
 \item{weights}{an optional \code{varFunc} object or one-sided formula
   describing the within-group heteroscedasticity structure. If given as
   a formula, it is used as the argument to \code{varFixed},
   corresponding to fixed variance weights. See the documentation on
   \code{varClasses} for a description of the available \code{varFunc}
   classes. Defaults to \code{NULL}, corresponding to homocesdatic
   within-group errors.} 
 \item{subset}{an optional expression indicating the subset of the rows of
   \code{data} that should be used in the fit. This can be a logical
   vector, or a numeric vector indicating which observation numbers are
   to be included, or a  character  vector of the row names to be
   included.  All observations are included by default.}
 \item{method}{a character string.  If \code{"REML"} the model is fit by
   maximizing the restricted log-likelihood.  If \code{"ML"} the
   log-likelihood is maximized.  Defaults to \code{"REML"}.}
 \item{na.action}{a function that indicates what should happen when the
   data contain \code{NA}s.  The default action (\code{na.fail}) causes
   \code{lme} to print an error message and terminate if there are any
   incomplete observations.}
 \item{control}{a list of control values for the estimation algorithm to
   replace the default values returned by the function \code{lmeControl}.
   Defaults to an empty list.}
 \item{contrasts}{an optional list. See the \code{contrasts.arg}
   of \code{model.matrix.default}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This generic function fits a linear mixed-effects model in the
  formulation described in Laird and Ware (1982) but allowing for nested
  random effects. The within-group errors are allowed to be correlated
  and/or have unequal variances. 
}
\value{
  an object of class \code{lme} representing the linear mixed-effects
  model fit. Generic functions such as \code{print}, \code{plot} and
  \code{summary} have methods to show the results of the fit. See
  \code{lmeObject} for the components of the fit. The functions
  \code{resid}, \code{coef}, \code{fitted}, \code{fixed.effects}, and
  \code{random.effects}  can be used to extract some of its components.
}
\references{
 The computational methods are described in Bates, D.M. and Pinheiro
 (1998) and follow on the general framework of Lindstrom, M.J. and Bates,
 D.M. (1988). The model formulation is described in Laird, N.M. and Ware,
 J.H. (1982).  The variance-covariance parametrizations are described in
 <Pinheiro, J.C. and Bates., D.M.  (1996).   The different correlation
 structures available for the \code{correlation} argument are described
 in Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994), Littel, R.C.,
 Milliken, G.A., Stroup, W.W., and Wolfinger, R.D. (1996), and Venables,
 W.N. and Ripley, B.D. (1997). The use of variance functions for linear
 and nonlinear mixed effects models is presented in detail in Davidian,
 M. and Giltinan, D.M. (1995). 

 Bates, D.M. and Pinheiro, J.C. (1998) "Computational methods for
 multilevel models" available in PostScript or PDF formats at
 http://franz.stat.wisc.edu/pub/NLME/
 
 Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994) "Time Series
 Analysis: Forecasting and Control", 3rd Edition, Holden-Day. 

 Davidian, M. and Giltinan, D.M. (1995) "Nonlinear Mixed Effects Models
 for Repeated Measurement Data", Chapman and Hall.

 Laird, N.M. and Ware, J.H. (1982) "Random-Effects Models for
 Longitudinal Data", Biometrics, 38, 963-974.  

 Lindstrom, M.J. and Bates, D.M. (1988) "Newton-Raphson and EM
 Algorithms for Linear Mixed-Effects Models for Repeated-Measures
 Data", Journal of the American Statistical Association, 83,
 1014-1022. 

 Littel, R.C., Milliken, G.A., Stroup, W.W., and Wolfinger, R.D. (1996)
 "SAS Systems for Mixed Models", SAS Institute.

 Pinheiro, J.C. and Bates., D.M.  (1996) "Unconstrained
 Parametrizations for Variance-Covariance Matrices", Statistics and
 Computing, 6, 289-296.

 Venables, W.N. and Ripley, B.D. (1997) "Modern Applied Statistics with
 S-plus", 2nd Edition, Springer-Verlag.
}
\author{Jose Pinheiro \email{jose.pinheiro@pharma.novartis.com} and
  Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{
  \code{\link{lmeControl}}, \code{\link{lme.lmList}},
  \code{\link{lme.groupedData}}, \code{\link{lmeObject}},
  \code{\link{lmList}}, \code{\link{reStruct}}, \code{\link{reStruct}},
  \code{\link{varFunc}}, \code{\link{pdClasses}},
  \code{\link{corClasses}}, \code{\link{varClasses}}
}
\examples{
data(Orthodont)
fm1 <- lme(distance ~ age, data = Orthodont) # random is ~ age
fm2 <- lme(distance ~ age + Sex, data = Orthodont, random = ~ 1)
summary(fm1)
summary(fm2)
}
\keyword{models}

\eof
% $Id: lme.groupedData.Rd,v 1.8.2.1 2003/01/30 18:35:14 bates Exp $
\name{lme.groupedData}
\title{LME fit from groupedData Object}
\usage{
\method{lme}{groupedData}(fixed, data, random, correlation, weights, 
    subset, method, na.action, control, contrasts)
}
\alias{lme.groupedData}
\arguments{
  \item{fixed}{a data frame inheriting from class \code{groupedData}.}
  \item{data}{this argument is included for consistency with the generic
    function. It is ignored in this method function.}
 \item{random}{optionally, any of the following: (i) a one-sided formula
   of the form \code{~x1+...+xn | g1/.../gm}, with \code{x1+...+xn}
   specifying the model for the random effects and \code{g1/.../gm} the
   grouping structure (\code{m} may be equal to 1, in which case no
   \code{/} is required). The random effects formula will be repeated
   for all levels of grouping, in the case of multiple levels of
   grouping; (ii) a list of one-sided formulas of the form
   \code{~x1+...+xn | g}, with possibly different random effects models
   for each grouping level. The order of nesting will be assumed the
   same as the order of the elements in the list; (iii) a one-sided
   formula of the form \code{~x1+...+xn}, or a \code{pdMat} object with
   a formula (i.e. a non-\code{NULL} value for \code{formula(object)}),
   or a list of such formulas or \code{pdMat} objects. In this case, the
   grouping structure formula will be derived from the data used to
   fit the linear mixed-effects model, which should inherit from class
   \code{groupedData}; (iv) a named list of formulas or \code{pdMat}
   objects as in (iii), with the grouping factors as names. The order of
   nesting will be assumed the same as the order of the order of the
   elements in the list; (v) an \code{reStruct} object. See the
   documentation on \code{pdClasses} for a description of the available
   \code{pdMat} classes. Defaults to a formula consisting of the right
   hand side of \code{fixed}.}  
 \item{correlation}{an optional \code{corStruct} object describing the
   within-group correlation structure. See the documentation of
   \code{corClasses} for a description of the available \code{corStruct}
   classes. Defaults to \code{NULL},
   corresponding to no within-group correlations.}
 \item{weights}{an optional \code{varFunc} object or one-sided formula
   describing the within-group heteroscedasticity structure. If given as
   a formula, it is used as the argument to \code{varFixed},
   corresponding to fixed variance weights. See the documentation on
   \code{varClasses} for a description of the available \code{varFunc}
   classes. Defaults to \code{NULL}, corresponding to homocesdatic
   within-group errors.} 
 \item{subset}{an optional expression indicating the subset of the rows of
   \code{data} that should be used in the fit. This can be a logical
   vector, or a numeric vector indicating which observation numbers are
   to be included, or a  character  vector of the row names to be
   included.  All observations are included by default.}
 \item{method}{a character string.  If \code{"REML"} the model is fit by
   maximizing the restricted log-likelihood.  If \code{"ML"} the
   log-likelihood is maximized.  Defaults to \code{"REML"}.}
 \item{na.action}{a function that indicates what should happen when the
   data contain \code{NA}s.  The default action (\code{na.fail}) causes
   \code{lme} to print an error message and terminate if there are any
   incomplete observations.}
 \item{control}{a list of control values for the estimation algorithm to
   replace the default values returned by the function \code{lmeControl}.
   Defaults to an empty list.}
 \item{contrasts}{an optional list. See the \code{contrasts.arg}
   of \code{model.matrix.default}.}
}
\description{
  The response variable and primary covariate in \code{formula(fixed)}
  are used to construct the fixed effects model formula. This formula
  and the   \code{groupedData} object are passed as the \code{fixed} and
  \code{data} arguments to \code{lme.formula}, together with any other
  additional arguments in the function call. See the documentation on
  \code{lme.formula} for a description of that function.
}
\value{
  an object of class \code{lme} representing the linear mixed-effects
  model fit. Generic functions such as \code{print}, \code{plot} and
  \code{summary} have methods to show the results of the fit. See
  \code{lmeObject} for the components of the fit. The functions
  \code{resid}, \code{coef}, \code{fitted}, \code{fixed.effects}, and
  \code{random.effects}  can be used to extract some of its components.
}
\references{
 The computational methods are described in Bates, D.M. and Pinheiro
 (1998) and follow on the general framework of Lindstrom, M.J. and Bates,
 D.M. (1988). The model formulation is described in Laird, N.M. and Ware,
 J.H. (1982).  The variance-covariance parametrizations are described in
 Pinheiro, J.C. and Bates., D.M.  (1996).   The different correlation
 structures available for the \code{correlation} argument are described
 in Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994), Littel, R.C.,
 Milliken, G.A., Stroup, W.W., and Wolfinger, R.D. (1996), and Venables,
 W.N. and Ripley, B.D. (1997). The use of variance functions for linear
 and nonlinear mixed effects models is presented in detail in Davidian,
 M. and Giltinan, D.M. (1995). 

 Bates, D.M. and Pinheiro, J.C. (1998) "Computational methods for
 multilevel models" available in PostScript or PDF formats at
 http://franz.stat.wisc.edu/pub/NLME/
 
 Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994) "Time Series
 Analysis: Forecasting and Control", 3rd Edition, Holden-Day. 

 Davidian, M. and Giltinan, D.M. (1995) "Nonlinear Mixed Effects Models
 for Repeated Measurement Data", Chapman and Hall.

 Laird, N.M. and Ware, J.H. (1982) "Random-Effects Models for
 Longitudinal Data", Biometrics, 38, 963-974.  

 Lindstrom, M.J. and Bates, D.M. (1988) "Newton-Raphson and EM
 Algorithms for Linear Mixed-Effects Models for Repeated-Measures
 Data", Journal of the American Statistical Association, 83,
 1014-1022. 

 Littel, R.C., Milliken, G.A., Stroup, W.W., and Wolfinger, R.D. (1996)
 "SAS Systems for Mixed Models", SAS Institute.

 Pinheiro, J.C. and Bates., D.M.  (1996) "Unconstrained
 Parametrizations for Variance-Covariance Matrices", Statistics and
 Computing, 6, 289-296.

 Venables, W.N. and Ripley, B.D. (1997) "Modern Applied Statistics with
 S-plus", 2nd Edition, Springer-Verlag.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lme}}, \code{\link{groupedData}},
  \code{\link{lmeObject}} } 

\examples{
data(Orthodont)
fm1 <- lme(Orthodont)
summary(fm1)
}
\keyword{models}

\eof
% $Id: lme.lmList.Rd,v 1.8.2.1 2003/01/30 18:35:33 bates Exp $
\name{lme.lmList}
\title{LME fit from lmList Object}
\usage{
\method{lme}{lmList}(fixed, data, random, correlation, weights, subset, method,
    na.action, control, contrasts)
}
\alias{lme.lmList}
\arguments{
  \item{fixed}{an object inheriting from class \code{lmList},
    representing a list of \code{lm} fits with a common model.}
  \item{data}{this argument is included for consistency with the generic
    function. It is ignored in this method function.}
 \item{random}{an optional one-sided linear formula with no conditioning
   expression, or a \code{pdMat} object with a \code{formula}
   attribute. Multiple levels of grouping are not allowed with this
   method function.  Defaults to a formula consisting of the right hand
   side of \code{formula(fixed)}.} 
 \item{correlation}{an optional \code{corStruct} object describing the
   within-group correlation structure. See the documentation of
   \code{corClasses} for a description of the available \code{corStruct}
   classes. Defaults to \code{NULL},
   corresponding to no within-group correlations.}
 \item{weights}{an optional \code{varFunc} object or one-sided formula
   describing the within-group heteroscedasticity structure. If given as
   a formula, it is used as the argument to \code{varFixed},
   corresponding to fixed variance weights. See the documentation on
   \code{varClasses} for a description of the available \code{varFunc}
   classes. Defaults to \code{NULL}, corresponding to homocesdatic
   within-group errors.} 
 \item{subset}{an optional expression indicating the subset of the rows of
   \code{data} that should be used in the fit. This can be a logical
   vector, or a numeric vector indicating which observation numbers are
   to be included, or a  character  vector of the row names to be
   included.  All observations are included by default.}
 \item{method}{a character string.  If \code{"REML"} the model is fit by
   maximizing the restricted log-likelihood.  If \code{"ML"} the
   log-likelihood is maximized.  Defaults to \code{"REML"}.}
 \item{na.action}{a function that indicates what should happen when the
   data contain \code{NA}s.  The default action (\code{na.fail}) causes
   \code{lme} to print an error message and terminate if there are any
   incomplete observations.}
 \item{control}{a list of control values for the estimation algorithm to
   replace the default values returned by the function \code{lmeControl}.
   Defaults to an empty list.}
 \item{contrasts}{an optional list. See the \code{contrasts.arg}
   of \code{model.matrix.default}.}
}
\description{
  If the random effects names defined in \code{random} are a subset of
  the \code{lmList} object coefficient names, initial estimates for the
  covariance matrix of the random effects are obtained (overwriting any
  values given in \code{random}). \code{formula(fixed)} and the
  \code{data} argument in the calling sequence used to obtain
  \code{fixed} are passed as the \code{fixed} and \code{data} arguments
  to \code{lme.formula}, together with any other additional arguments in
  the function call. See the documentation on \code{lme.formula} for a
  description of that function. 
}
\value{
  an object of class \code{lme} representing the linear mixed-effects
  model fit. Generic functions such as \code{print}, \code{plot} and
  \code{summary} have methods to show the results of the fit. See
  \code{lmeObject} for the components of the fit. The functions
  \code{resid}, \code{coef}, \code{fitted}, \code{fixed.effects}, and
  \code{random.effects}  can be used to extract some of its components.
}
\references{
 The computational methods are described in Bates, D.M. and Pinheiro
 (1998) and follow on the general framework of Lindstrom, M.J. and Bates,
 D.M. (1988). The model formulation is described in Laird, N.M. and Ware,
 J.H. (1982).  The variance-covariance parametrizations are described in
 <Pinheiro, J.C. and Bates., D.M.  (1996).   The different correlation
 structures available for the \code{correlation} argument are described
 in Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994), Littel, R.C.,
 Milliken, G.A., Stroup, W.W., and Wolfinger, R.D. (1996), and Venables,
 W.N. and Ripley, B.D. (1997). The use of variance functions for linear
 and nonlinear mixed effects models is presented in detail in Davidian,
 M. and Giltinan, D.M. (1995). 

 Bates, D.M. and Pinheiro, J.C. (1998) "Computational methods for
 multilevel models" available in PostScript or PDF formats at
 http://franz.stat.wisc.edu/pub/NLME/
 
 Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994) "Time Series
 Analysis: Forecasting and Control", 3rd Edition, Holden-Day. 

 Davidian, M. and Giltinan, D.M. (1995) "Nonlinear Mixed Effects Models
 for Repeated Measurement Data", Chapman and Hall.

 Laird, N.M. and Ware, J.H. (1982) "Random-Effects Models for
 Longitudinal Data", Biometrics, 38, 963-974.  

 Lindstrom, M.J. and Bates, D.M. (1988) "Newton-Raphson and EM
 Algorithms for Linear Mixed-Effects Models for Repeated-Measures
 Data", Journal of the American Statistical Association, 83,
 1014-1022. 

 Littel, R.C., Milliken, G.A., Stroup, W.W., and Wolfinger, R.D. (1996)
 "SAS Systems for Mixed Models", SAS Institute.

 Pinheiro, J.C. and Bates., D.M.  (1996) "Unconstrained
 Parametrizations for Variance-Covariance Matrices", Statistics and
 Computing, 6, 289-296.

 Venables, W.N. and Ripley, B.D. (1997) "Modern Applied Statistics with
 S-plus", 2nd Edition, Springer-Verlag.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lme}}, \code{\link{lmList}},
  \code{\link{lmeObject}} } 

\examples{
data(Orthodont)
fm1 <- lmList(Orthodont)
fm2 <- lme(fm1)
summary(fm1)
summary(fm2)
}
\keyword{models}

\eof
% $Id: lmeControl.Rd,v 1.8 2002/03/05 14:59:39 bates Exp $
\name{lmeControl}
\title{Control Values for lme Fit}
\usage{
lmeControl(maxIter, msMaxIter, tolerance, niterEM, msTol,
           msScale, msVerbose, returnObject, gradHess, apVar,
           .relStep, minAbsParApVar, nlmStepMax, natural)
}
\alias{lmeControl}
\arguments{
 \item{maxIter}{maximum number of iterations for the \code{lme}
   optimization algorithm. Default is 50.} 
 \item{msMaxIter}{maximum number of iterations
   for the \code{nlm} optimization step inside the \code{lme}
   optimization. Default is 50.}
 \item{tolerance}{tolerance for the convergence criterion in the
   \code{lme} algorithm. Default is 1e-6.}
 \item{niterEM}{number of iterations for the EM algorithm used to refine
   the initial estimates of the random effects variance-covariance
   coefficients. Default is 25.}
 \item{msTol}{tolerance for the convergence criterion in \code{nlm},
   passed as the \code{rel.tolerance} argument to the function (see
   documentation on \code{nlm}). Default is 1e-7. }
 \item{msScale}{scale function passed as the \code{scale} argument to
   the \code{nlm} function (see documentation on that function). Default
   is \code{lmeScale}.}
 \item{msVerbose}{a logical value passed as the \code{trace} argument to
   \code{nlm} (see documentation on that function). Default is
   \code{FALSE}.} 
 \item{returnObject}{a logical value indicating whether the fitted
   object should be returned when the maximum number of iterations is
   reached without convergence of the algorithm. Default is
   \code{FALSE}.} 
 \item{gradHess}{a logical value indicating whether numerical gradient
   vectors and Hessian matrices of the log-likelihood function should
   be used in the \code{nlm} optimization. This option is only available
   when the correlation structure (\code{corStruct}) and the variance
   function structure (\code{varFunc}) have no "varying" parameters and
   the \code{pdMat} classes used in the random effects structure are
   \code{pdSymm} (general positive-definite), \code{pdDiag} (diagonal),
   \code{pdIdent} (multiple of the identity),  or
   \code{pdCompSymm} (compound symmetry). Default is \code{TRUE}.}
 \item{apVar}{a logical value indicating whether the approximate
   covariance matrix of the variance-covariance parameters should be
   calculated. Default is \code{TRUE}.}
 \item{.relStep}{relative step for numerical derivatives
   calculations. Default is \code{.Machine$double.eps^(1/3)}.}
 \item{minAbsParApVar}{numeric value - minimum absolute parameter value
   in the approximate variance calculation.  The default is \code{0.05}.}
 \item{nlmStepMax}{stepmax value to be passed to nlm. See
     \code{\link{nlm}} for details. Default is 100.0}
 \item{natural}{a logical value indicating whether the \code{pdNatural}
   parametrization should be used for general positive-definite matrices
   (\code{pdSymm}) in \code{reStruct}, when the approximate covariance
   matrix of the estimators is calculated. Default is \code{TRUE}.}
}
\description{
  The values supplied in the function call replace the defaults and a
  list with all possible arguments is returned. The returned list is
  used as the \code{control} argument to the \code{lme} function.
}
\value{
  a list with components for each of the possible arguments.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{lme}}, \code{\link{nlm}}, \code{\link{optim}},
  \code{\link{lmeScale}}}
\examples{
# decrease the maximum number iterations in the ms call and
# request that information on the evolution of the ms iterations be printed
lmeControl(msMaxIter = 20, msVerbose = TRUE)
}
\keyword{models}

\eof
% $Id: lmeObject.Rd,v 1.4 2002/03/05 14:59:39 bates Exp $
\name{lmeObject}
\title{Fitted lme Object}
\alias{lmeObject}
\description{
  An object returned by the \code{lme} function, inheriting from class
  \code{lme} and representing a fitted linear mixed-effects
  model. Objects of this class have methods for the generic functions 
  \code{anova}, \code{coef}, \code{fitted}, \code{fixed.effects},
  \code{formula}, \code{getGroups}, \code{getResponse},
  \code{intervals}, \code{logLik}, \code{pairs}, \code{plot},
  \code{predict}, \code{print}, \code{random.effects}, \code{residuals},
  \code{summary}, and \code{update}.   

}
\value{
  The following components must be included in a legitimate \code{lme}
  object. 
  \item{apVar}{an approximate covariance matrix for the
    variance-covariance coefficients. If \code{apVar = FALSE} in the list
    of control values used in the call to \code{lme}, this
    component is equal to \code{NULL}.}
  \item{call}{a list containing an image of the \code{lme} call that
    produced the object.}
  \item{coefficients}{a list with two components, \code{fixed} and
    \code{random}, where the first is a vector containing the estimated
    fixed effects and the second is a list of matrices with the estimated
    random effects for each level of grouping. For each matrix in the
    \code{random} list, the columns refer to the random effects and the
    rows to the groups.}
  \item{contrasts}{a list with the contrasts used to represent factors
    in the fixed effects formula  and/or random effects formula. This
    information is important for making predictions from a new data
    frame in which not all levels of the original factors are
    observed. If no factors are used in the lme model, this component
    will be an empty list.}
  \item{dims}{a list with basic dimensions used in the lme fit,
    including the components \code{N} - the number of observations in
    the data, \code{Q} - the number of grouping levels, \code{qvec} -
    the number of random effects at each level from innermost to
    outermost (last two values are equal to zero and correspond to the
    fixed effects and the response), \code{ngrps} - the number of groups
    at each level from innermost to outermost (last two values are one
    and correspond to the fixed effects and the response), and
    \code{ncol} - the number of columns in the model matrix for each
    level of grouping from innermost to outermost (last two values are
    equal to the number of fixed effects and one).}
  \item{fitted}{a data frame with the fitted values as columns. The
    leftmost column corresponds to the population fixed effects
    (corresponding to the fixed effects only) and successive columns
    from left to right correspond to increasing levels of grouping.}
  \item{fixDF}{a list with components \code{X} and \code{terms}
    specifying the denominator degrees of freedom for, respectively,
    t-tests for the individual fixed effects and F-tests for the
    fixed-effects terms in the models.}
  \item{groups}{a data frame with the grouping factors as
    columns. The grouping level increases from left to right.}
  \item{logLik}{the (restricted) log-likelihood at convergence.}
  \item{method}{the estimation method: either \code{"ML"} for maximum
    likelihood, or \code{"REML"} for restricted maximum likelihood.}
  \item{modelStruct}{an object inheriting from class \code{lmeStruct},
    representing a list of mixed-effects model components, such
    as \code{reStruct}, \code{corStruct}, and \code{varFunc} objects.} 
  \item{numIter}{the number of iterations used in the iterative
    algorithm.} 
  \item{residuals}{a data frame with the residuals as columns. The
    leftmost column corresponds to the population residuals
    and successive columns from left to right correspond to increasing
    levels of grouping.} 
  \item{sigma}{the estimated within-group error standard deviation.}
  \item{varFix}{an approximate covariance matrix of the
    fixed effects estimates.}
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lme}}, \code{lmeStruct}}

\keyword{models}

\eof
\name{lmeScale}
\title{Scale for lme Optimization}
\usage{
lmeScale(start)
}
\alias{lmeScale}
\arguments{
  \item{start}{the starting values for the coefficients to be estimated.}
}
\description{
  This function calculates the scales to be used for each coefficient
  estimated through an \code{nlm} optimization in the \code{lme}
  function. If all initial values are zero, the scale is set to one for
  all coefficients; else, the scale for a coefficient with non-zero
  initial value is equal to the inverse of its initial value and the
  scale for the coefficients with initial value equal to zero is set to
  the median of the non-zero initial value coefficients.
}
\value{
  a vector with the scales to be used in \code{nlm} for estimating the
  coefficients. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{nlm}}}
\keyword{models}

\eof
% $Id: lmeStruct.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{lmeStruct}
\title{Linear Mixed-Effects Structure}
\usage{
lmeStruct(reStruct, corStruct, varStruct)
}
\alias{lmeStruct}
\arguments{
 \item{reStruct}{a \code{reStruct} representing a random effects
   structure.}
 \item{corStruct}{an optional \code{corStruct} object, representing a
   correlation structure. Default is \code{NULL}.}
 \item{varStruct}{an optional \code{varFunc} object, representing a
   variance function structure. Default is \code{NULL}.}
   
}
\description{
  A linear mixed-effects structure is a list of model components
  representing different sets of parameters in the linear mixed-effects
  model. An \code{lmeStruct} list must contain at least a
  \code{reStruct} object, but may also contain \code{corStruct} and
  \code{varFunc} objects. \code{NULL} arguments are not included in the
  \code{lmeStruct} list. 
}
\value{
  a list of model components determining the parameters to be estimated
  for the associated linear mixed-effects model.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lme}}, \code{\link{reStruct}},
  \code{\link{corClasses}}, \code{\link{varFunc}} }

\examples{
lms1 <- lmeStruct(reStruct(~age), corAR1(), varPower())
}
\keyword{models}

\eof
% $Id: logDet.Rd,v 1.4.2.1 2003/04/17 22:28:45 bates Exp $
\name{logDet}
\title{Extract the Logarithm of the Determinant}
\usage{
logDet(object, \dots)
}
\alias{logDet}
\arguments{
 \item{object}{any object from which a matrix, or list of matrices, can
   be extracted}
 \item{\dots}{some methods for this generic function require additional
   arguments.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include: \code{corStruct}, several \code{pdMat} classes,
  and \code{reStruct}. 
}
\value{
  will depend on the method function used; see the appropriate
  documentation. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{logLik}}}
\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: logDet.corStruct.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{logDet.corStruct}
\title{Extract corStruct Log-Determinant}
\usage{
\method{logDet}{corStruct}(object, covariate, \dots)
}
\alias{logDet.corStruct}
\alias{logDet.corIdent}
\arguments{
 \item{object}{an object inheriting from class \code{corStruct},
   representing a correlation structure.}
 \item{covariate}{an optional covariate vector (matrix), or list of
   covariate vectors (matrices), at which values the correlation matrix,
   or list of correlation  matrices, are to be evaluated. Defaults to
   \code{getCovariate(object)}.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts the logarithm of the determinant of a
  square-root factor of the correlation matrix associated with
  \code{object}, or the sum of the log-determinants of square-root
  factors of the list of correlation matrices associated with
  \code{object}.  
}
\value{
  the log-determinant of a square-root factor of the correlation matrix
  associated with \code{object}, or the sum of the log-determinants of
  square-root factors of the list of correlation matrices associated
  with \code{object}. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{logLik.corStruct}}, \code{\link{corMatrix.corStruct}}}

\examples{
cs1 <- corAR1(0.3)
logDet(cs1, covariate = 1:4)
}
\keyword{models}

\eof
% $Id: logDet.pdMat.Rd,v 1.8.2.2 2003/04/17 22:29:35 bates Exp $
\name{logDet.pdMat}
\title{Extract Log-Determinant from a pdMat Object}
\usage{
\method{logDet}{pdMat}(object, \dots)
}
\alias{logDet.pdMat}
\alias{logDet.pdBlocked}
\alias{logDet.pdCompSymm}
\alias{logDet.pdDiag}
\alias{logDet.pdIdent}
\alias{logDet.pdNatural}
\alias{logDet.pdSymm}
\arguments{
 \item{object}{an object inheriting from class \code{pdMat}, representing
   a positive definite matrix.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts the logarithm of the determinant of a
  square-root factor of the positive-definite matrix represented by
  \code{object}.  
}
\value{
  the log-determinant of a square-root factor of the positive-definite
  matrix represented by \code{object}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{pdMat}}}

\examples{
pd1 <- pdSymm(diag(1:3))
logDet(pd1)
}
\keyword{models}

\eof
% $Id: logDet.reStruct.Rd,v 1.6 2001/10/30 20:51:15 bates Exp $
\name{logDet.reStruct}
\title{Extract reStruct Log-Determinants}
\usage{
\method{logDet}{reStruct}(object, \dots)
}
\alias{logDet.reStruct}
\arguments{
 \item{object}{an object inheriting from class \code{reStruct},
   representing a random effects structure and consisting of a list of
   \code{pdMat} objects.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  Calculates, for each of the \code{pdMat} components of \code{object},
  the logarithm of the determinant of a square-root factor.  
}
\value{
  a vector with the log-determinants of square-root factors of the
  \code{pdMat} components of \code{object}.
}

\author{Jose Pinheiro}

\seealso{\code{\link{reStruct}}, \code{\link{pdMat}}}

\examples{
rs1 <- reStruct(list(A = pdSymm(diag(1:3), form = ~Score),
  B = pdDiag(2 * diag(4), form = ~Educ)))
logDet(rs1)
}
\keyword{models}

\eof
% $Id: logLik.corStruct.Rd,v 1.7.2.1 2002/08/09 19:45:29 bates Exp $
\name{logLik.corStruct}
\title{Extract corStruct Log-Likelihood}
\usage{
\method{logLik}{corStruct}(object, data, \dots)
}
\alias{logLik.corStruct}
\arguments{
 \item{object}{an object inheriting from class \code{corStruct},
   representing a correlation structure.}
 \item{data}{this argument is included to make this method function
   compatible with other \code{logLik} methods and will be ignored.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts the component of a Gaussian
  log-likelihood associated with the correlation structure, which is
  equal to the negative of the logarithm of the determinant (or sum of
  the logarithms of the determinants) of the matrix (or matrices)
  represented by \code{object}.
}
\value{
  the negative of the logarithm of the determinant (or sum of
  the logarithms of the determinants) of the correlation matrix (or
  matrices) represented by \code{object}.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{logDet.corStruct}} }

\examples{
data(Orthodont)
cs1 <- corAR1(0.2)
cs1 <- Initialize(cs1, data = Orthodont)
logLik(cs1)
}
\keyword{models}

\eof
% $Id: logLik.glsStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{logLik.glsStruct}
\title{Log-Likelihood of a glsStruct Object}
\usage{
\method{logLik}{glsStruct}(object, Pars, conLin, \dots)
}
\alias{logLik.glsStruct}

\arguments{
 \item{object}{an object inheriting from class \code{glsStruct},
   representing a list of linear model components, such as
   \code{corStruct} and \code{varFunc} objects.}
 \item{Pars}{the parameter values at which the (restricted)
   log-likelihood is to be evaluated.}
 \item{conLin}{an optional condensed linear model object, consisting of
   a list with components \code{"Xy"}, corresponding to a regression
   matrix (\code{X}) combined with a response vector (\code{y}), and 
   \code{"logLik"}, corresponding to the log-likelihood of the
   underlying linear model. Defaults to \code{attr(object, "conLin")}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  \code{Pars} is used to update the coefficients of the model components
  of \code{object} and the individual (restricted) log-likelihood
  contributions of each component are added together. The type of
  log-likelihood (restricted or not) is determined by the
  \code{settings} attribute of \code{object}.
}
\value{
  the (restricted) log-likelihood for the linear model described by
  \code{object}, evaluated at \code{Pars}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gls}}, \code{\link{glsStruct}}}

\keyword{models}

\eof
% $Id: logLik.gnls.Rd,v 1.8 2002/03/05 14:59:39 bates Exp $
\name{logLik.gnls}
\title{Log-Likelihood of a gnls Object}
\usage{
\method{logLik}{gnls}(object, REML, \dots)
}
\alias{logLik.gnls}
\arguments{
 \item{object}{an object inheriting from class \code{gnls}, representing
   a generalized nonlinear least squares fitted model.}
 \item{REML}{an optional logical value. If \code{TRUE} the restricted
   log-likelihood is returned, else, if \code{FALSE}, the log-likelihood
   is returned. Defaults to \code{FALSE}. }
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  Returns the log-likelihood value of the nonlinear model represented by
  \code{object} evaluated at the estimated coefficients.  
}
\value{
  the log-likelihood of the linear model represented by
  \code{object} evaluated at the estimated coefficients.  
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{gnls}}}
\examples{
data(Soybean)
fm1 <- gnls(weight ~ SSlogis(Time, Asym, xmid, scal), Soybean,
            weights = varPower())
logLik(fm1)
}
\keyword{models}

\eof
% $Id: logLik.gnlsStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{logLik.gnlsStruct}
\title{Log-Likelihood of a gnlsStruct Object}
\usage{
\method{logLik}{gnlsStruct}(object, Pars, conLin, \dots)
}
\alias{logLik.gnlsStruct}

\arguments{
 \item{object}{an object inheriting from class \code{gnlsStruct},
   representing a list of model components, such as
   \code{corStruct} and \code{varFunc} objects, and attributes
   specifying the underlying nonlinear model and the response variable.}
 \item{Pars}{the parameter values at which the log-likelihood is to be
   evaluated.} 
 \item{conLin}{an optional condensed linear model object, consisting of
   a list with components \code{"Xy"}, corresponding to a regression
   matrix (\code{X}) combined with a response vector (\code{y}), and 
   \code{"logLik"}, corresponding to the log-likelihood of the
   underlying nonlinear model. Defaults to \code{attr(object,
     "conLin")}.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  \code{Pars} is used to update the coefficients of the model components
  of \code{object} and the individual log-likelihood
  contributions of each component are added together.
}
\value{
  the log-likelihood for the linear model described by \code{object},
  evaluated at \code{Pars}.  
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gnls}}, \code{\link{gnlsStruct}}}

\keyword{models}

\eof
% $Id: logLik.lmList.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{logLik.lmList}
\title{Log-Likelihood of an lmList Object}
\usage{
\method{logLik}{lmList}(object, REML, pool, \dots)
}
\alias{logLik.lmList}
\arguments{
 \item{object}{an object inheriting from class \code{lmList}, representing
   a list of \code{lm} objects with a common model.
 }
 \item{REML}{an optional logical value. If \code{TRUE} the restricted
   log-likelihood is returned, else, if \code{FALSE}, the log-likelihood
   is returned. Defaults to \code{FALSE}.
 }
 \item{pool}{an optional logical value indicating whether all \code{lm}
   components of \code{object} may be assumed to have the same error
   variance. Default is \code{attr(object, "pool")}.
 }
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
 If \code{pool=FALSE}, the (restricted) log-likelihoods of the \code{lm}
 components of \code{object} are summed together. Else, the (restricted)
 log-likelihood of the \code{lm} fit with different coefficients for
 each level of the grouping factor associated with the partitioning of
 the \code{object} components is obtained.
}
\value{
  either the sum of the (restricted) log-likelihoods of each \code{lm}
  component in \code{object}, or the (restricted) log-likelihood for the
  \code{lm} fit with separate coefficients for each component of
  \code{object}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lmList}}}

\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age | Subject, Orthodont)
logLik(fm1)   # returns NA when it should not
}
\keyword{models}

\eof
% $Id: logLik.lme.Rd,v 1.7.2.1 2003/04/17 22:28:45 bates Exp $
\name{logLik.lme}
\title{Log-Likelihood of an lme Object}
\usage{
\method{logLik}{lme}(object, REML, \dots)
}
\alias{logLik.lme}
\alias{logLik.gls}
\arguments{
 \item{object}{an object inheriting from class \code{lme}, representing
   a fitted linear mixed-effects model.}
 \item{REML}{an optional logical value. If \code{TRUE} the restricted
   log-likelihood is returned, else, if \code{FALSE}, the log-likelihood
   is returned. Defaults to \code{FALSE}. }
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  If \code{REML=FALSE}, returns the log-likelihood value of the linear
  mixed-effects model represented by \code{object} evaluated at the
  estimated  coefficients; else, the restricted log-likelihood evaluated
  at the estimated coefficients is returned.
}
\value{
  the (restricted) log-likelihood of the model represented by
  \code{object} evaluated at the estimated coefficients.
}
\references{
  Harville, D.A. (1974) ``Bayesian Inference for Variance Components
  Using Only Error Contrasts'', \emph{Biometrika}, \bold{61}, 383--385.
}
\author{Jose Pinheiro and Douglas Bates}
\seealso{\code{\link{lme}},\code{\link{gls}}}
\examples{
data(Orthodont)
fm1 <- lme(distance ~ Sex * age, Orthodont, random = ~ age, method = "ML")
logLik(fm1)
logLik(fm1, REML = TRUE)
}
\keyword{models}

\eof
% $Id: logLik.lmeStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{logLik.lmeStruct}
\title{Log-Likelihood of an lmeStruct Object}
\usage{
\method{logLik}{lmeStruct}(object, Pars, conLin, \dots)
}
\alias{logLik.lmeStruct}
\alias{logLik.lmeStructInt}

\arguments{
 \item{object}{an object inheriting from class \code{lmeStruct},
   representing a list of linear mixed-effects model components, such as
   \code{reStruct}, \code{corStruct}, and \code{varFunc} objects.}
 \item{Pars}{the parameter values at which the (restricted)
   log-likelihood is to be evaluated.}
 \item{conLin}{an optional condensed linear model object, consisting of
   a list with components \code{"Xy"}, corresponding to a regression
   matrix (\code{X}) combined with a response vector (\code{y}), and 
   \code{"logLik"}, corresponding to the log-likelihood of the
   underlying lme model. Defaults to \code{attr(object, "conLin")}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  \code{Pars} is used to update the coefficients of the model components
  of \code{object} and the individual (restricted) log-likelihood
  contributions of each component are added together. The type of
  log-likelihood (restricted or not) is determined by the
  \code{settings} attribute of \code{object}.
}
\value{
  the (restricted) log-likelihood for the linear mixed-effects model
  described by \code{object}, evaluated at \code{Pars}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lme}}, \code{\link{lmeStruct}}}

\keyword{models}

\eof
% $Id: logLik.reStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{logLik.reStruct}
\title{Calculate reStruct Log-Likelihood}
\usage{
\method{logLik}{reStruct}(object, conLin, \dots)
}
\alias{logLik.reStruct}
\arguments{
 \item{object}{an object inheriting from class \code{reStruct},
   representing a random effects structure and consisting of a list of
   \code{pdMat} objects.} 
 \item{conLin}{a condensed linear model object, consisting of a list
   with components \code{"Xy"}, corresponding to a regression matrix
   (\code{X}) combined with a response vector (\code{y}), and
   \code{"logLik"}, corresponding to the log-likelihood of the
   underlying model.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  Calculates the log-likelihood, or restricted log-likelihood, of the
  Gaussian linear mixed-effects model represented by \code{object} and
  \code{conLin} (assuming spherical within-group covariance structure),
  evaluated at \code{coef(object)}. The \code{settings} attribute of
  \code{object} determines whether the log-likelihood, or the restricted
  log-likelihood, is to be calculated. The computational methods are 
  described in Bates and Pinheiro (1998).
}
\value{
  the log-likelihood, or restricted log-likelihood, of linear
  mixed-effects model represented by \code{object} and \code{conLin},
  evaluated at \code{coef{object}}.
}

\references{
  Bates, D.M. and Pinheiro, J.C. (1998) "Computational methods for
  multilevel models" available in PostScript or PDF formats at
  http://franz.stat.wisc.edu/pub/NLME/
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{reStruct}}, \code{\link{pdMat}}}

\keyword{models}

\eof
% $Id: logLik.varFunc.Rd,v 1.7.2.1 2002/08/09 19:45:29 bates Exp $
\name{logLik.varFunc}
\title{Extract varFunc logLik}
\usage{
\method{logLik}{varFunc}(object, data, \dots)
}
\alias{logLik.varFunc}
\alias{logLik.varComb}
\arguments{
 \item{object}{an object inheriting from class \code{varFunc},
   representing a variance function structure.} 
 \item{data}{this argument is included to make this method function
   compatible with other \code{logLik} methods and will be ignored.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function extracts the component of a Gaussian
  log-likelihood associated with the variance function structure
  represented by \code{object}, which is equal to the sum of the
  logarithms of the corresponding weights. 
}
\value{
  the sum of the logarithms of the weights corresponding to the variance
  function structure represented by \code{object}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}


\examples{
data(Orthodont)
vf1 <- varPower(form = ~age)
vf1 <- Initialize(vf1, Orthodont)
coef(vf1) <- 0.1
logLik(vf1)
}
\keyword{models}

\eof
% $Id: model.matrix.reStruct.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{model.matrix.reStruct}
\title{reStruct Model Matrix}
\usage{
\method{model.matrix}{reStruct}(object, data, contrast, \dots)
}
\alias{model.matrix.reStruct}
\arguments{
 \item{object}{an object inheriting from class \code{reStruct},
   representing a random effects structure and consisting of a list of
   \code{pdMat} objects.} 
 \item{data}{a data frame in which to evaluate the variables defined in
   \code{formula(object)}.}
 \item{contrast}{an optional named list specifying the contrasts to be used
   for representing the \code{factor} variables in \code{data}. The
   components names should match the names of the variables in
   \code{data} for which the contrasts are to be specified. The
   components of this list will be used as the \code{contrasts}
   attribute of the corresponding factor. If missing, the default
   contrast specification is used.}  
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The model matrices for each element of \code{formula(object)},
  calculated using \code{data}, are bound together column-wise. When
  multiple grouping levels are present (i.e. when \code{length(object) >
    1}), the individual model matrices are combined from innermost (at
  the leftmost position) to outermost (at the rightmost position). 
}
\value{
  a matrix obtained by binding together, column-wise, the model matrices
  for each element of \code{formula(object)}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{model.matrix}}, \code{\link{contrasts}},
  \code{\link{reStruct}}, \code{\link{formula.reStruct}}} 

\examples{
data(Pixel)
rs1 <- reStruct(list(Dog = ~day, Side = ~1), data = Pixel)
model.matrix(rs1, Pixel)
}
\keyword{models}

\eof
% $Id: needUpdate.Rd,v 1.5.2.1 2002/08/09 19:45:29 bates Exp $
\name{needUpdate}
\title{Check if Update is Needed}
\usage{
needUpdate(object)
}
\alias{needUpdate}
\alias{needUpdate.default}
\alias{needUpdate.varComb}
\alias{needUpdate.varIdent}
\arguments{
 \item{object}{any object}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. By default, it tries to extract a
  \code{needUpdate} attribute of \code{object}. If this is \code{NULL}
  or \code{FALSE} it returns \code{FALSE}; else it returns \code{TRUE}.
  Updating of objects usually takes place in iterative algorithms in
  which auxiliary quantities associated with the object, and not being
  optimized over, may change.
}
\value{
  a logical value indicating whether \code{object} needs to be updated. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\examples{
data(Orthodont)
vf1 <- varExp()
vf1 <- Initialize(vf1, data = Orthodont)
needUpdate(vf1)
}
\keyword{models}

\eof
% $Id: needUpdate.modelStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{needUpdate.modelStruct}
\title{Check if a modelStruct Object Needs Updating}
\usage{
\method{needUpdate}{modelStruct}(object)
}
\alias{needUpdate.modelStruct}
\alias{needUpdate.corStruct}
\alias{needUpdate.reStruct}
\arguments{
 \item{object}{an object inheriting from class \code{modelStruct},
   representing a list of model components, such as \code{corStruct} and
   \code{varFunc} objects.}
}
\description{
  This method function checks if any of the elements of \code{object}
  needs to be updated. Updating of objects usually takes place in
  iterative algorithms in which auxiliary quantities associated with the
  object, and not being optimized over, may change.
}
\value{
  a logical value indicating whether any element of \code{object} needs
  to be updated.  
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\examples{
lms1 <- lmeStruct(reStruct = reStruct(pdDiag(diag(2), ~age)),
   varStruct = varPower(form = ~age))
needUpdate(lms1)
}
\keyword{models}

\eof
% $Id: nlme.Rd,v 1.8.2.3 2003/06/12 14:58:30 bates Exp $
\name{nlme}
\title{Nonlinear Mixed-Effects Models}
\usage{
nlme(model, data, fixed, random, groups, start, correlation, weights,
     subset, method, na.action, naPattern, control, verbose)
}
\alias{nlme}
\alias{nlme.formula}
\arguments{
  \item{model}{a nonlinear model formula, with the response on the left
    of a \code{~} operator and an expression involving parameters and
    covariates on the right, or an \code{nlsList} object.  If
    \code{data} is given, all names used in the formula should be
    defined as parameters or variables in the data frame. The method
    function \code{nlme.nlsList} is documented separately.} 
 \item{data}{an optional data frame containing the variables named in
   \code{model}, \code{fixed}, \code{random}, \code{correlation},
   \code{weights}, \code{subset}, and \code{naPattern}.  By default the
   variables are taken from the environment from which \code{nlme} is
   called.} 
 \item{fixed}{a two-sided linear formula of the form
   \code{f1+...+fn~x1+...+xm}, or a list of two-sided formulas of the form
   \code{f1~x1+...+xm}, with possibly different models for different
   parameters. The \code{f1,...,fn} are the names of parameters included on
   the right hand side of \code{model} and the \code{x1+...+xm}
   expressions define linear models for these parameters (when the left
   hand side of the formula contains several parameters, they all are
   assumed to follow the same linear model, described by the right hand
   side expression).
   A \code{1} on the right hand side of the formula(s) indicates a single
   fixed effects for the corresponding parameter(s).}
 \item{random}{optionally, any of the following: (i) a two-sided formula
   of the form \code{r1+...+rn~x1+...+xm | g1/.../gQ}, with
   \code{r1,...,rn} naming parameters included on the right
   hand side of \code{model}, \code{x1+...+xm} specifying the
   random-effects model for
   these parameters and \code{g1/.../gQ} the grouping structure
   (\code{Q} may be equal to 1, in which case no \code{/} is
   required). The random effects formula will be repeated 
   for all levels of grouping, in the case of multiple levels of
   grouping; (ii) a two-sided formula of the form
   \code{r1+...+rn~x1+..+xm}, a list of two-sided formulas of the form
   \code{r1~x1+...+xm}, with possibly different random-effects models
   for different parameters, a \code{pdMat} object with a two-sided
   formula, or list of two-sided formulas (i.e. a non-\code{NULL} value for
   \code{formula(random)}), or a list of pdMat objects with two-sided
   formulas, or lists of two-sided formulas. In this case, the grouping
   structure formula will be given in \code{groups}, or derived from the
   data used to fit the nonlinear mixed-effects model, which should
   inherit from class  \code{groupedData},; (iii) a named list
   of formulas, lists of formulas, or \code{pdMat} objects as in (ii),
   with the grouping factors as names. The order of nesting will be
   assumed the same as the order of the order of the elements in the
   list; (iv) an \code{reStruct} object. See the documentation on
   \code{pdClasses} for a description of the available \code{pdMat}
   classes. Defaults to \code{fixed}, 
   resulting in all fixed effects having also random effects.}
 \item{groups}{an optional one-sided formula of the form \code{~g1}
   (single level of nesting) or \code{~g1/.../gQ} (multiple levels of
   nesting), specifying the partitions of the data over which the random
   effects vary. \code{g1,...,gQ} must evaluate to factors in
   \code{data}. The order of nesting, when multiple levels are present,
   is taken from left to right (i.e. \code{g1} is the first level,
   \code{g2} the second, etc.).}
 \item{start}{an optional numeric vector, or list of initial estimates
   for the fixed effects and random effects. If declared as a numeric
   vector, it is converted internally to a list with a single component
   \code{fixed}, given by the vector. The \code{fixed} component
   is required, unless the model function inherits from class
   \code{selfStart}, in which case initial values will be derived from a
   call to \code{nlsList}. An optional \code{random} component is used to specify
   initial values for the random effects and should consist of a matrix,
   or a list of matrices with length equal to the number of grouping
   levels. Each matrix should have as many rows as the number of groups
   at the corresponding level and as many columns as the number of
   random effects in that level.}
 \item{correlation}{an optional \code{corStruct} object describing the
   within-group correlation structure. See the documentation of
   \code{corClasses} for a description of the available \code{corStruct}
   classes. Defaults to \code{NULL}, corresponding to no within-group
   correlations.} 
 \item{weights}{an optional \code{varFunc} object or one-sided formula
   describing the within-group heteroscedasticity structure. If given as
   a formula, it is used as the argument to \code{varFixed},
   corresponding to fixed variance weights. See the documentation on
   \code{varClasses} for a description of the available \code{varFunc}
   classes. Defaults to \code{NULL}, corresponding to homoscesdatic
   within-group errors.} 
 \item{subset}{an optional expression indicating the subset of the rows of
   \code{data} that should be used in the fit. This can be a logical
   vector, or a numeric vector indicating which observation numbers are
   to be included, or a  character  vector of the row names to be
   included.  All observations are included by default.}
 \item{method}{a character string.  If \code{"REML"} the model is fit by
   maximizing the restricted log-likelihood.  If \code{"ML"} the
   log-likelihood is maximized.  Defaults to \code{"ML"}.}
 \item{na.action}{a function that indicates what should happen when the
   data contain \code{NA}s.  The default action (\code{na.fail}) causes
   \code{nlme} to print an error message and terminate if there are any
   incomplete observations.}
 \item{naPattern}{an expression or formula object, specifying which returned
   values are to be regarded as missing.}
 \item{control}{a list of control values for the estimation algorithm to
   replace the default values returned by the function \code{nlmeControl}.
   Defaults to an empty list.}
 \item{verbose}{an optional logical value. If \code{TRUE} information on
   the evolution of the iterative algorithm is printed. Default is
   \code{FALSE}.}
}
\description{
  This generic function fits a nonlinear mixed-effects model in the
  formulation described in Lindstrom and Bates (1990) but allowing for nested
  random effects. The within-group errors are allowed to be correlated
  and/or have unequal variances. 
}
\value{
  an object of class \code{nlme} representing the nonlinear
  mixed-effects model fit. Generic functions such as \code{print},
  \code{plot} and \code{summary} have methods to show the results of the
  fit. See \code{nlmeObject} for the components of the fit. The functions
  \code{resid}, \code{coef}, \code{fitted}, \code{fixed.effects}, and
  \code{random.effects}  can be used to extract some of its components.
}
\references{
 The model formulation and computational methods are described in
 Lindstrom, M.J. and Bates, D.M. (1990). The variance-covariance
 parametrizations are described in Pinheiro, J.C. and Bates., D.M.
 (1996).   The different correlation structures available for the
 \code{correlation} argument are described in Box, G.E.P., Jenkins,
 G.M., and Reinsel G.C. (1994), Littel, R.C., Milliken, G.A., Stroup,
 W.W., and Wolfinger, R.D. (1996), and Venables, W.N. and Ripley,
 B.D. (1997). The use of variance functions for linear and nonlinear
 mixed effects models is presented in detail in Davidian, M. and
 Giltinan, D.M. (1995).  

 Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994) "Time Series
 Analysis: Forecasting and Control", 3rd Edition, Holden-Day. 

 Davidian, M. and Giltinan, D.M. (1995) "Nonlinear Mixed Effects Models
 for Repeated Measurement Data", Chapman and Hall.

 Laird, N.M. and Ware, J.H. (1982) "Random-Effects Models for
 Longitudinal Data", Biometrics, 38, 963-974.  

 Littel, R.C., Milliken, G.A., Stroup, W.W., and Wolfinger, R.D. (1996)
 "SAS Systems for Mixed Models", SAS Institute.

 Lindstrom, M.J. and Bates, D.M. (1990) "Nonlinear Mixed Effects Models
 for Repeated Measures Data", Biometrics, 46, 673-687.

 Pinheiro, J.C. and Bates., D.M.  (1996) "Unconstrained
 Parametrizations for Variance-Covariance Matrices", Statistics and
 Computing, 6, 289-296.

 Venables, W.N. and Ripley, B.D. (1997) "Modern Applied Statistics with
 S-plus", 2nd Edition, Springer-Verlag.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{
  \code{\link{nlmeControl}}, \code{\link{nlme.nlsList}},
  \code{\link{nlmeObject}}, \code{\link{nlsList}},
  \code{\link{reStruct}}, \code{\link{varFunc}}, \code{\link{pdClasses}},
  \code{\link{corClasses}}, \code{\link{varClasses}}
}
\examples{
data(Loblolly)
fm1 <- nlme(height ~ SSasymp(age, Asym, R0, lrc),
            data = Loblolly,
            fixed = Asym + R0 + lrc ~ 1,
            random = Asym ~ 1,
            start = c(Asym = 103, R0 = -8.5, lrc = -3.3))
summary(fm1)
fm2 <- update(fm1, random = pdDiag(Asym + lrc ~ 1))
summary(fm2)
}
\keyword{models}

\eof
% $Id: nlme.nlsList.Rd,v 1.8 2002/03/05 14:59:39 bates Exp $
\name{nlme.nlsList}
\title{NLME fit from nlsList Object}
\usage{
\method{nlme}{nlsList}(model, data, fixed, random, groups, start, correlation, weights,
     subset, method, na.action, naPattern, control, verbose)
}
\alias{nlme.nlsList}
\arguments{
  \item{model}{an object inheriting from class \code{nlsList},
    representing a list of \code{nls} fits with a common model.}
  \item{data}{this argument is included for consistency with the generic
    function. It is ignored in this method function.}
  \item{fixed}{this argument is included for consistency with the generic
    function. It is ignored in this method function.}
  \item{random}{an optional one-sided linear formula with no conditioning
   expression, or a \code{pdMat} object with a \code{formula}
   attribute. Multiple levels of grouping are not allowed with this
   method function.  Defaults to a formula consisting of the right hand
   side of \code{formula(fixed)}.} 
 \item{groups}{an optional one-sided formula of the form \code{~g1}
   (single level of nesting) or \code{~g1/.../gQ} (multiple levels of
   nesting), specifying the partitions of the data over which the random
   effects vary. \code{g1,...,gQ} must evaluate to factors in
   \code{data}. The order of nesting, when multiple levels are present,
   is taken from left to right (i.e. \code{g1} is the first level,
   \code{g2} the second, etc.).}
 \item{start}{an optional numeric vector, or list of initial estimates
   for the fixed effects and random effects. If declared as a numeric
   vector, it is converted internally to a list with a single component
   \code{fixed}, given by the vector. The \code{fixed} component
   is required, unless the model function inherits from class
   \code{selfStart}, in which case initial values will be derived from a
   call to \code{nlsList}. An optional \code{random} component is used to specify
   initial values for the random effects and should consist of a matrix,
   or a list of matrices with length equal to the number of grouping
   levels. Each matrix should have as many rows as the number of groups
   at the corresponding level and as many columns as the number of
   random effects in that level.}
 \item{correlation}{an optional \code{corStruct} object describing the
   within-group correlation structure. See the documentation of
   \code{corClasses} for a description of the available \code{corStruct}
   classes. Defaults to \code{NULL}, corresponding to no within-group
   correlations.} 
 \item{weights}{an optional \code{varFunc} object or one-sided formula
   describing the within-group heteroscedasticity structure. If given as
   a formula, it is used as the argument to \code{varFixed},
   corresponding to fixed variance weights. See the documentation on
   \code{varClasses} for a description of the available \code{varFunc}
   classes. Defaults to \code{NULL}, corresponding to homoscesdatic
   within-group errors.} 
 \item{subset}{an optional expression indicating the subset of the rows of
   \code{data} that should be used in the fit. This can be a logical
   vector, or a numeric vector indicating which observation numbers are
   to be included, or a  character  vector of the row names to be
   included.  All observations are included by default.}
 \item{method}{a character string.  If \code{"REML"} the model is fit by
   maximizing the restricted log-likelihood.  If \code{"ML"} the
   log-likelihood is maximized.  Defaults to \code{"ML"}.}
 \item{na.action}{a function that indicates what should happen when the
   data contain \code{NA}s.  The default action (\code{na.fail}) causes
   \code{nlme} to print an error message and terminate if there are any
   incomplete observations.}
 \item{naPattern}{an expression or formula object, specifying which returned
   values are to be regarded as missing.}
 \item{control}{a list of control values for the estimation algorithm to
   replace the default values returned by the function \code{nlmeControl}.
   Defaults to an empty list.}
 \item{verbose}{an optional logical value. If \code{TRUE} information on
   the evolution of the iterative algorithm is printed. Default is
   \code{FALSE}.}
}
\description{
  If the random effects names defined in \code{random} are a subset of
  the \code{lmList} object coefficient names, initial estimates for the
  covariance matrix of the random effects are obtained (overwriting any
  values given in \code{random}). \code{formula(fixed)} and the
  \code{data} argument in the calling sequence used to obtain
  \code{fixed} are passed as the \code{fixed} and \code{data} arguments
  to \code{nlme.formula}, together with any other additional arguments in
  the function call. See the documentation on \code{nlme.formula} for a
  description of that function. 
}
\value{
  an object of class \code{nlme} representing the linear mixed-effects
  model fit. Generic functions such as \code{print}, \code{plot} and
  \code{summary} have methods to show the results of the fit. See
  \code{nlmeObject} for the components of the fit. The functions
  \code{resid}, \code{coef}, \code{fitted}, \code{fixed.effects}, and
  \code{random.effects}  can be used to extract some of its components.
}
\references{
 The computational methods are described in Bates, D.M. and Pinheiro
 (1998) and follow on the general framework of Lindstrom, M.J. and Bates,
 D.M. (1988). The model formulation is described in Laird, N.M. and Ware,
 J.H. (1982).  The variance-covariance parametrizations are described in
 <Pinheiro, J.C. and Bates., D.M.  (1996).   The different correlation
 structures available for the \code{correlation} argument are described
 in Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994), Littel, R.C.,
 Milliken, G.A., Stroup, W.W., and Wolfinger, R.D. (1996), and Venables,
 W.N. and Ripley, B.D. (1997). The use of variance functions for linear
 and nonlinear mixed effects models is presented in detail in Davidian,
 M. and Giltinan, D.M. (1995). 

 Bates, D.M. and Pinheiro, J.C. (1998) "Computational methods for
 multilevel models" available in PostScript or PDF formats at
 http://franz.stat.wisc.edu/pub/NNLME/
 
 Box, G.E.P., Jenkins, G.M., and Reinsel G.C. (1994) "Time Series
 Analysis: Forecasting and Control", 3rd Edition, Holden-Day. 

 Davidian, M. and Giltinan, D.M. (1995) "Nonlinear Mixed Effects Models
 for Repeated Measurement Data", Chapman and Hall.

 Laird, N.M. and Ware, J.H. (1982) "Random-Effects Models for
 Longitudinal Data", Biometrics, 38, 963-974.  

 Lindstrom, M.J. and Bates, D.M. (1988) "Newton-Raphson and EM
 Algorithms for Linear Mixed-Effects Models for Repeated-Measures
 Data", Journal of the American Statistical Association, 83,
 1014-1022. 

 Littel, R.C., Milliken, G.A., Stroup, W.W., and Wolfinger, R.D. (1996)
 "SAS Systems for Mixed Models", SAS Institute.

 Pinheiro, J.C. and Bates., D.M.  (1996) "Unconstrained
 Parametrizations for Variance-Covariance Matrices", Statistics and
 Computing, 6, 289-296.

 Venables, W.N. and Ripley, B.D. (1997) "Modern Applied Statistics with
 S-plus", 2nd Edition, Springer-Verlag.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{nlme}}, \code{\link{lmList}},
  \code{\link{nlmeObject}} } 

\examples{
data(Loblolly)
fm1 <- nlsList(SSasymp, data = Loblolly)
fm2 <- nlme(fm1, random = Asym ~ 1)
summary(fm1)
summary(fm2)
}
\keyword{models}

\eof
% $Id: nlmeControl.Rd,v 1.8 2002/03/05 14:59:39 bates Exp $
\name{nlmeControl}
\title{Control Values for nlme Fit}
\usage{
nlmeControl(maxIter, pnlsMaxIter, msMaxIter, minScale,
            tolerance, niterEM, pnlsTol, msTol, msScale,
            returnObject, msVerbose, gradHess, apVar, .relStep,
            nlmStepMax, minAbsParApVar, natural)
}
\alias{nlmeControl}
\arguments{
 \item{maxIter}{maximum number of iterations for the \code{nlme}
   optimization algorithm. Default is 50.} 
 \item{pnlsMaxIter}{maximum number of iterations
   for the \code{PNLS} optimization step inside the \code{nlme}
   optimization. Default is 7.}
 \item{msMaxIter}{maximum number of iterations
   for the \code{nlm} optimization step inside the \code{nlme}
   optimization. Default is 50.}
 \item{minScale}{minimum factor by which to shrink the default step size
   in an attempt to decrease the sum of squares in the \code{PNLS} step.
   Default 0.001.}
 \item{tolerance}{tolerance for the convergence criterion in the
   \code{nlme} algorithm. Default is 1e-6.}
 \item{niterEM}{number of iterations for the EM algorithm used to refine
   the initial estimates of the random effects variance-covariance
   coefficients. Default is 25.}
 \item{pnlsTol}{tolerance for the convergence criterion in \code{PNLS}
   step. Default is 1e-3.} 
 \item{msTol}{tolerance for the convergence criterion in \code{nlm},
   passed as the \code{rel.tolerance} argument to the function (see
   documentation on \code{nlm}). Default is 1e-7. }
 \item{msScale}{scale function passed as the \code{scale} argument to
   the \code{nlm} function (see documentation on that function). Default
   is \code{lmeScale}.}
 \item{returnObject}{a logical value indicating whether the fitted
   object should be returned when the maximum number of iterations is
   reached without convergence of the algorithm. Default is
   \code{FALSE}.} 
 \item{msVerbose}{a logical value passed as the \code{trace} argument to
   \code{nlm} (see documentation on that function). Default is
   \code{FALSE}.} 
 \item{gradHess}{a logical value indicating whether numerical gradient
   vectors and Hessian matrices of the log-likelihood function should
   be used in the \code{nlm} optimization. This option is only available
   when the correlation structure (\code{corStruct}) and the variance
   function structure (\code{varFunc}) have no "varying" parameters and
   the \code{pdMat} classes used in the random effects structure are
   \code{pdSymm} (general positive-definite), \code{pdDiag} (diagonal),
   \code{pdIdent} (multiple of the identity),  or
   \code{pdCompSymm} (compound symmetry). Default is \code{TRUE}.}
 \item{apVar}{a logical value indicating whether the approximate
   covariance matrix of the variance-covariance parameters should be
   calculated. Default is \code{TRUE}.}
 \item{.relStep}{relative step for numerical derivatives
   calculations. Default is \code{.Machine$double.eps^(1/3)}.}
 \item{nlmStepMax}{stepmax value to be passed to nlm. See
     \code{\link{nlm}} for details. Default is 100.0}
 \item{minAbsParApVar}{numeric value - minimum absolute parameter value
   in the approximate variance calculation.  The default is \code{0.05}.}
 \item{natural}{a logical value indicating whether the \code{pdNatural}
   parametrization should be used for general positive-definite matrices
   (\code{pdSymm}) in \code{reStruct}, when the approximate covariance
   matrix of the estimators is calculated. Default is \code{TRUE}.}
}
\description{
  The values supplied in the function call replace the defaults and a
  list with all possible arguments is returned. The returned list is
  used as the \code{control} argument to the \code{nlme} function.
}
\value{
  a list with components for each of the possible arguments.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{nlme}}, \code{\link{nlm}}, \code{\link{optim}},
  \code{\link{nlmeStruct}}}
\examples{
# decrease the maximum number iterations in the ms call and
# request that information on the evolution of the ms iterations be printed
nlmeControl(msMaxIter = 20, msVerbose = TRUE)
}
\keyword{models}

\eof
% $Id: nlmeObject.Rd,v 1.4 2002/03/05 14:59:39 bates Exp $
\name{nlmeObject}
\title{Fitted nlme Object}
\alias{nlmeObject}
\description{
  An object returned by the \code{nlme} function, inheriting from class
  \code{nlme}, also inheriting from class \code{lme}, and representing a
  fitted nonlinear mixed-effects model. Objects of this class have
  methods for the generic functions  \code{anova}, \code{coef},
  \code{fitted}, \code{fixed.effects}, \code{formula}, \code{getGroups},
  \code{getResponse}, \code{intervals}, \code{logLik}, \code{pairs},
  \code{plot}, \code{predict}, \code{print}, \code{random.effects},
  \code{residuals}, \code{summary}, and \code{update}.   

}
\value{
  The following components must be included in a legitimate \code{nlme}
  object. 
  \item{apVar}{an approximate covariance matrix for the
    variance-covariance coefficients. If \code{apVar = FALSE} in the list
    of control values used in the call to \code{nlme}, this
    component is equal to \code{NULL}.}
  \item{call}{a list containing an image of the \code{nlme} call that
    produced the object.}
  \item{coefficients}{a list with two components, \code{fixed} and
    \code{random}, where the first is a vector containing the estimated
    fixed effects and the second is a list of matrices with the estimated
    random effects for each level of grouping. For each matrix in the
    \code{random} list, the columns refer to the random effects and the
    rows to the groups.}
  \item{contrasts}{a list with the contrasts used to represent factors
    in the fixed effects formula  and/or random effects formula. This
    information is important for making predictions from a new data
    frame in which not all levels of the original factors are
    observed. If no factors are used in the nlme model, this component
    will be an empty list.}
  \item{dims}{a list with basic dimensions used in the nlme fit,
    including the components \code{N} - the number of observations in
    the data, \code{Q} - the number of grouping levels, \code{qvec} -
    the number of random effects at each level from innermost to
    outermost (last two values are equal to zero and correspond to the
    fixed effects and the response), \code{ngrps} - the number of groups
    at each level from innermost to outermost (last two values are one
    and correspond to the fixed effects and the response), and
    \code{ncol} - the number of columns in the model matrix for each
    level of grouping from innermost to outermost (last two values are
    equal to the number of fixed effects and one).}
  \item{fitted}{a data frame with the fitted values as columns. The
    leftmost column corresponds to the population fixed effects
    (corresponding to the fixed effects only) and successive columns
    from left to right correspond to increasing levels of grouping.}
  \item{fixDF}{a list with components \code{X} and \code{terms}
    specifying the denominator degrees of freedom for, respectively,
    t-tests for the individual fixed effects and F-tests for the
    fixed-effects terms in the models.}
  \item{groups}{a data frame with the grouping factors as
    columns. The grouping level increases from left to right.}
  \item{logLik}{the (restricted) log-likelihood at convergence.}
  \item{map}{a list with components \code{fmap}, \code{rmap},
    \code{rmapRel}, and \code{bmap}, specifying various mappings for the
    fixed and random effects, used to generate predictions from the
    fitted object.}
  \item{method}{the estimation method: either \code{"ML"} for maximum
    likelihood, or \code{"REML"} for restricted maximum likelihood.}
  \item{modelStruct}{an object inheriting from class \code{nlmeStruct},
    representing a list of mixed-effects model components, such
    as \code{reStruct}, \code{corStruct}, and \code{varFunc} objects.} 
  \item{numIter}{the number of iterations used in the iterative
    algorithm.} 
  \item{residuals}{a data frame with the residuals as columns. The
    leftmost column corresponds to the population residuals
    and successive columns from left to right correspond to increasing
    levels of grouping.} 
  \item{sigma}{the estimated within-group error standard deviation.}
  \item{varFix}{an approximate covariance matrix of the
    fixed effects estimates.}
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{nlme}}, \code{nlmeStruct}}

\keyword{models}

\eof
% $Id: nlmeStruct.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{nlmeStruct}
\title{Nonlinear Mixed-Effects Structure}
\usage{
nlmeStruct(reStruct, corStruct, varStruct)
}
\alias{nlmeStruct}
\arguments{
 \item{reStruct}{a \code{reStruct} representing a random effects
   structure.}
 \item{corStruct}{an optional \code{corStruct} object, representing a
   correlation structure. Default is \code{NULL}.}
 \item{varStruct}{an optional \code{varFunc} object, representing a
   variance function structure. Default is \code{NULL}.}
}
\description{
  A nonlinear mixed-effects structure is a list of model components
  representing different sets of parameters in the nonlinear mixed-effects
  model. An \code{nlmeStruct} list must contain at least a
  \code{reStruct} object, but may also contain \code{corStruct} and
  \code{varFunc} objects. \code{NULL} arguments are not included in the
  \code{nlmeStruct} list. 
}
\value{
  a list of model components determining the parameters to be estimated
  for the associated nonlinear mixed-effects model.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{nlme}}, \code{\link{reStruct}},
  \code{\link{corClasses}}, \code{\link{varFunc}} }

\examples{
nlms1 <- nlmeStruct(reStruct(~age), corAR1(), varPower())
}
\keyword{models}

\eof
% $Id: nlsList.Rd,v 1.7 2002/01/28 03:44:17 bates Exp $
\name{nlsList}
\title{List of nls Objects with a Common Model}
\usage{
nlsList(model, data, start, control, level, subset, na.action, pool)
\method{update}{nlsList}(object, model, data, start, control, level,
      subset, na.action, pool, \dots)
}
\alias{nlsList}
\alias{nlsList.formula}
\alias{update.nlsList}
\arguments{
  \item{object}{an object inheriting from class \code{nlsList}, representing
      a list of fitted \code{nls} objects.}
  \item{model}{either a nonlinear model formula, with the response on
    the left of a \code{~} operator and an expression involving
    parameters, covariates, and a grouping factor separated by the
    \code{|} operator on the right, or a \code{selfStart} function.  The
    method function \code{nlsList.selfStart} is documented separately.
  }
  \item{data}{
    a data frame in which to interpret the variables named in
    \code{model}.
  }
  \item{start}{an optional named list with initial values for the
    parameters to be estimated in \code{model}. It is passed as the
    \code{start} argument to each \code{nls} call and is required when
    the nonlinear function in \code{model} does not inherit from class
    \code{selfStart}.
  }
  \item{control}{a list of control values passed as the \code{control}
    argument to \code{nls}. Defaults to an empty list.
  }
  \item{level}{an optional integer specifying the level of grouping to
    be used when multiple nested levels of grouping are present.}
 \item{subset}{an optional expression indicating the subset of the rows of
   \code{data} that should be used in the fit. This can be a logical
   vector, or a numeric vector indicating which observation numbers are
   to be included, or a  character  vector of the row names to be
   included.  All observations are included by default.}
  \item{na.action}{a function that indicates what should happen when the
    data contain \code{NA}s.  The default action (\code{na.fail}) causes
    \code{nlsList} to print an error message and terminate if there are any
    incomplete observations.
  }
  \item{pool}{
    an optional logical value that is preserved as an attribute of the
    returned value.  This will be used as the default for \code{pool} in
    calculations of standard deviations or standard errors for summaries.
  }
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  \code{Data} is partitioned according to the levels of the grouping
  factor defined in \code{model} and individual \code{nls} fits are
  obtained for each \code{data} partition, using the model defined in
  \code{model}.
}
\value{
  a list of \code{nls} objects with as many components as the number of
  groups defined by the grouping factor. Generic functions such as
  \code{coef}, \code{fixed.effects}, \code{lme}, \code{pairs},
  \code{plot}, \code{predict}, \code{random.effects}, \code{summary},
  and \code{update} have methods that can be applied to an \code{nlsList}
  object. 
}
\seealso{
\code{\link{nls}}, \code{\link{nlme.nlsList}}.
}
\examples{
data(CO2)
fm1 <- nlsList(uptake ~ SSasympOff(conc, Asym, lrc, c0),
   data = CO2, start = c(Asym = 30, lrc = -4.5, c0 = 52))
summary(fm1)
}
\keyword{models}


\eof
% $Id: nlsList.selfStart.Rd,v 1.7 2002/01/28 03:44:17 bates Exp $
\name{nlsList.selfStart}
\title{nlsList Fit from a selfStart Function}
\usage{
\method{nlsList}{selfStart}(model, data, start, control, level, subset, na.action, pool)
}
\alias{nlsList.selfStart}
\arguments{
  \item{model}{a \code{selfStart} model function, which calculates
    initial estimates for the model parameters from \code{data}.}
  \item{data}{a data frame in which to interpret the variables in
    \code{model}. Because no grouping factor can be specified in
    \code{model}, \code{data} must inherit from class
    \code{groupedData}. 
  }
  \item{start}{an optional named list with initial values for the
    parameters to be estimated in \code{model}. It is passed as the
    \code{start} argument to each \code{nls} call and is required when
    the nonlinear function in \code{model} does not inherit from class
    \code{selfStart}.
  }
  \item{control}{a list of control values passed as the \code{control}
    argument to \code{nls}. Defaults to an empty list.
  }
  \item{level}{an optional integer specifying the level of grouping to
    be used when multiple nested levels of grouping are present.}
 \item{subset}{an optional expression indicating the subset of the rows of
   \code{data} that should be used in the fit. This can be a logical
   vector, or a numeric vector indicating which observation numbers are
   to be included, or a  character  vector of the row names to be
   included.  All observations are included by default.}
  \item{na.action}{a function that indicates what should happen when the
    data contain \code{NA}s.  The default action (\code{na.fail}) causes
    \code{nlsList} to print an error message and terminate if there are any
    incomplete observations.
  }
  \item{pool}{
    an optional logical value that is preserved as an attribute of the
    returned value.  This will be used as the default for \code{pool} in
    calculations of standard deviations or standard errors for summaries.
  }
}
\description{
  The response variable and primary covariate in \code{formula(data)}
  are used together with \code{model} to construct the nonlinear model
  formula. This is used in the \code{nls} calls and, because a
  selfStarting model function can calculate initial estimates for its
  parameters from the data, no starting estimates need to be provided.
}
\value{
  a list of \code{nls} objects with as many components as the number of
  groups defined by the grouping factor. A \code{NULL} value is assigned
  to the components corresponding to clusters for which the \code{nls}
  algorithm failed to converge. Generic functions such as \code{coef},
  \code{fixed.effects}, \code{lme}, \code{pairs}, \code{plot},
  \code{predict}, \code{random.effects}, \code{summary}, and
  \code{update} have methods that can be applied to an \code{nlsList}
  object.  
}

\seealso{
  \code{\link{selfStart}}, \code{\link{groupedData}}, \code{\link{nls}},
  \code{\link{nlme.nlsList}}, \code{\link{nlsList.formula}}
}
\examples{
data(CO2)
fm1 <- nlsList(SSasympOff, CO2)
summary(fm1)
}
\keyword{models}

\eof
% $Id: pairs.compareFits.Rd,v 1.9 2002/04/16 14:28:53 bates Exp $
\name{pairs.compareFits}
\title{Pairs Plot of compareFits Object}
\usage{
\method{pairs}{compareFits}(x, subset, key, \dots)
}
\alias{pairs.compareFits}
\arguments{
 \item{x}{an object of class \code{compareFits}.}
 \item{subset}{an optional logical or integer vector specifying which
   rows of \code{x} should be used in the plots. If missing, all
   rows are used.}
 \item{key}{an optional logical value, or list. If \code{TRUE}, a legend
 is included at the top of the plot indicating which symbols (colors)
 correspond to which objects being compared. If \code{FALSE}, no legend
 is included. If given as a list, \code{key} is passed down as an
 argument to the \code{trellis} function generating the plots
 (\code{splom} or  \code{xyplot}). Defaults to \code{TRUE}.}
 \item{\dots}{optional arguments passed down to the \code{trellis}
   function generating the plots.}
}
\description{
  Scatter plots of the values being compared are generated for each pair
  of coefficients in \code{x}. Different symbols (colors) are used
  for each object being compared and values corresponding to the same
  group are joined by a line, to facilitate comparison of fits. If only
  two coefficients are present, the \code{trellis} function
  \code{xyplot} is used; otherwise the \code{trellis} function \code{splom}
  is used.
}
\value{
  Pairwise scatter plots of the values being compared, with different
  symbols (colors) used for each object under comparison.
}
\author{Jose Pinheiro and Douglas Bates }
\seealso{
  \code{\link{compareFits}},
  \code{\link{plot.compareFits}},
  \code{\link[lattice]{xyplot}},
  \code{\link[lattice]{splom}}
}
\examples{
data(Orthodont)
fm1 <- lmList(Orthodont)
fm2 <- lme(Orthodont)
pairs(compareFits(coef(fm1), coef(fm2)))
}
\keyword{models}

\eof
% $Id: pairs.lmList.Rd,v 1.10.2.1 2003/04/17 22:28:45 bates Exp $
\name{pairs.lmList}
\title{Pairs Plot of an lmList Object}
\usage{
\method{pairs}{lmList}(x, form, label, id, idLabels, grid, \dots)
}
\alias{pairs.lmList}
\arguments{
 \item{x}{an object inheriting from class \code{lmList}, representing
   a list of \code{lm} objects with a common model.
 }
 \item{form}{an optional one-sided formula specifying the desired type of
   plot. Any variable present in the original data frame used to obtain
   \code{x} can be referenced. In addition, \code{x} itself
   can be referenced in the formula using the symbol
   \code{"."}. Conditional expressions on the right of a \code{|}
   operator can be used to define separate panels in a Trellis
   display. The expression on the right hand side of \code{form}, and to
   the left of the \code{|} operator, must evaluate to a data frame with
   at least two columns. Default is \code{~ coef(.) }, corresponding to
   a pairs plot of the coefficients of \code{x}.
 }    
 \item{label}{an optional character vector of labels for the variables
   in the pairs plot.}
 \item{id}{an optional numeric value, or one-sided formula. If given as
   a value, it is used as a significance level for an outlier
   test based on the Mahalanobis distances of the estimated random
   effects. Groups with random effects distances greater than the
   \eqn{1-value} percentile of the appropriate chi-square distribution
   are identified in the plot using \code{idLabels}. If given as a
   one-sided formula, its right hand side must evaluate to a  logical,
   integer, or character vector which is used to identify points in the
   plot. If missing, no points are identified.
 }
 \item{idLabels}{an optional vector, or one-sided formula. If given as a
   vector, it is converted to character and used to label the
   points identified according to \code{id}. If given as a
   one-sided formula, its right hand side must evaluate to a vector
   which is converted to character and used to label the identified
   points. Default is the innermost grouping factor. 
 }
 \item{grid}{an optional logical value indicating whether a grid should
   be added to plot. Default is \code{FALSE}.}
 \item{\dots}{optional arguments passed to the Trellis plot function.
 }
}
\description{
  Diagnostic plots for the linear model fits corresponding to the
  \code{x}  components  are obtained. The \code{form} argument
  gives considerable  flexibility in the type of plot specification. A
  conditioning  expression (on the right side of a \code{|} operator)
  always implies  that different panels are used for  each level of the
  conditioning  factor, according to a Trellis display. The expression
  on the right  hand side of the formula, before a \code{|} operator,
  must evaluate to  a data frame with at least two columns. If the data
  frame has two  columns, a scatter plot of the two variables is
  displayed (the Trellis  function \code{xyplot} is used). Otherwise, if
  more than two columns  are present, a scatter plot matrix with
  pairwise scatter plots of the  columns in the data frame is displayed
  (the Trellis function  \code{splom} is used).   
}
\value{
  a diagnostic Trellis plot.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{lmList}}, \code{\link[lattice]{xyplot}},
  \code{\link[lattice]{splom}}}
\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age | Subject, Orthodont)
# scatter plot of coefficients by gender, identifying unusual subjects
pairs(fm1, ~coef(.) | Sex, id = 0.1, adj = -0.5)    
# scatter plot of estimated random effects
\dontrun{
pairs(fm1, ~ranef(.))
}
}
\keyword{models}

\eof
% $Id: pairs.lme.Rd,v 1.10 2002/04/16 14:28:53 bates Exp $
\name{pairs.lme}
\title{Pairs Plot of an lme Object}
\usage{
\method{pairs}{lme}(x, form, label, id, idLabels, grid, \dots)
}
\alias{pairs.lme}
\arguments{
 \item{x}{an object inheriting from class \code{lme}, representing
   a fitted linear mixed-effects model.}
 \item{form}{an optional one-sided formula specifying the desired type of
   plot. Any variable present in the original data frame used to obtain
   \code{x} can be referenced. In addition, \code{x} itself
   can be referenced in the formula using the symbol
   \code{"."}. Conditional expressions on the right of a \code{|}
   operator can be used to define separate panels in a Trellis
   display. The expression on the right hand side of \code{form}, and to
   the left of the \code{|} operator, must evaluate to a data frame with
   at least two columns. Default is \code{~ coef(.) }, corresponding to
   a pairs plot of the coefficients evaluated at the innermost level of
   nesting.}    
 \item{label}{an optional character vector of labels for the variables
   in the pairs plot.}
 \item{id}{an optional numeric value, or one-sided formula. If given as
   a value, it is used as a significance level for an outlier
   test based on the Mahalanobis distances of the estimated random
   effects. Groups with random effects distances greater than the
   \eqn{1-value} percentile of the appropriate chi-square distribution
   are identified in the plot using \code{idLabels}. If given as a
   one-sided formula, its right hand side must evaluate to a  logical,
   integer, or character vector which is used to identify points in the
   plot. If missing, no points are identified.}
 \item{idLabels}{an optional vector, or one-sided formula. If given as a
   vector, it is converted to character and used to label the
   points identified according to \code{id}. If given as a
   one-sided formula, its right hand side must evaluate to a vector
   which is converted to character and used to label the identified
   points. Default is the innermost grouping factor. 
 }
 \item{grid}{an optional logical value indicating whether a grid should
   be added to plot. Default is \code{FALSE}.}
 \item{\dots}{optional arguments passed to the Trellis plot function.}
}
\description{
  Diagnostic plots for the linear mixed-effects fit are obtained. The
  \code{form} argument gives considerable flexibility in the type of
  plot specification. A conditioning expression (on the right side of a
  \code{|} operator) always implies that different panels are used for
  each level of the conditioning factor, according to a Trellis
  display. The expression on the right hand side of the formula, before
  a \code{|} operator, must evaluate to a data frame with at least two
  columns. If the data frame has two columns, a scatter plot of the two
  variables is displayed (the Trellis function \code{xyplot} is
  used). Otherwise, if more than two columns are present, a scatter plot
  matrix with pairwise scatter plots of the columns in the data frame is
  displayed (the Trellis function \code{splom} is used). 
}
\value{
  a diagnostic Trellis plot.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{lme}}, \code{\link[lattice]{xyplot}},
  \code{\link[lattice]{splom}}}
\examples{
data(Orthodont)
fm1 <- lme(distance ~ age, Orthodont, random = ~ age | Subject)
# scatter plot of coefficients by gender, identifying unusual subjects
pairs(fm1, ~coef(., augFrame = TRUE) | Sex, id = 0.1, adj = -0.5)    
# scatter plot of estimated random effects
\dontrun{
pairs(fm1, ~ranef(.))
}
}
\keyword{models}

\eof
% $Id: pdBlocked.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{pdBlocked}
\title{Positive-Definite Block Diagonal Matrix}
\usage{
pdBlocked(value, form, nam, data, pdClass)
}
\alias{pdBlocked}
\arguments{
 \item{value}{an optional list with elements to be used as the
   \code{value} argument to other \code{pdMat} constructors. These
   include: \code{pdMat} objects, positive-definite
   matrices, one-sided linear formulas, vectors of character strings, or
   numeric vectors. All elements in the list must be similar (e.g. all
   one-sided formulas, or all numeric vectors). Defaults to
   \code{numeric(0)}, corresponding to an uninitialized object.}
 \item{form}{an optional list of one-sided linear formulas specifying the
   row/column names for the block-diagonal elements of the matrix
   represented by \code{object}. Because factors may be present in
   \code{form}, the formulas needs to be evaluated on a data.frame to
   resolve the names they define. This argument is ignored when
   \code{value} is a list of one-sided formulas. Defaults to \code{NULL}.}
 \item{nam}{an optional list of vector of character strings specifying the
   row/column names for the block-diagonal elements of the matrix
   represented by object. Each of its components must have  
   length equal to the dimension of the corresponding block-diagonal
   element and unreplicated elements. This argument is ignored when 
   \code{value} is a list of vector of character strings. Defaults to 
   \code{NULL}.}
 \item{data}{an optional data frame in which to evaluate the variables
   named in \code{value} and \code{form}. It is used to
   obtain the levels for \code{factors}, which affect the
   dimensions and the row/column names of the underlying matrix. If
   \code{NULL}, no attempt is made to obtain information on any
   \code{factors} appearing in the formulas. Defaults to the parent
   frame from which the function was called.}
 \item{pdClass}{an optional vector of character strings naming the
   \code{pdMat} classes to be assigned to the individual blocks in the
   underlying matrix. If a single class is specified, it is used for all
   block-diagonal elements. This argument will only be used when
   \code{value} is missing, or its elements are not \code{pdMat}
   objects. Defaults to \code{"pdSymm"}.} 
}
\description{
  This function is a constructor for the \code{pdBlocked} class,
  representing a positive-definite block-diagonal matrix. Each
  block-diagonal element of the underlying  matrix is itself a
  positive-definite matrix and is represented internally as an
  individual \code{pdMat} object. When \code{value} is
  \code{numeric(0)}, a list of uninitialized \code{pdMat} objects, a
  list of one-sided formulas, or a list of vectors 
  of character strings,  \code{object} is returned
  as an uninitialized \code{pdBlocked} object (with just some of its
  attributes and its class defined) and needs to have its coefficients
  assigned later, generally using the \code{coef} or \code{matrix} replacement
  functions. If \code{value} is a list of  initialized \code{pdMat}
  objects, \code{object} will be constructed from the list obtained by
  applying \code{as.matrix} to each of the \code{pdMat} elements of
  \code{value}. Finally, if \code{value} is a list of numeric vectors,
  they are assumed to represent the unrestricted coefficients
  of the block-diagonal elements of the  underlying positive-definite
  matrix.  
}
\value{
  a \code{pdBlocked} object representing a positive-definite
  block-diagonal matrix, also inheriting from class \code{pdMat}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}


\seealso{\code{\link{as.matrix.pdMat}}, \code{\link{coef.pdMat}},
  \code{\link{matrix<-.pdMat}}} 

\examples{
pd1 <- pdBlocked(list(diag(1:2), diag(c(0.1, 0.2, 0.3))),
                 nam = list(c("A","B"), c("a1", "a2", "a3")))
pd1
}
\keyword{models}

\eof
% $Id: pdClasses.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{pdClasses}
\title{Positive-Definite Matrix Classes}
\alias{pdClasses}
\description{
  Standard classes of positive-definite matrices (\code{pdMat})
  structures  available in the \code{nlme} library. 
}
\value{
  Available standard classes:
  \item{pdSymm}{general positive-definite matrix, with no additional
    structure}
  \item{pdLogChol}{general positive-definite matrix, with no additional
    structure, using a log-Cholesky parameterization}
  \item{pdDiag}{diagonal}
  \item{pdIdent}{multiple of an identity}
  \item{pdCompSymm}{compound symmetry structure (constant diagonal and
    constant off-diagonal elements)}
  \item{pdBlocked}{block-diagonal matrix, with diagonal blocks of any
    "atomic" \code{pdMat} class}
  \item{pdNatural}{general positive-definite matrix in natural
    parametrization (i.e. parametrized in terms of standard deviations
    and correlations). The underlying coefficients are not unrestricted,
    so this class should NOT be used for optimization.}
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  Users may define their own \code{pdMat} classes by specifying a
  \code{constructor} function and, at a minimum, methods for the
  functions \code{pdConstruct}, \code{pdMatrix} and \code{coef}. For
  examples of these functions, see the methods for classes \code{pdSymm}
  and \code{pdDiag}. 
}
\seealso{\code{\link{pdCompSymm}}, \code{\link{pdDiag}},
  \code{\link{pdIdent}}, \code{\link{pdNatural}}, \code{\link{pdSymm}},
  \code{\link{pdLogChol}}
}
\keyword{models}

\eof
% $Id: pdCompSymm.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{pdCompSymm}
\title{Positive-Definite Matrix with Compound Symmetry Structure}
\usage{
pdCompSymm(value, form, nam, data)
}
\alias{pdCompSymm}
\arguments{
 \item{value}{an optional initialization value, which can be any of the
   following: a \code{pdMat} object, a positive-definite
   matrix, a one-sided linear formula (with variables separated by
   \code{+}), a vector of character strings, or a numeric
   vector of length 2. Defaults to \code{numeric(0)}, corresponding to
   an uninitialized object.} 
 \item{form}{an optional one-sided linear formula specifying the
   row/column names for the matrix represented by \code{object}. Because
   factors may be present in \code{form}, the formula needs to be
   evaluated on a data.frame to resolve the names it defines. This
   argument is ignored when \code{value} is a one-sided
   formula. Defaults to \code{NULL}.}
 \item{nam}{an optional vector of character strings specifying the
   row/column names for the matrix represented by object. It must have 
   length equal to the dimension of the underlying positive-definite
   matrix and unreplicated elements. This argument is ignored when
   \code{value} is a vector of character strings. Defaults to
   \code{NULL}.}
 \item{data}{an optional data frame in which to evaluate the variables
   named in \code{value} and \code{form}. It is used to
   obtain the levels for \code{factors}, which affect the
   dimensions and the row/column names of the underlying matrix. If
   \code{NULL}, no attempt is made to obtain information on 
   \code{factors} appearing in the formulas. Defaults to the
   parent frame from which the function was called.}
}
\description{
  This function is a constructor for the \code{pdCompSymm} class,
  representing a positive-definite matrix with compound symmetry
  structure (constant diagonal and constant off-diagonal elements). The
  underlying matrix is represented by 2 unrestricted parameters.
  When \code{value} is \code{numeric(0)}, an unitialized \code{pdMat}
  object, a one-sided formula, or a vector of character strings,
  \code{object} is returned  as an uninitialized \code{pdCompSymm}
  object (with just some of its  attributes and its class defined) and
  needs to have its coefficients assigned later, generally using the
  \code{coef} or \code{matrix} replacement functions. If \code{value} is
  an initialized
  \code{pdMat} object, \code{object} will be constructed from 
  \code{as.matrix(value)}. Finally, if \code{value} is a numeric vector
  of length 2, it is assumed to represent the unrestricted coefficients
  of the underlying positive-definite matrix.  
}
\value{
  a \code{pdCompSymm} object representing a positive-definite
  matrix with compound symmetry structure, also inheriting from class
  \code{pdMat}.  
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{as.matrix.pdMat}}, \code{\link{coef.pdMat}},
  \code{\link{matrix<-.pdMat}}} 

\examples{
pd1 <- pdCompSymm(diag(3) + 1, nam = c("A","B","C"))
pd1
}
\keyword{models}

\eof
% $Id: pdConstruct.Rd,v 1.7.2.1 2003/04/17 22:28:45 bates Exp $
\name{pdConstruct}
\title{Construct pdMat Objects}
\usage{
pdConstruct(object, value, form, nam, data, \dots)
}
\alias{pdConstruct}
\alias{pdConstruct.pdCompSymm}
\alias{pdConstruct.pdDiag}
\alias{pdConstruct.pdIdent}
\alias{pdConstruct.pdMat}
\alias{pdConstruct.pdNatural}
\alias{pdConstruct.pdSymm}
\alias{pdConstruct.pdLogChol}
\arguments{
 \item{object}{an object inheriting from class \code{pdMat}, representing
   a positive definite matrix.} 
 \item{value}{an optional initialization value, which can be any of the
   following: a \code{pdMat} object, a positive-definite
   matrix, a one-sided linear formula (with variables separated by
   \code{+}), a vector of character strings, or a numeric
   vector. Defaults to \code{numeric(0)}, corresponding to an
   uninitialized object.} 
 \item{form}{an optional one-sided linear formula specifying the
   row/column names for the matrix represented by \code{object}. Because
   factors may be present in \code{form}, the formula needs to be
   evaluated on a data.frame to resolve the names it defines. This
   argument is ignored when \code{value} is a one-sided
   formula. Defaults to \code{NULL}.}
 \item{nam}{an optional vector of character strings specifying the
   row/column names for the matrix represented by object. It must have 
   length equal to the dimension of the underlying positive-definite
   matrix and unreplicated elements. This argument is ignored when
   \code{value} is a vector of character strings. Defaults to
   \code{NULL}.}
 \item{data}{an optional data frame in which to evaluate the variables
   named in \code{value} and \code{form}. It is used to
   obtain the levels for \code{factors}, which affect the
   dimensions and the row/column names of the underlying matrix. If
   \code{NULL}, no attempt is made to obtain information on 
   \code{factors} appearing in the formulas. Defaults to the
   parent frame from which the function was called.}
 \item{\dots}{optional arguments for some methods.}
}
\description{
  This function is an alternative constructor for the \code{pdMat}
  class associated with \code{object} and is mostly used internally in other
  functions. See the documentation on the principal constructor
  function, generally with the same name as the \code{pdMat} class of
  object. 
}
\value{
  a \code{pdMat} object representing a positive-definite matrix,
  inheriting from the same classes as \code{object}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{pdCompSymm}}, \code{\link{pdDiag}},
  \code{\link{pdIdent}}, \code{\link{pdNatural}}, \code{\link{pdSymm}}}

\examples{
pd1 <- pdSymm()
pdConstruct(pd1, diag(1:4))
}
\keyword{models}

\eof
% $Id: pdConstruct.pdBlocked.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{pdConstruct.pdBlocked}
\title{Construct pdBlocked Objects}
\usage{
\method{pdConstruct}{pdBlocked}(object, value, form, nam, data, pdClass,
\dots)
}
\alias{pdConstruct.pdBlocked}
\arguments{
 \item{object}{an object inheriting from class \code{pdBlocked},
   representing a positive definite block-diagonal matrix.} 
 \item{value}{an optional list with elements to be used as the
   \code{value} argument to other \code{pdMat} constructors. These
   include: \code{pdMat} objects, positive-definite
   matrices, one-sided linear formulas, vectors of character strings, or
   numeric vectors. All elements in the list must be similar (e.g. all
   one-sided formulas, or all numeric vectors). Defaults to
   \code{numeric(0)}, corresponding to an uninitialized object.}
 \item{form}{an optional list of one-sided linear formula specifying the
   row/column names for the block-diagonal elements of the matrix
   represented by \code{object}. Because factors may be present in
   \code{form}, the formulas needs to be evaluated on a data.frame to
   resolve the names they defines. This argument is ignored when
   \code{value} is a list of one-sided formulas. Defaults to \code{NULL}.}
 \item{nam}{an optional list of vector of character strings specifying the
   row/column names for the block-diagonal elements of the matrix
   represented by object. Each of its components must have  
   length equal to the dimension of the corresponding block-diagonal
   element and unreplicated elements. This argument is ignored when 
   \code{value} is a list of vector of character strings. Defaults to 
   \code{NULL}.}
 \item{data}{an optional data frame in which to evaluate the variables
   named in \code{value} and \code{form}. It is used to
   obtain the levels for \code{factors}, which affect the
   dimensions and the row/column names of the underlying matrix. If
   \code{NULL}, no attempt is made to obtain information on 
   \code{factors} appearing in the formulas. Defaults to the
   parent frame from which the function was called.}
 \item{pdClass}{an optional vector of character strings naming the
   \code{pdMat} classes to be assigned to the individual blocks in the
   underlying matrix. If a single class is specified, it is used for all
   block-diagonal elements. This argument will only be used when
   \code{value} is missing, or its elements are not \code{pdMat}
   objects. Defaults to \code{"pdSymm"}.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This function give an alternative constructor for the \code{pdBlocked}
  class, representing a positive-definite block-diagonal matrix. Each
  block-diagonal element of the underlying  matrix is itself a
  positive-definite matrix and is represented internally as an
  individual \code{pdMat} object. When \code{value} is
  \code{numeric(0)}, a list of uninitialized \code{pdMat} objects, a
  list of one-sided formulas, or a list of vectors of character strings,
  \code{object} is returned as an uninitialized \code{pdBlocked} object
  (with just some of its attributes and its class defined) and needs to
  have its coefficients assigned later, generally using the \code{coef}
  or \code{matrix} replacement functions. If \code{value} is a list of  initialized
  \code{pdMat} objects, \code{object} will be constructed from the list
  obtained by applying \code{as.matrix} to each of the \code{pdMat}
  elements of \code{value}. Finally, if \code{value} is a list of
  numeric vectors, they are assumed to represent the unrestricted
  coefficients of the block-diagonal elements of the  underlying
  positive-definite matrix.  
}
\value{
  a \code{pdBlocked} object representing a positive-definite
  block-diagonal matrix, also inheriting from class \code{pdMat}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}


\seealso{\code{\link{as.matrix.pdMat}}, \code{\link{coef.pdMat}},
  \code{\link{matrix<-.pdMat}}} 

\examples{
pd1 <- pdBlocked(list(c("A","B"), c("a1", "a2", "a3")))
pdConstruct(pd1, list(diag(1:2), diag(c(0.1, 0.2, 0.3))))
}
\keyword{models}

\eof
% $Id: pdDiag.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{pdDiag}
\title{Diagonal Positive-Definite Matrix}
\usage{
pdDiag(value, form, nam, data)
}
\alias{pdDiag}
\arguments{
 \item{value}{an optional initialization value, which can be any of the
   following: a \code{pdMat} object, a positive-definite
   matrix, a one-sided linear formula (with variables separated by
   \code{+}), a vector of character strings, or a numeric
   vector of length equal to the dimension of the underlying
   positive-definite matrix. Defaults to \code{numeric(0)}, corresponding
   to an uninitialized object.} 
 \item{form}{an optional one-sided linear formula specifying the
   row/column names for the matrix represented by \code{object}. Because
   factors may be present in \code{form}, the formula needs to be
   evaluated on a data.frame to resolve the names it defines. This
   argument is ignored when \code{value} is a one-sided
   formula. Defaults to \code{NULL}.}
 \item{nam}{an optional vector of character strings specifying the
   row/column names for the matrix represented by object. It must have 
   length equal to the dimension of the underlying positive-definite
   matrix and unreplicated elements. This argument is ignored when
   \code{value} is a vector of character strings. Defaults to
   \code{NULL}.}
 \item{data}{an optional data frame in which to evaluate the variables
   named in \code{value} and \code{form}. It is used to
   obtain the levels for \code{factors}, which affect the
   dimensions and the row/column names of the underlying matrix. If
   \code{NULL}, no attempt is made to obtain information on 
   \code{factors} appearing in the formulas. Defaults to the
   parent frame from which the function was called.}
}

\description{
  This function is a constructor for the \code{pdDiag} class,
  representing a diagonal positive-definite matrix. If the matrix
  associated with \code{object} is of dimension \eqn{n}, it is
  represented by \eqn{n} unrestricted parameters, given by the logarithm
  of the square-root of the diagonal values. When \code{value} is
  \code{numeric(0)}, an uninitialized \code{pdMat} object, a one-sided
  formula, or a vector of character strings, \code{object} is returned
  as an uninitialized \code{pdDiag} object (with just some of its
  attributes and its class defined) and needs to have its coefficients
  assigned later, generally using the \code{coef} or \code{matrix} replacement
  functions. If \code{value} is an initialized \code{pdMat} object,
  \code{object} will be constructed from
  \code{as.matrix(value)}. Finally, if \code{value} is a numeric vector,
  it is assumed to represent the unrestricted coefficients of the
  underlying positive-definite
  matrix.  
}
\value{
  a \code{pdDiag} object representing a diagonal positive-definite
  matrix, also inheriting from class \code{pdMat}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{as.matrix.pdMat}}, \code{\link{coef.pdMat}},
  \code{\link{matrix<-.pdMat}}} 

\examples{
pd1 <- pdDiag(diag(1:3), nam = c("A","B","C"))
pd1
}
\keyword{models}

\eof
% $Id: pdFactor.Rd,v 1.8 2002/03/05 14:59:39 bates Exp $
\name{pdFactor}
\title{Square-Root Factor of a Positive-Definite Matrix}
\usage{
pdFactor(object)
}
\alias{pdFactor}
\alias{pdFactor.pdBlocked}
\alias{pdFactor.pdCompSymm}
\alias{pdFactor.pdDiag}
\alias{pdFactor.pdIdent}
\alias{pdFactor.pdMat}
\alias{pdFactor.pdNatural}
\alias{pdFactor.pdSymm}
\alias{pdFactor.pdLogChol}
\arguments{
 \item{object}{an object inheriting from class \code{pdMat}, representing
   a positive definite matrix, which must have been initialized
   (i.e. \code{length(coef(object)) > 0}).} 
}
\description{
  A square-root factor of the positive-definite matrix represented by
  \code{object} is obtained. Letting \eqn{\Sigma}{S} denote a
  positive-definite matrix, a square-root factor of \eqn{\Sigma}{S} is
  any square matrix \eqn{L}{L} such that \eqn{\Sigma = L'L}{S =
    L'L}. This function extracts \eqn{L}.
}
\value{
  a vector with a square-root factor of the positive-definite matrix
  associated with \code{object} stacked column-wise.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  This function is used intensively in optimization
  algorithms and its value is returned as a vector for efficiency
  reasons. The \code{pdMatrix} function can be used to obtain
  square-root factors in matrix form.
}
\seealso{\code{\link{pdMatrix}}}
\examples{
pd1 <- pdCompSymm(4 * diag(3) + 1)
pdFactor(pd1)
}
\keyword{models}

\eof
% $Id: pdFactor.reStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{pdFactor.reStruct}
\title{Extract Square-Root Factor from Components of an reStruct Object}
\usage{
\method{pdFactor}{reStruct}(object)
}
\alias{pdFactor.reStruct}
\arguments{
 \item{object}{an object inheriting from class \code{reStruct},
   representing a random effects structure and consisting of a list of
   \code{pdMat} objects.} 
}
\description{
  This method function extracts square-root factors of the
  positive-definite matrices corresponding to the \code{pdMat} elements
  of \code{object}.  
}
\value{
  a vector with square-root factors of the positive-definite matrices
  corresponding to the elements of \code{object} stacked column-wise.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  This function is used intensively in optimization
  algorithms and its value is returned as a vector for efficiency
  reasons. The \code{pdMatrix} function can be used to obtain
  square-root factors in matrix form.
}
\seealso{\code{\link{pdMatrix.reStruct}}, \code{\link{pdFactor.pdMat}}}
\examples{
data(Orthodont)
rs1 <- reStruct(pdSymm(diag(3), ~age+Sex, data = Orthodont))
pdFactor(rs1)
}
\keyword{models}

\eof
% $Id: pdIdent.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{pdIdent}
\title{Multiple of the Identity Positive-Definite Matrix}
\usage{
pdIdent(value, form, nam, data)
}
\alias{pdIdent}
\arguments{
 \item{value}{an optional initialization value, which can be any of the
   following: a \code{pdMat} object, a positive-definite
   matrix, a one-sided linear formula (with variables separated by
   \code{+}), a vector of character strings, or a numeric
   value. Defaults to \code{numeric(0)}, corresponding
   to an uninitialized object.} 
 \item{form}{an optional one-sided linear formula specifying the
   row/column names for the matrix represented by \code{object}. Because
   factors may be present in \code{form}, the formula needs to be
   evaluated on a data.frame to resolve the names it defines. This
   argument is ignored when \code{value} is a one-sided
   formula. Defaults to \code{NULL}.}
 \item{nam}{an optional vector of character strings specifying the
   row/column names for the matrix represented by object. It must have 
   length equal to the dimension of the underlying positive-definite
   matrix and unreplicated elements. This argument is ignored when
   \code{value} is a vector of character strings. Defaults to
   \code{NULL}.}
 \item{data}{an optional data frame in which to evaluate the variables
   named in \code{value} and \code{form}. It is used to
   obtain the levels for \code{factors}, which affect the
   dimensions and the row/column names of the underlying matrix. If
   \code{NULL}, no attempt is made to obtain information on 
   \code{factors} appearing in the formulas. Defaults to the
   parent frame from which the function was called.}
}

\description{
  This function is a constructor for the \code{pdIdent} class,
  representing a multiple of the identity positive-definite matrix. 
  The matrix associated with \code{object} is represented by 1
  unrestricted parameter, given by the logarithm of the square-root of
  the diagonal value. When \code{value} is 
  \code{numeric(0)}, an uninitialized \code{pdMat} object, a one-sided
  formula, or a vector of character strings, \code{object} is returned
  as an uninitialized \code{pdIdent} object (with just some of its
  attributes and its class defined) and needs to have its coefficients
  assigned later, generally using the \code{coef} or \code{matrix} replacement
  functions. If \code{value} is an initialized \code{pdMat} object,
  \code{object} will be constructed from
  \code{as.matrix(value)}. Finally, if \code{value} is a numeric value,
  it is assumed to represent the unrestricted coefficient of the
  underlying positive-definite  matrix.  
}
\value{
  a \code{pdIdent} object representing a multiple of the identity
  positive-definite matrix, also inheriting from class \code{pdMat}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{as.matrix.pdMat}}, \code{\link{coef.pdMat}},
  \code{\link{matrix<-.pdMat}}} 

\examples{
pd1 <- pdIdent(4 * diag(3), nam = c("A","B","C"))
pd1
}
\keyword{models}

\eof
% $Id: pdLogChol.Rd,v 1.2 2002/03/05 14:59:39 bates Exp $
\name{pdLogChol}
\title{General Positive-Definite Matrix}
\usage{
pdLogChol(value, form, nam, data)
}
\alias{pdLogChol}
\arguments{
 \item{value}{an optional initialization value, which can be any of the
   following: a \code{pdMat} object, a positive-definite
   matrix, a one-sided linear formula (with variables separated by
   \code{+}), a vector of character strings, or a numeric
   vector. Defaults to \code{numeric(0)}, corresponding to an
   uninitialized object.} 
 \item{form}{an optional one-sided linear formula specifying the
   row/column names for the matrix represented by \code{object}. Because
   factors may be present in \code{form}, the formula needs to be
   evaluated on a data.frame to resolve the names it defines. This
   argument is ignored when \code{value} is a one-sided
   formula. Defaults to \code{NULL}.}
 \item{nam}{an optional vector of character strings specifying the
   row/column names for the matrix represented by object. It must have 
   length equal to the dimension of the underlying positive-definite
   matrix and unreplicated elements. This argument is ignored when
   \code{value} is a vector of character strings. Defaults to
   \code{NULL}.}
 \item{data}{an optional data frame in which to evaluate the variables
   named in \code{value} and \code{form}. It is used to
   obtain the levels for \code{factors}, which affect the
   dimensions and the row/column names of the underlying matrix. If
   \code{NULL}, no attempt is made to obtain information on 
   \code{factors} appearing in the formulas. Defaults to the
   parent frame from which the function was called.}
}

\description{
  This function is a constructor for the \code{pdLogChol} class,
  representing a general positive-definite matrix. If the matrix
  associated with \code{object} is of dimension \eqn{n}, it is
  represented by \eqn{n(n+1)/2}{n*(n+1)/2} unrestricted parameters,
  using the log-Cholesky parametrization described in Pinheiro and
  Bates (1996). When \code{value} is \code{numeric(0)}, an uninitialized
  \code{pdMat} object, a one-sided
  formula, or a vector of character strings, \code{object} is returned
  as an uninitialized \code{pdLogChol} object (with just some of its
  attributes and its class defined) and needs to have its coefficients
  assigned later, generally using the \code{coef} or \code{matrix} replacement
  functions. If \code{value} is an initialized \code{pdMat} object,
  \code{object} will be constructed from
  \code{as.matrix(value)}. Finally, if \code{value} is a numeric vector,
  it is assumed to represent the unrestricted coefficients of the
  matrix-logarithm parametrization of the underlying positive-definite
  matrix.  
}
\value{
  a \code{pdLogChol} object representing a general positive-definite
  matrix, also inheriting from class \code{pdMat}. 
}
\references{
  Pinheiro, J.C. and Bates., D.M.  (1996) "Unconstrained
  Parametrizations for Variance-Covariance Matrices", Statistics and
  Computing, 6, 289-296.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{as.matrix.pdMat}}, \code{\link{coef.pdMat}},
  \code{\link{matrix<-.pdMat}}} 
\examples{
pd1 <- pdLogChol(diag(1:3), nam = c("A","B","C"))
pd1
}
\keyword{models}

\eof
% $Id: pdMat.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{pdMat}
\title{Positive-Definite Matrix}
\usage{
pdMat(value, form, nam, data, pdClass)
}
\alias{pdMat}
\alias{plot.pdMat}
\arguments{
 \item{value}{an optional initialization value, which can be any of the
   following: a \code{pdMat} object, a positive-definite
   matrix, a one-sided linear formula (with variables separated by
   \code{+}), a vector of character strings, or a numeric
   vector. Defaults to \code{numeric(0)}, corresponding to an
   uninitialized object.} 
 \item{form}{an optional one-sided linear formula specifying the
   row/column names for the matrix represented by \code{object}. Because
   factors may be present in \code{form}, the formula needs to be
   evaluated on a data.frame to resolve the names it defines. This
   argument is ignored when \code{value} is a one-sided
   formula. Defaults to \code{NULL}.}
 \item{nam}{an optional vector of character strings specifying the
   row/column names for the matrix represented by object. It must have 
   length equal to the dimension of the underlying positive-definite
   matrix and unreplicated elements. This argument is ignored when
   \code{value} is a vector of character strings. Defaults to
   \code{NULL}.}
 \item{data}{an optional data frame in which to evaluate the variables
   named in \code{value} and \code{form}. It is used to
   obtain the levels for \code{factors}, which affect the
   dimensions and the row/column names of the underlying matrix. If
   \code{NULL}, no attempt is made to obtain information on 
   \code{factors} appearing in the formulas. Defaults to the
   parent frame from which the function was called.}
 \item{pdClass}{an optional character string naming the
   \code{pdMat} class to be assigned to the returned object. This
   argument will only be used when \code{value} is not a \code{pdMat} 
   object. Defaults to \code{"pdSymm"}.} 
}

\description{
  This function gives an alternative way of constructing an object
  inheriting from the \code{pdMat} class named in \code{pdClass}, or
  from \code{data.class(object)} if \code{object} inherits from
  \code{pdMat},  and is mostly used internally in other functions. See
  the documentation on the principal constructor function, generally
  with the same name as the \code{pdMat} class of object. 
}
\value{
  a \code{pdMat} object representing a positive-definite matrix,
  inheriting from the class named in \code{pdClass}, or from
  \code{class(object)}, if \code{object} inherits from \code{pdMat}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{pdCompSymm}}, \code{\link{pdDiag}},
  \code{\link{pdIdent}}, \code{\link{pdNatural}}, \code{\link{pdSymm}}}

\examples{
pd1 <- pdMat(diag(1:4), pdClass = "pdDiag")
pd1
}
\keyword{models}

\eof
% $Id: pdMatrix.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{pdMatrix}
\title{Extract Matrix or Square-Root Factor from a pdMat Object}
\usage{
pdMatrix(object, factor)
}
\alias{pdMatrix}
\alias{pdMatrix.pdBlocked}
\alias{pdMatrix.pdCompSymm}
\alias{pdMatrix.pdDiag}
\alias{pdMatrix.pdIdent}
\alias{pdMatrix.pdMat}
\alias{pdMatrix.pdSymm}
\alias{pdMatrix.pdNatural}
\arguments{
 \item{object}{an object inheriting from class \code{pdMat}, representing
   a positive definite matrix.}
 \item{factor}{an optional logical value. If \code{TRUE}, a square-root
   factor of the positive-definite matrix represented by \code{object}
   is returned; else, if \code{FALSE}, the positive-definite matrix is
   returned. Defaults to \code{FALSE}.}
}
\description{
  The positive-definite matrix represented by \code{object}, or a
  square-root factor of it is obtained. Letting \eqn{\Sigma}{S} denote a
  positive-definite matrix, a square-root factor of \eqn{\Sigma}{S} is
  any square matrix \eqn{L}{L} such that \eqn{\Sigma = L'L}{S =
    L'L}. This function extracts \eqn{S} or \eqn{L}.
}
\value{
  if \code{fact} is \code{FALSE} the positive-definite matrix
  represented by \code{object} is returned; else a square-root of the
  positive-definite matrix is returned.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{as.matrix.pdMat}}, \code{\link{pdFactor}},
  \code{\link{corMatrix}}}

\examples{
pd1 <- pdSymm(diag(1:4))
pdMatrix(pd1)
}
\keyword{models}

\eof
% $Id: pdMatrix.reStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{pdMatrix.reStruct}
\title{Extract Matrix or Square-Root Factor from Components of an
  reStruct Object} 
\usage{
\method{pdMatrix}{reStruct}(object, factor)
}
\alias{pdMatrix.reStruct}

\arguments{
 \item{object}{an object inheriting from class \code{reStruct},
   representing a random effects structure and consisting of a list of
   \code{pdMat} objects.} 
 \item{factor}{an optional logical value. If \code{TRUE}, square-root
   factors of the positive-definite matrices represented by the elements
   of \code{object} are returned; else, if \code{FALSE}, the
   positive-definite matrices are returned. Defaults to \code{FALSE}.} 
}
\description{
  This method function extracts the positive-definite  matrices
  corresponding to the \code{pdMat} elements of \code{object}, or
  square-root factors of the positive-definite matrices. 
}
\value{
  a list with components given by the positive-definite matrices
  corresponding to the elements of \code{object}, or square-root factors
  of the positive-definite matrices.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{as.matrix.reStruct}}, \code{\link{reStruct}},
  \code{\link{pdMatrix.pdMat}}}

\examples{
data(Orthodont)
rs1 <- reStruct(pdSymm(diag(3), ~age+Sex, data = Orthodont))
pdMatrix(rs1)
}
\keyword{models}

\eof
% $Id: pdNatural.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{pdNatural}
\title{General Positive-Definite Matrix in Natural Parametrization}
\usage{
pdNatural(value, form, nam, data)
}
\alias{pdNatural}
\arguments{
 \item{value}{an optional initialization value, which can be any of the
   following: a \code{pdMat} object, a positive-definite
   matrix, a one-sided linear formula (with variables separated by
   \code{+}), a vector of character strings, or a numeric
   vector. Defaults to \code{numeric(0)}, corresponding to an
   uninitialized object.} 
 \item{form}{an optional one-sided linear formula specifying the
   row/column names for the matrix represented by \code{object}. Because
   factors may be present in \code{form}, the formula needs to be
   evaluated on a data.frame to resolve the names it defines. This
   argument is ignored when \code{value} is a one-sided
   formula. Defaults to \code{NULL}.}
 \item{nam}{an optional vector of character strings specifying the
   row/column names for the matrix represented by object. It must have 
   length equal to the dimension of the underlying positive-definite
   matrix and unreplicated elements. This argument is ignored when
   \code{value} is a vector of character strings. Defaults to
   \code{NULL}.}
 \item{data}{an optional data frame in which to evaluate the variables
   named in \code{value} and \code{form}. It is used to
   obtain the levels for \code{factors}, which affect the
   dimensions and the row/column names of the underlying matrix. If
   \code{NULL}, no attempt is made to obtain information on 
   \code{factors} appearing in the formulas. Defaults to the
   parent frame from which the function was called.}
}

\description{
  This function is a constructor for the \code{pdNatural} class,
  representing a general positive-definite matrix, using a natural
  parametrization . If the matrix associated with \code{object} is of
  dimension \eqn{n}, it is represented by \eqn{n(n+1)/2}{n*(n+1)/2}
  parameters. Letting \eqn{\sigma_{ij}}{S(i,j)} denote the \eqn{ij}-th
  element of the underlying positive definite matrix and
  \eqn{\rho_{ij}=\sigma_{i}/\sqrt{\sigma_{ii}\sigma_{jj}},\;i\neq j}{r(i,j) =
    S(i,j)/sqrt(S(i,i)S(j,j)), i not equal to j} denote the associated
  "correlations", the "natural" parameters are given by
  \eqn{\sqrt{\sigma_{ii}}, \;i=1,\ldots,n}{sqrt(Sii), i=1,..,n} and
  \eqn{\log((1+\rho_{ij})/(1-\rho_{ij})),\; i \neq
    j}{log((1+r(i,j))/(1-r(i,j))), i not equal to j}. Note that all
  natural parameters are individually unrestricted, but not jointly
  unrestricted (meaning that not all unrestricted vectors would give
  positive-definite matrices). Therefore, this parametrization should
  NOT be used for optimization. It is mostly used for deriving
  approximate confidence intervals on parameters following the
  optimization of an objective function. When \code{value} is
  \code{numeric(0)}, an uninitialized \code{pdMat} object, a one-sided
  formula, or a vector of character strings, \code{object} is returned
  as an uninitialized \code{pdSymm} object (with just some of its
  attributes and its class defined) and needs to have its coefficients
  assigned later, generally using the \code{coef} or \code{matrix} replacement
  functions. If \code{value} is an initialized \code{pdMat} object,
  \code{object} will be constructed from
  \code{as.matrix(value)}. Finally, if \code{value} is a numeric 
  vector, it is assumed to represent the natural parameters of the
  underlying positive-definite matrix.  
}
\value{
  a \code{pdNatural} object representing a general positive-definite
  matrix in natural parametrization, also inheriting from class
  \code{pdMat}.  
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{as.matrix.pdMat}}, \code{\link{coef.pdMat}},
  \code{\link{matrix<-.pdMat}}} 
\examples{
pdNatural(diag(1:3))
}
\keyword{models}

\eof
% $Id: pdSymm.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{pdSymm}
\title{General Positive-Definite Matrix}
\usage{
pdSymm(value, form, nam, data)
}
\alias{pdSymm}
\arguments{
 \item{value}{an optional initialization value, which can be any of the
   following: a \code{pdMat} object, a positive-definite
   matrix, a one-sided linear formula (with variables separated by
   \code{+}), a vector of character strings, or a numeric
   vector. Defaults to \code{numeric(0)}, corresponding to an
   uninitialized object.} 
 \item{form}{an optional one-sided linear formula specifying the
   row/column names for the matrix represented by \code{object}. Because
   factors may be present in \code{form}, the formula needs to be
   evaluated on a data.frame to resolve the names it defines. This
   argument is ignored when \code{value} is a one-sided
   formula. Defaults to \code{NULL}.}
 \item{nam}{an optional vector of character strings specifying the
   row/column names for the matrix represented by object. It must have 
   length equal to the dimension of the underlying positive-definite
   matrix and unreplicated elements. This argument is ignored when
   \code{value} is a vector of character strings. Defaults to
   \code{NULL}.}
 \item{data}{an optional data frame in which to evaluate the variables
   named in \code{value} and \code{form}. It is used to
   obtain the levels for \code{factors}, which affect the
   dimensions and the row/column names of the underlying matrix. If
   \code{NULL}, no attempt is made to obtain information on 
   \code{factors} appearing in the formulas. Defaults to the
   parent frame from which the function was called.}
}

\description{
  This function is a constructor for the \code{pdSymm} class,
  representing a general positive-definite matrix. If the matrix
  associated with \code{object} is of dimension \eqn{n}, it is
  represented by \eqn{n(n+1)/2}{n*(n+1)/2} unrestricted parameters,
  using the matrix-logarithm parametrization described in Pinheiro and
  Bates (1996). When \code{value} is \code{numeric(0)}, an uninitialized
  \code{pdMat} object, a one-sided
  formula, or a vector of character strings, \code{object} is returned
  as an uninitialized \code{pdSymm} object (with just some of its
  attributes and its class defined) and needs to have its coefficients
  assigned later, generally using the \code{coef} or \code{matrix} replacement
  functions. If \code{value} is an initialized \code{pdMat} object,
  \code{object} will be constructed from
  \code{as.matrix(value)}. Finally, if \code{value} is a numeric vector,
  it is assumed to represent the unrestricted coefficients of the
  matrix-logarithm parametrization of the underlying positive-definite
  matrix.  
}
\value{
  a \code{pdSymm} object representing a general positive-definite
  matrix, also inheriting from class \code{pdMat}. 
}
\references{
  Pinheiro, J.C. and Bates., D.M.  (1996) "Unconstrained
  Parametrizations for Variance-Covariance Matrices", Statistics and
  Computing, 6, 289-296.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}


\seealso{\code{\link{as.matrix.pdMat}}, \code{\link{coef.pdMat}},
  \code{\link{matrix<-.pdMat}}} 

\examples{
pd1 <- pdSymm(diag(1:3), nam = c("A","B","C"))
pd1
}
\keyword{models}

\eof
\name{phenoModel}
\alias{phenoModel}
\title{Model function for the Phenobarb data}
\description{
  A model function for a model used with the \code{Phenobarb} data.
  This function uses compiled C code to improve execution speed.
}
\usage{
phenoModel(Subject, time, dose, lCl, lV)
}
\arguments{
  \item{Subject}{an integer vector of subject identifiers.  These should
    be sorted in increasing order.}
  \item{time}{numeric. A vector of the times at which the sample was drawn or
      the drug administered (hr).
    }
  \item{dose}{numeric. A vector of doses of drug administered
      (\eqn{u}{\mu}g/kg).
    }
  \item{lCl}{numeric.  A vector of values of the natural log of the
    clearance parameter according to \code{Subject} and \code{time}.}
  \item{lV}{numeric.  A vector of values of the natural log of the
    effective volume of distribution according to \code{Subject} and
    \code{time}.}
}
\details{
  See the details section of \code{\link{Phenobarb}} for a description
  of the model function that \code{phenoModel} evaluates.
}
\value{
  a numeric vector of predicted phenobarbital concentrations.
}
\references{
    Pinheiro, J. C. and Bates, D. M. (2000) \emph{Mixed-effects Models in
	S and S-PLUS}, Springer. (section 6.4)
}
\author{Jose Pinheiro \email{jose.pinheiro@pharma.novartis.com}
  and Douglas Bates \email{bates@stat.wisc.edu}
}
\examples{
}
\keyword{models}

\eof
% $Id: plot.ACF.Rd,v 1.9 2002/04/16 14:28:53 bates Exp $
\name{plot.ACF}
\title{Plot an ACF Object}
\usage{
\method{plot}{ACF}(x, alpha, xlab, ylab, grid, \dots)
}
\alias{plot.ACF}
\arguments{
 \item{x}{an object inheriting from class \code{ACF},
   consisting of a data frame with two columns named \code{lag} and
   \code{ACF}, representing the autocorrelation values and the
   corresponding lags. 
 }
 \item{alpha}{an optional numeric value with the significance level for
   testing if the autocorrelations are zero. Lines corresponding to the
   lower and upper critical values for a test of level \code{alpha} are
   added to the plot. Default is \code{0}, in which case no lines are
   plotted.
 }
 \item{xlab,ylab}{optional character strings with the x- and y-axis
   labels. Default respectively to \code{"Lag"} and 
   \code{"Autocorrelation"}. 
 }
 \item{grid}{an optional logical value indicating whether a grid should
   be added to plot. Default is \code{FALSE}.}
 \item{\dots}{optional arguments passed to the \code{xyplot} function.} 
}
\description{
  an \code{xyplot} of the autocorrelations versus the lags, with
  \code{type = "h"}, is produced. If \code{alpha > 0}, curves
  representing the critical limits for a two-sided test of level
  \code{alpha} for the autocorrelations are added to the plot.
}
\value{
  an \code{xyplot} Trellis plot.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{ACF}}, \code{\link[lattice]{xyplot}}}
\examples{
data(Ovary)
fm1 <- lme(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), Ovary)
plot(ACF(fm1, maxLag = 10), alpha = 0.01)
}
\keyword{models}

\eof
% $Id: plot.Variogram.Rd,v 1.9.2.1 2003/04/17 22:28:45 bates Exp $
\name{plot.Variogram}
\title{Plot a Variogram Object}
\usage{
\method{plot}{Variogram}(x, smooth, showModel, sigma, span, xlab,
     ylab, type, ylim, grid, \dots)
}
\alias{plot.Variogram}
\arguments{
 \item{x}{an object inheriting from class \code{Variogram},
   consisting of a data frame with two columns named \code{variog} and
   \code{dist}, representing the semi-variogram values and the corresponding
   distances.
 }
 \item{smooth}{an optional logical value controlling whether a
   \code{loess} smoother should be added to the plot. Defaults to
   \code{TRUE}, when \code{showModel} is \code{FALSE}.
 }
 \item{showModel}{an optional logical value controlling whether the
   semi-variogram corresponding to an \code{"modelVariog"} attribute of
   \code{x}, if any is present, should be added to the
   plot. Defaults to \code{TRUE}, when the \code{"modelVariog"}
   attribute is present.
 }
 \item{sigma}{an optional numeric value used as the height of a
   horizontal line displayed in the plot. Can be used to represent the
   process standard deviation. Default is \code{NULL}, implying that no
   horizontal line is drawn. 
 }
 \item{span}{an optional numeric value with the smoothing parameter for
   the \code{loess} fit. Default is 0.6.
 }
 \item{xlab,ylab}{optional character strings with the x- and y-axis
   labels. Default respectively to \code{"Distance"} and
   \code{"SemiVariogram"}. 
 }
 \item{type}{an optional character indicating the type of plot. Defaults
     to \code{"p"}.
 }
 \item{ylim}{an optional numeric vector with the limits for the
   y-axis. Defaults to \code{c(0, max(x$variog))}.
 }
 \item{grid}{an optional logical value indicating whether a grid should
   be added to plot. Default is \code{FALSE}.}
 \item{\dots}{optional arguments passed to the Trellis \code{xyplot} function.}
}
\description{
  an \code{xyplot} of the semi-variogram versus the distances is
  produced. If \code{smooth = TRUE}, a \code{loess} smoother is added to
  the plot. If \code{showModel = TRUE} and \code{x} includes an
  \code{"modelVariog"} attribute, the corresponding semi-variogram
  is added to the plot.
}
\value{
  an \code{xyplot} Trellis plot.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{Variogram}}, \code{\link[lattice]{xyplot}},
  \code{\link[modreg]{loess}}}
\examples{
data(Ovary)
fm1 <- lme(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), Ovary)
plot(Variogram(fm1, form = ~ Time | Mare, maxDist = 0.7))
}
\keyword{models}

\eof
% $Id: plot.augPred.Rd,v 1.8.2.1 2003/04/17 22:28:45 bates Exp $
\name{plot.augPred}
\title{Plot an augPred Object}
\usage{
\method{plot}{augPred}(x, key, grid, \dots)
}
\alias{plot.augPred}
\arguments{
 \item{x}{an object of class \code{augPred}.}
 \item{key}{an optional logical value, or list. If \code{TRUE}, a legend
 is included at the top of the plot indicating which symbols (colors)
 correspond to which prediction levels. If \code{FALSE}, no legend
 is included. If given as a list, \code{key} is passed down as an
 argument to the \code{trellis} function generating the plots
 (\code{xyplot}). Defaults to \code{TRUE}.}
 \item{grid}{an optional logical value indicating whether a grid should
   be added to plot. Default is \code{FALSE}.}
 \item{\dots}{optional arguments passed down to the \code{trellis}
   function generating the plots.}
}
\description{
  A Trellis \code{xyplot} of predictions versus the primary covariate is
  generated, with a different panel for each value of the grouping
  factor. Predicted values are joined by lines, with  
  different line types (colors) being used for each level of
  grouping. Original observations are represented by circles.
}
\value{
  A Trellis plot of predictions versus the primary covariate, with panels
  determined by the grouping factor.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{augPred}}, \code{\link[lattice]{xyplot}}}

\examples{
data(Orthodont)
fm1 <- lme(Orthodont)
plot(augPred(fm1, level = 0:1, length.out = 2))
}
\keyword{models}

\eof
% $Id: plot.compareFits.Rd,v 1.8.2.1 2003/04/17 22:28:45 bates Exp $
\name{plot.compareFits}
\title{Plot a compareFits Object}
\usage{
\method{plot}{compareFits}(x, subset, key, mark, \dots)
}
\alias{plot.compareFits}
\arguments{
 \item{x}{an object of class \code{compareFits}.}
 \item{subset}{an optional logical or integer vector specifying which
   rows of \code{x} should be used in the plots. If missing, all
   rows are used.}
 \item{key}{an optional logical value, or list. If \code{TRUE}, a legend
 is included at the top of the plot indicating which symbols (colors)
 correspond to which objects being compared. If \code{FALSE}, no legend
 is included. If given as a list, \code{key} is passed down as an
 argument to the \code{trellis} function generating the plots
 (\code{dotplot}). Defaults to \code{TRUE}.}
 \item{mark}{an optional numeric vector, of length equal to the number of
   coefficients being compared, indicating where vertical lines should
   be drawn in the plots. If missing, no lines are drawn.} 
 \item{\dots}{optional arguments passed down to the \code{trellis}
   function generating the plots.}
}
\description{
  A Trellis \code{dotplot} of the values being compared, with different
  rows per group, is generated, with a different panel for each
  coefficient. Different symbols (colors) are used for each object being
  compared.
}
\value{
  A Trellis \code{dotplot} of the values being compared, with rows
  determined by the groups and panels by the coefficients.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{compareFits}}, \code{\link{pairs.compareFits}},
  \code{\link{dotplot}} }
\examples{
\dontrun{
data(Orthodont)
fm1 <- lmList(Orthodont)
fm2 <- lme(Orthodont)
plot(compareFits(coef(fm1), coef(fm2)))
}
}
\keyword{models}

\eof
% $Id: plot.gls.Rd,v 1.10 2002/04/16 14:28:53 bates Exp $
\name{plot.gls}
\title{Plot a gls Object}
\usage{
\method{plot}{gls}(x, form, abline, id, idLabels, idResType, grid, \dots)
}
\alias{plot.gls}
\arguments{
 \item{x}{an object inheriting from class \code{gls}, representing
   a generalized least squares fitted linear model.}
 \item{form}{an optional formula specifying the desired type of
   plot. Any variable present in the original data frame used to obtain
   \code{x} can be referenced. In addition, \code{x} itself
   can be referenced in the formula using the symbol
   \code{"."}. Conditional expressions on the right of a \code{|}
   operator can be used to define separate panels in a Trellis
   display. Default is \code{resid(., type = "p") ~ fitted(.) },
   corresponding to a plot of the standardized residuals versus fitted
   values, both evaluated at the innermost level of nesting.} 
 \item{abline}{an optional numeric value, or numeric vector of length
   two. If given as a single value, a horizontal line will be added to the
   plot at that coordinate; else, if given as a vector, its values are
   used as the intercept and slope for a line added to the plot. If
   missing, no lines are added to the plot.}
 \item{id}{an optional numeric value, or one-sided formula. If given as
   a value, it is used as a significance level for a two-sided outlier
   test for the standardized residuals. Observations with
   absolute standardized residuals greater than the \eqn{1 - value/2}
   quantile of the standard normal distribution are identified in the
   plot using \code{idLabels}. If given as a one-sided formula, its
   right hand side must evaluate to a  logical, integer, or character
   vector which is used to identify observations in the plot. If
   missing, no observations are identified.}
 \item{idLabels}{an optional vector, or one-sided formula. If given as a
   vector, it is converted to character mode and used to label the
   observations identified according to \code{id}. If given as a
   one-sided formula, its right hand side must evaluate to a vector
   which is converted to character mode and used to label the identified
   observations. Default is the innermost grouping factor.}
 \item{idResType}{an optional character string specifying the type of
   residuals to be used in identifying outliers, when \code{id} is a
   numeric value. If \code{"pearson"}, the standardized residuals (raw 
   residuals divided by the corresponding standard errors) are used;
   else, if \code{"normalized"}, the normalized residuals (standardized
   residuals pre-multiplied by the inverse square-root factor of the
   estimated error correlation matrix) are used. Partial matching of
   arguments is used, so only the first character needs to be
   provided. Defaults to \code{"pearson"}.}
 \item{grid}{an optional logical value indicating whether a grid should
   be added to plot. Default depends on the type of Trellis plot used:
   if \code{xyplot} defaults to \code{TRUE}, else defaults to
   \code{FALSE}.}
 \item{\dots}{optional arguments passed to the Trellis plot function.}
}
\description{
  Diagnostic plots for the linear model fit are obtained. The
  \code{form} argument gives considerable flexibility in the type of
  plot specification. A conditioning expression (on the right side of a
  \code{|} operator) always implies that different panels are used for
  each level of the conditioning factor, according to a Trellis
  display. If \code{form} is a one-sided formula, histograms of the
  variable on the right hand side of the formula, before a \code{|}
  operator, are displayed (the Trellis function \code{histogram} is
  used). If \code{form} is two-sided and both its left and
  right hand side variables are numeric, scatter plots are displayed
  (the Trellis function \code{xyplot} is used). Finally, if \code{form}
  is two-sided and its left had side variable is a factor, box-plots of
  the right hand side variable by the levels of the left hand side
  variable are displayed (the Trellis function  \code{bwplot} is used). 
}
\value{
  a diagnostic Trellis plot.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{gls}}, \code{\link[lattice]{xyplot}},
   \code{\link[lattice]{bwplot}},
   \code{\link[lattice]{histogram}}}
\examples{
data(Ovary)
fm1 <- gls(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), Ovary,
           correlation = corAR1(form = ~ 1 | Mare))
# standardized residuals versus fitted values by Mare
plot(fm1, resid(., type = "p") ~ fitted(.) | Mare, abline = 0)
# box-plots of residuals by Mare
plot(fm1, Mare ~ resid(.))
# observed versus fitted values by Mare
plot(fm1, follicles ~ fitted(.) | Mare, abline = c(0,1))
}
\keyword{models}

\eof
% $Id: plot.intervals.lmList.Rd,v 1.9.2.1 2003/04/17 22:28:45 bates Exp $
\name{plot.intervals.lmList}
\title{Plot lmList Confidence Intervals}
\usage{
\method{plot}{intervals.lmList}(x, \dots)
}
\alias{plot.intervals.lmList}
\arguments{
 \item{x}{an object inheriting from class \code{intervals.lmList},
   representing confidence intervals and estimates for the coefficients
   in the \code{lm} components of the \code{lmList} object used to
   produce \code{x}.
 }
 \item{\dots}{optional arguments passed to the Trellis \code{dotplot}
   function.
 } 
}
\description{
  A Trellis dot-plot of the confidence intervals on the linear model
  coefficients is generated, with a different panel for each
  coefficient. Rows in the dot-plot correspond to the names of the
  \code{lm} components of the \code{lmList} object used to produce
  \code{x}. The lower and upper confidence limits are connected by
  a line segment and the estimated coefficients are marked with a
  \code{"+"}. The Trellis function \code{dotplot} is used in this method
  function.  
}

\value{
  a Trellis plot with the confidence intervals on the coefficients of
  the individual \code{lm} components of the \code{lmList} that
  generated \code{x}. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{intervals.lmList}}, \code{\link{lmList}},
  \code{\link{dotplot}}} 
\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age | Subject, Orthodont)
plot(intervals(fm1))
}
\keyword{models}

\eof
% $Id: plot.lmList.Rd,v 1.10.2.1 2003/04/17 22:28:45 bates Exp $
\name{plot.lmList}
\title{Plot an lmList Object}
\usage{
\method{plot}{lmList}(x, form, abline, id, idLabels, grid, \dots)
}
\alias{plot.lmList}
\arguments{
 \item{x}{an object inheriting from class \code{lmList}, representing
   a list of \code{lm} objects with a common model.}
 \item{form}{an optional formula specifying the desired type of
   plot. Any variable present in the original data frame used to obtain
   \code{x} can be referenced. In addition, \code{x} itself
   can be referenced in the formula using the symbol
   \code{"."}. Conditional expressions on the right of a \code{|}
   operator can be used to define separate panels in a Trellis
   display. Default is \code{resid(., type = "pool") ~ fitted(.) },
   corresponding to a plot of the standardized residuals (using a pooled
   estimate for the residual standard error) versus fitted values.
 } 
 \item{abline}{an optional numeric value, or numeric vector of length
   two. If given as a single value, a horizontal line will be added to the
   plot at that coordinate; else, if given as a vector, its values are
   used as the intercept and slope for a line added to the plot. If
   missing, no lines are added to the plot.
 }
 \item{id}{an optional numeric value, or one-sided formula. If given as
   a value, it is used as a significance level for a two-sided outlier
   test for the standardized residuals. Observations with
   absolute standardized residuals greater than the \eqn{1 - value/2}
   quantile of the standard normal distribution are identified in the
   plot using \code{idLabels}. If given as a one-sided formula, its
   right hand side must evaluate to a  logical, integer, or character
   vector which is used to identify observations in the plot. If
   missing, no observations are identified.
 }
 \item{idLabels}{an optional vector, or one-sided formula. If given as a
   vector, it is converted to character and used to label the
   observations identified according to \code{id}. If given as a
   one-sided formula, its right hand side must evaluate to a vector
   which is converted to character and used to label the identified
   observations. Default is \code{getGroups(x)}. 
 }
 \item{grid}{an optional logical value indicating whether a grid should
   be added to plot. Default depends on the type of Trellis plot used:
   if \code{xyplot} defaults to \code{TRUE}, else defaults to
   \code{FALSE}.
 }
 \item{\dots}{optional arguments passed to the Trellis plot function.}
}
\description{
  Diagnostic plots for the linear model fits corresponding to the
  \code{x}  components are obtained. The \code{form} argument gives
  considerable flexibility in the type of  plot specification. A
  conditioning expression (on the right side of a  \code{|} operator)
  always implies that different panels are used for  each level of the
  conditioning factor, according to a Trellis  display. If \code{form}
  is a one-sided formula, histograms of the  variable on the right hand
  side of the formula, before a \code{|}  operator, are displayed (the
  Trellis function \code{histogram} is  used). If \code{form} is
  two-sided and both its left and  right hand side variables are
  numeric, scatter plots are displayed  (the Trellis function
  \code{xyplot} is used). Finally, if \code{form}  is two-sided and its
  left had side variable is a factor, box-plots of  the right hand side
  variable by the levels of the left hand side  variable are displayed
  (the Trellis function  \code{bwplot} is used).  
}
\value{
  a diagnostic Trellis plot.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{lmList}}, \code{\link[lattice]{xyplot}},
   \code{\link[lattice]{bwplot}}, \code{\link[lattice]{histogram}} }
\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age | Subject, Orthodont)
# standardized residuals versus fitted values by gender
plot(fm1, resid(., type = "pool") ~ fitted(.) | Sex, abline = 0, id = 0.05)
# box-plots of residuals by Subject
plot(fm1, Subject ~ resid(.))
# observed versus fitted values by Subject
plot(fm1, distance ~ fitted(.) | Subject, abline = c(0,1))
}
\keyword{models}

\eof
% $Id: plot.lme.Rd,v 1.11 2002/04/16 14:28:53 bates Exp $
\name{plot.lme}
\title{Plot an lme or nls object}
\alias{plot.lme}
\alias{plot.nls}
\usage{
\method{plot}{lme}(x, form, abline, id, idLabels, idResType, grid, \dots)
\method{plot}{nls}(x, form, abline, id, idLabels, idResType, grid, \dots)
}
\arguments{
 \item{x}{an object inheriting from class \code{lme}, representing
   a fitted linear mixed-effects model, or from \code{nls}, representing
   an fitted nonlinear least squares model.}
 \item{form}{an optional formula specifying the desired type of
   plot. Any variable present in the original data frame used to obtain
   \code{x} can be referenced. In addition, \code{x} itself
   can be referenced in the formula using the symbol
   \code{"."}. Conditional expressions on the right of a \code{|}
   operator can be used to define separate panels in a Trellis
   display. Default is \code{resid(., type = "p") ~ fitted(.) },
   corresponding to a plot of the standardized residuals versus fitted
   values, both evaluated at the innermost level of nesting.} 
 \item{abline}{an optional numeric value, or numeric vector of length
   two. If given as a single value, a horizontal line will be added to the
   plot at that coordinate; else, if given as a vector, its values are
   used as the intercept and slope for a line added to the plot. If
   missing, no lines are added to the plot.}
 \item{id}{an optional numeric value, or one-sided formula. If given as
   a value, it is used as a significance level for a two-sided outlier
   test for the standardized, or normalized residuals. Observations with
   absolute standardized (normalized) residuals greater than the \eqn{1
     - value/2} quantile of the standard normal distribution are
   identified in the plot using \code{idLabels}. If given as a one-sided
   formula, its right hand side must evaluate to a  logical, integer, or
   character vector which is used to identify observations in the
   plot. If missing, no observations are identified.}
 \item{idLabels}{an optional vector, or one-sided formula. If given as a
   vector, it is converted to character and used to label the
   observations identified according to \code{id}. If given as a
   one-sided formula, its right hand side must evaluate to a vector
   which is converted to character and used to label the identified
   observations. Default is the innermost grouping factor.}
 \item{idResType}{an optional character string specifying the type of
   residuals to be used in identifying outliers, when \code{id} is a
   numeric value. If \code{"pearson"}, the standardized residuals (raw 
   residuals divided by the corresponding standard errors) are used;
   else, if \code{"normalized"}, the normalized residuals (standardized
   residuals pre-multiplied by the inverse square-root factor of the
   estimated error correlation matrix) are used. Partial matching of
   arguments is used, so only the first character needs to be
   provided. Defaults to \code{"pearson"}.}  
 \item{grid}{an optional logical value indicating whether a grid should
   be added to plot. Default depends on the type of Trellis plot used:
   if \code{xyplot} defaults to \code{TRUE}, else defaults to
   \code{FALSE}.}
 \item{\dots}{optional arguments passed to the Trellis plot function.}
}
\description{
  Diagnostic plots for the linear mixed-effects fit are obtained. The
  \code{form} argument gives considerable flexibility in the type of
  plot specification. A conditioning expression (on the right side of a
  \code{|} operator) always implies that different panels are used for
  each level of the conditioning factor, according to a Trellis
  display. If \code{form} is a one-sided formula, histograms of the
  variable on the right hand side of the formula, before a \code{|}
  operator, are displayed (the Trellis function \code{histogram} is
  used). If \code{form} is two-sided and both its left and
  right hand side variables are numeric, scatter plots are displayed
  (the Trellis function \code{xyplot} is used). Finally, if \code{form}
  is two-sided and its left had side variable is a factor, box-plots of
  the right hand side variable by the levels of the left hand side
  variable are displayed (the Trellis function  \code{bwplot} is used). 
}
\value{
  a diagnostic Trellis plot.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{lme}}, \code{\link[lattice]{xyplot}},
   \code{\link[lattice]{bwplot}}, \code{\link[lattice]{histogram}}}
\examples{
data(Orthodont)
fm1 <- lme(distance ~ age, Orthodont, random = ~ age | Subject)
# standardized residuals versus fitted values by gender
plot(fm1, resid(., type = "p") ~ fitted(.) | Sex, abline = 0)
# box-plots of residuals by Subject
plot(fm1, Subject ~ resid(.))
# observed versus fitted values by Subject
plot(fm1, distance ~ fitted(.) | Subject, abline = c(0,1))
}
\keyword{models}
\keyword{hplot}

\eof
% $Id: plot.nffGroupedData.Rd,v 1.7.2.1 2003/04/17 22:28:45 bates Exp $
\name{plot.nffGroupedData}
\title{Plot an nffGroupedData Object}
\usage{
\method{plot}{nffGroupedData}(x, outer, inner, innerGroups, xlab, ylab, strip, panel, key,
     grid, \dots)
}
\alias{plot.nffGroupedData}
\arguments{
 \item{x}{an object inheriting from class \code{nffGroupedData},
   representing a \code{groupedData} object with a factor primary
   covariate and a single grouping level.
 }
 \item{outer}{an optional logical value or one-sided formula,
   indicating covariates that are outer to the grouping factor, which
   are used to determine the panels of the Trellis plot. If
   equal to \code{TRUE}, \code{attr(object, "outer")} is used to indicate the
   outer covariates. An outer covariate is invariant within the sets
   of rows defined by the grouping factor.  Ordering of the groups is
   done in such a way as to preserve adjacency of groups with the same
   value of the outer variables. Defaults to \code{NULL}, meaning that
   no outer covariates are to be used.
 }  
 \item{inner}{an optional logical value or one-sided formula, indicating
   a covariate that is inner to the grouping factor, which is used to
   associate points within each panel of the Trellis plot. If
   equal to \code{TRUE}, \code{attr(object, "inner")} is used to indicate the
   inner covariate. An inner covariate can change within the sets of
   rows defined by the grouping  factor. Defaults to \code{NULL},
   meaning that no inner covariate  is present. 
 } 
 \item{innerGroups}{an optional one-sided formula specifying a factor
   to be used for grouping the levels of the \code{inner}
   covariate. Different colors, or symbols, are used for each level
   of the \code{innerGroups} factor. Default is \code{NULL}, meaning
   that no \code{innerGroups} covariate is present.
 }
 \item{xlab}{an optional character string with the label for the
   horizontal axis. Default is the \code{y} elements of \code{attr(object,
     "labels")} and \code{attr(object, "units")} pasted together.
 }
 \item{ylab}{an optional character string with the label for the
   vertical axis. Default is the grouping factor name.
 }
 \item{strip}{an optional function passed as the \code{strip} argument to
   the \code{dotplot} function. Default is \code{strip.default(..., style
     = 1)} (see \code{trellis.args}).
 }
 \item{panel}{an optional function used to generate the individual
   panels in the Trellis display, passed as the \code{panel} argument
   to the \code{dotplot} function.
 }
 \item{key}{an optional logical function or function. If \code{TRUE}
   and either \code{inner} or \code{innerGroups} are non-\code{NULL}, a
   legend for the different \code{inner} (\code{innerGroups}) levels is 
   included at the top of the plot. If given as a function, it is passed
   as the \code{key} argument to the \code{dotplot} function. Default is
   \code{TRUE} is either \code{inner} or \code{innerGroups} are
   non-\code{NULL} and \code{FALSE} otherwise.
 }
 \item{grid}{this argument is included for consistency with the
	 \code{plot.nfnGroupedData} method calling sequence. It is
	 ignored in this method function.} 
 \item{\dots}{optional arguments passed to the \code{dotplot} function.}
}
\description{
  A Trellis dot-plot of the response by group is generated. If outer
  variables are specified, the combination of their levels are used to
  determine the panels of the Trellis display. The Trellis function
  \code{dotplot} is used.
}
\value{
  a Trellis dot-plot of the response by group.
}

\references{
Bates, D.M. and Pinheiro, J.C. (1997), "Software Design for Longitudinal
Data", in "Modelling Longitudinal and Spatially Correlated Data:
Methods, Applications and Future Directions", T.G. Gregoire (ed.),
Springer-Verlag, New York.

Pinheiro, J.C. and Bates, D.M. (1997) "Future Directions in
Mixed-Effects Software: Design of NLME 3.0" available at
http://franz.stat.wisc.edu/pub/NLME.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{groupedData}}, \code{\link[lattice]{dotplot}}}
\examples{
data(Machines)
plot(Machines)
plot(Machines, inner = TRUE)
}
\keyword{models}

\eof
% $Id: plot.nfnGroupedData.Rd,v 1.7.2.1 2003/04/17 22:28:45 bates Exp $
\name{plot.nfnGroupedData}
\title{Plot an nfnGroupedData Object}
\usage{
\method{plot}{nfnGroupedData}(x, outer, inner, innerGroups, xlab, ylab, strip, aspect, panel,
     key, grid, \dots)
}
\alias{plot.nfnGroupedData}
\arguments{
 \item{x}{an object inheriting from class \code{nfnGroupedData},
   representing a \code{groupedData} object with a numeric primary
   covariate and a single grouping level.
 }
 \item{outer}{an optional logical value or one-sided formula,
   indicating covariates that are outer to the grouping factor, which
   are used to determine the panels of the Trellis plot. If
   equal to \code{TRUE}, \code{attr(object, "outer")} is used to
   indicate the outer covariates. An outer covariate is invariant within
   the sets of rows defined by the grouping factor.  Ordering of the
   groups is done in such a way as to preserve adjacency of groups with
   the same value of the outer variables. Defaults to \code{NULL},
   meaning that no outer covariates are to be used.
 }  
 \item{inner}{an optional logical value or one-sided formula, indicating
   a covariate that is inner to the grouping factor, which is used to
   associate points within each panel of the Trellis plot. If
   equal to \code{TRUE}, \code{attr(object, "inner")} is used to
   indicate the inner covariate. An inner covariate can change within
   the sets of rows defined by the grouping  factor. Defaults to
   \code{NULL}, meaning that no inner covariate  is present. 
 } 
 \item{innerGroups}{an optional one-sided formula specifying a factor
   to be used for grouping the levels of the \code{inner}
   covariate. Different colors, or line types, are used for each level
   of the \code{innerGroups} factor. Default is \code{NULL}, meaning
   that no \code{innerGroups} covariate is present.
 }
 \item{xlab, ylab}{optional character strings with the labels for the
   plot. Default is the corresponding elements of \code{attr(object,
     "labels")} and \code{attr(object, "units")} pasted together.
 }
 \item{strip}{an optional function passed as the \code{strip} argument to
   the \code{xyplot} function. Default is \code{strip.default(\dots, style
     = 1)} (see \code{trellis.args}).
   }
 \item{aspect}{an optional character string indicating the aspect ratio
   for the plot passed as the \code{aspect} argument to the
   \code{xyplot} function. Default is \code{"xy"} (see
   \code{trellis.args}). 
 }
 \item{panel}{an optional function used to generate the individual
   panels in the Trellis display, passed as the \code{panel} argument
   to the \code{xyplot} function.}
 \item{key}{an optional logical function or function. If \code{TRUE}
   and \code{innerGroups} is non-\code{NULL}, a legend for the
   different \code{innerGroups} levels is included at the top of the
   plot. If given as a function, it is passed as the \code{key} argument
   to the \code{xyplot} function.  Default is \code{TRUE} if
   \code{innerGroups} is non-\code{NULL} and \code{FALSE} otherwise.
 }
 \item{grid}{an optional logical value indicating whether a grid should
   be added to plot. Default is \code{TRUE}.}
 \item{\dots}{optional arguments passed to the \code{xyplot} function.}
}
\description{
  A Trellis plot of the response versus the primary covariate is
  generated. If outer variables are specified, the combination of their
  levels are used to determine the panels of the Trellis
  display. Otherwise, the levels of the grouping variable determine the
  panels. A scatter plot of the response versus the primary covariate is
  displayed in each panel, with observations corresponding to same
  inner group joined by line segments. The Trellis function
  \code{xyplot} is used. 
}
\value{
  a Trellis plot of the response versus the primary covariate.
}

\references{
Bates, D.M. and Pinheiro, J.C. (1997), "Software Design for Longitudinal
Data", in "Modelling Longitudinal and Spatially Correlated Data:
Methods, Applications and Future Directions", T.G. Gregoire (ed.),
Springer-Verlag, New York.

Pinheiro, J.C. and Bates, D.M. (1997) "Future Directions in
Mixed-Effects Software: Design of NLME 3.0" available at
http://franz.stat.wisc.edu/pub/NLME.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{groupedData}}, \code{\link[lattice]{xyplot}}}
\examples{
data(Orthodont)
# different panels per Subject
plot(Orthodont)
# different panels per gender
plot(Orthodont, outer = TRUE)
}
\keyword{models}

\eof
% $Id: plot.nmGroupedData.Rd,v 1.7.2.1 2003/04/17 22:28:45 bates Exp $
\name{plot.nmGroupedData}
\title{Plot an nmGroupedData Object}
\usage{
\method{plot}{nmGroupedData}(x, collapseLevel, displayLevel, outer, inner,
    preserve, FUN, subset, key, grid, \dots)
}
\alias{plot.nmGroupedData}
\arguments{
 \item{x}{an object inheriting from class \code{nmGroupedData},
   representing a \code{groupedData} object with multiple grouping
   factors.
 }
 \item{collapseLevel}{an optional positive integer or character string
   indicating the grouping level to use when collapsing the data. Level
   values increase from outermost to innermost grouping. Default is the
   highest or innermost level of grouping.
 }
 \item{displayLevel}{an optional positive integer or character string
   indicating the grouping level to use for determining the panels in
   the Trellis display, when \code{outer} is missing. Default is
   \code{collapseLevel}.
 }
 \item{outer}{an optional logical value or one-sided formula,
   indicating covariates that are outer to the \code{displayLevel}
   grouping factor, which are used to determine the panels of the
   Trellis plot. If equal to \code{TRUE}, the \code{displayLevel}
   element \code{attr(object, "outer")} is used to indicate the 
   outer covariates. An outer covariate is invariant within the sets
   of rows defined by the grouping factor.  Ordering of the groups is
   done in such a way as to preserve adjacency of groups with the same
   value of the outer variables. Defaults to \code{NULL}, meaning that
   no outer covariates are to be used.
 }  
 \item{inner}{an optional logical value or one-sided formula, indicating
   a covariate that is inner to the \code{displayLevel} grouping factor,
   which is used to associate points within each panel of the Trellis
   plot. If equal to \code{TRUE}, \code{attr(object, "outer")} is used
   to indicate the inner covariate. An inner covariate can change within
   the sets of rows defined by the grouping  factor. Defaults to
   \code{NULL}, meaning that no inner covariate  is present.  
 } 
 \item{preserve}{an optional one-sided formula indicating a covariate
   whose levels should be preserved when collapsing the data according
   to the \code{collapseLevel} grouping factor. The collapsing factor is
   obtained by pasting together the levels of the \code{collapseLevel}
   grouping factor and the values of the covariate to be
   preserved. Default is \code{NULL}, meaning that no covariates need to
   be preserved.
 }
  \item{FUN}{an optional summary function or a list of summary functions
   to be used for collapsing the data.  The function or functions are
   applied only to variables in \code{object} that vary within the
   groups defined by \code{collapseLevel}.  Invariant variables are 
   always summarized by group using the unique value that they assume
   within that group.  If \code{FUN} is a single
   function it will be applied to each non-invariant variable by group
   to produce the summary for that variable.  If \code{FUN} is a list of
   functions, the names in the list should designate classes of
   variables in the data such as \code{ordered}, \code{factor}, or
   \code{numeric}.  The indicated function will be applied to any
   non-invariant variables of that class.  The default functions to be
   used are \code{mean} for numeric factors, and \code{Mode} for both
   \code{factor} and \code{ordered}.  The \code{Mode} function, defined
   internally in \code{gsummary}, returns the modal or most popular
   value of the variable.  It is different from the \code{mode} function
   that returns the S-language mode of the variable.}
 \item{subset}{an optional named list. Names can be either positive
   integers representing grouping levels, or names of grouping
   factors. Each element in the list is a vector indicating the levels
   of the corresponding grouping factor to be used for plotting the
   data. Default is \code{NULL}, meaning that all levels are
   used.} 
 \item{key}{an optional logical value, or list. If \code{TRUE}, a legend
   is included at the top of the plot indicating which symbols (colors)
   correspond to which prediction levels. If \code{FALSE}, no legend
   is included. If given as a list, \code{key} is passed down as an
   argument to the \code{trellis} function generating the plots
   (\code{xyplot}). Defaults to \code{TRUE}.}
 \item{grid}{an optional logical value indicating whether a grid should
   be added to plot. Default is \code{TRUE}.}
 \item{\dots}{optional arguments passed to the Trellis plot function.}
}
\description{
 The \code{groupedData} object is summarized by the values of the
 \code{displayLevel} grouping factor (or the combination of its values
 and the values of the covariate indicated in \code{preserve}, if any is
 present). The collapsed data is used to produce a new
 \code{groupedData} object, with grouping factor given by the
 \code{displayLevel} factor, which is plotted using the
 appropriate \code{plot} method for \code{groupedData} objects with
 single level of grouping. 
}
\value{
  a Trellis display of the data collapsed over the values of the
  \code{collapseLevel} grouping factor and grouped according to the
  \code{displayLevel} grouping factor. 
}

\references{
Bates, D.M. and Pinheiro, J.C. (1997), "Software Design for Longitudinal
Data", in "Modelling Longitudinal and Spatially Correlated Data:
Methods, Applications and Future Directions", T.G. Gregoire (ed.),
Springer-Verlag, New York.

Pinheiro, J.C. and Bates, D.M. (1997) "Future Directions in
Mixed-Effects Software: Design of NLME 3.0" available at
\url{http://franz.stat.wisc.edu/pub/NLME}.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{groupedData}}, \code{\link{collapse.groupedData}},
  \code{\link{plot.nfnGroupedData}}, \code{\link{plot.nffGroupedData}}
}  
\examples{
data(Pixel)
# no collapsing, panels by Dog
plot(Pixel, display = "Dog", inner = ~Side)
# collapsing by Dog, preserving day
plot(Pixel, collapse = "Dog", preserve = ~day)
}
\keyword{models}

\eof
% $Id: plot.ranef.lmList.Rd,v 1.9.2.1 2003/04/17 22:28:45 bates Exp $
\name{plot.ranef.lmList}
\title{Plot a ranef.lmList Object}
\usage{
\method{plot}{ranef.lmList}(x, form, grid, control, \dots)
}
\alias{plot.ranef.lmList}
\arguments{
 \item{x}{an object inheriting from class
   \code{ranef.lmList}, representing the estimated coefficients or
   estimated random effects for the \code{lmList} object from which it was
   produced.
 } 
 \item{form}{an optional formula specifying the desired type of plot. If
     given as a one-sided formula, a \code{dotplot} of the estimated
     random effects (coefficients) grouped according to all combinations of
     the levels of the factors named in \code{form} is returned. Single
     factors (\code{~g}) or crossed factors (\code{~g1*g2}) are
     allowed. If given as a two-sided formula, the left hand side must
     be a single random effects (coefficient) and the right hand side
     is formed by covariates in \code{x} separated by \code{+}. A
     Trellis display of the random effect (coefficient) versus the named
     covariates is returned in this case. Default is \code{NULL}, in
     which case  the row names of the random effects (coefficients) are
     used.}
 \item{grid}{an optional logical value indicating whether a grid should
   be added to plot. Only applies to plots associated with two-sided
   formulas in \code{form}. Default is \code{FALSE}.}
 \item{control}{an optional list with control values for the
     plot, when \code{form} is given as a two-sided formula. The control
     values are referenced by name in the \code{control} list and  only
     the ones to be modified from the default need to be
     specified. Available values include: \code{drawLine}, a logical
     value indicating whether a \code{loess} smoother should be added to
     the scatter plots and a line connecting the medians should be added
     to the boxplots (default is \code{TRUE}); \code{span.loess}, used
     as the \code{span} argument in the call to \code{panel.loess}
     (default is \code{2/3}); \code{degree.loess}, used as the
     \code{degree} argument in the call to \code{panel.loess} (default
     is \code{1}); \code{cex.axis}, the character expansion factor for
     the x-axis (default is \code{0.8}); \code{srt.axis}, the rotation
     factor for the x-axis (default is \code{0}); and \code{mgp.axis}, the
     margin parameters for the x-axis (default is \code{c(2, 0.5, 0)}).}
 \item{\dots}{optional arguments passed to the Trellis \code{dotplot}
   function.
 } 
}
\description{
  If \code{form} is missing, or is given as a one-sided formula, a
  Trellis dot-plot of the random effects is generated, with a different
  panel for each random effect (coefficient). Rows in the dot-plot are
  determined by the \code{form} argument (if not missing) or by the row
  names of the random effects (coefficients). If a single factor is
  specified in \code{form}, its levels determine the dot-plot rows
  (with possibly multiple dots per row); otherwise, if \code{form}
  specifies a crossing of factors, the dot-plot rows are determined by
  all combinations of the levels of the individual factors in the 
  formula. The Trellis function \code{dotplot} is used in this method
  function.

  If \code{form} is a two-sided formula, a Trellis display is generated,
  with a different panel for each variable listed in the right hand side
  of \code{form}. Scatter plots are generated for numeric variables and
  boxplots are generated for categorical (\code{factor} or
  \code{ordered}) variables. 
}
\value{
  a Trellis plot of the estimated random-effects (coefficients) versus
  covariates, or groups.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{lmList}}, \code{\link[lattice]{dotplot}}}
\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age | Subject, Orthodont)
plot(ranef(fm1))
fm1RE <- ranef(fm1, aug = TRUE)
plot(fm1RE, form = ~ Sex)
\dontrun{
plot(fm1RE, form = age ~ Sex)
}
}
\keyword{models}


\eof
% $Id: plot.ranef.lme.Rd,v 1.8.2.1 2003/04/17 22:28:45 bates Exp $
\name{plot.ranef.lme}
\title{Plot a ranef.lme Object}
\usage{
\method{plot}{ranef.lme}(x, form, omitFixed, level, grid, control, \dots)
}
\alias{plot.ranef.lme}
\arguments{
 \item{x}{an object inheriting from class
   \code{ranef.lme}, representing the estimated coefficients or
   estimated random effects for the \code{lme} object from which it was
   produced.} 
 \item{form}{an optional formula specifying the desired type of plot. If
     given as a one-sided formula, a \code{dotplot} of the estimated
     random effects (coefficients) grouped according to all combinations of
     the levels of the factors named in \code{form} is returned. Single
     factors (\code{~g}) or crossed factors (\code{~g1*g2}) are
     allowed. If given as a two-sided formula, the left hand side must
     be a single random effects (coefficient) and the right hand side
     is formed by covariates in \code{x} separated by \code{+}. A
     Trellis display of the random effect (coefficient) versus the named
     covariates is returned in this case. Default is \code{NULL}, in
     which case  the row names of the random effects (coefficients) are
     used.}
 \item{omitFixed}{an optional logical value indicating whether
   columns with values that are constant across groups should be
   omitted. Default is \code{TRUE}.}
 \item{level}{an optional integer value giving the level of grouping
   to be used for \code{x}. Only used when \code{x} is a list
   with different components for each grouping level. Defaults to the
   highest or innermost level of grouping.}
 \item{grid}{an optional logical value indicating whether a grid should
   be added to plot. Only applies to plots associated with two-sided
   formulas in \code{form}. Default is \code{FALSE}.}
 \item{control}{an optional list with control values for the
     plot, when \code{form} is given as a two-sided formula. The control
     values are referenced by name in the \code{control} list and  only
     the ones to be modified from the default need to be
     specified. Available values include: \code{drawLine}, a logical
     value indicating whether a \code{loess} smoother should be added to
     the scatter plots and a line connecting the medians should be added
     to the boxplots (default is \code{TRUE}); \code{span.loess}, used
     as the \code{span} argument in the call to \code{panel.loess}
     (default is \code{2/3}); \code{degree.loess}, used as the
     \code{degree} argument in the call to \code{panel.loess} (default
     is \code{1}); \code{cex.axis}, the character expansion factor for
     the x-axis (default is \code{0.8}); \code{srt.axis}, the rotation
     factor for the x-axis (default is \code{0}); and \code{mgp.axis}, the
     margin parameters for the x-axis (default is \code{c(2, 0.5, 0)}).}
 \item{\dots}{optional arguments passed to the Trellis \code{dotplot}
   function.} 
}
\description{
  If \code{form} is missing, or is given as a one-sided formula, a
  Trellis dot-plot of the random effects is generated, with a different
  panel for each random effect (coefficient). Rows in the dot-plot are
  determined by the \code{form} argument (if not missing) or by the row
  names of the random effects (coefficients). If a single factor is
  specified in \code{form}, its levels determine the dot-plot rows
  (with possibly multiple dots per row); otherwise, if \code{form}
  specifies a crossing of factors, the dot-plot rows are determined by
  all combinations of the levels of the individual factors in the 
  formula. The Trellis function \code{dotplot} is used in this method
  function.

  If \code{form} is a two-sided formula, a Trellis display is generated,
  with a different panel for each variable listed in the right hand side
  of \code{form}. Scatter plots are generated for numeric variables and
  boxplots are generated for categorical (\code{factor} or
  \code{ordered}) variables. 
}
\value{
  a Trellis plot of the estimated random-effects (coefficients) versus
  covariates, or groups.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{ranef.lme}}, \code{\link{lme}},
  \code{\link{dotplot}}} 
\examples{
\dontrun{
data(Orthodont)
fm1 <- lme(distance ~ age, Orthodont, random = ~ age | Subject)
plot(ranef(fm1))
fm1RE <- ranef(fm1, aug = TRUE)
plot(fm1RE, form = ~ Sex)
plot(fm1RE, form = age ~ Sex)
}
}
\keyword{models}


\eof
% $Id: pooledSD.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{pooledSD}
\title{Extract Pooled Standard Deviation}
\usage{
pooledSD(object)
}
\alias{pooledSD}
\arguments{
 \item{object}{an object inheriting from class \code{lmList}.}
}
\description{
  The pooled estimated standard deviation is obtained by adding together
  the residual sum of squares for each non-null element of
  \code{object}, dividing by the sum of the corresponding residual
  degrees-of-freedom, and taking the square-root.
}
\value{
  the pooled standard deviation for the non-null elements of
  \code{object}, with an attribute \code{df} with the number of
  degrees-of-freedom used in the estimation.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lmList}}, \code{\link{lm}}}

\examples{
data(Orthodont)
fm1 <- lmList(Orthodont)
pooledSD(fm1)
}
\keyword{models}

\eof
% $Id: predict.gls.Rd,v 1.8.2.1 2003/05/17 21:59:38 bates Exp $
\name{predict.gls}
\title{Predictions from a gls Object}
\usage{
\method{predict}{gls}(object, newdata, na.action, \dots)
}
\alias{predict.gls}
\arguments{
 \item{object}{an object inheriting from class \code{gls}, representing
   a generalized least squares fitted linear model.}
 \item{newdata}{an optional data frame to be used for obtaining the
   predictions. All variables used in the linear model must be present
   in the data frame. If missing, the fitted values are returned.}
 \item{na.action}{a function that indicates what should happen when
   \code{newdata} contains \code{NA}s.  The default action
   (\code{na.fail}) causes the function to print an error message and
   terminate if there are any incomplete observations.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The predictions for the linear model represented by \code{object} are
  obtained at the covariate values defined in \code{newdata}.
}
\value{
  a vector with the predicted values.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gls}}}

\examples{
data(Ovary)
fm1 <- gls(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), Ovary,
           correlation = corAR1(form = ~ 1 | Mare))
newOvary <- data.frame(Time = c(-0.75, -0.5, 0, 0.5, 0.75))
predict(fm1, newOvary)
}
\keyword{models}

\eof
% $Id: predict.gnls.Rd,v 1.8.2.1 2003/05/17 22:00:48 bates Exp $
\name{predict.gnls}
\title{Predictions from a gnls Object}
\usage{
\method{predict}{gnls}(object, newdata, na.action, naPattern, \dots)
}
\alias{predict.gnls}
\arguments{
 \item{object}{an object inheriting from class \code{gnls}, representing
   a generalized nonlinear least squares fitted model.}
 \item{newdata}{an optional data frame to be used for obtaining the
   predictions. All variables used in the nonlinear model must be present
   in the data frame. If missing, the fitted values are returned.}
 \item{na.action}{a function that indicates what should happen when
   \code{newdata} contains \code{NA}s.  The default action
   (\code{na.fail}) causes the function to print an error message and
   terminate if there are any incomplete observations.}
 \item{naPattern}{an expression or formula object, specifying which returned
   values are to be regarded as missing.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The predictions for the nonlinear model represented by \code{object} are
  obtained at the covariate values defined in \code{newdata}.
}
\value{
  a vector with the predicted values.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gnls}}}

\examples{
data(Soybean)
fm1 <- gnls(weight ~ SSlogis(Time, Asym, xmid, scal), Soybean,
            weights = varPower())
newSoybean <- data.frame(Time = c(10,30,50,80,100))
predict(fm1, newSoybean)
}
\keyword{models}

\eof
% $Id: predict.lmList.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{predict.lmList}
\title{Predictions from an lmList Object}
\usage{
\method{predict}{lmList}(object, newdata, subset, pool, asList, se.fit, \dots)
}
\alias{predict.lmList}
\arguments{
 \item{object}{an object inheriting from class \code{lmList}, representing
   a list of \code{lm} objects with a common model.
 }
 \item{newdata}{an optional data frame to be used for obtaining the
   predictions. All variables used in the \code{object} model formula
   must be present in the data frame. If missing, the same data frame
   used to produce \code{object} is used.
 } 
 \item{subset}{an optional character or integer vector naming the
   \code{lm} components of \code{object} from which the predictions
   are to be extracted. Default is \code{NULL}, in which case all
   components are used.
 }
 \item{asList}{an optional logical value. If \code{TRUE}, the returned
   object is a list with the predictions split by groups; else the
   returned value is a vector. Defaults to \code{FALSE}.
 }
 \item{pool}{an optional logical value indicating whether a pooled
   estimate of the residual standard error should be used. Default is
   \code{attr(object, "pool")}.
 }
 \item{se.fit}{an optional logical value indicating whether pointwise
   standard errors should be computed along with the
   predictions. Default is \code{FALSE}.
 }
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  If the grouping factor corresponding to \code{object} is included
  in \code{newdata}, the data frame is partitioned according to the
  grouping factor levels; else, \code{newdata} is repeated for all
  \code{lm} components. The predictions and, optionally, the standard
  errors for the predictions, are obtained for each \code{lm}
  component of \code{object}, using the corresponding element of the
  partitioned \code{newdata}, and arranged into a list with as many
  components as \code{object}, or combined into a single vector or data
  frame (if \code{se.fit=TRUE}).
}
\value{
  a list with components given by the predictions (and, optionally, the
  standard errors for the predictions) from each \code{lm}
  component of \code{object},  a vector with the predictions from all 
  \code{lm} components of \code{object}, or a data frame with columns
  given by the predictions and their corresponding standard errors.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lmList}}, \code{\link{predict.lm}}}

\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age | Subject, Orthodont)
predict(fm1, se.fit = TRUE)
}
\keyword{models}

\eof
% $Id: predict.lme.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{predict.lme}
\title{Predictions from an lme Object}
\usage{
\method{predict}{lme}(object, newdata, level, asList, na.action, \dots)
}
\alias{predict.lme}
\arguments{
 \item{object}{an object inheriting from class \code{lme}, representing
   a fitted linear mixed-effects model.}
 \item{newdata}{an optional data frame to be used for obtaining the
   predictions. All variables used in the fixed and random effects
   models, as well as the grouping factors, must be present in the data
   frame. If missing, the fitted values are returned.}
 \item{level}{an optional integer vector giving the level(s) of grouping
   to be used in obtaining the predictions. Level values increase from
   outermost to innermost grouping, with level zero corresponding to the
   population predictions. Defaults to the highest or innermost level of
   grouping.}  
 \item{asList}{an optional logical value. If \code{TRUE} and a single
   value is given in \code{level}, the returned object is a list with
   the predictions split by groups; else the returned value is
   either a vector or a data frame, according to the length of
   \code{level}. }
 \item{na.action}{a function that indicates what should happen when
   \code{newdata} contains \code{NA}s.  The default action
   (\code{na.fail}) causes the function to print an error message and
   terminate if there are any incomplete observations.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The predictions at level \eqn{i} are obtained by adding together the
  population predictions (based only on the fixed effects estimates)
  and the estimated contributions of the random effects to the
  predictions at grouping levels less or equal to \eqn{i}. The resulting
  values estimate the best linear unbiased predictions (BLUPs) at level
  \eqn{i}. If group values not included in the original grouping factors
  are present in \code{newdata}, the corresponding predictions will be
  set to \code{NA} for levels greater or equal to the level at which the
  unknown groups occur.
}
\value{
  if a single level of grouping is specified in \code{level}, the
  returned value is either a list with the predictions split by groups
  (\code{asList = TRUE}) or a vector with the predictions
  (\code{asList = FALSE}); else, when multiple grouping levels are
  specified in \code{level}, the returned object is a data frame with
  columns given by the predictions at different levels and the grouping
  factors. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lme}}, \code{\link{fitted.lme}}}

\examples{
data(Orthodont)
fm1 <- lme(distance ~ age, Orthodont, random = ~ age | Subject)
newOrth <- data.frame(Sex = c("Male","Male","Female","Female","Male","Male"),
                      age = c(15, 20, 10, 12, 2, 4),
                      Subject = c("M01","M01","F30","F30","M04","M04"))
predict(fm1, newOrth, level = 0:1)
}
\keyword{models}

\eof
% $Id: predict.nlme.Rd,v 1.9 2002/03/05 14:59:39 bates Exp $
\name{predict.nlme}
\title{Predictions from an nlme Object}
\usage{
\method{predict}{nlme}(object, newdata, level, asList, na.action,
naPattern, \dots)
}
\alias{predict.nlme}
\arguments{
 \item{object}{an object inheriting from class \code{nlme}, representing
   a fitted nonlinear mixed-effects model.}
 \item{newdata}{an optional data frame to be used for obtaining the
   predictions. All variables used in the nonlinear model, the fixed and
   the random effects models, as well as the grouping factors, must be
   present in the data frame. If missing, the fitted values are returned.}
 \item{level}{an optional integer vector giving the level(s) of grouping
   to be used in obtaining the predictions. Level values increase from
   outermost to innermost grouping, with level zero corresponding to the
   population predictions. Defaults to the highest or innermost level of
   grouping.}  
 \item{asList}{an optional logical value. If \code{TRUE} and a single
   value is given in \code{level}, the returned object is a list with
   the predictions split by groups; else the returned value is
   either a vector or a data frame, according to the length of
   \code{level}. }
 \item{na.action}{a function that indicates what should happen when
   \code{newdata} contains \code{NA}s.  The default action
   (\code{na.fail}) causes the function to print an error message and
   terminate if there are any incomplete observations.}
 \item{naPattern}{an expression or formula object, specifying which returned
   values are to be regarded as missing.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The predictions at level \eqn{i} are obtained by adding together the
  contributions from the estimated fixed effects and the estimated
  random effects at levels less or equal to \eqn{i} and evaluating the
  model function at the resulting estimated parameters. If group values
  not included in the original grouping factors  are present in
  \code{newdata}, the corresponding predictions will be  set to
  \code{NA} for levels greater or equal to the level at which the 
  unknown groups occur.
}
\value{
  if a single level of grouping is specified in \code{level}, the
  returned value is either a list with the predictions split by groups
  (\code{asList = TRUE}) or a vector with the predictions
  (\code{asList = FALSE}); else, when multiple grouping levels are
  specified in \code{level}, the returned object is a data frame with
  columns given by the predictions at different levels and the grouping
  factors. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{nlme}}, \code{\link{fitted.lme}}}
\examples{
data(Loblolly)
fm1 <- nlme(height ~ SSasymp(age, Asym, R0, lrc),
            data = Loblolly,
            fixed = Asym + R0 + lrc ~ 1,
            random = Asym ~ 1,
            start = c(Asym = 103, R0 = -8.5, lrc = -3.3))
newLoblolly <- data.frame(age = c(5,10,15,20,25,30),
                    Seed = rep(301,6))
predict(fm1, newLoblolly, level = 0:1)
}
\keyword{models}

\eof
% $Id: print.summary.pdMat.Rd,v 1.7.2.1 2003/04/17 22:28:45 bates Exp $
\name{print.summary.pdMat}
\title{Print a summary.pdMat Object}
\usage{
\method{print}{summary.pdMat}(x, sigma, rdig, Level, resid, \dots)
}
\alias{print.summary.pdMat}
\arguments{
 \item{x}{an object inheriting from class \code{summary.pdMat},
   generally resulting from applying \code{summary} to an object
   inheriting from class \code{pdMat}.}
 \item{sigma}{an optional numeric value used as a multiplier for
   the square-root factor of the positive-definite matrix represented by
   \code{object} (usually the estimated within-group standard deviation
   from a mixed-effects model). Defaults to 1.} 
 \item{rdig}{an optional integer value with the number of significant
   digits to be used in printing correlations. Defaults to 3.}
 \item{Level}{an optional character string with a description of the
   grouping level associated with \code{object} (generally corresponding
   to levels of grouping in a mixed-effects model). Defaults to NULL.}
 \item{resid}{an optional logical value. If \code{TRUE} an extra row
   with the \code{"residual"} standard deviation given in \code{sigma}
   will be included in the output. Defaults to \code{FALSE}.}
 \item{\dots}{optional arguments passed to \code{print.default}; see
   the documentation on that method function.}
}
\description{
  The standard deviations and correlations associated with the
  positive-definite matrix represented by \code{object} (considered as a
  variance-covariance matrix) are printed, together with the formula and
  the grouping level associated \code{object}, if any are present.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{summary.pdMat}},\code{\link{pdMat}}}

\examples{
data(Orthodont)
%\dontrun{
pd1 <- pdCompSymm(3 * diag(2) + 1, form = ~age + age^2,
         data = Orthodont)
print(summary(pd1), sigma = 1.2, resid = TRUE)
%}
}
\keyword{models}

\eof
% $Id: print.varFunc.Rd,v 1.6.2.3 2003/05/17 22:02:03 bates Exp $
\name{print.varFunc}
\title{Print a varFunc Object}
\usage{
\method{print}{varFunc}(x, \dots)
}
\alias{print.varFunc}
\alias{print.varComb}
\arguments{
 \item{x}{an object inheriting from class \code{varFunc}, representing a
 variance function structure.}
 \item{\dots}{optional arguments passed to \code{print.default}; see
   the documentation on that method function.}
}
\description{
  The class and the coefficients associated with \code{x} are printed.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{summary.varFunc}}}

\examples{
data(Orthodont)
vf1 <- varPower(0.3, form = ~age)
vf1 <- Initialize(vf1, Orthodont)
print(vf1)
}
\keyword{models}

\eof
% $Id: qqnorm.gls.Rd,v 1.9.2.1 2003/04/17 22:28:45 bates Exp $
\name{qqnorm.gls}
\title{Normal Plot of Residuals from a gls Object}
\usage{
\method{qqnorm}{gls}(y, form, abline, id, idLabels, grid, \dots)
}
\alias{qqnorm.gls}
\arguments{
 \item{y}{an object inheriting from class \code{gls}, representing
   a generalized least squares fitted model.}
 \item{form}{an optional one-sided formula specifying the desired type of
   plot. Any variable present in the original data frame used to obtain
   \code{y} can be referenced. In addition, \code{y} itself
   can be referenced in the formula using the symbol
   \code{"."}. Conditional expressions on the right of a \code{|}
   operator can be used to define separate panels in a Trellis
   display. The expression on the right hand side of \code{form} and to
   the left of a \code{|} operator must evaluate to a residuals
   vector. Default is \code{~ resid(., type = "p")}, 
   corresponding to a normal plot of the standardized residuals.}  
 \item{abline}{an optional numeric value, or numeric vector of length
   two. If given as a single value, a horizontal line will be added to the
   plot at that coordinate; else, if given as a vector, its values are
   used as the intercept and slope for a line added to the plot. If
   missing, no lines are added to the plot.}
 \item{id}{an optional numeric value, or one-sided formula. If given as
   a value, it is used as a significance level for a two-sided outlier
   test for the standardized residuals (random effects). Observations with
   absolute standardized residuals (random effects) greater than the
   \eqn{1 - value/2} quantile of the standard normal distribution are
   identified in the plot using \code{idLabels}. If given as a one-sided
   formula, its right hand side must evaluate to a  logical, integer, or
   character vector which is used to identify observations in the
   plot. If missing, no observations are identified.}
 \item{idLabels}{an optional vector, or one-sided formula. If given as a
   vector, it is converted to character and used to label the
   observations identified according to \code{id}. If given as a
   one-sided formula, its right hand side must evaluate to a vector
   which is converted to character and used to label the identified
   observations. Default is the innermost grouping factor.}
 \item{grid}{an optional logical value indicating whether a grid should
   be added to plot. Default depends on the type of Trellis plot used:
   if \code{xyplot} defaults to \code{TRUE}, else defaults to
   \code{FALSE}.}
 \item{\dots}{optional arguments passed to the Trellis plot function.}
}
\description{
  Diagnostic plots for assessing the normality of residuals the
  generalized least squares fit are obtained. The \code{form} argument
  gives considerable flexibility in the type of plot specification. A
  conditioning expression (on the right side of a \code{|} operator)
  always implies that different panels are used for each level of the
  conditioning factor, according to a Trellis display. 
}
\value{
  a diagnostic Trellis plot for assessing normality of residuals.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{gls}}, \code{\link{plot.gls}}}
\examples{
data(Ovary)
fm1 <- gls(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), Ovary,
           correlation = corAR1(form = ~ 1 | Mare))
qqnorm(fm1, abline = c(0,1))
}
\keyword{models}

\eof
% $Id: qqnorm.lme.Rd,v 1.8.2.1 2003/04/17 22:28:45 bates Exp $
\name{qqnorm.lme}
\title{Normal Plot of Residuals or Random Effects from an lme Object}
\usage{
\method{qqnorm}{lme}(y, form, abline, id, idLabels, grid, \dots)
}
\alias{qqnorm.lm}
\alias{qqnorm.lme}
\alias{qqnorm.lmList}
\alias{qqnorm.nls}
\arguments{
 \item{y}{an object inheriting from class \code{lme}, representing
   a fitted linear mixed-effects model or from class \code{lmList},
   representing a list of \code{lm} objects, or from class \code{lm},
   representing a fitted linear model, or from class \code{nls},
   representing a nonlinear least squares fitted model.}
 \item{form}{an optional one-sided formula specifying the desired type of
   plot. Any variable present in the original data frame used to obtain
   \code{y} can be referenced. In addition, \code{y} itself
   can be referenced in the formula using the symbol
   \code{"."}. Conditional expressions on the right of a \code{|}
   operator can be used to define separate panels in a Trellis
   display. The expression on the right hand side of \code{form} and to
   the left of a \code{|} operator must evaluate to a residuals vector,
   or a random effects matrix. Default is \code{~ resid(., type = "p")},
   corresponding to a normal plot of the standardized residuals
   evaluated at the innermost level of nesting.}  
 \item{abline}{an optional numeric value, or numeric vector of length
   two. If given as a single value, a horizontal line will be added to the
   plot at that coordinate; else, if given as a vector, its values are
   used as the intercept and slope for a line added to the plot. If
   missing, no lines are added to the plot.}
 \item{id}{an optional numeric value, or one-sided formula. If given as
   a value, it is used as a significance level for a two-sided outlier
   test for the standardized residuals (random effects). Observations with
   absolute standardized residuals (random effects) greater than the
   \eqn{1 - value/2} quantile of the standard normal distribution are
   identified in the plot using \code{idLabels}. If given as a one-sided
   formula, its right hand side must evaluate to a  logical, integer, or
   character vector which is used to identify observations in the
   plot. If missing, no observations are identified.}
 \item{idLabels}{an optional vector, or one-sided formula. If given as a
   vector, it is converted to character and used to label the
   observations identified according to \code{id}. If given as a
   one-sided formula, its right hand side must evaluate to a vector
   which is converted to character and used to label the identified
   observations. Default is the innermost grouping factor.}
 \item{grid}{an optional logical value indicating whether a grid should
   be added to plot. Default is \code{FALSE}.}
 \item{\dots}{optional arguments passed to the Trellis plot function.}
}
\description{
  Diagnostic plots for assessing the normality of residuals and random
  effects in the linear mixed-effects fit are obtained. The
  \code{form} argument gives considerable flexibility in the type of
  plot specification. A conditioning expression (on the right side of a
  \code{|} operator) always implies that different panels are used for
  each level of the conditioning factor, according to a Trellis
  display. 
}
\value{
  a diagnostic Trellis plot for assessing normality of residuals or
  random effects.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{lme}}, \code{\link{plot.lme}}}
\examples{
\dontrun{
data(Orthodont)
fm1 <- lme(distance ~ age, Orthodont, random = ~ age | Subject)
# normal plot of standardized residuals by gender
qqnorm(fm1, ~ resid(., type = "p") | Sex, abline = c(0, 1))
# normal plots of random effects
qqnorm(fm1, ~ranef(.))
}
}
\keyword{models}

\eof
\name{quinModel}
\alias{quinModel}
\title{Model function for the Quinidine data}
\description{
  A model function for a model used with the \code{Quinidine} data.
  This function calls compiled C code.
}
\usage{
quinModel(Subject, time, conc, dose, interval, lV, lKa, lCl)
}
\arguments{
  \item{Subject}{
    a factor identifying the patient on whom the data were collected.
  }
  \item{time}{
    a numeric vector giving the time (hr) at which the drug was
    administered or the blood sample drawn.  This is measured
    from the time the patient entered the study.
  }
  \item{conc}{
    a numeric vector giving the serum quinidine concentration (mg/L).
  }
  \item{dose}{
    a numeric vector giving the dose of drug administered (mg).
    Although there were two different forms of quinidine
    administered, the doses were adjusted for differences in
    salt content by conversion to milligrams of quinidine base.
  }
  \item{interval}{
    a numeric vector giving the when the drug has been given at
    regular intervals for a sufficiently long period of time to
    assume steady state behavior, the interval is recorded.
  }
  \item{lV}{numeric.  A vector of values of the natural log of the
    effective volume of distribution according to \code{Subject} and
    \code{time}.}
  \item{lKa}{numeric.  A vector of values of the natural log of the
    absorption rate constant according to \code{Subject} and \code{time}.}
  \item{lCl}{numeric.  A vector of values of the natural log of the
    clearance parameter according to \code{Subject} and \code{time}.}
}
\details{
  See the details section of \code{\link{Quinidine}} for a description
  of the model function that \code{quinModel} evaluates.
}
\value{
  a numeric vector of predicted quinidine concentrations.
}
\references{
    Pinheiro, J. C. and Bates, D. M. (2000) \emph{Mixed-effects Models in
	S and S-PLUS}, Springer. (section 8.2)
}
\author{Jose Pinheiro \email{jose.pinheiro@pharma.novartis.com}
  and Douglas Bates \email{bates@stat.wisc.edu}
}
\examples{
}
\keyword{models}

\eof
% $Id: random.effects.Rd,v 1.4.4.1 2003/04/17 22:28:45 bates Exp $
\name{random.effects}
\title{Extract Random Effects}
\usage{
random.effects(object, \dots)
ranef(object, \dots)
}
\alias{random.effects}
\alias{ranef}
\alias{print.ranef}
\arguments{
 \item{object}{any fitted model object from which random effects
   estimates can be extracted.}
 \item{\dots}{some methods for this generic function require additional
   arguments.}
}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include \code{lmList} and \code{lme}.
}
\value{
  will depend on the method function used; see the appropriate documentation.
}
\seealso{\code{\link{ranef.lmList}},\code{\link{ranef.lme}}} 
\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: ranef.lmList.Rd,v 1.4.2.1 2003/06/12 14:59:10 bates Exp $
\name{ranef.lmList}
\title{Extract lmList Random Effects}
\usage{
\method{ranef}{lmList}(object, augFrame, data, which, FUN, standard,
                 omitGroupingFactor, \dots)
}
\alias{random.effects.lmList}
\alias{ranef.lmList}
\arguments{
 \item{object}{an object inheriting from class \code{lmList}, representing
   a list of \code{lm} objects with a common model.
 }
 \item{augFrame}{an optional logical value. If \code{TRUE}, the returned
   data frame is augmented with variables defined in \code{data}; else,
   if \code{FALSE}, only the coefficients are returned. Defaults to
   \code{FALSE}.}
 \item{data}{an optional data frame with the variables to be used for
   augmenting the returned data frame when \code{augFrame =
     TRUE}. Defaults to the data frame used to fit \code{object}.}
 \item{which}{an optional positive integer vector specifying which
   columns of \code{data} should be used in the augmentation of the
   returned data frame. Defaults to all columns in \code{data}.}
 \item{FUN}{an optional summary function or a list of summary functions
   to be applied to group-varying variables, when collapsing \code{data}
   by groups.  Group-invariant variables are always summarized by the
   unique value that they assume within that group. If \code{FUN} is a
   single function it will be applied to each non-invariant variable by
   group to produce the summary for that variable.  If \code{FUN} is a
   list of functions, the names in the list should designate classes of
   variables in the frame such as \code{ordered}, \code{factor}, or
   \code{numeric}.  The indicated function will be applied to any
   group-varying variables of that class.  The default functions to be
   used are \code{mean} for numeric factors, and \code{Mode} for both
   \code{factor} and \code{ordered}.  The \code{Mode} function, defined
   internally in \code{gsummary}, returns the modal or most popular
   value of the variable.  It is different from the \code{mode} function
   that returns the S-language mode of the variable.}
 \item{standard}{an optional logical value indicating whether the
   estimated random effects should be "standardized" (i.e. divided by
   the corresponding estimated standard error). Defaults to
   \code{FALSE}.} 
 \item{omitGroupingFactor}{an optional logical value.  When \code{TRUE}
   the grouping factor itself will be omitted from the group-wise
   summary of \code{data} but the levels of the grouping factor will
   continue to be used as the row names for the returned data frame.
   Defaults to \code{FALSE}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The difference between the individual \code{lm} components
  coefficients and their average is calculated.
}
\value{
  a vector with the differences between the individual \code{lm}
  coefficients in \code{object} and their average. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lmList}}, \code{\link{fixed.effects.lmList}}}
 
\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age | Subject, Orthodont)
ranef(fm1)
random.effects(fm1)              # same as above
}
\keyword{models}

\eof
% $Id: ranef.lme.Rd,v 1.4.2.1 2003/05/17 22:03:12 bates Exp $
\name{ranef.lme}
\title{Extract lme Random Effects}
\usage{
\method{ranef}{lme}(object, augFrame, level, data, which, FUN,
      standard, omitGroupingFactor, subset, \dots)
}
\alias{ranef.lme}
\alias{random.effects.lme}
\alias{print.ranef.lme}
\arguments{
 \item{object}{an object inheriting from class \code{lme}, representing
   a fitted linear mixed-effects model.}
 \item{augFrame}{an optional logical value. If \code{TRUE}, the returned
   data frame is augmented with variables defined in \code{data}; else,
   if \code{FALSE}, only the coefficients are returned. Defaults to
   \code{FALSE}.}
 \item{level}{an optional vector of positive integers giving the levels
   of grouping to be used in extracting the random effects from an
   object with multiple nested grouping levels. Defaults to all levels
   of grouping.} 
 \item{data}{an optional data frame with the variables to be used for
   augmenting the returned data frame when \code{augFrame =
     TRUE}. Defaults to the data frame used to fit \code{object}.}
 \item{which}{an optional positive integer vector specifying which
   columns of \code{data} should be used in the augmentation of the
   returned data frame. Defaults to all columns in \code{data}.}
 \item{FUN}{an optional summary function or a list of summary functions
   to be applied to group-varying variables, when collapsing \code{data}
   by groups.  Group-invariant variables are always summarized by the
   unique value that they assume within that group. If \code{FUN} is a
   single function it will be applied to each non-invariant variable by
   group to produce the summary for that variable.  If \code{FUN} is a
   list of functions, the names in the list should designate classes of
   variables in the frame such as \code{ordered}, \code{factor}, or
   \code{numeric}.  The indicated function will be applied to any
   group-varying variables of that class.  The default functions to be
   used are \code{mean} for numeric factors, and \code{Mode} for both
   \code{factor} and \code{ordered}.  The \code{Mode} function, defined
   internally in \code{gsummary}, returns the modal or most popular
   value of the variable.  It is different from the \code{mode} function
   that returns the S-language mode of the variable.}
 \item{standard}{an optional logical value indicating whether the
   estimated random effects should be "standardized" (i.e. divided by
   the corresponding estimated standard error). Defaults to
   \code{FALSE}.} 
 \item{omitGroupingFactor}{an optional logical value.  When \code{TRUE}
   the grouping factor itself will be omitted from the group-wise
   summary of \code{data} but the levels of the grouping factor will
   continue to be used as the row names for the returned data frame.
   Defaults to \code{FALSE}.}
 \item{subset}{an optional expression indicating for which rows the
   random effects should be extracted.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The estimated random effects at level \eqn{i} are represented as a
  data frame with rows given by the different groups at that level and
  columns given by the random effects. If a single level of
  grouping is specified, the returned object is a data frame; else, the
  returned object is a list of such data frames. Optionally, the
  returned data frame(s) may be augmented with covariates summarized
  over groups.  
}
\value{
  a data frame, or list of data frames, with the estimated 
  random effects at the grouping level(s) specified in \code{level} and,
  optionally, other covariates  summarized over groups. The returned
  object inherits from classes \code{random.effects.lme} and
  \code{data.frame}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lme}}, %\code{\link{fixed.effects.lme}},
  \code{\link{coef.lme}}, \code{\link{plot.ranef.lme}},
  \code{\link{gsummary}}}  

\examples{
data(Orthodont)
fm1 <- lme(distance ~ age, Orthodont, random = ~ age | Subject)
ranef(fm1)
random.effects(fm1)             # same as above
random.effects(fm1, augFrame = TRUE)
}
\keyword{models}

\eof
% $Id: reStruct.Rd,v 1.6.2.1 2003/04/17 22:28:45 bates Exp $
\name{reStruct}
\title{Random Effects Structure}
\usage{
reStruct(object, pdClass, REML, data)
\method{print}{reStruct}(x, sigma, reEstimates, verbose, \dots)
}
\alias{reStruct}
\alias{[.reStruct}
\alias{print.reStruct}
\arguments{
 \item{object}{any of the following: (i) a one-sided formula of the form
   \code{~x1+...+xn | g1/.../gm}, with \code{x1+...+xn} specifying the
   model for the random effects and \code{g1/.../gm} the grouping
   structure (\code{m} may be equal to 1, in which case no \code{/} is
   required). The random effects formula will be repeated for all levels
   of grouping, in the case of multiple levels of grouping; (ii) a list of
   one-sided formulas of the form \code{~x1+...+xn | g}, with possibly
   different random effects models for each grouping level. The order of
   nesting will be assumed the same as the order of the elements in the
   list; (iii) a one-sided formula of the form \code{~x1+...+xn}, or a
   \code{pdMat} object with a formula (i.e. a non-\code{NULL} value for
   \code{formula(object)}), or a list of such formulas or \code{pdMat}
   objects. In this case, the grouping structure formula will be derived
   from the data used to to fit the mixed-effects model, which should
   inherit from class \code{groupedData}; (iv) a named list of formulas or
   \code{pdMat} objects as in (iii), with the grouping factors as
   names. The order of nesting will be assumed the same as the order of
   the order of the elements in the list; (v) an \code{reStruct} object.}
 \item{pdClass}{an optional character string with the name of the
   \code{pdMat} class to be used for the formulas in
   \code{object}. Defaults to \code{"pdSymm"} which corresponds to a
   general positive-definite matrix.}
 \item{REML}{an optional logical value. If \code{TRUE}, the associated
   mixed-effects model will be fitted using restricted maximum
   likelihood; else, if \code{FALSE}, maximum likelihood will be
   used. Defaults to \code{FALSE}.}
 \item{data}{an optional data frame in which to evaluate the variables
   used in the random effects formulas in \code{object}. It is used to
   obtain the levels for \code{factors}, which affect the dimensions and
   the row/column names of the underlying \code{pdMat} objects. If
   \code{NULL}, no attempt is made to obtain information on
   \code{factors} appearing in the formulas. Defaults to the
   parent frame from which the function was called.}
 \item{x}{an object inheriting from class \code{reStruct} to be printed.}
 \item{sigma}{an optional numeric value used as a multiplier for
   the square-root factors of the \code{pdMat} components (usually the
   estimated within-group standard deviation from a mixed-effects
   model). Defaults to 1.} 
 \item{reEstimates}{an optional list with the random effects estimates
   for each level of grouping. Only used when \code{verbose = TRUE}.}
 \item{verbose}{an optional logical value determining if the random
   effects estimates should be printed. Defaults to \code{FALSE}.}
 \item{\dots}{Optional arguments can be given to other methods for this
   generic.  None are used in this method.}
}
\description{
  This function is a constructor for the \code{reStruct} class,
  representing a random effects structure and consisting of a list of
  \code{pdMat} objects, plus a \code{settings} attribute containing
  information for the optimization algorithm used to fit the associated
  mixed-effects model.
}
\value{
  an object inheriting from class \code{reStruct}, representing a random
  effects structure.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{pdMat}}, \code{\link{lme}},
  \code{\link{groupedData}}} 

\examples{
data(Pixel)
rs1 <- reStruct(list(Dog = ~day, Side = ~1), data = Pixel)
rs1
}
\keyword{models}

\eof
% $Id: recalc.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{recalc}
\title{Recalculate Condensed Linear Model Object}
\usage{
recalc(object, conLin, \dots)
}
\alias{recalc}
\arguments{
 \item{object}{any object which induces a recalculation of the condensed
 linear model object \code{conLin}.}
 \item{conLin}{a condensed linear model object, consisting of a list
   with components \code{"Xy"}, corresponding to a regression matrix
   (\code{X}) combined with a response vector (\code{y}), and
   \code{"logLik"}, corresponding to the log-likelihood of the
   underlying model.}
 \item{\dots}{some methods for this generic can take additional
   arguments.} 

}
\description{
  This function is generic; method functions can be written to handle
  specific classes of objects. Classes which already have methods for
  this function include: \code{corStruct}, \code{modelStruct},
  \code{reStruct}, and \code{varFunc}.
}
\value{
  the recalculated condensed linear model object.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{This function is only used inside model fitting functions, such as
    \code{lme} and \code{gls}, that require recalculation of a condensed
    linear model object.}
\examples{
## see the method function documentation
}
\keyword{models}

\eof
% $Id: recalc.corStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{recalc.corStruct}
\title{Recalculate for corStruct Object}
\usage{
\method{recalc}{corStruct}(object, conLin, \dots)
}
\alias{recalc.corStruct}
\alias{recalc.corAR1}
\alias{recalc.corARMA}
\alias{recalc.corCAR1}
\alias{recalc.corCompSymm}
\alias{recalc.corHF}
\alias{recalc.corIdent}
\alias{recalc.corNatural}
\alias{recalc.corSpatial}
\alias{recalc.corSymm}

\arguments{
 \item{object}{an object inheriting from class \code{corStruct},
   representing a correlation structure.}
 \item{conLin}{a condensed linear model object, consisting of a list
   with components \code{"Xy"}, corresponding to a regression matrix
   (\code{X}) combined with a response vector (\code{y}), and
   \code{"logLik"}, corresponding to the log-likelihood of the
   underlying model.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function pre-multiples the \code{"Xy"} component of
  \code{conLin} by the transpose square-root factor(s) of the
  correlation matrix (matrices) associated with \code{object} and adds
  the log-likelihood contribution of \code{object}, given by
  \code{logLik(object)}, to the \code{"logLik"} component of
  \code{conLin}. 
}
\value{
  the recalculated condensed linear model object.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{This method function is only used inside model fitting functions,
    such as \code{lme} and \code{gls}, that allow correlated error terms.}
\seealso{\code{\link{corFactor}}, \code{\link{logLik.corStruct}}}
\keyword{models}

\eof
% $Id: recalc.modelStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{recalc.modelStruct}
%\title{Recalculate Condensed Linear Model for a \code{modelStruct} Object}
\title{Recalculate for a modelStruct Object}
\usage{
\method{recalc}{modelStruct}(object, conLin, \dots)
}
\alias{recalc.modelStruct}
\arguments{
 \item{object}{an object inheriting from class \code{modelStruct},
   representing a list of model components, such as \code{corStruct} and
   \code{varFunc} objects.}
 \item{conLin}{an optional  condensed linear model object, consisting of
   a list with components \code{"Xy"}, corresponding to a regression
   matrix (\code{X}) combined with a response vector (\code{y}), and 
   \code{"logLik"}, corresponding to the log-likelihood of the
   underlying model. Defaults to \code{attr(object, "conLin")}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function recalculates the condensed linear model object
  using each element of \code{object} sequentially from last to first.
}
\value{
  the recalculated condensed linear model object.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{This method function is generally only used inside model fitting
  functions, such as \code{lme} and \code{gls}, that allow model
  components, such as correlated error terms and variance functions.
}
\seealso{\code{\link{recalc.corStruct}}, \code{\link{recalc.reStruct}},
    \code{\link{recalc.varFunc}}}
\keyword{models}

\eof
% $Id: recalc.reStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{recalc.reStruct}
\title{Recalculate for an reStruct Object}
\usage{
\method{recalc}{reStruct}(object, conLin, \dots)
}
\alias{recalc.reStruct}
\arguments{
 \item{object}{an object inheriting from class \code{reStruct},
   representing a random effects structure and consisting of a list of
   \code{pdMat} objects.} 
 \item{conLin}{a condensed linear model object, consisting of a list
   with components \code{"Xy"}, corresponding to a regression matrix
   (\code{X}) combined with a response vector (\code{y}), and
   \code{"logLik"}, corresponding to the log-likelihood of the
   underlying model.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The log-likelihood, or restricted log-likelihood, of the
  Gaussian linear mixed-effects model represented by \code{object} and
  \code{conLin} (assuming spherical within-group covariance structure),
  evaluated at \code{coef(object)} is calculated and added to the
  \code{logLik} component of \code{conLin}. The \code{settings}
  attribute of \code{object} determines whether the log-likelihood, or
  the restricted log-likelihood, is to be calculated. The computational
  methods for the (restricted) log-likelihood calculations are described
  in Bates and Pinheiro (1998). 
}
\value{
  the condensed linear model with its \code{logLik} component updated.
}

\references{
  Bates, D.M. and Pinheiro, J.C. (1998) "Computational methods for
  multilevel models" available in PostScript or PDF formats at
  http://franz.stat.wisc.edu/pub/NLME/
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{reStruct}}, \code{\link{logLik}},
  \code{\link{lme}}} 

\keyword{models}

\eof
% $Id: recalc.varFunc.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{recalc.varFunc}
\title{Recalculate for varFunc Object}
\usage{
\method{recalc}{varFunc}(object, conLin, \dots)
}
\alias{recalc.varFunc}
\alias{recalc.varIdent}
\arguments{
 \item{object}{an object inheriting from class \code{varFunc},
   representing a variance function structure.} 
 \item{conLin}{a condensed linear model object, consisting of a list
   with components \code{"Xy"}, corresponding to a regression matrix
   (\code{X}) combined with a response vector (\code{y}), and
   \code{"logLik"}, corresponding to the log-likelihood of the
   underlying model.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function pre-multiples the \code{"Xy"} component of
  \code{conLin} by a diagonal matrix with diagonal elements given by the
  weights corresponding to the variance structure represented by
  \code{object}e and adds the log-likelihood contribution of
  \code{object}, given by \code{logLik(object)}, to the \code{"logLik"}
  component of \code{conLin}.  
}
\value{
  the recalculated condensed linear model object.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{This method function is only used inside model fitting functions,
    such as \code{lme} and \code{gls}, that allow heteroscedastic error
    terms.}
\seealso{\code{\link{varWeights}}, \code{\link{logLik.varFunc}}}
\keyword{models}

\eof
% $Id: residuals.gls.Rd,v 1.8.2.2 2003/05/17 22:04:29 bates Exp $
\name{residuals.gls}
\title{Extract gls Residuals}
\usage{
\method{residuals}{gls}(object, type, \dots)
}
\alias{residuals.gls}
\alias{residuals.gnls}
\arguments{
 \item{object}{an object inheriting from class \code{gls}, representing
   a generalized least squares fitted linear model, or from class
   \code{gnls}, representing a generalized nonlinear least squares
   fitted linear model.}
  \item{type}{an optional character string specifying the type of
    residuals to be used. If \code{"response"}, the "raw" residuals
    (observed - fitted) are used; else, if \code{"pearson"}, the
    standardized residuals (raw residuals divided by the corresponding
    standard errors) are used; else, if \code{"normalized"}, the
    normalized residuals (standardized residuals pre-multiplied by the
    inverse square-root factor of the estimated error correlation
    matrix) are used. Partial matching of arguments is used, so only the
    first character needs to be provided. Defaults to \code{"pearson"}.}
  \item{\dots}{some methods for this generic function require additional
    arguments.  None are used in this method.}
}
\description{
  The residuals for the linear model represented by \code{object}
  are extracted.
}
\value{
  a vector with the residuals for the linear model represented by
  \code{object}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gls}}}

\examples{
data(Ovary)
fm1 <- gls(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), Ovary,
           correlation = corAR1(form = ~ 1 | Mare))
residuals(fm1)
}
\keyword{models}

\eof
% $Id: residuals.glsStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{residuals.glsStruct}
\title{Calculate glsStruct Residuals}
\usage{
\method{residuals}{glsStruct}(object, glsFit, \dots)
}
\alias{residuals.glsStruct}
\arguments{
 \item{object}{an object inheriting from class \code{glsStruct},
   representing a list of linear model components, such as
   \code{corStruct} and \code{varFunc} objects.}
 \item{glsFit}{an optional list with components \code{logLik}   
   (log-likelihood), \code{beta} (coefficients), \code{sigma} (standard
   deviation for error term), \code{varBeta} (coefficients' covariance
   matrix), \code{fitted} (fitted values), and \code{residuals}
   (residuals). Defaults to \code{attr(object, "glsFit")}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The residuals for the linear model represented by \code{object}
  are extracted.
}
\value{
  a vector with the residuals for the linear model represented by
  \code{object}. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  This method function is primarily used inside \code{gls} and 
  \code{residuals.gls}.
}
\seealso{\code{\link{gls}}, \code{\link{residuals.gls}},
  \code{\link{fitted.glsStruct}} } 

\keyword{models}

\eof
% $Id: residuals.gnlsStruct.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{residuals.gnlsStruct}
\title{Calculate gnlsStruct Residuals}
\usage{
\method{residuals}{gnlsStruct}(object, \dots)
}
\alias{residuals.gnlsStruct}
\arguments{
 \item{object}{an object inheriting from class \code{gnlsStruct},
   representing a list of model components, such as
   \code{corStruct} and \code{varFunc} objects, and attributes
   specifying the underlying nonlinear model and the response variable.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The residuals for the nonlinear model represented by \code{object}
  are extracted.
}
\value{
  a vector with the residuals for the nonlinear model represented by
  \code{object}. 
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  This method function is primarily used inside \code{gnls} and 
  \code{residuals.gnls}.
}
\seealso{\code{\link{gnls}}, \code{\link{residuals.gnls}},
  \code{\link{fitted.gnlsStruct}} } 
\keyword{models}

\eof
% $Id: residuals.lmList.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{residuals.lmList}
\title{Extract lmList Residuals}
\usage{
\method{residuals}{lmList}(object, type, subset, asList, \dots)
}
\alias{residuals.lmList}
\arguments{
 \item{object}{an object inheriting from class \code{lmList}, representing
   a list of \code{lm} objects with a common model.
 }
 \item{subset}{an optional character or integer vector naming the
   \code{lm} components of \code{object} from which the residuals 
   are to be extracted. Default is \code{NULL}, in which case all
   components are used.
 }
 \item{type}{an optional character string specifying the type of
   residuals to be extracted. Options include \code{"response"} for the
   "raw" residuals (observed - fitted), \code{"pearson"} for the
   standardized residuals (raw residuals divided by the estimated
   residual standard error) using different standard errors for each
   \code{lm} fit, and \code{"pooled.pearson"} for the standardized
   residuals using a pooled estimate of the residual standard
   error. Partial matching of arguments is used, so only the first 
   character needs to be provided. Defaults to \code{"response"}.
 }
 \item{asList}{an optional logical value. If \code{TRUE}, the returned
   object is a list with the residuals split by groups; else the
   returned value is a vector. Defaults to \code{FALSE}.
 }
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The residuals are extracted from each \code{lm} component of
  \code{object} and arranged into a list with as many components as
  \code{object}, or combined into a single vector.
}
\value{
  a list with components given by the residuals of each \code{lm}
  component of \code{object}, or a vector with the residuals for all
  \code{lm} components of \code{object}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lmList}}, \code{\link{fitted.lmList}}}

\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age | Subject, Orthodont)
residuals(fm1)
}
\keyword{models}

\eof
% $Id: residuals.lme.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{residuals.lme}
\title{Extract lme Residuals}
\usage{
\method{residuals}{lme}(object, level, type, asList, \dots)
}
\alias{residuals.lme}
\arguments{
  \item{object}{an object inheriting from class \code{lme}, representing
    a fitted linear mixed-effects model.}
  \item{level}{an optional integer vector giving the level(s) of grouping
    to be used in extracting the residuals from \code{object}. Level
    values increase from outermost to innermost grouping, with
    level zero corresponding to the population residuals. Defaults to
    the highest or innermost level of grouping.}
  \item{type}{an optional character string specifying the type of
    residuals to be used. If \code{"response"}, the "raw" residuals
    (observed - fitted) are used; else, if \code{"pearson"}, the
    standardized residuals (raw residuals divided by the corresponding
    standard errors) are used; else, if \code{"normalized"}, the
    normalized residuals (standardized residuals pre-multiplied by the
    inverse square-root factor of the estimated error correlation
    matrix) are used. Partial matching of arguments is used, so only the
    first character needs to be provided. Defaults to \code{"pearson"}.} 
  \item{asList}{an optional logical value. If \code{TRUE} and a single
    value is given in \code{level}, the returned object is a list with
    the residuals split by groups; else the returned value is
    either a vector or a data frame, according to the length of
    \code{level}. Defaults to \code{FALSE}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The residuals at level \eqn{i} are obtained by subtracting the fitted
  levels at that level from the response vector (and dividing
  by the estimated within-group standard error, if
  \code{type="pearson"}). The fitted values at level \eqn{i} are obtained
  by adding together the population fitted values (based only on the
  fixed effects estimates) and the estimated contributions of the random
  effects to the fitted values at grouping levels less or equal to
  \eqn{i}.  
}
\value{
  if a single level of grouping is specified in \code{level}, the
  returned value is either a list with the residuals split by groups
  (\code{asList = TRUE}) or a vector with the residuals
  (\code{asList = FALSE}); else, when multiple grouping levels are
  specified in \code{level}, the returned object is a data frame with
  columns given by the residuals at different levels and the grouping
  factors.  
}
\references{
  Bates, D.M. and Pinheiro, J.C. (1998) "Computational methods for
  multilevel models" available in PostScript or PDF formats at
  http://franz.stat.wisc.edu/pub/NLME/
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lme}}, \code{\link{fitted.lme}} }

\examples{
data(Orthodont)
fm1 <- lme(distance ~ age + Sex, data = Orthodont, random = ~ 1)
residuals(fm1, level = 0:1)
}
\keyword{models}

\eof
% $Id: residuals.lmeStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{residuals.lmeStruct}
\title{Calculate lmeStruct Residuals}
\usage{
\method{residuals}{lmeStruct}(object, level, conLin, lmeFit, \dots)
}
\alias{residuals.lmeStruct}
\arguments{
 \item{object}{an object inheriting from class \code{lmeStruct},
   representing a list of linear mixed-effects model components, such as
   \code{reStruct}, \code{corStruct}, and \code{varFunc} objects.}
 \item{level}{an optional integer vector giving the level(s) of grouping
   to be used in extracting the residuals from \code{object}. Level
   values increase from outermost to innermost grouping, with
   level zero corresponding to the population fitted values. Defaults to
   the highest or innermost level of grouping.} 
 \item{conLin}{an optional condensed linear model object, consisting of
   a list with components \code{"Xy"}, corresponding to a regression
   matrix (\code{X}) combined with a response vector (\code{y}), and 
   \code{"logLik"}, corresponding to the log-likelihood of the
   underlying lme model. Defaults to \code{attr(object, "conLin")}.}
 \item{lmeFit}{an optional list with components \code{beta} and \code{b}
   containing respectively the fixed effects estimates and the random
   effects estimates to be used to calculate the residuals. Defaults
   to \code{attr(object, "lmeFit")}.}
 \item{\dots}{some methods for this generic accept optional arguments.}
}
\description{
  The residuals at level \eqn{i} are obtained by subtracting the fitted
  values at that level from the response vector. The fitted values at
  level \eqn{i} are obtained by adding together the population fitted
  values (based only on the fixed effects estimates) and the estimated
  contributions of the random effects to the fitted values at grouping
  levels less or equal to \eqn{i}.  
}
\value{
  if a single level of grouping is specified in \code{level},
  the returned value is a vector with the residuals at the desired
  level; else, when multiple grouping levels are specified in
  \code{level}, the returned object is a matrix with 
  columns given by the residuals at different levels. 
}
\references{
  Bates, D.M. and Pinheiro, J.C. (1998) "Computational methods for
  multilevel models" available in PostScript or PDF formats at
  http://franz.stat.wisc.edu/pub/NLME/
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  This method function is primarily used within the \code{lme}
  function. 
}
\seealso{\code{\link{lme}}, \code{\link{residuals.lme}},
  \code{\link{fitted.lmeStruct}} } 
\keyword{models}

\eof
% $Id: residuals.nlmeStruct.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{residuals.nlmeStruct}
\title{Calculate nlmeStruct Residuals}
\usage{
\method{residuals}{nlmeStruct}(object, level, conLin, \dots)
}
\alias{residuals.nlmeStruct}
\arguments{
 \item{object}{an object inheriting from class \code{nlmeStruct},
   representing a list of mixed-effects model components, such as
   \code{reStruct}, \code{corStruct}, and \code{varFunc} objects.}
 \item{level}{an optional integer vector giving the level(s) of grouping
   to be used in extracting the residuals from \code{object}. Level
   values increase from outermost to innermost grouping, with
   level zero corresponding to the population fitted values. Defaults to
   the highest or innermost level of grouping.} 
 \item{conLin}{an optional condensed linear model object, consisting of
   a list with components \code{"Xy"}, corresponding to a regression
   matrix (\code{X}) combined with a response vector (\code{y}), and 
   \code{"logLik"}, corresponding to the log-likelihood of the
   underlying nlme model. Defaults to \code{attr(object, "conLin")}.}
 \item{\dots}{optional arguments to the residuals generic.  Not
   used.}
}
\description{
  The residuals at level \eqn{i} are obtained by subtracting the fitted
  values at that level from the response vector. The fitted values at
  level \eqn{i} are obtained by adding together the contributions from
  the estimated fixed effects and the estimated random effects at levels
  less or equal to \eqn{i} and evaluating the model function at the
  resulting estimated parameters. 
}
\value{
  if a single level of grouping is specified in \code{level},
  the returned value is a vector with the residuals at the desired
  level; else, when multiple grouping levels are specified in
  \code{level}, the returned object is a matrix with 
  columns given by the residuals at different levels. 
}
\references{
  Bates, D.M. and Pinheiro, J.C. (1998) "Computational methods for
  multilevel models" available in PostScript or PDF formats at
  http://nlme.stat.wisc.edu
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  This method function is primarily used within the \code{nlme}
  function. 
}
\seealso{\code{\link{nlme}}, \code{\link{fitted.nlmeStruct}} } 
\keyword{models}

\eof
% $Id: simulate.lme.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{simulate.lme}
\title{simulate lme models}
\usage{
simulate.lme(m1, m2, Random.seed, method, nsim, niterEM, useGen)
}
\alias{simulate.lme}
\alias{plot.simulate.lme}
\alias{print.simulate.lme}
\arguments{
  \item{m1}{
    an object inheriting from class \code{lme}, representing a fitted
    linear mixed-effects model, or a list containing an lme model
    specification.  If given as a list, it should contain
    components \code{fixed}, \code{data}, and \code{random}
    with values suitable for a call to \code{\link{lme}}. This argument
    defines the null model.
  }
  \item{m2}{
    an \code{lme} object, or a list, like \code{m1} containing a second
    lme model specification. This argument defines the alternative model.
    If given as a list, only those parts of the specification that
    change between model \code{m1} and \code{m2} need to be specified.
  }
  \item{Random.seed}{
    an optional vector to seed the random number generator so as to
    reproduce a simulation.  This vector should be the same form as the
    \code{.Random.seed} object.
  }
  \item{method}{
   an optional character array.  If it includes \code{"REML"} the models
   are fit by maximizing the restricted log-likelihood. If it includes
   \code{"ML"} the log-likelihood is maximized.  Defaults to
   \code{c("REML", "ML")}, in which case both methods are used.
  }
  \item{nsim}{
    an optional positive integer specifying the number of simulations to
    perform.  Defaults to 1000.
  }
  \item{niterEM}{
    an optional integer vector of length 2 giving the number of
    iterations of the EM algorithm to apply when fitting the \code{m1}
    and \code{m2} to each simulated set of data. Defaults to
    \code{c(40,200)}. 
  }
  \item{useGen}{
    an optional logical value. If \code{TRUE}, numerical derivatives are
    used to obtain the gradient and the Hessian of the log-likelihood in
    the optimization algorithm in the \code{ms} function. If
    \code{FALSE}, the default algorithm in \code{ms} for functions that
    do not incorporate gradient and Hessian attributes is used. Default
    depends on the \code{pdMat} classes used in \code{m1} and \code{m2}:
    if both are standard classes (see \code{\link{pdClasses}}) then
    defaults to \code{TRUE}, otherwise defaults to \code{FALSE}.
 }
}
\description{
  The model \code{m1} is fit to the data.  Using
  the fitted values of the parameters, \code{nsim} new data vectors from
  this model are simulated.  Both \code{m1} and \code{m2} are fit by
  maximum likelihood (ML) and/or by restricted maximum likelihood (REML)
  to each of the simulated data vectors.
}
\value{
  an object of class \code{simulate.lme} with components \code{null} and
  \code{alt}.  Each of these has components \code{ML} and/or \code{REML}
  which are matrices.  An attribute called \code{Random.seed} contains
  the seed that was used for the random number generator.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{lme}}
}
\examples{
data(Orthodont)
\dontrun{
orthSim <-
 simulate.lme(m1 = list(fixed = distance ~ age, data = Orthodont,
                        random = ~ 1 | Subject),
              m2 = list(random = ~ age | Subject))
}
}
\keyword{models}

\eof
% $Id: solve.pdMat.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{solve.pdMat}
\title{Calculate Inverse of a Positive-Definite Matrix}
\usage{
\method{solve}{pdMat}(a, b, \dots)
}
\alias{solve.pdMat}
\alias{solve.pdBlocked}
\alias{solve.pdDiag}
\alias{solve.pdIdent}
\alias{solve.pdLogChol}
\alias{solve.pdNatural}
\alias{solve.pdSymm}
\arguments{
 \item{a}{an object inheriting from class \code{pdMat}, representing
   a positive definite matrix.} 
 \item{b}{this argument is only included for consistency with the generic
   function and is not used in this method function.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The positive-definite matrix represented by \code{a} is inverted and
  assigned to \code{a}.
}
\value{
  a \code{pdMat} object similar to \code{a}, but with coefficients
  corresponding to the inverse of the positive-definite matrix
  represented by \code{a}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}


\seealso{\code{\link{pdMat}}}

\examples{
pd1 <- pdCompSymm(3 * diag(3) + 1)
solve(pd1)
}
\keyword{models}

\eof
% $Id: solve.reStruct.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{solve.reStruct}
\title{Apply Solve to an reStruct Object}
\usage{
\method{solve}{reStruct}(a, b, \dots)
}
\alias{solve.reStruct}
\arguments{
 \item{a}{an object inheriting from class \code{reStruct},
   representing a random effects structure and consisting of a list of
   \code{pdMat} objects.} 
 \item{b}{this argument is only included for consistency with the
   generic function and is not used in this method function.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  \code{Solve} is applied to each \code{pdMat} component of \code{a},
  which results in inverting the positive-definite matrices they
  represent. 
}
\value{
  an \code{reStruct} object similar to \code{a}, but with the
  \code{pdMat} components representing the inverses of the
  matrices represented by the components of \code{a}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{solve.pdMat}}, \code{\link{reStruct}}}

\examples{
rs1 <- reStruct(list(A = pdSymm(diag(1:3), form = ~Score),
  B = pdDiag(2 * diag(4), form = ~Educ)))
solve(rs1)
}
\keyword{models}

\eof
% $Id: splitFormula.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{splitFormula}
\title{Split a Formula}
\usage{
splitFormula(form, sep)
}
\alias{splitFormula}
\arguments{
 \item{form}{a \code{formula} object.}
 \item{sep}{an optional character string specifying the separator to be
   used for splitting the formula. Defaults to \code{"/"}. }
}
\description{
  Splits the right hand side of \code{form} into a list of subformulas
  according to the presence of \code{sep}. The left hand side of
  \code{form}, if present, will be ignored. The length of the returned
  list will be equal to the number of occurrences of \code{sep} in
  \code{form} plus one. 
}
\value{
  a list of formulas, corresponding to the split of \code{form}
  according to \code{sep}.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{formula}}}

\examples{
splitFormula(~ g1/g2/g3)
}
\keyword{models}

\eof
\name{summary.corStruct}
\title{Summarize a corStruct Object}
\usage{
\method{summary}{corStruct}(object, structName, \dots)
}
\alias{summary.corStruct}
\alias{summary.corAR1}
\alias{summary.corARMA}
\alias{summary.corCAR1}
\alias{summary.corCompSymm}
\alias{summary.corExp}
\alias{summary.corGaus}
\alias{summary.corIdent}
\alias{summary.corLin}
\alias{summary.corNatural}
\alias{summary.corRatio}
\alias{summary.corSpher}
\alias{summary.corSymm}

\arguments{
 \item{object}{an object inheriting from class \code{corStruct},
   representing a correlation structure.}
 \item{structName}{an optional character string defining the type of
   correlation structure associated with \code{object}, to be used in
   the \code{print.summary} method. Defaults to
   \code{class(object)[1]}.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function prepares \code{object} to be printed using the
  \code{print.summary} method, by changing its class and adding a
  \code{structName} attribute to it.
}
\value{
  an object identical to \code{object}, but with its class changed to
  \code{summary.corStruct} and an additional attribute
  \code{structName}. The returned value inherits from the same classes
  as \code{object}.
}
\author{Jose Pinheiro and Douglas Bates }

%\seealso{\code{\link{print.summary.corStruct}}}

\examples{
cs1 <- corAR1(0.2)
summary(cs1)
}
\keyword{models}

\eof
% $Id: summary.gls.Rd,v 1.8 2002/03/05 14:59:39 bates Exp $
\name{summary.gls}
\title{Summarize a gls Object}
\usage{
\method{summary}{gls}(object, verbose, \dots)
}
\alias{summary.gls}
\arguments{
 \item{object}{an object inheriting from class \code{gls}, representing
   a generalized least squares fitted linear model.}
 \item{verbose}{an optional logical value used to control the amount of
   output in the \code{print.summary.gls} method. Defaults to
   \code{FALSE}.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  Additional information about the linear model fit represented
  by \code{object} is extracted and included as components of
  \code{object}. The returned object is suitable for printing with the
  \code{print.summary.gls} method.
}
\value{
  an object inheriting from class \code{summary.gls} with all components
  included in \code{object} (see \code{\link{glsObject}} for a full
  description of the components) plus the following components:
  \item{corBeta}{approximate correlation matrix for the coefficients
    estimates}
  \item{tTable}{a data frame with columns \code{Value},
    \code{Std. Error}, \code{t-value}, and \code{p-value} representing
    respectively the coefficients estimates, their approximate standard
    errors, the ratios between the estimates and their standard errors,
    and the associated p-value under a \eqn{t} approximation. Rows
    correspond to the different coefficients.}
  \item{residuals}{if more than five observations are used in the
    \code{gls} fit, a vector with the minimum, first quartile, median, third
    quartile, and maximum of the residuals distribution; else the
    residuals.} 
  \item{AIC}{the Akaike Information Criterion corresponding to
    \code{object}.} 
  \item{BIC}{the Bayesian Information Criterion corresponding to
    \code{object}.} 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{gls}}, \code{\link{AIC}}, \code{\link{BIC}},
  \code{print.summary.gls}}

\examples{
data(Ovary)
fm1 <- gls(follicles ~ sin(2*pi*Time) + cos(2*pi*Time), Ovary,
           correlation = corAR1(form = ~ 1 | Mare))
summary(fm1)
}
\keyword{models}

\eof
% $Id: summary.lmList.Rd,v 1.7.2.1 2003/05/17 22:06:46 bates Exp $
\name{summary.lmList}
\title{Summarize an lmList Object}
\usage{
\method{summary}{lmList}(object, pool, \dots)
}
\alias{summary.lmList}
\arguments{
 \item{object}{an object inheriting from class \code{lmList}, representing
   a list of \code{lm} fitted objects.
 }
 \item{pool}{an optional logical value indicating whether a pooled
   estimate of the residual standard error should be used. Default is
   \code{attr(object, "pool")}.
 }
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  The \code{summary.lm} method is applied to each \code{lm} component of
  \code{object} to produce summary information on the individual fits,
  which is organized into a list of summary statistics. The returned
  object is suitable for printing with the \code{print.summary.lmList}
  method. 
}
\value{
  a list with summary statistics obtained by applying \code{summary.lm}
  to the elements of \code{object}, inheriting from class
  \code{summary.lmList}. The components of \code{value} are:
  \item{call}{a list containing an image of the \code{lmList} call that
    produced \code{object}.
  }
  \item{coefficients}{a three dimensional array with summary information
    on the \code{lm} coefficients. The first dimension corresponds to
    the names of the \code{object} components, the second dimension is
    given by   \code{"Value"}, \code{"Std. Error"}, \code{"t value"},
    and \code{"Pr(>|t|)"}, corresponding, respectively, to the
    coefficient estimates and their associated standard errors,
    t-values, and p-values. The third dimension is given by the
    coefficients names.
  }
  \item{correlation}{a three dimensional array with the 
    correlations between the individual \code{lm} coefficient
    estimates. The first dimension corresponds to the names of the
    \code{object} components. The third dimension is given by the
    coefficients names. For each coefficient, the rows of the associated
    array give the correlations between that coefficient and the
    remaining coefficients, by \code{lm} component.
  }
  \item{cov.unscaled}{a three dimensional array with the unscaled
    variances/covariances for the individual \code{lm} coefficient
    estimates (giving the estimated variance/covariance for the
    coefficients, when multiplied by the estimated residual errors). The
    first dimension corresponds to the names of the \code{object}
    components. The third dimension is given by the
    coefficients names. For each coefficient, the rows of the associated
    array give the unscaled covariances between that coefficient and the
    remaining coefficients, by \code{lm} component.
  }
  \item{df}{an array with the number of degrees of freedom for the model
    and for residuals, for each \code{lm} component.
  }
  \item{df.residual}{the total number of degrees of freedom for
    residuals, corresponding to the sum of residuals df of all \code{lm}
    components.
  }
  \item{fstatistics}{an array with the F test statistics and
    corresponding degrees of freedom, for each \code{lm} component.
  }
  \item{pool}{the value of the \code{pool} argument to the function.
  }
  \item{r.squared}{a vector with the multiple R-squared statistics for
    each \code{lm} component.
  }
  \item{residuals}{a list with components given by the residuals from
    individual \code{lm} fits.
  }
  \item{RSE}{the pooled estimate of the residual standard error.}
  \item{sigma}{a vector with the residual standard error estimates for
    the individual \code{lm} fits.
  }
  \item{terms}{the terms object used in fitting the individual \code{lm}
    components.
  }
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lmList}}}

\examples{
data(Orthodont)
fm1 <- lmList(distance ~ age | Subject, Orthodont)
summary(fm1)
}
\keyword{models}

\eof
% $Id: summary.lme.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{summary.lme}
\title{Summarize an lme Object}
\usage{
\method{summary}{lme}(object, adjustSigma, verbose, \dots)
}
\alias{summary.lme}
\arguments{
 \item{object}{an object inheriting from class \code{lme}, representing
   a fitted linear mixed-effects model.}
 \item{adjustSigma}{an optional logical value. If \code{TRUE} and the
   estimation method used to obtain \code{object} was maximum
   likelihood, the residual standard error is multiplied by
   \eqn{\sqrt{n_{obs}/(n_{obs} - n_{par})}}{sqrt(nobs/(nobs - npar))},
   converting it to a REML-like estimate. This argument is only used
   when a single fitted object is passed to the function. Default is
   \code{TRUE}.} 
 \item{verbose}{an optional logical value used to control the amount of
   output in the \code{print.summary.lme} method. Defaults to
   \code{FALSE}.} 
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  Additional information about the linear mixed-effects fit represented
  by \code{object} is extracted and included as components of
  \code{object}. The returned object is suitable for printing with the
  \code{print.summary.lme} method.
}
\value{
  an object inheriting from class \code{summary.lme} with all components
  included in \code{object} (see \code{\link{lmeObject}} for a full
  description of the components) plus the following components:
  \item{corFixed}{approximate correlation matrix for the fixed effects
    estimates }
  \item{tTable}{a data frame with columns \code{Value},
    \code{Std. Error}, \code{DF}, \code{t-value}, and \code{p-value} representing
    respectively the fixed effects estimates, their approximate standard
    errors, the denominator degrees of freedom, the ratios between the
    estimates and their standard errors, 
    and the associated p-value from a t distribution. Rows
    correspond to the different fixed effects.}
  \item{residuals}{if more than five observations are used in the
    \code{lme} fit, a vector with the minimum, first quartile, median, third
    quartile, and maximum of the innermost grouping level residuals
    distribution; else the innermost grouping level residuals.}
  \item{AIC}{the Akaike Information Criterion corresponding to
    \code{object}.} 
  \item{BIC}{the Bayesian Information Criterion corresponding to
    \code{object}.} 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{lme}}, \code{\link{AIC}}, \code{\link{BIC}},
  \code{print.summary.lme}}

\examples{
data(Orthodont)
fm1 <- lme(distance ~ age, Orthodont, random = ~ age | Subject)
summary(fm1)
}
\keyword{models}

\eof
% $Id: summary.modelStruct.Rd,v 1.8.2.1 2003/05/17 22:07:48 bates Exp $
\name{summary.modelStruct}
\alias{summary.modelStruct}
\alias{summary.reStruct}
\title{Summarize a modelStruct Object}
\usage{
\method{summary}{modelStruct}(object, \dots)
}
\arguments{
  \item{object}{an object inheriting from class \code{modelStruct},
    representing a list of model components, such as \code{reStruct},
    \code{corStruct} and \code{varFunc} objects.
  }
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  This method function applies \code{summary} to each element of
  \code{object}. 
}
\value{
  a list with elements given by the summarized components of
  \code{object}. The returned value is of class
  \code{summary.modelStruct}, also inheriting from the same classes as
  \code{object}.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
%\seealso{\code{\link{print.summary.modelStruct}}}
\examples{
lms1 <- lmeStruct(reStruct = reStruct(pdDiag(diag(2), ~age)),
   corStruct = corAR1(0.3))
summary(lms1)
}
\keyword{models}

\eof
% $Id: summary.nlsList.Rd,v 1.7.2.1 2003/03/25 15:15:35 bates Exp $
\name{summary.nlsList}
\title{Summarize an nlsList Object}
\usage{
\method{summary}{nlsList}(object, \dots)
}
\alias{summary.nlsList}
\arguments{
 \item{object}{an object inheriting from class \code{nlsList},
   representing a list of \code{nls} fitted objects.
 }
 \item{\dots}{optional arguments to the \code{summary.lmList} method.
   One such optional argument is \code{pool}, a logical value indicating
   whether a pooled estimate of the residual standard error should be
   used. Default is \code{attr(object, "pool")}.
 }
}
\description{
  The \code{summary} function is applied to each \code{nls} component of
  \code{object} to produce summary information on the individual fits,
  which is organized into a list of summary statistics. The returned
  object is suitable for printing with the \code{print.summary.nlsList}
  method. 
}
\value{
  a list with summary statistics obtained by applying \code{summary}
  to the elements of \code{object}, inheriting from class
  \code{summary.nlsList}. The components of \code{value} are:
  \item{call}{a list containing an image of the \code{nlsList} call that
    produced \code{object}.
  }
  \item{parameters}{a three dimensional array with summary information
    on the \code{nls} coefficients. The first dimension corresponds to
    the names of the \code{object} components, the second dimension is
    given by   \code{"Value"}, \code{"Std. Error"}, \code{"t value"},
    and \code{"Pr(>|t|)"}, corresponding, respectively, to the
    coefficient estimates and their associated standard errors,
    t-values, and p-values. The third dimension is given by the
    coefficients names.
  }
  \item{correlation}{a three dimensional array with the 
    correlations between the individual \code{nls} coefficient
    estimates. The first dimension corresponds to the names of the
    \code{object} components. The third dimension is given by the
    coefficients names. For each coefficient, the rows of the associated
    array give the correlations between that coefficient and the
    remaining coefficients, by \code{nls} component.
  }
  \item{cov.unscaled}{a three dimensional array with the unscaled
    variances/covariances for the individual \code{lm} coefficient
    estimates (giving the estimated variance/covariance for the
    coefficients, when multiplied by the estimated residual errors). The
    first dimension corresponds to the names of the \code{object}
    components. The third dimension is given by the
    coefficients names. For each coefficient, the rows of the associated
    array give the unscaled covariances between that coefficient and the
    remaining coefficients, by \code{nls} component.
  }
  \item{df}{an array with the number of degrees of freedom for the model
    and for residuals, for each \code{nls} component.
  }
  \item{df.residual}{the total number of degrees of freedom for
    residuals, corresponding to the sum of residuals df of all \code{nls}
    components.
  }
  \item{pool}{the value of the \code{pool} argument to the function.
  }
  \item{RSE}{the pooled estimate of the residual standard error.}
  \item{sigma}{a vector with the residual standard error estimates for
    the individual \code{lm} fits.
  }
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{nlsList}}, \code{\link{summary}}}

\examples{
data(Loblolly)
fm1 <- nlsList(SSasymp, Loblolly)
summary(fm1)
}
\keyword{models}

\eof
% $Id: summary.pdMat.Rd,v 1.8 2002/03/05 14:59:39 bates Exp $
\name{summary.pdMat}
\title{Summarize a pdMat Object}
\usage{
\method{summary}{pdMat}(object, structName, noCorrelation, \dots)
}
\alias{summary.pdMat}
\alias{summary.pdBlocked}
\alias{summary.pdCompSymm}
\alias{summary.pdDiag}
\alias{summary.pdIdent}
\alias{summary.pdNatural}
\alias{summary.pdSymm}
\alias{summary.pdLogChol}
\arguments{
 \item{object}{an object inheriting from class \code{pdMat}, representing
   a positive definite matrix.} 
 \item{structName}{an optional character string with a description of
   the \code{pdMat} class. Default depends on the method function:
   \code{"Blocked"} for \code{pdBlocked},
   \code{"Compound Symmetry"} for \code{pdCompSymm}, \code{"Diagonal"}
   for \code{pdDiag}, \code{"Multiple of an Identity"} for
   \code{pdIdent}, \code{"General Positive-Definite, Natural
     Parametrization"} for \code{pdNatural}, \code{"General
     Positive-Definite"} for \code{pdSymm}, and
     \code{data.class(object)} for \code{pdMat}.}
 \item{noCorrelation}{an optional logical value indicating whether
   correlations are to be printed in \code{print.summary.pdMat}. Default
   depends on the method function: \code{FALSE} for \code{pdDiag} and
   \code{pdIdent}, and \code{TRUE} for all other classes.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  Attributes \code{structName} and \code{noCorrelation}, with the values
  of the corresponding arguments to the method function, are appended to
  \code{object} and its class is changed to \code{summary.pdMat}. 
}
\value{
  an object similar to \code{object}, with additional attributes
  \code{structName} and \code{noCorrelation}, inheriting from class
  \code{summary.pdMat}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{print.summary.pdMat}}, \code{\link{pdMat}}}

\examples{
summary(pdSymm(diag(4)))
}
\keyword{models}

\eof
% $Id: summary.varFunc.Rd,v 1.7.2.2 2003/05/17 22:08:34 bates Exp $
\name{summary.varFunc}
\title{Summarize varFunc Object}
\usage{
\method{summary}{varFunc}(object, structName, \dots)
}
\alias{summary.varFunc}
\alias{summary.varComb}
\alias{summary.varConstPower}
\alias{summary.varExp}
\alias{summary.varFixed}
\alias{summary.varIdent}
\alias{summary.varPower}

\arguments{
 \item{object}{an object inheriting from class \code{varFunc},
   representing a variance function structure.} 
 \item{structName}{an optional character string with a description of
   the \code{varFunc} class. Default depends on the method function:
 \code{"Combination of variance functions"} for \code{varComb},
 \code{"Constant plus power of covariate"} for \code{varConstPower},
 \code{"Exponential of variance covariate"} for \code{varExp},
 \code{"Different standard deviations per stratum"} for \code{varIdent},
 \code{"Power of variance covariate"} for \code{varPower}, and
 \code{data.class(object)} for \code{varFunc}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  A \code{structName} attribute, with the value of corresponding
  argument, is appended to \code{object} and its class is changed to
  \code{summary.varFunc}.  
}
\value{
  an object similar to \code{object}, with an additional attribute
  \code{structName}, inheriting from class \code{summary.varFunc}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
%\seealso{\code{\link{print.summary.varFunc}}}
\examples{
data(Orthodont)
vf1 <- varPower(0.3, form = ~age)
vf1 <- Initialize(vf1, Orthodont)
summary(vf1)
}
\keyword{models}

\eof
% $Id: update.modelStruct.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{update.modelStruct}
\title{Update a modelStruct Object}
\usage{
\method{update}{modelStruct}(object, data, \dots)
}
\alias{update.modelStruct}
\alias{update.corStruct}
\alias{update.reStruct}
\arguments{
  \item{object}{an object inheriting from class \code{modelStruct},
    representing a list of model components, such as \code{corStruct} and
    \code{varFunc} objects.}
  \item{data}{a data frame in which to evaluate the variables needed for
    updating the elements of \code{object}.}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
    This method function updates each element of \code{object}, allowing
    the access to \code{data}.
}
\value{
    an object similar to \code{object} (same class, length, and names),
    but with updated elements.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{This method function is primarily used within model fitting
  functions, such as \code{lme} and \code{gls}, that allow model
  components such as variance functions.}
\keyword{models}

\eof
% $Id: update.varFunc.Rd,v 1.7 2002/03/05 14:59:39 bates Exp $
\name{update.varFunc}
\title{Update varFunc Object}
\usage{
\method{update}{varFunc}(object, data, \dots)
}
\alias{update.varExp}
\alias{update.varFunc}
\alias{update.varComb}
\alias{update.varConstPower}
\alias{update.varExpon}
\alias{update.varPower}
\arguments{
  \item{object}{an object inheriting from class \code{varFunc},
    representing a variance function structure.} 
  \item{data}{a list with a component named \code{"."} with the current
    version of the fitted object (from which fitted values, coefficients,
    and residuals can be extracted) and, if necessary, other variables
    used to evaluate the variance covariate(s).}
 \item{\dots}{some methods for this generic require additional
    arguments.  None are used in this method.} 
}
\description{
  If the \code{formula(object)} includes a \code{"."} term, representing
  a fitted object, the variance covariate needs to be updated upon
  completion of an optimization cycle (in which the variance function
  weights are kept fixed). This method function allows a reevaluation of
  the variance covariate using the current fitted object and,
  optionally, other variables in the original data.
}
\value{
  if \code{formula(object)} includes a \code{"."} term, an
  \code{varFunc} object similar to \code{object}, but with the 
  variance covariate reevaluated at the current fitted object value;
  else \code{object} is returned unchanged.
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\seealso{\code{\link{needUpdate}}, \code{\link{covariate<-.varFunc}}}
\keyword{models}

\eof
% $Id: varClasses.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{varClasses}
\title{Variance Function Classes}
\alias{varClasses}

\description{
  Standard classes of variance function structures (\code{varFunc}) 
  available in the \code{nlme} library. Covariates included in the
  variance function, denoted by variance covariates, may involve
  functions of the fitted model object, such as the fitted values and
  the residuals. Different coefficients may be assigned to the levels of
  a classification factor.
}
\value{
  Available standard classes ():
  \item{varExp}{exponential of a variance covariate.}
  \item{varPower}{power of a variance covariate.}
  \item{varConstPower}{constant plus power of a variance covariate.}
  \item{varIdent}{constant variance(s), generally used to allow
    different variances according to the levels of a classification
    factor.} 
  \item{varFixed}{fixed weights, determined by a variance covariate.}
  \item{varComb}{combination of variance functions.}
}
\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}
\note{
  Users may define their own \code{varFunc} classes by specifying a
  \code{constructor} function and, at a minimum, methods for the
  functions \code{coef}, \code{coef<-}, and \code{initialize}. For
  examples of these functions, see the methods for class
  \code{varPower}. 
}
\seealso{\code{\link{varExp}}, \code{\link{varPower}},
  \code{\link{varConstPower}}, \code{\link{varIdent}},
  \code{\link{varFixed}}, \code{\link{varComb}}
}
\keyword{models}

\eof
% $Id: varComb.Rd,v 1.5.2.1 2003/04/17 22:28:45 bates Exp $
\name{varComb}
\title{Combination of Variance Functions}
\usage{
varComb(\dots)
}
\alias{varComb}
\arguments{
 \item{\dots}{objects inheriting from class \code{varFunc} representing
   variance function structures.}
}
\description{
  This function is a constructor for the \code{varComb} class,
  representing a combination of variance functions. The corresponding
  variance function is equal to the product of the variance functions of
  the \code{varFunc} objects listed in \code{\dots}.
}
\value{
  a \code{varComb} object representing a combination of variance
  functions, also inheriting from class \code{varFunc}.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{varWeights.varComb}}, \code{\link{coef.varComb}}}

\examples{
vf1 <- varComb(varIdent(form = ~1|Sex), varPower())
}
\keyword{models}

\eof
% $Id: varConstPower.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{varConstPower}
\title{Constant Plus Power Variance Function}
\usage{
varConstPower(const, power, form, fixed)
}
\alias{varConstPower}
\arguments{
  \item{const, power}{optional numeric vectors, or lists of numeric
    values, with, respectively, the coefficients for the constant 
    and the power terms. Both arguments must have length one, unless a
    grouping factor is specified in \code{form}. If either argument has
    length greater than one, it must have names which identify its
    elements to the levels of the grouping factor defined in
    \code{form}. If a grouping factor is present in 
    \code{form} and the argument has length one, its value will be
    assigned to all grouping levels. Only positive values are allowed
    for \code{const}. Default is \code{numeric(0)}, which
    results in a vector of zeros of appropriate length being assigned to
    the coefficients when \code{object} is initialized (corresponding
    to constant variance equal to one).}
  \item{form}{an optional one-sided formula of the form \code{~ v}, or
    \code{~ v | g}, specifying a variance covariate \code{v} and,
    optionally, a grouping factor \code{g} for the coefficients. The
    variance covariate must evaluate to a numeric vector and may involve
    expressions using \code{"."}, representing  a fitted model object
    from which fitted values (\code{fitted(.)}) and residuals
    (\code{resid(.)}) can be extracted (this allows the variance
    covariate to be updated during the optimization of an object
    function). When a grouping factor is present in \code{form},
    a different coefficient value is used for each of its
    levels. Several grouping variables may be
    simultaneously specified, separated by the \code{*} operator, as
    in \code{~ v | g1 * g2 * g3}. In this case, the levels of each
    grouping variable are pasted together and the resulting factor is
    used to group the observations. Defaults to \code{~ fitted(.)}
    representing a variance covariate given by the fitted values of a
    fitted model object and no grouping factor. }
  \item{fixed}{an optional list with components \code{const} and/or
    \code{power}, consisting of numeric vectors, or lists of numeric
    values, specifying the values at which some or all of the
    coefficients in the variance function should be fixed. If a grouping
    factor is specified in \code{form}, the components of \code{fixed}
    must have names identifying which coefficients are to be
    fixed. Coefficients included in \code{fixed} are not allowed to vary
    during the optimization of an objective function. Defaults to
    \code{NULL}, corresponding to no fixed coefficients.}
}
\description{
  This function is a constructor for the \code{varConstPower} class,
  representing a constant plus power variance function
  structure. Letting \eqn{v} denote the variance covariate and
  \eqn{\sigma^2(v)}{s2(v)} denote the variance function evaluated at
  \eqn{v}, the constant plus power variance function is defined as
  \eqn{\sigma^2(v) = (\theta_1 + |v|^\theta_2)^2}{s2(v) = (t1 +
    |v|^t2)^2}, where \eqn{\theta_1,\theta_2}{t1, t2} are the variance  
  function coefficients. When a grouping factor is present, different
  \eqn{\theta_1,\theta_2}{t1, t2} are used for each factor level.
}
\value{
  a \code{varConstPower} object representing a constant plus power
  variance function structure, also inheriting from class
  \code{varFunc}.  
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{varWeights.varFunc}}, \code{\link{coef.varConstPower}}}

\examples{
vf1 <- varConstPower(1.2, 0.2, form = ~age|Sex)
}
\keyword{models}

\eof
% $Id: varExp.Rd,v 1.6 2002/03/05 14:59:39 bates Exp $
\name{varExp}
\title{Exponential Variance Function}
\usage{
varExp(value, form, fixed)
}
\alias{varExp}
\arguments{
  \item{value}{an optional numeric vector, or list of numeric values,
    with the variance function coefficients. \code{Value} must have
    length one, unless a grouping factor is specified in \code{form}.
    If \code{value} has length greater than one, it must have names
    which identify its elements to the levels of the grouping factor
    defined in \code{form}. If a grouping factor is present in
    \code{form} and \code{value} has length one, its value will be
    assigned to all grouping levels. Default is \code{numeric(0)}, which
    results in a vector of zeros of appropriate length being assigned to
    the coefficients when \code{object} is initialized (corresponding
    to constant variance equal to one).}
  \item{form}{an optional one-sided formula of the form \code{~ v}, or
    \code{~ v | g}, specifying a variance covariate \code{v} and,
    optionally, a grouping factor \code{g} for the coefficients. The
    variance covariate must evaluate to a numeric vector and may involve
    expressions using \code{"."}, representing  a fitted model object
    from which fitted values (\code{fitted(.)}) and residuals
    (\code{resid(.)}) can be extracted (this allows the variance
    covariate to be updated during the optimization of an object
    function). When a grouping factor is present in \code{form},
    a different coefficient value is used for each of its
    levels. Several grouping variables may be
    simultaneously specified, separated by the \code{*} operator, like
    in \code{~ v | g1 * g2 * g3}. In this case, the levels of each
    grouping variable are pasted together and the resulting factor is
    used to group the observations. Defaults to \code{~ fitted(.)}
    representing a variance covariate given by the fitted values of a
    fitted model object and no grouping factor. }
  \item{fixed}{an optional numeric vector, or list of numeric values,
    specifying the values at which some or all of the  coefficients in
    the variance function should be fixed. If a grouping factor is
    specified in \code{form}, \code{fixed} must have names identifying
    which coefficients are to be fixed. Coefficients included in
    \code{fixed} are not allowed to vary during the optimization of an
    objective function. Defaults to \code{NULL}, corresponding to no
    fixed coefficients.}
}
\description{
  This function is a constructor for the \code{varExp} class,
  representing an exponential variance function structure. Letting
  \eqn{v} denote the variance covariate and \eqn{\sigma^2(v)}{s2(v)}
  denote the variance function evaluated at \eqn{v}, the exponential
  variance function is defined as \eqn{\sigma^2(v) = \exp(2\theta
    v)}{s2(v) = exp(2* t * v)}, where \eqn{\theta}{t} is the variance
  function coefficient. When a grouping factor is present, a different
  \eqn{\theta}{t} is used for each factor level.
}
\value{
  a \code{varExp} object representing an exponential variance function
  structure, also inheriting from class \code{varFunc}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{varWeights.varFunc}}, \code{\link{coef.varExp}}}

\examples{
vf1 <- varExp(0.2, form = ~age|Sex)
}
\keyword{models}

\eof
% $Id: varFixed.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{varFixed}
\title{Fixed Variance Function}
\usage{
varFixed(value)
}
\alias{varFixed}
\arguments{
 \item{value}{a one-sided formula of the form \code{~ v} specifying a
   variance covariate \code{v}. Grouping factors are ignored.} 
}
\description{
  This function is a constructor for the \code{varFixed} class,
  representing a variance function with fixed variances. Letting \eqn{v}
  denote the variance covariate defined in \code{value}, the variance
  function \eqn{\sigma^2(v)}{s2(v)} for this class is
  \eqn{\sigma^2(v)=|v|}{s2(v)=|v|}. The variance covariate \eqn{v} is
  evaluated once at initialization and remains fixed thereafter. No
  coefficients are required to represent this variance function.
}
\value{
  a \code{varFixed} object representing a fixed variance function
  structure, also inheriting from class \code{varFunc}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}


\seealso{\code{\link{varWeights.varFunc}}, \code{\link{varFunc}}}

\examples{
vf1 <- varFixed(~age)
}
\keyword{models}

\eof
% $Id: varFunc.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{varFunc}
\title{Variance Function Structure}
\usage{
varFunc(object)
}
\alias{varFunc}
\arguments{
 \item{object}{either an one-sided formula specifying a variance
   covariate, or an object inheriting from class \code{varFunc},
   representing a variance function structure.} 
}
\description{
  If \code{object} is a one-sided formula, it is used as the argument to
  \code{varFixed} and the resulting object is returned. Else, if
  \code{object} inherits from class \code{varFunc}, it is returned
  unchanged. 
}
\value{
  an object from class \code{varFunc}, representing a variance function
  structure. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}


\seealso{\code{\link{varFixed}}, \code{\link{varWeights.varFunc}},
  \code{\link{coef.varFunc}}} 

\examples{
vf1 <- varFunc(~age)
}
\keyword{models}

\eof
% $Id: varIdent.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{varIdent}
\title{Constant Variance Function}
\usage{
varIdent(value, form, fixed)
}
\alias{varIdent}
\arguments{
  \item{value}{an optional numeric vector, or list of numeric values,
    with the variance function coefficients. If no grouping factor is
    present in \code{form}, this argument is ignored, as the resulting
    variance function contains no coefficients. If \code{value} has
    length one, its value is repeated for all coefficients in the
    variance function. If \code{value} has length greater than one, it
    must have length equal to the number of grouping levels minus one
    and names which identify its elements to the levels of the grouping
    factor. Only positive values are allowed for this argument. Default
    is \code{numeric(0)}, which results in a vector of zeros of
    appropriate length being assigned to the coefficients when
    \code{object} is initialized (corresponding to constant variance 
    equal to one).}  
  \item{form}{an optional one-sided formula of the form \code{~ v}, or
    \code{~ v | g}, specifying a variance covariate \code{v} and,
    optionally, a grouping factor \code{g} for the coefficients. The
    variance covariate is ignored in this variance function. When a
    grouping factor is present in \code{form}, a different coefficient
    value is used for each of its levels less one reference level (see
    description section below). Several grouping variables may be
    simultaneously specified, separated by the \code{*} operator, like
    in \code{~ v | g1 * g2 * g3}. In this case, the levels of each
    grouping variable are pasted together and the resulting factor is
    used to group the observations. Defaults to \code{~ 1}. }
  \item{fixed}{an optional numeric vector, or list of numeric values,
    specifying the values at which some or all of the  coefficients in
    the variance function should be fixed. It must have names
    identifying which coefficients are to be fixed. Coefficients
    included in \code{fixed} are not allowed to vary during the
    optimization of an objective function. Defaults to \code{NULL},
    corresponding to no fixed coefficients.}
}
\description{
  This function is a constructor for the \code{varIdent} class,
  representing a constant variance function structure. If no grouping
  factor is present in \code{form}, the variance function is constant
  and equal to one, and no coefficients required to represent it. When
  \code{form} includes a grouping factor with \eqn{M > 1} levels, the
  variance function allows M different variances, one  for each level of
  the factor. For identifiability reasons, the coefficients of the
  variance function represent the ratios between the variances and a
  reference variance (corresponding to a reference group
  level). Therefore, only \eqn{M-1} coefficients are needed to represent
  the variance function. By default, if the elements in \code{value} are
  unnamed, the first group level is taken as the reference level.
}
\value{
  a \code{varIdent} object representing a constant variance function
  structure, also inheriting from class \code{varFunc}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{varWeights.varFunc}}, \code{\link{coef.varIdent}}}

\examples{
vf1 <- varIdent(c(Female = 0.5), form = ~ 1 | Sex)
}
\keyword{models}

\eof
% $Id: varPower.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{varPower}
\title{Power Variance Function}
\usage{
varPower(value, form, fixed)
}
\alias{varPower}
\arguments{
  \item{value}{an optional numeric vector, or list of numeric values,
    with the variance function coefficients. \code{Value} must have
    length one, unless a grouping factor is specified in \code{form}.
    If \code{value} has length greater than one, it must have names
    which identify its elements to the levels of the grouping factor
    defined in \code{form}. If a grouping factor is present in
    \code{form} and \code{value} has length one, its value will be
    assigned to all grouping levels. Default is \code{numeric(0)}, which
    results in a vector of zeros of appropriate length being assigned to
    the coefficients when \code{object} is initialized (corresponding
    to constant variance equal to one).}
  \item{form}{an optional one-sided formula of the form \code{~ v}, or
    \code{~ v | g}, specifying a variance covariate \code{v} and,
    optionally, a grouping factor \code{g} for the coefficients. The
    variance covariate must evaluate to a numeric vector and may involve
    expressions using \code{"."}, representing  a fitted model object
    from which fitted values (\code{fitted(.)}) and residuals
    (\code{resid(.)}) can be extracted (this allows the variance
    covariate to be updated during the optimization of an object
    function). When a grouping factor is present in \code{form},
    a different coefficient value is used for each of its
    levels. Several grouping variables may be
    simultaneously specified, separated by the \code{*} operator, like
    in \code{~ v | g1 * g2 * g3}. In this case, the levels of each
    grouping variable are pasted together and the resulting factor is
    used to group the observations. Defaults to \code{~ fitted(.)}
    representing a variance covariate given by the fitted values of a
    fitted model object and no grouping factor. }
  \item{fixed}{an optional numeric vector, or list of numeric values,
    specifying the values at which some or all of the  coefficients in
    the variance function should be fixed. If a grouping factor is
    specified in \code{form}, \code{fixed} must have names identifying
    which coefficients are to be fixed. Coefficients included in
    \code{fixed} are not allowed to vary during the optimization of an
    objective function. Defaults to \code{NULL}, corresponding to no
    fixed coefficients.}
}
\description{
  This function is a constructor for the \code{varPower} class,
  representing a power variance function structure. Letting
  \eqn{v} denote the variance covariate and \eqn{\sigma^2(v)}{s2(v)}
  denote the variance function evaluated at \eqn{v}, the power
  variance function is defined as \eqn{\sigma^2(v) = |v|^{2\theta}}{s2(v) =
    |v|^(2*t)}, where \eqn{\theta}{t} is the variance 
  function coefficient. When a grouping factor is present, a different
  \eqn{\theta}{t} is used for each factor level.
}
\value{
  a \code{varPower} object representing a power variance function
  structure, also inheriting from class \code{varFunc}. 
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{varWeights.varFunc}}, \code{\link{coef.varPower}}}

\examples{
vf1 <- varPower(0.2, form = ~age|Sex)
}
\keyword{models}

\eof
% $Id: varWeights.Rd,v 1.5.2.1 2002/08/09 19:45:29 bates Exp $
\name{varWeights}
\title{Extract Variance Function Weights}
\usage{
varWeights(object)
}
\alias{varWeights}
\alias{varWeights.varComb}
\alias{varWeights.varFunc}
\arguments{
 \item{object}{an object inheriting from class \code{varFunc},
   representing a variance function structure.} 
}
\description{
The inverse of the standard deviations corresponding to the variance
function structure represented by \code{object} are returned.
}
\value{
  if \code{object} has a \code{weights} attribute, its value is
  returned; else \code{NULL} is returned.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{logLik.varFunc}}}

\examples{
data(Orthodont)
vf1 <- varPower(form=~age)
vf1 <- Initialize(vf1, Orthodont)
coef(vf1) <- 0.3
varWeights(vf1)[1:10]
}
\keyword{models}

\eof
% $Id: varWeights.glsStruct.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{varWeights.glsStruct}
\title{Variance Weights for glsStruct Object}
\usage{
\method{varWeights}{glsStruct}(object)
}
\alias{varWeights.glsStruct}
\arguments{
 \item{object}{an object inheriting from class \code{glsStruct},
   representing a list of linear model components, such as
   \code{corStruct} and \code{varFunc} objects.}
}
\description{
  If \code{object} includes a \code{varStruct} component, the inverse of
  the standard deviations of the variance function structure represented
  by the corresponding \code{varFunc} object are returned; else, a
  vector of ones of length equal to the number of observations in the
  data frame used to fit the associated linear model is returned. 
}
\value{
  if \code{object} includes a \code{varStruct} component, a vector with
  the corresponding variance weights; else, or a vector of ones.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{varWeights}}}

\keyword{models}

\eof
% $Id: varWeights.lmeStruct.Rd,v 1.5 2002/03/05 14:59:39 bates Exp $
\name{varWeights.lmeStruct}
\title{Variance Weights for lmeStruct Object}
\usage{
\method{varWeights}{lmeStruct}(object)
}
\alias{varWeights.lmeStruct}
\arguments{
  \item{object}{an object inheriting from class \code{lmeStruct},
    representing a list of linear mixed-effects model components, such as
    \code{reStruct}, \code{corStruct}, and \code{varFunc} objects.}
}
\description{
  If \code{object} includes a \code{varStruct} component, the inverse of
  the standard deviations of the variance function structure represented
  by the corresponding \code{varFunc} object are returned; else, a
  vector of ones of length equal to the number of observations in the
  data frame used to fit the associated linear mixed-effects model is
  returned. 
}
\value{
  if \code{object} includes a \code{varStruct} component, a vector with
  the corresponding variance weights; else, or a vector of ones.
}

\author{Jose Pinheiro \email{Jose.Pinheiro@pharma.novartis.com} and Douglas Bates \email{bates@stat.wisc.edu}}

\seealso{\code{\link{varWeights}}}

\keyword{models}

\eof
