\name{addisolates}
\alias{addisolates}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Add Isolates to a Graph }
\description{
  Adds \code{n} isolates to the adjacency matrix (or matrices) in \code{dat}.
}
\usage{
addisolates(dat, n)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ One or more adjacency matrices }
  \item{n}{ The number of isolates to add }
}
\details{
  If \code{dat} contains more than one adjacency matrix, the \code{n} isolates are added to each member of \code{dat}.
}
\value{
   The updated graph stack.
}
\references{ Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Inter-Structural Analysis.''  CASOS Working Paper, Carnegie Mellon University. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ Isolate addition is particularly useful when computing structural distances between graphs of different orders; see the above reference for details. }

\seealso{ \code{\link{isolates}} }

\examples{

g<-rgraph(10,5)		#Produce some random graphs

dim(g)			#Get the dimensions of g

g<-addisolates(g,2)	#Add 2 isolates to each graph in g

dim(g)			#Now examine g
g

}
\keyword{ manip }
\keyword{ math }

\eof
\name{bbnam}
\alias{bbnam}
\alias{bbnam.actor}
\alias{bbnam.pooled}
\alias{bbnam.fixed}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Butts' (Hierarchical) Bayesian Network Accuracy Model }
\description{
 Takes posterior draws from Butts' bayesian network accuracy/estimation model for multiple participant/observers (conditional on observed data and priors), using a Gibbs sampler.
}
\usage{
bbnam(dat, model="actor", ...)
bbnam.fixed(dat, nprior=matrix(rep(0.5,dim(dat)[2]^2),
    nrow=dim(dat)[2],ncol=dim(dat)[2]), em=0.25, ep=0.25, diag=FALSE,
    mode="digraph", draws=1500, outmode="draws", anames=paste("a",
    1:dim(dat)[2],sep=""), onames=paste("o",1:dim(dat)[1], sep=""))
bbnam.pooled(dat, nprior=matrix(rep(0.5,dim(dat)[2]*dim(dat)[3]),
    nrow=dim(dat)[2],ncol=dim(dat)[3]), emprior=c(1,1), 
    epprior=c(1,1), diag=FALSE, mode="digraph", reps=5, draws=1500, 
    burntime=500, quiet=TRUE, anames=paste("a",1:dim(dat)[2],sep=""),
    onames=paste("o",1:dim(dat)[1],sep=""), compute.sqrtrhat=TRUE)
bbnam.actor(dat, nprior=matrix(rep(0.5,dim(dat)[2]*dim(dat)[3]),
    nrow=dim(dat)[2],ncol=dim(dat)[3]), 
    emprior=cbind(rep(1,dim(dat)[1]),rep(1,dim(dat)[1])), 
    epprior=cbind(rep(1,dim(dat)[1]),rep(1,dim(dat)[1])), diag=FALSE,
    mode="digraph", reps=5, draws=1500, burntime=500, quiet=TRUE, 
    anames=paste("a",1:dim(dat)[2],sep=""), 
    onames=paste("o",1:dim(dat)[1],sep=""), compute.sqrtrhat=TRUE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  This array must be of dimension n x n x n, where n is |V(G)|, the first dimension indexes the observer, the second indexes the sender of the relation, and the third dimension indexes the recipient of the relation.  (E.g., \code{dat[i,j,k]==1} implies that i observed j sending the relation in question to k.)  Note that only dichotomous data is supported at present, and missing values are permitted; the data collection pattern, however, is assumed to be ignorable, and hence the posterior draws are implicitly conditional on the observation pattern. }
  \item{model}{ String containing the error model to use; options are ``actor,'' ``pooled,'' and ``fixed'' }
  \item{...}{Arguments to be passed by \code{bbnam} to the particular model method}
  \item{nprior}{ Network prior matrix.  This must be a matrix of dimension n x n, containing the arc/edge priors for the criterion network.  (E.g., \code{nprior[i,j]} gives the prior probability of i sending the relation to j in the criterion graph.)  If no network prior is provided, an uninformative prior on the space of networks will be assumed (i.e., p(i->j)=0.5).  Missing values are not allowed. }
  \item{em}{ Probability of a false negative; this may be in the form of a single number, one number per observation slice, one number per (directed) dyad, or one number per dyadic observation (fixed model only) }
  \item{ep}{ Probability of a false positive; this may be in the form of a single number, one number per observation slice, one number per (directed) dyad, or one number per dyadic observation (fixed model only) }
  \item{emprior}{ Parameters for the (beta) false negative prior; these should be in the form of an (alpha,beta) pair for the pooled model, and of an n x 2 matrix of (alpha,beta) pairs for the actor model. If no emprior is given, an uninformative prior (1,1) will be assumed; note that this is usually inappropriate, as described below.  Missing values are not allowed. }
  \item{epprior}{ Parameters for the (beta) false positive prior; these should be in the form of an (alpha,beta) pair for the pooled model, and of an n x 2 matrix of (alpha,beta) pairs for the actor model. If no epprior is given, an uninformative prior (1,1) will be assumed; note that this is usually inappropriate, as described below.  Missing values are not allowed. }
  \item{diag}{ Boolean indicating whether loops (matrix diagonals) should be counted as data }
  \item{mode}{ A string indicating whether the data in question forms a ``graph'' or a ``digraph'' }
  \item{reps}{ Number of replicate chains for the Gibbs sampler (pooled and actor models only) }
  \item{draws}{ Integer indicating the total number of draws to take from the posterior distribution.  Draws are taken evenly from each replication (thus, the number of draws from a given chain is draws/reps), and are randomly reordered to minimize dependence associated with position in the chain. }
  \item{burntime}{ Integer indicating the burn-in time for the Markov Chain.  Each replication is iterated burntime times before taking draws (with these initial iterations being discarded); hence, one should realize that each increment to burntime increases execution time by a quantity proportional to reps. (pooled and actor models only) }
  \item{quiet}{ Boolean indicating whether MCMC diagnostics should be displayed (pooled and actor models only) }
  \item{outmode}{ ``posterior'' indicates that the exact posterior probability matrix for the criterion graph should be returned, otherwise draws from the joint posterior are returned instead (fixed model only) }
  \item{anames}{ A vector of names for the actors (vertices) in the graph }
  \item{onames}{ A vector of names for the observers (possibly the actors themselves) whose reports are contained in the CSS}
  \item{compute.sqrtrhat}{ A boolean indicating whether or not Gelman et al.'s potential scale reduction measure (an MCMC convergence diagnostic) should be computed (pooled and actor models only) }
}
\details{
  The bbnam models a set of network data as reflecting a series of (noisy) observations by a set of participant/observers regarding an uncertain criterion structure.  Each observer is assumed to send false positives (i.e., reporting a tie when none exists in the criterion structure) with probability \eqn{e^+}{e^+}, and false negatives (i.e., reporting that no tie exists when one does in fact exist in the criterion structure) with probability \eqn{e^-}{e^-}.  The criterion network itself is taken to be a Bernoulli (di)graph.  Note that the present model includes three variants:

\enumerate{
\item Fixed error probabilities: Each edge is associated with a known pair of false negative/false positive error probabilities (provided by the researcher).  In this case, the posterior for the criterion graph takes the form of a matrix of Bernoulli parameters, with each edge being independent conditional on the parameter matrix.

\item Pooled error probabilities: One pair of (uncertain) false negative/false positive error probabilities is assumed to hold for all observations.  Here, we assume that the researcher's prior information regarding these parameters can be expressed as a pair of Beta distributions, with the additional assumption of independence in the prior distribution.  Note that error rates and edge probabilities are \emph{not} independent in the joint posterior, but the posterior marginals take the form of Beta mixtures and Bernoulli parameters, respectively. 

\item Per observer (``actor'') error probabilities: One pair of (uncertain) false negative/false positive error probabilities is assumed to hold for each observation slice.  Again, we assume that prior knowledge can be expressed in terms of independent Beta distributions (along with the Bernoulli prior for the criterion graph) and the resulting posterior marginals are Beta mixtures and a Bernoulli graph.  (Again, it should be noted that independence in the priors does \emph{not} imply independence in the joint posterior!)
}

By default, the bbnam routine returns (approximately) independent draws from the joint posterior distribution, each draw yielding one realization of the criterion network and one collection of accuracy parameters (i.e., probabilities of false positives/negatives).  This is accomplished via a Gibbs sampler in the case of the pooled/actor model, and by direct sampling for the fixed probability model. In the special case of the fixed probability model, it is also possible to obtain directly the posterior for the criterion graph (expressed as a matrix of Bernoulli parameters); this can be controlled by the \code{outmode} parameter.

As noted, the taking of posterior draws in the nontrivial case is accomplished via a Markov Chain Monte Carlo method, in particular the Gibbs sampler; the high dimensionality of the problem (\eqn{O(n^2+2n)}{O(n^2+2n)}) tends to preclude more direct approaches.  At present, chain burn-in is determined ex ante on a more or less arbitrary basis by specification of the burntime parameter.  Eventually, a more systematic approach will be utilized.  Note that insufficient burn-in will result in inaccurate posterior sampling, so it's not wise to skimp on burn time where otherwise possible.  Similarly, it is wise to employ more than one Markov Chain (set by reps), since it is possible for trajectories to become ``trapped'' in metastable regions of the state space.  Number of draws per chain being equal, more replications are usually better than few; consult Gelman et al. for details.  A useful measure of chain convergence, Gelman and Rubin's potential scale reduction (\eqn{\sqrt{\hat{R}}}{\sqrt{\hat{R}}}), can be computed using the \code{compute.sqrtrhat} parameter.  The potential scale reduction measure is an ANOVA-like comparison of within-chain versus between-chain variance; it approaches 1 (from above) as the chain converges, and longer burn-in times are strongly recommended for chains with scale reductions in excess of 1.1 or thereabouts.  

Finally, a cautionary concerning prior distributions: it is important that the specified priors actually reflect the prior knowledge of the researcher; otherwise, the posterior will be inadequately informed.  In particular, note that an uninformative prior on the accuracy probabilities implies that it is a priori equally probable that any given actor's observations will be informative or \emph{negatively} informative (i.e., that i observing j sending a tie to k \emph{reduces} p(j->k)).  This is a highly unrealistic assumption, and it will tend to produce posteriors which are bimodal (one mode being related to the ``informative'' solution, the other to the ``negatively informative'' solution).  A more plausible but still fairly diffuse prior would be Beta(3,5), which reduces the prior probability of an actor's being negatively informative to 0.16, and the prior probability of any given actor's being more than 50\% likely to make a particular error (on average) to around 0.22.  (This prior also puts substantial mass near the 0.5 point, which would seem consonant with the BKS studies.)  Butts(1999) discusses a number of issues related to choice of priors for the bbnam, and users should consult this reference if matters are unclear before defaulting to the uninformative solution.
}

\value{
An object of class bbnam, containing the posterior draws.  The components of the output are as follows:

\item{anames}{
A vector of actor names.
}
\item{draws}{
An integer containing the number of draws.
}
\item{em}{
A matrix containing the posterior draws for probability of producing false negatives, by actor.
}
\item{ep}{
A matrix containing the posterior draws for probability of producing false positives, by actor.
}
\item{nactors}{
An integer containing the number of actors.
}
\item{net}{
An array containing the posterior draws for the criterion network.
}
\item{reps}{
An integer indicating the number of replicate chains used by the Gibbs sampler.
}
}
\references{ 
Butts, C. T.  (2003).  ``Network Inference, Error, and Informant (In)Accuracy: A Bayesian Approach.''  \emph{Social Networks}, 25(2), 103-140.

Gelman, A.; Carlin, J.B.; Stern, H.S.; and Rubin, D.B.  (1995).  \emph{Bayesian Data Analysis.}  London: Chapman and Hall.

Gelman, A., and Rubin, D.B.  (1992).  ``Inference from Iterative Simulation Using Multiple Sequences.''  \emph{Statistical Science,} 7, 457-511.

Krackhardt, D.  (1987).  ``Cognitive Social Structures.'' \emph{Social Networks,} 9, 109-134.
}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ As indicated, the posterior draws are conditional on the observed data, and hence on the data collection mechanism if the collection design is non-ignorable.  Complete data (e.g., a CSS) and random tie samples are examples of ignorable designs; see Gelman et al. for more information concerning ignorability.}
 
\seealso{\code{\link{npostpred}}, \code{\link{event2dichot}}, \code{\link{bbnam.bf}}}

\examples{
#Create some random data
g<-rgraph(5)
g.p<-0.8*g+0.2*(1-g)
dat<-rgraph(5,5,tprob=g.p)

#Define a network prior
pnet<-matrix(ncol=5,nrow=5)
pnet[,]<-0.5
#Define em and ep priors
pem<-matrix(nrow=5,ncol=2)
pem[,1]<-3
pem[,2]<-5
pep<-matrix(nrow=5,ncol=2)
pep[,1]<-3
pep[,2]<-5

#Draw from the posterior
b<-bbnam(dat,model="actor",nprior=pnet,emprior=pem,epprior=pep,
    burntime=100,draws=100)
#Print a summary of the posterior draws
summary(b)
}

\keyword{ models}
\keyword{ math }


\eof
\name{bbnam.bf}
\alias{bbnam.bf}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Estimate Bayes Factors for the bbnam }
\description{
  This function uses monte carlo integration to estimate the BFs, and tests the fixed probability, pooled, and pooled by actor models. (See \code{\link{bbnam}} for details.)
}
\usage{
bbnam.bf(dat, nprior=matrix(rep(0.5, dim(dat)[1]^2), 
    nrow = dim(dat)[1], ncol = dim(dat)[1]), em.fp=0.5, ep.fp=0.5, 
    emprior.pooled=c(1, 1), epprior.pooled=c(1, 1), 
    emprior.actor=cbind(rep(1, dim(dat)[1]), rep(1, dim(dat)[1])), 
    epprior.actor=cbind(rep(1, dim(dat)[1]), rep(1, dim(dat)[1])), 
    diag=FALSE, mode="digraph", reps=1000)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  This array must be of dimension m x n x n, where n is |V(G)|, the first dimension indexes the observer, the second indexes the sender of the relation, and the third dimension indexes the recipient of the relation.  (E.g., \code{dat[i,j,k]==1} implies that i observed j sending the relation in question to k.)  Note that only dichotomous data is supported at present, and missing values are permitted; the data collection pattern, however, is assumed to be ignorable, and hence the posterior inferences are implicitly conditional on the observation pattern. }
  \item{nprior}{ Network prior matrix.  This must be a matrix of dimension n x n, containing the arc/edge priors for the criterion network.  (E.g., \code{nprior[i,j]} gives the prior probability of i sending the relation to j in the criterion graph.)  If no network prior is provided, an uninformative prior on the space of networks will be assumed (i.e., p(i->j)=0.5).  Missing values are not allowed. }
  \item{em.fp}{ Probability of false negatives for the fixed probability model }
  \item{ep.fp}{ Probability of false positives for the fixed probability model }
  \item{emprior.pooled}{ (alpha,beta) pairs for the (beta) false negative prior under the pooled model }
  \item{epprior.pooled}{ (alpha,beta) pairs for the (beta) false positive prior under the pooled model }
  \item{emprior.actor}{ Matrix of per observer (alpha,beta) pairs for the (beta) false negative prior under the per observer/actor model }
  \item{epprior.actor}{ Matrix of per observer (alpha,beta) pairs for the (beta) false negative prior under the per observer/actor model }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the criterion graph can contain loops.  Diag is false by default. }
  \item{mode}{ String indicating the type of graph being evaluated.  "Digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  Mode is set to "digraph" by default. }
  \item{reps}{ Number of monte carlo draws to take }
}
\details{
  The bbnam model (detailed in the \code{\link{bbnam}} function help) is a fairly simple model for integrating informant reports regarding social network data.  \code{bbnam.bf} computes Bayes Factors (integrated likelihood ratios) for the three error submodels of the bbnam: fixed error probabilities, pooled error probabilities, and per observer/actor error probabilities.  
}
\value{
   An object of class \code{bayes.factor}.
}
\references{
Butts, C. T.  (2003).  ``Network Inference, Error, and Informant (In)Accuracy: A Bayesian Approach.''  \emph{Social Networks}, 25(2), 103-140.

Robert, C.  (1994).  \emph{The Bayesian Choice: A Decision-Theoretic Motivation.}  Springer. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ It is important to be aware that the model parameter priors are essential components of the models to be compared; inappropriate parameter priors will result in misleading Bayes Factors. }


\seealso{ \code{\link{bbnam}} }

\examples{
}

\keyword{models}
\keyword{ math }

\eof
\name{betweenness}
\alias{betweenness}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute the Betweenness Centrality Scores of Network Positions }
\description{
   \code{betweenness} takes a graph stack (\code{dat}) and returns the betweenness centralities of positions within one graph (indicated by \code{nodes} and \code{g}, respectively).  Depending on the specified mode, betweenness on directed or undirected geodesics will be returned; this function is compatible with \code{\link{centralization}}, and will return the theoretical maximum absolute deviation (from maximum) conditional on size (which is used by \code{\link{centralization}} to normalize the observed centralization score).
}
\usage{
betweenness(dat, g=1, nodes=c(1:dim(dat)[2]), gmode="digraph", 
    diag=FALSE, tmaxdev=FALSE, cmode="directed", 
    geodist.precomp=NULL, rescale=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors. Alternately, this can be an n x n matrix (if only one graph is involved). }
  \item{g}{ Integer indicating the index of the graph for which centralities are to be calculated.  By default, \code{g}=1. }
  \item{nodes}{ List indicating which nodes are to be included in the calculation.  By default, all nodes are included. }
  \item{gmode}{ String indicating the type of graph being evaluated.  "digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{dmode} is set to "digraph" by default. }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{tmaxdev}{ Boolean indicating whether or not the theoretical maximum absolute deviation from the maximum nodal centrality should be returned.  By default, \code{tmaxdev}==\code{FALSE}. }
  \item{cmode}{ String indicating the type of betweenness centrality being computed (directed or undirected geodesics). }
  \item{geodist.precomp}{ A \code{\link{geodist}} object precomputed for the graph to be analyzed (optional) }
  \item{rescale}{ If true, centrality scores are rescaled such that they sum to 1. }
}
\details{
The betweenness of a vertex, v, is given by

\deqn{
C_B(v) = \sum_{i,j : i \neq j, i \neq v, j \neq v} \frac{g_{ivj}}{g_{ij}}}{%
C_B(v) = sum( g_ivj / g_ij, i,j: i!=j,i!=v,j!=v )}

where \eqn{g_{ijk}}{g_ijk} is the number of geodesics from i to k through j.  Conceptually, high-betweenness vertices lie on a large number of non-redundant shortest paths between other vertices; they can thus be thought of as ``bridges'' or ``boundary spanners.'' 
}
\value{
A vector containing the betweenness scores.
}
\references{ Freeman, L.C.  (1979).  ``Centrality in Social Networks I: Conceptual Clarification.'' \emph{Social Networks}, 1, 215-239.}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ Judicious use of \code{geodist.precomp} can save a great deal of time when computing multiple path-based indices on the same network. }

\section{Warning }{Rescale may cause unexpected results if all actors have zero betweenness.}

\seealso{ \code{\link{centralization}} }

\examples{
g<-rgraph(10)     #Draw a random graph with 10 members
betweenness(g)    #Compute betweenness scores
}

\keyword{univar}
\keyword{ math }

\eof
\name{blockmodel}
\alias{blockmodel}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Generate Blockmodels Based on Partitions of Network Positions }
\description{
  Given an equivalence clustering object and a graph stack, \code{blockmodel} will form a blockmodel of the graph stack based on the classes in question, using the specified block content type.
}
\usage{
blockmodel(dat, ec, k=NULL, h=NULL, block.content="density", 
    plabels=ec$plabels, glabels=ec$glabels, rlabels=NULL, 
    mode="digraph", diag=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph stack }
  \item{ec}{ An object of class \code{equiv.clust} }
  \item{k}{ The number of classes to form (using \code{\link[mva]{cutree}}) }
  \item{h}{ The height at which to split classes (using \code{\link[mva]{cutree}}) }
  \item{block.content}{ String indicating block content type (see below)}
  \item{plabels}{ A vector of labels to be applied to the individual nodes}
  \item{glabels}{ A vector of labels to be applied to the graphs being modeled }
  \item{rlabels}{ A vector of labels to be applied to the (reduced) roles }
  \item{mode}{ A string indicating whether we are dealing with graphs or digraphs }
  \item{diag}{ A boolean indicating whether loops are permitted }
}
\details{
\code{blockmodel} forms its eponymous models by using \code{\link[mva]{cutree}} to cut an equivalence clustering in the fashion specified by \code{k} and \code{h}.  After forming clusters (roles), the input graphs are reordered and blockmodel reduction is applied.  Currently supported reductions are:

\enumerate{
\item \code{density}: block density, computed as the mean value of the block

\item \code{meanrowsum}: mean row sums for the block

\item \code{meancolsum}: mean column sums for the block

\item \code{sum}: total block sum

\item \code{median}: median block value

\item \code{min}: minimum block value

\item \code{max}: maximum block value

\item \code{types}: semi-intelligent coding of blocks by ``type.''  Currently recognized types are (in order of precedence) ``\code{NA}'' (i.e., blocks with no valid data), ``null'' (i.e., all values equal to zero), ``complete'' (i.e., all values equal to 1), ``1 covered'' (i.e., all rows/cols contain a 1), ``1 row-covered'' (i.e., all rows contain a 1), ``1 col-covered'' (i.e., all cols contain a 1), and ``other'' (i.e., none of the above). 
}

Density or median-based reductions are probably the most interpretable for most conventional analyses, though type-based reduction can be useful in examining certain equivalence class hypotheses (e.g., 1 covered and null blocks can be used to infer regular equivalence classes).  Once a given reduction is performed, the model can be analyzed and/or expansion can be used to generate new graphs based on the inferred role structure.
}
\value{
An object of class \code{blockmodel}.
}
\references{White, H.C.; Boorman, S.A.; and Breiger, R.L.  (1976).  ``Social Structure from Multiple Networks I: Blockmodels of Roles and Positions.''  \emph{American Journal of Sociology}, 81, 730-779.}

\author{ Carter T. Butts \email{buttsc@uci.edu} }

\note{ The \code{mva} library is required for \code{blockmodel} to function. }


\seealso{ \code{\link{equiv.clust}}, \code{\link{blockmodel.expand}} }

\examples{
#Create a random graph with _some_ edge structure
g.p<-sapply(runif(20,0,1),rep,20)  #Create a matrix of edge 
                                   #probabilities
g<-rgraph(20,tprob=g.p)            #Draw from a Bernoulli graph 
                                   #distribution

#Cluster based on structural equivalence
eq<-equiv.clust(g)

#Form a blockmodel with distance relaxation of 10
b<-blockmodel(g,eq,h=10)
plot(b)                            #Plot it
}
\keyword{ classif }%-- one or more ...
\keyword{ math }

\eof
\name{blockmodel.expand}
\alias{blockmodel.expand}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Generate a Graph (or Stack) from a Given Blockmodel Using Particular Expansion Rules }
\description{
  \code{blockmodel.expand} takes a blockmodel and an expansion vector, and expands the former by making copies of the vertices.
}
\usage{
blockmodel.expand(b, ev, mode="digraph", diag=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{b}{ Blockmodel object }
  \item{ev}{ A vector indicating the number of copies to make of each class (respectively) }
  \item{mode}{ A string indicating whether the result should be a ``graph'' or ``digraph'' }
  \item{diag}{ A boolean indicating whether or not loops should be permitted }
}
\details{
  The primary use of blockmodel expansion is in generating test data from a blockmodeling hypothesis.  Expansion is performed depending on the content type of the blockmodel; at present, only density is supported.  For the density content type, expansion is performed by interpreting the interclass density as an edge probability, and by drawing random graphs from the Bernoulli parameter matrix formed by expanding the density model.  Thus, repeated calls to \code{blockmodel.expand} can be used to generate a sample for monte carlo null hypothesis tests under a Bernoulli graph model.
}
\value{
An adjacency matrix, or stack thereof.
}
\references{ White, H.C.; Boorman, S.A.; and Breiger, R.L.  (1976).  ``Social Structure from Multiple Networks I: Blockmodels of Roles and Positions.''  \emph{American Journal of Sociology}, 81, 730-779. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ Eventually, other content types will be supported. }


\seealso{ \code{\link{blockmodel}} }

\examples{
#Create a random graph with _some_ edge structure
g.p<-sapply(runif(20,0,1),rep,20)  #Create a matrix of edge 
                                   #probabilities
g<-rgraph(20,tprob=g.p)            #Draw from a Bernoulli graph 
                                   #distribution

#Cluster based on structural equivalence
eq<-equiv.clust(g)

#Form a blockmodel with distance relaxation of 15
b<-blockmodel(g,eq,h=15)

#Draw from an expanded density blockmodel
g.e<-blockmodel.expand(b,rep(2,length(b$rlabels)))  #Two of each class
g.e
}
\keyword{ manip }%-- one or more ...
\keyword{ math }

\eof
\name{bonpow}
\alias{bonpow}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find Bonacich Power Centrality Scores of Network Positions }
\description{
  \code{bonpow} takes a graph stack (\code{dat}) and returns the Bonacich power centralities of positions within one graph (indicated by \code{nodes} and \code{g}, respectively).  The decay rate for power contributions is specified by \code{exponent} (1 by default).  This function is compatible with \code{\link{centralization}}, and will return the theoretical maximum absolute deviation (from maximum) conditional on size (which is used by \code{\link{centralization}} to normalize the observed centralization score).
}
\usage{
bonpow(dat, g=1, nodes=c(1:dim(dat)[2]), gmode="digraph", diag=FALSE,
    tmaxdev=FALSE, exponent=1, rescale=FALSE, tol=1e-07)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors. Alternately, this can be an n x n matrix (if only one graph is involved). }
  \item{g}{ Integer indicating the index of the graph for which centralities are to be calculated.  By default, \code{g}=1. }
  \item{nodes}{ List indicating which nodes are to be included in the calculation.  By default, all nodes are included. }
  \item{gmode}{ String indicating the type of graph being evaluated.  "Digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  This is currently ignored. }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{Diag} is \code{FALSE} by default. }
  \item{tmaxdev}{ Boolean indicating whether or not the theoretical maximum absolute deviation from the maximum nodal centrality should be returned.  By default, \code{tmaxdev}=\code{FALSE}. }
  \item{exponent}{ Exponent (decay rate) for the Bonacich power centrality score; can be negative }
  \item{rescale}{ If true, centrality scores are rescaled such that they sum to 1. }
  \item{tol}{ Tolerance for near-singularities during matrix inversion (see \code{\link{solve}}) }
}
\details{
Bonacich's power centrality measure is defined by \eqn{C_{BP}\left(\alpha,\beta\right)=\alpha\left(\mathbf{I}-\beta\mathbf{A}\right)^{-1}\mathbf{A}\mathbf{1}}{C_BP(alpha,beta)=alpha (I-A)^-1 A 1}, where \eqn{\beta}{beta} is an attenuation parameter (set here by \code{exponent}) and \eqn{\mathbf{A}}{A} is the graph adjacency matrix.  (The coefficient \eqn{\alpha}{alpha} acts as a scaling parameter, and is set here (following Bonacich (1987)) such that the sum of squared scores is equal to the number of vertices.  This allows 1 to be used as a reference value for the ``middle'' of the centrality range.)  When \eqn{\beta \rightarrow 1/\lambda_{\mathbf{A}1}}{beta->1/lambda_A1} (the reciprocal of the largest eigenvalue of \eqn{\mathbf{A}}{A}), this is to within a constant multiple of the familiar eigenvector centrality score; for other values of \eqn{\beta}, the behavior of the measure is quite different.  In particular, \eqn{\beta} gives positive and negative weight to even and odd walks, respectively, as can be seen from the series expansion \eqn{C_{BP}\left(\alpha,\beta\right)=\alpha \sum_{k=0}^\infty \beta^k \mathbf{A}^{k+1} \mathbf{1}}{C_BP(alpha,beta) = alpha sum( beta^k A^(k+1) 1, k in 0..infinity )} which converges so long as \eqn{|\beta| < 1/\lambda_{\mathbf{A}1}}{|beta|<1/lambda_A1}.  The magnitude of \eqn{\beta}{beta} controls the influence of distant actors on ego's centrality score, with larger magnitudes indicating slower rates of decay.  (High rates, hence, imply a greater sensitivity to edge effects.)

Interpretively, the Bonacich power measure corresponds to the notion that the power of a vertex is recursively defined by the sum of the power of its alters.  The nature of the recursion involved is then controlled by the power exponent: positive values imply that vertices become more powerful as their alters become more powerful (as occurs in cooperative relations), while negative values imply that vertices become more powerful only as their alters become \emph{weaker} (as occurs in competitive or antagonistic relations).  The magnitude of the exponent indicates the tendency of the effect to decay across long walks; higher magnitudes imply slower decay.  One interesting feature of this measure is its relative instability to changes in exponent magnitude (particularly in the negative case).  If your theory motivates use of this measure, you should be very careful to choose a decay parameter on a non-ad hoc basis.
}
\value{
A vector containing the centrality scores.
}
\references{ Bonacich, P.  (1972).  ``Factoring and Weighting Approaches to Status Scores and Clique Identification.''  \emph{Journal of Mathematical Sociology}, 2, 113-120.

Bonacich, P.  (1987).  ``Power and Centrality: A Family of Measures.'' \emph{American Journal of Sociology}, 92, 1170-1182.
 }
\author{ Carter T. Butts \email{buttsc@uci.edu}}
\note{ The theoretical maximum deviation used here is not obtained with the star network, in general.  For positive exponents, at least, the symmetric maximum occurs for an empty graph with one complete dyad (the asymmetric maximum is generated by the outstar).  UCINET V seems not to adjust for this fact, which can cause some oddities in their centralization scores (thus, don't expect to get the same numbers with both packages).}

\section{Warning }{Singular adjacency matrices cause no end of headaches for this algorithm; thus, the routine may fail in certain cases.  This will be fixed when I get a better algorithm.  \code{bonpow} will not symmetrize your data before extracting eigenvectors; don't send this routine asymmetric matrices unless you really mean to do so.}

\seealso{ \code{\link{centralization}}, \code{\link{evcent}} }

\examples{
#Generate some test data
dat<-rgraph(10,mode="graph")
#Compute Bonpow scores
bonpow(dat,exponent=1,tol=1e-20)
bonpow(dat,exponent=-1,tol=1e-20)
}
\keyword{ univar }%-- one or more ...
\keyword{ math }

\eof
\name{centralgraph}
\alias{centralgraph}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find the Central Graph of a Labeled Graph Set }
\description{
  Returns the central graph of a set of labeled graphs, i.e. that graph in which i->j iff i->j in >=50\% of the graphs within the set.  If \code{normalize==TRUE}, then the value of the i,jth edge is given as the proportion of graphs in which i->j.
}
\usage{
centralgraph(dat, normalize=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors. }
  \item{normalize}{ Boolean indicating whether the results should be normalized.  The result of this is the "mean matrix".  By default, \code{normalize==FALSE}.}
}
\details{
  The central graph of a set of graphs S is that graph C which minimizes the sum of Hamming distances between C and G in S.  As such, it turns out (for the dichotomous case, at least), to be analogous to both the mean and median for sets of graphs.  The central graph is useful in a variety of contexts; see the references below for more details.
}
\value{
A matrix containing the central graph (or mean matrix)
}
\references{ 
Banks, D.L., and Carley, K.M.  (1994).  ``Metric Inference for Social Networks.''  \emph{Journal of Classification}, 11(1), 121-49. 
 }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ 0.5 is used as the cutoff value regardless of whether or not the data is dichotomous (as is tacitly assumed).  The routine is unaffected by data type when \code{normalize==TRUE}. }

\seealso{ \code{\link{hdist}} }

\examples{
#Generate some random graphs
dat<-rgraph(10,5)
#Find the central graph
cg<-centralgraph(dat)
#Plot the central graph
gplot(cg)
#Now, look at the mean matrix
cg<-centralgraph(dat,normalize=TRUE)
print(cg)
}
\keyword{ math }%-- one or more ...

\eof
\name{centralization}
\alias{centralization}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find the Centralization of a Given Network, for Some Measure of Centrality  }
\description{
  \code{Centralization} returns the centralization GLI (graph-level index) for a given graph in \code{dat}, given a (node) centrality measure \code{FUN}.  \code{Centralization} follows Freeman's (1979) generalized definition of network centralization, and can be used with any properly defined centrality measure.  This measure must be implemented separately; see the references below for examples.
}
\usage{
centralization(dat, FUN, g=1, mode="digraph", diag=FALSE, 
    normalize=TRUE, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors.  Provided that \code{FUN} is well-behaved, this can be an n x n matrix if only one graph is involved. }
  \item{FUN}{ Function to return nodal centrality scores.}
  \item{g}{ Integer indicating the index of the graph for which centralization should be computed.  By default, \code{g}=1. }
  \item{mode}{ String indicating the type of graph being evaluated.  "digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{mode} is set to "digraph" by default. }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{normalize}{ Boolean indicating whether or not the centralization score should be normalized to the theoretical maximum.  (Note that this function relies on \code{FUN} to return this value when called with \code{tmaxdev==TRUE}.)  By default, \code{tmaxdev==TRUE}. }
  \item{\dots}{ Additional arguments to \code{FUN}. }
}
\details{
  The centralization of a graph G for centrality measure \eqn{C(v)}{C(v)} is defined (as per Freeman (1979)) to be:

\deqn{C^*(G) = \sum_{i \in V(G)} \left|\max_{v \in V(G)}(C(v))-C(i)\right|}{%
C^*(G) = sum( |max(C(v))-C(i)|, i in V(G) )}

Or, equivalently, the absolute deviation from the maximum of C on G.  Generally, this value is normalized by the theoretical maximum centralization score, conditional on \eqn{|V(G)|}{|V(G)|}.  (Here, this functionality is activated by \code{normalize}.)  \code{Centralization} depends on the function specified by \code{FUN} to return the vector of nodal centralities when called with \code{dat} and \code{g}, and to return the theoretical maximum value when called with the above and \code{tmaxdev==TRUE}.  For an example of such a centrality routine, see \code{\link{degree}}.
}
\value{
The centralization of the specified graph.
}
\references{Freeman, L.C.  (1979).  ``Centrality in Social Networks I: Conceptual Clarification.'' \emph{Social Networks}, 1, 215-239.

Wasserman, S., and Faust, K.  (1994).  \emph{Social Network Analysis: Methods and Applications.}  Cambridge: Cambridge University Press.
 }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ See \code{\link{cugtest}} for null hypothesis tests involving centralization scores. }

\seealso{ \code{\link{cugtest}} }

\examples{
#Generate some random graphs
dat<-rgraph(5,10)
#How centralized is the third one on indegree?
centralization(dat,g=3,degree,cmode="indegree")
#How about on total (Freeman) degree?
centralization(dat,g=3,degree)
}
\keyword{ univar }%-- one or more ...
\keyword{ math }

\eof
\name{closeness}
\alias{closeness}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute the Closeness Centrality Scores of Network Positions }
\description{
 \code{closeness} takes a graph stack (\code{dat}) and returns the closeness centralities of positions within one graph (indicated by \code{nodes} and \code{g}, respectively).  Depending on the specified mode, closeness on directed or undirected geodesics will be returned; this function is compatible with \code{\link{centralization}}, and will return the theoretical maximum absolute deviation (from maximum) conditional on size (which is used by \code{\link{centralization}} to normalize the observed centralization score).
}
\usage{
closeness(dat, g=1, nodes=c(1:dim(dat)[2]), gmode="digraph", 
    diag=FALSE, tmaxdev=FALSE, cmode="directed", 
    geodist.precomp=NULL, rescale=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors. Alternately, this can be an n x n matrix (if only one graph is involved). }
  \item{g}{ Integer indicating the index of the graph for which centralities are to be calculated.  By default, \code{g}=1. }
  \item{nodes}{ List indicating which nodes are to be included in the calculation.  By default, all nodes are included. }
  \item{gmode}{ String indicating the type of graph being evaluated.  "digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{gmode} is set to "digraph" by default. }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{tmaxdev}{ Boolean indicating whether or not the theoretical maximum absolute deviation from the maximum nodal centrality should be returned.  By default, \code{tmaxdev==FALSE}. }
  \item{cmode}{ String indicating the type of closeness centrality being computed (distances on directed or undirected geodesics). }
  \item{geodist.precomp}{ A \code{\link{geodist}} object precomputed for the graph to be analyzed (optional) }
  \item{rescale}{ If true, centrality scores are rescaled such that they sum to 1. }
}
\details{
  The closeness of a vertex v is defined as

\deqn{C_C(v) = \frac{\left|V\left(G\right)\right|-1}{\sum_{i : i \neq v} d(v,i)}}{%
C_C(v) = (|V(G)|-1)/sum( d(v,i), i in V(G), i!=v )}

where \eqn{d(i,j)}{d(i,j)} is the geodesic distance between i and j (where defined).  Closeness is ill-defined on disconnected graphs; in such cases, this routine substitutes a number one greater than the maximum path length (i.e., |V(G)|) for the geodesic distance).  It should be understood that this modification is not canonical, but can be avoided by not attempting to measure closeness on disconnected graphs in the first place!  Intuitively, closeness provides an index of the extent to which a given vertex has short paths to all other vertices in the graph; this is one reasonable measure of the extent to which a vertex is in the ``middle'' of a given structure.
}
\value{
A vector containing the closeness scores.
}
\references{Freeman, L.C.  (1979).  ``Centrality in Social Networks I: Conceptual Clarification.'' \emph{Social Networks}, 1, 215-239. }
\author{ Carter T. Butts, \email{buttsc@uci.edu} }
\note{ Judicious use of \code{geodist.precomp} can save a great deal of time when computing multiple path-based indices on the same network. }



\seealso{ \code{\link{centralization}} }

\examples{
g<-rgraph(10)     #Draw a random graph with 10 members
closeness(g)      #Compute closeness scores

}
\keyword{univar }%-- one or more ...
\keyword{ math }

\eof
\name{component.dist}
\alias{component.dist}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Calculate the Component Size Distribution of a Graph }
\description{
  \code{component.dist} returns a data frame containing a vector of length n such that the ith element contains the number of components of G having size i, and a vector of length n giving component membership.  Component strength is determined by the \code{connected} parameter; see below for details.
}
\usage{
component.dist(dat, connected=c("strong","weak","unilateral",
     "recursive"))
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A single nxn adjacency matrix }
  \item{connected}{ A string selecting strong, weak, unilateral or recursively connected components; by default, \code{"strong"} components are used.}
}
\details{
Components are maximal sets of mutually connected vertices; depending on the definition of ``connected'' one employs, one can arrive at several types of components.  Those supported here are as follows (in increasing order of restrictiveness):

\enumerate{
\item \code{weak}:  \eqn{v_1} is connected to \eqn{v_2} iff there exists a semi-path from \eqn{v_1} to \eqn{v_2} (i.e., a path in the weakly symmetrized graph)

\item \code{unilateral}:  \eqn{v_1} is connected to \eqn{v_2} iff there exists a directed path from \eqn{v_1} to \eqn{v_2} \emph{or} a directed path from \eqn{v_2} to \eqn{v_1}

\item \code{strong}:  \eqn{v_1} is connected to \eqn{v_2} iff there exists a directed path from \eqn{v_1} to \eqn{v_2} \emph{and} a directed path from \eqn{v_2} to \eqn{v_1}

\item \code{recursive}:  \eqn{v_1} is connected to \eqn{v_2} iff there exists a vertex sequence \eqn{v_a,\ldots,v_z}{v_a,...,v_z} such that \eqn{v_1,v_a,\ldots,v_z,v_2}{v_1,v_a,...,v_z,v_2} and \eqn{v_2,v_z,\ldots,v_a,v_1}{v_2,v_z,...,v_a,v_1} are directed paths
}

Note that the above definitions are distinct for directed graphs only; if \code{dat} is symmetric, then the \code{connected} parameter has no effect.  
}
\value{
A data frame containing:

  \item{membership }{A vector of component memberships, by vertex}
  \item{csize}{A vector of component sizes, by component}
  \item{cdist}{A vector of length |V(G)| with the (unnormalized) empirical distribution function of component sizes}
}
\references{ West, D.B.  (1996).  \emph{Introduction to Graph Theory.}  Upper Saddle River, N.J.: Prentice Hall.}
\author{ Carter T. Butts \email{buttsc@uci.edu} }

\section{Note}{Unilaterally connected component partitions may not be well-defined, since it is possible for a given vertex to be unilaterally connected to two vertices which are not unilaterally connected with one another.  Consider, for instance, the graph \eqn{a \rightarrow b \leftarrow c \rightarrow d}{a->b<-c<-d}.  In this case, the maximal unilateral components are \eqn{ab} and \eqn{bcd}, with vertex \eqn{b} properly belonging to both components.  For such graphs, a unique partition of vertices by component does not exist, and we ``solve'' the problem by allocating each ``problem vertex'' to one of its components on an essentially arbitrary basis.  (\code{component.dist} generates a warning when this occurs.)  It is recommended that the \code{unilateral} option be avoided where possible.}

\seealso{ \code{\link{components}}, \code{\link{symmetrize}}, \code{\link{reachability}} \code{\link{geodist}} }

\examples{
g<-rgraph(20,tprob=0.075)   #Generate a sparse random graph

#Find weak components
cd<-component.dist(g,connected="weak")
cd$membership              #Who's in what component?
cd$csize                   #What are the component sizes?
                           #Plot the size distribution
plot(1:length(cd$cdist),cd$cdist/sum(cd$cdist),ylim=c(0,1),type="h")  

#Find strong components
cd<-component.dist(g,connected="strong")
cd$membership              #Who's in what component?
cd$csize                   #What are the component sizes?
                           #Plot the size distribution
plot(1:length(cd$cdist),cd$cdist/sum(cd$cdist),ylim=c(0,1),type="h")

}
\keyword{ math }%-- one or more ...

\eof
\name{components}
\alias{components}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find the Number of (Maximal) Components Within a Given Graph }
\description{
  Returns the number of components within \code{dat}, using the connectedness rule given in \code{connected}.
}
\usage{
components(dat, connected="strong", comp.dist.precomp=NULL)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ An nxn adjacency matrix }
  \item{connected}{ The symmetrizing rule to be used by \code{\link{symmetrize}} before components are extracted }
  \item{comp.dist.precomp}{ A component size distribution object from \code{\link{component.dist}} (optional) }
}
\details{
  The \code{connected} parameter corresponds to the \code{rule} parameter of \code{\link{component.dist}}.  By default, \code{components} returns the number of strong components, but weak components can be returned if so desired.  For symmetric matrices, this is obviously a moot point.
}
\value{
  The number of components of \code{dat}
}
\references{ West, D.B.  (1996).  \emph{Introduction to Graph Theory}.  Upper Saddle River, NJ: Prentice Hall. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }

\seealso{ \code{\link{component.dist}}, \code{link{symmetrize}} }

\examples{
g<-rgraph(20,tprob=0.05)   #Generate a sparse random graph

#Find weak components
components(g,connected="weak")

#Find strong components
components(g,connected="strong")

}
\keyword{ math }%-- one or more ...

\eof
\name{connectedness}
\alias{connectedness}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute Graph Connectedness Scores }
\description{
\code{connectedness} takes a graph stack (\code{dat}) and returns the Krackhardt connectedness scores for the graphs selected by \code{g}. 
}
\usage{
connectedness(dat, g=1:stackcount(dat))
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph or graph stack }
  \item{g}{ Index values for the graphs to be utilized; by default, all graphs are selected }
}
\details{
Krackhardt's connectedness for a digraph \eqn{G} is equal to the fraction of all dyads, \eqn{\{i,j\}}, such that there exists an undirected path from \eqn{i} to \eqn{j} in \eqn{G}.  (This, in turn, is just the density of the weak \code{\link{reachability}} graph of \eqn{G}.)  Obviously, the connectedness score ranges from 0 (for the null graph) to 1 (for weakly connected graphs).

Connectedness is one of four measures (\code{\link{connectedness}}, \code{\link{efficiency}}, \code{\link{hierarchy}}, and \code{\link{lubness}}) suggested by Krackhardt for summarizing hierarchical structures.  Each corresponds to one of four axioms which are necessary and sufficient for the structure in question to be an outtree; thus, the measures will be equal to 1 for a given graph iff that graph is an outtree.  Deviations from unity can be interpreted in terms of failure to satisfy one or more of the outtree conditions, information which may be useful in classifying its structural properties.
}
\value{
A vector containing the connectedness scores
}
\references{ Krackhardt, David.  (1994).  ``Graph Theoretical Dimensions of Informal Organizations.'' In K. M. Carley and M. J. Prietula (Eds.), \emph{Computational Organization Theory}, 89-111. Hillsdale, NJ: Lawrence Erlbaum and Associates.}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{
The four Krackhardt indices are, in general, nondegenerate for a relatively narrow band of size/density combinations (efficiency being the sole exception).  This is primarily due to their dependence on the reachability graph, which tends to become complete rapidly as size/density increase.  See Krackhardt (1994) for a useful simulation study.
}
\seealso{ \code{\link{connectedness}}, \code{\link{efficiency}}, \code{\link{hierarchy}}, \code{\link{lubness}}, \code{\link{reachability}} }

\examples{
#Get connectedness scores for graphs of varying densities
connectedness(rgraph(10,5,tprob=c(0.1,0.25,0.5,0.75,0.9)))
}
\keyword{ math }% at least one, from doc/KEYWORDS
\keyword{ univar }% __ONLY ONE__ keyword per line

\eof
\name{consensus}
\alias{consensus}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Estimate a Consensus Structure from Multiple Observations }
\description{
  \code{consensus} estimates a central or consensus structure given multiple observations, using one of several algorithms.
}
\usage{
consensus(dat, mode="digraph", diag=FALSE, method="central.graph", 
    tol=0.01)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ An m x n x n graph stack}
  \item{mode}{ \code{"digraph"} for directed data, else \code{"graph"} }
  \item{diag}{ A boolean indicating whether the diagonals (loops) should be treated as data }
  \item{method}{ One of \code{"central.graph"}, \code{"single.reweight"}, \code{"PCA.reweight"}, \code{"LAS.intersection"}, \code{"LAS.union"}, \code{"OR.row"}, or \code{"OR.col"}  }
  \item{tol}{ Tolerance for the iterative reweighting algorithm (not currently supported)}
}
\details{

The term ``consensus structure'' is used by a number of authors to reflect a notion of shared or common perceptions of social structure among a set of observers.  As there are many interpretations of what is meant by ``consensus'' (and as to how best to estimate it), several algorithms are employed here:

\enumerate{
\item \code{central.graph}: Estimate the consensus structure using the central graph.  This correponds to a ``median response'' notion of consensus.
\item \code{single.reweight}: Estimate the consensus structure using subject responses, reweighted by mean graph correlation.  This corresponds to an ``expertise-weighted vote'' notion of consensus.
\item \code{PCA.reweight}: Estimate the consensus using the (scores on the) first component of a network PCA.  This corresponds to a ``shared theme'' or ``common element'' notion of consensus.
\item \code{LAS.intersection}: Estimate the concensus structure using the locally aggregated structure (intersection rule).  In this model, an i->j edge exists iff i \emph{and} j agree that it exists.
\item \code{LAS.union}: Estimate the concensus structure using the locally aggregated structure (union rule).  In this model, an i->j edge exists iff i \emph{or} j agree that it exists.
\item \code{OR.row}: Estimate the consensus structure using own report.  Here, we take each informant's outgoing tie reports to be correct.
\item \code{OR.col}: Estimate the consensus structure using own report.  Here, we take each informant's incoming tie reports to be correct.
}

Note that the reweighted algorithms are not dichotomized by default; since these return valued graphs, dichotomization may be desirable prior to use.

It should be noted that a model for estimating an underlying criterion structure from multiple informant reports is provided in \code{\link{bbnam}}; if your goal is to reconstruct an ``objective'' network from informant reports, this may prove more useful.
}
\value{
An adjacency matrix representing the consensus structure
}
\references{ 
Banks, D.L., and Carley, K.M.  (1994).  ``Metric Inference for Social Networks.''  \emph{Journal of Classification,}  11(1), 121-49.

Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Inter-Structural Analysis.''  CASOS Working Paper, Carnegie Mellon University.

Krackhardt, D.  (1987).  ``Cognitive Social Structures.'' \emph{Social Networks,} 9, 109-134. 
}
\author{ Carter T. Butts \email{buttsc@uci.edu} }

\note{ Eventually, this routine will also support the (excellent) consensus methods of Romney and Batchelder; since these are similar in many respects to the \code{\link{bbnam}} model, users may wish to try this alternative for now. 
}

\seealso{ \code{\link{bbnam}}, \code{\link{centralgraph}} }

\examples{

#Generate some test data
g<-rgraph(5)
g.pobs<-g*0.9+(1-g)*0.5
g.obs<-rgraph(5,5,tprob=g.pobs)

#Find some consensus structures
consensus(g.obs)                           #Central graph
consensus(g.obs,method="single.reweight")  #Single reweighting
consensus(g.obs,method="PCA.reweight")     #1st component in network PCA
}
\keyword{ graphs }%-- one or more ...

\eof
\name{cugtest}
\alias{cugtest}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{Perform Conditional Uniform Graph (CUG) Hypothesis Tests for Graph-Level Indices}
\description{
 \code{cugtest} tests an arbitrary GLI (computed on \code{dat} by \code{FUN}) against a conditional uniform graph null hypothesis, via Monte Carlo simulation of likelihood quantiles.  Some variation in the nature of the conditioning is available; currently, conditioning only on size, conditioning jointly on size and estimated tie probability (via density), and conditioning jointly on size and (bootstrapped) edge value distributions are implemented.  Note that fair amount of flexibility is possible regarding CUG tests on functions of GLIs (Anderson et al., 1999).  See below for more details. 
}
\usage{
cugtest(dat, FUN, reps=1000, gmode="digraph", cmode="density", 
    diag=FALSE, g1=1, g2=2, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors.  Provided that \code{FUN} is well-behaved, this can be an n x n matrix if only one graph is involved. }
  \item{FUN}{ Function to compute GLIs, or functions thereof.  \code{FUN} must accept \code{dat} and the specified \code{g} arguments, and should return a real number. }
  \item{reps}{ Integer indicating the number of draws to use for quantile estimation.  Note that, as for all Monte Carlo procedures, convergence is slower for more extreme quantiles.  By default, \code{reps==1000}. }
  \item{gmode}{ String indicating the type of graph being evaluated.  "digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{gmode} is set to "digraph" by default. }
  \item{cmode}{ String indicating the type of conditioning assumed by the null hypothesis.  If \code{cmode} is set to "density", then the density of the graph in question is used to determine the tie probabilities of the Bernoulli graph draws (which are also conditioned on |V(G)|).  If\code{cmode=="ties"}, then draws are bootstrapped from the distribution of edge values within the data matrices.  If \code{cmode="order"}, then draws are uniform over all graphs of the same order (size) as the graphs within the input stack.  By default, \code{cmode} is set to \code{"density"}.}
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{g1}{ Integer indicating the index of the first graph input to the GLI.  By default, \code{g1==1}. }
  \item{g2}{ Integer indicating the index of the second graph input to the GLI.  (\code{FUN} can ignore this, if one wishes to test the GLI value of a single graph, but it should recognize the argument.)  By default, \code{g2==2}. }
  \item{\dots}{ Additional arguments to \code{FUN}. }
}
\details{
  The null hypothesis of the CUG test is that the observed GLI (or function thereof) was drawn from a distribution isomorphic to that of said GLI evaluated (uniformly) on the space of all graphs conditional on one or more features.  The most common "features" used for conditioning purposes are order (size) and density, both of which are known to have strong and nontrivial effects on other GLIs (Anderson et al., 1999) and which are, in many cases, exogenously determined.  Since theoretical results regarding functions of arbitrary GLIs on the space of graphs are not available, the standard approach to CUG testing is to approximate the quantiles of the likelihood associated with the null hypothesis using Monte Carlo methods.  This is the technique utilized by \code{cugtest}, which takes appropriately conditioned draws from the set of graphs and computes on them the GLI specified in \code{FUN}, thereby accumulating an approximation to the true likelihood.

The \code{cugtest} procedure returns a \code{cugtest} object containing the estimated likelihood (distribution of the test GLI under the null hypothesis), the observed GLI value of the data, and the one-tailed p-values (estimated quantiles) associated with said observation.  As usual, the (upper tail) null hypothesis is rejected for significance level alpha if p>=observation is less than alpha (or p<=observation, for the lower tail); if the hypothesis is undirected, then one rejects if either p<=observation or p>=observation is less then alpha/2.  Standard caveats regarding the use of null hypothesis testing procedures are relevant here: in particular, bear in mind that a significant result does not necessarily imply that the likelihood ratio of the null model and the alternative hypothesis favors the latter.

Informative and aesthetically pleasing portrayals of \code{cugtest} objects are available via the \code{\link{print.cugtest}} and \code{\link{summary.cugtest}} methods.  The \code{\link{plot.cugtest}} method displays the estimated distribution, with a reference line signifying the observed value.
}
\value{
An object of class \code{cugtest}, containing

\item{testval}{
The observed GLI value.
}
\item{dist}{
A vector containing the Monte Carlo draws. 
}
\item{pgreq}{
The proportion of draws which were greater than or equal to the observed GLI value.
}
\item{pleeq}{
The proportion of draws which were less than or equal to the observed GLI value.
}

}
\references{ 
Anderson, B.S.; Butts, C.T.; and Carley, K.M. (1999). ``The Interaction of Size and Density with Graph-Level Indices.'' \emph{Social Networks}, 21(3), 239-267.
 }
\author{ Carter T. Butts \email{buttsc@uci.edu} }

\seealso{ \code{\link{qaptest}}, \code{\link{gliop}} }

\examples{
#Draw two random graphs, with different tie probabilities
dat<-rgraph(20,2,tprob=c(0.2,0.8))
#Is their correlation higher than would be expected, conditioning 
#only on size?
cug<-cugtest(dat,gcor,cmode="order")
summary(cug)
#Now, let's try conditioning on density as well.
cug<-cugtest(dat,gcor)
summary(cug)
}
\keyword{htest}
\keyword{ math }

\eof
\name{degree}
\alias{degree}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute the Degree Centrality Scores of Network Positions }
\description{
  \code{Degree} takes a graph stack (\code{dat}) and returns the degree centralities of positions within one graph (indicated by \code{nodes} and \code{g}, respectively).  Depending on the specified mode, indegree, outdegree, or total (Freeman) degree will be returned; this function is compatible with \code{\link{centralization}}, and will return the theoretical maximum absolute deviation (from maximum) conditional on size (which is used by \code{\link{centralization}} to normalize the observed centralization score).
}
\usage{
degree(dat, g=1, nodes=c(1:dim(dat)[2]), gmode="digraph", diag=FALSE,
    tmaxdev=FALSE, cmode="freeman", rescale=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors.  Provided that \code{FUN} is well-behaved, this can be an n x n matrix if only one graph is involved. }
  \item{g}{ Integer indicating the index of the graph for which centralities are to be calculated.  By default, \code{g==1}. }
  \item{nodes}{ List indicating which nodes are to be included in the calculation.  By default, all nodes are included. }
  \item{gmode}{ String indicating the type of graph being evaluated.  "digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{gmode} is set to "digraph" by default. }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{tmaxdev}{ Boolean indicating whether or not the theoretical maximum absolute deviation from the maximum nodal centrality should be returned.  By default, \code{tmaxdev==FALSE}. }
  \item{cmode}{ String indicating the type of degree centrality being computed.  "indegree", "outdegree", and "freeman" refer to the indegree, outdegree, and total (Freeman) degree measures, respectively.  The default for \code{cmode} is "freeman". }
  \item{rescale}{ If true, centrality scores are rescaled such that they sum to 1. }
}
\details{
  Degree centrality is the social networker's term for various permutations of the graph theoretic notion of vertex degree: indegree of a vertex, v, corresponds to the cardinality of the vertex set \eqn{N^+(v)=\{i \in V(G) : (i,v) \in E(G)\}}{N^+(v) = \{i in V(G) : (i,v) in E(G)\}}; outdegree corresponds to the cardinality of the vertex set \eqn{N^-(v)=\{i \in V(G) : (v,i) \in E(G)\}}{N^-(v) = \{i in V(G) : (v,i) in E(G)\}}; and total (or "Freeman") degree corresponds to \eqn{\left|N^+(v)\right| + \left|N^-(v)\right|}{|N^+(v)|+|N^-(v)|}.  (Note that, for simple graphs, indegree=outdegree=total degree/2.)  Obviously, degree centrality can be interpreted in terms of the sizes of actors' neighborhoods within the larger structure.  See the references below for more details.
}
\value{
  A vector containing the degree centrality scores
}
\references{ Freeman, L.C.  (1979).  ``Centrality in Social Networks I: Conceptual Clarification.'' \emph{Social Networks}, 1, 215-239. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }


\seealso{ \code{\link{centralization}} }

\examples{
#Create a random directed graph
dat<-rgraph(10)
#Find the indegrees, outdegrees, and total degrees
degree(dat,cmode="indegree")
degree(dat,cmode="outdegree")
degree(dat)
}
\keyword{univar}
\keyword{ math }

\eof
\name{diag.remove}
\alias{diag.remove}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Remove the Diagonals of Adjacency Matrices in a Graph Stack }
\description{
  Returns the input graph stack, with the diagonal entries removed/replaced as indicated.
}
\usage{
diag.remove(dat, remove.val=NA)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph stack }
  \item{remove.val}{ The value with which to replace the existing diagonals }
}
\details{
  \code{diag.remove} is simply a convenient way to apply \code{\link{diag}} to an entire stack of adjacency matrices at once.
}
\value{
The updated graph stack
}

\author{ Carter T. Butts \email{buttsc@uci.edu} }


\seealso{ \code{\link{diag}}, \code{\link{upper.tri.remove}}, \code{\link{lower.tri.remove}} }

\examples{
#Generate a random graph stack
g<-rgraph(3,5)
#Remove the diagonals
g<-diag.remove(g)
}
\keyword{ manip }%-- one or more ...
\keyword{ math }

\eof
\name{dyad.census}
\alias{dyad.census}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute a Holland and Leinhardt MAN Dyad Census }
\description{
  \code{dyad.census} computes a Holland and Leinhardt dyad census on the graphs of \code{dat} selected by \code{g}.
}
\usage{
dyad.census(dat, g=1:stackcount(dat))
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph or graph stack }
  \item{g}{ The elements of \code{dat} to be included; by default, all graphs are processed. }
}
\details{
Each dyad in a directed graph may be in one of four states: the null state (\eqn{a \not\leftrightarrow b}{a <-!-> b}), the complete or mutual state (\eqn{a \leftrightarrow b}{a <--> b}), and either of two asymmetric states (\eqn{a \leftarrow b}{a --> b} or \eqn{a \rightarrow b}{a <-- b}).  Holland and Leinhardt's dyad census classifies each dyad into the mutual, asymmetric, or null categories, counting the number of each within the digraph.  These counts can be used as the basis for null hypothesis tests (since their distributions are known under assumptions such as constant edge probability), or for the generation of random graphs (e.g., via the U|MAN distribution, which conditions on the numbers of mutual, asymmetric, and null dyads in each graph).  
}
\value{
A matrix whose three columns contain the counts of mutual, asymmetric, and null dyads (respectively) for each graph
}
\references{ Holland, P.W. and Leinhardt, S.  (1970).  ``A Method for Detecting Structure in Sociometric Data.''  \emph{American Journal of Sociology}, 70, 492-513. 

Wasserman, S., and Faust, K.  (1994).  ``Social Network Analysis: Methods and Applications.''  Cambridge: Cambridge University Press.}
\author{ Carter T. Butts \email{buttsc@uci.edu} }

\seealso{ \code{\link{mutuality}}, \code{\link{grecip}}, \code{\link{triad.census}} }

\examples{
#Generate a dyad census of random data with varying densities
dyad.census(rgraph(15,5,tprob=c(0.1,0.25,0.5,0.75,0.9)))
}
\keyword{ math }% __ONLY ONE__ keyword per line

\eof
\name{efficiency}
\alias{efficiency}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute Graph Efficiency Scores }
\description{
  \code{efficiency} takes a graph stack (\code{dat}) and returns the Krackhardt efficiency scores for the graphs selected by \code{g}.
}
\usage{
efficiency(dat, g=1:stackcount(dat), diag=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph or graph stack }
  \item{g}{ Index values for the graphs to be utilized; by default, all graphs are selected }
  \item{diag}{ \code{TRUE} if the diagonal contains valid data; by default, \code{diag==FALSE} }
}
\details{
Let \eqn{G=\{G_1,G_2,\dots,G_n\}}{G=\{G_1,G_2,...,G_n\}} be a digraph with weak components \eqn{G_1,G_2,\dots,G_n}{G_1,G_2,...,G_n}.  For convenience, we denote the cardinalities of these graphs' vertex sets by \eqn{|V(G)|=N} and \eqn{|V(G_i)|=N_i}, \eqn{\forall i \in 1,\dots,n}{for i in 1,...,n}.  Then the Krackhardt efficiency of \eqn{G} is given by

\deqn{
1-\frac{|E(G)| - \sum_{i=1}^n \left(N_i-1\right)}{\frac{N^2-N}{2}\sum_{i=1}^n \left(N_i-1\right)}}{%
1 - ( |E(G)| - Sum(N_i-1,i=1,..,n) )/( (N^2-N)/2 - Sum(N_i-1,i=1,..,n) )}

which can be interpreted as 1 minus the proportion of possible ``extra'' edges (above those needed to connect the existing components) actually present in the graph.  A graph which an efficiency of 1 has precisely as many edges as are needed to connect its components; as additional edges are added, efficiency gradually falls towards 0.

Efficiency is one of four measures (\code{\link{connectedness}}, \code{\link{efficiency}}, \code{\link{hierarchy}}, and \code{\link{lubness}}) suggested by Krackhardt for summarizing hierarchical structures.  Each corresponds to one of four axioms which are necessary and sufficient for the structure in question to be an outtree; thus, the measures will be equal to 1 for a given graph iff that graph is an outtree.  Deviations from unity can be interpreted in terms of failure to satisfy one or more of the outtree conditions, information which may be useful in classifying its structural properties.
}
\value{
A vector of efficiency scores
}
\references{ Krackhardt, David.  (1994).  ``Graph Theoretical Dimensions of Informal Organizations.'' In K. M. Carley and M. J. Prietula (Eds.), \emph{Computational Organization Theory}, 89-111. Hillsdale, NJ: Lawrence Erlbaum and Associates. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ The four Krackhardt indices are, in general, nondegenerate for a relatively narrow band of size/density combinations (efficiency being the sole exception).  This is primarily due to their dependence on the reachability graph, which tends to become complete rapidly as size/density increase.  See Krackhardt (1994) for a useful simulation study. }


\seealso{ \code{\link{connectedness}}, \code{\link{efficiency}}, \code{\link{hierarchy}}, \code{\link{lubness}}, \code{\link{gden}} }

\examples{
#Get efficiency scores for graphs of varying densities
efficiency(rgraph(10,5,tprob=c(0.1,0.25,0.5,0.75,0.9)))
}
\keyword{ math }% at least one, from doc/KEYWORDS
\keyword{ univar }% __ONLY ONE__ keyword per line

\eof
\name{equiv.clust}
\alias{equiv.clust}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find Clusters of Positions Based on an Equivalence Relation }
\description{
  \code{equiv.clust} uses a definition of approximate equivalence (\code{equiv.fun}) to form a hierarchical clustering of network positions.  Where \code{dat} consists of multiple relations, all specified relations are considered jointly in forming the equivalence clustering.
}
\usage{
equiv.clust(dat, g=c(1:dim(dat)[1]), equiv.fun="sedist", 
   method="hamming", mode="digraph", diag=FALSE, 
   cluster.method="complete", glabels=dimnames(dat)[[1]][g], 
   plabels=dimnames(dat)[[2]], ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph adjacency matrix, or stack thereof }
  \item{g}{ The elements of \code{dat} to use in clustering the vertices; by default, all structures are used }
  \item{equiv.fun}{ The distance function to use in clustering vertices (defaults to \code{\link{sedist}}) }
  \item{method}{ \code{method} parameter to be passed to \code{equiv.fun} }
  \item{mode}{ ``graph'' or ``digraph,'' as appropriate }
  \item{diag}{ A boolean indicating whether or not matrix diagonals (loops) should be interpreted as useful data }
  \item{cluster.method}{ The hierarchical clustering method to use (see \code{\link[mva]{hclust}} in package \code{mva}) }
  \item{glabels}{ Labels for the various graphs in \code{dat} }
  \item{plabels}{ Labels for the vertices of \code{dat} }
  \item{\dots}{ Additional arguments to \code{equiv.dist} }
}
\details{
This routine is essentially a joint front-end to \code{\link{hclust}} and various positional distance functions, though it defaults to structural equivalence in particular.  Taking the specified graphs as input, \code{equiv.clust} computes the distances between all pairs of positions using \code{equiv.dist}, and then performs a cluster analysis of the result.  The return value is an object of class \code{equiv.clust}, for which various secondary analysis methods exist.
}
\value{
An object of class \code{equiv.clust}
}
\references{ 
Breiger, R.L.; Boorman, S.A.; and Arabie, P.  (1975).  ``An Algorithm for Clustering Relational Data with Applications to Social Network Analysis and Comparison with Multidimensional Scaling.''  \emph{Journal of Mathematical Psychology}, 12, 328-383.

Burt, R.S.  (1976).  ``Positions in Networks.''  \emph{Social Forces}, 55, 93-122.

Wasserman, S., and Faust, K.  \emph{Social Network Analysis: Methods and Applications.}  Cambridge: Cambridge University Press. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ See \code{\link{sedist}} for an example of a distance function compatible with \code{equiv.clust}. }

\section{Requires}{\code{mva}}

\seealso{ \code{\link{sedist}}, \code{\link{blockmodel}}}

\examples{

#Create a random graph with _some_ edge structure
g.p<-sapply(runif(20,0,1),rep,20)  #Create a matrix of edge 
                                   #probabilities
g<-rgraph(20,tprob=g.p)            #Draw from a Bernoulli graph 
                                   #distribution

#Cluster based on structural equivalence
eq<-equiv.clust(g)
plot(eq)
}
\keyword{ cluster }%-- one or more ...
\keyword{ math }

\eof
\name{eval.edgeperturbation}
\alias{eval.edgeperturbation}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute the Effects of Single-Edge Perturbations on Structural Indices }
\description{
  Evaluates a given function on an input graph with and without a specified edge, returning the difference between the results in each case.
}
\usage{
eval.edgeperturbation(dat, i, j, FUN, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A single adjacency matrix }
  \item{i}{ The row(s) of the edge(s) to be perturbed }
  \item{j}{ The column(s) of the edge(s) to be perturbed }
  \item{FUN}{ The function to be computed }
  \item{\dots}{ Additional arguments to \code{FUN} }
}
\details{
Although primarily a back-end utility for \code{\link{pstar}}, \code{eval.edgeperturbation} may be useful in any circumstance in which one wishes to assess the stability of a given structural index with respect to single edge perturbations. The function to be evaluated is calculated first on the input graph with all marked edges set to present, and then on the same graph with said edges absent.  (Obviously, this is sensible only for dichotomous data.)  The difference is then returned.  

In \code{\link{pstar}}, calls to \code{eval.edgeperturbation} are used to construct a perturbation effect matrix for the GLM. 
}
\value{
The difference in the values of \code{FUN} as computed on the perturbed graphs.
}
\references{ Anderson, C.; Wasserman, S.; and Crouch, B. (1999).  ``A p* Primer:  Logit Models for Social Networks.  \emph{Social Networks,} 21,37-66. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ \code{length(i)} and \code{length(j)} must be equal; where multiple edges are specified, the row and column listings are interpreted as pairs. }

\seealso{ \code{\link{pstar}} }

\examples{

#Create a random graph
g<-rgraph(5)

#How much does a one-edge change affect reciprocity?
eval.edgeperturbation(g,1,2,grecip)
}
\keyword{ math }%-- one or more ...

\eof
\name{evcent}
\alias{evcent}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find Eigenvector Centrality Scores of Network Positions }
\description{
  \code{evcent} takes a graph stack (\code{dat}) and returns the eigenvector centralities of positions within one graph (indicated by \code{nodes} and \code{g}, respectively).  This function is compatible with \code{\link{centralization}}, and will return the theoretical maximum absolute deviation (from maximum) conditional on size (which is used by \code{\link{centralization}} to normalize the observed centralization score).
}
\usage{
evcent(dat, g=1, nodes=c(1:dim(dat)[2]), gmode="digraph", diag=FALSE,
    tmaxdev=FALSE, rescale=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors. Alternately, this can be an n x n matrix (if only one graph is involved). }
  \item{g}{ Integer indicating the index of the graph for which centralities are to be calculated.  By default, \code{g==1}. }
  \item{nodes}{ List indicating which nodes are to be included in the calculation.  By default, all nodes are included. }
  \item{gmode}{ String indicating the type of graph being evaluated.  "digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  This is currently ignored. }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{tmaxdev}{ Boolean indicating whether or not the theoretical maximum absolute deviation from the maximum nodal centrality should be returned.  By default, \code{tmaxdev==FALSE}. }
  \item{rescale}{ If true, centrality scores are rescaled such that they sum to 1. }
}
\details{
Eigenvector centrality scores correspond to the values of the first eigenvector of the graph adjacency matrix; these scores may, in turn, be interpreted as arising from a reciprocal process in which the centrality of each actor is proportional to the sum of the centralities of those actors to whom he or she is connected.  In general, vertices with high eigenvector centralities are those which are connected to many other vertices which are, in turn, connected to many others (and so on).  (The perceptive may realize that this implies that the largest values will be obtained by individuals in large cliques (or high-density substructures).  This is also intelligible from an algebraic point of view, with the first eigenvector being closely related to the best rank-1 approximation of the adjacency matrix (a relationship which is easy to see in the special case of a diagonalizable symmetric real matrix via the \eqn{S \Lambda S^{-1}}{SLS^-1} decomposition).) 

The simple eigenvector centrality is generalized by the Bonacich power centrality measure; see \code{\link{bonpow}} for more details.
}
\value{
  A vector containing the centrality scores
}
\references{ Bonacich, P.  (1987).  ``Power and Centrality: A Family of Measures.'' \emph{American Journal of Sociology}, 92, 1170-1182.

Katz, L.  (1953).  ``A New Status Index Derived from Sociometric Analysis.''  \emph{Psychometrika}, 18, 39-43.
 }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ The theoretical maximum deviation used here is not obtained with the star network, in general.  For symmetric data, the maximum occurs for an empty graph with one complete dyad; the maximum deviation for asymmetric data is generated by the outstar.  UCINET V seems not to adjust for this fact, which can cause some oddities in their centralization scores (and results in a discrepancy in centralizations between the two packages). }

\section{WARNING }{\code{evcent} will not symmetrize your data before extracting eigenvectors; don't send this routine asymmetric matrices unless you really mean to do so.}

\seealso{ \code{\link{centralization}}, \code{\link{bonpow}} }

\examples{

#Generate some test data
dat<-rgraph(10,mode="graph")
#Compute eigenvector centrality scores
evcent(dat)
}
\keyword{ univar }%-- one or more ...
\keyword{ math }

\eof
\name{event2dichot}
\alias{event2dichot}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Convert an Observed Event Matrix to a Dichotomous matrix }
\description{
  Given a valued adjacency matrix (possibly derived from observed interaction ``events''), \code{event2dichot} returns a dichotomous adjacency matrix.
}
\usage{
event2dichot(m, method="quantile", thresh=0.5, leq=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{m}{ An adjacency matrix }
  \item{method}{ One of ``quantile,'' ``rquantile,'' ``cquantile,'' ``mean,'' ``rmean,'' ``cmean,'' ``absolute,'' ``rank,'' ``rrank,'' or ``crank''  }
  \item{thresh}{ Dichotomization thresholds for ranks or quantiles }
  \item{leq}{ Boolean indicating whether values less than or equal to the threshold should be taken as existing edges; the alternative is to use values strictly greater than the threshold }
}
\details{
The methods used for choosing dichotomization thresholds are as follows:

\enumerate{
\item quantile: specified quantile over the distribution of all edge values

\item rquantile: specified quantile by row

\item cquantile: specified quantile by column

\item mean: grand mean

\item rmean: row mean

\item cmean: column mean

\item absolute: the value of \code{thresh} itself

\item rank: specified rank over the distribution of all edge values

\item rrank: specified rank by row

\item crank: specified rank by column
}

Note that when a quantile, rank, or value is said to be ``specified,'' this refers to the value of \code{thresh}.  
}
\value{
The dichotomized data matrix
}
\references{ ~put references to the literature/web site here ~ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

%\seealso{ }

\examples{
#Draw a matrix of normal values
n<-matrix(rnorm(25),nrow=5,ncol=5)

#Dichotomize by the mean value
event2dichot(n,"mean")

#Dichotomize by the 0.95 quantile
event2dichot(n,"quantile",0.95)

}
\keyword{ manip }%-- one or more ...
\keyword{ math }

\eof
\name{gapply}
\alias{gapply}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Apply Functions Over Vertex Neighborhoods }
\description{
  Returns a vector or array or list of values obtained by applying a function to vertex neighborhoods of a given order.
}
\usage{
gapply(X, MARGIN, STATS, FUN, ..., mode = "digraph", diag = FALSE, 
    distance = 1, thresh = 0, simplify = TRUE)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{X}{ the adjacency matrix to be used. }
  \item{MARGIN}{ a vector giving the ``margin'' of \code{X} to be used in calculating neighborhoods.  1 indicates rows (out-neighbors), 2 indicates columns (in-neighbors), and c(1,2) indicates rows and columns (total neighborhood). }
  \item{STATS}{ the vector or matrix of vertex statistics to be used. }
  \item{FUN}{ the function to be applied.  In the case of operators, the function name must be quoted. }
  \item{\dots}{ additional arguments to \code{FUN}. }
  \item{mode}{ \code{"graph"} if \code{X} is a simple graph, else \code{"digraph"}. }
  \item{diag}{ boolean; are the diagonals of \code{X} meaningful? }
  \item{distance}{ the maximum geodesic distance at which neighborhoods are to be taken.  1 signifies first-order neighborhoods, 2 signifies second-order neighborhoods, etc. }
  \item{thresh}{ the threshold to be used in dichotomizing \code{X}. }
  \item{simplify}{ boolean; should we attempt to coerce output to a vector if possible? }
}
\details{
  For each vertex in \code{X}, \code{gapply} first identifies all members of the relevant neighborhood (as determined by \code{MARGIN} and \code{distance}) and pulls the rows of \code{STATS} associated with each.  \code{FUN} is then applied to this collection of values.  This provides a very quick and easy way to answer questions like:
\itemize{
\item How many persons are in each ego's 3rd-order neighborhood?
\item What fraction of each ego's alters are female?
\item What is the mean income for each ego's trading partners?
\item etc.
}
With clever use of \code{FUN} and \code{STATS}, a wide range of functionality can be obtained.
}
\value{
The result of the iterated application of \code{FUN} to each vertex neighborhood's \code{STATS}.
}
%\references{ ~put references to the literature/web site here ~ }
\author{ Carter T. Butts \email{buttsc@uci.edu}}
%\note{ ~~further notes~~ }
%
% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{ \code{\link{apply}}, \code{\link{sapply}} }
\examples{
#Generate a random graph
g<-rgraph(6)

#Calculate the degree of g using gapply
all(gapply(g,1,rep(1,6),sum)==degree(g,cmode="outdegree"))
all(gapply(g,2,rep(1,6),sum)==degree(g,cmode="degree"))
all(gapply(g,c(1,2),rep(1,6),sum)==degree(symmetrize(g),cmode="freeman")/2)

#Find first and second order neighborhood means on some variable
gapply(g,c(1,2),1:6,mean)
gapply(g,c(1,2),1:6,mean,distance=2)

}
\keyword{ iteration }% at least one, from doc/KEYWORDS
\keyword{ manip }% __ONLY ONE__ keyword per line

\eof
\name{gclust.boxstats}
\alias{gclust.boxstats}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Plot Statistics Associated with Graph Clusters }
\description{
  \code{gclust.boxstats} creates side-by-side boxplots of graph statistics based on a hierarchical clustering of networks (cut into \code{k} sets).
}
\usage{
gclust.boxstats(h, k, meas, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{h}{ An \code{\link[mva]{hclust}} object, presumably formed by clustering a set of structural distances }
  \item{k}{ The number of groups to evaluate }
  \item{meas}{ A vector of length equal to the number of graphs in \code{h}, containing a GLI to be evaluated }
  \item{\dots}{ Additional parameters to \code{\link{boxplot}} }
}
\details{
  \code{gclust.boxstats} simply takes the \code{\link[mva]{hclust}} object in \code{h}, applies \code{\link[mva]{cutree}} to form \code{k} groups, and then uses \code{\link{boxplot}} on the distribution of \code{meas} by group.  This can be quite handy for assessing graph clusters.
}
\value{
None
}
\references{ Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Interstructural Analysis.''  CASOS working paper, Carnegie Mellon University. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ Actually, this function will work with any \code{\link{hclust}} object and measure matrix; the data need not originate with social networks.  For this reason, the clever may also employ this function in conjunction with \code{\link{sedist}} or \code{\link{equiv.clust}} to plot NLIs against clusters of positions within a graph.}

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\section{Requires}{\code{mva}}

\seealso{ \code{\link{gclust.centralgraph}}, \code{\link{gdist.plotdiff}}, \code{\link{gdist.plotstats}}  }

\examples{
#Create some random graphs
g<-rgraph(10,20,tprob=c(rbeta(10,15,2),rbeta(10,2,15)))

#Find the Hamming distances between them
g.h<-hdist(g)

#Cluster the graphs via their Hamming distances
library(mva)     #Load the mva library
g.c<-hclust(as.dist(g.h))

#Now display boxplots of density by cluster for a two cluster solution
gclust.boxstats(g.c,2,gden(g))
}
\keyword{ hplot }%-- one or more ...

\eof
\name{gclust.centralgraph}
\alias{gclust.centralgraph}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Get Central Graphs Associated with Graph Clusters }
\description{
  Calculates central graphs associated with particular graph clusters (as indicated by the \code{k} partition of \code{h}).
}
\usage{
gclust.centralgraph(h, k, mats, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{h}{ An \code{\link[mva]{hclust}} object, the based on a graph set }
  \item{k}{ The number of groups to evaluate }
  \item{mats}{ A graph stack containing the adjacency matrices for the graphs on which the clustering was performed }
  \item{\dots}{ Additional arguments to \code{\link{centralgraph}} }
}
\details{
\code{gclust.centralgraph} uses \code{\link[mva]{cutree}} to cut the hierarchical clustering in \code{h} into \code{k} groups.  \code{\link{centralgraph}} is then called on each cluster, and the results are returned as a graph stack.  This is a useful tool for interpreting clusters of (labeled) graphs, with the resulting central graphs being subsequently analyzed using standard SNA methods.
}
\value{
An array containing the stack of central graph adjacency matrices
}
\references{ Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Interstructural Analysis.''  CASOS working paper, Carnegie Mellon University. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{hclust}}, \code{\link{centralgraph}}, \code{\link{gclust.boxstats}}, \code{\link{gdist.plotdiff}}, \code{\link{gdist.plotstats}} }

\examples{
#Create some random graphs
g<-rgraph(10,20,tprob=c(rbeta(10,15,2),rbeta(10,2,15)))

#Find the Hamming distances between them
g.h<-hdist(g)

#Cluster the graphs via their Hamming distances
library(mva)          #Load the mva library
g.c<-hclust(as.dist(g.h))

#Now find central graphs by cluster for a two cluster solution
g.cg<-gclust.centralgraph(g.c,2,g)

#Plot the central graphs
gplot(g.cg[1,,])
gplot(g.cg[2,,])
}
\keyword{ cluster }%-- one or more ...
\keyword{ math }

\eof
\name{gcor}
\alias{gcor}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ 
Find the (Product-Moment) Correlation Between Two or More Labeled Graphs }
\description{
 \code{gcor} finds the product-moment correlation between the adjacency matrices of graphs indicated by \code{g1} and \code{g2} in stack \code{dat} (or possibly \code{dat2}).  Missing values are permitted. 
}
\usage{
gcor(dat, dat2=NULL, g1=c(1:dim(dat)[1]), g2=c(1:dim(dat)[1]), 
    diag=FALSE, mode="digraph")
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph stack }
  \item{dat2}{ Optionally, a second graph stack }
  \item{g1}{ The indices of \code{dat} reflecting the first set of graphs to be compared; by default, all members of \code{dat} are included }
  \item{g2}{ The indices or \code{dat} (or \code{dat2}, if applicable) reflecting the second set of graphs to be compared; by default, all members of \code{dat} are included }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{mode}{ String indicating the type of graph being evaluated.  "Digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{mode} is set to "digraph" by default. }
}
\details{

The (product moment) graph correlation between labeled graphs G and H is given by 
\deqn{cor(G,H) = \frac{cov(G,H)}{\sqrt{cov(G,G) cov(H,H)}} }{%
cor(G,H) = cov(G,V)/sqrt(cov(G,G)cov(H,H))}
where the graph covariance is defined as
\deqn{cov(G,H) = \frac{1}{{|V| \choose 2}} \sum_{\{i,j\}} \left(A^G_{ij}-\mu_G\right)\left(A^H_{ij}-\mu_H\right)}{%
cov(G,H) = sum( (A^G_ij-mu_G)(A^H_ij-mu_H), \{i,j\} )/Choose(|V|,2)}
(with \eqn{A^G}{A^G} being the adjacency matrix of G).  The graph correlation/covariance is at the center of a number of graph comparison methods, including network variants of regression analysis, PCA, CCA, and the like.

Note that \code{gcor} computes only the correlation between \emph{uniquely labeled} graphs.  For the more general case, \code{\link{gscor}} is recommended.
}
\value{
A graph correlation matrix
}
\references{ Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Interstructural Analysis.''  CASOS Working Paper, Carnegie Mellon University.

Krackhardt, D.  (1987).  ``QAP Partialling as a Test of Spuriousness.''  \emph{Social Networks}, 9, 171-86
}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{  The \code{gcor} routine is really just a front-end to the standard \code{\link{cor}} method; the primary value-added is the transparent vectorization of the input graphs (with intelligent handling of simple versus directed graphs, diagonals, etc.).  As noted, the correlation coefficient returned is a standard Pearson's product-moment coefficient, and output should be interpreted accordingly.  Classical null hypothesis testing procedures are not recommended for use with graph correlations; for nonparametric null hypothesis testing regarding graph correlations, see \code{\link{cugtest}} and \code{\link{qaptest}}.  For multivariate correlations among graph sets, try \code{\link{netcancor}}. 
}

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{gscor}}, \code{\link{gcov}}, \code{\link{gscov}}  }

\examples{
#Generate two random graphs each of low, medium, and high density
g<-rgraph(10,6,tprob=c(0.2,0.2,0.5,0.5,0.8,0.8))

#Examine the correlation matrix
gcor(g)
}
\keyword{ univar }
\keyword{ multivariate }%-- one or more ...

\eof
\name{gcov}
\alias{gcov}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find the Covariance(s) Between Two or More Labeled Graphs}
\description{
   \code{gcov} finds the covariances between the adjacency matrices of graphs indicated by \code{g1} and \code{g2} in stack \code{dat} (or possibly \code{dat2}).  Missing values are permitted.
}
\usage{
gcov(dat, dat2=NULL, g1=c(1:dim(dat)[1]), g2=c(1:dim(dat)[1]), 
    diag=FALSE, mode="digraph")
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph stack }
  \item{dat2}{ Optionally, a second graph stack }
  \item{g1}{ The indices of \code{dat} reflecting the first set of graphs to be compared; by default, all members of \code{dat} are included }
  \item{g2}{ The indices or \code{dat} (or \code{dat2}, if applicable) reflecting the second set of graphs to be compared; by default, all members of \code{dat} are included }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{mode}{ String indicating the type of graph being evaluated.  "digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{mode} is set to "digraph" by default. }
}
\details{
  The graph covariance between two labeled graphs is defined as
\deqn{cov(G,H) = \frac{1}{{|V| \choose 2}} \sum_{\{i,j\}} \left(A^G_{ij}-\mu_G\right)\left(A^H_{ij}-\mu_H\right)}{%
cov(G,H) = sum( (A^G_ij-mu_G)(A^H_ij-mu_H), \{i,j\} )/Choose(|V|,2)}
(with \eqn{A^G}{A^G} being the adjacency matrix of G).  The graph correlation/covariance is at the center of a number of graph comparison methods, including network variants of regression analysis, PCA, CCA, and the like.

Note that \code{gcov} computes only the covariance between \emph{uniquely labeled} graphs.  For the more general case, \code{\link{gscov}} is recommended.
}
\value{
  A graph covariance matrix
}
\references{ Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Interstructural Analysis.''  CASOS Working Paper, Carnegie Mellon University. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ The \code{gcov} routine is really just a front-end to the standard \code{\link{cov}} method; the primary value-added is the transparent vectorization of the input graphs (with intelligent handling of simple versus directed graphs, diagonals, etc.).  Classical null hypothesis testing procedures are not recommended for use with graph covariance; for nonparametric null hypothesis testing regarding graph covariance, see \code{\link{cugtest}} and \code{\link{qaptest}}.
}

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{gscov}}, \code{\link{gcor}}, \code{\link{gscor}} }

\examples{
#Generate two random graphs each of low, medium, and high density
g<-rgraph(10,6,tprob=c(0.2,0.2,0.5,0.5,0.8,0.8))

#Examine the covariance matrix
gcov(g)
}
\keyword{ univar }
\keyword{ multivariate }%-- one or more ...

\eof
\name{gden}
\alias{gden}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find the Density of a Graph }
\description{
  \code{gden} computes the density of graph \code{g} in stack \code{dat}, adjusting for the type of graph in question.
}
\usage{
gden(dat, g=NULL, diag=FALSE, mode="digraph")
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors.  If \code{dat} is a matrix, \code{g} will be ignored. }
  \item{g}{ Integer indicating the index of the graphs for which the density is to be calculated.  If \code{g==NULL} (the default), density is calculated for all graphs in \code{dat}. }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{mode}{ String indicating the type of graph being evaluated.  "digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{mode} is set to "digraph" by default. }
}
\details{
  The density of a graph is here taken to be the sum of tie values divided by the number of possible ties (i.e., an unbiased estimator of the graph mean); hence, the result is interpretable for valued graphs as the mean tie value.  The number of possible ties is determined by the graph type (and by \code{diag}) in the usual fashion.
}
\value{
  The graph density
}
\references{ Wasserman, S., and Faust, K.  (1994).  ``Social Network Analysis: Methods and Applications.''  Cambridge: Cambridge University Press. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

%\seealso{  }

\examples{
#Draw three random graphs
dat<-rgraph(10,3)
#Find their densities
gden(dat)
}
\keyword{ univar }%-- one or more ...
\keyword{ math }

\eof
\name{gdist.plotdiff}
\alias{gdist.plotdiff}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Plot Differences in Graph-level Statistics Against Inter-graph Distances}
\description{
  For a given graph set, \code{gdist.plotdiff} plots the distances between graphs against their distances (or differences) on a set of graph-level measures.
}
\usage{
gdist.plotdiff(d, meas, method="manhattan", jitter=TRUE, 
    xlab="Inter-Graph Distance", ylab="Measure Distance", 
    lm.line=FALSE, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{d}{ A matrix containing the inter-graph distances }
  \item{meas}{ An n x m matrix containing the graph-level indices; rows of this matrix must correspond to graphs, and columns to indices }
  \item{method}{ The distance method used by \code{\link[mva]{dist}} to establish differences/distances between graph GLI values.  By default, absolute ("manhattan") differences are used. }
  \item{jitter}{ Should values be jittered prior to display? }
  \item{xlab}{ A label for the X axis }
  \item{ylab}{ A label for the Y axis }
  \item{lm.line}{ Include a least-squares line? }
  \item{\dots}{ Additional arguments to \code{\link{plot}} }
}
\details{
  \code{gdist.plotdiff} works by taking the distances between all graphs on \code{meas} and then plotting these distances against \code{d} for all pairs of graphs (with, optionally, an added least-squares line for reference value).  This can be a useful exploratory tool for relating inter-graph distances (e.g., Hamming distances) to differences on other attributes.  
}
\value{
None
}
\references{ Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Interstructural Analysis.''  CASOS working paper, Carnegie Mellon University. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ This function is actually quite generic, and can be used with node-level -- or even non-network -- data as well. }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\section{Requires}{\code{mva}}

\seealso{ \code{\link{gdist.plotstats}}, \code{\link{gclust.boxstats}}, \code{\link{gclust.centralgraph}} }

\examples{
#Generate some random graphs with varying densities
g<-rgraph(10,20,tprob=runif(20,0,1))

#Find the Hamming distances between graphs
g.h<-hdist(g)

#Plot the relationship between distance and differences in density
gdist.plotdiff(g.h,gden(g),lm.line=TRUE)
}
\keyword{ hplot }%-- one or more ...

\eof
\name{gdist.plotstats}
\alias{gdist.plotstats}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Plot Various Graph Statistics Over a Network MDS }
\description{
  Plots a two-dimensional metric MDS of \code{d}, with the corresponding values of \code{meas} indicated at each point.  Various options are available for controlling how \code{meas} is to be displayed.
}
\usage{
gdist.plotstats(d, meas, siz.lim=c(0, 0.15), rescale="quantile", 
    display.scale="radius", display.type="circleray", cex=0.5, pch=1,
    labels=NULL, pos=1, labels.cex=1, legend=NULL, legend.xy=NULL, 
    legend.cex=1, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{d}{ A matrix containing the inter-graph distances }
  \item{meas}{ An nxm matrix containing the graph-level measures; each row must correspond to a graph, and each column must correspond to an index }
  \item{siz.lim}{ The minimum and maximum sizes (respectively) of the plotted symbols, given as fractions of the total plotting range }
  \item{rescale}{ One of ``quantile'' for ordinal scaling, ``affine'' for max-min scaling, and ``normalize'' for rescaling by maximum value; these determine the scaling rule to be used in sizing the plotting symbols }
  \item{display.scale}{ One of ``area'' or ``radius''; this controls the attribute of the plotting symbol which is rescaled by the value of \code{meas}}
  \item{display.type}{ One of ``circle'', ``ray'', ``circleray'', ``poly'', or ``polyray''; this determines the type of plotting symbol used (circles, rays, polygons, or come combination of these) }
  \item{cex}{ Character expansion coefficient }
  \item{pch}{ Point types for the base plotting symbol (not the expanded symbols which are used to indicate \code{meas} values) }
  \item{labels}{ Point labels, if desired }
  \item{pos}{ Relative position of labels (see \code{\link{par}}) }
  \item{labels.cex}{ Character expansion factor for labels }
  \item{legend}{ Add a legend? }
  \item{legend.xy}{ x,y coordinates for legend }
  \item{legend.cex}{ Character expansion factor for legend }
  \item{\dots}{ Additional arguments to \code{\link{plot}} }
}
\details{
\code{gdist.plotstats} works by performing an MDS (using \code{\link[mva]{cmdscale}}) on \code{d}, and then using the values in \code{meas} to determine the shape of the points at each MDS coordinate.  Typically, these shapes involve rays of varying color and length indicating \code{meas} magnitude, with circles and polygons of the appropriate radius and/or error being options as well.  Various options are available (described above) to govern the details of the data display; some tinkering may be needed in order to produce an aesthetically pleasing visualization.

The primary use of \code{gdist.plotstats} is to explore broad relationships between graph properties and inter-graph distances.  This routine complements others in the \code{gdist} and \code{gclust} family of interstructural visualization tools.
}
\value{
None
}
\references{ Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Interstructural Analysis.'' CASOS working paper, Carnegie Mellon University. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ This routine does not actually depend on the data's being graphic in origin, and can be used with any distance matrix/measure matrix combination. }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\section{Requires}{\code{mva}}

\seealso{ \code{\link{gdist.plotdiff}}, \code{\link{gclust.boxstats}}, \code{\link{gclust.centralgraph}} }

\examples{
#Generate random graphs with varying density
g<-rgraph(10,20,tprob=runif(20,0,1))

#Get Hamming distances between graphs
g.h<-hdist(g)

#Plot the association of distance, density, and reciprocity
gdist.plotstats(g.h,cbind(gden(g),grecip(g)))
}
\keyword{ hplot }%-- one or more ...

\eof
\name{geodist}
\alias{geodist}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Fund the Numbers and Lengths of Geodesics Among Nodes in a Graph }
\description{
  \code{geodist} uses a BFS to find the number and lengths of geodesics between all nodes of \code{dat}.  Where geodesics do not exist, the value in \code{inf.replace} is substituted for the distance in question.
}
\usage{
geodist(dat, inf.replace=dim(dat)[2])
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ An adjacency matrix }
  \item{inf.replace}{ The value to use for geodesic distances between disconnected nodes; by default, this is equal to one greater than the maximum path length }
}
\details{
  This routine is used by a variety of other functions; many of these will allow the user to provide manually precomputed \code{geodist} output so as to prevent expensive recomputation.  Note that the choice of one greater than the maximum path length for disconnected vertex pairs is non-canonical, and some may prefer to simply treat these as missing values.  \code{geodist} (without loss of generality) treats all paths as directed, a fact which should be kept in mind when interpreting \code{geodist} output.
}
\value{
A data frame containing:

  \item{counts}{A matrix containing the number of geodesics between each pair of vertices}
  \item{sigma}{A matrix containing the geodesic distances between each pair of vertices}
}
\references{ 
Brandes, U.  (2000).  ``Faster Evaluation of Shortest-Path Based Centrality Indices.'' \emph{Konstanzer Schriften in Mathematik und Informatik}, 120.

West, D.B.  (1996).  \emph{Introduction to Graph Theory.}  Upper Saddle River, N.J.: Prentice Hall. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{component.dist}}, \code{\link{components}} }

\examples{
#Find geodesics on a random graph
gd<-geodist(rgraph(15))

#Examine the number of geodesics
gd$counts

#Examine the geodesic distances
gd$gdist
}
\keyword{ math }%-- one or more ...

\eof
\name{gliop}
\alias{gliop}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Return a Binary Operation on GLI Values Computed on Two Graphs }
\description{
  \code{gliop} is a wrapper which allows for an arbitrary binary operation on GLIs to be treated as a single call.  This is particularly useful for test routines such as \code{\link{cugtest}} and \code{\link{qaptest}}.
}
\usage{
gliop(dat, GFUN, OP="-", g1=1, g2=2, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph stack }
  \item{GFUN}{ A function taking single graphs as input }
  \item{OP}{ The operator to use on the output of \code{GFUN} }
  \item{g1}{ The index of the first input graph }
  \item{g2}{ The index of the second input graph }
  \item{\dots}{ Additional arguments to \code{GFUN} }
}
\details{
  \code{gliop} operates by evaluating \code{GFUN} on the graphs indexed by \code{g1} and \code{g2} and returning the result of \code{OP} as applied to the \code{GFUN} output.
}
\value{
\code{OP(GFUN(dat[g1, , ],...),GFUN(dat[g2, , ],...))}
}
\references{ Anderson, B.S.; Butts, C.T.; and Carley, K.M. (1999). ``The Interaction of Size and Density with Graph-Level Indices.'' \emph{Social Networks}, 21(3), 239-267. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ If the output of \code{GFUN} is not sufficiently well-behaved, undefined behavior may occur.  Common sense is advised. }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{cugtest}}, \code{\link{qaptest}} }

\examples{
#Draw two random graphs
g<-rgraph(10,2,tprob=c(0.2,0.5))

#What is their difference in density?
gliop(g,gden,"-",1,2)
}
\keyword{ utilities }%-- one or more ...
\keyword{ math }

\eof
\name{gplot}
\alias{gplot}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Two-Dimensional Visualization of Graphs }
\description{
  \code{gplot} produces a simple two-dimensional plot of graph \code{g} in stack \code{dat}.  A variety of options are available to control vertex placement, display details, color, etc.
}
\usage{
gplot(dat, g = 1, gmode = "digraph", diag = FALSE, label = c(1:dim(dat)[2]),
    coord = NULL, jitter = TRUE, thresh = 0, usearrows = TRUE, mode = "segeo",
    displayisolates = TRUE, interactive = FALSE, boxed.labels = TRUE, 
    xlab = NULL, ylab = NULL, pad = 0.2, vertex.pch = 19, label.cex = 1,
    vertex.cex = 1, label.col = 1, edge.col = 1, vertex.col = 1,
    arrowhead.length = 0.2, edge.lty = 1, edge.lwd = 0, edge.len = 0.5,
    edge.curve = 0.1, edge.steps = 50, diag.size = 0.025, uselen = FALSE,
    usecurve = FALSE, suppress.axes = TRUE, new = TRUE, layout.par = NULL, ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{dat}{ a graph or graph stack.  This data may be valued. }
  \item{g}{ integer indicating the index of the graph which is to be plotted.  By default, \code{g==1}. }
  \item{gmode}{ String indicating the type of graph being evaluated.  \code{"digraph"} indicates that edges should be interpreted as directed; \code{"graph"} indicates that edges are undirected; \code{"twomode"} indicates that data should be interpreted as bimodal (i.e., rows and columns are distinct vertex sets).  \code{gmode} is set to \code{"digraph"} by default. }
  \item{diag}{ boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{label}{ a vector of vertex labels, if desired; defaults to the vertex index number. }
  \item{coord}{ user-specified vertex coordinates, in an NCOL(dat)x2 matrix.  Where this is specified, it will override the \code{mode} setting. }
  \item{jitter}{ boolean; should the output be jittered? }
  \item{thresh}{ real number indicating the lower threshold for tie values.  Only ties of value >\code{thresh} are displayed.  By default, \code{thresh}=0.}
  \item{usearrows}{ boolean; should arrows (rather than line segments) be used to indicate edges? }
  \item{mode}{ the vertex placement algorithm; this must correspond to a \code{\link{gplot.layout}} function. }
  \item{displayisolates}{ boolean; should isolates be displayed? }
  \item{interactive}{ boolean; should interactive adjustment of vertex placement be attempted? }
  \item{boxed.labels}{ boolean; place vertex labels within boxes? }
  \item{xlab}{ X axis label. }
  \item{ylab}{ Y axis label. }
  \item{pad}{ amount to pad the plotting range; useful if labels are being clipped. }
  \item{vertex.pch}{ character type for vertices; may be given as a vector, if vertices are to be of different types. }
  \item{label.cex}{ character expansion factor for label text. }
  \item{vertex.cex}{ character expansion factor for vertices; may be given as a vector, if vertices are to be of different types. }
  \item{label.col}{ color for vertex labels; may be given as a vector, if labels are to be of different colors. }
  \item{edge.col}{ color for edges; may be given as a vector or adjacency matrix, if edges are to be of different colors. }
  \item{vertex.col}{ color for vertices; may be given as a vector, if vertices are to be of different colors. }
  \item{arrowhead.length}{ arrowhead length to use with \code{arrow}, if \code{usearrows==TRUE}. }
  \item{edge.lty}{ line type for edges; may be given as a vector or adjacency matrix, if edges are to have different line types. }
  \item{edge.lwd}{ line width scale for edges; if set greater than 0, edge widths are given by \code{edge.lwd*dat}.  May be given as a vector or adjacency matrix, if edges are to have different line widths. }
  \item{edge.len}{ if \code{uselen==TRUE}, curved edge lengths are scaled by \code{edge.len}. }
  \item{edge.curve}{ if \code{usecurve==TRUE}, the extent of edge curvature is controlled by \code{edge.curv}.  May be given as a fixed value, vector, or adjacency matrix, if edges are to have different levels of curvature.  }
  \item{edge.steps}{ for curved edges (including loops), the number of line segments to use for the curve approximation. }
  \item{diag.size}{ loop size, if \code{diag==TRUE}. }
  \item{uselen}{ boolean; should we use \code{edge.len} to rescale edge lengths? }
  \item{usecurve}{ boolean; should we use \code{edge.curve}? }
  \item{suppress.axes}{ boolean; suppress plotting of axes? }
  \item{new}{ boolean; create a new plot?  If \code{new==FALSE}, vertices and edges will be added to the existing plot. }
  \item{layout.par}{ parameters to the \code{\link{gplot.layout}} function specified in \code{mode}. }
  \item{\dots}{ additional arguments to \code{\link{plot}}. }
}
\details{
\code{gplot} is the standard network visualization tool within the \code{sna} library.  By means of clever selection of display parameters, a fair amount of display flexibility can be obtained.  Graph layout -- if not specified directly using \code{coord} -- is determined via one of the various available algorithms.  These should be specified via the \code{mode} argument; see \code{\link{gplot.layout}} for a full list.  User-supplied layout functions are also possible -- see the aforementioned man page for details.

Note that where \code{gmode=="twomode"}, the supplied two-mode matrix is converted to bipartite adjacency form prior to computing coordinates.  If \code{interactive==TRUE}, then the user may modify the initial graph layout by selecting an individual vertex and then clicking on the location to which this vertex is to be moved; this process may be repeated until the layout is satisfactory.
}
\value{
A two-column matrix containing the vertex positions as x,y coordinates.  
}
\references{ Wasserman, S. and Faust, K.  (1994)  \emph{Social Network Analysis: Methods and Applications.}  Cambridge: Cambridge University Press. }
\author{ 
Carter T. Butts \email{buttsc@uci.edu} 

Alex Montgomery \email{ahm@stanford.edu}
}
%\note{ ~~further notes~~ }

% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{ \code{\link{plot}}, \code{\link{gplot.layout}} }
\examples{
gplot(rgraph(5))               #Plot a random graph
gplot(rgraph(5),usecurv=TRUE)  #This time, use curved edges
gplot(rgraph(5),mode="mds")    #Try an alternative layout scheme
}
\keyword{ graphs }% at least one, from doc/KEYWORDS
\keyword{ hplot }% __ONLY ONE__ keyword per line

\eof
\name{gplot.layout}
\alias{gplot.layout}
\alias{gplot.layout.princoord}
\alias{gplot.layout.mds}
\alias{gplot.layout.eigen}
\alias{gplot.layout.rmds}
\alias{gplot.layout.geodist}
\alias{gplot.layout.adj}
\alias{gplot.layout.seham}
\alias{gplot.layout.segeo}
\alias{gplot.layout.random}
\alias{gplot.layout.circrand}
\alias{gplot.layout.circle}
\alias{gplot.layout.spring}
\alias{gplot.layout.springrepulse}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Vertex Layout Functions for gplot }
\description{
  Various functions which generate vertex layouts for the \code{\link{gplot}} visualization routine.
}
\usage{
gplot.layout.adj(d, layout.par)
gplot.layout.circle(d, layout.par)
gplot.layout.circrand(d, layout.par)
gplot.layout.eigen(d, layout.par)
gplot.layout.geodist(d, layout.par)
gplot.layout.mds(d, layout.par)
gplot.layout.princoord(d, layout.par)
gplot.layout.random(d, layout.par)
gplot.layout.rmds(d, layout.par)
gplot.layout.segeo(d, layout.par)
gplot.layout.seham(d, layout.par)
gplot.layout.spring(d, layout.par)
gplot.layout.springrepulse(d, layout.par)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{d}{ an adjacency matrix, as passed by \code{\link{gplot}}. }
  \item{layout.par}{ a list of parameters. }
}
\details{
  Vertex layouts for network visualization pose a difficult problem -- there is no single, ``good'' layout algorithm, and many different approaches may be valuable under different circumstances.  With this in mind, \code{\link{gplot}} allows for the use of arbitrary vertex layout algorithms via the \code{gplot.layout.*} family of routines.  When called, \code{\link{gplot}} searches for a \code{graph.layout} function whose third name matches its \code{mode} argument (see \code{\link{gplot}} help for more information); this function is then used to generate the layout for the resulting plot.  In addition to the routines documented here, users may add their own layout functions as needed.  The requirements for a \code{graph.layout} function are as follows:
  \enumerate{
  \item the first argument, \code{d}, must be the (dichotomous) graph adjacency matrix;
  \item the second argument, \code{layout.par}, must be a list of parameters (or \code{NULL}, if no parameters are specified); and
  \item the return value must be a real matrix of dimension \code{c(2,NROW(d))}, whose rows contain the vertex coordinates.
  }
  Other than this, anything goes.  (In particular, note that \code{layout.par} could be used to pass additional matrices, if needed.)  
  
  The \code{graph.layout} functions currently supplied by default are as follows:
  \describe{
  \item{circle}{ This function places vertices uniformly in a circle; it takes no arguments.}
  \item{eigen}{ This function places vertices based on the eigenstructure of the adjacency matrix.  It takes the following arguments:
    \describe{
    \item{layout.par\$var}{ This argument controls the matrix to be used for the eigenanalysis.  \code{"symupper"}, \code{"symlower"}, \code{"symstrong"}, \code{"symweak"} invoke \code{\link{symmetrize}} on \code{d} with the respective symmetrizing rule.  \code{"user"} indicates a user-supplied matrix (see below), while \code{"raw"} indicates that \code{d} should be used as-is.  (Defaults to \code{"raw"}.)}
    \item{layout.par\$evsel}{ If \code{"first"}, the first two eigenvectors are used; if \code{"size"}, the two eigenvectors whose eigenvalues have the largest magnitude are used instead. Note that only the real portion of the associated eigenvectors is used.  (Defaults to \code{"first"}.)}
    \item{layout.par\$mat}{  If \code{layout.par\$var=="user"}, this matrix is used for the eigenanalysis. (No default.)}
    }
  }
  \item{mds}{ This function places vertices based on a metric multidimensional scaling of a specified distance matrix.  It takes the following arguments:
    \describe{
    \item{layout.par\$var}{ This argument controls the raw variable matrix to be used for the subsequent distance calculation and scaling.  \code{"rowcol"}, \code{"row"}, and \code{"col"} indicate that the rows and columns (concatenated), rows, or columns (respectively) of \code{d} should be used.  \code{"rcsum"} and \code{"rcdiff"} result in the sum or difference of \code{d} and its transpose being employed.  \code{"invadj"} indicates that \code{max{d}-d} should be used, while \code{"geodist"} uses \code{\link{geodist}} to generate a matrix of geodesic distances from \code{d}. Alternately, an arbitrary matrix can be provided using \code{"user"}.  (Defaults to \code{"rowcol"}.)}
    \item{layout.par\$dist}{ The distance function to be calculated on the rows of the variable matrix.  This must be one of the \code{method} parameters to \code{\list{dist}} (\code{"euclidean"}, \code{"maximum"}, \code{"manhattan"}, or \code{"canberra"}), or else \code{"none"}.  In the latter case, no distance function is calculated, and the matrix in question must be square (with dimension \code{dim(d)}) for the routine to work properly.  (Defaults to \code{"euclidean"}.)}
    \item{layout.par\$exp}{ The power to which distances should be raised prior to scaling.  (Defaults to 2.)}
    \item{layout.par\$vm}{ If \code{layout.par\$var=="user"}, this matrix is used for the distance calculation. (No default.)}
    }
  
  Note: the following layout functions are based on \code{mds}:
    \describe{
    \item{adj}{ scaling of the raw adjacency matrix, treated as similarities (using \code{"invadj"}).}
    \item{geodist}{ scaling of the matrix of geodesic distances.}
    \item{rmds}{ euclidean scaling of the rows of \code{d}.}
    \item{segeo}{ scaling of the squared euclidean distances between row-wise geodesic distances (i.e., approximate structural equivalence).}
    \item{seham}{ scaling of the Hamming distance between rows/columns of \code{d} (i.e., another approximate structural equivalence scaling).}
    }
  }
  \item{princoord}{ This function places vertices based on the eigenstructure of a given correlation/covariance matrix. It takes the following arguments:
    \describe{
    \item{layout.par\$var}{ The matrix of variables to be used for the correlation/covariance calculation.  \code{"rowcol"}, \code{"col"}, and \code{"row"} indicate that the rows/cols, columns, or rows (respectively) of \code{d} should be employed.  \code{"rcsum"} \code{"rcdiff"} result in the sum or difference of \code{d} and \code{t(d)} being used.  \code{"user"} allows for an arbitrary variable matrix to be supplied.  (Defaults to \code{"rowcol"}.)}
    \item{layout.par\$cor}{ Should the correlation matrix (rather than the covariance matrix) be used?  (Defaults to \code{TRUE}.)}
    \item{layout.par\$vm}{ If \code{layout.par\$var=="user"}, this matrix is used for the correlation/covariance calculation. (No default.)}
    }
  }
  \item{random}{ This function places vertices randomly.  It takes the following argument:
    \describe{
    \item{layout.par\$dist}{ The distribution to be used for vertex placement.  Currently, the options are \code{"unif"} (for uniform distribution on the square), \code{"uniang"} (for a ``gaussian donut'' configuration), and \code{"normal"} (for a straight Gaussian distribution).  (Defaults to \code{"unif"}.)}
    }
  Note: \code{circrand}, which is a frontend to the \code{"uniang"} option, is based on this function.
  }
  \item{spring}{ This function places vertices using a spring embedder.  It takes the following arguments:
    \describe{
    \item{layout.par\$mass}{ The vertex mass (in ``quasi-kilograms'').  (Defaults to \code{0.1}.)}
    \item{layout.par\$equil}{ The equilibrium spring extension (in ``quasi-meters''). (Defaults to \code{1}.)}
    \item{layout.par\$k}{ The spring coefficient (in ``quasi-Newtons per quasi-meter''). (Defaults to \code{0.001}.)}
    \item{layout.par\$repeqdis}{ The point at which repulsion (if employed) balances out the spring extension force (in ``quasi-meters''). (Defaults to \code{0.1}.)}
    \item{layout.par\$kfr}{ The base coefficient of kinetic friction (in ``quasi-Newton quasi-kilograms''). (Defaults to \code{0.01}.)}
    \item{layout.par\$repulse}{ Should repulsion be used?  (Defaults to \code{FALSE}.)}
    }
  Note: \code{springrepulse} is a frontend to \code{spring}, with repulsion turned on.
  }
  }
}
\value{
A matrix whose rows contain the x,y coordinates of the vertices of \code{d}.
}
%\references{ ~put references to the literature/web site here ~ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ ~~further notes~~ }

% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{ \code{\link{gplot}}, \code{\link{cmdscale}}, \code{\link{eigen}} }
\examples{
}
\keyword{ graphs }% at least one, from doc/KEYWORDS
\keyword{ dplot }% __ONLY ONE__ keyword per line

\eof
\name{graphcent}
\alias{graphcent}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute the (Harary) Graph Centrality Scores of Network Positions }
\description{
  \code{graphcent} takes a graph stack (\code{dat}) and returns the Harary graph centralities of positions within one graph (indicated by \code{nodes} and \code{g}, respectively).  Depending on the specified mode, graph centrality on directed or undirected geodesics will be returned; this function is compatible with \code{\link{centralization}}, and will return the theoretical maximum absolute deviation (from maximum) conditional on size (which is used by \code{\link{centralization}} to normalize the observed centralization score).
}
\usage{
graphcent(dat, g=1, nodes=c(1:dim(dat)[2]), gmode="digraph", 
    diag=FALSE, tmaxdev=FALSE, cmode="directed", 
    geodist.precomp=NULL, rescale=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors. Alternately, this can be an n x n matrix (if only one graph is involved). }
  \item{g}{ Integer indicating the index of the graph for which centralities are to be calculated.  By default, \code{g==1}. }
  \item{nodes}{ List indicating which nodes are to be included in the calculation.  By default, all nodes are included. }
  \item{gmode}{ String indicating the type of graph being evaluated.  "digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{gmode} is set to "digraph" by default. }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{tmaxdev}{ Boolean indicating whether or not the theoretical maximum absolute deviation from the maximum nodal centrality should be returned.  By default, \code{tmaxdev==FALSE}. }
  \item{cmode}{ String indicating the type of graph centrality being computed (directed or undirected geodesics). }
  \item{geodist.precomp}{ A \code{\link{geodist}} object precomputed for the graph to be analyzed (optional) }
  \item{rescale}{ If true, centrality scores are rescaled such that they sum to 1. }
}
\details{
  The Harary graph centrality of a vertex v is equal to \eqn{\frac{1}{\max_u d(v,u)}}{1/(max_u d(v,u))}, where \eqn{d(v,u)}{d(v,u)} is the geodesic distance from v to u.  Vertices with low graph centrality scores are likely to be near the ``edge'' of a graph, while those with high scores are likely to be near the ``middle.''  Compare this with \code{\link{closeness}}, which is based on the reciprocal of the sum of distances to all other vertices (rather than simply the maximum).
}
\value{
  A vector containing the centrality scores
}
\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ Judicious use of \code{geodist.precomp} can save a great deal of time when computing multiple path-based indices on the same network. }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{centralization}} }

\examples{
g<-rgraph(10)     #Draw a random graph with 10 members
graphcent(g)    #Compute centrality scores
}
\keyword{ univar }%-- one or more ...
\keyword{ math }

\eof
\name{grecip}
\alias{grecip}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute the Reciprocity of an Input Graph or Graph Stack }
\description{
  \code{grecip} calculates the dyadic reciprocity of the elements of \code{dat} selected by \code{g}.
}
\usage{
grecip(dat, g = NULL, measure = c("dyadic", "edgewise"))
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ An adjacency matrix, or stack thereof }
  \item{g}{ A vector indicating which matrices to evaluate (optional) }
  \item{measure}{ One of \code{"dyadic"} (default) or \code{"edgewise"} }
}
\details{
  The dyadic reciprocity of a graph is the proportion of dyads which are symmetric; this is computed and returned by \code{grecip} for the graphs indicated.  Note that the dyadic reciprocity is distinct from the \emph{edgewise} or \emph{tie reciprocity}, which is the proportion of \emph{edges} which are reciprocated.  This latter form may be obtained by setting \code{measure="edgewise"}.
}
\value{
  The graph reciprocity value(s)
}
\references{ Wasserman, S., and Faust, K.  (1994).  ``Social Network Analysis: Methods and Applications.''  Cambridge: Cambridge University Press. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{mutuality}}, \code{\link{symmetrize}} }

\examples{
#Calculate the dyadic reciprocity scores for some random graphs
grecip(rgraph(10,5))
}
\keyword{ math }%-- one or more ...
\keyword{ univar }%-- one or more ...

\eof
\name{gscor}
\alias{gscor}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find the Structural Correlations Between Two or More Graphs }
\description{
  \code{gscor} finds the product-moment structural correlation between the adjacency matrices of graphs indicated by \code{g1} and \code{g2} in stack \code{dat} (or possibly \code{dat2}) given exchangeability list \code{exchange.list}.  Missing values are permitted. 
}
\usage{
gscor(dat, dat2=NULL, g1=c(1:dim(dat)[1]), g2=c(1:dim(dat)[1]), 
    diag=FALSE, mode="digraph", method="anneal", reps=1000, 
    prob.init=0.9, prob.decay=0.85, freeze.time=25, 
    full.neighborhood=TRUE, exchange.list=rep(0, dim(dat)[2]))
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph stack }
  \item{dat2}{ Optionally, a second graph stack }
  \item{g1}{ The indices of \code{dat} reflecting the first set of graphs to be compared; by default, all members of \code{dat} are included }
  \item{g2}{ The indices or \code{dat} (or \code{dat2}, if applicable) reflecting the second set of graphs to be compared; by default, all members of \code{dat} are included. }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{mode}{ String indicating the type of graph being evaluated.  "digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{mode} is set to "digraph" by default. }
  \item{method}{ Method to be used to search the space of accessible permutations; must be one of ``none'', ``exhaustive'', ``anneal'', ``hillclimb'', or ``mc''.  }
  \item{reps}{ Number of iterations for monte carlo method. }
  \item{prob.init}{ Initial acceptance probability for the annealing routine. }
  \item{prob.decay}{ Cooling multiplier for the annealing routine. }
  \item{freeze.time}{ Freeze time for the annealing routine. }
  \item{full.neighborhood}{ Should the annealer evaluate the full neighborhood of pair exchanges at each iteration? }
  \item{exchange.list}{ Information on which vertices are exchangeable (see below); this must be a single number, a vector of length n, or a nx2 matrix. }
}
\details{

The structural correlation coefficient between two graphs G and H is defined as
\deqn{scor\left(G,H \left| L_G,L_H\right.\right) = \max_{L_G,L_H} cor(\ell(G),\ell(H))}{%
scor(G,H | L_G,L_H) = max_[L_G,L_H] cor(l(G),l(H))}
where \eqn{L_G} is the set of accessible permutations/labelings of G, \eqn{\ell(G)}{l(G)} is a permutation/relabeling of G, and \eqn{\ell(G) \in L_G}{l(G) in L_G}.  The set of accessible permutations on a given graph is determined by the \emph{theoretical exchangeability} of its vertices; in a  nutshell, two vertices are considered to be theoretically exchangeable for a given problem if all predictions under the conditioning theory are invariant to a relabeling of the vertices in question (see Butts and Carley (2001) for a more formal exposition).  Where no vertices are exchangeable, the structural correlation becomes the simple graph correlation.  Where \emph{all} vertices are exchangeable, the structural correlation reflects the correlation between unlabeled graphs; other cases correspond to correlation under partial labeling.  

The accessible permutation set is determined by the \code{exchange.list} argument, which is dealt with in the following manner. First, \code{exchange.list} is expanded to fill an nx2 matrix.  If \code{exchange.list} is a single number, this is trivially accomplished by replication; if \code{exchange.list} is a vector of length n, the matrix is formed by cbinding two copies together.  If \code{exchange.list} is already an nx2 matrix, it is left as-is.  Once the nx2 exchangeability matrix has been formed, it is interpreted as follows: columns refer to graphs 1 and 2, respectively; rows refer to their corresponding vertices in the original adjacency matrices; and vertices are taken to be theoretically exchangeable iff their corresponding exchangeability matrix values are identical.  To obtain an unlabeled graph correlation (the default), then, one could simply let \code{exchange.list} equal any single number.  To obtain the standard graph correlation, one would use the vector \code{1:n}.

Because the set of accessible permutations is, in general, very large (\eqn{o(n!)}), searching the set for the maximum correlation is a non-trivial affair.  Currently supported methods for estimating the structural correlation are hill climbing, simulated annealing, blind monte carlo search, or exhaustive search (it is also possible to turn off searching entirely).  Exhaustive search is not recommended for graphs larger than size 8 or so, and even this may take days; still, this is a valid alternative for small graphs.  Blind monte carlo search and hill climbing tend to be suboptimal for this problem and are not, in general recommended, but they are available if desired.  The preferred (and default) option for permutation search is simulated annealing, which seems to work well on this problem (though some tinkering with the annealing parameters may be needed in order to get optimal performance).  See the help for \code{\link{lab.optimize}} for more information regarding these options.

Structural correlation matrices are p.s.d., and are p.d. so long as no graph within the set is a linear combination of any other under any accessible permutation.  Their eigendecompositions are meaningful and they may be used in linear subspace analyses, so long as the researcher is careful to interpret the results in terms of the appropriate set of accessible labelings.  Classical null hypothesis tests should not be employed with structural correlations, and QAP tests are almost never appropriate (save in the uniquely labeled case).  See \code{\link{cugtest}} for a more reasonable alternative.
}
\value{
An estimate of the structural correlation matrix
}
\references{ Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Interstructural Analysis.''  CASOS Working Paper, Carnegie Mellon University.}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ Consult Butts and Carley (2001) for advice and examples on theoretical exchangeability. }

\section{Warning }{The search process can be \emph{very slow}, particularly for large graphs.  In particular, the \emph{exhaustive} method is order factorial, and will take approximately forever for unlabeled graphs of size greater than about 7-9.}

\seealso{ \code{\link{gscov}}, \code{\link{gcor}}, \code{\link{gcov}} }

\examples{
#Generate two random graphs
g.1<-rgraph(5)
g.2<-rgraph(5)

#Copy one of the graphs and permute it
perm<-sample(1:5)
g.3<-g.2[perm,perm]

#What are the structural correlations between the labeled graphs?
gscor(g.1,g.2,exchange.list=1:5)
gscor(g.1,g.3,exchange.list=1:5)
gscor(g.2,g.3,exchange.list=1:5)

#What are the structural correlations between the underlying 
#unlabeled graphs?
gscor(g.1,g.2)
gscor(g.1,g.3)
gscor(g.2,g.3)
}
\keyword{ univar }
\keyword{ multivariate }%-- one or more ...

\eof
\name{gscov}
\alias{gscov}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find the Structural Covariance(s) Between Two or More Graphs }
\description{
  \code{gscov} finds the structural covariance between the adjacency matrices of graphs indicated by \code{g1} and \code{g2} in stack \code{dat} (or possibly \code{dat2}) given exchangeability list \code{exchange.list}.  Missing values are permitted. 
}
\usage{
gscov(dat, dat2=NULL, g1=c(1:dim(dat)[1]), g2=c(1:dim(dat)[1]), 
    diag=FALSE, mode="digraph", method="anneal", reps=1000, 
    prob.init=0.9, prob.decay=0.85, freeze.time=25, 
    full.neighborhood=TRUE, exchange.list=rep(0, dim(dat)[2]))
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph stack }
  \item{dat2}{ Optionally, a second graph stack }
  \item{g1}{ The indices of \code{dat} reflecting the first set of graphs to be compared; by default, all members of \code{dat} are included }
  \item{g2}{ The indices or \code{dat} (or \code{dat2}, if applicable) reflecting the second set of graphs to be compared; by default, all members of \code{dat} are included. }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{mode}{ String indicating the type of graph being evaluated.  "digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{mode} is set to "digraph" by default.}
  \item{method}{ Method to be used to search the space of accessible permutations; must be one of ``none'', ``exhaustive'', ``anneal'', ``hillclimb'', or ``mc''. }
  \item{reps}{ Number of iterations for monte carlo method. }
  \item{prob.init}{ Initial acceptance probability for the annealing routine. }
  \item{prob.decay}{ Cooling multiplier for the annealing routine. }
  \item{freeze.time}{ Freeze time for the annealing routine. }
  \item{full.neighborhood}{ Should the annealer evaluate the full neighborhood of pair exchanges at each iteration? }
  \item{exchange.list}{ Information on which vertices are exchangeable (see below); this must be a single number, a vector of length n, or a nx2 matrix. }
}
\details{
  The structural covariance between two graphs G and H is defined as
\deqn{scov\left(G,H \left| L_G,L_H\right.\right) = \max_{L_G,L_H} cov(\ell(G),\ell(H))}{%
scov(G,H | L_G,L_H) = max_[L_G,L_H] cov(l(G),l(H))}
where \eqn{L_G} is the set of accessible permutations/labelings of G, \eqn{\ell(G)}{l(G)} is a permutation/labeling of G, and \eqn{\ell(G) \in L_G}{l(G) in L_G}.  The set of accessible permutations on a given graph is determined by the \emph{theoretical exchangeability} of its vertices; in a  nutshell, two vertices are considered to be theoretically exchangeable for a given problem if all predictions under the conditioning theory are invariant to a relabeling of the vertices in question (see Butts and Carley (2001) for a more formal exposition).  Where no vertices are exchangeable, the structural covariance becomes the simple graph covariance.  Where \emph{all} vertices are exchangeable, the structural covariance reflects the covariance between unlabeled graphs; other cases correspond to covariance under partial labeling.  

The accessible permutation set is determined by the \code{exchange.list} argument, which is dealt with in the following manner. First, \code{exchange.list} is expanded to fill an nx2 matrix.  If \code{exchange.list} is a single number, this is trivially accomplished by replication; if \code{exchange.list} is a vector of length n, the matrix is formed by cbinding two copies together.  If \code{exchange.list} is already an nx2 matrix, it is left as-is.  Once the nx2 exchangeabiliy matrix has been formed, it is interpreted as follows: columns refer to graphs 1 and 2, respectively; rows refer to their corresponding vertices in the original adjacency matrices; and vertices are taken to be theoretically exchangeable iff their corresponding exchangeability matrix values are identical.  To obtain an unlabeled graph covariance (the default), then, one could simply let \code{exchange.list} equal any single number.  To obtain the standard graph covariance, one would use the vector \code{1:n}.

Because the set of accessible permutations is, in general, very large (\eqn{o(n!)}), searching the set for the maximum covariance is a non-trivial affair.  Currently supported methods for estimating the structural covariance are hill climbing, simulated annealing, blind monte carlo search, or exhaustive search (it is also possible to turn off searching entirely).  Exhaustive search is not recommended for graphs larger than size 8 or so, and even this may take days; still, this is a valid alternative for small graphs.  Blind monte carlo search and hill climbing tend to be suboptimal for this problem and are not, in general recommended, but they are available if desired.  The preferred (and default) option for permutation search is simulated annealing, which seems to work well on this problem (though some tinkering with the annealing parameters may be needed in order to get optimal performance).  See the help for \code{\link{lab.optimize}} for more information regarding these options.

Structural covariance matrices are p.s.d., and are p.d. so long as no graph within the set is a linear combination of any other under any accessible permutation.  Their eigendecompositions are meaningful and they may be used in linear subspace analyses, so long as the researcher is careful to interpret the results in terms of the appropriate set of accessible labelings.  Classical null hypothesis tests should not be employed with structural covariances, and QAP tests are almost never appropriate (save in the uniquely labeled case).  See \code{\link{cugtest}} for a more reasonable alternative.
}
\value{
 An estimate of the structural covariance matrix
}
\references{ Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Interstructural Analysis.''  CASOS Working Paper, Carnegie Mellon University. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ Consult Butts and Carley (2001) for advice and examples on theoretical exchangeability. }

\section{Warning }{The search process can be \emph{very slow}, particularly for large graphs.  In particular, the \emph{exhaustive} method is order factorial, and will take approximately forever for unlabeled graphs of size greater than about 7-9.}

\seealso{ \code{\link{gscor}}, \code{\link{gcov}}, \code{\link{gcor}} }

\examples{
#Generate two random graphs
g.1<-rgraph(5)
g.2<-rgraph(5)

#Copy one of the graphs and permute it
perm<-sample(1:5)
g.3<-g.2[perm,perm]

#What are the structural covariances between the labeled graphs?
gscov(g.1,g.2,exchange.list=1:5)
gscov(g.1,g.3,exchange.list=1:5)
gscov(g.2,g.3,exchange.list=1:5)

#What are the structural covariances between the underlying 
#unlabeled graphs?
gscov(g.1,g.2)
gscov(g.1,g.3)
gscov(g.2,g.3)
}
\keyword{ univar }
\keyword{ multivariate }%-- one or more ...

\eof
\name{gtrans}
\alias{gtrans}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute the Transitivity of an Input Graph or Graph Stack }
\description{
  \code{gtrans} returns the transitivity of the elements of \code{dat} selected by \code{g}, using the definition of \code{measure}.  Triads involving missing values are omitted from the analysis.
}
\usage{
gtrans(dat, g=NULL, diag=FALSE, mode="digraph", measure = c("weak", 
    "strong", "weakcensus", "strongcensus"))
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph or graph stack }
  \item{g}{ A vector indicating the graphs which are to be analyzed; by default, all graphs are analyzed }
  \item{diag}{ A boolean indicating whether or not diagonal entries (loops) are to be taken as valid data }
  \item{mode}{ \code{"digraph"} if directed triads are sought, or else \code{"graph"} }
  \item{measure}{ One of \code{"weak"} (default), \code{"strong"}, \code{"weakcensus"}, or \code{"strongcensus"} }
}
\details{
Transitivity is a triadic, algebraic structural constraint.  In its weak form, the transitive constraint corresponds to \eqn{a \rightarrow b \rightarrow c \Rightarrow a \rightarrow c}{a->b->c => a->c}.  In the corresponding strong form, the constraint is \eqn{a \rightarrow b \rightarrow c \Leftrightarrow a \rightarrow c}{a->b->c <=> a->c}.  (Note that the weak form is that most commonly employed.)  Where \code{measure=="weak"}, the fraction of potentially intransitive triads obeying the weak condition is returned.  With the \code{measure=="weakcensus"} setting, by contrast, the total \emph{number} of transitive triads is computed.  The \code{strong} versions of the measures are similar to the above, save in that the set of all triads is considered (since all are ``at risk'' for intransitivity).
}
\value{
A vector of transitivity scores
}
\references{ Holland, P.W., and Leinhardt, S.  (1972).  ``Some Evidence on the Transitivity of Positive Interpersonal Sentiment.'' \emph{American Journal of Sociology,} 72, 1205-1209.

Wasserman, S., and Faust, K.  (1994).  \emph{Social Network Analysis: Methods and Applications}.  Cambridge: Cambridge University Press. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ In version 0.3, the strong form was the default (and only) option for \code{gtrans}. }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{triad.classify}}, \code{\link{cugtest}} }

\examples{
#Draw some random graphs
g<-rgraph(5,10)

#Find transitivity scores
gtrans(g)
}
\keyword{ algebra }%-- one or more ...

\eof
\name{gvectorize}
\alias{gvectorize}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Vectorization of Adjacency Matrices}
\description{
\code{gvectorize} takes an input graph stack and converts it into a corresponding number of vectors by row concatenation.
}
\usage{
gvectorize(mats, mode="digraph", diag=FALSE, censor.as.na=TRUE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{mats}{ One or more adjacency matrices (in a graph stack) }
  \item{mode}{ ``digraph'' if data is taken to be directed, else ``graph'' }
  \item{diag}{ Boolean indicating whether diagonal entries (loops) are taken to contain meaningful data }
  \item{censor.as.na}{ If \code{TRUE}, code unused parts of the adjacency matrix as \code{NA}s prior to vectorizing; otherwise, unused parts are simply removed }
}
\details{
The output of \code{gvectorize} is a matrix in which each column corresponds to an input graph, and each row corresponds to an edge.  The columns of the output matrix are formed by simple row-concatenation of the original adjacency matrices, possibly after removing cells which are not meaningful (if \code{censor.as.na==FALSE}).  This is useful when preprocessing edge sets for use with \code{glm} or the like.
}
\value{
An nxk matrix, where n is the number of arcs and k is the number of graphs; if \code{censor.as.na==FALSE}, n will be reflect the relevant number of uncensored arcs.
}
%\references{ }
\author{ Carter T. Butts \email{buttsc@uci.edu}}
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

%\seealso{  }

\examples{
#Draw two random graphs
g<-rgraph(10,2)

#Examine the vectorized form of the adjacency structure
gvectorize(g)
}
\keyword{ manip }%-- one or more ...
\keyword{ math }

\eof
\name{hdist}
\alias{hdist}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find the Hamming Distances Between Two or More Graphs }
\description{
  \code{hdist} returns the Hamming distance between the labeled graphs \code{g1} and \code{g2} in stack \code{dat} for dichotomous data, or else the absolute (manhattan) distance.  If \code{normalize} is true, this distance is divided by its dichotomous theoretical maximum (conditional on |V(G)|).
}
\usage{
hdist(dat, dat2=NULL, g1=c(1:dim(dat)[1]), g2=c(1:dim(dat)[1]), 
    normalize=FALSE, diag=FALSE, mode="digraph")
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors.  This data need not be dichotomous, and missing values are allowed. }
  \item{dat2}{ A second data array (optional) }
  \item{g1}{ A vector indicating which graphs to compare (by default, all elements of \code{dat}) }
  \item{g2}{ A vector indicating against which the graphs of \code{g1} should be compared (by default, all graphs) }
  \item{normalize}{ Divide by the number of available dyads? }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{mode}{ String indicating the type of graph being evaluated.  "digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{mode} is set to "digraph" by default. }
}
\details{
  The Hamming distance between two labeled graphs \eqn{G_1} and \eqn{G_2} is equal to \eqn{|\{e : (e \in E(G_1), e \not\in E(G_2)) \wedge (e \not\in E(G_1), e \in E(G_2))\}|}{|\{e : (e in E(G_1) and e not in E(G_2)) or (e not in E(G_1) and e in E(G_2))\}|}.  In more prosaic terms, this may be thought of as the number of addition/deletion operations required to turn the edge set of \eqn{G_1} into that of \eqn{G_2}.  The Hamming distance is a highly general measure of structural similarity, and forms a metric on the space of graphs (simple or directed).  Users should be reminded, however, that the Hamming distance is extremely sensitive to nodal labeling, and should not be employed directly when nodes are interchangeable.  The structural distance (Butts and Carley (2001)), implemented in \code{\link{structdist}}, provides a natural generalization of the Hamming distance to the more general case of unlabeled graphs.

Null hypothesis testing for Hamming distances is available via \code{\link{cugtest}}, and \code{\link{qaptest}}; graphs which minimize the Hamming distances to all members of a graph set can be found by \code{\link{centralgraph}}.  For an alternative means of comparing the similarity of graphs, consider \code{\link{gcor}}.
}
\value{
A matrix of Hamming distances
}
\references{ 
Banks, D., and Carley, K.M.  (1994).  ``Metric Inference for Social Networks.''  \emph{Journal of Classification}, 11(1), 121-49.

Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Interstructural Analysis.'' CASOS working paper, Carnegie Mellon University. 

Hamming, R.W. (1950). ``Error Detecting and Error Correcting Codes.'' \emph{Bell System Technical Journal,} 29, 147-160.
}

\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ For non-dichotomous data, the distance which is returned is simply the sum of the absolute edge-wise differences. }


 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{sdmat}}, \code{\link{structdist}} }

\examples{
#Get some random graphs
g<-rgraph(5,5,tprob=runif(5,0,1))

#Find the Hamming distances
hdist(g)
}
\keyword{ univar }
\keyword{ multivariate }%-- one or more ...

\eof
\name{hierarchy}
\alias{hierarchy}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute Graph Hierarchy Scores }
\description{
  \code{hierarchy} takes a graph stack (\code{dat}) and returns reciprocity or Krackhardt hierarchy scores for the graphs selected by \code{g}.
}
\usage{
hierarchy(dat, g=1:stackcount(dat), measure=c("reciprocity", 
    "krackhardt"))
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph or graph stack }
  \item{g}{ Index values for the graphs to be utilized; by default, all graphs are selected }
  \item{measure}{ One of \code{"reciprocity"} or \code{"krackhardt"} }
}
\details{
Hierarchy measures quantify the extent of asymmetry in a structure; the greater the extent of asymmetry, the more hierarchical the structure is said to be.  (This should not be confused with how \emph{centralized} the structure is, i.e., the extent to which centralities of vertex positions are highly concentrated.)  \code{hierarchy} provides two measures (selected by the \code{measure} argument) as follows:

\enumerate{
\item \code{reciprocity}: This setting returns the dyadic reciprocity for each input graph (see \code{\link{grecip}})

\item \code{krackhardt}: This setting returns the Krackhardt hierarchy score for each input graph.  The Krackhardt hierarchy is defined as the fraction of non-null dyads in the \code{\link{reachability}} graph which are asymmetric.  Thus, when no directed paths are reciprocated (e.g., in an in/outtree), Krackhardt hierarchy is equal to 1; when all such paths are reciprocated, by contrast (e.g., in a cycle or clique), the measure falls to 0. 

Hierarchy is one of four measures (\code{\link{connectedness}}, \code{\link{efficiency}}, \code{\link{hierarchy}}, and \code{\link{lubness}}) suggested by Krackhardt for summarizing hierarchical structures.  Each corresponds to one of four axioms which are necessary and sufficient for the structure in question to be an outtree; thus, the measures will be equal to 1 for a given graph iff that graph is an outtree.  Deviations from unity can be interpreted in terms of failure to satisfy one or more of the outtree conditions, information which may be useful in classifying its structural properties.
}

Note that hierarchy is inherently density-constrained: as densities climb above 0.5, the proportion of mutual dyads must (by the pigeonhole principle) increase rapidly, thereby reducing possibilities for asymmetry.  Thus, the interpretation of hierarchy scores should take density into account, particularly if density is artifactual (e.g., due to a particular dichotomization procedure).  
}
\value{
A vector of hierarchy scores
}
\references{ Krackhardt, David.  (1994).  ``Graph Theoretical Dimensions of Informal Organizations.'' In K. M. Carley and M. J. Prietula (Eds.), \emph{Computational Organization Theory}, 89-111. Hillsdale, NJ: Lawrence Erlbaum and Associates. 

Wasserman, S., and Faust, K.  (1994).  ``Social Network Analysis: Methods and Applications.''  Cambridge: Cambridge University Press.}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ The four Krackhardt indices are, in general, nondegenerate for a relatively narrow band of size/density combinations (efficiency being the sole exception).  This is primarily due to their dependence on the reachability graph, which tends to become complete rapidly as size/density increase.  See Krackhardt (1994) for a useful simulation study. }

\seealso{ \code{\link{connectedness}}, \code{\link{efficiency}}, \code{\link{hierarchy}}, \code{\link{lubness}}, \code{\link{grecip}}, \code{\link{mutuality}}, \code{\link{dyad.census}} }

\examples{
#Get hierarchy scores for graphs of varying densities
hierarchy(rgraph(10,5,tprob=c(0.1,0.25,0.5,0.75,0.9)),
    measure="reciprocity")
hierarchy(rgraph(10,5,tprob=c(0.1,0.25,0.5,0.75,0.9)),
    measure="krackhardt")
}
\keyword{ math }% at least one, from doc/KEYWORDS
\keyword{ univar }% __ONLY ONE__ keyword per line

\eof
\name{infocent}
\alias{infocent}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find Information Centrality Scores of Network Positions }
\description{
  \code{infocent} takes a graph stack (\code{dat}) and returns the information centralities of positions within one graph (indicated by \code{nodes} and \code{g}, respectively).  This function is compatible with \code{\link{centralization}}, and will return the theoretical maximum absolute deviation (from maximum) conditional on size (which is used by \code{\link{centralization}} to normalize the observed centralization score).
}
\usage{
infocent(dat, g=1, nodes=c(1:dim(dat)[2]), gmode="digraph", 
    diag=FALSE, cmode="weak", tmaxdev=FALSE, rescale=FALSE,tol=1e-20)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors. Alternately, this can be an n x n matrix (if only one graph is involved). }
  \item{g}{ Integer indicating the index of the graph for which centralities are to be calculated.  By default, \code{g==1}. }
  \item{nodes}{ List indicating which nodes are to be included in the calculation.  By default, all nodes are included. }
  \item{gmode}{ String indicating the type of graph being evaluated.  \code{"digraph"} indicates that edges should be interpreted as directed; \code{"graph"} indicates that edges are undirected.  This is currently ignored. }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{cmode}{ The rule to be used by \code{\link{symmetrize}} when symmetrizing dichotomous data; must be one of \code{"weak"} (for an \code{OR} rule), \code{"strong"} for an \code{AND} rule), \code{"upper"} (for a \code{max} rule), or \code{"lower"} (for a \code{min} rule).  Set to \code{"weak"} by default, this parameter obviously has no effect on symmetric data. }
  \item{tmaxdev}{ Boolean indicating whether or not the theoretical maximum absolute deviation from the maximum nodal centrality should be returned.  By default, \code{tmaxdev==FALSE}. }
  \item{rescale}{ If true, centrality scores are rescaled such that they sum to 1. }
  \item{tol}{ Tolerance for near-singularities during matrix inversion (see \code{\link{solve}}) }
}
\details{
  Actor information centrality is a hybrid measure which relates to both path-length indices (e.g., closeness, graph centrality) and to walk-based eigenmeasures (e.g., eigenvector centrality, Bonacich power).  In particular, the information centrality of a given actor can be understood to be the harmonic average of the ``bandwidth'' for all paths originating with said individual (where the bandwidth is taken to be inversely related to path length).  Formally, the index is constructed as follows.  First, we take \eqn{G} to be an undirected (but possibly valued) graph -- symmetrizing if necessary -- with (possibly valued) adjacency matrix \eqn{\mathbf{A}}{A}.  From this, we remove all isolates (whose information centralities are zero in any event) and proceed to create the weighted connection matrix

\deqn{
\mathbf{C} = \mathbf{B}^{-1}}{%
C = B^-1}

where \eqn{\mathbf{B}}{B} is a pseudo-adjacency matrix formed by replacing the diagonal of \eqn{1-\mathbf{A}}{1-A} with one plus each actor's degree.  Given the above, let \eqn{T} be the trace of \eqn{\mathbf{C}}{C} with sum \eqn{S_T}, and let \eqn{S_R} be an arbitrary row sum (all rows of \eqn{\mathbf{C}}{C} have the same sum).  The information centrality scores are then equal to

\deqn{
C_I = \frac{1}{T + \frac{S_T-2S_R}{|V(G)|}}}{%
C_I = ( T + (S_T-2S_R)/|V(G)| )^-1
}

(recalling that the scores for any omitted vertices are 0).

In general, actors with higher information centrality are predicted to have greater control over the flow of information within a network; highly information-central individuals tend to have a large number of short paths to many others within the social structure.  Because the raw centrality values can be difficult to interpret directly, rescaled values are sometimes preferred (see the \code{rescale} option).  Though the use of path weights suggest information centrality as a possible replacement for closeness, the problem of inverting the \eqn{\mathbf{B}}{B} matrix poses problems of its own; as with all such measures, caution is advised on disconnected or degenerate structures. 
}
\value{
A vector containing the centrality scores
}
\references{ Stephenson, K., and Zelen, M. (1989).  ``Rethinking Centrality: Methods and Applications.''  \emph{Social Networks}, 11, 1-37.

Wasserman, S., and Faust, K.  (1994).  ``Social Network Analysis: Methods and Applications.''  Cambridge: Cambridge University Press.}
\author{ David Barron \email{david.barron@jesus.ox.ac.uk}

Carter T. Butts \email{buttsc@uci.edu} }
\note{ The theoretical maximum deviation used here is not obtained with the star network; rather, the maximum occurs for an empty graph with one complete dyad, which is the model used here. }

\seealso{ \code{\link{evcent}}, \code{\link{bonpow}}, \code{\link{closeness}}, \code{\link{graphcent}}, \code{\link{centralization}} }

\examples{
#Generate some test data
dat<-rgraph(10,mode="graph")
#Compute information centrality scores
infocent(dat)
}
\keyword{ math }% at least one, from doc/KEYWORDS
\keyword{ univar }% __ONLY ONE__ keyword per line

\eof
\name{interval.graph}
\alias{interval.graph}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Convert Spell Data to Interval Graphs }
\description{
  Constructs one or more interval graphs (and exchangeability vectors) from a set of spells.
}
\usage{
interval.graph(slist, type="simple", diag=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{slist}{ A spell list.  This must consist of an nxmx3 array, with n being the number of actors, m being the maximum number of spells (one per row) and with the three columns of the last dimension containing a (categorical) spell type code, the time of spell onset (any units), and the time of spell termination (same units), respectively. }
  \item{type}{ One of ``simple'', ``overlap'', ``fracxy'', ``fracyx'', or ``jntfrac''. }
  \item{diag}{ Include the dyadic entries? }
}
\details{
  Given some ordering dimension T (usually time), a ``spell'' is defined as the interval between a specified onset and a specified termination (with onset preceding the termination).  An interval graph, then, on spell set V, is \eqn{G=\{V,E\}}, where \eqn{\{i,j\} \in E}{\{i,j\} in E} iff there exists some point \eqn{t \in T}{e \in T} such that \eqn{t \in i}{t \in i} and \eqn{t \in j}{t \in j}.  In more prosaic terms, an interval graph on a given spell set has each spell as a vertex, with vertices adjacent iff they overlap.  Such structures are useful for quantifying life history data (where spells might represent marriages, periods of child custody/co-residence, periods of employment, etc.), organizational history data (where spells might reflect periods of strategic alliances, participation in a particular product market, etc.), task scheduling (with spells representing the dedication of a particular resource to a given task), etc.  By giving complex historical data a graphic representation, it is possible to easily perform a range of analyses which would otherwise be difficult and/or impossible (see Butts and Pixley (2001) for examples).

In addition to the simple interval graph (described above), \code{interval.graph} can also generate valued interval graphs using a number of different edge definitions.  This is controlled by the \code{type} argument, with edge values as follows:
\enumerate{
\item simple: dichotomous coding based on simple overlap (i.e., (x,y)=1 iff x overlaps y)

\item overlap: edge value equals the total magnitude of the overlap between spells

\item fracxy: the (x,y) edge value equals the fraction of the duration of y which is covered by x

\item fracyx: the (x,y) edge value equals the fraction of the duration of x which is covered by y

\item jntfrac: edge value equals the total magnitude of the overlap between spells divided by the mean of the spells' lengths
}
Note that ``simple,'' ``overlap,'' and ``jntfrac'' are symmetric relations, while ``fracxy'' and ``fracyx'' are directed.  As always, the specific edge type used should reflect the application to which the interval graph is being put.
}
\value{
  A data frame containing:

  \item{graph }{A graph stack containing the interval graphs}
  \item{exchange.list }{Matrix containing the vector of spell types associated with each interval graph}
}
\references{ 
Butts, C.T., and Pixley, J.E.  ``A Structural Approach to the Representation of Life History Data.'' CASOS working paper, Carnegie Mellon University.

West, D.B.  (1996).  \emph{Introduction to Graph Theory}.  Upper Saddle River, NJ: Prentice Hall. }
\author{ Carter T. Butts \email{buttsc@uci.edu}}
%\note{ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

%\seealso{ }

\examples{
}
\keyword{ manip }%-- one or more ...
\keyword{ math }

\eof
\name{is.isolate}
\alias{is.isolate}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Is Ego an Isolate? }
\description{
  Returns TRUE iff ego is an isolate in graph \code{g} of \code{dat}.
}
\usage{
is.isolate(dat, ego, g=1, diag=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph or graph stack }
  \item{ego}{ Index of the vertex (or a vector of vertices) to check }
  \item{g}{ Which graph should be examined? }
  \item{diag}{ Boolean indicating whether adjacency matrix diagonals (i.e., loops) contain meaningful data }
}
\details{
In the valued case, any non-zero edge value is taken as sufficient to establish a tie.
}
\value{
A boolean value (or vector thereof) indicating isolate status
}
\references{
Wasserman, S., and Faust, K.  (1994).  ``Social Network Analysis: Methods and Applications.''  Cambridge: Cambridge University Press.

West, D.B.  (1996).  \emph{Introduction to Graph Theory}.  Upper Saddle River, NJ: Prentice Hall. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{isolates}}, \code{\link{addisolates}} }

\examples{
#Generate a test graph
g<-rgraph(20)
g[,4]<-0          #Create an isolate
g[4,]<-0

#Check for isolates
is.isolate(g,2)  #2 is almost surely not an isolate
is.isolate(g,4)  #4 is, by construction
}
\keyword{ logic }%-- one or more ...

\eof
\name{isolates}
\alias{isolates}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ List the Isolates in a Graph or Graph Stack }
\description{
  Returns a list of the isolates in the graph or graph stack given by \code{dat}.
}
\usage{
isolates(dat, diag=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph or graph stack }
  \item{diag}{ Boolean indicating whether adjacency matrix diagonals (i.e., loops) contain meaningful data }
}
%\details{
%}
\value{
  A vector containing the isolates, or a list of vectors if more than one graph was specified
}
\references{ Wasserman, S., and Faust, K.  (1994).  ``Social Network Analysis: Methods and Applications.''  Cambridge: Cambridge University Press.

West, D.B.  (1996).  \emph{Introduction to Graph Theory}.  Upper Saddle River, NJ: Prentice Hall.}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{is.isolate}}, \code{\link{addisolates}} }

\examples{
#Generate a test graph
g<-rgraph(20)
g[,4]<-0          #Create an isolate
g[4,]<-0

#List the isolates
isolates(g)
}
\keyword{ math }%-- one or more ...

\eof
\name{lab.optimize}
\alias{lab.optimize}
\alias{lab.optimize.anneal}
\alias{lab.optimize.exhaustive}
\alias{lab.optimize.gumbel}
\alias{lab.optimize.hillclimb}
\alias{lab.optimize.mc}

%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Optimize a Bivariate Graph Statistic Across a Set of Accessible Permutations}
\description{
  \code{lab.optimize} is the front-end to a series of heuristic optimization routines (see below), all of which seek to maximize/minimize some bivariate graph statistic (e.g., graph correlation) across a set of vertex relabelings.
}
\usage{
lab.optimize(d1, d2, FUN, exchange.list=0, seek="min", 
    opt.method=c("anneal", "exhaustive", "mc", "hillclimb", 
    "gumbel"), ...)
lab.optimize.anneal(d1, d2, FUN, exchange.list=0, seek="min", 
    prob.init=1, prob.decay=0.99, freeze.time=1000, 
    full.neighborhood=TRUE, ...)
lab.optimize.exhaustive(d1, d2, FUN, exchange.list=0, seek="min", ...)
lab.optimize.gumbel(d1, d2, FUN, exchange.list=0, seek="min", 
    draws=500, tol=1e-5, estimator="median", ...)
lab.optimize.hillclimb(d1, d2, FUN, exchange.list=0, seek="min", ...)
lab.optimize.mc(d1, d2, FUN, exchange.list=0, seek="min", 
    draws=1000, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{d1}{ A single adjacency matrix }
  \item{d2}{ Another single adjacency matrix }
  \item{FUN}{ A function taking two graphs as its first two arguments, and returning a numeric value }
  \item{exchange.list}{ Information on which vertices are exchangeable (see below); this must be a single number, a vector of length n, or a nx2 matrix. }
  \item{seek}{ "min" if the optimizer should seek a minimum, or "max" if a maximum should be sought }
  \item{opt.method}{ The particular optimization method to use }
  \item{prob.init}{Initial acceptance probability for a downhill move (\code{lab.optimize.anneal} only)}
  \item{prob.decay}{The decay (cooling) multiplier for the probability of accepting a downhill move (\code{lab.optimize.anneal} only)}
  \item{freeze.time}{Number of iterations at which the annealer should be frozen (\code{lab.optimize.anneal} only)}
  \item{full.neighborhood}{Should all moves in the binary-exchange neighborhood be evaluated at each iteration? (\code{lab.optimize.anneal} only)}
  \item{tol}{Tolerance for estimation of gumbel distribution parameters (\code{lab.optimize.gumbel} only)}
  \item{estimator}{Gumbel distribution statistic to use as optimal value prediction; must be one of ``mean'', ``median'', or ``mode'' (\code{lab.optimize.gumbel} only)}
  \item{draws}{Number of draws to take for gumbel and mc methods }
  \item{\dots}{ Additional arguments to \code{FUN} }
}
\details{
\code{lab.optimize} is the front-end to a family of routines for optimizing a bivariate graph statistic over a set of permissible relabelings (or equivalently, permutations).  The accessible permutation set is determined by the \code{exchange.list} argument, which is dealt with in the following manner. First, \code{exchange.list} is expanded to fill an nx2 matrix.  If \code{exchange.list} is a single number, this is trivially accomplished by replication; if \code{exchange.list} is a vector of length n, the matrix is formed by cbinding two copies together.  If \code{exchange.list} is already an nx2 matrix, it is left as-is.  Once the nx2 exchangeabiliy matrix has been formed, it is interpreted as follows: columns refer to graphs 1 and 2, respectively; rows refer to their corresponding vertices in the original adjacency matrices; and vertices are taken to be theoretically exchangeable iff their corresponding exchangeability matrix values are identical.  To obtain an unlabeled graph statistic (the default), then, one could simply let \code{exchange.list} equal any single number.  To obtain the labeled statistic, one would use the vector \code{1:n}.

Assuming a non-degenerate set of accessible permutations/relabelings, optimization proceeds via the algorithm specified in \code{opt.method}. The optimization routines which are currently implemented use a variety of different techniques, each with certain advantages and disadvantages.  A brief summary of each is as follows:

\enumerate{
\item exhaustive search (``exhaustive''): Under exhaustive search, the entire space of accessible permutations is combed for the global optimum.  This guarantees a correct answer, but at a very high price: the set of all permutations grows with the factorial of the number of vertices, and even substantial exchangeability constraints are unlikely to keep the number of permutations from growing out of control.  While exhaustive search \emph{is} possible for small graphs, unlabeled structures of size approximately 10 or greater cannot be treated using this algorithm within a reasonable time frame.  

Approximate complexity: on the order of \eqn{\prod_{i \in L}|V_i|!}{prod( |V_i|!, i in L )}, where L is the set of exchangeability classes.

\item hill climbing (``hillclimb''): The hill climbing algorithm employed here searches, at each iteration, the set of all permissible binary exchanges of vertices.  If one or more exchanges are found which are superior to the current permutation, the best alternative is taken.  If no superior alternative is found, then the algorithm terminates.  As one would expect, this algorithm is guaranteed to terminate on a local optimum; unfortunately, however, it is quite prone to becoming ``stuck'' in suboptimal solutions.  In general, hill climbing is not recommended for permutation search, but the method may prove useful in certain circumstances.  

Approximate complexity: on the order of \eqn{|V(G)|^2} per iteration, total complexity dependent on the number of iterations.

\item simulated annealing (``anneal''): The (fairly simple) annealing procedure here employed proceeds as follows.  At each iteration, the set of all permissible binary exchanges (if \code{full.neighborhood==TRUE}) or a random selection from this set is evaluated.  If a superior option is identified, the best of these is chosen.  If no superior options are found, then the algorithm chooses randomly from the set of alternatives with probability equal to the current temperature, otherwise retaining its prior solution.  After each iteration, the current temperature is reduced by a factor equal to \code{prob.decay}; the initial temperature is set by \code{prob.init}.  When a number of iterations equal to \code{freeze.time} have been completed, the algorithm ``freezes.''  Once ``frozen,'' the annealer hillclimbs from its present location until no improvement is found, and terminates.  At termination, the best permutation identified so far is utilized; this need not be the most recent position (though it sometimes is).  

Simulated annealing is sometimes called ``noisy hill climbing'' because it uses the introduction of random variation to a hill climbing routine to avoid convergence to local optima; it works well on reasonably correlated search spaces with well-defined solution neighborhoods, and is far more robust than hill climbing algorithms.  As a general rule, simulated annealing is recommended here for most graphs up to size approximately 50.  At this point, computational complexity begins to become a serious barrier, and alternative methods may be more practical.  

Approximate complexity: on the order of \eqn{|V(G)|^2}*\code{freeze.time} if \code{full.neighborhood==TRUE}, otherwise complexity scales approximately linearly with \code{freeze.time}.  This can be misleading, however, since failing to search the full neighborhood generally requires that \code{freeze.time} be greatly increased.)

\item blind monte carlo search (``mc''): Blind monte carlo search, as the name implies, consists of randomly drawing a sample of permutations from the accessible permutation set and selecting the best.  Although this not such a bad option when A) a large fraction of points are optimal or nearly optimal and B) the search space is largely uncorrelated, these conditions do not seem to characterize most permutation search problems.  Blind monte carlo search is not generally recommended, but it is provided as an option should it be desired (e.g., when it is absolutely necessary to control the number of permutations examined).  

Approximate complexity: linear in \code{draws}.

\item extreme value estimation (``gumbel''): Extreme value estimation attempts to estimate a global optimum via stochastic modeling of the distribution of the graph statistic over the space of accessible permutations.  The algorithm currently proceeds as follows.  First, a random sample is taken from the accessible permutation set (as with monte carlo search, above).  Next, this sample is used to fit an extreme value (gumbel) model; the gumbel distribution is the limiting distribution of the extreme values from samples under a continuous, unbounded distribution, and we use it here as an approximation.  Having fit the model, an associated statistic (the mean, median, or mode as determined by \code{estimator}) is then used as an estimator of the global optimum.

Obviously, this approach has certain drawbacks.  First of all, our use of the gumbel model in particular assumes an unbounded, continuous underlying distribution, which may or may not be approximately true for any given problem.  Secondly, the inherent non-robustness of extremal problems makes the fact that our prediction rests on a string of approximations rather worrisome: our idea of the shape of the underlying distribution could be distorted by a bad sample, our parameter estimation could be somewhat off, etc., any of which could have serious consequences for our extremal prediction.  Finally, the prediction which is made by the extreme value model is \emph{nonconstructive}, in the sense that \emph{no permutation need have been found by the algorithm which induces the predicted value}.  On the bright side, this \emph{could} allow one to estimate the optimum without having to find it directly; on the dark side, this means that the reported optimum could be a numerical chimera.

At this time, extreme value estimation should be considered \emph{experimental}, and \emph{is not recommended for use on substantive problems.}  \code{lab.optimize.gumbel} is not guaranteed to work properly, or to produce intelligible results; this may eventually change in future revisions, or the routine may be scrapped altogether.

Approximate complexity: linear in \code{draws}.
}

This list of algorithms is itself somewhat unstable: some additional techniques (canonical labeling and genetic algorithms, for instance) may be added, and some existing methods (e.g., extreme value estimation) may be modified or removed.  Every attempt will be made to keep the command format as stable as possible for other routines (e.g., \code{\link{gscov}}, \code{\link{structdist}}) which depend on \code{lab.optimize} to do their heavy-lifting.  In general, it is not expected that the end-user will call \code{lab.optimize} directly; instead, most end-user interaction with these routines will be via the structural distance/covariance functions which used them.
}
\value{
The estimated global optimum of \code{FUN} over the set of relabelings permitted by \code{exchange.list}
}
\references{ Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Interstructural Analysis.''  CASOS Working Paper, Carnegie Mellon University. }
\author{ Carter T. Butts \email{buttsc@uci.edu}  }
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{gscov}}, \code{\link{gscor}}, \code{\link{structdist}}, \code{\link{sdmat}} }

\examples{
#Generate a random graph and copy it
g<-rgraph(10)
g2<-rmperm(g)  #Permute the copy randomly

#Seek the maximum correlation
lab.optimize(g,g2,gcor,seek="max",opt.method="anneal",freeze.time=50,
    prob.decay=0.9)

#These two don't do so well...
lab.optimize(g,g2,gcor,seek="max",opt.method="hillclimb")     
lab.optimize(g,g2,gcor,seek="max",opt.method="mc",draws=1000)

}
\keyword{ optimize }%-- one or more ...
\keyword{ math }

\eof
\name{lnam}
\alias{lnam}
\alias{coef.lnam}
\alias{se.lnam}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Fit a Linear Network Autocorrelation Model }
\description{
  \code{lnam} is used to fit linear network autocorrelation models.  These include standard OLS as a special case, although \code{\link{lm}} is to be preferred for such analyses.
}
\usage{
lnam(y, x = NULL, W1 = NULL, W2 = NULL, theta.seed = NULL, 
    null.model = c("meanstd", "mean", "std", "none"), method = "BFGS", 
    control = list())
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{y}{ a vector of responses. }
  \item{x}{ a vector or matrix of covariates; if the latter, each column should contain a single covariate. }
  \item{W1}{ a (possibly valued) adjacency matrix on the elements of \code{y}. }
  \item{W2}{ another (possibly valued) adjacency matrix on the elements of \code{y}. }
  \item{theta.seed}{ an optional seed value for \code{\link{optim}}. }
  \item{null.model}{ the null model to be fit; must be one of \code{"meanstd"}, \code{"mean"}, \code{"std"}, or \code{"none"}. }
  \item{method}{ method to be used with \code{\link{optim}}. }
  \item{control}{ optional control parameters for \code{\link{optim}}. }
}
\details{
  \code{lnam} fits the linear network autocorrelation model given by
  
  \deqn{y = \rho_1 W_1 y + X \beta + e, \quad e=\rho_2 W_2 e + \nu}{%
  y = rho1 * W1 \%*\% y + X \%*\% beta + e, e = rho2 * W2 \%*\% e + nu}

  where \eqn{y} is a vector of responses, \eqn{X} is a covariate matrix, \eqn{W_1}{W1} and \eqn{W_2}{W2} are (possibly valued) adjacency matrices, and \eqn{\nu \sim N(0,\sigma^2)}{nu ~ Norm(0,sigma^2)}.  Intuitively, \eqn{\rho_1}{rho1} is an ``AR''-like parameter (parameterizing the autoregression of each \eqn{y} value on its neighbors in \eqn{W_1}{W1}) while \eqn{\rho_2}{rho2} is an ``MA''-like parameter (parameterizing the autocorrelation of each \emph{disturbance} in \eqn{y} on its neighbors in \eqn{W_2}{W2}).  In general, the two models are distinct, and either or both effects may be selected by including the appropriate matrix arguments.

Model parameters are estimated by maximum likelihood, and asymptotic standard errors are provided as well; all of the above (and more) can be obtained by means of the appropriate \code{print} and \code{summary} methods.  A plotting method is also provided, which supplies fit basic diagnostics for the estimated model.  For purposes of comparison, fits may be evaluated against one of four null models:

\enumerate{
\item \code{meanstd}: mean and standard deviation estimated (default).
\item \code{mean}: mean estimated; standard deviation assumed equal to 1.
\item \code{std}: standard deviation estimated; mean assumed equal to 0.
\item \code{none}: no parameters estimated; data assumed to be drawn from a standard normal density.
}

The default setting should be appropriate for the vast majority of cases, although the others may have use when fitting ``pure'' autoregressive models (e.g., without covariates).  Although a major use of the \code{lnam} is in controlling for network autocorrelation within a regression context, the model is subtle and has a variety of uses.  (See the references below for suggestions.)
}
\value{
  An object of class \code{"lnam"} containing the following elements:

  \item{y}{the response vector used.}
  \item{x}{if supplied, the coefficient matrix.}
  \item{W1}{if supplied, the W1 matrix.}
  \item{W2}{if supplied, the W2 matrix.}
  \item{model}{a code indicating the model terms fit.}
  \item{infomat}{the estimated Fisher information matrix for the fitted model.}
  \item{acvm}{the estimated asymptotic covariance matrix for the model parameters.}
  \item{null.model}{a string indicating the null model fit.}
  \item{lnlik.null}{the log-likelihood of y under the null model.}
  \item{df.null.resid}{the residual degrees of freedom under the null model.}
  \item{df.null}{the model degrees of freedom under the null model.}
  \item{null.param}{parameter estimates for the null model.}
  \item{lnlik.model}{the log-likelihood of y under the fitted model.}
  \item{df.model}{the model degrees of freedom.}
  \item{df.residual}{the residual degrees of freedom.}
  \item{df.total}{the total degrees of freedom.}
  \item{rho1}{if applicable, the MLE for rho1.}
  \item{rho1.se}{if applicable, the asymptotic standard error for rho1.}
  \item{rho2}{if applicable, the MLE for rho2.}
  \item{rho2.se}{if applicable, the asymptotic standard error for rho2.}
  \item{sigma}{the MLE for sigma.}
  \item{sigma.se}{the standard error for sigma}
  \item{beta}{if applicable, the MLE for beta.}
  \item{beta.se}{if applicable, the asymptotic standard errors for beta.}
  \item{fitted.values}{the fitted mean values.}
  \item{residuals}{the residuals (response minus fitted); note that these correspond to \eqn{\hat{e}}{e-hat} in the model equation, not \eqn{\hat{\nu}}{nu-hat}.}
  \item{disturbances}{the estimated disturbances, i.e., \eqn{\hat{\nu}}{nu-hat}.}
  \item{call}{the matched call.}
}
\references{ 
Leenders, T.Th.A.J.  (2002)  ``Modeling Social Influence Through Network Autocorrelation: Constructing the Weight Matrix''  \emph{Social Networks}, 24(1), 21-47. 

Anselin, L.  (1988)  \emph{Spatial Econometrics: Methods and Models.}  Norwell, MA: Kluwer.
 }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ Actual optimization is performed by calls to \code{\link{optim}}.  Information on algorithms and control parameters can be found via the appropriate man pages. }

% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{ \code{\link{lm}}, \code{\link{optim}} }
\examples{
#Construct a simple, random example:
w1<-rgraph(100)               #Draw the AR matrix
w2<-rgraph(100)               #Draw the MA matrix
x<-matrix(rnorm(100*5),100,5) #Draw some covariates
r1<-0.2                       #Set the model parameters
r2<-0.1
sigma<-0.1
beta<-rnorm(5)
#Assemble y from its components:
nu<-rnorm(100,0,sigma)          #Draw the disturbances
e<-qr.solve(diag(100)-r2*w2,nu) #Draw the effective errors
y<-qr.solve(diag(100)-r1*w1,x\%*\%beta+e)  #Compute y

#Now, fit the autocorrelation model:
fit<-lnam(y,x,w1,w2)
summary(fit)
plot(fit)
}
\keyword{ regression }% at least one, from doc/KEYWORDS
\keyword{ multivariate }% __ONLY ONE__ keyword per line

\eof
\name{lower.tri.remove}
\alias{lower.tri.remove}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Remove the Lower Triangles of Adjacency Matrices in a Graph Stack }
\description{
  Returns the input graph stack, with the lower triangle entries removed/replaced as indicated.
}
\usage{
lower.tri.remove(dat, remove.val=NA)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph stack }
  \item{remove.val}{ The value with which to replace the existing lower triangles }
}
\details{
  \code{lower.tri.remove} is simply a convenient way to apply \code{g[lower.tri(g)]<-remove.val} to an entire stack of adjacency matrices at once.
}
\value{
  The updated graph stack
}
\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu}}
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{lower.tri}}, \code{\link{upper.tri.remove}}, \code{\link{diag.remove}} }

\examples{
#Generate a random graph stack
g<-rgraph(3,5)
#Remove the lower triangles
g<-lower.tri.remove(g)

}
\keyword{ manip }%-- one or more ...
\keyword{ math }

\eof
\name{lubness}
\alias{lubness}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute Graph LUBness Scores }
\description{
  \code{lubness} takes a graph stack (\code{dat}) and returns the Krackhardt LUBness scores for the graphs selected by \code{g}.
}
\usage{
lubness(dat, g=1:stackcount(dat))
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph or graph stack }
  \item{g}{ Index values for the graphs to be utilized; by default, all graphs are selected }
}
\details{
In the context of a directed graph \eqn{G}, two actors \eqn{i} and \eqn{j} may be said to have an \emph{upper bound} iff there exists some actor \eqn{k} such that directed \eqn{ki} and \eqn{kj} paths are belong to \eqn{G}.  An upper bound \eqn{\ell} is known as a \emph{least upper bound} for \eqn{i} and \eqn{j} iff it belongs to at least one \eqn{ki} and \eqn{kj} path (respectively) for all \eqn{i,j} upper bounds \eqn{k}; let \eqn{L(i,j)} be an indicator which returns 1 iff such an $\ell$ exists, otherwise returning 0.  Now, let \eqn{G_1,G_2,\dots,G_n}{G_1,G_2,...,G_n} represent the weak components of \eqn{G}.  For convenience, we denote the cardinalities of these graphs' vertex sets by \eqn{|V(G)|=N} and \eqn{|V(G_i)|=N_i}, \eqn{\forall i \in 1,\dots,n}{for i in 1,...,n}.  Given this, the Krackhardt LUBness of \eqn{G} is given by

\deqn{
1-\frac{\sum_{i=1}^n \sum_{v_j,v_k \in V(G_i)} \Bigl(1-L(v_j,v_k)\Bigr)}{\sum_{i=1}^n \frac{1}{2}(N_i-1)(N_i-2)}}{%
1-Sum(Sum(1-L(v_j,v_k),v_j,v_k in V(G_i)),i=1,...,n)/Sum((N_i-1)(N_i-2)/2,i=1,...,n)}


Where all vertex pairs possess a least upper bound, Krackhardt's LUBness is equal to 1; in general, it approaches 0 as this condition is broached.  (This convergence is problematic in certain cases due to the requirement that we sum violations across components; where a graph contains no components of size three or greater, Krackhardt's LUBness is not well-defined.  \code{lubness} returns a \code{NaN} in these cases.)  

LUBness is one of four measures (\code{\link{connectedness}}, \code{\link{efficiency}}, \code{\link{hierarchy}}, and \code{\link{lubness}}) suggested by Krackhardt for summarizing hierarchical structures.  Each corresponds to one of four axioms which are necessary and sufficient for the structure in question to be an outtree; thus, the measures will be equal to 1 for a given graph iff that graph is an outtree.  Deviations from unity can be interpreted in terms of failure to satisfy one or more of the outtree conditions, information which may be useful in classifying its structural properties.
}
\value{
A vector of LUBness scores
}
\references{ Krackhardt, David.  (1994).  ``Graph Theoretical Dimensions of Informal Organizations.'' In K. M. Carley and M. J. Prietula (Eds.), \emph{Computational Organization Theory}, 89-111. Hillsdale, NJ: Lawrence Erlbaum and Associates. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ The four Krackhardt indices are, in general, nondegenerate for a relatively narrow band of size/density combinations (efficiency being the sole exception).  This is primarily due to their dependence on the reachability graph, which tends to become complete rapidly as size/density increase.  See Krackhardt (1994) for a useful simulation study. }

\seealso{ \code{\link{connectedness}}, \code{\link{efficiency}}, \code{\link{hierarchy}}, \code{\link{lubness}}, \code{\link{reachability}} }

\examples{
#Get LUBness scores for graphs of varying densities
lubness(rgraph(10,5,tprob=c(0.1,0.25,0.5,0.75,0.9)))

}
\keyword{ math }% at least one, from doc/KEYWORDS
\keyword{ univar }% __ONLY ONE__ keyword per line

\eof
\name{make.stochastic}
\alias{make.stochastic}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Make a Graph Stack Row, Column, or Row-column Stochastic }
\description{
  Returns a graph stack in which each adjacency matrix in \code{dat} has been normalized to row stochastic, column stochastic, or row-column stochastic form, as specified by \code{mode}.
}
\usage{
make.stochastic(dat, mode="rowcol", tol=0.005, 
    maxiter=prod(dim(dat)) * 100, anneal.decay=0.01, errpow=1)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph or graph stack }
  \item{mode}{ One of ``row,'' ``col,'' or ``rowcol'' }
  \item{tol}{ Tolerance parameter for the row-column normalization algorithm }
  \item{maxiter}{ Maximum iterations for the rwo-column normalization algorithm }
  \item{anneal.decay}{ Probability decay factor for the row-column annealer }
  \item{errpow}{ Power to which absolute row-column normalization errors should be raised for the annealer (i.e., the penalty function) }
}
\details{
  Row and column stochastic matrices are those whose rows and columns sum to 1 (respectively).  These are quite straightforwardly produced here by dividing each row (or column) by its sum.  Row-column stochastic matrices, by contrast, are those in which each row \emph{and} each column sums to 1.  Here, we try to produce row-column stochastic matrices whose values are as close in proportion to the original data as possible by means of an annealing algorithm.  This is probably not optimal in the long term, but the results seem to be consistent where row-column stochasticization of the original data is possible (which it is not in all cases).
}
\value{
The stochasticized adjacency matrices
}
\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu}}
%\note{  }

\section{Warning }{Rows or columns which sum to 0 in the original data will generate undefined results.  This can happen if, for instance, your input graphs contain in- or out-isolates.} 

%\seealso{  }

\examples{
#Generate a test matrix
g<-rgraph(15)

#Make it row stochastic
make.stochastic(g,mode="row")

#Make it column stochastic
make.stochastic(g,mode="col")

#(Try to) make it row-column stochastic
make.stochastic(g,mode="rowcol")
}
\keyword{ manip }
\keyword{ array }
\keyword{ algebra }%-- one or more ...

\eof
\name{mutuality}
\alias{mutuality}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find the Mutuality of a Graph }
\description{
  Returns the mutuality scores of the graphs indicated by \code{g} in \code{dat}.
}
\usage{
mutuality(dat, g=NULL)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph or graph stack }
  \item{g}{ A vector indicating which elements of \code{dat} should be analyzed; by default, all graphs are included }
}
\details{
  The mutuality of a digraph G is defined as the number of complete dyads (i.e., i<->j) within G.  (Compare this to dyadic reciprocity, the fraction of dyads within G which are symmetric.)  Mutuality is commonly employed as a measure of reciprocal tendency within the p* literature; although mutuality can be very hard to interpret in practice, it is much better behaved than many alternative measures.
}
\value{
One or more mutuality scores
}
\references{ Moreno, J.L., and Jennings, H.H.  (1938).  ``Statistics of Social Configurations.''  \emph{Sociometry}, 1, 342-374. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{grecip}} }

\examples{
#Create some random graphs
g<-rgraph(15,3)

#Get mutuality and reciprocity scores
mutuality(g)
grecip(g)         #Compare with mutuality
}
\keyword{ univar }%-- one or more ...
\keyword{ math }

\eof
\name{netcancor}
\alias{netcancor}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Canonical Correlation for Labeled Graphs }
\description{
 \code{netcancor} finds the canonical correlation(s) between the graph sets \code{x} and \code{y}, testing the result using either conditional uniform graph (CUG) or quadratic assignment procedure (QAP) null hypotheses.
}
\usage{
netcancor(y, x, mode="digraph", diag=FALSE, nullhyp="cugtie", 
    reps=1000)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{y}{ First data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors.  Missing values are not allowed. }
  \item{x}{ Second data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors.  Missing values are not allowed. }
  \item{mode}{ String indicating the type of graph being evaluated.  "digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{mode} is set to "digraph" by default. }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{nullhyp}{ String indicating the particular null hypothesis against which to test the observed estimands.  A value of "cug" implies a conditional uniform graph test (see \code{\link{cugtest}}) controlling for order \emph{only}; "cugden" controls for both order and tie probability; "cugtie" controls for order and tie distribution (via bootstrap); and "qap" implies that the QAP null hypothesis (see \code{\link{qaptest}}) should be used. }
  \item{reps}{Integer indicating the number of draws to use for quantile estimation.  (Relevant to the null hypothesis test only - the analysis itself is unaffected by this parameter.)  Note that, as for all Monte Carlo procedures, convergence is slower for more extreme quantiles. }
}
\details{
The \code{netcancor} routine is actually a front-end to the \code{mva} library's \code{\link[mva]{cancor}} routine for computing canonical correlations between sets of vectors.  \code{netcancor} itself vectorizes the network variables (as per its graph type) and manages the appropriate null hypothesis tests; the actual canonical correlation is handled by \code{\link[mva]{cancor}}.  

Canonical correlation itself is a multivariate generalization of the product-moment correlation.  Specifically, the analysis seeks linear combinations of the variables in \code{y} which are well-explained by linear combinations of the variables in \code{x}.  The network version of this technique is performed elementwise on the adjacency matrices of the graphs in question; as usual, the result should be interpreted with an eye to the relationship between the type of data used and the assumptions of the underlying model.

Intelligent printing and summarizing of netcancor objects is provided by \code{\link{print.netcancor}} and \code{\link{summary.netcancor}}.  
}
\value{
An object of class \code{netcancor} with the following properties:

\item{xdist}{
Array containing the distribution of the X coefficients under the null hypothesis test.
}
\item{ydist}{
Array containing the distribution of the Y coefficients under the null hypothesis test.
}
\item{cdist}{
Array containing the distribution of the canonical correlation coefficients under the null hypothesis test.
}
\item{cor}{
Vector containing the observed canonical correlation coefficients.
}
\item{xcoef}{
Vector containing the observed X coefficients.
}
\item{ycoef}{
Vector containing the observed Y coefficients.
}
\item{cpgreq}{
Vector containing the estimated upper tail quantiles (p>=obs) for the observed canonical correlation coefficients under the null hypothesis.
}
\item{cpleeq}{
Vector containing the estimated lower tail quantiles (p<=obs) for the observed canonical correlation coefficients under the null hypothesis.
}
\item{xpgreq}{
Matrix containing the estimated upper tail quantiles (p>=obs) for the observed X coefficients under the null hypothesis.
}
\item{xpleeq}{
Matrix containing the estimated lower tail quantiles (p<=obs) for the observed X coefficients under the null hypothesis.
}
\item{ypgreq}{
Matrix containing the estimated upper tail quantiles (p>=obs) for the observed Y coefficients under the null hypothesis.
}
\item{ypleeq}{
Matrix containing the estimated lower tail quantiles (p<=obs) for the observed Y coefficients under the null hypothesis.
}
\item{cnames}{
Vector containing names for the canonical correlation coefficients.
}
\item{xnames}{
Vector containing names for the X vars.
}
\item{ynames}{
Vector containing names for the Y vars.
}
\item{xcenter}{
Values used to adjust the X variables.
}
\item{xcenter}{
Values used to adjust the Y variables.
}
\item{nullhyp}{
String indicating the null hypothesis employed.
}
}
\references{ Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Interstructural Analysis.''  CASOS working paper, Carnegie Mellon University. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ This will eventually be replaced with a superior cancor procedure with more interpretable output; the new version will handle arbitrary labeling as well. }

% ~Make other sections like WARNING with \section{WARNING }{....} ~
\section{Requires }{\code{mva}}

\seealso{ \code{\link{gcor}}, \code{\link{cugtest}}, \code{\link{qaptest}}, \code{\link{cancor}} }

\examples{
#Generate a valued seed structure
cv<-matrix(rnorm(100),nrow=10,ncol=10)
#Produce two sets of valued graphs
x<-array(dim=c(3,10,10))
x[1,,]<-3*cv+matrix(rnorm(100,0,0.1),nrow=10,ncol=10)
x[2,,]<--1*cv+matrix(rnorm(100,0,0.1),nrow=10,ncol=10)
x[3,,]<-x[1,,]+2*x[2,,]+5*cv+matrix(rnorm(100,0,0.1),nrow=10,ncol=10)
y<-array(dim=c(2,10,10))
y[1,,]<--5*cv+matrix(rnorm(100,0,0.1),nrow=10,ncol=10)
y[2,,]<--2*cv+matrix(rnorm(100,0,0.1),nrow=10,ncol=10)
#Perform a canonical correlation analysis
nc<-netcancor(y,x,reps=100)
summary(nc)
}
\keyword{ multivariate }%-- one or more ...
\keyword{ math }

\eof
\name{netlm}
\alias{netlm}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Linear Regression for Network Data }
\description{
  \code{netlm} regresses the network variable in \code{y} on the network variables in stack \code{x} using ordinary least squares.  The resulting fits (and coefficients) are then tested against the indicated null hypothesis.
}
\usage{
netlm(y, x, mode="digraph", diag=FALSE, nullhyp="cugtie", reps=1000)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{y}{ Dependent network variable.  This should be a matrix, for obvious reasons; NAs are allowed, but dichotomous data is strongly discouraged due to the assumptions of the analysis. }
  \item{x}{ Data array containing the stack of independent network variables.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors.  Note that NAs are permitted, as is dichotomous data. }
  \item{mode}{ String indicating the type of graph being evaluated.  "digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{mode} is set to "digraph" by default. }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{Diag} is \code{FALSE} by default. }
  \item{nullhyp}{ String indicating the particular null hypothesis against which to test the observed estimands.  A value of "cug" implies a conditional uniform graph test (see \code{\link{cugtest}}) controlling for order \emph{only}; "cugden" controls for both order and tie probability; "cugtie" controls for order and tie distribution (via bootstrap); and "qap" implies that the QAP null hypothesis (see \code{\link{qaptest}}) should be used. }
  \item{reps}{ Integer indicating the number of draws to use for quantile estimation.  (Relevant to the null hypothesis test only - the analysis itself is unaffected by this parameter.)  Note that, as for all Monte Carlo procedures, convergence is slower for more extreme quantiles.  By default, \code{reps}=1000. }
}
\details{
 Unsurprisingly, \code{netlm} is really a front-end to the built-in \code{\link{lm}} routine.  \code{netlm} handles vectorization and null hypothesis testing; the actual fitting is taken care of by \code{\link{lm}}.  

Network regression using OLS is directly analogous to standard OLS regression elementwise on the appropriately vectorized adjacency matrices of the networks involved.  In particular, the network regression attempts to fit the model:

\deqn{\mathbf{A_y} = b_0 \mathbf{A_1} + b_1 \mathbf{A_{x_1}} + b_2 \mathbf{A_{x_2}} + \dots + \mathbf{Z}}{%
A_y = b_0 A_1 + b_1 A_x1 + b_2 A_x2 + \dots + Z}

where \eqn{\mathbf{A_y}}{A_y} is the dependent adjacency matrix, \eqn{\mathbf{A_{x_i}}}{A_xi} is the ith independent adjacency matrix, \eqn{\mathbf{A_1}}{A_1} is an n x n matrix of 1's, and \eqn{\mathbf{Z}}{Z} is an n x n matrix of independent normal random variables with mean 0 and variance \eqn{\sigma^2}{sigma^2}.  Clearly, this model is nonoptimal when \eqn{\mathbf{A_y}}{A_y} is dichotomous (or, for that matter, categorical in general); an alternative such as \code{netlogit} should be employed in such cases.  (Note that \code{netlm} will still attempt to fit such data...the user should consider him or herself to have been warned.)

Null hypothesis tests for the network regression are handled using either the conditional uniform graph hypothesis (the default) or QAP.  See the help pages for these tests for a fuller description of each.  Reasonable printing and summarizing of \code{netlm} objects is provided by \code{\link{print.netlm}} and \code{\link{summary.netlm}}, respectively.  No plot methods exist at this time, alas. 
}
\value{
  An object of class \code{netlm}
}
\references{Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Interstructural Analysis.''  CASOS working paper, Carnegie Mellon University. 

Krackhardt, D.  (1987).  ``QAP Partialling as a Test of Spuriousness.'' \emph{Social Networks}, 9 171-186.

Krackhardt, D.  (1988).  ``Predicting With Networks: Nonparametric Multiple Regression Analyses of Dyadic Data.''  \emph{Social Networks}, 10, 359-382.}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{lm}}, \code{\link{netlogit}} }

\examples{
#Create some input graphs
x<-rgraph(20,4)

#Create a response structure
y<-x[1,,]+4*x[2,,]+2*x[3,,]   #Note that the fourth graph is unrelated

#Fit a netlm model
nl<-netlm(y,x,reps=100)

#Examine the results
summary(nl)
}
\keyword{ regression }%-- one or more ...
\keyword{ math }

\eof
\name{netlogit}
\alias{netlogit}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Logistic Regression for Network Data }
\description{
  \code{netlogit} performs a logistic regression of the network variable in \code{y} on the network variables in stack \code{x}.  The resulting fits (and coefficients) are then tested against the indicated null hypothesis.
}
\usage{
netlogit(y, x, mode="digraph", diag=FALSE, nullhyp="cugtie", 
    reps=1000)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{y}{ Dependent network variable.  This should be a matrix, for obvious reasons; \code{NA}s are allowed, and the data should be dichotomous. }
  \item{x}{ Data array containing the stack of independent network variables.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors.  Note that \code{NA}s are permitted, as is dichotomous data. }
  \item{mode}{ String indicating the type of graph being evaluated.  "Digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{mode} is set to "digraph" by default. }
  \item{diag}{  Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{nullhyp}{ String indicating the particular null hypothesis against which to test the observed estimands.  A value of "cug" implies a conditional uniform graph test (see \code{\link{cugtest}}) controlling for order \emph{only}; "cugden" controls for both order and tie probability; "cugtie" controls for order and tie distribution (via bootstrap); and "qap" implies that the QAP null hypothesis (see \code{\link{qaptest}}) should be used. }
  \item{reps}{ Integer indicating the number of draws to use for quantile estimation.  (Relevant to the null hypothesis test only -- the analysis itself is unaffected by this parameter.)  Note that, as for all Monte Carlo procedures, convergence is slower for more extreme quantiles.  By default, \code{reps}=1000. }
}
\details{
  \code{netlogit} is primarily a front-end to the built-in \code{\link{glm}} routine.  \code{netlogit} handles vectorization, sets up \code{\link{glm}} options, and deals with null hypothesis testing; the actual fitting is taken care of by \code{\link{glm}}.  

Logistic network regression using is directly analogous to standard logistic regression elementwise on the appropriately vectorized adjacency matrices of the networks involved.  As such, it is often a more appropriate model for fitting dichotomous response networks than is linear network regression.  

Null hypothesis tests for logistic network regression are handled using either the conditional uniform graph hypothesis (the default) or QAP.  See the help pages for these tests for a fuller description of each.  Reasonable printing and summarizing of \code{netlogit} objects is provided by \code{\link{print.netlogit}} and \code{\link{summary.netlogit}}, respectively.  No plot methods exist at this time.
}
\value{
  An object of class \code{netlogit}
}
\references{Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Interstructural Analysis.''  CASOS working paper, Carnegie Mellon University.}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{}

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{glm}}, \code{\link{netlm}} }

\examples{
#Create some input graphs
x<-rgraph(20,4)

#Create a response structure
y.l<-x[1,,]+4*x[2,,]+2*x[3,,]   #Note that the fourth graph is 
                                #unrelated
y.p<-apply(y.l,c(1,2),function(a){1/(1+exp(-a))})
y<-rgraph(20,tprob=y.p)

#Fit a netlogit model
nl<-netlogit(y,x,reps=100)

#Examine the results
summary(nl)
}
\keyword{ regression }%-- one or more ...
\keyword{ math }

\eof
\name{npostpred}
\alias{npostpred}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Take Posterior Predictive Draws for Functions of Networks }
\description{
  \code{npostpred} takes a list or data frame, \code{b}, and applies the function \code{FUN} to each element of \code{b}'s \code{net} member.
}
\usage{
npostpred(b, FUN, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{b}{ A list or data frame containing posterior network draws; these draws must take the form of a graph stack, and must be the member of \code{b} referenced by "\code{net}" }
  \item{FUN}{ Function for which posterior predictive is to be estimated }
  \item{\dots}{ Additional arguments to \code{FUN} }
}
\details{
Although created to work with \code{\link{bbnam}}, \code{npostpred} is quite generic.  The form of the posterior draws will vary with the output of \code{FUN}; since invocation is handled by \code{\link{apply}}, check there if unsure.
}
\value{
A series of posterior predictive draws
}
\references{ Gelman, A.; Carlin, J.B.; Stern, H.S.; and Rubin, D.B.  (1995).  \emph{Bayesian Data Analysis.}  London: Chapman and Hall. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ ~~further notes~~ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{bbnam}} }

\examples{
#Create some random data
g<-rgraph(5)
g.p<-0.8*g+0.2*(1-g)
dat<-rgraph(5,5,tprob=g.p)

#Define a network prior
pnet<-matrix(ncol=5,nrow=5)
pnet[,]<-0.5
#Define em and ep priors
pem<-matrix(nrow=5,ncol=2)
pem[,1]<-3
pem[,2]<-5
pep<-matrix(nrow=5,ncol=2)
pep[,1]<-3
pep[,2]<-5

#Draw from the posterior
b<-bbnam(dat,model="actor",nprior=pnet,emprior=pem,epprior=pep,
    burntime=100,draws=100)
#Plot a summary of the posterior predictive of reciprocity
hist(npostpred(b,grecip))
}
\keyword{ models }%-- one or more ...
\keyword{ math }

\eof
\name{nties}
\alias{nties}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find the Number of Possible Ties in a Given Graph or Stack }
\description{
\code{nties} returns the number of possible edges in each element of \code{dat}, given \code{mode} and \code{diag}.
}
\usage{
nties(dat, mode="digraph", diag=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph or stack thereof }
  \item{mode}{ One of ``digraph'', ``graph'', and ``hgraph'' }
  \item{diag}{ A boolean indicating whether or not diagonal entries (loops) should be treated as valid data; ignored for hypergraphic (``hgraph'') data }
}
\details{
\code{nties} is used primarily to automate maximum edge counts for use with normalization routines.
}
\value{
The number of possible edges, or a vector of the same
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ For two-mode (hypergraphic) data, the value returned isn't technically the number of edges per se, but rather the number of edge memberships. }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

%\seealso{ }

\examples{
#How many possible edges in a loopless digraph of order 15?
nties(rgraph(15),diag=FALSE)
}
\keyword{ univar }%-- one or more ...
\keyword{ math }

\eof
\name{numperm}
\alias{numperm}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Get the nth Permutation Vector by Periodic Placement }
\description{
  \code{numperm} implicitly numbers all permutations of length \code{olength}, returning the \code{permnum}th of these.
}
\usage{
numperm(olength, permnum)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{olength}{ The number of items to permute }
  \item{permnum}{ The number of the permutation to use (in \code{1:olength!}) }
}
\details{
  The n! permutations on n items can be deterministically ordered via a factorization process in which there are n slots for the first element, n-1 for the second, and n-i for the ith.  This fact is quite handy if you want to visit each permutation in turn, or if you wish to sample without replacement from the set of permutations on some number of elements: one just enumerates or samples from the integers on [1,n!], and then find the associated permutation.  \code{numperm} performs exactly this last operation, returning the \code{permnum}th permutation on \code{olength} items.
}
\value{
A permutation vector
}
\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ Permutation search is central to the estimation of structural distances, correlations, and covariances on partially labeled graphs.  \code{numperm} is hence used by \code{\link{structdist}}, \code{\link{gscor}}, \code{\link{gscov}}, etc. }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{rperm}}, \code{\link{rmperm}} }

\examples{
#Draw a graph
g<-rgraph(5)

#Permute the rows and columns
p.1<-numperm(5,1)
p.2<-numperm(5,2)
p.3<-numperm(5,3)
g[p.1,p.1]
g[p.2,p.2]
g[p.3,p.3]
}
\keyword{ array }%-- one or more ...
\keyword{ math }

\eof
\name{plot.bbnam}
\alias{plot.bbnam}
\alias{plot.bbnam.fixed}
\alias{plot.bbnam.pooled}
\alias{plot.bbnam.actor}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Plotting for bbnam Objects}
\description{
  Generates various plots of posterior draws from the \code{\link{bbnam}} model.
}
\usage{
plot.bbnam(x, mode="density", intlines=TRUE, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ A \code{bbnam} object }
  \item{mode}{ ``density'' for kernel density estimators of posterior marginals; otherwise, histograms are used }
  \item{intlines}{ Plot lines for the 0.9 central posterior probability intervals? }
  \item{\dots}{ Additional arguments to \code{\link{plot}} }
}
\details{
  \code{plot.bbnam} provides plots of the estimated posterior marginals for the criterion graph and error parameters (as appropriate).  Plotting may run into difficulties when dealing with large graphs, due to the problem of getting all of the various plots on the page; the routine handles these issues reasonably intelligently, but there is doubtless room for improvement.
}
\value{
None
}
\references{Butts, C.T. (1999). ``Informant (In)Accuracy and Network Estimation: A Bayesian Approach.'' CASOS Working Paper, Carnegie Mellon University. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{bbnam}}}

\examples{
#Create some random data
g<-rgraph(5)
g.p<-0.8*g+0.2*(1-g)
dat<-rgraph(5,5,tprob=g.p)

#Define a network prior
pnet<-matrix(ncol=5,nrow=5)
pnet[,]<-0.5
#Define em and ep priors
pem<-matrix(nrow=5,ncol=2)
pem[,1]<-3
pem[,2]<-5
pep<-matrix(nrow=5,ncol=2)
pep[,1]<-3
pep[,2]<-5

#Draw from the posterior
b<-bbnam(dat,model="actor",nprior=pnet,emprior=pem,epprior=pep,
    burntime=100,draws=100)
#Print a summary of the posterior draws
summary(b)
#Plot the result
plot(b)
}
\keyword{ hplot}%-- one or more ...

\eof
\name{plot.blockmodel}
\alias{plot.blockmodel}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Plotting for blockmodel Objects }
\description{
  Displays a plot of the blocked data matrix, given a blockmodel object.
}
\usage{
plot.blockmodel(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{blockmodel} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
\details{
Plots of the blocked data matrix (i.e., the data matrix with rows and columns permuted to match block membership) can be useful in assessing the strength of the block solution (particularly for clique detection and/or regular equivalence).
}
\value{
  None
}
\references{ White, H.C.; Boorman, S.A.; and Breiger, R.L.  (1976).  ``Social Structure from Multiple Networks I: Blockmodels of Roles and Positions.''  \emph{American Journal of Sociology}, 81, 730-779.

Wasserman, S., and Faust, K.  (1994).  \emph{Social Network Analysis: Methods and Applications.}  Cambridge: Cambridge University Press.}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{blockmodel}}, \code{\link{plot.sociomatrix}} }

\examples{
#Create a random graph with _some_ edge structure
g.p<-sapply(runif(20,0,1),rep,20)  #Create a matrix of edge 
                                   #probabilities
g<-rgraph(20,tprob=g.p)            #Draw from a Bernoulli graph 
                                   #distribution

#Cluster based on structural equivalence
eq<-equiv.clust(g)

#Form a blockmodel with distance relaxation of 10
b<-blockmodel(g,eq,h=10)
plot(b)                            #Plot it
}
\keyword{ hplot }%-- one or more ...

\eof
\name{plot.cugtest}
\alias{plot.cugtest}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Plotting for cugtest Objects }
\description{
  Plots the distribution of a CUG test statistic.
}
\usage{
plot.cugtest(x, mode="density", ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ A \code{\link{cugtest}} object }
  \item{mode}{ ``density'' for kernel density estimation, ``hist'' for histogram }
  \item{\dots}{ Additional arguments to \code{\link{plot}} }
}
\details{
  In addition to the quantiles associated with a CUG test, it is often useful to examine the form of the distribution of the test statistic.  \code{plot.cugtest} facilitates this.
}
\value{
None
}
\references{ Anderson, B.S.; Butts, C.T.; and Carley, K.M. (1999). ``The Interaction of Size and Density with Graph-Level Indices.'' \emph{Social Networks}, 21(3), 239-267. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{cugtest}} }

\examples{
#Draw two random graphs, with different tie probabilities
dat<-rgraph(20,2,tprob=c(0.2,0.8))

#Is their correlation higher than would be expected, conditioning 
#only on size?
cug<-cugtest(dat,gcor,cmode="order")
summary(cug)
plot(cug)

#Now, let's try conditioning on density as well.
cug<-cugtest(dat,gcor)
plot(cug)
}
\keyword{ hplot }%-- one or more ...

\eof
\name{plot.equiv.clust}
\alias{plot.equiv.clust}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Plot an equiv.clust Object }
\description{
 Plots a hierarchical clustering of node positions as generated by \code{\link{equiv.clust}}.
}
\usage{
plot.equiv.clust(x, labels=x$plabels, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An \code{\link{equiv.clust}} object }
  \item{labels}{ A vector of vertex labels }
  \item{\dots}{ Additional arguments to \code{\link[mva]{plot.hclust}} }
}
\details{
\code{plot.equiv.clust} is actually a front-end to \code{\link[mva]{plot.hclust}}; see the latter for more additional documentation.
}
\value{
None.
}
\references{ Breiger, R.L.; Boorman, S.A.; and Arabie, P.  (1975).  ``An Algorithm for Clustering Relational Data with Applications to Social Network Analysis and Comparison with Multidimensional Scaling.''  \emph{Journal of Mathematical Psychology}, 12, 328-383.

Burt, R.S.  (1976).  ``Positions in Networks.''  \emph{Social Forces}, 55, 93-122.

Wasserman, S., and Faust, K.  \emph{Social Network Analysis: Methods and Applications.}  Cambridge: Cambridge University Press.}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\section{Requires}{\code{mva}}

\seealso{ \code{\link{equiv.clust}}, \code{\link{plot.hclust}} }

\examples{
#Create a random graph with _some_ edge structure
g.p<-sapply(runif(20,0,1),rep,20)  #Create a matrix of edge 
                                   #probabilities
g<-rgraph(20,tprob=g.p)            #Draw from a Bernoulli graph 
                                   #distribution

#Cluster based on structural equivalence
eq<-equiv.clust(g)
plot(eq)
}
\keyword{ hplot }%-- one or more ...

\eof
\name{plot.lnam}
\alias{plot.lnam}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Plotting for lnam Objects }
\description{
  Generates various diagnostic plots for \code{\link{lnam} objects.}
}
\usage{
plot.lnam(x, ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{x}{ an object of class \code{lnam}. }
  \item{\dots}{ additional arguments to \code{\link{plot}}. }
}
%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}
\value{
None
}
%\references{ ~put references to the literature/web site here ~ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ ~~further notes~~ }
%
% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{ \code{\link{lnam}} }
%\examples{
%}
\keyword{ hplot }% at least one, from doc/KEYWORDS
%\keyword{ ~kwd2 }% __ONLY ONE__ keyword per line

\eof
\name{plot.qaptest}
\alias{plot.qaptest}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Plotting for qaptest Objects }
\description{
  Plots the Distribution of a QAP Test Statistic.
}
\usage{
plot.qaptest(x, mode="density", ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ A \code{\link{qaptest}} object }
  \item{mode}{ ``density'' for kernel density estimation, ``hist'' for histogram }
  \item{\dots}{ Additional arguments to \code{\link{plot}} }
}
\details{
  In addition to the quantiles associated with a QAP test, it is often useful to examine the form of the distribution of the test statistic.  \code{plot.qaptest} facilitates this.
}
\value{
  None
}
\references{
Hubert, L.J., and Arabie, P.  (1989).  ``Combinatorial Data Analysis: Confirmatory Comparisons Between Sets of Matrices.''  \emph{Applied Stochastic Models and Data Analysis}, 5, 273-325.

Krackhardt, D.  (1987).  ``QAP Partialling as a Test of Spuriousness.'' \emph{Social Networks}, 9 171-186.

Krackhardt, D.  (1988).  ``Predicting With Networks: Nonparametric Multiple Regression Analyses of Dyadic Data.''  \emph{Social Networks}, 10, 359-382.}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{qaptest}} }

\examples{

#Generate three graphs
g<-array(dim=c(3,10,10))
g[1,,]<-rgraph(10)
g[2,,]<-rgraph(10,tprob=g[1,,]*0.8)
g[3,,]<-1; g[3,1,2]<-0              #This is nearly a clique

#Perform qap tests of graph correlation
q.12<-qaptest(g,gcor,g1=1,g2=2)
q.13<-qaptest(g,gcor,g1=1,g2=3)

#Examine the results
summary(q.12)
plot(q.12)
summary(q.13)
plot(q.13)
}
\keyword{ hplot }%-- one or more ...

\eof
\name{plot.sociomatrix}
\alias{plot.sociomatrix}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Plot Matrices Using a Color/Intensity Grid }
\description{
  Plots a matrix, \code{m}, associating the magnitude of the i,jth cell of \code{m} with the color of the i,jth cell of an \code{nrow(m)} by \code{ncol(m)} grid.
}
\usage{
plot.sociomatrix(x, labels=list(seq(1:dim(x)[1]), seq(1:dim(x)[2])), 
    drawlab=TRUE, diaglab=TRUE, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An input matrix }
  \item{labels}{ A list containing the vectors of row and column labels (respectively) }
  \item{drawlab}{ Add labels to the plot? }
  \item{diaglab}{ Label the diagonal? }
  \item{\dots}{ Additional arguments to \code{\link{plot}} }
}
\details{
  \code{plot.sociomatrix} is particularly valuable for examining large adjacency matrices, whose structure can be non-obvious otherwise.
}
\value{
None
}
%\references{ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

%\seealso{  }

\examples{
#Plot a small adjacency matrix
plot.sociomatrix(rgraph(5))

#Plot a much larger one
plot.sociomatrix(rgraph(100),drawlab=FALSE,diaglab=FALSE)
}
\keyword{ hplot }%-- one or more ...

\eof
\name{potscalered.mcmc}
\alias{potscalered.mcmc}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute Gelman and Rubin's Potential Scale Reduction Measure for a Markov Chain Monte Carlo Simulation}
\description{
  Computes Gelman and Rubin's (simplified) measure of scale reduction for draws of a single scalar estimand from parallel MCMC chains. 
}
\usage{
potscalered.mcmc(psi)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{psi}{ An nxm matrix, with columns corresponding to chains and rows corresponding to iterations. }
}
\details{
  The Gelman and Rubin potential scale reduction (\eqn{\sqrt{\hat{R}}}{sqrt(hat(R))}) provides an ANOVA-like comparison of the between-chain to within-chain variance on a given scalar estimand; the disparity between these gives an indication of the extent to which the scale of the simulated distribution can be reduced via further sampling.  As the parallel chains converge \eqn{\sqrt{\hat{R}}}{sqrt(hat(R))} approaches 1 (from above), and it is generally recommended that values of 1.2 or less be obtained before a series of draws can be considered well-mixed.  (Even so, one should ideally examine other indicators of chain mixing, and verify that the properties of the draws are as they should be.  There is currently no fool-proof way to verify burn-in of an MCMC, but using multiple indicators should help one avoid falling prey to the idiosyncrasies of any one index.)

Note that the particular estimators used in the \eqn{\sqrt{\hat{R}}}{sqrt(hat(R))} formulation are based on normal-theory results, and as such have been criticized vis a vis their behavior on other distributions.  Where simulating distributions whose properties differ greatly from the normal, an alternative form of the measure using robust measures of scale (e.g., the IQR) may be preferable.
}
\value{
The potential scale reduction measure
}
\references{ Gelman, A.; Carlin, J.B.; Stern, H.S.; and Rubin, D.B.  (1995).  \emph{Bayesian Data Analysis.}  London: Chapman and Hall.

Gelman, A., and Rubin, D.B.  (1992).  ``Inference from Iterative Simulation Using Multiple Sequences.''  \emph{Statistical Science,} 7, 457-511. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{bbnam}}}

\examples{

}
\keyword{ models }%-- one or more ...
\keyword{univar}

\eof
\name{prestige}
\alias{prestige}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Calculate the Vertex Prestige Scores }
\description{
  \code{prestige} takes a graph stack (\code{dat}) and returns the prestige scores of positions within one graph (indicated by \code{nodes} and \code{g}, respectively).  Depending on the specified mode, prestige based on any one of a number of different definitions will be returned. This function is compatible with \code{\link{centralization}}, and will return the theoretical maximum absolute deviation (from maximum) conditional on size (which is used by \code{\link{centralization}} to normalize the observed centralization score).
}
\usage{
prestige(dat, g=1, nodes=c(1:dim(dat)[2]), gmode="digraph", 
    diag=FALSE, cmode="indegree", tmaxdev=FALSE, rescale=FALSE, 
    tol=1e-07)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors. Alternately, this can be an n x n matrix (if only one graph is involved). }
  \item{g}{ Integer indicating the index of the graph for which centralities are to be calculated.  By default, \code{g==1}. }
  \item{nodes}{ List indicating which nodes are to be included in the calculation.  By default, all nodes are included. }
  \item{gmode}{ String indicating the type of graph being evaluated.  "digraph" indicates that edges should be interpreted as directed; "graph" indicates that edges are undirected.  \code{gmode} is set to "digraph" by default.}
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{cmode}{ One of "indegree", "indegree.rownorm", "indegree.rowcolnorm", "eigenvector", "eigenvector.rownorm", "eigenvector.colnorm", "eigenvector.rowcolnorm", "domain", or "domain.proximity" }
  \item{tmaxdev}{ Boolean indicating whether or not the theoretical maximum absolute deviation from the maximum nodal centrality should be returned.  By default, \code{tmaxdev==FALSE}. }
  \item{rescale}{ If true, centrality scores are rescaled such that they sum to 1. }
  \item{tol}{ Currently ignored }
}
\details{
  "Prestige" is the name collectively given to a range of centrality scores which focus on the extent to which one is nominated by others.  The definitions supported here are as follows:
\enumerate{
\item indegree: indegree centrality

\item indegree.rownorm: indegree within the row-normalized graph

\item indegree.rowcolnorm: indegree within the row-column normalized graph

\item eigenvector: eigenvector centrality within the transposed graph (i.e., incoming ties recursively determine prestige)

\item eigenvector.rownorm: eigenvector centrality within the transposed row-normalized graph

\item eigenvector.colnorm: eigenvector centrality within the transposed column-normalized graph

\item eigenvector.rowcolnorm: eigenvector centrality within the transposed row/column-normalized graph

\item domain: indegree within the reachability graph (Lin's unweighted measure)

\item domain.proximity: Lin's proximity-weighted domain prestige
}

Note that the centralization of prestige is simply the extent to which one actor has substantially greater prestige than others; the underlying definition is the same.
}
\value{
  A vector of prestige scores
}
\references{ Lin, N.  (1976).  \emph{Foundations of Social Research}.  New York: McGraw Hill.

Wasserman, S., and Faust, K.  (1994).  \emph{Social Network Analysis: Methods and Applications.}  Cambridge: Cambridge University Press. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 \section{Warning }{Making adjacency matrices doubly stochastic (row-column normalization) is not guaranteed to work.  In general, be wary of attempting to try normalizations on graphs with degenerate rows and columns.} 

\seealso{ \code{\link{centralization}}}

\examples{
g<-rgraph(10)                 #Draw a random graph with 10 members
prestige(g,cmode="domain")    #Compute domain prestige scores
}
\keyword{univar}
\keyword{ math }

\eof
\name{print.bayes.factor}
\alias{print.bayes.factor}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for Bayes Factor Objects }
\description{
  Prints a quick summary of a Bayes Factor object.
}
\usage{
print.bayes.factor(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{bayes.factor} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
None
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{bbnam.bf}}}

\examples{
}
\keyword{ print }%-- one or more ...

\eof
\name{print.bbnam}
\alias{print.bbnam}
\alias{print.bbnam.fixed}
\alias{print.bbnam.pooled}
\alias{print.bbnam.actor}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for bbnam Objects }
\description{
Prints a quick summary of posterior draws from \code{\link{bbnam}}.
}
\usage{
print.bbnam(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ A \code{\link{bbnam}} object }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
None
}
%\references{ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{bbnam}} }

\examples{
}
\keyword{ print }%-- one or more ...

\eof
\name{print.blockmodel}
\alias{print.blockmodel}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for blockmodel Objects }
\description{
Prints a quick summary of a \code{\link{blockmodel}} object.
}
\usage{
print.blockmodel(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{blockmodel} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
None
}
%\references{ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{blockmodel}} }

\examples{
}
\keyword{ print }%-- one or more ...

\eof
\name{print.cugtest}
\alias{print.cugtest}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for cugtest Objects}
\description{
  Prints a quick summary of objects produced by \code{\link{cugtest}}.
}
\usage{
print.cugtest(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{cugtest} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
None.
}
%\references{}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{cugtest}} }

\examples{
}
\keyword{ print }%-- one or more ...

\eof
\name{print.lnam}
\alias{print.lnam}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Printing for lnam Objects }
\description{
  Prints an objsect of class \code{lnam} 
}
\usage{
print.lnam(x, digits = max(3, getOption("digits") - 3), ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{x}{ an object of class \code{lnam}. }
  \item{digits}{ number of digits to display. }
  \item{\dots}{ additional arguments. }
}
%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}
\value{
 None.
}
%\references{ ~put references to the literature/web site here ~ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ ~~further notes~~ }
%
% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{ \code{\link{lnam}} }
%\examples{
%}
\keyword{ print }% at least one, from doc/KEYWORDS
%\keyword{ ~kwd2 }% __ONLY ONE__ keyword per line

\eof
\name{print.netcancor}
\alias{print.netcancor}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for netcancor Objects }
\description{
  Prints a quick summary of objects produced by \code{\link{netcancor}}.
}
\usage{
print.netcancor(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{netcancor}}
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
}
%\references{}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{netcancor}} }

\examples{
}
\keyword{ print }%-- one or more ...

\eof
\name{print.netlm}
\alias{print.netlm}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for netlm Objects }
\description{
  Prints a quick summary of objects produced by \code{\link{netlm}}.
}
\usage{
print.netlm(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{netlm}}
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{netlm}}}

\examples{
}
\keyword{ print }%-- one or more ...

\eof
\name{print.netlogit}
\alias{print.netlogit}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for netlogit Objects }
\description{
  Prints a quick summary of objects produced by \code{\link{netlogit}}.
}
\usage{
print.netlogit(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{netlogit} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{netlogit}} }

\examples{
}
\keyword{ print }%-- one or more ...

\eof
\name{print.qaptest}
\alias{print.qaptest}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for qaptest Objects }
\description{
Prints a quick summary of objects produced by \code{\link{qaptest}}.
}
\usage{
print.qaptest(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{qaptest} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{qaptest}}}

\examples{
}
\keyword{ print }%-- one or more ...

\eof
\name{print.summary.bayes.factor}
\alias{print.summary.bayes.factor}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for summary.bayes.factor Objects }
\description{
  Prints an object of class \code{summary.bayes.factor}.
}
\usage{
print.summary.bayes.factor(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{summary.bayes.factor} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{summary.bayes.factor}} }

\examples{
}
\keyword{ print }%-- one or more ...

\eof
\name{print.summary.bbnam}
\alias{print.summary.bbnam}
\alias{print.summary.bbnam.fixed}
\alias{print.summary.bbnam.pooled}
\alias{print.summary.bbnam.actor}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for summary.bbnam Objects }
\description{
  Prints an object of class \code{summary.bbnam}.
}
\usage{
print.summary.bbnam(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{summary.bbnam} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
}
%\references{ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{bbnam}} }

\examples{

}
\keyword{ print }%-- one or more ...

\eof
\name{print.summary.blockmodel}
\alias{print.summary.blockmodel}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for summary.blockmodel Objects }
\description{
  Prints an object of class \code{summary.blockmodel}.
}
\usage{
print.summary.blockmodel(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{summary.blockmodel} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{summary.blockmodel}}}

\examples{

}
\keyword{ print }%-- one or more ...

\eof
\name{print.summary.cugtest}
\alias{print.summary.cugtest}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for summary.cugtest Objects }
\description{
  Prints an object of class \code{summary.cugtest}.
}
\usage{
print.summary.cugtest(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{summary.cugtest} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{summary.cugtest}} }

\examples{

}
\keyword{ print }%-- one or more ...

\eof
\name{print.summary.lnam}
\alias{print.summary.lnam}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Printing for summary.lnam Objects }
\description{
  Prints an object of class \code{summary.lnam}.
}
\usage{
print.summary.lnam(x, digits = max(3, getOption("digits") - 3), 
    signif.stars = getOption("show.signif.stars"), ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{x}{ an object of class \code{summary.lnam}. }
  \item{digits}{ number of digits to display. }
  \item{signif.stars}{ show significance stars? }
  \item{\dots}{ additional arguments. }
}
%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}
\value{
None
}
%\references{ ~put references to the literature/web site here ~ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ ~~further notes~~ }
%
% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{ \code{\link{summary.lnam}}, \code{\link{lnam}} }
%\examples{
%}
\keyword{ print }% at least one, from doc/KEYWORDS
%\keyword{ ~kwd2 }% __ONLY ONE__ keyword per line

\eof
\name{print.summary.netcancor}
\alias{print.summary.netcancor}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for summary.netcancor Objects }
\description{
  Prints an object of class \code{summary.netcancor}.
}
\usage{
print.summary.netcancor(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{summary.netcancor}}
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
}
%\references{ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{summary.netcancor}} }

\examples{

}
\keyword{ print }%-- one or more ...

\eof
\name{print.summary.netlm}
\alias{print.summary.netlm}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for summary.netlm Objects }
\description{
  Prints an object of class \code{summary.netlm}.
}
\usage{
print.summary.netlm(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{summary.netlm} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
}
%\references{ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{summary.netlm}} }

\examples{

}
\keyword{ print }%-- one or more ...

\eof
\name{print.summary.netlogit}
\alias{print.summary.netlogit}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for summary.netlogit Objects }
\description{
  Prints an object of class \code{summary.netlogit}.
}
\usage{
print.summary.netlogit(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{summary.netlogit}~ }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{summary.netlogit}} }

\examples{

}
\keyword{ print }%-- one or more ...

\eof
\name{print.summary.qaptest}
\alias{print.summary.qaptest}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Printing for summary.qaptest Objects  }
\description{
  Prints an object of class \code{summary.qaptest}.
}
\usage{
print.summary.qaptest(x, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{x}{ An object of class \code{summary.qaptest} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{summary.qaptest}} }

\examples{

}
\keyword{ print }%-- one or more ...

\eof
\name{pstar}
\alias{pstar}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Fit a p*/ERG Model Using a Logistic Approximation }
\description{
  Fits a p*/ERG model to the adjacency matrix in \code{dat} containing the effects listed in \code{effects}.  The result is returned as a \code{\link{glm}} object.
}
\usage{
pstar(dat, effects=c("choice", "mutuality", "density", "reciprocity",
    "stransitivity", "wtransitivity", "stranstri",  "wtranstri", 
    "outdegree", "indegree", "betweenness", "closeness", 
    "degcentralization", "betcentralization", "clocentralization",
    "connectedness", "hierarchy", "lubness", "efficiency"), 
    attr=NULL, memb=NULL, diag=FALSE, mode="digraph")
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A single adjacency matrix }
  \item{effects}{ A vector of strings indicating which effects should be fit }
  \item{attr}{ A matrix whose columns contain individual attributes (one row per vertex) whose differences should be used as supplemental predictors }
  \item{memb}{ A matrix whose columns contain group memberships whose categorical similarities (same group/not same group) should be used as supplemental predictors}
  \item{diag}{ A boolean indicating whether or not diagonal entries (loops) should be counted as meaningful data }
  \item{mode}{ \code{"digraph"} if \code{dat} is directed, else \code{"graph"} }
}
\details{
p* (also called the Exponential Random Graph (ERG) family) is an exponential family specification for network data.  Under p*, it is assumed that 
\deqn{p(G=g) \propto \exp(\beta_0 \gamma_0(g) + \beta_1 \gamma_1(g) + \dots)}{%
p(G=g) propto exp(beta_0 gamma_0(g) + beta_1 gamma_1(g) + \dots)}
for all g, where the betas represent real coefficients and the gammas represent functions of g.  Unfortunately, the unknown normalizing factor in the above expression makes evaluation difficult in the general case.  One solution to this problem is to operate instead on the edgewise log odds; in this case, the p* can be approximated by a logistic regression of each edge on the \emph{differences} in the gamma scores induced by the presence and absence of said edge in the graph (conditional on all other edges).  It is this approximation (known as autologistic regression, or maximum pseudo-likelihood estimation) which is employed here.  

Using the \code{effects} argument, a range of different potential parameters can be estimated.  The network measure associated with each is, in turn, the edge-perturbed difference in:
\enumerate{
\item \code{choice}: the number of edges in the graph (acts as a constant)

\item \code{mutuality}: the number of reciprocated dyads in the graph

\item \code{density}: the density of the graph

\item \code{reciprocity}: the edgewise reciprocity of the graph

\item \code{stransitivity}: the strong transitivity of the graph

\item \code{wtransitivity}: the weak transitivity of the graph

\item \code{stranstri}: the number of strongly transitive triads in the graph

\item \code{wtranstri}: the number of weakly transitive triads in the graph

\item \code{outdegree}: the outdegree of each actor (|V| parameters)

\item \code{indegree}: the indegree of each actor (|V| parameters)

\item \code{betweenness}: the betweenness of each actor (|V| parameters)

\item \code{closeness}: the closeness of each actor (|V| parameters)

\item \code{degcentralization}: the Freeman degree centralization of the graph

\item \code{betcentralization}: the betweenness centralization of the graph

\item \code{clocentralization}: the closeness centralization of the graph

\item \code{connectedness}: the Krackhardt connectedness of the graph

\item \code{hierarchy}: the Krackhardt hierarchy of the graph

\item \code{efficiency}: the Krackhardt efficiency of the graph

\item \code{lubness}: the Krackhardt LUBness of the graph
}
(Note that some of these do differ somewhat from the common p* parameter formulation, e.g. quantities such as density and reciprocity are computed as per the \code{\link{gden}} and \code{\link{grecip}} functions rather than via the unnormalized "choice" and "mutual" quantities one often finds in the p* literature.)  \emph{Please do not attempt to use all effects simultaneously!!!}  In addition to the above, the user may specify a matrix of individual attributes whose absolute dyadic differences are to be used as predictors, as well as a matrix of individual memberships whose dyadic categorical similarities (same/different) are used in the same manner.

Although the p* framework is quite versatile in its ability to accommodate a range of structural predictors, it should be noted that the \emph{substantial} collinearity of many of the standard p* predictors can lead to very unstable model fits.  Measurement and specification errors compound this problem; thus, it is somewhat risky to use p* in an exploratory capacity (i.e., when there is little prior knowledge to constrain choice of parameters).  While raw instability due to multicollinearity should decline with graph size, improper specification will still result in biased coefficient estimates so long as an omitted predictor correlates with an included predictor.  Caution is advised.
}
\value{
A \code{\link{glm}} object
}
\references{ Anderson, C.; Wasserman, S.; and Crouch, B. (1999).  ``A p* Primer:  Logit Models for Social Networks.  \emph{Social Networks,} 21,37-66.

Holland, P.W., and Leinhardt, S. (1981).  ``An Exponential Family of Probability Distributions for Directed Graphs.'' \emph{Journal of the American statistical Association}, 81, 51-67.

Wasserman, S., and Pattison, P. (1996).  ``Logit Models and Logistic Regressions for Social Networks:  I.  An introduction to Markov Graphs and p*.''  \emph{Psychometrika,} 60, 401-426.
 }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ In the long run, support will be included for p* models involving arbitrary functions (much like the system used with \code{\link{cugtest}} and \code{\link{qaptest}}).}

\section{WARNING }{Estimation of p* models by maximum pseudo-likelihood is now known to be a dangerous practice.  Use at your own risk.} 

\seealso{ \code{\link{eval.edgeperturbation}} }

\examples{
#Create a graph with expansiveness and popularity effects
in.str<-rnorm(20,0,3)
out.str<-rnorm(20,0,3)
tie.str<-outer(out.str,in.str,"+")
tie.p<-apply(tie.str,c(1,2),function(a){1/(1+exp(-a))})
g<-rgraph(20,tprob=tie.p)

#Fit a model with expansiveness only
p1<-pstar(g,effects="outdegree")
#Fit a model with expansiveness and popularity
p2<-pstar(g,effects=c("outdegree","indegree"))
#Fit a model with expansiveness, popularity, and mutuality
p3<-pstar(g,effects=c("outdegree","indegree","mutuality"))

#Compare the model AICs
extractAIC(p1)
extractAIC(p2)
extractAIC(p3)
}
\keyword{ models }
\keyword{ multivariate }
\keyword{ regression }%-- one or more ...
\keyword{ graphs }

\eof
\name{qaptest}
\alias{qaptest}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Perform Quadratic Assignment Procedure (QAP) Hypothesis Tests for Graph-Level Statistics }
\description{
   \code{qaptest} tests an arbitrary graph-level statistic (computed on \code{dat} by \code{FUN}) against a QAP null hypothesis, via Monte Carlo simulation of likelihood quantiles.  Note that fair amount of flexibility is possible regarding QAP tests on functions of such statistics (see an equivalent discussion with respect to CUG null hypothesis tests in Anderson et al. (1999)).  See below for more details. 
}
\usage{
qaptest(dat, FUN, reps=1000, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors.  Though this could in principle be an array containing only one graph, this is rarely if ever sensible in a QAP-test context.}
  \item{FUN}{ Function to generate the test statistic.  \code{FUN} must accept \code{dat} and the specified \code{g} arguments, and should return a real number. }
  \item{reps}{ Integer indicating the number of draws to use for quantile estimation.  Note that, as for all Monte Carlo procedures, convergence is slower for more extreme quantiles.  By default, \code{reps}=1000. }
  \item{\dots}{ Additional arguments to \code{FUN}. }
}
\details{
  The null hypothesis of the QAP test is that the observed graph-level statistic on graphs \eqn{G_1,G_2,\dots} was drawn from the distribution of said statistic evaluated (uniformly) on the set of all relabelings of \eqn{G_1,G_2,\dots}.  Pragmatically, this test is performed by repeatedly (randomly) relabeling the input graphs, recalculating the test statistic, and then evaluating the fraction of draws greater than or equal to (and less than or equal to) the observed value.  This accumulated fraction approximates the integral of the distribution of the test statistic over the set of unlabeled input graphs.

The \code{qaptest} procedure returns a \code{qaptest} object containing the estimated likelihood (distribution of the test statistic under the null hypothesis), the observed value of the test statistic on the input data, and the one-tailed p-values (estimated quantiles) associated with said observation.  As usual, the (upper tail) null hypothesis is rejected for significance level alpha if p>=observation is less than alpha (or p<=observation, for the lower tail); if the hypothesis is undirected, then one rejects if either p<=observation or p>=observation is less then alpha/2.  Standard caveats regarding the use of null hypothesis testing procedures are relevant here: in particular, bear in mind that a significant result does not necessarily imply that the likelihood ratio of the null model and the alternative hypothesis favors the latter.

In interpreting a QAP test, it is important to bear in mind the nature of the QAP null hypothesis.  The QAP test should \emph{not} be interpreted as evaluating underlying structural differences; indeed, QAP is more accurately understood as testing differences induced by a particular vertex labeling \emph{controlling for} underlying structure.  Where there is substantial automorphism in the underling structures, QAP will tend to given non-significant results.  (In fact, it is \emph{impossible} to obtain a one-tailed significance level in excess of \eqn{\max_{g \in \{G,H\}} \frac{|Aut(g)|}{|Perm(g)|}}{max_[g in \{G,H\}] |Aut(g)|/|Perm(g)|} when using a QAP test on a bivariate graph statistic \eqn{f(G,H)}, where Aut(g) and Perm(g) are the automorphism and permutation groups on g, respectively.  This follows from the fact that all members of Aut(g) will induce the same values of \eqn{f()}.)  By turns, significance under QAP does not necessarily imply that the observed structural relationship is unusual relative to what one would expect from typical structures with (for instance) the sizes and densities of the graphs in question.  In contexts in which one's research question implies a particular labeling of vertices (e.g., "within this group of individuals, do friends also tend to give advice to one another"), QAP can be a very useful way of ruling out spurious structural influences (e.g., some respondents tend to indiscriminately nominate many people (without regard to whom), resulting in a structural similarity which has nothing to do with the identities of those involved).  Where one's question does not imply a labeled relationship (e.g., is the \emph{shape} of this group's friendship network similar to that of its advice network), the QAP null hypothesis is inappropriate.
}
\value{
  An object of class \code{qaptest}, containing

\item{testval}{
The observed value of the test statistic.
}
\item{dist}{
A vector containing the Monte Carlo draws. 
}
\item{pgreq}{
The proportion of draws which were greater than or equal to the observed value.
}
\item{pleeq}{
The proportion of draws which were less than or equal to the observed value.
}
}
\references{ 
Anderson, B.S.; Butts, C.T.; and Carley, K.M. (1999). ``The Interaction of Size and Density with Graph-Level Indices.'' \emph{Social Networks}, 21(3), 239-267.

Hubert, L.J., and Arabie, P.  (1989).  ``Combinatorial Data Analysis: Confirmatory Comparisons Between Sets of Matrices.''  \emph{Applied Stochastic Models and Data Analysis}, 5, 273-325.

Krackhardt, D.  (1987).  ``QAP Partialling as a Test of Spuriousness.'' \emph{Social Networks}, 9 171-186.

Krackhardt, D.  (1988).  ``Predicting With Networks: Nonparametric Multiple Regression Analyses of Dyadic Data.''  \emph{Social Networks}, 10, 359-382. 
}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{cugtest}} }

\examples{

#Generate three graphs
g<-array(dim=c(3,10,10))
g[1,,]<-rgraph(10)
g[2,,]<-rgraph(10,tprob=g[1,,]*0.8)
g[3,,]<-1; g[3,1,2]<-0              #This is nearly a clique

#Perform qap tests of graph correlation
q.12<-qaptest(g,gcor,g1=1,g2=2)
q.13<-qaptest(g,gcor,g1=1,g2=3)

#Examine the results
summary(q.12)
plot(q.12)
summary(q.13)
plot(q.13)
}
\keyword{ htest }%-- one or more ...
\keyword{ math }

\eof
\name{reachability}
\alias{reachability}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find the Reachability Matrix of a Graph }
\description{
  \code{reachability} takes a (possibly directed) graph as input, producing the associated reachability matrix.
}
\usage{
reachability(dat, geodist.precomp=NULL)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph adjacency matrix (directed or otherwise) }
  \item{geodist.precomp}{ Optionally, a precomputed \code{\link{geodist}} object }
}
\details{
For a digraph \eqn{G=(V,E)} with vertices \eqn{i} and \eqn{j}, let \eqn{P_{ij}}{P_ij} represent a directed \eqn{ij} path.  Then the graph

\deqn{
R = \left(V\left(G\right),\left\{\left(i,j\right):i,j \in V\left(G\right), P_{ij} \in G\right\}\right)}{%
R = ( V(G), \{ (i,j): i,j in V(G), P_ij in G \} )}

is said to be the \emph{reachability graph} of \eqn{G}, and the adjacency matrix of \eqn{R} is said to be \eqn{G}'s \emph{reachability matrix}.  (Note that when \eqn{G} is undirected, we simply take each undirected edge to be bidirectional.)  Vertices which are adjacent in the reachability graph are connected by one or more directed paths in the original graph; thus, structural equivalence classes in the reachability graph are synonymous with strongly connected components in the original structure.

Bear in mind that -- as with all matters involving connectedness -- reachability is strongly related to size and density.  Since, for any given density, almost all structures of sufficiently large size are connected, reachability graphs associated with large structures will generally be complete.  Measures based on the reachability graph, then, will tend to become degenerate in the large \eqn{|V(G)|} limit (assuming constant positive density).
}
\value{
A reachability matrix
}
\references{ Wasserman, S., and Faust, K.  (1994).  ``Social Network Analysis: Methods and Applications.''  Cambridge: Cambridge University Press. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }

\seealso{ \code{\link{geodist}} }

\examples{
#Find the reachability matrix for a sparse random graph
g<-rgraph(10,tprob=0.15)
rg<-reachability(g)
g  #Compare the two structures
rg

#Compare to the output of geodist
all(rg==(geodist(g)$counts>0))
}
\keyword{ algebra }% __ONLY ONE__ keyword per line

\eof
\name{read.nos}
\alias{read.nos}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Read (N)eo-(O)rg(S)tat Input Files}
\description{
  Reads an input file in NOS format, returning the result as a graph stack.
}
\usage{
read.nos(file)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{file}{ The file to be imported }
}
\details{
  NOS format consists of three header lines, followed by a whitespace delimited stack of raw adjacency matrices; the format is not particularly elegant, but turns up in certain legacy applications (mostly at CMU).  \code{read.nos} provides a quick and dirty way of reading in these files, without the headache of messing with \code{\link{read.table}} settings.

The content of the NOS format is as follows:

<m>

<n> <o>

<kr1> <kr2> ... <krn> <kc1> <kc2> ... <kcn>

<a111> <a112> ... <a11o>

<a121> <a122> ... <a12o>

...

<a1n1> <a1n2> ... <a1no>

<a211> <a212> ... <a21o>

...

<a2n1> <a2n2> ... <a2no>

...

<amn1> <amn2> ... <amno>


where <abcd> is understood to be the value of the c->d edge in the bth graph of the file.  (As one might expect, m, n, and o are the numbers of graphs (matrices), rows, and columns for the data, respectively.)  The "k" line contains a list of row and column "colors", categorical variables associated with each row and column, respectively.  Although originally intended to communicate exchangability information, these can be used for other purposes (though there are easier ways to deal with attribute data these days).
}
\value{
  The imported graph stack
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ \code{read.nos} currently ignores the coloring information. }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{scan}}, \code{\link{read.table}} }

\examples{

}
\keyword{ file }%-- one or more ...

\eof
\name{rgraph}
\alias{rgraph}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Generate Bernoulli Random Graphs }
\description{
  \code{rgraph} generates random draws from a Bernoulli graph distribution, with various parameters for controlling the nature of the data so generated.
}
\usage{
rgraph(n, m=1, tprob=0.5, mode="digraph", diag=FALSE, replace=FALSE,
    tielist=NULL)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{n}{ The size of the vertex set (|V(G)|) for the random graphs }
  \item{m}{ The number of graphs to generate }
  \item{tprob}{ Information regarding tie (edge) probabilities; see below }
  \item{mode}{ ``digraph'' for directed data, ``graph'' for undirected data }
  \item{diag}{ Should the diagonal entries (loops) be set to zero? }
  \item{replace}{ Sample with or without replacement from a tie list (ignored if \code{tielist==NULL} }
  \item{tielist}{ A vector of edge values, from which the new graphs should be bootstrapped }
}
\details{
   \code{rgraph} is a reasonably versatile routine for generating random network data.  The graphs so generated are either Bernoulli graphs (graphs in which each edge is a Bernoulli trial, independent conditional on the Bernoulli parameters), or are bootstrapped from a user-provided edge distribution (very handy for CUG tests).  In the latter case, edge data should be provided using the \code{tielist} argument; the exact form taken by the data is irrelevant, so long as it can be coerced to a vector.  In the former case, Bernoulli graph probabilities are set by the \code{tprob} argument as follows:
\enumerate{
\item If \code{tprob} contains a single number, this number is used as the probability of all edges.

\item If \code{tprob} contains a vector, each entry is assumed to correspond to a separate graph (in order).  Thus, each entry is used as the probability of all edges within its corresponding graph.

\item If \code{tprob} contains a matrix, then each entry is assumed to correspond to a separate edge.  Thus, each entry is used as the probability of its associated edge in each graph which is generated.

\item Finally, if \code{tprob} contains a three-dimensional array, then each entry is assumed to correspond to a particular edge in a particular graph, and is used as the associated probability parameter.
}

Finally, note that \code{rgraph} will symmetrize all generated networks if \code{mode} is set to ``graph'' by copying down the upper triangle.  The lower half of \code{tprob}, where applicable, must still be specified, however.
}
\value{
  A graph stack
}
\references{ Wasserman, S., and Faust, K.  \emph{Social Network Analysis: Methods and Applications}.  Cambridge: Cambridge University Press. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{rmperm}} }

\examples{

#Generate three graphs with different densities
g<-rgraph(10,3,tprob=c(0.1,0.9,0.5))

#Generate from a matrix of Bernoulli parameters
g.p<-matrix(runif(25,0,1),nrow=5)
g<-rgraph(5,2,tprob=g.p)
}
\keyword{ distribution }%-- one or more ...
\keyword{ math }

\eof
\name{rmperm}
\alias{rmperm}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Randomly Permute the Rows and Columns of an Input Matrix }
\description{
Given an input matrix (or stack thereof), \code{rmperm} performs a (random) simultaneous row/column permutation of the input data.
}
\usage{
rmperm(m)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{m}{ A matrix, or stack thereof }
}
\details{
Random matrix permutations are the essence of the QAP test; see \code{\link{qaptest}} for details.
}
\value{
The permuted matrix (or matrices)
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{rperm}} }

\examples{
#Generate an input matrix
g<-rgraph(5)
g             #Examine it

#Examine a random permutation
rmperm(g)
}
\keyword{ array }
\keyword{ distribution }%-- one or more ...

\eof
\name{rperm}
\alias{rperm}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Draw a Random Permutation Vector with Exchangeability Constraints }
\description{
  Draws a random permutation on \code{1:length(exchange.list)} such that no two elements whose corresponding \code{exchange.list} values are different are interchanged.
}
\usage{
rperm(exchange.list)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{exchange.list}{ A vector such that the permutation vector may exchange the ith and jth positions iff \code{exchange.list[i]==exchange.list[j]} }
}
\details{
\code{rperm} draws random permutation vectors given the constraints of exchangeability described above.  Thus, \code{rperm(c(0,0,0,0))} returns a random permutation of four elements in which all exchanges are allowed, while \code{rperm(c(1,1,"a","a")} (or similar) returns a random permutation of four elements in which only the first/second and third/fourth elements may be exchanged.  This turns out to be quite useful for searching permutation spaces with exchangeability constraints (e.g., for structural distance estimation).
}
\value{
A random permutation vector satisfying the given constraints
}
%\references{ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{rmperm}} }

\examples{
rperm(c(0,0,0,0))  #All elements may be exchanged
rperm(c(0,0,0,1))  #Fix the fourth element
rperm(c(0,0,1,1))  #Allow \{1,2\} and \{3,4\} to be swapped
rperm(c("a",4,"x",2))  #Fix all elements (the identity permutation)
}
\keyword{ distribution }
\keyword{ array }%-- one or more ...

\eof
\name{sdmat}
\alias{sdmat}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Estimate the Structural Distance Matrix for a Graph Set}
\description{
  Estimates the structural distances among all elements of \code{dat} using the method specified in \code{method}.
}
\usage{
sdmat(dat, normalize=FALSE, diag=FALSE, mode="digraph", 
    output="matrix", method="mc", exchange.list=rep(0, 
    dim(dat)[2]), ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors.  This data need not be dichotomous, and missing values are allowed. }
  \item{normalize}{ Divide by the number of available dyads? }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{mode}{ String indicating the type of graph being evaluated.  \code{?digraph"} indicates that edges should be interpreted as directed; \code{"graph"} indicates that edges are undirected.  \code{mode} is set to \code{"digraph"} by default. }
  \item{output}{ \code{"matrix"} for matrix output, \code{"dist"} for a \code{\link[mva]{dist}} object (requires \code{mva}). }
  \item{method}{ Method to be used to search the space of accessible permutations; must be one of \code{?none?}, \code{?exhaustive?}, \code{?anneal?}, \code{?hillclimb?}, or \code{?mc?}. }
  \item{exchange.list}{ Information on which vertices are exchangeable (see below); this must be a single number, a vector of length n, or a nx2 matrix. }
  \item{\dots}{ Additional arguments to \code{\link{lab.optimize}}}
}
\details{
  The structural distance between two graphs G and H is defined as
\deqn{d_S\left(G,H \left| L_G,L_H\right.\right) = \min_{L_G,L_H} d\left(\ell\left(G\right),\ell\left(H\right)\right)}{%
d_S(G,H | L_G,L_H) = min_[L_G,L_H] d(l(G),l(H))}
where \eqn{L_G} is the set of accessible permutations/labelings of G, and \eqn{\ell(G)}{l(G)} is a permuation/relabeling of the vertices of G (\eqn{\ell(G) \in L_G}{l(G) in L_G}).  The set of accessible permutations on a given graph is determined by the \emph{theoretical exchangeability} of its vertices; in a  nutshell, two vertices are considered to be theoretically exchangeable for a given problem if all predictions under the conditioning theory are invariant to a relabeling of the vertices in question (see Butts and Carley (2001) for a more formal exposition).  Where no vertices are exchangeable, the structural distance becomes the its labeled counterpart (here, the Hamming distance).  Where \emph{all} vertices are exchangeable, the structural distance reflects the distance between unlabeled graphs; other cases correspond to distance under partial labeling.  

The accessible permutation set is determined by the \code{exchange.list} argument, which is dealt with in the following manner. First, \code{exchange.list} is expanded to fill an nx2 matrix.  If \code{exchange.list} is a single number, this is trivially accomplished by replication; if \code{exchange.list} is a vector of length n, the matrix is formed by cbinding two copies together.  If \code{exchange.list} is already an nx2 matrix, it is left as-is.  Once the nx2 exchangeabiliy matrix has been formed, it is interpreted as follows: columns refer to graphs 1 and 2, respectively; rows refer to their corresponding vertices in the original adjacency matrices; and vertices are taken to be theoretically exchangeable iff their corresponding exchangeability matrix values are identical.  To obtain an unlabeled distance (the default), then, one could simply let \code{exchange.list} equal any single number.  To obtain the Hamming distance, one would use the vector \code{1:n}.

Because the set of accessible permutations is, in general, very large (\eqn{o(n!)}), searching the set for the minimum distance is a non-trivial affair.  Currently supported methods for estimating the structural distance are hill climbing, simulated annealing, blind monte carlo search, or exhaustive search (it is also possible to turn off searching entirely).  Exhaustive search is not recommended for graphs larger than size 8 or so, and even this may take days; still, this is a valid alternative for small graphs.  Blind monte carlo search and hill climbing tend to be suboptimal for this problem and are not, in general recommended, but they are available if desired.  The preferred (and default) option for permutation search is simulated annealing, which seems to work well on this problem (though some tinkering with the annealing parameters may be needed in order to get optimal performance).  See the help for \code{\link{lab.optimize}} for more information regarding these options.

Structural distance matrices may be used in the same manner as any other distance matrices (e.g., with multidimensional scaling, cluster analysis, etc.)  Classical null hypothesis tests should not be employed with structural distances, and QAP tests are almost never appropriate (save in the uniquely labeled case).  See \code{\link{cugtest}} for a more reasonable alternative.
}
\value{
A matrix of distances (or an object of class \code{dist})
}
\references{ Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Interstructural Analysis.''  CASOS Working Paper, Carnegie Mellon University.}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ For most applications, \code{sdmat} is dominated by \code{\link{structdist}}; the former is retained largely for reasons of compatibility.}

\section{Warning }{The search process can be \emph{very slow}, particularly for large graphs.  In particular, the \emph{exhaustive} method is order factorial, and will take approximately forever for unlabeled graphs of size greater than about 7-9.}

\seealso{ \code{\link{hdist}}, \code{\link{structdist}} }

\examples{
#Generate two random graphs
g<-array(dim=c(3,5,5))
g[1,,]<-rgraph(5)
g[2,,]<-rgraph(5)

#Copy one of the graphs and permute it
g[3,,]<-rmperm(g[2,,])

#What are the structural distances between the labeled graphs?
sdmat(g,exchange.list=1:5)

#What are the structural distances between the underlying unlabeled 
#graphs?
sdmat(g,method="anneal", prob.init=0.9, prob.decay=0.85, 
    freeze.time=50, full.neighborhood=TRUE)
}
\keyword{ univar }
\keyword{ multivariate }%-- one or more ...
\keyword{ math }

\eof
\name{sedist}
\alias{sedist}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find a Matrix of Distances Between Positions Based on Structural Equivalence }
\description{
\code{sedist} uses the graphs indicated by \code{g} in \code{dat} to assess the extent to which each vertex is structurally equivalent; \code{joint.analysis} determines whether this analysis is simultaneous, and \code{method} determines the measure of approximate equivalence which is used.
}
\usage{
sedist(dat, g=c(1:dim(dat)[1]), method="hamming", 
    joint.analysis=FALSE, mode="digraph", diag=FALSE, code.diss=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph or graph stack }
  \item{g}{ A vector indicating which elements of \code{dat} should be examined }
  \item{method}{ One of \code{?correlation?}, \code{?euclidean?}, \code{?hamming?}, and \code{?gamma?}}
  \item{joint.analysis}{ Should equivalence be assessed across all networks jointly (\code{TRUE}), or individually within each (\code{FALSE}) }
  \item{mode}{ \code{?digraph?} for directed data, otherwise \code{?graph?} }
  \item{diag}{ Boolean indicating whether diagonal entries (loops) should be treated as meaningful data }
  \item{code.diss}{ Reverse-code the raw comparison values}
}
\details{
\code{sedist} provides a basic tool for assessing the (approximate) structural equivalence of actors.  (Two vertices i and j are said to be structurally equivalent if i->k iff j->k for all k.)  SE similarity/difference scores are computed by comparing vertex rows and columns using the measure indicated by \code{method}:
\enumerate{
\item correlation: the product-moment correlation

\item euclidean: the euclidean distance

\item hamming: the Hamming distance

\item gamma: the gamma correlation
}
Once these similarities/differences are calculated, the results can be used with a clustering routine (such as \code{\link{equiv.clust}}) or an MDS (such as \code{\link[mva]{cmdscale}}).
}
\value{
A matrix of similarity/difference scores
}
\references{Breiger, R.L.; Boorman, S.A.; and Arabie, P.  (1975).  ``An Algorithm for Clustering Relational Data with Applications to Social Network Analysis and Comparison with Multidimensional Scaling.''  \emph{Journal of Mathematical Psychology}, 12, 328-383.

Burt, R.S.  (1976).  ``Positions in Networks.''  \emph{Social Forces}, 55, 93-122.

Wasserman, S., and Faust, K.  \emph{Social Network Analysis: Methods and Applications.}  Cambridge: Cambridge University Press.   }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ Be careful to verify that you have computed what you meant to compute, with respect to similarities/differences. Also, note that (despite its popularity) the product-moment correlation can give rather strange results in some cases.
}

\seealso{ \code{\link{equiv.clust}}, \code{\link{blockmodel}} }

\examples{
#Create a random graph with _some_ edge structure
g.p<-sapply(runif(20,0,1),rep,20)  #Create a matrix of edge 
                                   #probabilities
g<-rgraph(20,tprob=g.p)            #Draw from a Bernoulli graph 
                                   #distribution

#Get SE distances
g.se<-sedist(g)

#Plot a metric MDS of vertex positions in two dimensions
library(mva)
plot(cmdscale(as.dist(g.se)))
}
\keyword{ classif }
\keyword{ cluster }%-- one or more ...
\keyword{ math }

\eof
\name{sna-internal}
\alias{sna-internal}
\alias{bbnam.jntlik}
\alias{bbnam.jntlik.slice} 
\alias{bbnam.probtie}     
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Internal sna Functions }
\description{
  Internal \code{sna} functions.
}
\usage{
bbnam.jntlik(dat, log=FALSE, ...)
bbnam.jntlik.slice(s, dat, a, em, ep, log=FALSE)
bbnam.probtie(dat, i, j, npriorij, em, ep)
}
%- maybe also `usage' for other objects documented here.
\details{
These are not to be called by the end user.
}
\keyword{internal}

\eof
\name{sna.operators}
\alias{sna.operators}
\alias{\%c\%}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Graphical Operators }
\description{
 These operators allow for algebraic manupulation of graph adjacency matrices.
}
\usage{
x \%c\% y
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{x}{ an (unvalued) adjacency matrix. }
  \item{y}{ another (unvalued) adjacency matrix. }
}
\details{
  Currently, only one operator is supported.  \code{x \%c\% y} returns the adjacency matrix of the composition of graphs with adjacency matrices \code{x} and \code{y} (respectively).  (Note that this may contain loops.)
}
\value{
   The resulting adjacency matrix.
}
\references{ Wasserman, S. and Faust, K.  (1994).  \emph{Social Network Analysis: Methods and Applications.}  Cambridge: University of Cambridge Press. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ ~~further notes~~ }

% ~Make other sections like Warning with \section{Warning }{....} ~

%\seealso{ ~~objects to See Also as \code{\link{~~fun~~}}, ~~~ }
\examples{
#Create an in-star
g<-matrix(0,6,6)
g[2:6,1]<-1
gplot(g)

#Compose g with its transpose
gcgt<-g\%c\%t(g)
gplot(gcgt,diag=TRUE)
gcgt
}
\keyword{ math }% at least one, from doc/KEYWORDS
\keyword{ graphs }% __ONLY ONE__ keyword per line

\eof
\name{sr2css}
\alias{sr2css}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Convert a Row-wise Self-Report Matrix to a CSS Matrix with Missing Observations }
\description{
  Given a matrix in which the ith row corresponds to i's reported relations, \code{sr2css} creates a graph stack in which each element represents a CSS slice with missing observations.
}
\usage{
sr2css(net)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{net}{ An adjacency matrix }
}
\details{
 A cognitive social structure (CSS) is an nxnxn array in which the ith matrix corresponds to the ith actor's perception of the entire network.  Here, we take a conventional self-report data structure and put it in CSS format for routines (such as \code{\link{bbnam}}) which require this.
}
\value{
An array (graph stack) containing the CSS
}
\references{ Krackhardt, D.  (1987).  \emph{Cognitive Social Structures}, 9, 109-134.  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ A row-wise self-report matrix doesn't contain a great deal of data, and the data in question is certainly not an ignorable sample of the individual's CSS for most purposes.  The provision of this routine should not be perceived as license to substitute SR for CSS data at will. }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

%\seealso{ }

\examples{
#Start with some random reports
g<-rgraph(10)

#Transform to CSS format
c<-sr2css(g)
}
\keyword{ manip }%-- one or more ...
\keyword{ math }

\eof
\name{stackcount}
\alias{stackcount}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ How Many Graphs are in a Graph Stack? }
\description{
Returns the number of graphs in the stack provided by \code{d}.
}
\usage{
stackcount(d)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{d}{ A graph or graph stack }
}
\details{
}
\value{
The number of graphs in \code{d}
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{nties}} }

\examples{
stackcount(rgraph(4,8))==8
}
\keyword{ utilities }%-- one or more ...
\keyword{ math }

\eof
\name{stresscent}
\alias{stresscent}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute the Stress Centrality Scores of Network Positions }
\description{
  \code{stresscent} takes a graph stack (\code{dat}) and returns the stress centralities of positions within one graph (indicated by \code{nodes} and \code{g}, respectively).  Depending on the specified mode, stress on directed or undirected geodesics will be returned; this function is compatible with \code{\link{centralization}}, and will return the theoretical maximum absolute deviation (from maximum) conditional on size (which is used by \code{\link{centralization}} to normalize the observed centralization score).
}
\usage{
stresscent(dat, g=1, nodes=c(1:dim(dat)[2]), gmode="digraph", 
    diag=FALSE, tmaxdev=FALSE, cmode="directed", 
    geodist.precomp=NULL, rescale=FALSE)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors. Alternately, this can be an n x n matrix (if only one graph is involved). }
  \item{g}{ Integer indicating the index of the graph for which centralities are to be calculated.  By default, \code{g==1}. }
  \item{nodes}{ List indicating which nodes are to be included in the calculation.  By default, all nodes are included. }
  \item{gmode}{ String indicating the type of graph being evaluated.  \code{"digraph"} indicates that edges should be interpreted as directed; \code{"graph"} indicates that edges are undirected.  \code{gmode} is set to \code{"digraph"} by default.}
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{tmaxdev}{ Boolean indicating whether or not the theoretical maximum absolute deviation from the maximum nodal centrality should be returned.  By default, \code{tmaxdev==FALSE}. }
  \item{cmode}{ String indicating the type of betweenness centrality being computed (directed or undirected geodesics). }
  \item{geodist.precomp}{A \code{\link{geodist}} object precomputed for the graph to be analyzed (optional)  }
  \item{rescale}{ If true, centrality scores are rescaled such that they sum to 1. }
}
\details{
 The stress of a vertex, v, is given by

\deqn{C_S(v) = \sum_{i,j : i \neq j,i \neq v,j \neq v} g_{ivj}}{%
C_S(v) = sum( g_ivj, i,j: i!=j,i!=v,j!=v)}

where \eqn{g_{ijk}}{g_ijk} is the number of geodesics from i to k through j.  Conceptually, high-stress vertices lie on a large number of shortest paths between other vertices; they can thus be thought of as ``bridges'' or ``boundary spanners.''  Compare this with \code{\link{betweenness}}, which weights shortest paths by the inverse of their redundancy.
}
\value{
A vector of centrality scores
}
%\references{ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ Judicious use of \code{geodist.precomp} can save a great deal of time when computing multiple path-based indices on the same network. }

% ~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{centralization}}}

\examples{
g<-rgraph(10)     #Draw a random graph with 10 members
stresscent(g)     #Compute stress scores
}
\keyword{univar}
\keyword{ math }

\eof
\name{structdist}
\alias{structdist}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Find the Structural Distances Between Two or More Graphs }
\description{
  \code{structdist} returns the structural distance between the labeled graphs \code{g1} and \code{g2} in stack \code{dat} based on Hamming distance for dichotomous data, or else the absolute (manhattan) distance.  If \code{normalize} is true, this distance is divided by its dichotomous theoretical maximum (conditional on |V(G)|).
}
\usage{
structdist(dat, g1=c(1:dim(dat)[1]), g2=c(1:dim(dat)[1]), 
    normalize=FALSE, diag=FALSE, mode="digraph", method="anneal", 
    reps=1000, prob.init=0.9, prob.decay=0.85, freeze.time=25, 
    full.neighborhood=TRUE, mut=0.05, pop=20, trials=5, 
    exchange.list=rep(0, dim(dat)[2]))
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ Data array to be analyzed.  By assumption, the first dimension of the array indexes the graph, with the next two indexing the actors.  This data need not be dichotomous, and missing values are allowed. }
  \item{g1}{ A vector indicating which graphs to compare (by default, all elements of \code{dat}  }
  \item{g2}{ A vector indicating against which the graphs of \code{g1} should be compared (by default, all graphs) }
  \item{normalize}{ Divide by the number of available dyads? }
  \item{diag}{ Boolean indicating whether or not the diagonal should be treated as valid data.  Set this true if and only if the data can contain loops.  \code{diag} is \code{FALSE} by default. }
  \item{mode}{ String indicating the type of graph being evaluated.  \code{"digraph"} indicates that edges should be interpreted as directed; \code{"graph"} indicates that edges are undirected.  \code{mode} is set to \code{"digraph"} by default.}
  \item{method}{ Method to be used to search the space of accessible permutations; must be one of \code{?none?}, \code{?exhaustive?}, \code{?anneal?}, \code{?hillclimb?}, or \code{?mc?}. }
  \item{reps}{ Number of iterations for monte carlo method.}
  \item{prob.init}{ Initial acceptance probability for the annealing routine. }
  \item{prob.decay}{ Cooling multiplier for the annealing routine. }
  \item{freeze.time}{ Freeze time for the annealing routine. }
  \item{full.neighborhood}{ Should the annealer evaluate the full neighborhood of pair exchanges at each iteration? }
  \item{mut}{ GA Mutation rate (currently ignored) }
  \item{pop}{ GA population (currently ignored) }
  \item{trials}{ Number of GA populations (currently ignored) }
  \item{exchange.list}{ Information on which vertices are exchangeable (see below); this must be a single number, a vector of length n, or a nx2 matrix. }
}
\details{
The structural distance between two graphs G and H is defined as
\deqn{d_S\left(G,H \left| L_G,L_H\right.\right) = \min_{L_G,L_H} d\left(\ell\left(G\right),\ell\left(H\right)\right)}{%
d_S(G,H | L_G,L_H) = min_[L_G,L_H] d(l(G),l(H))}
where \eqn{L_G} is the set of accessible permutations/labelings of G, and \eqn{\ell(G)}{l(G)} is a permuation/relabeling of the vertices of G (\eqn{\ell(G) \in L_G}{l(G) in L_G}).  The set of accessible permutations on a given graph is determined by the \emph{theoretical exchangeability} of its vertices; in a  nutshell, two vertices are considered to be theoretically exchangeable for a given problem if all predictions under the conditioning theory are invariant to a relabeling of the vertices in question (see Butts and Carley (2001) for a more formal exposition).  Where no vertices are exchangeable, the structural distance becomes the its labeled counterpart (here, the Hamming distance).  Where \emph{all} vertices are exchangeable, the structural distance reflects the distance between unlabeled graphs; other cases correspond to distance under partial labeling.  

The accessible permutation set is determined by the \code{exchange.list} argument, which is dealt with in the following manner. First, \code{exchange.list} is expanded to fill an nx2 matrix.  If \code{exchange.list} is a single number, this is trivially accomplished by replication; if \code{exchange.list} is a vector of length n, the matrix is formed by \code{cbind}ing two copies together.  If \code{exchange.list} is already an nx2 matrix, it is left as-is.  Once the nx2 exchangeabiliy matrix has been formed, it is interpreted as follows: columns refer to graphs 1 and 2, respectively; rows refer to their corresponding vertices in the original adjacency matrices; and vertices are taken to be theoretically exchangeable iff their corresponding exchangeability matrix values are identical.  To obtain an unlabeled distance (the default), then, one could simply let \code{exchange.list} equal any single number.  To obtain the Hamming distance, one would use the vector \code{1:n}.

Because the set of accessible permutations is, in general, very large (\eqn{o(n!)}), searching the set for the minimum distance is a non-trivial affair.  Currently supported methods for estimating the structural distance are hill climbing, simulated annealing, blind monte carlo search, or exhaustive search (it is also possible to turn off searching entirely).  Exhaustive search is not recommended for graphs larger than size 8 or so, and even this may take days; still, this is a valid alternative for small graphs.  Blind monte carlo search and hill climbing tend to be suboptimal for this problem and are not, in general recommended, but they are available if desired.  The preferred (and default) option for permutation search is simulated annealing, which seems to work well on this problem (though some tinkering with the annealing parameters may be needed in order to get optimal performance).  See the help for \code{\link{lab.optimize}} for more information regarding these options.

Structural distance matrices may be used in the same manner as any other distance matrices (e.g., with multidimensional scaling, cluster analysis, etc.)  Classical null hypothesis tests should not be employed with structural distances, and QAP tests are almost never appropriate (save in the uniquely labeled case).  See \code{\link{cugtest}} for a more reasonable alternative.
}
\value{
  A structural distance matrix
}
\references{ Butts, C.T., and Carley, K.M.  (2001).  ``Multivariate Methods for Interstructural Analysis.''  CASOS Working Paper, Carnegie Mellon University.}
\author{ Carter T. Butts \email{buttsc@uci.edu} }
\note{ Consult Butts and Carley (2001) for advice and examples on theoretical exchangeability. }

\section{Warning }{The search process can be \emph{very slow}, particularly for large graphs.  In particular, the \emph{exhaustive} method is order factorial, and will take approximately forever for unlabeled graphs of size greater than about 7-9.}

\seealso{ \code{\link{hdist}}, \code{\link{sdmat}} }

\examples{
#Generate two random graphs
g<-array(dim=c(3,5,5))
g[1,,]<-rgraph(5)
g[2,,]<-rgraph(5)

#Copy one of the graphs and permute it
g[3,,]<-rmperm(g[2,,])

#What are the structural distances between the labeled graphs?
structdist(g,exchange.list=1:5)

#What are the structural distances between the underlying unlabeled 
#graphs?
structdist(g,method="anneal", prob.init=0.9, prob.decay=0.85, 
    freeze.time=50, full.neighborhood=TRUE)
}
\keyword{ univar }
\keyword{ multivariate }%-- one or more ...
\keyword{ math }

\eof
\name{summary.bayes.factor}
\alias{summary.bayes.factor}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Detailed Summaries of Bayes Factor Objects }
\description{
Returns a \code{bayes.factor} summary object.
}
\usage{
summary.bayes.factor(object, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{object}{ An object of class \code{bayes.factor} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
An object of class \code{summary.bayes.factor}
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{bbnam.bf}} }

\examples{
}
\keyword{ math }%-- one or more ...

\eof
\name{summary.bbnam}
\alias{summary.bbnam}
\alias{summary.bbnam.fixed}
\alias{summary.bbnam.pooled}
\alias{summary.bbnam.actor}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Detailed Summaries of bbnam Objects }
\description{
  Returns a \code{bbnam} summary object
}
\usage{
summary.bbnam(object, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{object}{ An object of class \code{bbnam} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
An object of class \code{summary.bbnam}
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{bbnam}} }

\examples{

}
\keyword{ math }%-- one or more ...

\eof
\name{summary.blockmodel}
\alias{summary.blockmodel}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Detailed Summaries of blockmodel Objects }
\description{
  Returns a \code{blockmodel} summary object.
}
\usage{
summary.blockmodel(object, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{object}{ An object of class \code{blockmodel} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%
%}
\value{
An object of class \code{summary.blockmodel}
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ ~ }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{blockmodel}} }

\examples{

}
\keyword{ math }%-- one or more ...

\eof
\name{summary.cugtest}
\alias{summary.cugtest}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Detailed Summaries of cugtest Objects }
\description{
  Returns a \code{cugtest} summary object
}
\usage{
summary.cugtest(object, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{object}{ An object of class \code{cugtest} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%
%}
\value{
An object of class \code{summary.cugtest}
}
%\references{ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{cugtest}} }

\examples{

}
\keyword{ math }%-- one or more ...

\eof
\name{summary.lnam}
\alias{summary.lnam}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{ Detailed Summaries of lnam Objects }
\description{
  Returns a \code{lnam} summary object.
}
\usage{
summary.lnam(object, ...)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
  \item{object}{ an object of class \code{lnam}. }
  \item{\dots}{ additional arguments. }
}
%\details{
%  ~~ If necessary, more details than the __description__  above ~~
%}
\value{
An object of class \code{summary.lnam}.
}
%\references{ ~put references to the literature/web site here ~ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ ~~further notes~~ }
%
% ~Make other sections like Warning with \section{Warning }{....} ~

\seealso{ \code{\link{lnam}} }
%\examples{
%}
\keyword{ methods }% at least one, from doc/KEYWORDS
%\keyword{ ~kwd2 }% __ONLY ONE__ keyword per line

\eof
\name{summary.netcancor}
\alias{summary.netcancor}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Detailed Summaries of netcancor Objects }
\description{
  Returns a \code{netcancor} summary object
}
\usage{
summary.netcancor(object, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{object}{ An object of class \code{netcancor} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%  
%}
\value{
  An object of class \code{summary.netcancor}
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu}~ }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{netcancor}} }

\examples{

}
\keyword{ math }%-- one or more ...

\eof
\name{summary.netlm}
\alias{summary.netlm}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Detailed Summaries of netlm Objects }
\description{
  Returns a \code{netlm} summary object
}
\usage{
summary.netlm(object, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{object}{ An object of class \code{netlm} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%  
%}
\value{
  An object of class \code{summary.netlm}
}
%\references{ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{netlm}} }

\examples{

}
\keyword{ math }%-- one or more ...

\eof
\name{summary.netlogit}
\alias{summary.netlogit}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Detailed Summaries of netlogit Objects  }
\description{
  Returns a \code{netlogit} summary object~
}
\usage{
summary.netlogit(object, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{object}{  An object of class \code{netlogit}  }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%}
\value{
  An object of class \code{summary.netlogit}
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{\code{\link{netlogit}} }

\examples{

}
\keyword{ math }%-- one or more ...

\eof
\name{summary.qaptest}
\alias{summary.qaptest}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Detailed Summaries of qaptest Objects }
\description{
  Returns a \code{qaptest} summary object
}
\usage{
summary.qaptest(object, ...)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{object}{ An object of class \code{qaptest} }
  \item{\dots}{ Further arguments passed to or from other methods }
}
%\details{
%  
%}
\value{
  An object of class \code{summary.qaptest}
}
%\references{  }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{qaptest}} }

\examples{

}
\keyword{ math }%-- one or more ...

\eof
\name{symmetrize}
\alias{symmetrize}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Symmetrize an Adjacency Matrix }
\description{
 Symmetrizes the elements of \code{mats} according to the rule in \code{rule}.
}
\usage{
symmetrize(mats, rule="weak")
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{mats}{ A graph or graph stack}
  \item{rule}{ One of ``upper'', ``lower'', ``strong'' or ``weak'' }
}
\details{
 The rules used by \code{symmetrize} are as follows:
\enumerate{
\item upper: Copy the upper triangle over the lower triangle

\item lower: Copy the lower triangle over the upper triangle

\item strong: i<->j iff i->j and i<-j  (AND rule)

\item weak: i<->j iff i->j or i<-j  (OR rule)
}
}
\value{
The symmetrized graph stack
}
\references{ Wasserman, S., and Faust, K.  (1994).  \emph{Social Network Analysis: Methods and Applications}.  Cambridge: Cambridge University Press. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{  }

 %~Make other sections like WARNING with \section{WARNING }{....} ~

%\seealso{ }

\examples{
#Generate a graph
g<-rgraph(5)

#Weak symmetrization
symmetrize(g)

#Strong symmetrization
symmetrize(g,rule="strong")
}
\keyword{ manip }
\keyword{ array }%-- one or more ...
\keyword{ math }

\eof
\name{triad.census}
\alias{triad.census}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute the Davis and Leinhardt Triad Census }
\description{
  \code{triad.census} returns the Davis and Leinhardt triad census of the elements of \code{dat} indicated by \code{g}.
}
\usage{
triad.census(dat, g=1:stackcount(dat))
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph or graph stack }
  \item{g}{ The elements of \code{dat} to process }
}
\details{
The Davis and Leinhardt triad census consists of a classification of all triads into one of 16 different categories; the resulting distribution can be compared against various null models to test for the presence of configural biases (e.g., transitivity bias).  \code{triad.census} is a front end for the \code{\link{triad.classify}} routine, performing the classification for all triads within the selected graphs.  The results are placed in the order indicated by the column names; this is the same order as presented in the \code{\link{triad.classify}} documentation, to which the reader is referred for additional details.

Compare \code{\link{triad.census}} to \code{\link{dyad.census}}, the dyadic equivalent.
}
\value{
A matrix whose 16 columns contain the counts of triads by class for each graph
}
\references{ Davis, J.A. and Leinhardt, S.  (1972).  ``The Structure of Positive Interpersonal Relations in Small Groups.''  In J. Berger (Ed.), \emph{Sociological Theories in Progress, Volume 2}, 218-251.  Boston: Houghton Mifflin.

Wasserman, S., and Faust, K.  (1994).  ``Social Network Analysis: Methods and Applications.''  Cambridge: Cambridge University Press. }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{ ~~further notes~~ }

\seealso{ \code{\link{triad.classify}}, \code{\link{dyad.census}}, \code{\link{gtrans}} }

\section{Warning }{Valued data may cause strange behavior with this routine.  Dichotomize the data first.}

\examples{
#Generate a triad census of random data with varying densities
triad.census(rgraph(15,5,tprob=c(0.1,0.25,0.5,0.75,0.9)))
}
\keyword{ math }% __ONLY ONE__ keyword per line

\eof
\name{triad.classify}
\alias{triad.classify}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Compute the Davis and Leinhardt Classification of a Given Triad }
\description{
  \code{triad.classify} returns the Davis and Leinhardt classification of the triad indicated by \code{tri} in the \code{g}th graph of stack \code{dat}.
}
\usage{
triad.classify(dat, g=1, tri=c(1, 2, 3))
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph or graph stack }
  \item{g}{ The index of the graph to be analyzed }
  \item{tri}{ A triple containing the indices of the triad to be classified}
}
\details{
Every unoriented directed triad may occupy one of 16 distinct states.  These states were used by Davis and Leinhardt as a basis for classifying triads within a larger structure; the distribution of triads within a graph (see \code{\link{triad.census}}), for instance, is linked to a range of substantive hypotheses (e.g., concerning structural balance).  The Davis and Leinhardt classification scheme describes each triad by a string of four elements: the number of mutual (complete) dyads within the triad; the number of asymmetric dyads within the triad; the number of null (empty) dyads within the triad; and a configuration code for the triads which are not uniquely distinguished by the first three distinctions.  The complete list of classes is as follows.

\describe{
\item{\code{003}}{ \eqn{a \not\leftrightarrow b \not\leftrightarrow c, a \not\leftrightarrow c}{a<-!->b<-!->c, a<-!->c}}
\item{\code{012}}{ \eqn{a \rightarrow b \not\leftrightarrow c, a \not\leftrightarrow c}{a-->b<-!->c, a<-!->c}}
\item{\code{102}}{ \eqn{a \leftrightarrow b \not\leftrightarrow c, a \not\leftrightarrow c}{a<-->b<-!->c, a<-!->c}}
\item{\code{021D}}{ \eqn{a \leftarrow b \rightarrow c, a \not\leftrightarrow c}{a<--b-->c, a<-!->c}}
\item{\code{021U}}{ \eqn{a \rightarrow b \leftarrow c, a \not\leftrightarrow c}{a-->b<--c, a<-!->c}}
\item{\code{021C}}{ \eqn{a \rightarrow b \rightarrow c, a \not\leftrightarrow c}{a-->b-->c, a<-!->c}}
\item{\code{111D}}{ \eqn{a \not\leftrightarrow b \rightarrow c, a \leftrightarrow c}{a<-!->b-->c, a<-->c}}
\item{\code{111U}}{ \eqn{a \not\leftrightarrow b \leftarrow c, a \leftrightarrow c}{a<-!->b-->c, a<-->c}}
\item{\code{030T}}{ \eqn{a \rightarrow b \leftarrow c, a \rightarrow c}{a-->b<--c, a-->c}}
\item{\code{030C}}{ \eqn{a \leftarrow b \leftarrow c, a \rightarrow c}{a<--b<--c, a-->c}}
\item{\code{201}}{ \eqn{a \leftrightarrow b \not\leftrightarrow c, a \leftrightarrow c}{a<-->b<-!->c, a<-->c}}
\item{\code{120D}}{ \eqn{a \leftarrow b \rightarrow c, a \leftrightarrow c}{a<--b-->c, a<-->c}}
\item{\code{120U}}{ \eqn{a \rightarrow b \leftarrow c, a \leftrightarrow c}{a-->b<--c, a<-->c}}
\item{\code{120C}}{ \eqn{a \rightarrow b \rightarrow c, a \leftrightarrow c}{a-->b-->c, a<-->c}}
\item{\code{210}}{ \eqn{a \rightarrow b \leftrightarrow c, a \leftrightarrow c}{a-->b<-->c, a<-->c}}
\item{\code{300}}{ \eqn{a \leftrightarrow b \leftrightarrow c, a \leftrightarrow c}{a<-->b<-->c, a<-->c}}
}

These codes are returned by \code{triad.classify} as strings.
}
\value{
A string containing the triad classification, or \code{NA} if one or more edges were missing
}
\references{ Davis, J.A. and Leinhardt, S.  (1972).  ``The Structure of Positive Interpersonal Relations in Small Groups.''  In J. Berger (Ed.), \emph{Sociological Theories in Progress, Volume 2}, 218-251.  Boston: Houghton Mifflin.

Wasserman, S., and Faust, K.  (1994).  ``Social Network Analysis: Methods and Applications.''  Cambridge: Cambridge University Press.}
\author{ Carter T. Butts  \email{buttsc@uci.edu} }
%\note{ ~~further notes~~ }


\seealso{ \code{\link{triad.census}}, \code{\link{gtrans}} }

\section{Warning }{Valued data and/or loops may cause strange behavior with this routine.  Dichotomize/remove loops first.}

\examples{
#Generate a random graph
g<-rgraph(10)

#Classify the triads (1,2,3) and (2,3,4)
triad.classify(g,tri=c(1,2,3))
triad.classify(g,tri=c(1,2,3))

#Plot the triads in question
gplot(g[1:3,1:3])
gplot(g[2:4,2:4])
}
\keyword{ math }% __ONLY ONE__ keyword per line

\eof
\name{upper.tri.remove}
\alias{upper.tri.remove}
%- Also NEED an `\alias' for EACH other topic documented here.
\title{ Remove the Upper Triangles of Adjacency Matrices in a Graph Stack }
\description{
  Returns the input graph stack, with the upper triangle entries removed/replaced as indicated.
}
\usage{
upper.tri.remove(dat, remove.val=NA)
}
%- maybe also `usage' for other objects documented here.
\arguments{
  \item{dat}{ A graph stack }
  \item{remove.val}{ The value with which to replace the existing upper triangles }
}
\details{
  \code{upper.tri.remove} is simply a convenient way to apply \code{g[upper.tri(g)]<-remove.val} to an entire stack of adjacency matrices at once.
}
\value{
  The updated graph stack
}
%\references{ }
\author{ Carter T. Butts \email{buttsc@uci.edu} }
%\note{}

 %~Make other sections like WARNING with \section{WARNING }{....} ~

\seealso{ \code{\link{upper.tri}}, \code{\link{lower.tri.remove}}, \code{\link{diag.remove}} }

\examples{
#Generate a random graph stack
g<-rgraph(3,5)
#Remove the upper triangles
g<-upper.tri.remove(g)
}
\keyword{ manip }
\keyword{ array }%-- one or more ...
\keyword{ math }

\eof
