### R code from vignette source 'RcppEigen-Intro.Rnw'

###################################################
### code chunk number 1: version
###################################################
prettyVersion <- packageDescription("RcppEigen")$Version
prettyDate <- format(Sys.Date(), "%B %e, %Y")


###################################################
### code chunk number 2: preliminaries
###################################################
link <- function( f, package, text = f, root = "http://finzi.psych.upenn.edu/R/library/" ){
	h <- if( missing(package) ) {
		as.character( help( f ) )
	} else {
		as.character( help( f, package = paste( package, sep = "" ) ) )
	}
	if( ! length(h) ){
		sprintf( "\\\\textbf{%s}", f )
	} else {
		rx <- "^.*/([^/]*?)/help/(.*?)$"
		package <- sub( rx, "\\1", h, perl = TRUE )
		page <- sub( rx, "\\2", h, perl = TRUE )
		sprintf( "\\\\href{%s%s/html/%s.html}{\\\\texttt{%s}}", root, package, page, text )
	}
}
linkS4class <- function( cl, package, text = cl, root = "http://finzi.psych.upenn.edu/R/library/" ){
	link( sprintf("%s-class", cl), package, text, root )
}
require( inline )
require( RcppEigen )


###################################################
### code chunk number 3: Adef
###################################################
(A <- matrix(1:6, ncol=2))
str(A)


###################################################
### code chunk number 4: transCpp
###################################################
transCpp <-'
using Eigen::Map;
using Eigen::MatrixXi;
                 // Map the integer matrix AA from R
const Map<MatrixXi>  A(as<Map<MatrixXi> >(AA));
                 // evaluate and return the transpose of A
const MatrixXi      At(A.transpose());
return wrap(At);
'


###################################################
### code chunk number 5: transCppLst
###################################################
cat(transCpp, "\n")


###################################################
### code chunk number 6: ftrans
###################################################
ftrans <- cxxfunction(signature(AA="matrix"), transCpp, plugin="RcppEigen")
(At <- ftrans(A))
stopifnot(all.equal(At, t(A)))


###################################################
### code chunk number 7: prodCpp
###################################################
prodCpp <- '
using Eigen::Map;
using Eigen::MatrixXi;
const Map<MatrixXi>    B(as<Map<MatrixXi> >(BB));
const Map<MatrixXi>    C(as<Map<MatrixXi> >(CC));
return List::create(_["B %*% C"]         = B * C,
                    _["crossprod(B, C)"] = B.adjoint() * C);
'


###################################################
### code chunk number 8: prodCppLst
###################################################
cat(prodCpp, "\n")


###################################################
### code chunk number 9: prod
###################################################
fprod <- cxxfunction(signature(BB = "matrix", CC = "matrix"), prodCpp, "RcppEigen")
B <- matrix(1:4, ncol=2); C <- matrix(6:1, nrow=2)
str(fp <- fprod(B, C))
stopifnot(all.equal(fp[[1]], B %*% C), all.equal(fp[[2]], crossprod(B, C)))


###################################################
### code chunk number 10: RcppEigen-Intro.Rnw:297-298 (eval = FALSE)
###################################################
## t(X) %*% X


###################################################
### code chunk number 11: crossprod
###################################################
crossprodCpp <- '
using Eigen::Map;
using Eigen::MatrixXi;
using Eigen::Lower;

const Map<MatrixXi> A(as<Map<MatrixXi> >(AA));
const int           m(A.rows()), n(A.cols());                        
MatrixXi          AtA(MatrixXi(n, n).setZero().
                      selfadjointView<Lower>().rankUpdate(A.adjoint()));
MatrixXi          AAt(MatrixXi(m, m).setZero().
                      selfadjointView<Lower>().rankUpdate(A));

return List::create(_["crossprod(A)"]  = AtA,
                    _["tcrossprod(A)"] = AAt);
'


###################################################
### code chunk number 12: crossprodCppLst
###################################################
cat(crossprodCpp, "\n")


###################################################
### code chunk number 13: RcppEigen-Intro.Rnw:346-349
###################################################
fcprd <- cxxfunction(signature(AA = "matrix"), crossprodCpp, "RcppEigen")
str(crp <- fcprd(A))
stopifnot(all.equal(crp[[1]], crossprod(A)), all.equal(crp[[2]], tcrossprod(A)))


###################################################
### code chunk number 14: storage
###################################################
storage.mode(A) <- "double"


###################################################
### code chunk number 15: cholCpp
###################################################
cholCpp <- '
using Eigen::Map;
using Eigen::MatrixXd;
using Eigen::LLT;
using Eigen::Lower;

const Map<MatrixXd>   A(as<Map<MatrixXd> >(AA));
const int             n(A.cols());
const LLT<MatrixXd> llt(MatrixXd(n, n).setZero().
                        selfadjointView<Lower>().rankUpdate(A.adjoint()));

return List::create(_["L"] = MatrixXd(llt.matrixL()),
                    _["R"] = MatrixXd(llt.matrixU()));
'


###################################################
### code chunk number 16: cholCppLst
###################################################
cat(cholCpp, "\n")


###################################################
### code chunk number 17: fchol
###################################################
fchol <- cxxfunction(signature(AA = "matrix"), cholCpp, "RcppEigen")
(ll <- fchol(A))
stopifnot(all.equal(ll[[2]], chol(crossprod(A))))


###################################################
### code chunk number 18: RcppEigen-Intro.Rnw:440-458
###################################################
cholDetCpp <- '
using Eigen::Lower;
using Eigen::Map;
using Eigen::MatrixXd;
using Eigen::VectorXd;

const Map<MatrixXd>   A(as<Map<MatrixXd> >(AA));
const int             n(A.cols());
const MatrixXd      AtA(MatrixXd(n, n).setZero().
                        selfadjointView<Lower>().rankUpdate(A.adjoint()));
const MatrixXd     Lmat(AtA.llt().matrixL());
const double       detL(Lmat.diagonal().prod());
const VectorXd     Dvec(AtA.ldlt().vectorD());

return List::create(_["d1"] = detL * detL,
                    _["d2"] = Dvec.prod(),
                    _["ld"] = Dvec.array().log().sum());
'


###################################################
### code chunk number 19: cholDetCppLst
###################################################
cat(cholDetCpp, "\n")


###################################################
### code chunk number 20: fdet
###################################################
fdet <- cxxfunction(signature(AA = "matrix"), cholDetCpp, "RcppEigen")
unlist(ll <- fdet(A))


###################################################
### code chunk number 21: lltLSCpp
###################################################
lltLSCpp <- '
using Eigen::LLT;
using Eigen::Lower;
using Eigen::Map;
using Eigen::MatrixXd;
using Eigen::VectorXd;

const Map<MatrixXd>   X(as<Map<MatrixXd> >(XX));
const Map<VectorXd>   y(as<Map<VectorXd> >(yy));
const int             n(X.rows()), p(X.cols());
const LLT<MatrixXd> llt(MatrixXd(p, p).setZero().
                        selfadjointView<Lower>().rankUpdate(X.adjoint()));
const VectorXd  betahat(llt.solve(X.adjoint() * y));
const VectorXd   fitted(X * betahat);
const VectorXd    resid(y - fitted);
const int            df(n - p);
const double          s(resid.norm() / std::sqrt(double(df)));
const VectorXd       se(s * llt.matrixL().solve(MatrixXd::Identity(p, p)).
                        colwise().norm());
return     List::create(_["coefficients"]   = betahat,
                        _["fitted.values"]  = fitted,
                        _["residuals"]      = resid,
                        _["s"]              = s,
                        _["df.residual"]    = df,
                        _["rank"]           = p,
                        _["Std. Error"]     = se);
'


###################################################
### code chunk number 22: lltLSCppLst
###################################################
cat(lltLSCpp, "\n")


###################################################
### code chunk number 23: lltLS
###################################################
lltLS <- cxxfunction(signature(XX = "matrix", yy = "numeric"), lltLSCpp, "RcppEigen")
data(trees, package="datasets")
str(lltFit <- with(trees, lltLS(cbind(1, log(Girth)), log(Volume))))
str(lmFit <- with(trees, lm.fit(cbind(1, log(Girth)), log(Volume))))
for (nm in c("coefficients", "residuals", "fitted.values", "rank"))
    stopifnot(all.equal(lltFit[[nm]], unname(lmFit[[nm]])))
stopifnot(all.equal(lltFit[["Std. Error"]],
                    unname(coef(summary(lm(log(Volume) ~ log(Girth), trees)))[,2])))


###################################################
### code chunk number 24: missingcell
###################################################
dd <- data.frame(f1 = gl(4, 6, labels = LETTERS[1:4]),
                 f2 = gl(3, 2, labels = letters[1:3]))[-(7:8), ]
xtabs(~ f2 + f1, dd)                    # one missing cell
mm <- model.matrix(~ f1 * f2, dd)
kappa(mm)         # large condition number, indicating rank deficiency
rcond(mm)         # alternative evaluation, the reciprocal condition number
(c(rank=qr(mm)$rank, p=ncol(mm))) # rank as computed in R's qr function
set.seed(1)
dd$y <- mm %*% seq_len(ncol(mm)) + rnorm(nrow(mm), sd = 0.1)
                         # lm detects the rank deficiency
fm1 <- lm(y ~ f1 * f2, dd)
writeLines(capture.output(print(summary(fm1), signif.stars=FALSE))[9:22])


###################################################
### code chunk number 25: rankdeficientPQR
###################################################
print(summary(fmPQR <- fastLm(y ~ f1 * f2, dd)), signif.stars=FALSE)
all.equal(coef(fm1), coef(fmPQR))
all.equal(unname(fitted(fm1)), fitted(fmPQR))
all.equal(unname(residuals(fm1)), residuals(fmPQR))


###################################################
### code chunk number 26: rankdeficientSVD
###################################################
print(summary(fmSVD <- fastLm(y ~ f1 * f2, dd, method=4L)), signif.stars=FALSE)
all.equal(coef(fm1), coef(fmSVD))
all.equal(unname(fitted(fm1)), fitted(fmSVD))
all.equal(unname(residuals(fm1)), residuals(fmSVD))


###################################################
### code chunk number 27: rankdeficientVLV
###################################################
print(summary(fmVLV <- fastLm(y ~ f1 * f2, dd, method=5L)), signif.stars=FALSE)
all.equal(coef(fmSVD), coef(fmVLV))
all.equal(unname(fitted(fm1)), fitted(fmSVD))
all.equal(unname(residuals(fm1)), residuals(fmSVD))


###################################################
### code chunk number 28: benchmark (eval = FALSE)
###################################################
## source(system.file("examples", "lmBenchmark.R", package="RcppEigen"))


###################################################
### code chunk number 29: badtransCpp
###################################################
badtransCpp <- '
using Eigen::Map;
using Eigen::MatrixXi;
const Map<MatrixXi>  A(as<Map<MatrixXi> >(AA));
return wrap(A.transpose());
'


###################################################
### code chunk number 30: badtransCppLst
###################################################
cat(badtransCpp, "\n")


###################################################
### code chunk number 31: ftrans2
###################################################
Ai <- matrix(1:6, ncol=2L)
ftrans2 <- cxxfunction(signature(AA = "matrix"), badtransCpp, "RcppEigen")
(At <- ftrans2(Ai))
all.equal(At, t(Ai))


###################################################
### code chunk number 32: RcppEigen-Intro.Rnw:990-1002
###################################################
sparseProdCpp <- '
using Eigen::Map;
using Eigen::MappedSparseMatrix;
using Eigen::SparseMatrix;
using Eigen::VectorXd;

const MappedSparseMatrix<double>  A(as<MappedSparseMatrix<double> >(AA));
const Map<VectorXd>               y(as<Map<VectorXd> >(yy));
const SparseMatrix<double>       At(A.adjoint());
return List::create(_["At"]  = At,
                    _["Aty"] = At * y);
'


###################################################
### code chunk number 33: sparseProdCppLst
###################################################
cat(sparseProdCpp, "\n")


###################################################
### code chunk number 34: RcppEigen-Intro.Rnw:1010-1016
###################################################
sparse1 <- cxxfunction(signature(AA = "dgCMatrix", yy = "numeric"),
                       sparseProdCpp, "RcppEigen")
data(KNex, package="Matrix")
rr <- sparse1(KNex$mm, KNex$y)
stopifnot(all.equal(rr$At, t(KNex$mm)),
          all.equal(rr$Aty, as.vector(crossprod(KNex$mm, KNex$y))))


###################################################
### code chunk number 35: RcppEigen-Intro.Rnw:1032-1041 (eval = FALSE)
###################################################
## code <- '
## using Eigen::Map;
## using Eigen::MatrixXcd;
##   // Map the complex matrix A_ from R
## const Map<MatrixXcd>    A(as<Map<MatrixXcd> >(A_));
## return List::create(_["transpose"] = A.transpose(),
##                     _["adjoint"]   = A.adjoint());
## '
## writeLines( code, "code.cpp" )


###################################################
### code chunk number 36: RcppEigen-Intro.Rnw:1043-1044 (eval = FALSE)
###################################################
## ex_highlight( "code.cpp" )


###################################################
### code chunk number 37: RcppEigen-Intro.Rnw:1047-1053 (eval = FALSE)
###################################################
## fadj <-
##     cxxfunction(signature( A_ = "matrix"),
##                 paste(readLines( "code.cpp" ), collapse = "\n"),
##                 plugin = "RcppEigen")
## A <- matrix(1:6 + 1i*(6:1), nc=2)
## fadj(A)


