### Name: gcmrec
### Title: General Class of Models for recurrent event data
### Aliases: gcmrec
### Keywords: survival

### ** Examples



###################################
## Models using different data formats
###################################

#
#    Data input as a data frame
#

#   We use the well-known bladder cancer data set from survival package

library(survival)
data(bladder2)

# we compute the interocurrence time
bladder2$time<-bladder2$stop-bladder2$start

# If we execute:
#     gcmrec(Survr(id,time,event)~rx+size+number,data=bladder2,s=2060)

# We will obtain the following error message:
#   Error in Survr(id, time, event) : Data doesn't match...

# This means that we have some patients without right-censoring time. So,
# we understand that the last event coincides with the end of study. 
# Consequently,we need to add a line with time 0 and status value equal 
# to 0, too. To do so, we can use the function "addCenTime" as follows:

bladderOK<-addCenTime(bladder2) 

# Now, we can fit the model using this new data set:

gcmrec(Survr(id,time,event)~rx+size+number,data=bladderOK,s=2060)


#
#    Data as a list. See either GeneratedData or hydraulic data 
#                    sets as an example.
#

#
# We can fit the model by transforming our data in a data frame 
# using "List.to.Dataframe" function:
#

data(hydraulic)
hydraulicOK<-List.to.Dataframe(hydraulic)
gcmrec(Survr(id,time,event)~covar.1+covar.2,data=hydraulicOK,s=4753)


#
# Our model allows us to incorporate effective age information
#
# To illustrate this example, we will use a simulated data set generated 
# under the minimal repair model with probability of perfect repair equal to 0.6
#
# As we have the data in a list, first we need to obtain a data frame containing
# the time, event, and covariates information:
#

data(GeneratedData)
temp<-List.to.Dataframe(GeneratedData)

# then, we can fit the model incorporating the information about the effective 
#        age in the effageData argument:

gcmrec(Survr(id,time,event)~covar.1+covar.2, data=temp, 
                    effageData=GeneratedData, s=100)



#####################################################################
##  How to fit minimal or perfect repair models, with and without frailties
#####################################################################

# Model with frailties 

mod.Fra<-gcmrec(Survr(id,time,event)~rx+size+number,data=bladderOK,s=2060,Frailty=TRUE)
print(mod.Fra)

# effective age function: perfect repair and minimal repair models 
# (models without frailties)

data(readmission)

# perfect
mod.per<-gcmrec(Survr(id,time,event)~as.factor(dukes),data=readmission,
    s=3000,typeEffage="per")
print(mod.per)

# minimal
mod.min<-gcmrec(Survr(id,time,event)~as.factor(dukes),data=readmission,
   s=3000,typeEffage="min")
print(mod.min)

#####################################################################
##  How to fit models with \rho function equal to identity 
#####################################################################

data(lymphoma)

gcmrec(Survr(id, time, event) ~ as.factor(distrib), 
    data = lymphoma, s = 1000, Frailty = TRUE, rhoFunc = "Ident")

#####################################################################
##  How to fit cancer model
#####################################################################

mod.can<-gcmrec(Survr(id,time,event)~as.factor(distrib), data=lymphoma,
     s=1000, Frailty=TRUE,  cancer=lymphoma$effage)

#  standard errors can be obtained by adding se="Jacknife".
# This procedure can be very time consuming

 


