### Name: BayesianMCMC
### Title: Bayesian MCMC frequency analysis
### Aliases: BayesianMCMC print.BayesianMCMC plot.BayesianMCMC
### Keywords: htest

### ** Examples

set.seed(2988)
serie <- rand.GEV(120, xi=40, alfa=20, k=-0.4)
serie100 <- serie[1:100]
serie100[serie100 < 250] <- NA
serie20 <- serie[101:120]
serie <- c(serie100, serie20)

plot(serie, type="h", ylim=c(0, 600), xlab="", 
     ylab="Annual flood peaks [m3/s]", lwd=3)
abline(h=0)
points(serie100, col=2)

# Using only sistematic data
only_sist <- BayesianMCMC (xcont=serie20, xhist=NA, infhist=NA, suphist=NA, 
                           nbans=NA, seuil=NA,
                           nbpas=5000, nbchaines=3, 
                           confint=c(0.05, 0.95), dist="GEV")
plot(only_sist, which=c(1:3), ask=TRUE, ylim=c(1,600))


# Adding the information that the threshold 250 m3/s was exceeded 
#   3 times in the past 100 years
with_hist_thresh <- BayesianMCMC (xcont=serie20, xhist=NA, infhist=rep(250,3), 
                                  suphist=NA, nbans=100, seuil=250,
                                  nbpas=5000, nbchaines=3, 
                                  confint=c(0.05, 0.95), dist="GEV")
plot(with_hist_thresh, which=c(1:3), ask=TRUE, ylim=c(1,600))


# Assuming that the 3 historical events are known with high uncertainty
with_hist_limits <- BayesianMCMC (xcont=serie20, xhist=NA, 
                                  infhist=c(320,320,250), 
                                  suphist=c(360,400,270), 
                                  nbans=100, seuil=250,
                                  nbpas=5000, nbchaines=3, 
                                  confint=c(0.05, 0.95), dist="GEV")
plot(with_hist_limits, which=c(1:3), ask=TRUE, ylim=c(1,600))


# Assuming that the 3 historical events are perfectly known
with_hist_known <- BayesianMCMC (xcont=serie20, xhist=serie100[!is.na(serie100)], 
                                 infhist=NA, suphist=NA, 
                                 nbans=100, seuil=250,
                                 nbpas=5000, nbchaines=3, 
                                 confint=c(0.05, 0.95), dist="GEV")
plot(with_hist_known, which=c(1:3), ask=TRUE, ylim=c(1,600))



# Using one reasonable a-priori distribution
fNORM3 <- function (x) {
 # x = vector of values
 # mu = vector of means
 mu = c(44, 26, -0.40)
 # CM = covariance matrix
 CM = matrix(c(13, 7.8, -0.055,
               7.8, 15, -0.42,
               -0.055, -0.42, 0.056), nrow=3, ncol=3)
 CMm1 <- solve(CM)
 term2 <- exp(-((x - mu) %*% CMm1 %*% (x - mu))/2)
 term1 <- 1/(2*pi)^(3/2)/sqrt(det(CM))
 term1*term2
}

with_hist_known2 <- BayesianMCMC (xcont=serie20, xhist=serie100[!is.na(serie100)], 
                                  infhist=NA, suphist=NA,
                                  nbans=100, seuil=250,
                                  nbpas=5000, nbchaines=3, apriori=fNORM3,
                                  confint=c(0.05, 0.95), dist="GEV")
plot(with_hist_known2, 5)
plot(with_hist_known2, 4)
plot(with_hist_known, 4)
plot(with_hist_known)
plot(with_hist_known2)

## Not run: 
##D # Using one non-reasonable a-priori distribution
##D fNORM3 <- function (x) {
##D  # x = vector of values
##D  # mu = vector of means
##D  mu = c(30, 50, -0.10)
##D  # CM = covariance matrix
##D  CM = matrix(c(13, 7.8, -0.055,
##D                7.8, 15, -0.42,
##D                -0.055, -0.42, 0.056), nrow=3, ncol=3)
##D  CMm1 <- solve(CM)
##D  term2 <- exp(-((x - mu) %*% CMm1 %*% (x - mu))/2)
##D  term2
##D }
##D 
##D with_hist_known3 <- BayesianMCMC (xcont=serie20, xhist=serie100[!is.na(serie100)], 
##D                                   infhist=NA, suphist=NA,
##D                                   nbans=100, seuil=250,
##D                                   nbpas=5000, nbchaines=3, apriori=fNORM3,
##D                                   confint=c(0.05, 0.95), dist="GEV")
##D plot(with_hist_known3, 5)
##D plot(with_hist_known3, 4)
##D plot(with_hist_known, 4)
##D plot(with_hist_known)
##D plot(with_hist_known3)
## End(Not run)



