/*	$NetBSD: mvsoc.c,v 1.29 2017/03/10 15:44:24 skrll Exp $	*/
/*
 * Copyright (c) 2007, 2008, 2013, 2014, 2016 KIYOHARA Takashi
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: mvsoc.c,v 1.29 2017/03/10 15:44:24 skrll Exp $");

#include "opt_cputypes.h"
#include "opt_mvsoc.h"
#ifdef ARMADAXP
#include "mvxpe.h"
#include "mvxpsec.h"
#endif

#include <sys/param.h>
#include <sys/boot_flag.h>
#include <sys/systm.h>
#include <sys/bus.h>
#include <sys/device.h>
#include <sys/errno.h>

#include <dev/pci/pcidevs.h>
#include <dev/pci/pcireg.h>
#include <dev/marvell/marvellreg.h>
#include <dev/marvell/marvellvar.h>

#include <arm/marvell/mvsocreg.h>
#include <arm/marvell/mvsocvar.h>
#include <arm/marvell/orionreg.h>
#include <arm/marvell/kirkwoodreg.h>
#include <arm/marvell/mv78xx0reg.h>
#include <arm/marvell/dovereg.h>
#include <arm/marvell/armadaxpvar.h>
#include <arm/marvell/armadaxpreg.h>

#include <uvm/uvm.h>

#include "locators.h"

#ifdef MVSOC_CONSOLE_EARLY
#include <dev/ic/ns16550reg.h>
#include <dev/ic/comreg.h>
#include <dev/cons.h>
#endif

static int mvsoc_match(device_t, struct cfdata *, void *);
static void mvsoc_attach(device_t, device_t, void *);

static int mvsoc_print(void *, const char *);
static int mvsoc_search(device_t, cfdata_t, const int *, void *);

static int mvsoc_target_ddr(uint32_t, uint32_t *, uint32_t *);
static int mvsoc_target_ddr3(uint32_t, uint32_t *, uint32_t *);
static int mvsoc_target_axi(int, uint32_t *, uint32_t *);
static int mvsoc_target_peripheral(uint32_t, uint32_t, uint32_t *, uint32_t *);

uint32_t mvPclk, mvSysclk, mvTclk = 0;
int nwindow = 0, nremap = 0;
static vaddr_t regbase = 0xffffffff, dsc_base, pex_base;
vaddr_t mlmb_base;

void (*mvsoc_intr_init)(void);
int (*mvsoc_clkgating)(struct marvell_attach_args *);


#ifdef MVSOC_CONSOLE_EARLY
static vaddr_t com_base;

static inline uint32_t
uart_read(bus_size_t o)
{
	return *(volatile uint32_t *)(com_base + (o << 2));
}

static inline void
uart_write(bus_size_t o, uint32_t v)
{
	*(volatile uint32_t *)(com_base + (o << 2)) = v;
}

static int
mvsoc_cngetc(dev_t dv)
{
        if ((uart_read(com_lsr) & LSR_RXRDY) == 0)
		return -1;

	return uart_read(com_data) & 0xff;
}

static void
mvsoc_cnputc(dev_t dv, int c)
{
	int timo = 150000;

        while ((uart_read(com_lsr) & LSR_TXRDY) == 0 && --timo > 0)
		;

	uart_write(com_data, c);

	timo = 150000;
        while ((uart_read(com_lsr) & LSR_TSRE) == 0 && --timo > 0)
		;
}

static struct consdev mvsoc_earlycons = {
	.cn_putc = mvsoc_cnputc,
	.cn_getc = mvsoc_cngetc,
	.cn_pollc = nullcnpollc,
};
#endif


/* attributes */
static struct {
	int tag;
	uint32_t attr;
	uint32_t target;
} mvsoc_tags[] = {
	{ MARVELL_TAG_SDRAM_CS0,
	  MARVELL_ATTR_SDRAM_CS0,	MVSOC_UNITID_DDR },
	{ MARVELL_TAG_SDRAM_CS1,
	  MARVELL_ATTR_SDRAM_CS1,	MVSOC_UNITID_DDR },
	{ MARVELL_TAG_SDRAM_CS2,
	  MARVELL_ATTR_SDRAM_CS2,	MVSOC_UNITID_DDR },
	{ MARVELL_TAG_SDRAM_CS3,
	  MARVELL_ATTR_SDRAM_CS3,	MVSOC_UNITID_DDR },

	{ MARVELL_TAG_AXI_CS0,
	  MARVELL_ATTR_AXI_DDR,		MVSOC_UNITID_DDR },
	{ MARVELL_TAG_AXI_CS1,
	  MARVELL_ATTR_AXI_DDR,		MVSOC_UNITID_DDR },

	{ MARVELL_TAG_DDR3_CS0,
	  MARVELL_ATTR_SDRAM_CS0,	MVSOC_UNITID_DDR },
	{ MARVELL_TAG_DDR3_CS1,
	  MARVELL_ATTR_SDRAM_CS1,	MVSOC_UNITID_DDR },
	{ MARVELL_TAG_DDR3_CS2,
	  MARVELL_ATTR_SDRAM_CS2,	MVSOC_UNITID_DDR },
	{ MARVELL_TAG_DDR3_CS3,
	  MARVELL_ATTR_SDRAM_CS3,	MVSOC_UNITID_DDR },

#if defined(ORION)
	{ ORION_TAG_DEVICE_CS0,
	  ORION_ATTR_DEVICE_CS0,	MVSOC_UNITID_DEVBUS },
	{ ORION_TAG_DEVICE_CS1,
	  ORION_ATTR_DEVICE_CS1,	MVSOC_UNITID_DEVBUS },
	{ ORION_TAG_DEVICE_CS2,
	  ORION_ATTR_DEVICE_CS2,	MVSOC_UNITID_DEVBUS },
	{ ORION_TAG_DEVICE_BOOTCS,
	  ORION_ATTR_BOOT_CS,		MVSOC_UNITID_DEVBUS },
	{ ORION_TAG_FLASH_CS,
	  ORION_ATTR_FLASH_CS,		MVSOC_UNITID_DEVBUS },
	{ ORION_TAG_PEX0_MEM,
	  ORION_ATTR_PEX_MEM,		MVSOC_UNITID_PEX },
	{ ORION_TAG_PEX0_IO,
	  ORION_ATTR_PEX_IO,		MVSOC_UNITID_PEX },
	{ ORION_TAG_PEX1_MEM,
	  ORION_ATTR_PEX_MEM,		ORION_UNITID_PEX1 },
	{ ORION_TAG_PEX1_IO,
	  ORION_ATTR_PEX_IO,		ORION_UNITID_PEX1 },
	{ ORION_TAG_PCI_MEM,
	  ORION_ATTR_PCI_MEM,		ORION_UNITID_PCI },
	{ ORION_TAG_PCI_IO,
	  ORION_ATTR_PCI_IO,		ORION_UNITID_PCI },
	{ ORION_TAG_CRYPT,
	  ORION_ATTR_CRYPT,		ORION_UNITID_CRYPT },
#endif

#if defined(KIRKWOOD)
	{ KIRKWOOD_TAG_NAND,
	  KIRKWOOD_ATTR_NAND,		MVSOC_UNITID_DEVBUS },
	{ KIRKWOOD_TAG_SPI,
	  KIRKWOOD_ATTR_SPI,		MVSOC_UNITID_DEVBUS },
	{ KIRKWOOD_TAG_BOOTROM,
	  KIRKWOOD_ATTR_BOOTROM,	MVSOC_UNITID_DEVBUS },
	{ KIRKWOOD_TAG_PEX_MEM,
	  KIRKWOOD_ATTR_PEX_MEM,	MVSOC_UNITID_PEX },
	{ KIRKWOOD_TAG_PEX_IO,
	  KIRKWOOD_ATTR_PEX_IO,		MVSOC_UNITID_PEX },
	{ KIRKWOOD_TAG_PEX1_MEM,
	  KIRKWOOD_ATTR_PEX1_MEM,	MVSOC_UNITID_PEX },
	{ KIRKWOOD_TAG_PEX1_IO,
	  KIRKWOOD_ATTR_PEX1_IO,	MVSOC_UNITID_PEX },
	{ KIRKWOOD_TAG_CRYPT,
	  KIRKWOOD_ATTR_CRYPT,		KIRKWOOD_UNITID_CRYPT },
#endif

#if defined(MV78XX0)
	{ MV78XX0_TAG_DEVICE_CS0,
	  MV78XX0_ATTR_DEVICE_CS0,	MVSOC_UNITID_DEVBUS },
	{ MV78XX0_TAG_DEVICE_CS1,
	  MV78XX0_ATTR_DEVICE_CS1,	MVSOC_UNITID_DEVBUS },
	{ MV78XX0_TAG_DEVICE_CS2,
	  MV78XX0_ATTR_DEVICE_CS2,	MVSOC_UNITID_DEVBUS },
	{ MV78XX0_TAG_DEVICE_CS3,
	  MV78XX0_ATTR_DEVICE_CS3,	MVSOC_UNITID_DEVBUS },
	{ MV78XX0_TAG_DEVICE_BOOTCS,
	  MV78XX0_ATTR_BOOT_CS,		MVSOC_UNITID_DEVBUS },
	{ MV78XX0_TAG_SPI,
	  MV78XX0_ATTR_SPI,		MVSOC_UNITID_DEVBUS },
	{ MV78XX0_TAG_PEX0_MEM,
	  MV78XX0_ATTR_PEX_0_MEM,	MVSOC_UNITID_PEX },
	{ MV78XX0_TAG_PEX01_MEM,
	  MV78XX0_ATTR_PEX_1_MEM,	MVSOC_UNITID_PEX },
	{ MV78XX0_TAG_PEX02_MEM,
	  MV78XX0_ATTR_PEX_2_MEM,	MVSOC_UNITID_PEX },
	{ MV78XX0_TAG_PEX03_MEM,
	  MV78XX0_ATTR_PEX_3_MEM,	MVSOC_UNITID_PEX },
	{ MV78XX0_TAG_PEX0_IO,
	  MV78XX0_ATTR_PEX_0_IO,	MVSOC_UNITID_PEX },
	{ MV78XX0_TAG_PEX01_IO,
	  MV78XX0_ATTR_PEX_1_IO,	MVSOC_UNITID_PEX },
	{ MV78XX0_TAG_PEX02_IO,
	  MV78XX0_ATTR_PEX_2_IO,	MVSOC_UNITID_PEX },
	{ MV78XX0_TAG_PEX03_IO,
	  MV78XX0_ATTR_PEX_3_IO,	MVSOC_UNITID_PEX },
	{ MV78XX0_TAG_PEX1_MEM,
	  MV78XX0_ATTR_PEX_0_MEM,	MV78XX0_UNITID_PEX1 },
	{ MV78XX0_TAG_PEX11_MEM,
	  MV78XX0_ATTR_PEX_1_MEM,	MV78XX0_UNITID_PEX1 },
	{ MV78XX0_TAG_PEX12_MEM,
	  MV78XX0_ATTR_PEX_2_MEM,	MV78XX0_UNITID_PEX1 },
	{ MV78XX0_TAG_PEX13_MEM,
	  MV78XX0_ATTR_PEX_3_MEM,	MV78XX0_UNITID_PEX1 },
	{ MV78XX0_TAG_PEX1_IO,
	  MV78XX0_ATTR_PEX_0_IO,	MV78XX0_UNITID_PEX1 },
	{ MV78XX0_TAG_PEX11_IO,
	  MV78XX0_ATTR_PEX_1_IO,	MV78XX0_UNITID_PEX1 },
	{ MV78XX0_TAG_PEX12_IO,
	  MV78XX0_ATTR_PEX_2_IO,	MV78XX0_UNITID_PEX1 },
	{ MV78XX0_TAG_PEX13_IO,
	  MV78XX0_ATTR_PEX_3_IO,	MV78XX0_UNITID_PEX1 },
	{ MV78XX0_TAG_CRYPT,
	  MV78XX0_ATTR_CRYPT,		MV78XX0_UNITID_CRYPT },
#endif

#if defined(DOVE)
	{ DOVE_TAG_PEX0_MEM,
	  DOVE_ATTR_PEX_MEM,		MVSOC_UNITID_PEX },
	{ DOVE_TAG_PEX0_IO,
	  DOVE_ATTR_PEX_IO,		MVSOC_UNITID_PEX },
	{ DOVE_TAG_PEX1_MEM,
	  DOVE_ATTR_PEX_MEM,		DOVE_UNITID_PEX1 },
	{ DOVE_TAG_PEX1_IO,
	  DOVE_ATTR_PEX_IO,		DOVE_UNITID_PEX1 },
	{ DOVE_TAG_CRYPT,
	  DOVE_ATTR_SA,			DOVE_UNITID_SA },
	{ DOVE_TAG_SPI0,
	  DOVE_ATTR_SPI0,		MVSOC_UNITID_DEVBUS },
	{ DOVE_TAG_SPI1,
	  DOVE_ATTR_SPI1,		MVSOC_UNITID_DEVBUS },
	{ DOVE_TAG_BOOTROM,
	  DOVE_ATTR_BOOTROM,		MVSOC_UNITID_DEVBUS },
	{ DOVE_TAG_PMU,
	  DOVE_ATTR_NAND,		DOVE_UNITID_NAND },
	{ DOVE_TAG_PMU,
	  DOVE_ATTR_PMU,		DOVE_UNITID_PMU },
#endif

#if defined(ARMADAXP)
	{ ARMADAXP_TAG_PEX00_MEM,
	  ARMADAXP_ATTR_PEXx0_MEM,	ARMADAXP_UNITID_PEX0 },
	{ ARMADAXP_TAG_PEX00_IO,
	  ARMADAXP_ATTR_PEXx0_IO,	ARMADAXP_UNITID_PEX0 },

	{ ARMADAXP_TAG_PEX01_MEM,
	  ARMADAXP_ATTR_PEXx1_MEM,	ARMADAXP_UNITID_PEX0 },
	{ ARMADAXP_TAG_PEX01_IO,
	  ARMADAXP_ATTR_PEXx1_IO,	ARMADAXP_UNITID_PEX0 },

	{ ARMADAXP_TAG_PEX02_MEM,
	  ARMADAXP_ATTR_PEXx2_MEM,	ARMADAXP_UNITID_PEX0 },
	{ ARMADAXP_TAG_PEX02_IO,
	  ARMADAXP_ATTR_PEXx2_IO,	ARMADAXP_UNITID_PEX0 },

	{ ARMADAXP_TAG_PEX03_MEM,
	  ARMADAXP_ATTR_PEXx3_MEM,	ARMADAXP_UNITID_PEX0 },
	{ ARMADAXP_TAG_PEX03_IO,
	  ARMADAXP_ATTR_PEXx3_IO,	ARMADAXP_UNITID_PEX0 },

	{ ARMADAXP_TAG_PEX10_MEM,
	  ARMADAXP_ATTR_PEXx0_MEM,	ARMADAXP_UNITID_PEX1 },
	{ ARMADAXP_TAG_PEX10_IO,
	  ARMADAXP_ATTR_PEXx0_IO,	ARMADAXP_UNITID_PEX1 },

	{ ARMADAXP_TAG_PEX11_MEM,
	  ARMADAXP_ATTR_PEXx1_MEM,	ARMADAXP_UNITID_PEX1 },
	{ ARMADAXP_TAG_PEX11_IO,
	  ARMADAXP_ATTR_PEXx1_IO,	ARMADAXP_UNITID_PEX1 },

	{ ARMADAXP_TAG_PEX12_MEM,
	  ARMADAXP_ATTR_PEXx2_MEM,	ARMADAXP_UNITID_PEX1 },
	{ ARMADAXP_TAG_PEX12_IO,
	  ARMADAXP_ATTR_PEXx2_IO,	ARMADAXP_UNITID_PEX1 },

	{ ARMADAXP_TAG_PEX13_MEM,
	  ARMADAXP_ATTR_PEXx3_MEM,	ARMADAXP_UNITID_PEX1 },
	{ ARMADAXP_TAG_PEX13_IO,
	  ARMADAXP_ATTR_PEXx3_IO,	ARMADAXP_UNITID_PEX1 },

	{ ARMADAXP_TAG_PEX2_MEM,
	  ARMADAXP_ATTR_PEX2_MEM,	ARMADAXP_UNITID_PEX2 },
	{ ARMADAXP_TAG_PEX2_IO,
	  ARMADAXP_ATTR_PEX2_IO,	ARMADAXP_UNITID_PEX2 },

	{ ARMADAXP_TAG_PEX3_MEM,
	  ARMADAXP_ATTR_PEX3_MEM,	ARMADAXP_UNITID_PEX3 },
	{ ARMADAXP_TAG_PEX3_IO,
	  ARMADAXP_ATTR_PEX3_IO,	ARMADAXP_UNITID_PEX3 },

	{ ARMADAXP_TAG_CRYPT0,
	  ARMADAXP_ATTR_CRYPT0_NOSWAP,	ARMADAXP_UNITID_CRYPT },
	{ ARMADAXP_TAG_CRYPT1,
	  ARMADAXP_ATTR_CRYPT1_NOSWAP,	ARMADAXP_UNITID_CRYPT },
#endif
};

#if defined(ORION)
#define ORION_1(m)	MARVELL_ORION_1_ ## m
#define ORION_2(m)	MARVELL_ORION_2_ ## m
#endif
#if defined(KIRKWOOD)
#undef KIRKWOOD
#define KIRKWOOD(m)	MARVELL_KIRKWOOD_ ## m
#endif
#if defined(MV78XX0)
#undef MV78XX0
#define MV78XX0(m)	MARVELL_MV78XX0_ ## m
#endif
#if defined(DOVE)
#undef DOVE
#define DOVE(m)		MARVELL_DOVE_ ## m
#endif
#if defined(ARMADAXP)
#undef ARMADAXP
#define ARMADAXP(m)	MARVELL_ARMADAXP_ ## m
#define ARMADA370(m)	MARVELL_ARMADA370_ ## m
#endif
static struct {
	uint16_t model;
	uint8_t rev;
	const char *modelstr;
	const char *revstr;
	const char *typestr;
} nametbl[] = {
#if defined(ORION)
	{ ORION_1(88F1181),	0, "MV88F1181", NULL,	"Orion1" },
	{ ORION_1(88F5082),	2, "MV88F5082", "A2",	"Orion1" },
	{ ORION_1(88F5180N),	3, "MV88F5180N","B1",	"Orion1" },
	{ ORION_1(88F5181),	0, "MV88F5181",	"A0",	"Orion1" },
	{ ORION_1(88F5181),	1, "MV88F5181",	"A1",	"Orion1" },
	{ ORION_1(88F5181),	2, "MV88F5181",	"B0",	"Orion1" },
	{ ORION_1(88F5181),	3, "MV88F5181",	"B1",	"Orion1" },
	{ ORION_1(88F5181),	8, "MV88F5181L","A0",	"Orion1" },
	{ ORION_1(88F5181),	9, "MV88F5181L","A1",	"Orion1" },
	{ ORION_1(88F5182),	0, "MV88F5182",	"A0",	"Orion1" },
	{ ORION_1(88F5182),	1, "MV88F5182",	"A1",	"Orion1" },
	{ ORION_1(88F5182),	2, "MV88F5182",	"A2",	"Orion1" },
	{ ORION_1(88F6082),	0, "MV88F6082",	"A0",	"Orion1" },
	{ ORION_1(88F6082),	1, "MV88F6082",	"A1",	"Orion1" },
	{ ORION_1(88F6183),	0, "MV88F6183",	"A0",	"Orion1" },
	{ ORION_1(88F6183),	1, "MV88F6183",	"Z0",	"Orion1" },
	{ ORION_1(88W8660),	0, "MV88W8660",	"A0",	"Orion1" },
	{ ORION_1(88W8660),	1, "MV88W8660",	"A1",	"Orion1" },

	{ ORION_2(88F1281),	0, "MV88F1281",	"A0",	"Orion2" },
	{ ORION_2(88F5281),	0, "MV88F5281",	"A0",	"Orion2" },
	{ ORION_2(88F5281),	1, "MV88F5281",	"B0",	"Orion2" },
	{ ORION_2(88F5281),	2, "MV88F5281",	"C0",	"Orion2" },
	{ ORION_2(88F5281),	3, "MV88F5281",	"C1",	"Orion2" },
	{ ORION_2(88F5281),	4, "MV88F5281",	"D0",	"Orion2" },
#endif

#if defined(KIRKWOOD)
	{ KIRKWOOD(88F6180),	2, "88F6180",	"A0",	"Kirkwood" },
	{ KIRKWOOD(88F6180),	3, "88F6180",	"A1",	"Kirkwood" },
	{ KIRKWOOD(88F6192),	0, "88F619x",	"Z0",	"Kirkwood" },
	{ KIRKWOOD(88F6192),	2, "88F619x",	"A0",	"Kirkwood" },
	{ KIRKWOOD(88F6192),	3, "88F619x",	"A1",	"Kirkwood" },
	{ KIRKWOOD(88F6281),	0, "88F6281",	"Z0",	"Kirkwood" },
	{ KIRKWOOD(88F6281),	2, "88F6281",	"A0",	"Kirkwood" },
	{ KIRKWOOD(88F6281),	3, "88F6281",	"A1",	"Kirkwood" },
	{ KIRKWOOD(88F6282),	0, "88F6282",	"A0",	"Kirkwood" },
	{ KIRKWOOD(88F6282),	1, "88F6282",	"A1",	"Kirkwood" },
#endif

#if defined(MV78XX0)
	{ MV78XX0(MV78100),	1, "MV78100",	"A0",  "Discovery Innovation" },
	{ MV78XX0(MV78100),	2, "MV78100",	"A1",  "Discovery Innovation" },
	{ MV78XX0(MV78200),	1, "MV78200",	"A0",  "Discovery Innovation" },
#endif

#if defined(DOVE)
	{ DOVE(88AP510),	0, "88AP510",	"Z0",  "Dove" },
	{ DOVE(88AP510),	1, "88AP510",	"Z1",  "Dove" },
	{ DOVE(88AP510),	2, "88AP510",	"Y0",  "Dove" },
	{ DOVE(88AP510),	3, "88AP510",	"Y1",  "Dove" },
	{ DOVE(88AP510),	4, "88AP510",	"X0",  "Dove" },
	{ DOVE(88AP510),	6, "88AP510",	"A0",  "Dove" },
	{ DOVE(88AP510),	7, "88AP510",	"A1",  "Dove" },
#endif

#if defined(ARMADAXP)
	{ ARMADAXP(MV78130),	1, "MV78130",	"A0",  "Armada XP" },
	{ ARMADAXP(MV78160),	1, "MV78160",	"A0",  "Armada XP" },
	{ ARMADAXP(MV78230),	1, "MV78230",	"A0",  "Armada XP" },
	{ ARMADAXP(MV78260),	1, "MV78260",	"A0",  "Armada XP" },
	{ ARMADAXP(MV78260),	2, "MV78260",	"B0",  "Armada XP" },
	{ ARMADAXP(MV78460),	1, "MV78460",	"A0",  "Armada XP" },
	{ ARMADAXP(MV78460),	2, "MV78460",	"B0",  "Armada XP" },

	{ ARMADA370(MV6707),	0, "MV6707",	"A0",  "Armada 370" },
	{ ARMADA370(MV6707),	1, "MV6707",	"A1",  "Armada 370" },
	{ ARMADA370(MV6710),	0, "MV6710",	"A0",  "Armada 370" },
	{ ARMADA370(MV6710),	1, "MV6710",	"A1",  "Armada 370" },
	{ ARMADA370(MV6W11),	0, "MV6W11",	"A0",  "Armada 370" },
	{ ARMADA370(MV6W11),	1, "MV6W11",	"A1",  "Armada 370" },
#endif
};

enum marvell_tags ddr_tags[] = {
	MARVELL_TAG_SDRAM_CS0,
	MARVELL_TAG_SDRAM_CS1,
	MARVELL_TAG_SDRAM_CS2,
	MARVELL_TAG_SDRAM_CS3,

	MARVELL_TAG_UNDEFINED
};
enum marvell_tags ddr3_tags[] = {
	MARVELL_TAG_DDR3_CS0,
	MARVELL_TAG_DDR3_CS1,
	MARVELL_TAG_DDR3_CS2,
	MARVELL_TAG_DDR3_CS3,

	MARVELL_TAG_UNDEFINED
};
enum marvell_tags axi_tags[] = {
	MARVELL_TAG_AXI_CS0,
	MARVELL_TAG_AXI_CS1,

	MARVELL_TAG_UNDEFINED
};
static struct {
	uint16_t model;
	uint8_t rev;
	enum marvell_tags *tags;
} tagstbl[] = {
#if defined(ORION)
	{ ORION_1(88F1181),	0, ddr_tags },
	{ ORION_1(88F5082),	2, ddr_tags },
	{ ORION_1(88F5180N),	3, ddr_tags },
	{ ORION_1(88F5181),	0, ddr_tags },
	{ ORION_1(88F5181),	1, ddr_tags },
	{ ORION_1(88F5181),	2, ddr_tags },
	{ ORION_1(88F5181),	3, ddr_tags },
	{ ORION_1(88F5181),	8, ddr_tags },
	{ ORION_1(88F5181),	9, ddr_tags },
	{ ORION_1(88F5182),	0, ddr_tags },
	{ ORION_1(88F5182),	1, ddr_tags },
	{ ORION_1(88F5182),	2, ddr_tags },
	{ ORION_1(88F6082),	0, ddr_tags },
	{ ORION_1(88F6082),	1, ddr_tags },
	{ ORION_1(88F6183),	0, ddr_tags },
	{ ORION_1(88F6183),	1, ddr_tags },
	{ ORION_1(88W8660),	0, ddr_tags },
	{ ORION_1(88W8660),	1, ddr_tags },

	{ ORION_2(88F1281),	0, ddr_tags },
	{ ORION_2(88F5281),	0, ddr_tags },
	{ ORION_2(88F5281),	1, ddr_tags },
	{ ORION_2(88F5281),	2, ddr_tags },
	{ ORION_2(88F5281),	3, ddr_tags },
	{ ORION_2(88F5281),	4, ddr_tags },
#endif

#if defined(KIRKWOOD)
	{ KIRKWOOD(88F6180),	2, ddr_tags },
	{ KIRKWOOD(88F6180),	3, ddr_tags },
	{ KIRKWOOD(88F6192),	0, ddr_tags },
	{ KIRKWOOD(88F6192),	2, ddr_tags },
	{ KIRKWOOD(88F6192),	3, ddr_tags },
	{ KIRKWOOD(88F6281),	0, ddr_tags },
	{ KIRKWOOD(88F6281),	2, ddr_tags },
	{ KIRKWOOD(88F6281),	3, ddr_tags },
	{ KIRKWOOD(88F6282),	0, ddr_tags },
	{ KIRKWOOD(88F6282),	1, ddr_tags },
#endif

#if defined(MV78XX0)
	{ MV78XX0(MV78100),	1, ddr_tags },
	{ MV78XX0(MV78100),	2, ddr_tags },
	{ MV78XX0(MV78200),	1, ddr_tags },
#endif

#if defined(DOVE)
	{ DOVE(88AP510),	0, axi_tags },
	{ DOVE(88AP510),	1, axi_tags },
	{ DOVE(88AP510),	2, axi_tags },
	{ DOVE(88AP510),	3, axi_tags },
	{ DOVE(88AP510),	4, axi_tags },
	{ DOVE(88AP510),	5, axi_tags },
	{ DOVE(88AP510),	6, axi_tags },
	{ DOVE(88AP510),	7, axi_tags },
#endif

#if defined(ARMADAXP)
	{ ARMADAXP(MV78130),	1, ddr3_tags },
	{ ARMADAXP(MV78160),	1, ddr3_tags },
	{ ARMADAXP(MV78230),	1, ddr3_tags },
	{ ARMADAXP(MV78260),	1, ddr3_tags },
	{ ARMADAXP(MV78260),	2, ddr3_tags },
	{ ARMADAXP(MV78460),	1, ddr3_tags },
	{ ARMADAXP(MV78460),	2, ddr3_tags },

	{ ARMADA370(MV6707),	0, ddr3_tags },
	{ ARMADA370(MV6707),	1, ddr3_tags },
	{ ARMADA370(MV6710),	0, ddr3_tags },
	{ ARMADA370(MV6710),	1, ddr3_tags },
	{ ARMADA370(MV6W11),	0, ddr3_tags },
	{ ARMADA370(MV6W11),	1, ddr3_tags },
#endif
};


#define OFFSET_DEFAULT	MVA_OFFSET_DEFAULT
#define IRQ_DEFAULT	MVA_IRQ_DEFAULT
static const struct mvsoc_periph {
	int model;
	const char *name;
	int unit;
	bus_size_t offset;
	int irq;
} mvsoc_periphs[] = {
#if defined(ORION)
#define ORION_IRQ_TMR		(32 + MVSOC_MLMB_MLMBI_CPUTIMER0INTREQ)

    { ORION_1(88F1181),	"mvsoctmr",0, MVSOC_TMR_BASE,	ORION_IRQ_TMR },
    { ORION_1(88F1181),	"mvsocgpp",0, MVSOC_GPP_BASE,	ORION_IRQ_GPIO7_0 },
    { ORION_1(88F1181),	"com",     0, MVSOC_COM0_BASE,	ORION_IRQ_UART0 },
    { ORION_1(88F1181),	"com",     1, MVSOC_COM1_BASE,	ORION_IRQ_UART1 },
    { ORION_1(88F1181),	"gttwsi",  0, MVSOC_TWSI_BASE,	ORION_IRQ_TWSI },
    { ORION_1(88F1181),	"mvpex",   0, MVSOC_PEX_BASE,	ORION_IRQ_PEX0INT },
    { ORION_1(88F1181),	"mvpex",   1, ORION_PEX1_BASE,	ORION_IRQ_PEX1INT },

    { ORION_1(88F5082),	"mvsoctmr",0, MVSOC_TMR_BASE,	ORION_IRQ_TMR },
    { ORION_1(88F5082),	"mvsocgpp",0, MVSOC_GPP_BASE,	ORION_IRQ_GPIO7_0 },
    { ORION_1(88F5082),	"com",     0, MVSOC_COM0_BASE,	ORION_IRQ_UART0 },
    { ORION_1(88F5082),	"com",     1, MVSOC_COM1_BASE,	ORION_IRQ_UART1 },
    { ORION_1(88F5082),	"ehci",    0, ORION_USB0_BASE,	ORION_IRQ_USBCNT0 },
    { ORION_1(88F5082),	"ehci",    1, ORION_USB1_BASE,	ORION_IRQ_USBCNT1 },
    { ORION_1(88F5082),	"gtidmac", 0, ORION_IDMAC_BASE,	IRQ_DEFAULT },
    { ORION_1(88F5082),	"gttwsi",  0, MVSOC_TWSI_BASE,	ORION_IRQ_TWSI },
    { ORION_1(88F5082),	"mvcesa",  0, ORION_CESA_BASE,	ORION_IRQ_SECURITYINTR},
    { ORION_1(88F5082),	"mvgbec",  0, ORION_GBE_BASE,	IRQ_DEFAULT },
    { ORION_1(88F5082),	"mvpex",   0, MVSOC_PEX_BASE,	ORION_IRQ_PEX0INT },
    { ORION_1(88F5082),	"mvsata",  0, ORION_SATAHC_BASE,ORION_IRQ_SATAINTR },

    { ORION_1(88F5180N),"mvsoctmr",0, MVSOC_TMR_BASE,	ORION_IRQ_TMR },
    { ORION_1(88F5180N),"mvsocgpp",0, MVSOC_GPP_BASE,	ORION_IRQ_GPIO7_0 },
    { ORION_1(88F5180N),"com",     0, MVSOC_COM0_BASE,	ORION_IRQ_UART0 },
    { ORION_1(88F5180N),"com",     1, MVSOC_COM1_BASE,	ORION_IRQ_UART1 },
    { ORION_1(88F5180N),"ehci",    0, ORION_USB0_BASE,	ORION_IRQ_USBCNT0 },
    { ORION_1(88F5180N),"gtidmac", 0, ORION_IDMAC_BASE,	IRQ_DEFAULT },
    { ORION_1(88F5180N),"gtpci",   0, ORION_PCI_BASE,	ORION_IRQ_PEX0INT },
    { ORION_1(88F5180N),"gttwsi",  0, MVSOC_TWSI_BASE,	ORION_IRQ_TWSI },
    { ORION_1(88F5180N),"mvgbec",  0, ORION_GBE_BASE,	IRQ_DEFAULT },
    { ORION_1(88F5180N),"mvpex",   0, MVSOC_PEX_BASE,	ORION_IRQ_PEX0INT },

    { ORION_1(88F5181),	"mvsoctmr",0, MVSOC_TMR_BASE,	ORION_IRQ_TMR },
    { ORION_1(88F5181),	"mvsocgpp",0, MVSOC_GPP_BASE,	ORION_IRQ_GPIO7_0 },
    { ORION_1(88F5181),	"com",     0, MVSOC_COM0_BASE,	ORION_IRQ_UART0 },
    { ORION_1(88F5181),	"com",     1, MVSOC_COM1_BASE,	ORION_IRQ_UART1 },
    { ORION_1(88F5181),	"ehci",    0, ORION_USB0_BASE,	ORION_IRQ_USBCNT0 },
    { ORION_1(88F5181),	"gtidmac", 0, ORION_IDMAC_BASE,	IRQ_DEFAULT },
    { ORION_1(88F5181),	"gtpci",   0, ORION_PCI_BASE,	ORION_IRQ_PEX0INT },
    { ORION_1(88F5181),	"gttwsi",  0, MVSOC_TWSI_BASE,	ORION_IRQ_TWSI },
    { ORION_1(88F5181),	"mvcesa",  0, ORION_CESA_BASE,	ORION_IRQ_SECURITYINTR},
    { ORION_1(88F5181),	"mvgbec",  0, ORION_GBE_BASE,	IRQ_DEFAULT },
    { ORION_1(88F5181),	"mvpex",   0, MVSOC_PEX_BASE,	ORION_IRQ_PEX0INT },

    { ORION_1(88F5182),	"mvsoctmr",0, MVSOC_TMR_BASE,	ORION_IRQ_TMR },
    { ORION_1(88F5182),	"mvsocgpp",0, MVSOC_GPP_BASE,	ORION_IRQ_GPIO7_0 },
    { ORION_1(88F5182),	"com",     0, MVSOC_COM0_BASE,	ORION_IRQ_UART0 },
    { ORION_1(88F5182),	"com",     1, MVSOC_COM1_BASE,	ORION_IRQ_UART1 },
    { ORION_1(88F5182),	"ehci",    0, ORION_USB0_BASE,	ORION_IRQ_USBCNT0 },
    { ORION_1(88F5182),	"ehci",    1, ORION_USB1_BASE,	ORION_IRQ_USBCNT1 },
    { ORION_1(88F5182),	"gtidmac", 0, ORION_IDMAC_BASE,	IRQ_DEFAULT },
    { ORION_1(88F5182),	"gtpci",   0, ORION_PCI_BASE,	ORION_IRQ_PEX0INT },
    { ORION_1(88F5182),	"gttwsi",  0, MVSOC_TWSI_BASE,	ORION_IRQ_TWSI },
    { ORION_1(88F5182),	"mvgbec",  0, ORION_GBE_BASE,	IRQ_DEFAULT },
    { ORION_1(88F5182),	"mvsata",  0, ORION_SATAHC_BASE,ORION_IRQ_SATAINTR },
    { ORION_1(88F5182),	"mvpex",   0, MVSOC_PEX_BASE,	ORION_IRQ_PEX0INT },

    { ORION_1(88F6082),	"mvsoctmr",0, MVSOC_TMR_BASE,	ORION_IRQ_TMR },
    { ORION_1(88F6082),	"mvsocgpp",0, MVSOC_GPP_BASE,	ORION_IRQ_GPIO7_0 },
    { ORION_1(88F6082),	"com",     0, MVSOC_COM0_BASE,	ORION_IRQ_UART0 },
    { ORION_1(88F6082),	"com",     1, MVSOC_COM1_BASE,	ORION_IRQ_UART1 },
    { ORION_1(88F6082),	"ehci",    0, ORION_USB0_BASE,	ORION_IRQ_USBCNT0 },
    { ORION_1(88F6082),	"gttwsi",  0, MVSOC_TWSI_BASE,	ORION_IRQ_TWSI },
    { ORION_1(88F6082),	"mvcesa",  0, ORION_CESA_BASE,	ORION_IRQ_SECURITYINTR},
    { ORION_1(88F6082),	"mvgbec",  0, ORION_GBE_BASE,	IRQ_DEFAULT },
    { ORION_1(88F6082),	"mvsata",  0, ORION_SATAHC_BASE,ORION_IRQ_SATAINTR },
    { ORION_1(88F6082),	"mvpex",   0, MVSOC_PEX_BASE,	ORION_IRQ_PEX0INT },

    { ORION_1(88F6183),	"mvsoctmr",0, MVSOC_TMR_BASE,	ORION_IRQ_TMR },
    { ORION_1(88F6183),	"mvsocgpp",0, MVSOC_GPP_BASE,	ORION_IRQ_GPIO7_0 },
    { ORION_1(88F6183),	"gttwsi",  0, MVSOC_TWSI_BASE,	ORION_IRQ_TWSI },
    { ORION_1(88F6183),	"mvpex",   0, MVSOC_PEX_BASE,	ORION_IRQ_PEX0INT },

    { ORION_1(88W8660),	"mvsoctmr",0, MVSOC_TMR_BASE,	ORION_IRQ_TMR },
    { ORION_1(88W8660),	"mvsocgpp",0, MVSOC_GPP_BASE,	ORION_IRQ_GPIO7_0 },
    { ORION_1(88W8660),	"com",     0, MVSOC_COM0_BASE,	ORION_IRQ_UART0 },
    { ORION_1(88W8660),	"com",     1, MVSOC_COM1_BASE,	ORION_IRQ_UART1 },
    { ORION_1(88W8660),	"ehci",    0, ORION_USB0_BASE,	ORION_IRQ_USBCNT0 },
    { ORION_1(88W8660),	"gtidmac", 0, ORION_IDMAC_BASE,	IRQ_DEFAULT },
    { ORION_1(88W8660),	"gtpci",   0, ORION_PCI_BASE,	ORION_IRQ_PEX0INT },
    { ORION_1(88W8660),	"gttwsi",  0, MVSOC_TWSI_BASE,	ORION_IRQ_TWSI },
    { ORION_1(88W8660),	"mvgbec",  0, ORION_GBE_BASE,	IRQ_DEFAULT },
    { ORION_1(88W8660),	"mvpex",   0, MVSOC_PEX_BASE,	ORION_IRQ_PEX0INT },

    { ORION_2(88F1281),	"mvsoctmr",0, MVSOC_TMR_BASE,	ORION_IRQ_TMR },
    { ORION_2(88F1281),	"mvsocgpp",0, MVSOC_GPP_BASE,	ORION_IRQ_GPIO7_0 },
    { ORION_2(88F1281),	"com",     0, MVSOC_COM0_BASE,	ORION_IRQ_UART0 },
    { ORION_2(88F1281),	"com",     1, MVSOC_COM1_BASE,	ORION_IRQ_UART1 },
    { ORION_2(88F1281),	"gttwsi",  0, MVSOC_TWSI_BASE,	ORION_IRQ_TWSI },
    { ORION_2(88F1281),	"mvpex",   0, MVSOC_PEX_BASE,	ORION_IRQ_PEX0INT },
    { ORION_2(88F1281),	"mvpex",   1, ORION_PEX1_BASE,	ORION_IRQ_PEX1INT },

    { ORION_2(88F5281),	"mvsoctmr",0, MVSOC_TMR_BASE,	ORION_IRQ_TMR },
    { ORION_2(88F5281),	"mvsocgpp",0, MVSOC_GPP_BASE,	ORION_IRQ_GPIO7_0 },
    { ORION_2(88F5281),	"com",     0, MVSOC_COM0_BASE,	ORION_IRQ_UART0 },
    { ORION_2(88F5281),	"com",     1, MVSOC_COM1_BASE,	ORION_IRQ_UART1 },
    { ORION_2(88F5281),	"ehci",    0, ORION_USB0_BASE,	ORION_IRQ_USBCNT0 },
    { ORION_2(88F5281),	"gtidmac", 0, ORION_IDMAC_BASE,	IRQ_DEFAULT },
    { ORION_2(88F5281),	"gtpci",   0, ORION_PCI_BASE,	ORION_IRQ_PEX0INT },
    { ORION_2(88F5281),	"gttwsi",  0, MVSOC_TWSI_BASE,	ORION_IRQ_TWSI },
    { ORION_2(88F5281),	"mvgbec",  0, ORION_GBE_BASE,	IRQ_DEFAULT },
    { ORION_2(88F5281),	"mvpex",   0, MVSOC_PEX_BASE,	ORION_IRQ_PEX0INT },
#endif

#if defined(KIRKWOOD)
#define KIRKWOOD_IRQ_TMR	(64 + MVSOC_MLMB_MLMBI_CPUTIMER0INTREQ)

    { KIRKWOOD(88F6180),"mvsoctmr",0, MVSOC_TMR_BASE,	KIRKWOOD_IRQ_TMR },
    { KIRKWOOD(88F6180),"mvsocgpp",0, MVSOC_GPP_BASE,	KIRKWOOD_IRQ_GPIOLO7_0},
    { KIRKWOOD(88F6180),"mvsocrtc",0, KIRKWOOD_RTC_BASE,IRQ_DEFAULT },
    { KIRKWOOD(88F6180),"com",     0, MVSOC_COM0_BASE,	KIRKWOOD_IRQ_UART0INT },
    { KIRKWOOD(88F6180),"com",     1, MVSOC_COM1_BASE,	KIRKWOOD_IRQ_UART1INT },
    { KIRKWOOD(88F6180),"ehci",    0, KIRKWOOD_USB_BASE,KIRKWOOD_IRQ_USB0CNT },
    { KIRKWOOD(88F6180),"gtidmac", 0, KIRKWOOD_IDMAC_BASE,IRQ_DEFAULT },
    { KIRKWOOD(88F6180),"gttwsi",  0, MVSOC_TWSI_BASE,	KIRKWOOD_IRQ_TWSI },
    { KIRKWOOD(88F6180),"mvcesa",  0, KIRKWOOD_CESA_BASE,KIRKWOOD_IRQ_SECURITYINT},
    { KIRKWOOD(88F6180),"mvgbec",  0, KIRKWOOD_GBE0_BASE,IRQ_DEFAULT },
    { KIRKWOOD(88F6180),"mvpex",   0, MVSOC_PEX_BASE,	KIRKWOOD_IRQ_PEX0INT },
    { KIRKWOOD(88F6180),"mvsdio",  0, KIRKWOOD_SDIO_BASE,KIRKWOOD_IRQ_SDIOINT },

    { KIRKWOOD(88F6192),"mvsoctmr",0, MVSOC_TMR_BASE,	KIRKWOOD_IRQ_TMR },
    { KIRKWOOD(88F6192),"mvsocgpp",0, MVSOC_GPP_BASE,	KIRKWOOD_IRQ_GPIOLO7_0},
    { KIRKWOOD(88F6192),"mvsocrtc",0, KIRKWOOD_RTC_BASE,IRQ_DEFAULT },
    { KIRKWOOD(88F6192),"com",     0, MVSOC_COM0_BASE,	KIRKWOOD_IRQ_UART0INT },
    { KIRKWOOD(88F6192),"com",     1, MVSOC_COM1_BASE,	KIRKWOOD_IRQ_UART1INT },
    { KIRKWOOD(88F6192),"ehci",    0, KIRKWOOD_USB_BASE,KIRKWOOD_IRQ_USB0CNT },
    { KIRKWOOD(88F6192),"gtidmac", 0, KIRKWOOD_IDMAC_BASE,IRQ_DEFAULT },
    { KIRKWOOD(88F6192),"gttwsi",  0, MVSOC_TWSI_BASE,	KIRKWOOD_IRQ_TWSI },
    { KIRKWOOD(88F6192),"mvcesa",  0, KIRKWOOD_CESA_BASE,KIRKWOOD_IRQ_SECURITYINT},
    { KIRKWOOD(88F6192),"mvgbec",  0, KIRKWOOD_GBE0_BASE,IRQ_DEFAULT },
    { KIRKWOOD(88F6192),"mvgbec",  1, KIRKWOOD_GBE1_BASE,IRQ_DEFAULT },
    { KIRKWOOD(88F6192),"mvpex",   0, MVSOC_PEX_BASE,	KIRKWOOD_IRQ_PEX0INT },
    { KIRKWOOD(88F6192),"mvsata",  0, KIRKWOOD_SATAHC_BASE,KIRKWOOD_IRQ_SATA },
    { KIRKWOOD(88F6192),"mvsdio",  0, KIRKWOOD_SDIO_BASE,KIRKWOOD_IRQ_SDIOINT },

    { KIRKWOOD(88F6281),"mvsoctmr",0, MVSOC_TMR_BASE,	KIRKWOOD_IRQ_TMR },
    { KIRKWOOD(88F6281),"mvsocgpp",0, MVSOC_GPP_BASE,	KIRKWOOD_IRQ_GPIOLO7_0},
    { KIRKWOOD(88F6281),"mvsocrtc",0, KIRKWOOD_RTC_BASE,IRQ_DEFAULT },
    { KIRKWOOD(88F6281),"com",     0, MVSOC_COM0_BASE,	KIRKWOOD_IRQ_UART0INT },
    { KIRKWOOD(88F6281),"com",     1, MVSOC_COM1_BASE,	KIRKWOOD_IRQ_UART1INT },
    { KIRKWOOD(88F6281),"ehci",    0, KIRKWOOD_USB_BASE,KIRKWOOD_IRQ_USB0CNT },
    { KIRKWOOD(88F6281),"gtidmac", 0, KIRKWOOD_IDMAC_BASE,IRQ_DEFAULT },
    { KIRKWOOD(88F6281),"gttwsi",  0, MVSOC_TWSI_BASE,	KIRKWOOD_IRQ_TWSI },
    { KIRKWOOD(88F6281),"mvcesa",  0, KIRKWOOD_CESA_BASE,KIRKWOOD_IRQ_SECURITYINT },
    { KIRKWOOD(88F6281),"mvgbec",  0, KIRKWOOD_GBE0_BASE,IRQ_DEFAULT },
    { KIRKWOOD(88F6281),"mvgbec",  1, KIRKWOOD_GBE1_BASE,IRQ_DEFAULT },
    { KIRKWOOD(88F6281),"mvpex",   0, MVSOC_PEX_BASE,	KIRKWOOD_IRQ_PEX0INT },
    { KIRKWOOD(88F6281),"mvsata",  0, KIRKWOOD_SATAHC_BASE,KIRKWOOD_IRQ_SATA },
    { KIRKWOOD(88F6281),"mvsdio",  0, KIRKWOOD_SDIO_BASE,KIRKWOOD_IRQ_SDIOINT },

    { KIRKWOOD(88F6282),"mvsoctmr",0, MVSOC_TMR_BASE,	KIRKWOOD_IRQ_TMR },
    { KIRKWOOD(88F6282),"mvsocgpp",0, MVSOC_GPP_BASE,	KIRKWOOD_IRQ_GPIOLO7_0},
    { KIRKWOOD(88F6282),"mvsocrtc",0, KIRKWOOD_RTC_BASE,IRQ_DEFAULT },
    { KIRKWOOD(88F6282),"mvsocts", 0, KIRKWOOD_TS_BASE,	IRQ_DEFAULT },
    { KIRKWOOD(88F6282),"com",     0, MVSOC_COM0_BASE,	KIRKWOOD_IRQ_UART0INT },
    { KIRKWOOD(88F6282),"com",     1, MVSOC_COM1_BASE,	KIRKWOOD_IRQ_UART1INT },
    { KIRKWOOD(88F6282),"ehci",    0, KIRKWOOD_USB_BASE,KIRKWOOD_IRQ_USB0CNT },
    { KIRKWOOD(88F6282),"gtidmac", 0, KIRKWOOD_IDMAC_BASE,IRQ_DEFAULT },
    { KIRKWOOD(88F6282),"gttwsi",  0, MVSOC_TWSI_BASE,	KIRKWOOD_IRQ_TWSI },
    { KIRKWOOD(88F6282),"gttwsi",  1, KIRKWOOD_TWSI1_BASE,KIRKWOOD_IRQ_TWSI1 },
    { KIRKWOOD(88F6282),"mvcesa",  0, KIRKWOOD_CESA_BASE,KIRKWOOD_IRQ_SECURITYINT},
    { KIRKWOOD(88F6282),"mvgbec",  0, KIRKWOOD_GBE0_BASE,IRQ_DEFAULT },
    { KIRKWOOD(88F6282),"mvgbec",  1, KIRKWOOD_GBE1_BASE,IRQ_DEFAULT },
    { KIRKWOOD(88F6282),"mvpex",   0, MVSOC_PEX_BASE,	KIRKWOOD_IRQ_PEX0INT },
    { KIRKWOOD(88F6282),"mvpex",   1, KIRKWOOD_PEX1_BASE,KIRKWOOD_IRQ_PEX1INT },
    { KIRKWOOD(88F6282),"mvsata",  0, KIRKWOOD_SATAHC_BASE,KIRKWOOD_IRQ_SATA },
    { KIRKWOOD(88F6282),"mvsdio",  0, KIRKWOOD_SDIO_BASE,KIRKWOOD_IRQ_SDIOINT },
#endif

#if defined(MV78XX0)
    { MV78XX0(MV78100),	"mvsoctmr",0, MVSOC_TMR_BASE,	MV78XX0_IRQ_TIMER0 },
    { MV78XX0(MV78100),	"mvsocgpp",0, MVSOC_GPP_BASE,	MV78XX0_IRQ_GPIO0_7 },
    { MV78XX0(MV78100),	"com",	   0, MVSOC_COM0_BASE,	MV78XX0_IRQ_UART0 },
    { MV78XX0(MV78100),	"com",	   1, MVSOC_COM1_BASE,	MV78XX0_IRQ_UART1 },
    { MV78XX0(MV78100),	"com",	   2, MV78XX0_COM2_BASE,MV78XX0_IRQ_UART2 },
    { MV78XX0(MV78100),	"com",	   3, MV78XX0_COM3_BASE,MV78XX0_IRQ_UART3 },
    { MV78XX0(MV78100),	"gttwsi",  0, MVSOC_TWSI_BASE,	MV78XX0_IRQ_TWSI0 },
    { MV78XX0(MV78100),	"gttwsi",  1, MV78XX0_TWSI1_BASE,MV78XX0_IRQ_TWSI1 },
    { MV78XX0(MV78100), "mvgbec",  0, MV78XX0_GBE0_BASE,IRQ_DEFAULT },
    { MV78XX0(MV78100), "mvgbec",  1, MV78XX0_GBE1_BASE,IRQ_DEFAULT },
    { MV78XX0(MV78100), "mvsata",  0, MV78XX0_SATAHC_BASE,MV78XX0_IRQ_SATA },

    { MV78XX0(MV78200),	"mvsoctmr",0, MVSOC_TMR_BASE,	MV78XX0_IRQ_TIMER0 },
    { MV78XX0(MV78200),	"mvsocgpp",0, MVSOC_GPP_BASE,	MV78XX0_IRQ_GPIO0_7 },
    { MV78XX0(MV78200),	"com",     0, MVSOC_COM0_BASE,	MV78XX0_IRQ_UART0 },
    { MV78XX0(MV78200),	"com",     1, MVSOC_COM1_BASE,	MV78XX0_IRQ_UART1 },
    { MV78XX0(MV78200),	"com",	   2, MV78XX0_COM2_BASE,MV78XX0_IRQ_UART2 },
    { MV78XX0(MV78200),	"com",	   3, MV78XX0_COM3_BASE,MV78XX0_IRQ_UART3 },
    { MV78XX0(MV78200),	"gttwsi",  0, MVSOC_TWSI_BASE,	MV78XX0_IRQ_TWSI0 },
    { MV78XX0(MV78200),	"gttwsi",  1, MV78XX0_TWSI1_BASE,MV78XX0_IRQ_TWSI1 },
    { MV78XX0(MV78200), "mvgbec",  0, MV78XX0_GBE0_BASE,IRQ_DEFAULT },
    { MV78XX0(MV78200), "mvgbec",  1, MV78XX0_GBE1_BASE,IRQ_DEFAULT },
    { MV78XX0(MV78200), "mvgbec",  2, MV78XX0_GBE2_BASE,IRQ_DEFAULT },
    { MV78XX0(MV78200), "mvgbec",  3, MV78XX0_GBE3_BASE,IRQ_DEFAULT },
    { MV78XX0(MV78200), "mvsata",  0, MV78XX0_SATAHC_BASE,MV78XX0_IRQ_SATA },
#endif

#if defined(DOVE)
#define DOVE_IRQ_TMR		(64 + MVSOC_MLMB_MLMBI_CPUTIMER0INTREQ)

    { DOVE(88AP510),	"mvsoctmr",0, MVSOC_TMR_BASE,	DOVE_IRQ_TMR },
    { DOVE(88AP510),	"mvsocpmu",0, DOVE_PMU_BASE,	DOVE_IRQ_PMU },
    { DOVE(88AP510),	"com",     0, MVSOC_COM0_BASE,	DOVE_IRQ_UART0 },
    { DOVE(88AP510),	"com",     1, MVSOC_COM1_BASE,	DOVE_IRQ_UART1 },
    { DOVE(88AP510),	"com",     2, DOVE_COM2_BASE,	DOVE_IRQ_UART2 },
    { DOVE(88AP510),	"com",     3, DOVE_COM3_BASE,	DOVE_IRQ_UART3 },
    { DOVE(88AP510),	"gttwsi",  0, MVSOC_TWSI_BASE,	DOVE_IRQ_TWSI },
    { DOVE(88AP510),	"mvspi",   0, DOVE_SPI0_BASE,	DOVE_IRQ_SPI0 },
    { DOVE(88AP510),	"mvspi",   1, DOVE_SPI1_BASE,	DOVE_IRQ_SPI1 },
    { DOVE(88AP510),	"mvcesa",  0, DOVE_CESA_BASE,	DOVE_IRQ_SECURITYINT },
    { DOVE(88AP510),	"ehci",    0, DOVE_USB0_BASE,	DOVE_IRQ_USB0CNT },
    { DOVE(88AP510),	"ehci",    1, DOVE_USB1_BASE,	DOVE_IRQ_USB1CNT },
    { DOVE(88AP510),	"gtidmac", 0, DOVE_XORE_BASE,	IRQ_DEFAULT },
    { DOVE(88AP510),	"mvgbec",  0, DOVE_GBE_BASE,	IRQ_DEFAULT },
    { DOVE(88AP510),	"mvpex",   0, MVSOC_PEX_BASE,	DOVE_IRQ_PEX0_INT },
    { DOVE(88AP510),	"mvpex",   1, DOVE_PEX1_BASE,	DOVE_IRQ_PEX1_INT },
    { DOVE(88AP510),	"sdhc",    0, DOVE_SDHC0_BASE,	DOVE_IRQ_SD0 },
    { DOVE(88AP510),	"sdhc",    1, DOVE_SDHC1_BASE,	DOVE_IRQ_SD1 },
    { DOVE(88AP510),	"mvsata",  0, DOVE_SATAHC_BASE,	DOVE_IRQ_SATAINT },
//    { DOVE(88AP510),	"mvsocgpp",0, MVSOC_GPP_BASE,	IRQ_DEFAULT },
    { DOVE(88AP510),	"mvsocrtc",0, DOVE_RTC_BASE,	IRQ_DEFAULT },
#endif

#if defined(ARMADAXP)
    { ARMADAXP(MV78130), "mvsoctmr",0,MVSOC_TMR_BASE,	ARMADAXP_IRQ_TIMER0 },
    { ARMADAXP(MV78130), "com",    0, MVSOC_COM0_BASE,	ARMADAXP_IRQ_UART0 },
    { ARMADAXP(MV78130), "com",    1, MVSOC_COM1_BASE,	ARMADAXP_IRQ_UART1 },
    { ARMADAXP(MV78130), "com",    2, ARMADAXP_COM2_BASE,ARMADAXP_IRQ_UART2 },
    { ARMADAXP(MV78130), "com",    3, ARMADAXP_COM3_BASE,ARMADAXP_IRQ_UART3 },
    { ARMADAXP(MV78130), "mvsocrtc",0,ARMADAXP_RTC_BASE,ARMADAXP_IRQ_RTC },
    { ARMADAXP(MV78130), "gttwsi", 0, MVSOC_TWSI_BASE,	ARMADAXP_IRQ_TWSI0 },
    { ARMADAXP(MV78130), "gttwsi", 1, ARMADAXP_TWSI1_BASE,ARMADAXP_IRQ_TWSI1 },
    { ARMADAXP(MV78130), "gtidmac",0, ARMADAXP_XORE0_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78130), "gtidmac",1, ARMADAXP_XORE1_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78130), "mvsocts",0, ARMADAXP_TS_BASE,	ARMADAXP_IRQ_PMU },
    { ARMADAXP(MV78130), "ehci",   0, ARMADAXP_USB0_BASE,ARMADAXP_IRQ_USB0 },
    { ARMADAXP(MV78130), "ehci",   1, ARMADAXP_USB1_BASE,ARMADAXP_IRQ_USB1 },
    { ARMADAXP(MV78130), "ehci",   2, ARMADAXP_USB2_BASE,ARMADAXP_IRQ_USB2 },
    { ARMADAXP(MV78130), "mvpex",  0, MVSOC_PEX_BASE,	ARMADAXP_IRQ_PEX00 },
    { ARMADAXP(MV78130), "mvpex",  1, ARMADAXP_PEX01_BASE,ARMADAXP_IRQ_PEX01 },
    { ARMADAXP(MV78130), "mvpex",  2, ARMADAXP_PEX02_BASE,ARMADAXP_IRQ_PEX02 },
    { ARMADAXP(MV78130), "mvpex",  3, ARMADAXP_PEX03_BASE,ARMADAXP_IRQ_PEX03 },
    { ARMADAXP(MV78130), "mvsata", 0, ARMADAXP_SATAHC_BASE,ARMADAXP_IRQ_SATA0 },
    { ARMADAXP(MV78130), "mvspi",  0, ARMADAXP_SPI_BASE,ARMADAXP_IRQ_SPI },
    { ARMADAXP(MV78130), "mvsdio", 0, ARMADAXP_SDIO_BASE,ARMADAXP_IRQ_SDIO },
    { ARMADAXP(MV78130), "mvxpe", 0, ARMADAXP_GBE0_BASE,ARMADAXP_IRQ_GBE0_TH_RXTX },
#if NMVXPE > 0
    { ARMADAXP(MV78130), "mvxpbm", 0, MVA_OFFSET_DEFAULT,IRQ_DEFAULT },
    { ARMADAXP(MV78130), "mvxpe", 1, ARMADAXP_GBE1_BASE,ARMADAXP_IRQ_GBE1_TH_RXTX },
    { ARMADAXP(MV78130), "mvxpe", 2, ARMADAXP_GBE2_BASE,ARMADAXP_IRQ_GBE2_TH_RXTX },
#else
    { ARMADAXP(MV78130), "mvgbec", 1, ARMADAXP_GBE1_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78130), "mvgbec", 2, ARMADAXP_GBE2_BASE,IRQ_DEFAULT },
#endif
#if NMVXPSEC > 0
    { ARMADAXP(MV78130), "mvxpsec", 0, ARMADAXP_XPSEC0_BASE,ARMADAXP_IRQ_CESA0 },
    { ARMADAXP(MV78130), "mvxpsec", 1, ARMADAXP_XPSEC1_BASE,ARMADAXP_IRQ_CESA1 },
#else
    { ARMADAXP(MV78130), "mvcesa", 0, ARMADAXP_CESA0_BASE,ARMADAXP_IRQ_CESA0 },
    { ARMADAXP(MV78130), "mvcesa", 1, ARMADAXP_CESA1_BASE,ARMADAXP_IRQ_CESA1 },
#endif

    { ARMADAXP(MV78160), "mvsoctmr",0,MVSOC_TMR_BASE,	ARMADAXP_IRQ_TIMER0 },
    { ARMADAXP(MV78160), "com",    0, MVSOC_COM0_BASE,	ARMADAXP_IRQ_UART0 },
    { ARMADAXP(MV78160), "com",    1, MVSOC_COM1_BASE,	ARMADAXP_IRQ_UART1 },
    { ARMADAXP(MV78160), "com",    2, ARMADAXP_COM2_BASE,ARMADAXP_IRQ_UART2 },
    { ARMADAXP(MV78160), "com",    3, ARMADAXP_COM3_BASE,ARMADAXP_IRQ_UART3 },
    { ARMADAXP(MV78160), "mvsocrtc",0,ARMADAXP_RTC_BASE,ARMADAXP_IRQ_RTC },
    { ARMADAXP(MV78160), "gttwsi", 0, MVSOC_TWSI_BASE,	ARMADAXP_IRQ_TWSI0 },
    { ARMADAXP(MV78160), "gttwsi", 1, ARMADAXP_TWSI1_BASE,ARMADAXP_IRQ_TWSI1 },
    { ARMADAXP(MV78160), "gtidmac",0, ARMADAXP_XORE0_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78160), "gtidmac",1, ARMADAXP_XORE1_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78160), "mvsocts",0, ARMADAXP_TS_BASE,	ARMADAXP_IRQ_PMU },
    { ARMADAXP(MV78160), "ehci",   0, ARMADAXP_USB0_BASE,ARMADAXP_IRQ_USB0 },
    { ARMADAXP(MV78160), "ehci",   1, ARMADAXP_USB1_BASE,ARMADAXP_IRQ_USB1 },
    { ARMADAXP(MV78160), "ehci",   2, ARMADAXP_USB2_BASE,ARMADAXP_IRQ_USB2 },
    { ARMADAXP(MV78160), "mvpex",  0, MVSOC_PEX_BASE,	ARMADAXP_IRQ_PEX00 },
    { ARMADAXP(MV78160), "mvpex",  1, ARMADAXP_PEX01_BASE,ARMADAXP_IRQ_PEX01 },
    { ARMADAXP(MV78160), "mvpex",  2, ARMADAXP_PEX02_BASE,ARMADAXP_IRQ_PEX02 },
    { ARMADAXP(MV78160), "mvpex",  3, ARMADAXP_PEX03_BASE,ARMADAXP_IRQ_PEX03 },
    { ARMADAXP(MV78160), "mvpex",  4, ARMADAXP_PEX2_BASE,ARMADAXP_IRQ_PEX2 },
    { ARMADAXP(MV78160), "mvsata", 0, ARMADAXP_SATAHC_BASE,ARMADAXP_IRQ_SATA0 },
    { ARMADAXP(MV78160), "mvspi",  0, ARMADAXP_SPI_BASE,ARMADAXP_IRQ_SPI },
    { ARMADAXP(MV78160), "mvsdio", 0, ARMADAXP_SDIO_BASE,ARMADAXP_IRQ_SDIO },
#if NMVXPE > 0
    { ARMADAXP(MV78160), "mvxpbm", 0, MVA_OFFSET_DEFAULT,IRQ_DEFAULT },
    { ARMADAXP(MV78160), "mvxpe", 0, ARMADAXP_GBE0_BASE,ARMADAXP_IRQ_GBE0_TH_RXTX },
    { ARMADAXP(MV78160), "mvxpe", 1, ARMADAXP_GBE1_BASE,ARMADAXP_IRQ_GBE1_TH_RXTX },
    { ARMADAXP(MV78160), "mvxpe", 2, ARMADAXP_GBE2_BASE,ARMADAXP_IRQ_GBE2_TH_RXTX },
    { ARMADAXP(MV78160), "mvxpe", 3, ARMADAXP_GBE3_BASE,ARMADAXP_IRQ_GBE3_TH_RXTX },
#else
    { ARMADAXP(MV78160), "mvgbec", 0, ARMADAXP_GBE0_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78160), "mvgbec", 1, ARMADAXP_GBE1_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78160), "mvgbec", 2, ARMADAXP_GBE2_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78160), "mvgbec", 3, ARMADAXP_GBE3_BASE,IRQ_DEFAULT },
#endif
#if NMVXPSEC > 0
    { ARMADAXP(MV78160), "mvxpsec", 0, ARMADAXP_XPSEC0_BASE,ARMADAXP_IRQ_CESA0 },
    { ARMADAXP(MV78160), "mvxpsec", 1, ARMADAXP_XPSEC1_BASE,ARMADAXP_IRQ_CESA1 },
#else
    { ARMADAXP(MV78160), "mvcesa", 0, ARMADAXP_CESA0_BASE,ARMADAXP_IRQ_CESA0 },
    { ARMADAXP(MV78160), "mvcesa", 1, ARMADAXP_CESA1_BASE,ARMADAXP_IRQ_CESA1 },
#endif

    { ARMADAXP(MV78230), "mvsoctmr",0,MVSOC_TMR_BASE,	ARMADAXP_IRQ_TIMER0 },
    { ARMADAXP(MV78230), "com",    0, MVSOC_COM0_BASE,	ARMADAXP_IRQ_UART0 },
    { ARMADAXP(MV78230), "com",    1, MVSOC_COM1_BASE,	ARMADAXP_IRQ_UART1 },
    { ARMADAXP(MV78230), "com",    2, ARMADAXP_COM2_BASE,ARMADAXP_IRQ_UART2 },
    { ARMADAXP(MV78230), "com",    3, ARMADAXP_COM3_BASE,ARMADAXP_IRQ_UART3 },
    { ARMADAXP(MV78230), "mvsocrtc",0,ARMADAXP_RTC_BASE,ARMADAXP_IRQ_RTC },
    { ARMADAXP(MV78230), "gttwsi", 0, MVSOC_TWSI_BASE,	ARMADAXP_IRQ_TWSI0 },
    { ARMADAXP(MV78230), "gttwsi", 1, ARMADAXP_TWSI1_BASE,ARMADAXP_IRQ_TWSI1 },
    { ARMADAXP(MV78230), "gtidmac",0, ARMADAXP_XORE0_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78230), "gtidmac",1, ARMADAXP_XORE1_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78230), "mvsocts",0, ARMADAXP_TS_BASE,	ARMADAXP_IRQ_PMU },
    { ARMADAXP(MV78230), "ehci",   0, ARMADAXP_USB0_BASE,ARMADAXP_IRQ_USB0 },
    { ARMADAXP(MV78230), "ehci",   1, ARMADAXP_USB1_BASE,ARMADAXP_IRQ_USB1 },
    { ARMADAXP(MV78230), "ehci",   2, ARMADAXP_USB2_BASE,ARMADAXP_IRQ_USB2 },
    { ARMADAXP(MV78230), "mvpex",  0, MVSOC_PEX_BASE,	ARMADAXP_IRQ_PEX00 },
    { ARMADAXP(MV78230), "mvpex",  1, ARMADAXP_PEX01_BASE,ARMADAXP_IRQ_PEX01 },
    { ARMADAXP(MV78230), "mvpex",  2, ARMADAXP_PEX02_BASE,ARMADAXP_IRQ_PEX02 },
    { ARMADAXP(MV78230), "mvpex",  3, ARMADAXP_PEX03_BASE,ARMADAXP_IRQ_PEX03 },
    { ARMADAXP(MV78230), "mvpex",  4, ARMADAXP_PEX10_BASE,ARMADAXP_IRQ_PEX10 },
    { ARMADAXP(MV78230), "mvpex",  5, ARMADAXP_PEX2_BASE,ARMADAXP_IRQ_PEX2 },
    { ARMADAXP(MV78230), "mvpex",  6, ARMADAXP_PEX3_BASE,ARMADAXP_IRQ_PEX3 },
    { ARMADAXP(MV78230), "mvsata", 0, ARMADAXP_SATAHC_BASE,ARMADAXP_IRQ_SATA0 },
    { ARMADAXP(MV78230), "mvspi",  0, ARMADAXP_SPI_BASE,ARMADAXP_IRQ_SPI },
    { ARMADAXP(MV78230), "mvsdio", 0, ARMADAXP_SDIO_BASE,ARMADAXP_IRQ_SDIO },
#if NMVXPE > 0
    { ARMADAXP(MV78230), "mvxpbm", 0, MVA_OFFSET_DEFAULT,IRQ_DEFAULT },
    { ARMADAXP(MV78230), "mvxpe", 0, ARMADAXP_GBE0_BASE,ARMADAXP_IRQ_GBE0_TH_RXTX },
    { ARMADAXP(MV78230), "mvxpe", 1, ARMADAXP_GBE1_BASE,ARMADAXP_IRQ_GBE1_TH_RXTX },
    { ARMADAXP(MV78230), "mvxpe", 2, ARMADAXP_GBE2_BASE,ARMADAXP_IRQ_GBE2_TH_RXTX },
#else
    { ARMADAXP(MV78230), "mvgbec", 0, ARMADAXP_GBE0_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78230), "mvgbec", 1, ARMADAXP_GBE1_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78230), "mvgbec", 2, ARMADAXP_GBE2_BASE,IRQ_DEFAULT },
#endif
#if NMVXPSEC > 0
    { ARMADAXP(MV78230), "mvxpsec", 0, ARMADAXP_XPSEC0_BASE,ARMADAXP_IRQ_CESA0 },
    { ARMADAXP(MV78230), "mvxpsec", 1, ARMADAXP_XPSEC1_BASE,ARMADAXP_IRQ_CESA1 },
#else
    { ARMADAXP(MV78230), "mvcesa", 0, ARMADAXP_CESA0_BASE,ARMADAXP_IRQ_CESA0 },
    { ARMADAXP(MV78230), "mvcesa", 1, ARMADAXP_CESA1_BASE,ARMADAXP_IRQ_CESA1 },
#endif

    { ARMADAXP(MV78260), "mvsoctmr",0,MVSOC_TMR_BASE,	ARMADAXP_IRQ_TIMER0 },
    { ARMADAXP(MV78260), "com",    0, MVSOC_COM0_BASE,	ARMADAXP_IRQ_UART0 },
    { ARMADAXP(MV78260), "com",    1, MVSOC_COM1_BASE,	ARMADAXP_IRQ_UART1 },
    { ARMADAXP(MV78260), "com",    2, ARMADAXP_COM2_BASE,ARMADAXP_IRQ_UART2 },
    { ARMADAXP(MV78260), "com",    3, ARMADAXP_COM3_BASE,ARMADAXP_IRQ_UART3 },
    { ARMADAXP(MV78260), "mvsocrtc",0,ARMADAXP_RTC_BASE,ARMADAXP_IRQ_RTC },
    { ARMADAXP(MV78260), "gttwsi", 0, MVSOC_TWSI_BASE,	ARMADAXP_IRQ_TWSI0 },
    { ARMADAXP(MV78260), "gttwsi", 1, ARMADAXP_TWSI1_BASE,ARMADAXP_IRQ_TWSI1 },
    { ARMADAXP(MV78260), "gtidmac",0, ARMADAXP_XORE0_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78260), "gtidmac",1, ARMADAXP_XORE1_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78260), "mvsocts",0, ARMADAXP_TS_BASE,	ARMADAXP_IRQ_PMU },
    { ARMADAXP(MV78260), "ehci",   0, ARMADAXP_USB0_BASE,ARMADAXP_IRQ_USB0 },
    { ARMADAXP(MV78260), "ehci",   1, ARMADAXP_USB1_BASE,ARMADAXP_IRQ_USB1 },
    { ARMADAXP(MV78260), "ehci",   2, ARMADAXP_USB2_BASE,ARMADAXP_IRQ_USB2 },
    { ARMADAXP(MV78260), "mvpex",  0, MVSOC_PEX_BASE,	ARMADAXP_IRQ_PEX00 },
    { ARMADAXP(MV78260), "mvpex",  1, ARMADAXP_PEX01_BASE,ARMADAXP_IRQ_PEX01 },
    { ARMADAXP(MV78260), "mvpex",  2, ARMADAXP_PEX02_BASE,ARMADAXP_IRQ_PEX02 },
    { ARMADAXP(MV78260), "mvpex",  3, ARMADAXP_PEX03_BASE,ARMADAXP_IRQ_PEX03 },
    { ARMADAXP(MV78260), "mvpex",  4, ARMADAXP_PEX2_BASE,ARMADAXP_IRQ_PEX2 },
    { ARMADAXP(MV78260), "mvsata", 0, ARMADAXP_SATAHC_BASE,ARMADAXP_IRQ_SATA0 },
    { ARMADAXP(MV78260), "mvspi",  0, ARMADAXP_SPI_BASE,ARMADAXP_IRQ_SPI },
    { ARMADAXP(MV78260), "mvsdio", 0, ARMADAXP_SDIO_BASE,ARMADAXP_IRQ_SDIO },
#if NMVXPE > 0
    { ARMADAXP(MV78260), "mvxpbm", 0, MVA_OFFSET_DEFAULT,IRQ_DEFAULT },
    { ARMADAXP(MV78260), "mvxpe", 0, ARMADAXP_GBE0_BASE,ARMADAXP_IRQ_GBE0_TH_RXTX },
    { ARMADAXP(MV78260), "mvxpe", 1, ARMADAXP_GBE1_BASE,ARMADAXP_IRQ_GBE1_TH_RXTX },
    { ARMADAXP(MV78260), "mvxpe", 2, ARMADAXP_GBE2_BASE,ARMADAXP_IRQ_GBE2_TH_RXTX },
    { ARMADAXP(MV78260), "mvxpe", 3, ARMADAXP_GBE3_BASE,ARMADAXP_IRQ_GBE3_TH_RXTX },
#else
    { ARMADAXP(MV78260), "mvgbec", 0, ARMADAXP_GBE0_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78260), "mvgbec", 1, ARMADAXP_GBE1_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78260), "mvgbec", 2, ARMADAXP_GBE2_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78260), "mvgbec", 3, ARMADAXP_GBE3_BASE,IRQ_DEFAULT },
#endif
#if NMVXPSEC > 0
    { ARMADAXP(MV78260), "mvxpsec", 0, ARMADAXP_XPSEC0_BASE,ARMADAXP_IRQ_CESA0 },
    { ARMADAXP(MV78260), "mvxpsec", 1, ARMADAXP_XPSEC1_BASE,ARMADAXP_IRQ_CESA1 },
#else
    { ARMADAXP(MV78260), "mvcesa", 0, ARMADAXP_CESA0_BASE,ARMADAXP_IRQ_CESA0 },
    { ARMADAXP(MV78260), "mvcesa", 1, ARMADAXP_CESA1_BASE,ARMADAXP_IRQ_CESA1 },
#endif

    { ARMADAXP(MV78460), "mvsoctmr",0,MVSOC_TMR_BASE,	ARMADAXP_IRQ_TIMER0 },
    { ARMADAXP(MV78460), "com",    0, MVSOC_COM0_BASE,	ARMADAXP_IRQ_UART0 },
    { ARMADAXP(MV78460), "com",    1, MVSOC_COM1_BASE,	ARMADAXP_IRQ_UART1 },
    { ARMADAXP(MV78460), "com",    2, ARMADAXP_COM2_BASE,ARMADAXP_IRQ_UART2 },
    { ARMADAXP(MV78460), "com",    3, ARMADAXP_COM3_BASE,ARMADAXP_IRQ_UART3 },
    { ARMADAXP(MV78460), "mvsocrtc",0,ARMADAXP_RTC_BASE,ARMADAXP_IRQ_RTC },
    { ARMADAXP(MV78460), "gttwsi", 0, MVSOC_TWSI_BASE,	ARMADAXP_IRQ_TWSI0 },
    { ARMADAXP(MV78460), "gttwsi", 1, ARMADAXP_TWSI1_BASE,ARMADAXP_IRQ_TWSI1 },
    { ARMADAXP(MV78460), "gtidmac",0, ARMADAXP_XORE0_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78460), "gtidmac",1, ARMADAXP_XORE1_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78460), "mvsocts",0, ARMADAXP_TS_BASE,	ARMADAXP_IRQ_PMU },
    { ARMADAXP(MV78460), "ehci",   0, ARMADAXP_USB0_BASE,ARMADAXP_IRQ_USB0 },
    { ARMADAXP(MV78460), "ehci",   1, ARMADAXP_USB1_BASE,ARMADAXP_IRQ_USB1 },
    { ARMADAXP(MV78460), "ehci",   2, ARMADAXP_USB2_BASE,ARMADAXP_IRQ_USB2 },
    { ARMADAXP(MV78460), "mvpex",  0, MVSOC_PEX_BASE,	ARMADAXP_IRQ_PEX00 },
    { ARMADAXP(MV78460), "mvpex",  1, ARMADAXP_PEX01_BASE,ARMADAXP_IRQ_PEX01 },
    { ARMADAXP(MV78460), "mvpex",  2, ARMADAXP_PEX02_BASE,ARMADAXP_IRQ_PEX02 },
    { ARMADAXP(MV78460), "mvpex",  3, ARMADAXP_PEX03_BASE,ARMADAXP_IRQ_PEX03 },
    { ARMADAXP(MV78460), "mvpex",  4, ARMADAXP_PEX2_BASE,ARMADAXP_IRQ_PEX2 },
    { ARMADAXP(MV78460), "mvpex",  5, ARMADAXP_PEX3_BASE,ARMADAXP_IRQ_PEX3 },
    { ARMADAXP(MV78460), "mvsata", 0, ARMADAXP_SATAHC_BASE,ARMADAXP_IRQ_SATA0 },
    { ARMADAXP(MV78460), "mvspi",  0, ARMADAXP_SPI_BASE,ARMADAXP_IRQ_SPI },
    { ARMADAXP(MV78460), "mvsdio", 0, ARMADAXP_SDIO_BASE,ARMADAXP_IRQ_SDIO },
#if NMVXPE > 0
    { ARMADAXP(MV78460), "mvxpbm", 0, MVA_OFFSET_DEFAULT,IRQ_DEFAULT },
    { ARMADAXP(MV78460), "mvxpe", 0, ARMADAXP_GBE0_BASE,ARMADAXP_IRQ_GBE0_TH_RXTX },
    { ARMADAXP(MV78460), "mvxpe", 1, ARMADAXP_GBE1_BASE,ARMADAXP_IRQ_GBE1_TH_RXTX },
    { ARMADAXP(MV78460), "mvxpe", 2, ARMADAXP_GBE2_BASE,ARMADAXP_IRQ_GBE2_TH_RXTX },
    { ARMADAXP(MV78460), "mvxpe", 3, ARMADAXP_GBE3_BASE,ARMADAXP_IRQ_GBE3_TH_RXTX },
#else
    { ARMADAXP(MV78460), "mvgbec", 0, ARMADAXP_GBE0_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78460), "mvgbec", 1, ARMADAXP_GBE1_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78460), "mvgbec", 2, ARMADAXP_GBE2_BASE,IRQ_DEFAULT },
    { ARMADAXP(MV78460), "mvgbec", 3, ARMADAXP_GBE3_BASE,IRQ_DEFAULT },
#endif
#if NMVXPSEC > 0
    { ARMADAXP(MV78460), "mvxpsec", 0, ARMADAXP_XPSEC0_BASE,ARMADAXP_IRQ_CESA0 },
    { ARMADAXP(MV78460), "mvxpsec", 1, ARMADAXP_XPSEC1_BASE,ARMADAXP_IRQ_CESA1 },
#else
    { ARMADAXP(MV78460), "mvcesa", 0, ARMADAXP_CESA0_BASE,ARMADAXP_IRQ_CESA0 },
    { ARMADAXP(MV78460), "mvcesa", 1, ARMADAXP_CESA1_BASE,ARMADAXP_IRQ_CESA1 },
#endif

    { ARMADA370(MV6710), "mvsoctmr",0,MVSOC_TMR_BASE,	ARMADAXP_IRQ_TIMER0 },
    { ARMADA370(MV6710), "com",    0, MVSOC_COM0_BASE,	ARMADAXP_IRQ_UART0 },
    { ARMADA370(MV6710), "com",    1, MVSOC_COM1_BASE,	ARMADAXP_IRQ_UART1 },
    { ARMADA370(MV6710), "mvsocrtc",0,ARMADAXP_RTC_BASE,ARMADAXP_IRQ_RTC },
    { ARMADA370(MV6710), "gttwsi", 0, MVSOC_TWSI_BASE,	ARMADAXP_IRQ_TWSI0 },
    { ARMADA370(MV6710), "gttwsi", 1, ARMADAXP_TWSI1_BASE,ARMADAXP_IRQ_TWSI1 },
    { ARMADA370(MV6710), "gtidmac",0, ARMADAXP_XORE0_BASE,IRQ_DEFAULT },
    { ARMADA370(MV6710), "ehci",   0, ARMADAXP_USB0_BASE,ARMADAXP_IRQ_USB0 },
    { ARMADA370(MV6710), "ehci",   1, ARMADAXP_USB1_BASE,ARMADAXP_IRQ_USB1 },
    { ARMADA370(MV6710), "mvpex",  0, MVSOC_PEX_BASE,	ARMADAXP_IRQ_PEX00 },
    { ARMADA370(MV6710), "mvpex",  1, ARMADAXP_PEX01_BASE,ARMADAXP_IRQ_PEX01 },
    { ARMADA370(MV6710), "mvsata", 0, ARMADAXP_SATAHC_BASE,ARMADAXP_IRQ_SATA0 },
    { ARMADA370(MV6710), "mvspi",  0, ARMADAXP_SPI_BASE,ARMADAXP_IRQ_SPI },
    { ARMADA370(MV6710), "mvspi",  1, ARMADAXP_SPI_BASE,ARMADAXP_IRQ_SPI },
    { ARMADA370(MV6710), "mvsdio", 0, ARMADAXP_SDIO_BASE,ARMADAXP_IRQ_SDIO },
#if NMVXPE > 0
    { ARMADA370(MV6710), "mvxpbm", 0, MVA_OFFSET_DEFAULT,IRQ_DEFAULT },
    { ARMADA370(MV6710), "mvxpe", 0, ARMADAXP_GBE0_BASE,ARMADAXP_IRQ_GBE0_TH_RXTX },
    { ARMADA370(MV6710), "mvxpe", 1, ARMADAXP_GBE1_BASE,ARMADAXP_IRQ_GBE1_TH_RXTX },
#else
    { ARMADA370(MV6710), "mvgbec", 0, ARMADAXP_GBE0_BASE,IRQ_DEFAULT },
    { ARMADA370(MV6710), "mvgbec", 1, ARMADAXP_GBE1_BASE,IRQ_DEFAULT },
#endif
#if NMVXPSEC > 0
    { ARMADA370(MV6710), "mvxpsec", 0, ARMADAXP_XPSEC0_BASE,ARMADAXP_IRQ_CESA0 },
#else
    { ARMADA370(MV6710), "mvcesa", 0, ARMADAXP_CESA0_BASE,ARMADAXP_IRQ_CESA0 },
#endif
#endif
};


CFATTACH_DECL_NEW(mvsoc, sizeof(struct mvsoc_softc),
    mvsoc_match, mvsoc_attach, NULL, NULL);

/* ARGSUSED */
static int
mvsoc_match(device_t parent, struct cfdata *match, void *aux)
{

	return 1;
}

/* ARGSUSED */
static void
mvsoc_attach(device_t parent, device_t self, void *aux)
{
	struct mvsoc_softc *sc = device_private(self);
	struct marvell_attach_args mva;
	enum marvell_tags *tags;
	uint16_t model;
	uint8_t rev;
	int i;

	sc->sc_dev = self;
	sc->sc_iot = &mvsoc_bs_tag;
	sc->sc_addr = vtophys(regbase);
	sc->sc_dmat = &mvsoc_bus_dma_tag;
	if (bus_space_map(sc->sc_iot, sc->sc_addr, 0x100000, 0, &sc->sc_ioh) !=
	    0) {
		aprint_error_dev(self, "can't map registers\n");
		return;
	}

	model = mvsoc_model();
	rev = mvsoc_rev();
	for (i = 0; i < __arraycount(nametbl); i++)
		if (nametbl[i].model == model && nametbl[i].rev == rev)
			break;
	if (i >= __arraycount(nametbl))
		panic("unknown SoC: model 0x%04x, rev 0x%02x", model, rev);

	aprint_normal(": Marvell %s %s%s  %s\n",
	    nametbl[i].modelstr,
	    nametbl[i].revstr != NULL ? "Rev. " : "",
	    nametbl[i].revstr != NULL ? nametbl[i].revstr : "",
	    nametbl[i].typestr);
        aprint_normal("%s: CPU Clock %d.%03d MHz"
	    "  SysClock %d.%03d MHz  TClock %d.%03d MHz\n",
	    device_xname(self),
	    mvPclk / 1000000, (mvPclk / 1000) % 1000,
	    mvSysclk / 1000000, (mvSysclk / 1000) % 1000,
	    mvTclk / 1000000, (mvTclk / 1000) % 1000);
	aprint_naive("\n");

	mvsoc_intr_init();

	for (i = 0; i < __arraycount(tagstbl); i++)
		if (tagstbl[i].model == model && tagstbl[i].rev == rev)
			break;
	if (i >= __arraycount(tagstbl))
		panic("unknown SoC: model 0x%04x, rev 0x%02x", model, rev);
	tags = tagstbl[i].tags;

	if (boothowto & (AB_VERBOSE | AB_DEBUG))
		mvsoc_target_dump(sc);

	for (i = 0; i < __arraycount(mvsoc_periphs); i++) {
		if (mvsoc_periphs[i].model != model)
			continue;

		mva.mva_name = mvsoc_periphs[i].name;
		mva.mva_model = model;
		mva.mva_revision = rev;
		mva.mva_iot = sc->sc_iot;
		mva.mva_ioh = sc->sc_ioh;
		mva.mva_unit = mvsoc_periphs[i].unit;
		mva.mva_addr = sc->sc_addr;
		mva.mva_offset = mvsoc_periphs[i].offset;
		mva.mva_size = 0;
		mva.mva_dmat = sc->sc_dmat;
		mva.mva_irq = mvsoc_periphs[i].irq;
		mva.mva_tags = tags;

		/* Skip clock disabled devices */
		if (mvsoc_clkgating != NULL && mvsoc_clkgating(&mva)) {
			aprint_normal_dev(self, "%s%d clock disabled\n",
			    mvsoc_periphs[i].name, mvsoc_periphs[i].unit);
			continue;
		}

		config_found_sm_loc(sc->sc_dev, "mvsoc", NULL, &mva,
		    mvsoc_print, mvsoc_search);
	}
}

static int
mvsoc_print(void *aux, const char *pnp)
{
	struct marvell_attach_args *mva = aux;

	if (pnp)
		aprint_normal("%s at %s unit %d",
		    mva->mva_name, pnp, mva->mva_unit);
	else {
		if (mva->mva_unit != MVA_UNIT_DEFAULT)
			aprint_normal(" unit %d", mva->mva_unit);
		if (mva->mva_offset != MVA_OFFSET_DEFAULT) {
			aprint_normal(" offset 0x%04lx", mva->mva_offset);
			if (mva->mva_size > 0)
				aprint_normal("-0x%04lx",
				    mva->mva_offset + mva->mva_size - 1);
		}
		if (mva->mva_irq != MVA_IRQ_DEFAULT)
			aprint_normal(" irq %d", mva->mva_irq);
	}

	return UNCONF;
}

/* ARGSUSED */
static int
mvsoc_search(device_t parent, cfdata_t cf, const int *ldesc, void *aux)
{

	return config_match(parent, cf, aux);
}

/* ARGSUSED */
int
marvell_winparams_by_tag(device_t dev, int tag, int *target, int *attribute,
			 uint64_t *base, uint32_t *size)
{
	uint32_t base32;
	int rv;

	rv = mvsoc_target(tag, target, attribute, &base32, size);
	*base = base32;
	if (rv == -1)
		return -1;
	return 0;
}


/*
 * These functions is called before bus_space is initialized.
 */

void
mvsoc_bootstrap(bus_addr_t iobase)
{

	regbase = iobase;
	dsc_base = iobase + MVSOC_DSC_BASE;
	mlmb_base = iobase + MVSOC_MLMB_BASE;
	pex_base = iobase + MVSOC_PEX_BASE;
#ifdef MVSOC_CONSOLE_EARLY
	com_base = iobase + MVSOC_COM0_BASE;
	cn_tab = &mvsoc_earlycons;
	printf("Hello\n");
#endif
}

/*
 * We can read register of PCI configurations from (MVSOC_PEX_BASE + 0).
 */
uint16_t
mvsoc_model(void)
{
	/*
	 * We read product-id from vendor/device register of PCI-Express.
	 */
	uint32_t reg;
	uint16_t model;

	KASSERT(regbase != 0xffffffff);

	reg = *(volatile uint32_t *)(pex_base + PCI_ID_REG);
	model = PCI_PRODUCT(reg);

#if defined(ORION)
	if (model == PCI_PRODUCT_MARVELL_88F5182) {
		reg = *(volatile uint32_t *)(regbase + ORION_PMI_BASE +
		    ORION_PMI_SAMPLE_AT_RESET);
		if ((reg & ORION_PMISMPL_TCLK_MASK) == 0)
			model = PCI_PRODUCT_MARVELL_88F5082;
	}
#endif
#if defined(KIRKWOOD)
	if (model == PCI_PRODUCT_MARVELL_88F6281) {
		reg = *(volatile uint32_t *)(regbase + KIRKWOOD_MISC_BASE +
		    KIRKWOOD_MISC_DEVICEID);
		if (reg == 1)	/* 88F6192 is 1 */
			model = MARVELL_KIRKWOOD_88F6192;
	}
#endif

	return model;
}

uint8_t
mvsoc_rev(void)
{
	uint32_t reg;
	uint8_t rev;

	KASSERT(regbase != 0xffffffff);

	reg = *(volatile uint32_t *)(pex_base + PCI_CLASS_REG);
	rev = PCI_REVISION(reg);

	return rev;
}


int
mvsoc_target(int tag, uint32_t *target, uint32_t *attr, uint32_t *base,
	     uint32_t *size)
{
	int i;

	KASSERT(regbase != 0xffffffff);

	if (tag == MVSOC_TAG_INTERNALREG) {
		if (target != NULL)
			*target = 0;
		if (attr != NULL)
			*attr = 0;
		if (base != NULL)
			*base = read_mlmbreg(MVSOC_MLMB_IRBAR) &
			    MVSOC_MLMB_IRBAR_BASE_MASK;
		if (size != NULL)
			*size = 0;

		return 0;
	}

	/* sanity check */
	for (i = 0; i < __arraycount(mvsoc_tags); i++)
		if (mvsoc_tags[i].tag == tag)
			break;
	if (i >= __arraycount(mvsoc_tags))
		return -1;

	if (target != NULL)
		*target = mvsoc_tags[i].target;
	if (attr != NULL)
		*attr = mvsoc_tags[i].attr;

	if (mvsoc_tags[i].target == MVSOC_UNITID_DDR) {
		if (tag == MARVELL_TAG_SDRAM_CS0 ||
		    tag == MARVELL_TAG_SDRAM_CS1 ||
		    tag == MARVELL_TAG_SDRAM_CS2 ||
		    tag == MARVELL_TAG_SDRAM_CS3)
			return mvsoc_target_ddr(mvsoc_tags[i].attr, base, size);
		else if (tag == MARVELL_TAG_AXI_CS0 ||
		    tag == MARVELL_TAG_AXI_CS1)
			return mvsoc_target_axi(tag, base, size);
		else
			return mvsoc_target_ddr3(mvsoc_tags[i].attr, base,
			    size);
	} else
		return mvsoc_target_peripheral(mvsoc_tags[i].target,
		    mvsoc_tags[i].attr, base, size);
}

static int
mvsoc_target_ddr(uint32_t attr, uint32_t *base, uint32_t *size)
{
	uint32_t baseaddrreg, sizereg;
	int cs;

	/*
	 * Read DDR SDRAM Controller Address Decode Registers
	 */

	switch (attr) {
	case MARVELL_ATTR_SDRAM_CS0:
		cs = 0;
		break;
	case MARVELL_ATTR_SDRAM_CS1:
		cs = 1;
		break;
	case MARVELL_ATTR_SDRAM_CS2:
		cs = 2;
		break;
	case MARVELL_ATTR_SDRAM_CS3:
		cs = 3;
		break;
	default:
		aprint_error("unknwon ATTR: 0x%x", attr);
		return -1;
	}
	sizereg = *(volatile uint32_t *)(dsc_base + MVSOC_DSC_CSSR(cs));
	if (sizereg & MVSOC_DSC_CSSR_WINEN) {
		baseaddrreg =
		    *(volatile uint32_t *)(dsc_base + MVSOC_DSC_CSBAR(cs));

		if (base != NULL)
			*base = baseaddrreg & MVSOC_DSC_CSBAR_BASE_MASK;
		if (size != NULL)
			*size = (sizereg & MVSOC_DSC_CSSR_SIZE_MASK) +
			    (~MVSOC_DSC_CSSR_SIZE_MASK + 1);
	} else {
		if (base != NULL)
			*base = 0;
		if (size != NULL)
			*size = 0;
	}
	return 0;
}

static int
mvsoc_target_ddr3(uint32_t attr, uint32_t *base, uint32_t *size)
{
	uint32_t baseaddrreg, sizereg;
	int cs, i;

	/*
	 * Read DDR3 SDRAM Address Decoding Registers
	 */

	switch (attr) {
	case MARVELL_ATTR_SDRAM_CS0:
		cs = 0;
		break;
	case MARVELL_ATTR_SDRAM_CS1:
		cs = 1;
		break;
	case MARVELL_ATTR_SDRAM_CS2:
		cs = 2;
		break;
	case MARVELL_ATTR_SDRAM_CS3:
		cs = 3;
		break;
	default:
		aprint_error("unknwon ATTR: 0x%x", attr);
		return -1;
	}
	for (i = 0; i < MVSOC_MLMB_NWIN; i++) {
		sizereg = read_mlmbreg(MVSOC_MLMB_WINCR(i));
		if ((sizereg & MVSOC_MLMB_WINCR_EN) &&
		    MVSOC_MLMB_WINCR_WINCS(sizereg) == cs)
			break;
	}
	if (i == MVSOC_MLMB_NWIN) {
		if (base != NULL)
			*base = 0;
		if (size != NULL)
			*size = 0;
		return 0;
	}

	baseaddrreg = read_mlmbreg(MVSOC_MLMB_WINBAR(i));
	if (base != NULL)
		*base = baseaddrreg & MVSOC_MLMB_WINBAR_BASE_MASK;
	if (size != NULL)
		*size = (sizereg & MVSOC_MLMB_WINCR_SIZE_MASK) +
		    (~MVSOC_MLMB_WINCR_SIZE_MASK + 1);
	return 0;
}

static int
mvsoc_target_axi(int tag, uint32_t *base, uint32_t *size)
{
	uint32_t val;
	int cs;

	/*
	 * Read MMAP1 Chip Select N the other side of AXI DDR Registers
	 */

	switch (tag) {
	case MARVELL_TAG_AXI_CS0:
		cs = 0;
		break;
	case MARVELL_TAG_AXI_CS1:
		cs = 1;
		break;
	default:
		aprint_error("unknwon TAG: 0x%x", tag);
		return -1;
	}
	val = *(volatile uint32_t *)(regbase + MVSOC_AXI_MMAP1(cs));
	if (val & MVSOC_AXI_MMAP1_VALID) {
		if (base != NULL)
			*base = MVSOC_AXI_MMAP1_STARTADDRESS(val);
		if (size != NULL)
			*size = MVSOC_AXI_MMAP1_AREALENGTH(val);
	} else {
		if (base != NULL)
			*base = 0;
		if (size != NULL)
			*size = 0;
	}
	return 0;
}

static int
mvsoc_target_peripheral(uint32_t target, uint32_t attr, uint32_t *base,
			uint32_t *size)
{
	uint32_t basereg, ctrlreg, ta, tamask;
	int i;

	/*
	 * Read CPU Address Map Registers
	 */

	ta = MVSOC_MLMB_WCR_TARGET(target) | MVSOC_MLMB_WCR_ATTR(attr);
	tamask = MVSOC_MLMB_WCR_TARGET(MVSOC_UNITID_MASK) |
	    MVSOC_MLMB_WCR_ATTR(MARVELL_ATTR_MASK);

	if (base != NULL)
		*base = 0;
	if (size != NULL)
		*size = 0;

	for (i = 0; i < nwindow; i++) {
		ctrlreg = read_mlmbreg(MVSOC_MLMB_WCR(i));
		if ((ctrlreg & tamask) != ta)
			continue;
		if (ctrlreg & MVSOC_MLMB_WCR_WINEN) {
			basereg = read_mlmbreg(MVSOC_MLMB_WBR(i));

			if (base != NULL)
				*base = basereg & MVSOC_MLMB_WBR_BASE_MASK;
			if (size != NULL)
				*size = (ctrlreg &
				    MVSOC_MLMB_WCR_SIZE_MASK) +
				    (~MVSOC_MLMB_WCR_SIZE_MASK + 1);
		}
		break;
	}
	return i;
}

int
mvsoc_target_dump(struct mvsoc_softc *sc)
{
	uint32_t reg, base, size, target, attr, enable;
	int i, n;

	for (i = 0, n = 0; i < nwindow; i++) {
		reg = read_mlmbreg(MVSOC_MLMB_WCR(i));
		enable = reg & MVSOC_MLMB_WCR_WINEN;
		target = MVSOC_MLMB_WCR_GET_TARGET(reg);
		attr = MVSOC_MLMB_WCR_GET_ATTR(reg);
		size = MVSOC_MLMB_WCR_GET_SIZE(reg);

		reg = read_mlmbreg(MVSOC_MLMB_WBR(i));
		base = MVSOC_MLMB_WBR_GET_BASE(reg);

		if (!enable)
			continue;

		aprint_verbose_dev(sc->sc_dev,
		    "Mbus window %2d: Base 0x%08x Size 0x%08x ", i, base, size);
#ifdef ARMADAXP
		armadaxp_attr_dump(sc, target, attr);
#else
		mvsoc_attr_dump(sc, target, attr);
#endif
		printf("\n");
		n++;
	}

	return n;
}

int
mvsoc_attr_dump(struct mvsoc_softc *sc, uint32_t target, uint32_t attr)
{
	aprint_verbose_dev(sc->sc_dev, "target 0x%x(attr 0x%x)", target, attr);
	return 0;
}
