/*
 * Copyright (c) 2023 Matthew R. Green
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * stupid module that will set bit 9 in the DE_CFG register when loaded, and
 * unset if it wasn't set upon unload.  skips cpus that report being non-SMT
 * ID 0.
 */

#include <sys/types.h>
#include <sys/module.h>
#include <sys/xcall.h>
#include <sys/cpu.h>

#include <machine/cpufunc.h>
#include <machine/specialreg.h>

MODULE(MODULE_CLASS_MISC, zenbleed, NULL);

#define        DE_CFG_ERRATA_ZENBLEED  __BIT(9)

static bool orig[MAXCPUS];

static void
zenbleed_checkset(void *v1, void *v2)
{
	uint64_t val;
	unsigned num = cpu_number();

	if (curcpu()->ci_data.cpu_smt_id != 0)
		return;

	val = rdmsr_locked(MSR_DE_CFG);
	if ((val & DE_CFG_ERRATA_ZENBLEED) == 0) {
		wrmsr_locked(MSR_DE_CFG, val | DE_CFG_ERRATA_ZENBLEED);
		printf("%s: cpu%d set DE_CFG_ERRATA_ZENBLEED bit\n", __func__, num);
	} else {
		printf("%s: cpu%d DE_CFG_ERRATA_ZENBLEED bit already set\n", __func__, num);
		orig[num] = true;
	}
}

static void
zenbleed_checkunset(void *v1, void *v2)
{
	uint64_t val;
	unsigned num = cpu_number();

	if (curcpu()->ci_data.cpu_smt_id != 0)
		return;

	val = rdmsr_locked(MSR_DE_CFG);
	if ((val & DE_CFG_ERRATA_ZENBLEED) != 0) {
		if (orig[num]) {
			printf("%s: cpu%d left DE_CFG_ERRATA_ZENBLEED bit enabled (as at modload)\n", __func__, num);
		} else {
			val &= ~DE_CFG_ERRATA_ZENBLEED;
			wrmsr_locked(MSR_DE_CFG, val);
			printf("%s: cpu%d unset DE_CFG_ERRATA_ZENBLEED bit\n", __func__, num);
		}
	} else {
		printf("%s: cpu%d DE_CFG_ERRATA_ZENBLEED bit already not set\n", __func__, num);
	}
}

static int
zenbleed_modcmd(modcmd_t cmd, void *arg)
{
	uint64_t w;

	switch (cmd) {
	case MODULE_CMD_INIT:
		w = xc_broadcast(0, zenbleed_checkset, 0, 0);
		xc_wait(w);
		return 0;

	case MODULE_CMD_FINI:
		w = xc_broadcast(0, zenbleed_checkunset, 0, 0);
		xc_wait(w);
		return 0;

	default:
		return ENOTTY;
	}
}
