#
# monitor.py - monitor probing and install data
#
# Mike Fulbright <msf@redhat.com>
#
# Copyright 2001-2002 Red Hat, Inc.
#
# This software may be freely redistributed under the terms of the GNU
# library public license.
#
# You should have received a copy of the GNU Library Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#

import string
import exceptions

from rhpl.executil import execWithCapture
from rhpl.translate import _
import rhpl.translate as translate

translate.textdomain('rhpxl')

class MonitorInfo:

#
#  This class represents the monitor on the system. Values from ddcprobing
#  are provided if available.  LCDs are not currently probed.
#
#  Internal members (use methods to access):
#
#     monEisa     - probed monitor ID  (string)
#     monName     - human readable description (string)
#     monID       - human readable ID (string)
#     monHoriz    - horizontal rating (kHz)
#     monVert     - vertical rating (Hz)
#
    def readMonitorsDB (self, lines = None):
        if self.monlist:
            return self.monlist
        if not lines:
            db = open ('/usr/share/hwdata/MonitorsDB')
            lines = db.readlines ()
            db.close ()

        for line in lines:
            line = string.strip (line)
            if not line:
                continue
            if line and line[0] == '#':
                continue
            fields = string.split (line, ';')
            if len(fields) < 5:
                # ignore bad entries in MonitorsDB
                continue
            man = string.strip(fields[0])
            model = string.strip(fields[1])
            eisa = string.lower(string.strip(fields[2]))
            horiz = string.strip(fields[3])
            vert = string.strip(fields[4])
            if self.monlist.has_key(man):
                self.monlist[man].append((model, eisa, vert, horiz))
            else:
                self.monlist[man] = [(model, eisa, vert, horiz)]
            self.monids[eisa] = (man, model, eisa, vert, horiz)
        return self.monlist

    def monitorsDB(self):
        if not self.monlist:
            self.readMonitorsDB()

        return self.monlist

    def lookupMonitorByID(self, monID):
        if not self.monlist:
            self.readMonitorsDB()

        for man in self.monlist.keys():
            for model in self.monlist[man]:
		idlower = string.lower(monID)
		idupper = string.upper(monID)
                if idlower == model[1] or idupper == model[1]:
                    return model

        return 0

    def lookupMonitorByName(self, monName):
        if not self.monlist:
            self.readMonitorsDB()

        for man in self.monlist.keys():
            for model in self.monlist[man]:
		if monName == model[0]:
                    return model

        return None


    def __str__ (self):
        return  "monName: %s\nmonID: %s\nmonHoriz: %s\nmonVert: %s\nphysicalWidth: %s\nphysicalHeight: %s\n" % ( self.monName, self.monID, self.monHoriz, self.monVert, self.monPhysicalWidth, self.monPhysicalHeight)

    def setSpecs(self, horiz, vert, id=None, name = None, physicalWidth=0, physicalHeight=0):
        self.monPhysicalWidth = physicalWidth
        self.monPhysicalHeight = physicalHeight
        self.monHoriz = horiz
        self.monVert = vert
        if id:
            self.monID = id
            
        if name:
            self.monName = name

    def getMonitorPhysicalWidth(self):
        return self.monPhysicalWidth

    def getMonitorPhysicalHeight(self):
        return self.monPhysicalHeight

    def getMonitorHorizSync(self):
        return self.monHoriz

    def getMonitorVertSync(self):
        return self.monVert

    def getMonitorID(self):
        return self.monID

    def getMonitorName(self):
	return self.monName

    def __init__ (self):
        self.monName = None
        self.monID = "Unprobed Monitor"

        self.monHoriz = None
        self.monVert = None

        self.monlist = {}
        self.monids = {}

        self.monPhysicalWidth = 0
        self.monPhysicalHeight = 0

SYNC_TOLERANCE = 0.01    # 1 percent 

class ModeLine:
    def __init__(self, elements):
        self.clock = string.atof(elements[2])
        self.hdisp = string.atof(elements[3])
        self.hsyncstart = string.atof(elements[4])
        self.hsyncend = string.atof(elements[5])
        self.htotal = string.atof(elements[6])
        self.vdisp = string.atof(elements[7])
        self.vsyncstart = string.atof(elements[8])
        self.vsyncend = string.atof(elements[9])
        self.vtotal = string.atof(elements[10])

        self.flags = []
        for i in range(11, len(elements)):
            self.flags.append(string.upper(elements[i]))

    # Bascically copied from xf86CheckModeForMonitor
    def supports(self, monitor_hsync, monitor_vsync):
        hsync = self.clock * 1000 / self.htotal
        hsync_ok = 0
        for i in range(len(monitor_hsync)):
            if hsync > monitor_hsync[i][0] * (1.0 - SYNC_TOLERANCE) and hsync < monitor_hsync[i][1] * (1.0 + SYNC_TOLERANCE):
                hsync_ok = 1
                break;
        if not hsync_ok:
            return 0

        vrefresh = self.clock * 1000000.0 / (self.htotal * self.vtotal)
        if "INTERLACE" in self.flags:
            vrefresh = vrefresh * 2.0;
        if "DBLSCAN" in self.flags:
            vrefresh = vrefresh / 2.0;

        vsync_ok = 0
        for i in range(len(monitor_vsync)):
            if vrefresh > monitor_vsync[i][0] * (1.0 - SYNC_TOLERANCE) and vrefresh < monitor_vsync[i][1] * (1.0 + SYNC_TOLERANCE):
                vsync_ok = 1
                break;
        
        return vsync_ok

class Modes:
    def __init__(self):
        self.modelines = {}

        self.read_modes("vesamodes")
        self.read_modes("extramodes")

    def read_modes(self, filename):
        try:
            fd = open("/usr/share/xorg/" + filename, 'r')
        except IOError:
            fd = open("/usr/share/rhpxl/" + filename, 'r')

        lines = fd.readlines()
        fd.close()

        for line in lines:
            if line[0] != "#" and line[0] != '/':
                line = string.strip(line)
                elements = string.split(line)

                if line == "":
                    continue
                
                if len(elements) < 11 or string.lower(elements[0]) != "modeline":
                    raise exceptions.StandardError("Invalid modeline in file: %s"%(line))

                name = elements[1][1:-1]
                if self.modelines.has_key(name):
                    self.modelines[name].append(ModeLine(elements))
                else:
                    self.modelines[name] = [ModeLine(elements)]

    def monitor_supports_mode (self, monitor_hsync, monitor_vsync, modename):
        def list_from_string(str):
            l = []
            pieces = string.split(str, ",")
            for piece in pieces:
                tmp = string.split(piece, "-")
                if len(tmp) == 1:
                    l.append( (string.atof(tmp[0]), string.atof(tmp[0])) )
                else:
                    l.append( (string.atof(tmp[0]), string.atof(tmp[1])) )
            return l

        if not self.modelines.has_key(modename):
            return -1

        # believe randr before believing our sync range guesses
        try:
            import _pyrandr
        except ImportError:
            _pyrandr = None

        if _pyrandr and _pyrandr.randrAvailable():
            for num in range(0, _pyrandr.getNumRes()):
                res = "%sx%s" % _pyrandr.getRes(num)
                if res == modename:
                    return 1
        
        hsync_list = list_from_string (monitor_hsync)
        vsync_list = list_from_string (monitor_vsync)

        for modeline in self.modelines[modename]:
            if modeline.supports(hsync_list, vsync_list):
                return 1
        return 0
