# videocard.py - Install data and probing for video cards
#
# Matt Wilson <msw@redhat.com>
# Brent Fox <bfox@redhat.com>
# Mike Fulbright <msf@redhat.com>
# Bill Nottingham <notting@redhat.com>
# Adam Jackson <ajax@redhat.com>
#
# Copyright 2001-2008 Red Hat, Inc.
#
# This software may be freely redistributed under the terms of the GNU
# library public license.
#
# You should have received a copy of the GNU Library Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

import dbus
import string
import os

from rhpl.executil import execWithCapture
from rhpl.translate import _
import rhpl.translate as translate
import rhpl.alias as alias

translate.textdomain('rhpxl')

class VideoCard:

# This class represents the base data about a videocard. These are
# the internal members - PLEASE use methods to access values!
#
#    device   - if framebuffer running on card this is entry in /dev (string)
#    descr    - human readable description (string)
#    driver   - X driver
#    vidRam   - amount of video ram (in kB) (string)
#    pcibus   - PCI bus number of the card
#    pcidev   - PCI device number of the card
#    pcifn    - PCI fn number of the card
#
# These values will be None if undefined or not applicable.

    def __str__ (self):
        return "device: %s\ndriver : %s\ndescr : %s\nvidRam: %s\n" % (self.device, self.driver, self.descr, self.vidRam)

    def __init__ (self):
        self.device = None
        self.driver = None
        self.descr  = None
        self.vidRam = None
        self.pcibus = None
        self.pcidev = None
        self.pcifn = None
        
    def setDevice(self, device):
        self.device = device

    def setDescription(self, descr):
        self.descr = descr

    def setDriver(self, card):
        self.driver = card

    def setVideoRam(self, vidRam):
        self.vidRam = vidRam

    def setPCIBus(self, num):
        self.pcibus = num

    def setPCIDev(self, num):
        self.pcidev = num

    def setPCIFn(self, num):
        self.pcifn = num
        
    def getDriver(self):
        return self.driver
    
    def getVideoRam(self):
        return self.vidRam

    def getDescription(self):
        return self.descr

    def getDevice(self):
        return self.device

    def getPCIBus(self):
        return self.pcibus

    def getPCIDev(self):
        return self.pcidev

    def getPCIFn(self):
        return self.pcifn

class VideoCardInfo:

# This class represents the video cards on the system.
#
# Currently we only care about the primary card on the system.
# This can be found by using the VideoCardInfo::primaryCard() function.
#
# NOTE - X configuration is not represented here. This class is
#        intended only to represent the available hardware on the system

    def primaryCard(self):
	if self.videocards and self.primary < len(self.videocards):
	    return self.videocards[self.primary]
	else:
	    return None

    def __str__(self):
        retstr = "primary: %s\nvidCards: %s\n" % (self.primary, self.videocards)
        if self.primaryCard():
            retstr = retstr + ("Primary Video Card Info:\n%s" % (str(self.primaryCard())))
        return retstr

    def __init__ (self, forceDriver = None):
        # just use first video card we recognize
        # store as a list of class VideoCard
        self.videocards = []
        self.primary = None

	aliases = alias.Alias()
	aliases.addAliasDir("/usr/share/hwdata/videoaliases", "*.xinf")
	try:
		bus = dbus.SystemBus()
		hal = dbus.Interface(bus.get_object('org.freedesktop.Hal',
					    '/org/freedesktop/Hal/Manager'),
					    'org.freedesktop.Hal.Manager')
		# in principle, this should be .FindDeviceByCapability('video')
		devs = hal.GetAllDevices()
		devs = [bus.get_object('org.freedesktop.Hal', x) for x in devs]
		devs = [dbus.Interface(x, 'org.freedesktop.Hal.Device') for x in devs]
	except dbus.exceptions.DBusException:
		devs = []

	pdevs = [x for x in devs if x.PropertyExists('pci.device_class')]
	pdevs = [x for x in pdevs if x.GetPropertyInteger('pci.device_class') == 3]

        for dev in pdevs:
	    driver = None
            
	    vc = VideoCard()
	    vc.setDescription(dev.GetPropertyString('info.vendor') + " " +
			      dev.GetPropertyString('info.product'))
	    path = dev.GetPropertyString('pci.linux.sysfs_path')
	    # last path component is bus address; tupleize it
	    path = os.path.basename(path).replace('.', ':').split(':')
	    # vc.setPCIDom(path[0])
	    vc.setPCIBus(path[1])
	    vc.setPCIDev(path[2])
	    vc.setPCIFn(path[3])

	    # bit of a hack to skip secondary devices.  thanks redmond.
	    if path[3] != '0':
		continue

            if forceDriver:
                driver = forceDriver
	    else:
		v = dev.GetPropertyInteger('pci.vendor_id')
		d = dev.GetPropertyInteger('pci.product_id')
		sv = dev.GetPropertyInteger('pci.subsys_vendor_id')
		sd = dev.GetPropertyInteger('pci.subsys_product_id')
		# XXX bc, sc, i.  except these never matter
		driver = aliases.matchDevice(v, d, sv, sd)
		if not driver:
		    driver = 'vesa'

	    vc.setDriver(driver)
            self.videocards.append(vc)

	# if we didn't find anything native, hunt for fbdev.
	if len(self.videocards) == 0:
	    if os.access('/proc/fb', os.F_OK):
		file = open('/proc/fb')
		for line in file:
		    (num, drv) = line.split(' ',1)
		    device = '/dev/fb%s' % num
		    if not os.access(device, os.F_OK):
			continue
		    vc = VideoCard()
		    vc.setDescription('Framebuffer %s' % num)
		    vc.setDevice(device)
		    vc.setDriver('fbdev')
		    self.videocards.append(vc)

	if len(self.videocards) > 0:
	    self.primary = 0
