/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.brooklyn.core.location.dynamic;

import org.apache.brooklyn.api.entity.Entity;
import org.apache.brooklyn.api.location.Location;
import org.apache.brooklyn.api.location.LocationDefinition;
import org.apache.brooklyn.api.location.LocationRegistry;
import org.apache.brooklyn.config.ConfigKey;
import org.apache.brooklyn.core.config.ConfigKeys;
import org.apache.brooklyn.util.core.flags.SetFromFlag;

import com.google.common.annotations.Beta;

/**
 * A location that is created and owned by an entity at runtime.
 * <p>
 * The lifecycle of the location is managed by the owning entity.
 *
 * @param E the entity type
 * @param L the location type
 */
@Beta
public interface DynamicLocation<E extends Entity & LocationOwner<L, E>, L extends Location & DynamicLocation<E, L>> {

    @SetFromFlag("owner")
    ConfigKey<Entity> OWNER =
            ConfigKeys.newConfigKey(Entity.class, "owner", "The entity owning this location");

    @SetFromFlag("maxLocations")
    ConfigKey<Integer> MAX_SUB_LOCATIONS =
            ConfigKeys.newIntegerConfigKey("maxLocations", "The maximum number of sub-locations that can be created; 0 for unlimited", 0);

    E getOwner();

    /**
     * An opportunity to register this location (e.g. with the {@link LocationRegistry} or the 
     * catalog, so that it will be persisted).
     */
    LocationDefinition register();
    
    /**
     * The complement of {@link #register()}, to deregister this location.
     */
    void deregister();
}
