/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.ml.feature.lsh;

import org.apache.flink.ml.util.ParamUtils;
import org.apache.flink.ml.util.ReadWriteUtils;
import org.apache.flink.streaming.api.datastream.DataStream;
import org.apache.flink.table.api.bridge.java.StreamTableEnvironment;

import java.io.IOException;

/**
 * An Estimator that implements the MinHash LSH algorithm, which supports LSH for Jaccard distance.
 *
 * <p>The input could be dense or sparse vectors. Each input vector must have at least one non-zero
 * index and all non-zero values are treated as binary "1" values. The sizes of input vectors should
 * be same and not larger than a predefined prime (i.e., 2038074743).
 *
 * <p>See: <a href="https://en.wikipedia.org/wiki/MinHash">MinHash</a>.
 */
public class MinHashLSH extends LSH<MinHashLSH, MinHashLSHModel>
        implements MinHashLSHParams<MinHashLSH> {

    @Override
    protected MinHashLSHModel createModel(
            DataStream<Integer> inputDim, StreamTableEnvironment tEnv) {
        DataStream<MinHashLSHModelData> modelData =
                inputDim.map(
                        dim ->
                                MinHashLSHModelData.generateModelData(
                                        getNumHashTables(),
                                        getNumHashFunctionsPerTable(),
                                        dim,
                                        getSeed()));
        MinHashLSHModel model = new MinHashLSHModel().setModelData(tEnv.fromDataStream(modelData));
        ParamUtils.updateExistingParams(model, getParamMap());
        return model;
    }

    public static MinHashLSH load(StreamTableEnvironment tEnv, String path) throws IOException {
        return ReadWriteUtils.loadStageParam(path);
    }
}
