/*
 * Decompiled with CFR 0.152.
 */
package org.apache.gobblin.yarn;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Optional;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import com.google.common.io.Closer;
import com.google.common.util.concurrent.ServiceManager;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import java.io.Closeable;
import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.mail.EmailException;
import org.apache.gobblin.cluster.GobblinClusterUtils;
import org.apache.gobblin.cluster.HelixUtils;
import org.apache.gobblin.configuration.State;
import org.apache.gobblin.rest.JobExecutionInfoServer;
import org.apache.gobblin.runtime.app.ServiceBasedAppLauncher;
import org.apache.gobblin.util.ConfigUtils;
import org.apache.gobblin.util.EmailUtils;
import org.apache.gobblin.util.ExecutorsUtils;
import org.apache.gobblin.util.JvmUtils;
import org.apache.gobblin.util.io.StreamUtils;
import org.apache.gobblin.util.logs.LogCopier;
import org.apache.gobblin.yarn.GobblinApplicationMaster;
import org.apache.gobblin.yarn.HelixMessageSubTypes;
import org.apache.gobblin.yarn.YarnAppSecurityManager;
import org.apache.gobblin.yarn.YarnHelixUtils;
import org.apache.gobblin.yarn.event.ApplicationReportArrivalEvent;
import org.apache.gobblin.yarn.event.GetApplicationReportFailureEvent;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RawLocalFileSystem;
import org.apache.hadoop.io.DataOutputBuffer;
import org.apache.hadoop.security.Credentials;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.yarn.api.ApplicationConstants;
import org.apache.hadoop.yarn.api.protocolrecords.GetNewApplicationResponse;
import org.apache.hadoop.yarn.api.records.ApplicationId;
import org.apache.hadoop.yarn.api.records.ApplicationReport;
import org.apache.hadoop.yarn.api.records.ApplicationResourceUsageReport;
import org.apache.hadoop.yarn.api.records.ApplicationSubmissionContext;
import org.apache.hadoop.yarn.api.records.ContainerLaunchContext;
import org.apache.hadoop.yarn.api.records.FinalApplicationStatus;
import org.apache.hadoop.yarn.api.records.LocalResource;
import org.apache.hadoop.yarn.api.records.LocalResourceType;
import org.apache.hadoop.yarn.api.records.Priority;
import org.apache.hadoop.yarn.api.records.Resource;
import org.apache.hadoop.yarn.api.records.YarnApplicationState;
import org.apache.hadoop.yarn.client.api.YarnClient;
import org.apache.hadoop.yarn.client.api.YarnClientApplication;
import org.apache.hadoop.yarn.conf.YarnConfiguration;
import org.apache.hadoop.yarn.exceptions.YarnException;
import org.apache.hadoop.yarn.util.Records;
import org.apache.helix.Criteria;
import org.apache.helix.HelixManager;
import org.apache.helix.HelixManagerFactory;
import org.apache.helix.InstanceType;
import org.apache.helix.model.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GobblinYarnAppLauncher {
    private static final Logger LOGGER = LoggerFactory.getLogger(GobblinYarnAppLauncher.class);
    private static final Splitter SPLITTER = Splitter.on((char)',').omitEmptyStrings().trimResults();
    private static final String GOBBLIN_YARN_APPLICATION_TYPE = "GOBBLIN_YARN";
    private static final Set<String> APPLICATION_TYPES = ImmutableSet.of((Object)"GOBBLIN_YARN");
    private static final EnumSet<YarnApplicationState> RECONNECTABLE_APPLICATION_STATES = EnumSet.of(YarnApplicationState.NEW, YarnApplicationState.NEW_SAVING, YarnApplicationState.SUBMITTED, YarnApplicationState.ACCEPTED, YarnApplicationState.RUNNING);
    private final String applicationName;
    private final String appQueueName;
    private final Config config;
    private final HelixManager helixManager;
    private final Configuration yarnConfiguration;
    private final YarnClient yarnClient;
    private final FileSystem fs;
    private final EventBus eventBus = new EventBus(GobblinYarnAppLauncher.class.getSimpleName());
    private final ScheduledExecutorService applicationStatusMonitor;
    private final long appReportIntervalMinutes;
    private final Optional<String> appMasterJvmArgs;
    private final Path sinkLogRootDir;
    private final Closer closer = Closer.create();
    private volatile Optional<ApplicationId> applicationId = Optional.absent();
    private volatile Optional<ServiceManager> serviceManager = Optional.absent();
    private final int maxGetApplicationReportFailures;
    private final AtomicInteger getApplicationReportFailureCount = new AtomicInteger();
    private volatile boolean applicationCompleted = false;
    private volatile boolean stopped = false;
    private final boolean emailNotificationOnShutdown;

    public GobblinYarnAppLauncher(Config config, YarnConfiguration yarnConfiguration) throws IOException {
        this.config = config;
        this.applicationName = config.getString("gobblin.yarn.app.name");
        this.appQueueName = config.getString("gobblin.yarn.app.queue");
        String zkConnectionString = config.getString("gobblin.cluster.zk.connection.string");
        LOGGER.info("Using ZooKeeper connection string: " + zkConnectionString);
        this.helixManager = HelixManagerFactory.getZKHelixManager((String)config.getString("gobblin.cluster.helix.cluster.name"), (String)GobblinClusterUtils.getHostname(), (InstanceType)InstanceType.SPECTATOR, (String)zkConnectionString);
        this.yarnConfiguration = yarnConfiguration;
        this.yarnConfiguration.set("fs.automatic.close", "false");
        this.yarnClient = YarnClient.createYarnClient();
        this.yarnClient.init(this.yarnConfiguration);
        this.fs = config.hasPath("fs.uri") ? FileSystem.get((URI)URI.create(config.getString("fs.uri")), (Configuration)this.yarnConfiguration) : FileSystem.get((Configuration)this.yarnConfiguration);
        this.closer.register((Closeable)this.fs);
        this.applicationStatusMonitor = Executors.newSingleThreadScheduledExecutor(ExecutorsUtils.newThreadFactory((Optional)Optional.of((Object)LOGGER), (Optional)Optional.of((Object)"GobblinYarnAppStatusMonitor")));
        this.appReportIntervalMinutes = config.getLong("gobblin.yarn.app.report.interval.minutes");
        this.appMasterJvmArgs = config.hasPath("gobblin.yarn.app.master.jvm.args") ? Optional.of((Object)config.getString("gobblin.yarn.app.master.jvm.args")) : Optional.absent();
        this.sinkLogRootDir = new Path(config.getString("gobblin.yarn.logs.sink.root.dir"));
        this.maxGetApplicationReportFailures = config.getInt("gobblin.yarn.max.get.app.report.failures");
        this.emailNotificationOnShutdown = config.getBoolean("gobblin.yarn.email.notification.on.shutdown");
    }

    public void launch() throws IOException, YarnException {
        this.eventBus.register((Object)this);
        String clusterName = this.config.getString("gobblin.cluster.helix.cluster.name");
        HelixUtils.createGobblinHelixCluster((String)this.config.getString("gobblin.cluster.zk.connection.string"), (String)clusterName);
        LOGGER.info("Created Helix cluster " + clusterName);
        this.connectHelixManager();
        this.startYarnClient();
        this.applicationId = this.getApplicationId();
        this.applicationStatusMonitor.scheduleAtFixedRate(new Runnable(){

            @Override
            public void run() {
                try {
                    GobblinYarnAppLauncher.this.eventBus.post((Object)new ApplicationReportArrivalEvent(GobblinYarnAppLauncher.this.yarnClient.getApplicationReport((ApplicationId)GobblinYarnAppLauncher.this.applicationId.get())));
                }
                catch (IOException | YarnException e) {
                    LOGGER.error("Failed to get application report for Gobblin Yarn application " + GobblinYarnAppLauncher.this.applicationId.get(), e);
                    GobblinYarnAppLauncher.this.eventBus.post((Object)new GetApplicationReportFailureEvent(e));
                }
            }
        }, 0L, this.appReportIntervalMinutes, TimeUnit.MINUTES);
        ArrayList services = Lists.newArrayList();
        if (this.config.hasPath("gobblin.yarn.keytab.file.path")) {
            LOGGER.info("Adding YarnAppSecurityManager since login is keytab based");
            services.add(this.buildYarnAppSecurityManager());
        }
        if (!this.config.hasPath("gobblin.yarn.log.copier.disable.driver.copy") || !this.config.getBoolean("gobblin.yarn.log.copier.disable.driver.copy")) {
            services.add(this.buildLogCopier(this.config, new Path(this.sinkLogRootDir, this.applicationName + "/" + ((ApplicationId)this.applicationId.get()).toString()), GobblinClusterUtils.getAppWorkDirPath((FileSystem)this.fs, (String)this.applicationName, (String)((ApplicationId)this.applicationId.get()).toString())));
        }
        if (this.config.getBoolean("job.execinfo.server.enabled")) {
            LOGGER.info("Starting the job execution info server since it is enabled");
            Properties properties = ConfigUtils.configToProperties((Config)this.config);
            JobExecutionInfoServer executionInfoServer = new JobExecutionInfoServer(properties);
            services.add(executionInfoServer);
            if (this.config.getBoolean("admin.server.enabled")) {
                LOGGER.info("Starting the admin UI server since it is enabled");
                services.add(ServiceBasedAppLauncher.createAdminServer((Properties)properties, (URI)executionInfoServer.getAdvertisedServerUri()));
            }
        } else if (this.config.getBoolean("admin.server.enabled")) {
            LOGGER.warn("NOT starting the admin UI because the job execution info server is NOT enabled");
        }
        this.serviceManager = Optional.of((Object)new ServiceManager((Iterable)services));
        ((ServiceManager)this.serviceManager.get()).startAsync();
    }

    public synchronized void stop() throws IOException, TimeoutException {
        if (this.stopped) {
            return;
        }
        LOGGER.info("Stopping the " + GobblinYarnAppLauncher.class.getSimpleName());
        try {
            if (this.applicationId.isPresent() && !this.applicationCompleted) {
                this.sendShutdownRequest();
            }
            if (this.serviceManager.isPresent()) {
                ((ServiceManager)this.serviceManager.get()).stopAsync().awaitStopped(5L, TimeUnit.MINUTES);
            }
            ExecutorsUtils.shutdownExecutorService((ExecutorService)this.applicationStatusMonitor, (Optional)Optional.of((Object)LOGGER), (long)5L, (TimeUnit)TimeUnit.MINUTES);
            this.stopYarnClient();
            this.disconnectHelixManager();
        }
        finally {
            try {
                if (this.applicationId.isPresent()) {
                    this.cleanUpAppWorkDirectory((ApplicationId)this.applicationId.get());
                }
            }
            finally {
                this.closer.close();
            }
        }
        this.stopped = true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Subscribe
    public void handleApplicationReportArrivalEvent(ApplicationReportArrivalEvent applicationReportArrivalEvent) {
        ApplicationReport applicationReport = applicationReportArrivalEvent.getApplicationReport();
        YarnApplicationState appState = applicationReport.getYarnApplicationState();
        LOGGER.info("Gobblin Yarn application state: " + appState.toString());
        this.getApplicationReportFailureCount.set(0);
        if (appState == YarnApplicationState.FINISHED || appState == YarnApplicationState.FAILED || appState == YarnApplicationState.KILLED) {
            this.applicationCompleted = true;
            LOGGER.info("Gobblin Yarn application finished with final status: " + applicationReport.getFinalApplicationStatus().toString());
            if (applicationReport.getFinalApplicationStatus() == FinalApplicationStatus.FAILED) {
                LOGGER.error("Gobblin Yarn application failed for the following reason: " + applicationReport.getDiagnostics());
            }
            try {
                this.stop();
            }
            catch (IOException ioe) {
                LOGGER.error("Failed to close the " + GobblinYarnAppLauncher.class.getSimpleName(), (Throwable)ioe);
            }
            catch (TimeoutException te) {
                LOGGER.error("Timeout in stopping the service manager", (Throwable)te);
            }
            finally {
                if (this.emailNotificationOnShutdown) {
                    this.sendEmailOnShutdown((Optional<ApplicationReport>)Optional.of((Object)applicationReport));
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Subscribe
    public void handleGetApplicationReportFailureEvent(GetApplicationReportFailureEvent getApplicationReportFailureEvent) {
        int numConsecutiveFailures = this.getApplicationReportFailureCount.incrementAndGet();
        if (numConsecutiveFailures > this.maxGetApplicationReportFailures) {
            LOGGER.warn(String.format("Number of consecutive failures to get the ApplicationReport %d exceeds the threshold %d", numConsecutiveFailures, this.maxGetApplicationReportFailures));
            try {
                this.stop();
            }
            catch (IOException ioe) {
                LOGGER.error("Failed to close the " + GobblinYarnAppLauncher.class.getSimpleName(), (Throwable)ioe);
            }
            catch (TimeoutException te) {
                LOGGER.error("Timeout in stopping the service manager", (Throwable)te);
            }
            finally {
                if (this.emailNotificationOnShutdown) {
                    this.sendEmailOnShutdown((Optional<ApplicationReport>)Optional.absent());
                }
            }
        }
    }

    @VisibleForTesting
    void connectHelixManager() {
        try {
            this.helixManager.connect();
        }
        catch (Exception e) {
            LOGGER.error("HelixManager failed to connect", (Throwable)e);
            throw Throwables.propagate((Throwable)e);
        }
    }

    @VisibleForTesting
    void disconnectHelixManager() {
        if (this.helixManager.isConnected()) {
            this.helixManager.disconnect();
        }
    }

    @VisibleForTesting
    void startYarnClient() {
        this.yarnClient.start();
    }

    @VisibleForTesting
    void stopYarnClient() {
        this.yarnClient.stop();
    }

    private Optional<ApplicationId> getApplicationId() throws YarnException, IOException {
        Optional<ApplicationId> reconnectableApplicationId = this.getReconnectableApplicationId();
        if (reconnectableApplicationId.isPresent()) {
            LOGGER.info("Found reconnectable application with application ID: " + reconnectableApplicationId.get());
            return reconnectableApplicationId;
        }
        LOGGER.info("No reconnectable application found so submitting a new application");
        return Optional.of((Object)this.setupAndSubmitApplication());
    }

    @VisibleForTesting
    Optional<ApplicationId> getReconnectableApplicationId() throws YarnException, IOException {
        List applicationReports = this.yarnClient.getApplications(APPLICATION_TYPES, RECONNECTABLE_APPLICATION_STATES);
        if (applicationReports == null || applicationReports.isEmpty()) {
            return Optional.absent();
        }
        for (ApplicationReport applicationReport : applicationReports) {
            if (!this.applicationName.equals(applicationReport.getName())) continue;
            return Optional.of((Object)applicationReport.getApplicationId());
        }
        return Optional.absent();
    }

    @VisibleForTesting
    ApplicationId setupAndSubmitApplication() throws IOException, YarnException {
        YarnClientApplication gobblinYarnApp = this.yarnClient.createApplication();
        ApplicationSubmissionContext appSubmissionContext = gobblinYarnApp.getApplicationSubmissionContext();
        appSubmissionContext.setApplicationType(GOBBLIN_YARN_APPLICATION_TYPE);
        ApplicationId applicationId = appSubmissionContext.getApplicationId();
        GetNewApplicationResponse newApplicationResponse = gobblinYarnApp.getNewApplicationResponse();
        Resource resource = this.prepareContainerResource(newApplicationResponse);
        Map<String, LocalResource> appMasterLocalResources = this.addAppMasterLocalResources(applicationId);
        ContainerLaunchContext amContainerLaunchContext = (ContainerLaunchContext)Records.newRecord(ContainerLaunchContext.class);
        amContainerLaunchContext.setLocalResources(appMasterLocalResources);
        amContainerLaunchContext.setEnvironment(YarnHelixUtils.getEnvironmentVariables(this.yarnConfiguration));
        amContainerLaunchContext.setCommands((List)Lists.newArrayList((Object[])new String[]{this.buildApplicationMasterCommand(resource.getMemory())}));
        if (UserGroupInformation.isSecurityEnabled()) {
            this.setupSecurityTokens(amContainerLaunchContext);
        }
        appSubmissionContext.setApplicationName(this.applicationName);
        appSubmissionContext.setResource(resource);
        appSubmissionContext.setQueue(this.appQueueName);
        appSubmissionContext.setPriority(Priority.newInstance((int)0));
        appSubmissionContext.setAMContainerSpec(amContainerLaunchContext);
        this.addContainerLocalResources(applicationId);
        LOGGER.info("Submitting application " + applicationId);
        this.yarnClient.submitApplication(appSubmissionContext);
        LOGGER.info("Application successfully submitted and accepted");
        ApplicationReport applicationReport = this.yarnClient.getApplicationReport(applicationId);
        LOGGER.info("Application Name: " + applicationReport.getName());
        LOGGER.info("Application Tracking URL: " + applicationReport.getTrackingUrl());
        LOGGER.info("Application User: " + applicationReport.getUser() + " Queue: " + applicationReport.getQueue());
        return applicationId;
    }

    private Resource prepareContainerResource(GetNewApplicationResponse newApplicationResponse) {
        int maximumVirtualCoreCapacity;
        int vCores;
        int maximumMemoryCapacity;
        int memoryMbs = this.config.getInt("gobblin.yarn.app.master.memory.mbs");
        if (memoryMbs > (maximumMemoryCapacity = newApplicationResponse.getMaximumResourceCapability().getMemory())) {
            LOGGER.info(String.format("Specified AM memory [%d] is above the maximum memory capacity [%d] of the cluster, using the maximum memory capacity instead.", memoryMbs, maximumMemoryCapacity));
            memoryMbs = maximumMemoryCapacity;
        }
        if ((vCores = this.config.getInt("gobblin.yarn.app.master.cores")) > (maximumVirtualCoreCapacity = newApplicationResponse.getMaximumResourceCapability().getVirtualCores())) {
            LOGGER.info(String.format("Specified AM vcores [%d] is above the maximum vcore capacity [%d] of the cluster, using the maximum vcore capacity instead.", memoryMbs, maximumMemoryCapacity));
            vCores = maximumVirtualCoreCapacity;
        }
        return Resource.newInstance((int)memoryMbs, (int)vCores);
    }

    private Map<String, LocalResource> addAppMasterLocalResources(ApplicationId applicationId) throws IOException {
        Path appFilesDestDir;
        Path appWorkDir = GobblinClusterUtils.getAppWorkDirPath((FileSystem)this.fs, (String)this.applicationName, (String)applicationId.toString());
        Path appMasterWorkDir = new Path(appWorkDir, "appmaster");
        HashMap appMasterResources = Maps.newHashMap();
        if (this.config.hasPath("gobblin.yarn.lib.jars.dir")) {
            Path libJarsDestDir = new Path(appWorkDir, "_libjars");
            this.addLibJars(new Path(this.config.getString("gobblin.yarn.lib.jars.dir")), (Optional<Map<String, LocalResource>>)Optional.of((Object)appMasterResources), libJarsDestDir);
        }
        if (this.config.hasPath("gobblin.yarn.app.master.jars")) {
            Path appJarsDestDir = new Path(appMasterWorkDir, "_appjars");
            this.addAppJars(this.config.getString("gobblin.yarn.app.master.jars"), (Optional<Map<String, LocalResource>>)Optional.of((Object)appMasterResources), appJarsDestDir);
        }
        if (this.config.hasPath("gobblin.yarn.app.master.files.local")) {
            appFilesDestDir = new Path(appMasterWorkDir, "_appfiles");
            this.addAppLocalFiles(this.config.getString("gobblin.yarn.app.master.files.local"), (Optional<Map<String, LocalResource>>)Optional.of((Object)appMasterResources), appFilesDestDir);
        }
        if (this.config.hasPath("gobblin.yarn.app.master.files.remote")) {
            this.addAppRemoteFiles(this.config.getString("gobblin.yarn.app.master.files.remote"), appMasterResources);
        }
        if (this.config.hasPath("gobblin.cluster.job.conf.path")) {
            appFilesDestDir = new Path(appMasterWorkDir, "_appfiles");
            this.addJobConfPackage(this.config.getString("gobblin.cluster.job.conf.path"), appFilesDestDir, appMasterResources);
        }
        return appMasterResources;
    }

    private void addContainerLocalResources(ApplicationId applicationId) throws IOException {
        Path appWorkDir = GobblinClusterUtils.getAppWorkDirPath((FileSystem)this.fs, (String)this.applicationName, (String)applicationId.toString());
        Path containerWorkDir = new Path(appWorkDir, "container");
        if (this.config.hasPath("gobblin.yarn.container.jars")) {
            Path appJarsDestDir = new Path(containerWorkDir, "_appjars");
            this.addAppJars(this.config.getString("gobblin.yarn.container.jars"), (Optional<Map<String, LocalResource>>)Optional.absent(), appJarsDestDir);
        }
        if (this.config.hasPath("gobblin.yarn.container.files.local")) {
            Path appFilesDestDir = new Path(containerWorkDir, "_appfiles");
            this.addAppLocalFiles(this.config.getString("gobblin.yarn.container.files.local"), (Optional<Map<String, LocalResource>>)Optional.absent(), appFilesDestDir);
        }
    }

    private void addLibJars(Path srcLibJarDir, Optional<Map<String, LocalResource>> resourceMap, Path destDir) throws IOException {
        LocalFileSystem localFs = FileSystem.getLocal((Configuration)this.yarnConfiguration);
        FileStatus[] libJarFiles = localFs.listStatus(srcLibJarDir);
        if (libJarFiles == null || libJarFiles.length == 0) {
            return;
        }
        for (FileStatus libJarFile : libJarFiles) {
            Path destFilePath = new Path(destDir, libJarFile.getPath().getName());
            this.fs.copyFromLocalFile(libJarFile.getPath(), destFilePath);
            if (!resourceMap.isPresent()) continue;
            YarnHelixUtils.addFileAsLocalResource(this.fs, destFilePath, LocalResourceType.FILE, (Map)resourceMap.get());
        }
    }

    private void addAppJars(String jarFilePathList, Optional<Map<String, LocalResource>> resourceMap, Path destDir) throws IOException {
        for (String jarFilePath : SPLITTER.split((CharSequence)jarFilePathList)) {
            Path srcFilePath = new Path(jarFilePath);
            Path destFilePath = new Path(destDir, srcFilePath.getName());
            this.fs.copyFromLocalFile(srcFilePath, destFilePath);
            if (!resourceMap.isPresent()) continue;
            YarnHelixUtils.addFileAsLocalResource(this.fs, destFilePath, LocalResourceType.FILE, (Map)resourceMap.get());
        }
    }

    private void addAppLocalFiles(String localFilePathList, Optional<Map<String, LocalResource>> resourceMap, Path destDir) throws IOException {
        for (String localFilePath : SPLITTER.split((CharSequence)localFilePathList)) {
            Path srcFilePath = new Path(localFilePath);
            Path destFilePath = new Path(destDir, srcFilePath.getName());
            this.fs.copyFromLocalFile(srcFilePath, destFilePath);
            if (!resourceMap.isPresent()) continue;
            YarnHelixUtils.addFileAsLocalResource(this.fs, destFilePath, LocalResourceType.FILE, (Map)resourceMap.get());
        }
    }

    private void addAppRemoteFiles(String hdfsFileList, Map<String, LocalResource> resourceMap) throws IOException {
        for (String hdfsFilePath : SPLITTER.split((CharSequence)hdfsFileList)) {
            YarnHelixUtils.addFileAsLocalResource(this.fs, new Path(hdfsFilePath), LocalResourceType.FILE, resourceMap);
        }
    }

    private void addJobConfPackage(String jobConfPackagePath, Path destDir, Map<String, LocalResource> resourceMap) throws IOException {
        Path srcFilePath = new Path(jobConfPackagePath);
        Path destFilePath = new Path(destDir, srcFilePath.getName() + ".tar.gz");
        StreamUtils.tar((FileSystem)FileSystem.getLocal((Configuration)this.yarnConfiguration), (FileSystem)this.fs, (Path)srcFilePath, (Path)destFilePath);
        YarnHelixUtils.addFileAsLocalResource(this.fs, destFilePath, LocalResourceType.ARCHIVE, resourceMap);
    }

    private String buildApplicationMasterCommand(int memoryMbs) {
        String appMasterClassName = GobblinApplicationMaster.class.getSimpleName();
        return ApplicationConstants.Environment.JAVA_HOME.$() + "/bin/java" + " -Xmx" + memoryMbs + "M" + " " + JvmUtils.formatJvmArguments(this.appMasterJvmArgs) + " " + GobblinApplicationMaster.class.getName() + " --" + "app_name" + " " + this.applicationName + " 1>" + "<LOG_DIR>" + File.separator + appMasterClassName + "." + "stdout" + " 2>" + "<LOG_DIR>" + File.separator + appMasterClassName + "." + "stderr";
    }

    private void setupSecurityTokens(ContainerLaunchContext containerLaunchContext) throws IOException {
        Credentials credentials = UserGroupInformation.getCurrentUser().getCredentials();
        String tokenRenewer = this.yarnConfiguration.get("yarn.resourcemanager.principal");
        if (tokenRenewer == null || tokenRenewer.length() == 0) {
            throw new IOException("Failed to get master Kerberos principal for the RM to use as renewer");
        }
        Token[] tokens = this.fs.addDelegationTokens(tokenRenewer, credentials);
        if (tokens != null) {
            for (Token token : tokens) {
                LOGGER.info("Got delegation token for " + this.fs.getUri() + "; " + token);
            }
        }
        try (Closer closer = Closer.create();){
            DataOutputBuffer dataOutputBuffer = (DataOutputBuffer)closer.register((Closeable)new DataOutputBuffer());
            credentials.writeTokenStorageToStream((DataOutputStream)dataOutputBuffer);
            ByteBuffer fsTokens = ByteBuffer.wrap(dataOutputBuffer.getData(), 0, dataOutputBuffer.getLength());
            containerLaunchContext.setTokens(fsTokens);
        }
    }

    private LogCopier buildLogCopier(Config config, Path sinkLogDir, Path appWorkDir) throws IOException {
        FileSystem rawLocalFs = (FileSystem)this.closer.register((Closeable)new RawLocalFileSystem());
        rawLocalFs.initialize(URI.create("file:///"), new Configuration());
        LogCopier.Builder builder = LogCopier.newBuilder().useSrcFileSystem(this.fs).useDestFileSystem(rawLocalFs).readFrom(this.getHdfsLogDir(appWorkDir)).writeTo(sinkLogDir).acceptsLogFileExtensions((Set)ImmutableSet.of((Object)"stdout", (Object)"stderr"));
        if (config.hasPath("gobblin.yarn.log.copier.max.file.size")) {
            builder.useMaxBytesPerLogFile(config.getBytes("gobblin.yarn.log.copier.max.file.size").longValue());
        }
        if (config.hasPath("gobblin.yarn.log.copier.scheduler")) {
            builder.useScheduler(config.getString("gobblin.yarn.log.copier.scheduler"));
        }
        return builder.build();
    }

    private Path getHdfsLogDir(Path appWorkDir) throws IOException {
        Path logRootDir = new Path(appWorkDir, "_applogs");
        if (!this.fs.exists(logRootDir)) {
            this.fs.mkdirs(logRootDir);
        }
        return logRootDir;
    }

    private YarnAppSecurityManager buildYarnAppSecurityManager() throws IOException {
        Path tokenFilePath = new Path(this.fs.getHomeDirectory(), this.applicationName + "/" + ".token");
        return new YarnAppSecurityManager(this.config, this.helixManager, this.fs, tokenFilePath);
    }

    @VisibleForTesting
    void sendShutdownRequest() {
        Criteria criteria = new Criteria();
        criteria.setInstanceName("%");
        criteria.setResource("%");
        criteria.setPartition("%");
        criteria.setPartitionState("%");
        criteria.setRecipientInstanceType(InstanceType.CONTROLLER);
        criteria.setSessionSpecific(true);
        Message shutdownRequest = new Message("SHUTDOWN", HelixMessageSubTypes.APPLICATION_MASTER_SHUTDOWN.toString().toLowerCase() + UUID.randomUUID().toString());
        shutdownRequest.setMsgSubType(HelixMessageSubTypes.APPLICATION_MASTER_SHUTDOWN.toString());
        shutdownRequest.setMsgState(Message.MessageState.NEW);
        shutdownRequest.setTgtSessionId("*");
        int messagesSent = this.helixManager.getMessagingService().send(criteria, shutdownRequest);
        if (messagesSent == 0) {
            LOGGER.error(String.format("Failed to send the %s message to the controller", shutdownRequest.getMsgSubType()));
        }
    }

    @VisibleForTesting
    void cleanUpAppWorkDirectory(ApplicationId applicationId) throws IOException {
        Path appWorkDir = GobblinClusterUtils.getAppWorkDirPath((FileSystem)this.fs, (String)this.applicationName, (String)applicationId.toString());
        if (this.fs.exists(appWorkDir)) {
            LOGGER.info("Deleting application working directory " + appWorkDir);
            this.fs.delete(appWorkDir, true);
        }
    }

    private void sendEmailOnShutdown(Optional<ApplicationReport> applicationReport) {
        String subject = String.format("Gobblin Yarn application %s completed", this.applicationName);
        StringBuilder messageBuilder = new StringBuilder("Gobblin Yarn ApplicationReport:");
        if (applicationReport.isPresent()) {
            ApplicationResourceUsageReport resourceUsageReport;
            messageBuilder.append("\n");
            messageBuilder.append("\tApplication ID: ").append(((ApplicationReport)applicationReport.get()).getApplicationId()).append("\n");
            messageBuilder.append("\tApplication attempt ID: ").append(((ApplicationReport)applicationReport.get()).getCurrentApplicationAttemptId()).append("\n");
            messageBuilder.append("\tFinal application status: ").append(((ApplicationReport)applicationReport.get()).getFinalApplicationStatus()).append("\n");
            messageBuilder.append("\tStart time: ").append(((ApplicationReport)applicationReport.get()).getStartTime()).append("\n");
            messageBuilder.append("\tFinish time: ").append(((ApplicationReport)applicationReport.get()).getFinishTime()).append("\n");
            if (!Strings.isNullOrEmpty((String)((ApplicationReport)applicationReport.get()).getDiagnostics())) {
                messageBuilder.append("\tDiagnostics: ").append(((ApplicationReport)applicationReport.get()).getDiagnostics()).append("\n");
            }
            if ((resourceUsageReport = ((ApplicationReport)applicationReport.get()).getApplicationResourceUsageReport()) != null) {
                messageBuilder.append("\tUsed containers: ").append(resourceUsageReport.getNumUsedContainers()).append("\n");
                Resource usedResource = resourceUsageReport.getUsedResources();
                if (usedResource != null) {
                    messageBuilder.append("\tUsed memory (MBs): ").append(usedResource.getMemory()).append("\n");
                    messageBuilder.append("\tUsed vcores: ").append(usedResource.getVirtualCores()).append("\n");
                }
            }
        } else {
            messageBuilder.append(' ').append("Not available");
        }
        try {
            EmailUtils.sendEmail((State)ConfigUtils.configToState((Config)this.config), (String)subject, (String)messageBuilder.toString());
        }
        catch (EmailException ee) {
            LOGGER.error("Failed to send email notification on shutdown", (Throwable)ee);
        }
    }

    public static void main(String[] args) throws Exception {
        final GobblinYarnAppLauncher gobblinYarnAppLauncher = new GobblinYarnAppLauncher(ConfigFactory.load(), new YarnConfiguration());
        Runtime.getRuntime().addShutdownHook(new Thread(){

            @Override
            public void run() {
                try {
                    gobblinYarnAppLauncher.stop();
                }
                catch (IOException ioe) {
                    LOGGER.error("Failed to shutdown the " + GobblinYarnAppLauncher.class.getSimpleName(), (Throwable)ioe);
                }
                catch (TimeoutException te) {
                    LOGGER.error("Timeout in stopping the service manager", (Throwable)te);
                }
                finally {
                    if (gobblinYarnAppLauncher.emailNotificationOnShutdown) {
                        gobblinYarnAppLauncher.sendEmailOnShutdown((Optional<ApplicationReport>)Optional.absent());
                    }
                }
            }
        });
        gobblinYarnAppLauncher.launch();
    }
}

