/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

#pragma once

#include <map>
#include <memory>
#include <string>
#include <utility>

#include "graphar/util/macros.h"

// forward declaration
namespace arrow {
class DataType;
}

namespace graphar {

/** @brief Main data type enumeration. */
enum class Type {
  /** Boolean */
  BOOL = 0,

  /** Signed 32-bit integer */
  INT32,

  /** Signed 64-bit integer */
  INT64,

  /** 4-byte floating point value */
  FLOAT,

  /** 8-byte floating point value */
  DOUBLE,

  /** UTF8 variable-length string */
  STRING,

  /** List of some logical data type */
  LIST,

  /** int32_t days since the UNIX epoch */
  DATE,

  /** Exact timestamp encoded with int64 since UNIX epoch in milliseconds */
  TIMESTAMP,

  /** User-defined data type */
  USER_DEFINED,

  // Leave this at the end
  MAX_ID,
};

/**
 * @brief The DataType struct to provide enum type for data type and functions
 *   to parse data type.
 */
class DataType {
 public:
  DataType() : id_(Type::BOOL) {}

  explicit DataType(Type id, const std::string& user_defined_type_name = "")
      : id_(id),
        child_(nullptr),
        user_defined_type_name_(user_defined_type_name) {}

  explicit DataType(Type id, const std::shared_ptr<DataType>& child)
      : id_(id), child_(std::move(child)), user_defined_type_name_("") {}

  DataType(const DataType& other)
      : id_(other.id_),
        child_(other.child_),
        user_defined_type_name_(other.user_defined_type_name_) {}

  explicit DataType(DataType&& other)
      : id_(other.id_),
        child_(std::move(other.child_)),
        user_defined_type_name_(std::move(other.user_defined_type_name_)) {}

  inline DataType& operator=(const DataType& other) = default;

  bool Equals(const DataType& other) const {
    return id_ == other.id_ &&
           user_defined_type_name_ == other.user_defined_type_name_;
  }

  bool Equals(const std::shared_ptr<DataType>& other) const {
    if (!other) {
      return false;
    }
    return Equals(*other.get());
  }

  const std::shared_ptr<DataType>& value_type() const { return child_; }

  bool operator==(const DataType& other) const { return Equals(other); }

  bool operator!=(const DataType& other) const { return !Equals(other); }

  static std::shared_ptr<arrow::DataType> DataTypeToArrowDataType(
      const std::shared_ptr<DataType>& type);

  static std::shared_ptr<DataType> ArrowDataTypeToDataType(
      const std::shared_ptr<arrow::DataType>& type);

  static std::shared_ptr<DataType> TypeNameToDataType(const std::string& str);

  /** Return the type category of the DataType. */
  Type id() const { return id_; }

  std::string ToTypeName() const;

 private:
  Type id_;
  std::shared_ptr<DataType> child_;
  std::string user_defined_type_name_;
};  // struct DataType

// Define a Timestamp class to represent timestamp data type value
class Timestamp {
 public:
  using c_type = int64_t;
  explicit Timestamp(c_type value) : value_(value) {}

  c_type value() const { return value_; }

 private:
  c_type value_;
};

// Define a Date class to represent date data type value
class Date {
 public:
  using c_type = int32_t;
  explicit Date(c_type value) : value_(value) {}

  c_type value() const { return value_; }

 private:
  c_type value_;
};

}  // namespace graphar
