{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GInputStream@ is a base class for implementing streaming input.
-- 
-- It has functions to read from a stream ('GI.Gio.Objects.InputStream.inputStreamRead'),
-- to close a stream ('GI.Gio.Objects.InputStream.inputStreamClose') and to skip some content
-- ('GI.Gio.Objects.InputStream.inputStreamSkip').
-- 
-- To copy the content of an input stream to an output stream without
-- manually handling the reads and writes, use 'GI.Gio.Objects.OutputStream.outputStreamSplice'.
-- 
-- See the documentation for t'GI.Gio.Objects.IOStream.IOStream' for details of thread safety
-- of streaming APIs.
-- 
-- All of these functions have async variants too.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.InputStream
    ( 

-- * Exported types
    InputStream(..)                         ,
    IsInputStream                           ,
    toInputStream                           ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [clearPending]("GI.Gio.Objects.InputStream#g:method:clearPending"), [close]("GI.Gio.Objects.InputStream#g:method:close"), [closeAsync]("GI.Gio.Objects.InputStream#g:method:closeAsync"), [closeFinish]("GI.Gio.Objects.InputStream#g:method:closeFinish"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [hasPending]("GI.Gio.Objects.InputStream#g:method:hasPending"), [isClosed]("GI.Gio.Objects.InputStream#g:method:isClosed"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [read]("GI.Gio.Objects.InputStream#g:method:read"), [readAll]("GI.Gio.Objects.InputStream#g:method:readAll"), [readAllAsync]("GI.Gio.Objects.InputStream#g:method:readAllAsync"), [readAllFinish]("GI.Gio.Objects.InputStream#g:method:readAllFinish"), [readAsync]("GI.Gio.Objects.InputStream#g:method:readAsync"), [readBytes]("GI.Gio.Objects.InputStream#g:method:readBytes"), [readBytesAsync]("GI.Gio.Objects.InputStream#g:method:readBytesAsync"), [readBytesFinish]("GI.Gio.Objects.InputStream#g:method:readBytesFinish"), [readFinish]("GI.Gio.Objects.InputStream#g:method:readFinish"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [skip]("GI.Gio.Objects.InputStream#g:method:skip"), [skipAsync]("GI.Gio.Objects.InputStream#g:method:skipAsync"), [skipFinish]("GI.Gio.Objects.InputStream#g:method:skipFinish"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getData]("GI.GObject.Objects.Object#g:method:getData"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setPending]("GI.Gio.Objects.InputStream#g:method:setPending"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveInputStreamMethod                ,
#endif

-- ** clearPending #method:clearPending#

#if defined(ENABLE_OVERLOADING)
    InputStreamClearPendingMethodInfo       ,
#endif
    inputStreamClearPending                 ,


-- ** close #method:close#

#if defined(ENABLE_OVERLOADING)
    InputStreamCloseMethodInfo              ,
#endif
    inputStreamClose                        ,


-- ** closeAsync #method:closeAsync#

#if defined(ENABLE_OVERLOADING)
    InputStreamCloseAsyncMethodInfo         ,
#endif
    inputStreamCloseAsync                   ,


-- ** closeFinish #method:closeFinish#

#if defined(ENABLE_OVERLOADING)
    InputStreamCloseFinishMethodInfo        ,
#endif
    inputStreamCloseFinish                  ,


-- ** hasPending #method:hasPending#

#if defined(ENABLE_OVERLOADING)
    InputStreamHasPendingMethodInfo         ,
#endif
    inputStreamHasPending                   ,


-- ** isClosed #method:isClosed#

#if defined(ENABLE_OVERLOADING)
    InputStreamIsClosedMethodInfo           ,
#endif
    inputStreamIsClosed                     ,


-- ** read #method:read#

#if defined(ENABLE_OVERLOADING)
    InputStreamReadMethodInfo               ,
#endif
    inputStreamRead                         ,


-- ** readAll #method:readAll#

#if defined(ENABLE_OVERLOADING)
    InputStreamReadAllMethodInfo            ,
#endif
    inputStreamReadAll                      ,


-- ** readAllAsync #method:readAllAsync#

#if defined(ENABLE_OVERLOADING)
    InputStreamReadAllAsyncMethodInfo       ,
#endif
    inputStreamReadAllAsync                 ,


-- ** readAllFinish #method:readAllFinish#

#if defined(ENABLE_OVERLOADING)
    InputStreamReadAllFinishMethodInfo      ,
#endif
    inputStreamReadAllFinish                ,


-- ** readAsync #method:readAsync#

#if defined(ENABLE_OVERLOADING)
    InputStreamReadAsyncMethodInfo          ,
#endif
    inputStreamReadAsync                    ,


-- ** readBytes #method:readBytes#

#if defined(ENABLE_OVERLOADING)
    InputStreamReadBytesMethodInfo          ,
#endif
    inputStreamReadBytes                    ,


-- ** readBytesAsync #method:readBytesAsync#

#if defined(ENABLE_OVERLOADING)
    InputStreamReadBytesAsyncMethodInfo     ,
#endif
    inputStreamReadBytesAsync               ,


-- ** readBytesFinish #method:readBytesFinish#

#if defined(ENABLE_OVERLOADING)
    InputStreamReadBytesFinishMethodInfo    ,
#endif
    inputStreamReadBytesFinish              ,


-- ** readFinish #method:readFinish#

#if defined(ENABLE_OVERLOADING)
    InputStreamReadFinishMethodInfo         ,
#endif
    inputStreamReadFinish                   ,


-- ** setPending #method:setPending#

#if defined(ENABLE_OVERLOADING)
    InputStreamSetPendingMethodInfo         ,
#endif
    inputStreamSetPending                   ,


-- ** skip #method:skip#

#if defined(ENABLE_OVERLOADING)
    InputStreamSkipMethodInfo               ,
#endif
    inputStreamSkip                         ,


-- ** skipAsync #method:skipAsync#

#if defined(ENABLE_OVERLOADING)
    InputStreamSkipAsyncMethodInfo          ,
#endif
    inputStreamSkipAsync                    ,


-- ** skipFinish #method:skipFinish#

#if defined(ENABLE_OVERLOADING)
    InputStreamSkipFinishMethodInfo         ,
#endif
    inputStreamSkipFinish                   ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable

#else
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable

#endif

-- | Memory-managed wrapper type.
newtype InputStream = InputStream (SP.ManagedPtr InputStream)
    deriving (Eq)

instance SP.ManagedPtrNewtype InputStream where
    toManagedPtr (InputStream p) = p

foreign import ccall "g_input_stream_get_type"
    c_g_input_stream_get_type :: IO B.Types.GType

instance B.Types.TypedObject InputStream where
    glibType = c_g_input_stream_get_type

instance B.Types.GObject InputStream

-- | Type class for types which can be safely cast to t'InputStream', for instance with `toInputStream`.
class (SP.GObject o, O.IsDescendantOf InputStream o) => IsInputStream o
instance (SP.GObject o, O.IsDescendantOf InputStream o) => IsInputStream o

instance O.HasParentTypes InputStream
type instance O.ParentTypes InputStream = '[GObject.Object.Object]

-- | Cast to t'InputStream', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toInputStream :: (MIO.MonadIO m, IsInputStream o) => o -> m InputStream
toInputStream = MIO.liftIO . B.ManagedPtr.unsafeCastTo InputStream

-- | Convert t'InputStream' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe InputStream) where
    gvalueGType_ = c_g_input_stream_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr InputStream)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr InputStream)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject InputStream ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveInputStreamMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveInputStreamMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveInputStreamMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveInputStreamMethod "clearPending" o = InputStreamClearPendingMethodInfo
    ResolveInputStreamMethod "close" o = InputStreamCloseMethodInfo
    ResolveInputStreamMethod "closeAsync" o = InputStreamCloseAsyncMethodInfo
    ResolveInputStreamMethod "closeFinish" o = InputStreamCloseFinishMethodInfo
    ResolveInputStreamMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveInputStreamMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveInputStreamMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveInputStreamMethod "hasPending" o = InputStreamHasPendingMethodInfo
    ResolveInputStreamMethod "isClosed" o = InputStreamIsClosedMethodInfo
    ResolveInputStreamMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveInputStreamMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveInputStreamMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveInputStreamMethod "read" o = InputStreamReadMethodInfo
    ResolveInputStreamMethod "readAll" o = InputStreamReadAllMethodInfo
    ResolveInputStreamMethod "readAllAsync" o = InputStreamReadAllAsyncMethodInfo
    ResolveInputStreamMethod "readAllFinish" o = InputStreamReadAllFinishMethodInfo
    ResolveInputStreamMethod "readAsync" o = InputStreamReadAsyncMethodInfo
    ResolveInputStreamMethod "readBytes" o = InputStreamReadBytesMethodInfo
    ResolveInputStreamMethod "readBytesAsync" o = InputStreamReadBytesAsyncMethodInfo
    ResolveInputStreamMethod "readBytesFinish" o = InputStreamReadBytesFinishMethodInfo
    ResolveInputStreamMethod "readFinish" o = InputStreamReadFinishMethodInfo
    ResolveInputStreamMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveInputStreamMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveInputStreamMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveInputStreamMethod "skip" o = InputStreamSkipMethodInfo
    ResolveInputStreamMethod "skipAsync" o = InputStreamSkipAsyncMethodInfo
    ResolveInputStreamMethod "skipFinish" o = InputStreamSkipFinishMethodInfo
    ResolveInputStreamMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveInputStreamMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveInputStreamMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveInputStreamMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveInputStreamMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveInputStreamMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveInputStreamMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveInputStreamMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveInputStreamMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveInputStreamMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveInputStreamMethod "setPending" o = InputStreamSetPendingMethodInfo
    ResolveInputStreamMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveInputStreamMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveInputStreamMethod t InputStream, O.OverloadedMethod info InputStream p) => OL.IsLabel t (InputStream -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveInputStreamMethod t InputStream, O.OverloadedMethod info InputStream p, R.HasField t InputStream p) => R.HasField t InputStream p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveInputStreamMethod t InputStream, O.OverloadedMethodInfo info InputStream) => OL.IsLabel t (O.MethodProxy info InputStream) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList InputStream
type instance O.AttributeList InputStream = InputStreamAttributeList
type InputStreamAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList InputStream = InputStreamSignalList
type InputStreamSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method InputStream::clear_pending
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input stream" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_input_stream_clear_pending" g_input_stream_clear_pending :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    IO ()

-- | Clears the pending flag on /@stream@/.
inputStreamClearPending ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a) =>
    a
    -- ^ /@stream@/: input stream
    -> m ()
inputStreamClearPending stream = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    g_input_stream_clear_pending stream'
    touchManagedPtr stream
    return ()

#if defined(ENABLE_OVERLOADING)
data InputStreamClearPendingMethodInfo
instance (signature ~ (m ()), MonadIO m, IsInputStream a) => O.OverloadedMethod InputStreamClearPendingMethodInfo a signature where
    overloadedMethod = inputStreamClearPending

instance O.OverloadedMethodInfo InputStreamClearPendingMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamClearPending",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamClearPending"
        })


#endif

-- method InputStream::close
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GInputStream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_input_stream_close" g_input_stream_close :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Closes the stream, releasing resources related to it.
-- 
-- Once the stream is closed, all other operations will return 'GI.Gio.Enums.IOErrorEnumClosed'.
-- Closing a stream multiple times will not return an error.
-- 
-- Streams will be automatically closed when the last reference
-- is dropped, but you might want to call this function to make sure
-- resources are released as early as possible.
-- 
-- Some streams might keep the backing store of the stream (e.g. a file descriptor)
-- open after the stream is closed. See the documentation for the individual
-- stream for details.
-- 
-- On failure the first error that happened will be reported, but the close
-- operation will finish as much as possible. A stream that failed to
-- close will still return 'GI.Gio.Enums.IOErrorEnumClosed' for all operations. Still, it
-- is important to check and report the error to the user.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned.
-- Cancelling a close will still leave the stream closed, but some streams
-- can use a faster close that doesn\'t block to e.g. check errors.
inputStreamClose ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@stream@/: A t'GI.Gio.Objects.InputStream.InputStream'.
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
inputStreamClose stream cancellable = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_input_stream_close stream' maybeCancellable
        touchManagedPtr stream
        whenJust cancellable touchManagedPtr
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data InputStreamCloseMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod InputStreamCloseMethodInfo a signature where
    overloadedMethod = inputStreamClose

instance O.OverloadedMethodInfo InputStreamCloseMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamClose",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamClose"
        })


#endif

-- method InputStream::close_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GInputStream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "optional cancellable object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_input_stream_close_async" g_input_stream_close_async :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Requests an asynchronous closes of the stream, releasing resources related to it.
-- When the operation is finished /@callback@/ will be called.
-- You can then call 'GI.Gio.Objects.InputStream.inputStreamCloseFinish' to get the result of the
-- operation.
-- 
-- For behaviour details see 'GI.Gio.Objects.InputStream.inputStreamClose'.
-- 
-- The asynchronous methods have a default fallback that uses threads to implement
-- asynchronicity, so they are optional for inheriting classes. However, if you
-- override one you must override all.
inputStreamCloseAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@stream@/: A t'GI.Gio.Objects.InputStream.InputStream'.
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional cancellable object
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
inputStreamCloseAsync stream ioPriority cancellable callback = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_input_stream_close_async stream' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr stream
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data InputStreamCloseAsyncMethodInfo
instance (signature ~ (Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod InputStreamCloseAsyncMethodInfo a signature where
    overloadedMethod = inputStreamCloseAsync

instance O.OverloadedMethodInfo InputStreamCloseAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamCloseAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamCloseAsync"
        })


#endif

-- method InputStream::close_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GInputStream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_input_stream_close_finish" g_input_stream_close_finish :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes closing a stream asynchronously, started from 'GI.Gio.Objects.InputStream.inputStreamCloseAsync'.
inputStreamCloseFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.InputStream.InputStream'.
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'.
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
inputStreamCloseFinish stream result_ = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_input_stream_close_finish stream' result_'
        touchManagedPtr stream
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data InputStreamCloseFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsInputStream a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod InputStreamCloseFinishMethodInfo a signature where
    overloadedMethod = inputStreamCloseFinish

instance O.OverloadedMethodInfo InputStreamCloseFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamCloseFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamCloseFinish"
        })


#endif

-- method InputStream::has_pending
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input stream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_input_stream_has_pending" g_input_stream_has_pending :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    IO CInt

-- | Checks if an input stream has pending actions.
inputStreamHasPending ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a) =>
    a
    -- ^ /@stream@/: input stream.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@stream@/ has pending actions.
inputStreamHasPending stream = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    result <- g_input_stream_has_pending stream'
    let result' = (/= 0) result
    touchManagedPtr stream
    return result'

#if defined(ENABLE_OVERLOADING)
data InputStreamHasPendingMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsInputStream a) => O.OverloadedMethod InputStreamHasPendingMethodInfo a signature where
    overloadedMethod = inputStreamHasPending

instance O.OverloadedMethodInfo InputStreamHasPendingMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamHasPending",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamHasPending"
        })


#endif

-- method InputStream::is_closed
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input stream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_input_stream_is_closed" g_input_stream_is_closed :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    IO CInt

-- | Checks if an input stream is closed.
inputStreamIsClosed ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a) =>
    a
    -- ^ /@stream@/: input stream.
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the stream is closed.
inputStreamIsClosed stream = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    result <- g_input_stream_is_closed stream'
    let result' = (/= 0) result
    touchManagedPtr stream
    return result'

#if defined(ENABLE_OVERLOADING)
data InputStreamIsClosedMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsInputStream a) => O.OverloadedMethod InputStreamIsClosedMethodInfo a signature where
    overloadedMethod = inputStreamIsClosed

instance O.OverloadedMethodInfo InputStreamIsClosedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamIsClosed",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamIsClosed"
        })


#endif

-- method InputStream::read
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GInputStream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "buffer"
--           , argType = TCArray False (-1) 2 (TBasicType TUInt8)
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "\n  a buffer to read data into (which should be at least count bytes long)."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "count"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the number of bytes that will be read from the stream"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "count"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just "the number of bytes that will be read from the stream"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TBasicType TSSize)
-- throws : True
-- Skip return : False

foreign import ccall "g_input_stream_read" g_input_stream_read :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    Ptr Word8 ->                            -- buffer : TCArray False (-1) 2 (TBasicType TUInt8)
    FCT.CSize ->                            -- count : TBasicType TSize
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO DI.Int64

-- | Tries to read /@count@/ bytes from the stream into the buffer starting at
-- /@buffer@/. Will block during this read.
-- 
-- If count is zero returns zero and does nothing. A value of /@count@/
-- larger than @/G_MAXSSIZE/@ will cause a 'GI.Gio.Enums.IOErrorEnumInvalidArgument' error.
-- 
-- On success, the number of bytes read into the buffer is returned.
-- It is not an error if this is not the same as the requested size, as it
-- can happen e.g. near the end of a file. Zero is returned on end of file
-- (or if /@count@/ is zero),  but never otherwise.
-- 
-- The returned /@buffer@/ is not a nul-terminated string, it can contain nul bytes
-- at any position, and this function doesn\'t nul-terminate the /@buffer@/.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned. If an
-- operation was partially finished when the operation was cancelled the
-- partial result will be returned, without an error.
-- 
-- On error -1 is returned and /@error@/ is set accordingly.
inputStreamRead ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.InputStream.InputStream'.
    -> ByteString
    -- ^ /@buffer@/: 
    --   a buffer to read data into (which should be at least count bytes long).
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> m ((DI.Int64, ByteString))
    -- ^ __Returns:__ Number of bytes read, or -1 on error, or 0 on end of file. /(Can throw 'Data.GI.Base.GError.GError')/
inputStreamRead stream buffer cancellable = liftIO $ do
    let count = fromIntegral $ B.length buffer
    stream' <- unsafeManagedPtrCastPtr stream
    buffer' <- packByteString buffer
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_input_stream_read stream' buffer' count maybeCancellable
        buffer'' <- (unpackByteStringWithLength count) buffer'
        freeMem buffer'
        touchManagedPtr stream
        whenJust cancellable touchManagedPtr
        return (result, buffer'')
     ) (do
        freeMem buffer'
     )

#if defined(ENABLE_OVERLOADING)
data InputStreamReadMethodInfo
instance (signature ~ (ByteString -> Maybe (b) -> m ((DI.Int64, ByteString))), MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod InputStreamReadMethodInfo a signature where
    overloadedMethod = inputStreamRead

instance O.OverloadedMethodInfo InputStreamReadMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamRead",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamRead"
        })


#endif

-- method InputStream::read_all
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GInputStream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "buffer"
--           , argType = TCArray False (-1) 2 (TBasicType TUInt8)
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "\n  a buffer to read data into (which should be at least count bytes long)."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "count"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the number of bytes that will be read from the stream"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "bytes_read"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "location to store the number of bytes that was read from the stream"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "count"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just "the number of bytes that will be read from the stream"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_input_stream_read_all" g_input_stream_read_all :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    Ptr Word8 ->                            -- buffer : TCArray False (-1) 2 (TBasicType TUInt8)
    FCT.CSize ->                            -- count : TBasicType TSize
    Ptr FCT.CSize ->                        -- bytes_read : TBasicType TSize
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Tries to read /@count@/ bytes from the stream into the buffer starting at
-- /@buffer@/. Will block during this read.
-- 
-- This function is similar to 'GI.Gio.Objects.InputStream.inputStreamRead', except it tries to
-- read as many bytes as requested, only stopping on an error or end of stream.
-- 
-- On a successful read of /@count@/ bytes, or if we reached the end of the
-- stream,  'P.True' is returned, and /@bytesRead@/ is set to the number of bytes
-- read into /@buffer@/.
-- 
-- If there is an error during the operation 'P.False' is returned and /@error@/
-- is set to indicate the error status.
-- 
-- As a special exception to the normal conventions for functions that
-- use t'GError', if this function returns 'P.False' (and sets /@error@/) then
-- /@bytesRead@/ will be set to the number of bytes that were successfully
-- read before the error was encountered.  This functionality is only
-- available from C.  If you need it from another language then you must
-- write your own loop around 'GI.Gio.Objects.InputStream.inputStreamRead'.
inputStreamReadAll ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.InputStream.InputStream'.
    -> ByteString
    -- ^ /@buffer@/: 
    --   a buffer to read data into (which should be at least count bytes long).
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> m ((ByteString, FCT.CSize))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
inputStreamReadAll stream buffer cancellable = liftIO $ do
    let count = fromIntegral $ B.length buffer
    stream' <- unsafeManagedPtrCastPtr stream
    buffer' <- packByteString buffer
    bytesRead <- allocMem :: IO (Ptr FCT.CSize)
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_input_stream_read_all stream' buffer' count bytesRead maybeCancellable
        buffer'' <- (unpackByteStringWithLength count) buffer'
        freeMem buffer'
        bytesRead' <- peek bytesRead
        touchManagedPtr stream
        whenJust cancellable touchManagedPtr
        freeMem bytesRead
        return (buffer'', bytesRead')
     ) (do
        freeMem buffer'
        freeMem bytesRead
     )

#if defined(ENABLE_OVERLOADING)
data InputStreamReadAllMethodInfo
instance (signature ~ (ByteString -> Maybe (b) -> m ((ByteString, FCT.CSize))), MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod InputStreamReadAllMethodInfo a signature where
    overloadedMethod = inputStreamReadAll

instance O.OverloadedMethodInfo InputStreamReadAllMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamReadAll",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamReadAll"
        })


#endif

-- method InputStream::read_all_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GInputStream" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "buffer"
--           , argType = TCArray False (-1) 2 (TBasicType TUInt8)
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "\n  a buffer to read data into (which should be at least count bytes long)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "count"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the number of bytes that will be read from the stream"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 6
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "count"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just "the number of bytes that will be read from the stream"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_input_stream_read_all_async" g_input_stream_read_all_async :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    Ptr Word8 ->                            -- buffer : TCArray False (-1) 2 (TBasicType TUInt8)
    FCT.CSize ->                            -- count : TBasicType TSize
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Request an asynchronous read of /@count@/ bytes from the stream into the
-- buffer starting at /@buffer@/.
-- 
-- This is the asynchronous equivalent of [method/@inputStream@/.read_all].
-- 
-- Call [method/@inputStream@/.read_all_finish] to collect the result.
-- 
-- Any outstanding I\/O request with higher priority (lower numerical
-- value) will be executed before an outstanding request with lower
-- priority. Default priority is 'GI.GLib.Constants.PRIORITY_DEFAULT'.
-- 
-- /Since: 2.44/
inputStreamReadAllAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@stream@/: A t'GI.Gio.Objects.InputStream.InputStream'
    -> ByteString
    -- ^ /@buffer@/: 
    --   a buffer to read data into (which should be at least count bytes long)
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m (ByteString)
inputStreamReadAllAsync stream buffer ioPriority cancellable callback = liftIO $ do
    let count = fromIntegral $ B.length buffer
    stream' <- unsafeManagedPtrCastPtr stream
    buffer' <- packByteString buffer
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_input_stream_read_all_async stream' buffer' count ioPriority maybeCancellable maybeCallback userData
    buffer'' <- (unpackByteStringWithLength count) buffer'
    freeMem buffer'
    touchManagedPtr stream
    whenJust cancellable touchManagedPtr
    return buffer''

#if defined(ENABLE_OVERLOADING)
data InputStreamReadAllAsyncMethodInfo
instance (signature ~ (ByteString -> Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m (ByteString)), MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod InputStreamReadAllAsyncMethodInfo a signature where
    overloadedMethod = inputStreamReadAllAsync

instance O.OverloadedMethodInfo InputStreamReadAllAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamReadAllAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamReadAllAsync"
        })


#endif

-- method InputStream::read_all_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GInputStream" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "bytes_read"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "location to store the number of bytes that was read from the stream"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_input_stream_read_all_finish" g_input_stream_read_all_finish :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr FCT.CSize ->                        -- bytes_read : TBasicType TSize
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes an asynchronous stream read operation started with
-- [method/@inputStream@/.read_all_async].
-- 
-- As a special exception to the normal conventions for functions that
-- use t'GError', if this function returns 'P.False' (and sets /@error@/) then
-- /@bytesRead@/ will be set to the number of bytes that were successfully
-- read before the error was encountered.  This functionality is only
-- available from C.  If you need it from another language then you must
-- write your own loop around 'GI.Gio.Objects.InputStream.inputStreamReadAsync'.
-- 
-- /Since: 2.44/
inputStreamReadAllFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.InputStream.InputStream'
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'
    -> m (FCT.CSize)
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
inputStreamReadAllFinish stream result_ = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    result_' <- unsafeManagedPtrCastPtr result_
    bytesRead <- allocMem :: IO (Ptr FCT.CSize)
    onException (do
        _ <- propagateGError $ g_input_stream_read_all_finish stream' result_' bytesRead
        bytesRead' <- peek bytesRead
        touchManagedPtr stream
        touchManagedPtr result_
        freeMem bytesRead
        return bytesRead'
     ) (do
        freeMem bytesRead
     )

#if defined(ENABLE_OVERLOADING)
data InputStreamReadAllFinishMethodInfo
instance (signature ~ (b -> m (FCT.CSize)), MonadIO m, IsInputStream a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod InputStreamReadAllFinishMethodInfo a signature where
    overloadedMethod = inputStreamReadAllFinish

instance O.OverloadedMethodInfo InputStreamReadAllFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamReadAllFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamReadAllFinish"
        })


#endif

-- method InputStream::read_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GInputStream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "buffer"
--           , argType = TCArray False (-1) 2 (TBasicType TUInt8)
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "\n  a buffer to read data into (which should be at least count bytes long)."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "count"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the number of bytes that will be read from the stream"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority)\nof the request."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 6
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "count"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just "the number of bytes that will be read from the stream"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_input_stream_read_async" g_input_stream_read_async :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    Ptr Word8 ->                            -- buffer : TCArray False (-1) 2 (TBasicType TUInt8)
    FCT.CSize ->                            -- count : TBasicType TSize
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Request an asynchronous read of /@count@/ bytes from the stream into the buffer
-- starting at /@buffer@/. When the operation is finished /@callback@/ will be called.
-- You can then call 'GI.Gio.Objects.InputStream.inputStreamReadFinish' to get the result of the
-- operation.
-- 
-- During an async request no other sync and async calls are allowed on /@stream@/, and will
-- result in 'GI.Gio.Enums.IOErrorEnumPending' errors.
-- 
-- A value of /@count@/ larger than @/G_MAXSSIZE/@ will cause a 'GI.Gio.Enums.IOErrorEnumInvalidArgument' error.
-- 
-- On success, the number of bytes read into the buffer will be passed to the
-- callback. It is not an error if this is not the same as the requested size, as it
-- can happen e.g. near the end of a file, but generally we try to read
-- as many bytes as requested. Zero is returned on end of file
-- (or if /@count@/ is zero),  but never otherwise.
-- 
-- Any outstanding i\/o request with higher priority (lower numerical value) will
-- be executed before an outstanding request with lower priority. Default
-- priority is 'GI.GLib.Constants.PRIORITY_DEFAULT'.
-- 
-- The asynchronous methods have a default fallback that uses threads to implement
-- asynchronicity, so they are optional for inheriting classes. However, if you
-- override one you must override all.
inputStreamReadAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@stream@/: A t'GI.Gio.Objects.InputStream.InputStream'.
    -> Maybe (ByteString)
    -- ^ /@buffer@/: 
    --   a buffer to read data into (which should be at least count bytes long).
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority>
    -- of the request.
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ((Maybe ByteString))
inputStreamReadAsync stream buffer ioPriority cancellable callback = liftIO $ do
    let count = case buffer of
            Nothing -> 0
            Just jBuffer -> fromIntegral $ B.length jBuffer
    stream' <- unsafeManagedPtrCastPtr stream
    maybeBuffer <- case buffer of
        Nothing -> return FP.nullPtr
        Just jBuffer -> do
            jBuffer' <- packByteString jBuffer
            return jBuffer'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_input_stream_read_async stream' maybeBuffer count ioPriority maybeCancellable maybeCallback userData
    maybeMaybeBuffer <- convertIfNonNull maybeBuffer $ \maybeBuffer' -> do
        maybeBuffer'' <- (unpackByteStringWithLength count) maybeBuffer'
        freeMem maybeBuffer'
        return maybeBuffer''
    touchManagedPtr stream
    whenJust cancellable touchManagedPtr
    return maybeMaybeBuffer

#if defined(ENABLE_OVERLOADING)
data InputStreamReadAsyncMethodInfo
instance (signature ~ (Maybe (ByteString) -> Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ((Maybe ByteString))), MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod InputStreamReadAsyncMethodInfo a signature where
    overloadedMethod = inputStreamReadAsync

instance O.OverloadedMethodInfo InputStreamReadAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamReadAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamReadAsync"
        })


#endif

-- method InputStream::read_bytes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GInputStream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "count"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "maximum number of bytes that will be read from the stream. Common\nvalues include 4096 and 8192."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Bytes" })
-- throws : True
-- Skip return : False

foreign import ccall "g_input_stream_read_bytes" g_input_stream_read_bytes :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    FCT.CSize ->                            -- count : TBasicType TSize
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr GLib.Bytes.Bytes)

-- | Like 'GI.Gio.Objects.InputStream.inputStreamRead', this tries to read /@count@/ bytes from
-- the stream in a blocking fashion. However, rather than reading into
-- a user-supplied buffer, this will create a new t'GI.GLib.Structs.Bytes.Bytes' containing
-- the data that was read. This may be easier to use from language
-- bindings.
-- 
-- If count is zero, returns a zero-length t'GI.GLib.Structs.Bytes.Bytes' and does nothing. A
-- value of /@count@/ larger than @/G_MAXSSIZE/@ will cause a
-- 'GI.Gio.Enums.IOErrorEnumInvalidArgument' error.
-- 
-- On success, a new t'GI.GLib.Structs.Bytes.Bytes' is returned. It is not an error if the
-- size of this object is not the same as the requested size, as it
-- can happen e.g. near the end of a file. A zero-length t'GI.GLib.Structs.Bytes.Bytes' is
-- returned on end of file (or if /@count@/ is zero), but never
-- otherwise.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned. If an
-- operation was partially finished when the operation was cancelled the
-- partial result will be returned, without an error.
-- 
-- On error 'P.Nothing' is returned and /@error@/ is set accordingly.
-- 
-- /Since: 2.34/
inputStreamReadBytes ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.InputStream.InputStream'.
    -> FCT.CSize
    -- ^ /@count@/: maximum number of bytes that will be read from the stream. Common
    -- values include 4096 and 8192.
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> m GLib.Bytes.Bytes
    -- ^ __Returns:__ a new t'GI.GLib.Structs.Bytes.Bytes', or 'P.Nothing' on error /(Can throw 'Data.GI.Base.GError.GError')/
inputStreamReadBytes stream count cancellable = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_input_stream_read_bytes stream' count maybeCancellable
        checkUnexpectedReturnNULL "inputStreamReadBytes" result
        result' <- (wrapBoxed GLib.Bytes.Bytes) result
        touchManagedPtr stream
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data InputStreamReadBytesMethodInfo
instance (signature ~ (FCT.CSize -> Maybe (b) -> m GLib.Bytes.Bytes), MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod InputStreamReadBytesMethodInfo a signature where
    overloadedMethod = inputStreamReadBytes

instance O.OverloadedMethodInfo InputStreamReadBytesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamReadBytes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamReadBytes"
        })


#endif

-- method InputStream::read_bytes_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GInputStream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "count"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the number of bytes that will be read from the stream"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_input_stream_read_bytes_async" g_input_stream_read_bytes_async :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    FCT.CSize ->                            -- count : TBasicType TSize
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Request an asynchronous read of /@count@/ bytes from the stream into a
-- new t'GI.GLib.Structs.Bytes.Bytes'. When the operation is finished /@callback@/ will be
-- called. You can then call 'GI.Gio.Objects.InputStream.inputStreamReadBytesFinish' to get the
-- result of the operation.
-- 
-- During an async request no other sync and async calls are allowed
-- on /@stream@/, and will result in 'GI.Gio.Enums.IOErrorEnumPending' errors.
-- 
-- A value of /@count@/ larger than @/G_MAXSSIZE/@ will cause a
-- 'GI.Gio.Enums.IOErrorEnumInvalidArgument' error.
-- 
-- On success, the new t'GI.GLib.Structs.Bytes.Bytes' will be passed to the callback. It is
-- not an error if this is smaller than the requested size, as it can
-- happen e.g. near the end of a file, but generally we try to read as
-- many bytes as requested. Zero is returned on end of file (or if
-- /@count@/ is zero), but never otherwise.
-- 
-- Any outstanding I\/O request with higher priority (lower numerical
-- value) will be executed before an outstanding request with lower
-- priority. Default priority is 'GI.GLib.Constants.PRIORITY_DEFAULT'.
-- 
-- /Since: 2.34/
inputStreamReadBytesAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@stream@/: A t'GI.Gio.Objects.InputStream.InputStream'.
    -> FCT.CSize
    -- ^ /@count@/: the number of bytes that will be read from the stream
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
inputStreamReadBytesAsync stream count ioPriority cancellable callback = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_input_stream_read_bytes_async stream' count ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr stream
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data InputStreamReadBytesAsyncMethodInfo
instance (signature ~ (FCT.CSize -> Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod InputStreamReadBytesAsyncMethodInfo a signature where
    overloadedMethod = inputStreamReadBytesAsync

instance O.OverloadedMethodInfo InputStreamReadBytesAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamReadBytesAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamReadBytesAsync"
        })


#endif

-- method InputStream::read_bytes_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GInputStream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Bytes" })
-- throws : True
-- Skip return : False

foreign import ccall "g_input_stream_read_bytes_finish" g_input_stream_read_bytes_finish :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr GLib.Bytes.Bytes)

-- | Finishes an asynchronous stream read-into-t'GI.GLib.Structs.Bytes.Bytes' operation.
-- 
-- /Since: 2.34/
inputStreamReadBytesFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.InputStream.InputStream'.
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'.
    -> m GLib.Bytes.Bytes
    -- ^ __Returns:__ the newly-allocated t'GI.GLib.Structs.Bytes.Bytes', or 'P.Nothing' on error /(Can throw 'Data.GI.Base.GError.GError')/
inputStreamReadBytesFinish stream result_ = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        result <- propagateGError $ g_input_stream_read_bytes_finish stream' result_'
        checkUnexpectedReturnNULL "inputStreamReadBytesFinish" result
        result' <- (wrapBoxed GLib.Bytes.Bytes) result
        touchManagedPtr stream
        touchManagedPtr result_
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data InputStreamReadBytesFinishMethodInfo
instance (signature ~ (b -> m GLib.Bytes.Bytes), MonadIO m, IsInputStream a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod InputStreamReadBytesFinishMethodInfo a signature where
    overloadedMethod = inputStreamReadBytesFinish

instance O.OverloadedMethodInfo InputStreamReadBytesFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamReadBytesFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamReadBytesFinish"
        })


#endif

-- method InputStream::read_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GInputStream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TSSize)
-- throws : True
-- Skip return : False

foreign import ccall "g_input_stream_read_finish" g_input_stream_read_finish :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO DI.Int64

-- | Finishes an asynchronous stream read operation.
inputStreamReadFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.InputStream.InputStream'.
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'.
    -> m DI.Int64
    -- ^ __Returns:__ number of bytes read in, or -1 on error, or 0 on end of file. /(Can throw 'Data.GI.Base.GError.GError')/
inputStreamReadFinish stream result_ = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        result <- propagateGError $ g_input_stream_read_finish stream' result_'
        touchManagedPtr stream
        touchManagedPtr result_
        return result
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data InputStreamReadFinishMethodInfo
instance (signature ~ (b -> m DI.Int64), MonadIO m, IsInputStream a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod InputStreamReadFinishMethodInfo a signature where
    overloadedMethod = inputStreamReadFinish

instance O.OverloadedMethodInfo InputStreamReadFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamReadFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamReadFinish"
        })


#endif

-- method InputStream::set_pending
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "input stream" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_input_stream_set_pending" g_input_stream_set_pending :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Sets /@stream@/ to have actions pending. If the pending flag is
-- already set or /@stream@/ is closed, it will return 'P.False' and set
-- /@error@/.
inputStreamSetPending ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a) =>
    a
    -- ^ /@stream@/: input stream
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
inputStreamSetPending stream = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    onException (do
        _ <- propagateGError $ g_input_stream_set_pending stream'
        touchManagedPtr stream
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data InputStreamSetPendingMethodInfo
instance (signature ~ (m ()), MonadIO m, IsInputStream a) => O.OverloadedMethod InputStreamSetPendingMethodInfo a signature where
    overloadedMethod = inputStreamSetPending

instance O.OverloadedMethodInfo InputStreamSetPendingMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamSetPending",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamSetPending"
        })


#endif

-- method InputStream::skip
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GInputStream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "count"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the number of bytes that will be skipped from the stream"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TSSize)
-- throws : True
-- Skip return : False

foreign import ccall "g_input_stream_skip" g_input_stream_skip :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    FCT.CSize ->                            -- count : TBasicType TSize
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO DI.Int64

-- | Tries to skip /@count@/ bytes from the stream. Will block during the operation.
-- 
-- This is identical to 'GI.Gio.Objects.InputStream.inputStreamRead', from a behaviour standpoint,
-- but the bytes that are skipped are not returned to the user. Some
-- streams have an implementation that is more efficient than reading the data.
-- 
-- This function is optional for inherited classes, as the default implementation
-- emulates it using read.
-- 
-- If /@cancellable@/ is not 'P.Nothing', then the operation can be cancelled by
-- triggering the cancellable object from another thread. If the operation
-- was cancelled, the error 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned. If an
-- operation was partially finished when the operation was cancelled the
-- partial result will be returned, without an error.
inputStreamSkip ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.InputStream.InputStream'.
    -> FCT.CSize
    -- ^ /@count@/: the number of bytes that will be skipped from the stream
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> m DI.Int64
    -- ^ __Returns:__ Number of bytes skipped, or -1 on error /(Can throw 'Data.GI.Base.GError.GError')/
inputStreamSkip stream count cancellable = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_input_stream_skip stream' count maybeCancellable
        touchManagedPtr stream
        whenJust cancellable touchManagedPtr
        return result
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data InputStreamSkipMethodInfo
instance (signature ~ (FCT.CSize -> Maybe (b) -> m DI.Int64), MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod InputStreamSkipMethodInfo a signature where
    overloadedMethod = inputStreamSkip

instance O.OverloadedMethodInfo InputStreamSkipMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamSkip",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamSkip"
        })


#endif

-- method InputStream::skip_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GInputStream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "count"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the number of bytes that will be skipped from the stream"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the request"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback\n  to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_input_stream_skip_async" g_input_stream_skip_async :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    FCT.CSize ->                            -- count : TBasicType TSize
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Request an asynchronous skip of /@count@/ bytes from the stream.
-- When the operation is finished /@callback@/ will be called.
-- You can then call 'GI.Gio.Objects.InputStream.inputStreamSkipFinish' to get the result
-- of the operation.
-- 
-- During an async request no other sync and async calls are allowed,
-- and will result in 'GI.Gio.Enums.IOErrorEnumPending' errors.
-- 
-- A value of /@count@/ larger than @/G_MAXSSIZE/@ will cause a 'GI.Gio.Enums.IOErrorEnumInvalidArgument' error.
-- 
-- On success, the number of bytes skipped will be passed to the callback.
-- It is not an error if this is not the same as the requested size, as it
-- can happen e.g. near the end of a file, but generally we try to skip
-- as many bytes as requested. Zero is returned on end of file
-- (or if /@count@/ is zero), but never otherwise.
-- 
-- Any outstanding i\/o request with higher priority (lower numerical value)
-- will be executed before an outstanding request with lower priority.
-- Default priority is 'GI.GLib.Constants.PRIORITY_DEFAULT'.
-- 
-- The asynchronous methods have a default fallback that uses threads to
-- implement asynchronicity, so they are optional for inheriting classes.
-- However, if you override one, you must override all.
inputStreamSkipAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@stream@/: A t'GI.Gio.Objects.InputStream.InputStream'.
    -> FCT.CSize
    -- ^ /@count@/: the number of bytes that will be skipped from the stream
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the request
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback'
    --   to call when the request is satisfied
    -> m ()
inputStreamSkipAsync stream count ioPriority cancellable callback = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_input_stream_skip_async stream' count ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr stream
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data InputStreamSkipAsyncMethodInfo
instance (signature ~ (FCT.CSize -> Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsInputStream a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod InputStreamSkipAsyncMethodInfo a signature where
    overloadedMethod = inputStreamSkipAsync

instance O.OverloadedMethodInfo InputStreamSkipAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamSkipAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamSkipAsync"
        })


#endif

-- method InputStream::skip_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "stream"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InputStream" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GInputStream." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TSSize)
-- throws : True
-- Skip return : False

foreign import ccall "g_input_stream_skip_finish" g_input_stream_skip_finish :: 
    Ptr InputStream ->                      -- stream : TInterface (Name {namespace = "Gio", name = "InputStream"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO DI.Int64

-- | Finishes a stream skip operation.
inputStreamSkipFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsInputStream a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@stream@/: a t'GI.Gio.Objects.InputStream.InputStream'.
    -> b
    -- ^ /@result@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'.
    -> m DI.Int64
    -- ^ __Returns:__ the size of the bytes skipped, or @-1@ on error. /(Can throw 'Data.GI.Base.GError.GError')/
inputStreamSkipFinish stream result_ = liftIO $ do
    stream' <- unsafeManagedPtrCastPtr stream
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        result <- propagateGError $ g_input_stream_skip_finish stream' result_'
        touchManagedPtr stream
        touchManagedPtr result_
        return result
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data InputStreamSkipFinishMethodInfo
instance (signature ~ (b -> m DI.Int64), MonadIO m, IsInputStream a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod InputStreamSkipFinishMethodInfo a signature where
    overloadedMethod = inputStreamSkipFinish

instance O.OverloadedMethodInfo InputStreamSkipFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.InputStream.inputStreamSkipFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-InputStream.html#v:inputStreamSkipFinish"
        })


#endif


