-- Copyright (C) 2010-2011 John Millikin <jmillikin@gmail.com>
-- 
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- any later version.
-- 
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
-- 
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

module Network.Protocol.XMPP.Connections
	( Server (..)
	, xmlHeader
	, startOfStream
	, qnameStream
	, connectTo
	) where

import           Network.Socket (HostName, PortNumber, socketToHandle)
import           Network.Simple.TCP (connectSock)
import           Data.ByteString (ByteString)
import qualified Data.Text
import           Data.Text (Text)
import           Data.Text.Encoding (encodeUtf8)
import qualified System.IO as IO

import qualified Network.Protocol.XMPP.XML as X
import           Network.Protocol.XMPP.JID (JID, formatJID)
import           Network.Protocol.XMPP.String (s)

connectTo :: HostName -> PortNumber -> IO IO.Handle
connectTo host port =
	(`socketToHandle` IO.ReadWriteMode) =<<
	(fst <$> connectSock host (show port))

data Server = Server
	{ serverJID      :: JID
	, serverHostname :: HostName
	, serverPort     :: PortNumber
	}

-- Since only the opening tag should be written, normal XML
-- serialization cannot be used. Be careful to escape any embedded
-- attributes.
xmlHeader :: Text -> JID -> ByteString
xmlHeader ns jid = encodeUtf8 header where
	attr x = Data.Text.concat [s"\"", X.escape x, s"\""]
	header = Data.Text.concat
		[ s"<?xml version='1.0'?>\n"
		, s"<stream:stream xmlns=" , attr ns
		, s" to=", attr (formatJID jid)
		, s" version=\"1.0\""
		, s" xmlns:stream=\"http://etherx.jabber.org/streams\">"
		]

startOfStream :: Integer -> X.Event -> Bool
startOfStream depth event = case (depth, event) of
	(1, X.EventBeginElement elemName _) -> qnameStream == elemName
	_ -> False

qnameStream :: X.Name
qnameStream = s"{http://etherx.jabber.org/streams}stream"
