use super::GFeedID;
use glib::{Object, Properties, prelude::*};
use gtk4::subclass::prelude::*;
use news_flash::models::{Feed, Url};
use std::cell::{Cell, RefCell};
use std::fmt::Display;

mod imp {
    use super::*;

    #[derive(Default, Properties)]
    #[properties(wrapper_type = super::GFeed)]
    pub struct GFeed {
        #[property(get, set, name = "feed-id")]
        pub id: RefCell<GFeedID>,
        #[property(get, set)]
        pub label: RefCell<String>,
        #[property(get, set, nullable)]
        pub website: RefCell<Option<String>>,
        #[property(get, set, nullable, name = "feed-url")]
        pub feed_url: RefCell<Option<String>>,
        #[property(get, set, nullable, name = "icon-url")]
        pub icon_url: RefCell<Option<String>>,
        #[property(get, set, name = "error-count")]
        pub error_count: Cell<i32>,
        #[property(get, set, nullable, name = "error-message")]
        pub error_message: RefCell<Option<String>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for GFeed {
        const NAME: &'static str = "NewsFlashGFeed";
        type Type = super::GFeed;
    }

    #[glib::derived_properties]
    impl ObjectImpl for GFeed {}
}

glib::wrapper! {
    pub struct GFeed(ObjectSubclass<imp::GFeed>);
}

impl Default for GFeed {
    fn default() -> Self {
        Object::new()
    }
}

impl Display for GFeed {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "(id: {}, label: {})", self.feed_id(), self.label())
    }
}

impl From<Feed> for GFeed {
    fn from(value: Feed) -> Self {
        let obj = GFeed::default();
        let imp = obj.imp();
        imp.id.replace(value.feed_id.into());
        imp.label.replace(value.label);
        imp.website.replace(value.website.as_ref().map(ToString::to_string));
        imp.feed_url.replace(value.feed_url.as_ref().map(ToString::to_string));
        imp.icon_url.replace(value.icon_url.as_ref().map(ToString::to_string));
        imp.error_count.set(value.error_count);
        imp.error_message.replace(value.error_message);
        obj
    }
}

impl From<GFeed> for Feed {
    fn from(value: GFeed) -> Self {
        Feed {
            feed_id: value.feed_id().into(),
            label: value.label(),
            website: value.website().map(|url| Url::parse(&url).unwrap()),
            feed_url: value.feed_url().map(|url| Url::parse(&url).unwrap()),
            icon_url: value.icon_url().map(|url| Url::parse(&url).unwrap()),
            error_count: value.error_count(),
            error_message: value.error_message(),
        }
    }
}
