use glib::{Properties, prelude::*, subclass::prelude::*};
use serde::{Deserialize, Serialize};
use std::cell::{Cell, RefCell};

mod imp {
    use super::*;

    #[derive(Debug, Serialize, Deserialize, Properties)]
    #[properties(wrapper_type = super::ShareSettings)]
    pub struct ShareSettings {
        #[property(get, set, name = "instapaper-enabled", default = true)]
        #[serde(default)]
        pub instapaper_enabled: Cell<bool>,

        #[property(get, set, name = "mastodon-enabled", default = true)]
        #[serde(default)]
        pub mastodon_enabled: Cell<bool>,

        #[property(get, set, name = "reddit-enabled", default = true)]
        #[serde(default)]
        pub reddit_enabled: Cell<bool>,

        #[property(get, set, name = "telegram-enabled", default = true)]
        #[serde(default)]
        pub telegram_enabled: Cell<bool>,

        #[property(get, set, name = "bluesky-enabled", default = true)]
        #[serde(default)]
        pub bluesky_enabled: Cell<bool>,

        #[property(get, set, name = "buffer-enabled", default = true)]
        #[serde(default)]
        pub buffer_enabled: Cell<bool>,

        #[property(get, set, name = "custom-enabled", default = false)]
        #[serde(default)]
        pub custom_enabled: Cell<bool>,

        #[property(get, set, name = "custom-name", nullable)]
        #[serde(default)]
        pub custom_name: RefCell<Option<String>>,

        #[property(get, set, name = "custom-url", nullable)]
        #[serde(default)]
        pub custom_url: RefCell<Option<String>>,
    }

    impl Default for ShareSettings {
        fn default() -> Self {
            Self {
                instapaper_enabled: Cell::new(true),
                mastodon_enabled: Cell::new(true),
                reddit_enabled: Cell::new(true),
                telegram_enabled: Cell::new(true),
                bluesky_enabled: Cell::new(true),
                buffer_enabled: Cell::new(true),

                custom_enabled: Cell::new(false),
                custom_name: RefCell::new(None),
                custom_url: RefCell::new(None),
            }
        }
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ShareSettings {
        const NAME: &'static str = "ShareSettings";
        type Type = super::ShareSettings;
    }

    #[glib::derived_properties]
    impl ObjectImpl for ShareSettings {}
}

glib::wrapper! {
    pub struct ShareSettings(ObjectSubclass<imp::ShareSettings>);
}

impl Serialize for ShareSettings {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        self.imp().serialize(serializer)
    }
}

impl<'de> Deserialize<'de> for ShareSettings {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        let inner = imp::ShareSettings::deserialize(deserializer)?;
        Ok(inner.into())
    }
}

impl From<imp::ShareSettings> for ShareSettings {
    fn from(inner: imp::ShareSettings) -> Self {
        glib::Object::builder()
            .property("instapaper-enabled", inner.instapaper_enabled.get())
            .property("mastodon-enabled", inner.mastodon_enabled.get())
            .property("reddit-enabled", inner.reddit_enabled.get())
            .property("telegram-enabled", inner.telegram_enabled.get())
            .property("bluesky-enabled", inner.bluesky_enabled.get())
            .property("buffer-enabled", inner.buffer_enabled.get())
            .property("custom-enabled", inner.custom_enabled.get())
            .property("custom-name", inner.custom_name.borrow().clone())
            .property("custom-url", inner.custom_url.borrow().clone())
            .build()
    }
}

impl Default for ShareSettings {
    fn default() -> Self {
        imp::ShareSettings::default().into()
    }
}
