// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dk_dv.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dk_dv__A6__F236

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_3_T_F_0___gfx120x)xyzw" };
#else
{ R"xyzw(amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_3_T_F_0___gfx120x)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_48_3_T_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1201)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x1348badau, 0xa303379eu, 0, 6 }, // 1348badaa303379e = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_3_T_F_0___gfx120x__P__16_16__CO__wave2_warp2_stg1--Arch_gfx1201
    { 0x6bca2ef1u, 0x8e246323u, 0, 46 }, // 6bca2ef18e246323 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_3_T_F_0___gfx120x__P__16_16__CO__wave3_warp2_stg1--Arch_gfx1201
    { 0x42b26fe4u, 0x0fefbdf3u, 0, 63 }, // 42b26fe40fefbdf3 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_3_T_F_0___gfx120x__P__16_16__CO__wave4_warp2_stg1--Arch_gfx1201
    { 0x4697b2e9u, 0x72875eacu, 23, 29 }, // 4697b2e972875eac = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_3_T_F_0___gfx120x__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1201
    { 0x04c7a42bu, 0x5892ff30u, 23, 6 }, // 04c7a42b5892ff30 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_3_T_F_0___gfx120x__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1201
    { 0xaf59a41du, 0x563ef61fu, 23, 46 }, // af59a41d563ef61f = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_3_T_F_0___gfx120x__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1201
    { 0x83a03e8bu, 0x11e2a229u, 23, 63 }, // 83a03e8b11e2a229 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_3_T_F_0___gfx120x__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1201
    { 0xfe6db334u, 0x18f5a125u, 103, 109 }, // fe6db33418f5a125 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_3_T_F_0___gfx120x__P__64_16__CO__wave1_warp4_stg1--Arch_gfx1201
    { 0xcbb443b9u, 0xc9ed4c3cu, 103, 126 }, // cbb443b9c9ed4c3c = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_3_T_F_0___gfx120x__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1201
    { 0x6f986237u, 0x816e2e7bu, 103, 86 }, // 6f986237816e2e7b = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_3_T_F_0___gfx120x__P__64_16__CO__wave3_warp4_stg1--Arch_gfx1201
    { 0x7391bfceu, 0x39500f9fu, 160, 109 }, // 7391bfce39500f9f = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_3_T_F_0___gfx120x__P__64_32__CO__wave1_warp4_stg1--Arch_gfx1201
    { 0x1aab0907u, 0x5fbfa56fu, 160, 126 }, // 1aab09075fbfa56f = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_3_T_F_0___gfx120x__P__64_32__CO__wave2_warp4_stg1--Arch_gfx1201
    { 0xa454bde8u, 0xfc166d27u, 160, 86 }, // a454bde8fc166d27 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊bf16@16_48_3_T_F_0___gfx120x__P__64_32__CO__wave3_warp4_stg1--Arch_gfx1201
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dk_dv_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dk_dv_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1201_mod0
{{ 1, 2, 2, 0, 2, 2, 2, 2, 2, 2},
 { 6, 6, 3, 4, 2, 4, 5, 6, 6, 6},
 { 8, 6, 6, 6, 5, 6, 6, 6, 5, 6},
 { 7, 5, 5, 5, 5, 5, 6, 6, 6, 5},
 { 9, 4, 6, 4, 5, 4, 6, 6, 6, 6},
 { 7,10, 6, 5, 5, 6, 6, 4, 5, 3},
 { 7,10, 5,10,11,12, 5, 4,11,10},
 { 7,11, 5,11,11,11,12,12,11,10},
 { 7,11, 4,10,12,10,11,11,11,10},
 { 7,11, 6,10,12,12,11,10,11,10}}
// End of GPU gfx1201_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDkDvContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDkDvContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dk_dv__A6__F236 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 5;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dk_dv__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 5;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = "i32"
// BLOCK_DMODEL = 48
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = True
// PADDED_HEAD = False
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dk_dv WHERE gpu IN ('gfx1201_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 48 AND inputs$CAUSAL_TYPE = 3 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

