﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/guardduty/GuardDuty_EXPORTS.h>
#include <aws/guardduty/model/GetMalwareScanResultDetails.h>
#include <aws/guardduty/model/MalwareProtectionResourceType.h>
#include <aws/guardduty/model/MalwareProtectionScanStatus.h>
#include <aws/guardduty/model/MalwareProtectionScanType.h>
#include <aws/guardduty/model/ScanCategory.h>
#include <aws/guardduty/model/ScanConfiguration.h>
#include <aws/guardduty/model/ScanStatusReason.h>
#include <aws/guardduty/model/ScannedResource.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Json {
class JsonValue;
}  // namespace Json
}  // namespace Utils
namespace GuardDuty {
namespace Model {
class GetMalwareScanResult {
 public:
  AWS_GUARDDUTY_API GetMalwareScanResult() = default;
  AWS_GUARDDUTY_API GetMalwareScanResult(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);
  AWS_GUARDDUTY_API GetMalwareScanResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);

  ///@{
  /**
   * <p>A unique identifier associated with the malware scan. Each malware scan has a
   * corresponding scan ID. Using this scan ID, you can monitor the status of your
   * malware scan.</p>
   */
  inline const Aws::String& GetScanId() const { return m_scanId; }
  template <typename ScanIdT = Aws::String>
  void SetScanId(ScanIdT&& value) {
    m_scanIdHasBeenSet = true;
    m_scanId = std::forward<ScanIdT>(value);
  }
  template <typename ScanIdT = Aws::String>
  GetMalwareScanResult& WithScanId(ScanIdT&& value) {
    SetScanId(std::forward<ScanIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique ID of the detector that is associated with the request, if it
   * belongs to an account which is a GuardDuty customer.</p> <p>To find the
   * <code>detectorId</code> in the current Region, see the Settings page in the
   * GuardDuty console, or run the <a
   * href="https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListDetectors.html">ListDetectors</a>
   * API.</p>
   */
  inline const Aws::String& GetDetectorId() const { return m_detectorId; }
  template <typename DetectorIdT = Aws::String>
  void SetDetectorId(DetectorIdT&& value) {
    m_detectorIdHasBeenSet = true;
    m_detectorId = std::forward<DetectorIdT>(value);
  }
  template <typename DetectorIdT = Aws::String>
  GetMalwareScanResult& WithDetectorId(DetectorIdT&& value) {
    SetDetectorId(std::forward<DetectorIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique detector ID of the administrator account that the request is
   * associated with. If the account is an administrator, the
   * <code>AdminDetectorId</code> will be the same as the one used for
   * <code>DetectorId. If the customer is not a GuardDuty customer, this field will
   * not be present.</code>.</p> <p>To find the <code>detectorId</code> in the
   * current Region, see the Settings page in the GuardDuty console, or run the <a
   * href="https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListDetectors.html">ListDetectors</a>
   * API.</p>
   */
  inline const Aws::String& GetAdminDetectorId() const { return m_adminDetectorId; }
  template <typename AdminDetectorIdT = Aws::String>
  void SetAdminDetectorId(AdminDetectorIdT&& value) {
    m_adminDetectorIdHasBeenSet = true;
    m_adminDetectorId = std::forward<AdminDetectorIdT>(value);
  }
  template <typename AdminDetectorIdT = Aws::String>
  GetMalwareScanResult& WithAdminDetectorId(AdminDetectorIdT&& value) {
    SetAdminDetectorId(std::forward<AdminDetectorIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Amazon Resource Name (ARN) of the resource on which a malware scan was
   * invoked.</p>
   */
  inline const Aws::String& GetResourceArn() const { return m_resourceArn; }
  template <typename ResourceArnT = Aws::String>
  void SetResourceArn(ResourceArnT&& value) {
    m_resourceArnHasBeenSet = true;
    m_resourceArn = std::forward<ResourceArnT>(value);
  }
  template <typename ResourceArnT = Aws::String>
  GetMalwareScanResult& WithResourceArn(ResourceArnT&& value) {
    SetResourceArn(std::forward<ResourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of resource that was scanned for malware.</p>
   */
  inline MalwareProtectionResourceType GetResourceType() const { return m_resourceType; }
  inline void SetResourceType(MalwareProtectionResourceType value) {
    m_resourceTypeHasBeenSet = true;
    m_resourceType = value;
  }
  inline GetMalwareScanResult& WithResourceType(MalwareProtectionResourceType value) {
    SetResourceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The total number of resources that were successfully scanned. This is
   * dependent on the resource type.</p>
   */
  inline int GetScannedResourcesCount() const { return m_scannedResourcesCount; }
  inline void SetScannedResourcesCount(int value) {
    m_scannedResourcesCountHasBeenSet = true;
    m_scannedResourcesCount = value;
  }
  inline GetMalwareScanResult& WithScannedResourcesCount(int value) {
    SetScannedResourcesCount(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The total number of resources that were skipped during the scan.</p>
   */
  inline int GetSkippedResourcesCount() const { return m_skippedResourcesCount; }
  inline void SetSkippedResourcesCount(int value) {
    m_skippedResourcesCountHasBeenSet = true;
    m_skippedResourcesCount = value;
  }
  inline GetMalwareScanResult& WithSkippedResourcesCount(int value) {
    SetSkippedResourcesCount(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The total number of resources that failed to be scanned.</p>
   */
  inline int GetFailedResourcesCount() const { return m_failedResourcesCount; }
  inline void SetFailedResourcesCount(int value) {
    m_failedResourcesCountHasBeenSet = true;
    m_failedResourcesCount = value;
  }
  inline GetMalwareScanResult& WithFailedResourcesCount(int value) {
    SetFailedResourcesCount(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of resources along with their metadata that were scanned as part of
   * the malware scan operation.</p>
   */
  inline const Aws::Vector<ScannedResource>& GetScannedResources() const { return m_scannedResources; }
  template <typename ScannedResourcesT = Aws::Vector<ScannedResource>>
  void SetScannedResources(ScannedResourcesT&& value) {
    m_scannedResourcesHasBeenSet = true;
    m_scannedResources = std::forward<ScannedResourcesT>(value);
  }
  template <typename ScannedResourcesT = Aws::Vector<ScannedResource>>
  GetMalwareScanResult& WithScannedResources(ScannedResourcesT&& value) {
    SetScannedResources(std::forward<ScannedResourcesT>(value));
    return *this;
  }
  template <typename ScannedResourcesT = ScannedResource>
  GetMalwareScanResult& AddScannedResources(ScannedResourcesT&& value) {
    m_scannedResourcesHasBeenSet = true;
    m_scannedResources.emplace_back(std::forward<ScannedResourcesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the scan configuration used for the malware scan.</p>
   */
  inline const ScanConfiguration& GetScanConfiguration() const { return m_scanConfiguration; }
  template <typename ScanConfigurationT = ScanConfiguration>
  void SetScanConfiguration(ScanConfigurationT&& value) {
    m_scanConfigurationHasBeenSet = true;
    m_scanConfiguration = std::forward<ScanConfigurationT>(value);
  }
  template <typename ScanConfigurationT = ScanConfiguration>
  GetMalwareScanResult& WithScanConfiguration(ScanConfigurationT&& value) {
    SetScanConfiguration(std::forward<ScanConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The category of the malware scan, indicating the type of scan performed.</p>
   */
  inline ScanCategory GetScanCategory() const { return m_scanCategory; }
  inline void SetScanCategory(ScanCategory value) {
    m_scanCategoryHasBeenSet = true;
    m_scanCategory = value;
  }
  inline GetMalwareScanResult& WithScanCategory(ScanCategory value) {
    SetScanCategory(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value representing the current status of the malware scan.</p>
   */
  inline MalwareProtectionScanStatus GetScanStatus() const { return m_scanStatus; }
  inline void SetScanStatus(MalwareProtectionScanStatus value) {
    m_scanStatusHasBeenSet = true;
    m_scanStatus = value;
  }
  inline GetMalwareScanResult& WithScanStatus(MalwareProtectionScanStatus value) {
    SetScanStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Represents the reason for the current scan status, if applicable.</p>
   */
  inline ScanStatusReason GetScanStatusReason() const { return m_scanStatusReason; }
  inline void SetScanStatusReason(ScanStatusReason value) {
    m_scanStatusReasonHasBeenSet = true;
    m_scanStatusReason = value;
  }
  inline GetMalwareScanResult& WithScanStatusReason(ScanStatusReason value) {
    SetScanStatusReason(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value representing the initiator of the scan.</p>
   */
  inline MalwareProtectionScanType GetScanType() const { return m_scanType; }
  inline void SetScanType(MalwareProtectionScanType value) {
    m_scanTypeHasBeenSet = true;
    m_scanType = value;
  }
  inline GetMalwareScanResult& WithScanType(MalwareProtectionScanType value) {
    SetScanType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp representing when the malware scan was started.</p>
   */
  inline const Aws::Utils::DateTime& GetScanStartedAt() const { return m_scanStartedAt; }
  template <typename ScanStartedAtT = Aws::Utils::DateTime>
  void SetScanStartedAt(ScanStartedAtT&& value) {
    m_scanStartedAtHasBeenSet = true;
    m_scanStartedAt = std::forward<ScanStartedAtT>(value);
  }
  template <typename ScanStartedAtT = Aws::Utils::DateTime>
  GetMalwareScanResult& WithScanStartedAt(ScanStartedAtT&& value) {
    SetScanStartedAt(std::forward<ScanStartedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp representing when the malware scan was completed.</p>
   */
  inline const Aws::Utils::DateTime& GetScanCompletedAt() const { return m_scanCompletedAt; }
  template <typename ScanCompletedAtT = Aws::Utils::DateTime>
  void SetScanCompletedAt(ScanCompletedAtT&& value) {
    m_scanCompletedAtHasBeenSet = true;
    m_scanCompletedAt = std::forward<ScanCompletedAtT>(value);
  }
  template <typename ScanCompletedAtT = Aws::Utils::DateTime>
  GetMalwareScanResult& WithScanCompletedAt(ScanCompletedAtT&& value) {
    SetScanCompletedAt(std::forward<ScanCompletedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Detailed information about the results of the malware scan, if the scan
   * completed.</p>
   */
  inline const GetMalwareScanResultDetails& GetScanResultDetails() const { return m_scanResultDetails; }
  template <typename ScanResultDetailsT = GetMalwareScanResultDetails>
  void SetScanResultDetails(ScanResultDetailsT&& value) {
    m_scanResultDetailsHasBeenSet = true;
    m_scanResultDetails = std::forward<ScanResultDetailsT>(value);
  }
  template <typename ScanResultDetailsT = GetMalwareScanResultDetails>
  GetMalwareScanResult& WithScanResultDetails(ScanResultDetailsT&& value) {
    SetScanResultDetails(std::forward<ScanResultDetailsT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  GetMalwareScanResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_scanId;

  Aws::String m_detectorId;

  Aws::String m_adminDetectorId;

  Aws::String m_resourceArn;

  MalwareProtectionResourceType m_resourceType{MalwareProtectionResourceType::NOT_SET};

  int m_scannedResourcesCount{0};

  int m_skippedResourcesCount{0};

  int m_failedResourcesCount{0};

  Aws::Vector<ScannedResource> m_scannedResources;

  ScanConfiguration m_scanConfiguration;

  ScanCategory m_scanCategory{ScanCategory::NOT_SET};

  MalwareProtectionScanStatus m_scanStatus{MalwareProtectionScanStatus::NOT_SET};

  ScanStatusReason m_scanStatusReason{ScanStatusReason::NOT_SET};

  MalwareProtectionScanType m_scanType{MalwareProtectionScanType::NOT_SET};

  Aws::Utils::DateTime m_scanStartedAt{};

  Aws::Utils::DateTime m_scanCompletedAt{};

  GetMalwareScanResultDetails m_scanResultDetails;

  Aws::String m_requestId;
  bool m_scanIdHasBeenSet = false;
  bool m_detectorIdHasBeenSet = false;
  bool m_adminDetectorIdHasBeenSet = false;
  bool m_resourceArnHasBeenSet = false;
  bool m_resourceTypeHasBeenSet = false;
  bool m_scannedResourcesCountHasBeenSet = false;
  bool m_skippedResourcesCountHasBeenSet = false;
  bool m_failedResourcesCountHasBeenSet = false;
  bool m_scannedResourcesHasBeenSet = false;
  bool m_scanConfigurationHasBeenSet = false;
  bool m_scanCategoryHasBeenSet = false;
  bool m_scanStatusHasBeenSet = false;
  bool m_scanStatusReasonHasBeenSet = false;
  bool m_scanTypeHasBeenSet = false;
  bool m_scanStartedAtHasBeenSet = false;
  bool m_scanCompletedAtHasBeenSet = false;
  bool m_scanResultDetailsHasBeenSet = false;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace GuardDuty
}  // namespace Aws
