﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/medialive/MediaLiveRequest.h>
#include <aws/medialive/MediaLive_EXPORTS.h>
#include <aws/medialive/model/IpPoolCreateRequest.h>
#include <aws/medialive/model/RouteCreateRequest.h>

#include <utility>

namespace Aws {
namespace MediaLive {
namespace Model {

/**
 * A request to create a Network.<p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/medialive-2017-10-14/CreateNetworkRequest">AWS
 * API Reference</a></p>
 */
class CreateNetworkRequest : public MediaLiveRequest {
 public:
  AWS_MEDIALIVE_API CreateNetworkRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateNetwork"; }

  AWS_MEDIALIVE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * An array of IpPoolCreateRequests that identify a collection of IP addresses in
   * your network that you want to reserve for use in MediaLive Anywhere.
   * MediaLiveAnywhere uses these IP addresses for Push inputs (in both Bridge and
   * NATnetworks) and for output destinations (only in Bridge networks).
   * EachIpPoolUpdateRequest specifies one CIDR block.
   */
  inline const Aws::Vector<IpPoolCreateRequest>& GetIpPools() const { return m_ipPools; }
  inline bool IpPoolsHasBeenSet() const { return m_ipPoolsHasBeenSet; }
  template <typename IpPoolsT = Aws::Vector<IpPoolCreateRequest>>
  void SetIpPools(IpPoolsT&& value) {
    m_ipPoolsHasBeenSet = true;
    m_ipPools = std::forward<IpPoolsT>(value);
  }
  template <typename IpPoolsT = Aws::Vector<IpPoolCreateRequest>>
  CreateNetworkRequest& WithIpPools(IpPoolsT&& value) {
    SetIpPools(std::forward<IpPoolsT>(value));
    return *this;
  }
  template <typename IpPoolsT = IpPoolCreateRequest>
  CreateNetworkRequest& AddIpPools(IpPoolsT&& value) {
    m_ipPoolsHasBeenSet = true;
    m_ipPools.emplace_back(std::forward<IpPoolsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * Specify a name that is unique in the AWS account. We recommend that you assign a
   * name that hints at the type of traffic on the network. Names are case-sensitive.
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateNetworkRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * An ID that you assign to a create request. This ID ensures idempotency when
   * creating resources.
   */
  inline const Aws::String& GetRequestId() const { return m_requestId; }
  inline bool RequestIdHasBeenSet() const { return m_requestIdHasBeenSet; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  CreateNetworkRequest& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * An array of routes that MediaLive Anywhere needs to know about in order to route
   * encoding traffic.
   */
  inline const Aws::Vector<RouteCreateRequest>& GetRoutes() const { return m_routes; }
  inline bool RoutesHasBeenSet() const { return m_routesHasBeenSet; }
  template <typename RoutesT = Aws::Vector<RouteCreateRequest>>
  void SetRoutes(RoutesT&& value) {
    m_routesHasBeenSet = true;
    m_routes = std::forward<RoutesT>(value);
  }
  template <typename RoutesT = Aws::Vector<RouteCreateRequest>>
  CreateNetworkRequest& WithRoutes(RoutesT&& value) {
    SetRoutes(std::forward<RoutesT>(value));
    return *this;
  }
  template <typename RoutesT = RouteCreateRequest>
  CreateNetworkRequest& AddRoutes(RoutesT&& value) {
    m_routesHasBeenSet = true;
    m_routes.emplace_back(std::forward<RoutesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * A collection of key-value pairs.
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateNetworkRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateNetworkRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<IpPoolCreateRequest> m_ipPools;

  Aws::String m_name;

  Aws::String m_requestId{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::Vector<RouteCreateRequest> m_routes;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_ipPoolsHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_requestIdHasBeenSet = true;
  bool m_routesHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace MediaLive
}  // namespace Aws
