﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/neptune/Neptune_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace Neptune {
namespace Model {

/**
 * <p>The version of the database engine that a DB instance can be upgraded
 * to.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/neptune-2014-10-31/UpgradeTarget">AWS
 * API Reference</a></p>
 */
class UpgradeTarget {
 public:
  AWS_NEPTUNE_API UpgradeTarget() = default;
  AWS_NEPTUNE_API UpgradeTarget(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_NEPTUNE_API UpgradeTarget& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_NEPTUNE_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_NEPTUNE_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The name of the upgrade target database engine.</p>
   */
  inline const Aws::String& GetEngine() const { return m_engine; }
  inline bool EngineHasBeenSet() const { return m_engineHasBeenSet; }
  template <typename EngineT = Aws::String>
  void SetEngine(EngineT&& value) {
    m_engineHasBeenSet = true;
    m_engine = std::forward<EngineT>(value);
  }
  template <typename EngineT = Aws::String>
  UpgradeTarget& WithEngine(EngineT&& value) {
    SetEngine(std::forward<EngineT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version number of the upgrade target database engine.</p>
   */
  inline const Aws::String& GetEngineVersion() const { return m_engineVersion; }
  inline bool EngineVersionHasBeenSet() const { return m_engineVersionHasBeenSet; }
  template <typename EngineVersionT = Aws::String>
  void SetEngineVersion(EngineVersionT&& value) {
    m_engineVersionHasBeenSet = true;
    m_engineVersion = std::forward<EngineVersionT>(value);
  }
  template <typename EngineVersionT = Aws::String>
  UpgradeTarget& WithEngineVersion(EngineVersionT&& value) {
    SetEngineVersion(std::forward<EngineVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the database engine that a DB instance can be upgraded to.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpgradeTarget& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that indicates whether the target version is applied to any source DB
   * instances that have AutoMinorVersionUpgrade set to true.</p>
   */
  inline bool GetAutoUpgrade() const { return m_autoUpgrade; }
  inline bool AutoUpgradeHasBeenSet() const { return m_autoUpgradeHasBeenSet; }
  inline void SetAutoUpgrade(bool value) {
    m_autoUpgradeHasBeenSet = true;
    m_autoUpgrade = value;
  }
  inline UpgradeTarget& WithAutoUpgrade(bool value) {
    SetAutoUpgrade(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that indicates whether a database engine is upgraded to a major
   * version.</p>
   */
  inline bool GetIsMajorVersionUpgrade() const { return m_isMajorVersionUpgrade; }
  inline bool IsMajorVersionUpgradeHasBeenSet() const { return m_isMajorVersionUpgradeHasBeenSet; }
  inline void SetIsMajorVersionUpgrade(bool value) {
    m_isMajorVersionUpgradeHasBeenSet = true;
    m_isMajorVersionUpgrade = value;
  }
  inline UpgradeTarget& WithIsMajorVersionUpgrade(bool value) {
    SetIsMajorVersionUpgrade(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A value that indicates whether you can use Neptune global databases with the
   * target engine version.</p>
   */
  inline bool GetSupportsGlobalDatabases() const { return m_supportsGlobalDatabases; }
  inline bool SupportsGlobalDatabasesHasBeenSet() const { return m_supportsGlobalDatabasesHasBeenSet; }
  inline void SetSupportsGlobalDatabases(bool value) {
    m_supportsGlobalDatabasesHasBeenSet = true;
    m_supportsGlobalDatabases = value;
  }
  inline UpgradeTarget& WithSupportsGlobalDatabases(bool value) {
    SetSupportsGlobalDatabases(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_engine;

  Aws::String m_engineVersion;

  Aws::String m_description;

  bool m_autoUpgrade{false};

  bool m_isMajorVersionUpgrade{false};

  bool m_supportsGlobalDatabases{false};
  bool m_engineHasBeenSet = false;
  bool m_engineVersionHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_autoUpgradeHasBeenSet = false;
  bool m_isMajorVersionUpgradeHasBeenSet = false;
  bool m_supportsGlobalDatabasesHasBeenSet = false;
};

}  // namespace Model
}  // namespace Neptune
}  // namespace Aws
