﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kinesisanalyticsv2/KinesisAnalyticsV2Request.h>
#include <aws/kinesisanalyticsv2/KinesisAnalyticsV2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace KinesisAnalyticsV2 {
namespace Model {

/**
 */
class DeleteApplicationRequest : public KinesisAnalyticsV2Request {
 public:
  AWS_KINESISANALYTICSV2_API DeleteApplicationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteApplication"; }

  AWS_KINESISANALYTICSV2_API Aws::String SerializePayload() const override;

  AWS_KINESISANALYTICSV2_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the application to delete.</p>
   */
  inline const Aws::String& GetApplicationName() const { return m_applicationName; }
  inline bool ApplicationNameHasBeenSet() const { return m_applicationNameHasBeenSet; }
  template <typename ApplicationNameT = Aws::String>
  void SetApplicationName(ApplicationNameT&& value) {
    m_applicationNameHasBeenSet = true;
    m_applicationName = std::forward<ApplicationNameT>(value);
  }
  template <typename ApplicationNameT = Aws::String>
  DeleteApplicationRequest& WithApplicationName(ApplicationNameT&& value) {
    SetApplicationName(std::forward<ApplicationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Use the <code>DescribeApplication</code> operation to get this value.</p>
   */
  inline const Aws::Utils::DateTime& GetCreateTimestamp() const { return m_createTimestamp; }
  inline bool CreateTimestampHasBeenSet() const { return m_createTimestampHasBeenSet; }
  template <typename CreateTimestampT = Aws::Utils::DateTime>
  void SetCreateTimestamp(CreateTimestampT&& value) {
    m_createTimestampHasBeenSet = true;
    m_createTimestamp = std::forward<CreateTimestampT>(value);
  }
  template <typename CreateTimestampT = Aws::Utils::DateTime>
  DeleteApplicationRequest& WithCreateTimestamp(CreateTimestampT&& value) {
    SetCreateTimestamp(std::forward<CreateTimestampT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationName;

  Aws::Utils::DateTime m_createTimestamp{};
  bool m_applicationNameHasBeenSet = false;
  bool m_createTimestampHasBeenSet = false;
};

}  // namespace Model
}  // namespace KinesisAnalyticsV2
}  // namespace Aws
