/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.services.x509;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import org.jboss.logging.Logger;
import org.keycloak.common.util.Base64;
import org.keycloak.common.util.DerUtils;
import org.keycloak.http.HttpRequest;
import org.keycloak.services.x509.X509ClientCertificateLookup;

public class Rfc9440ClientCertificateLookup
implements X509ClientCertificateLookup {
    private static final Logger log = Logger.getLogger(Rfc9440ClientCertificateLookup.class);
    protected final String sslClientCertHttpHeader;
    protected final String sslCertChainHttpHeader;
    protected final int certificateChainLength;

    public Rfc9440ClientCertificateLookup(String sslClientCertHttpHeader, String sslCertChainHttpHeader, int certificateChainLength) {
        this.sslClientCertHttpHeader = Optional.ofNullable(sslClientCertHttpHeader).filter(s -> !s.isBlank()).orElseThrow(() -> new IllegalArgumentException("sslClientCertHttpHeader"));
        this.sslCertChainHttpHeader = Optional.ofNullable(sslCertChainHttpHeader).filter(s -> !s.isBlank()).orElseThrow(() -> new IllegalArgumentException("sslCertChainHttpHeader"));
        this.certificateChainLength = certificateChainLength;
    }

    @Override
    public X509Certificate[] getCertificateChain(HttpRequest httpRequest) throws GeneralSecurityException {
        if (!httpRequest.isProxyTrusted()) {
            log.warnf("HTTP header \"%s\" is not trusted", (Object)this.sslClientCertHttpHeader);
            return null;
        }
        try {
            ArrayList<X509Certificate> chain = new ArrayList<X509Certificate>();
            X509Certificate clientCertificate = this.getClientCertificateFromHeader(httpRequest);
            if (clientCertificate != null) {
                chain.add(clientCertificate);
                chain.addAll(this.getClientCertificateChainFromHeader(httpRequest));
            }
            return chain.toArray(new X509Certificate[0]);
        }
        catch (RfcViolationException e) {
            throw new GeneralSecurityException(e);
        }
    }

    public void close() {
    }

    protected X509Certificate getClientCertificateFromHeader(HttpRequest httpRequest) throws RfcViolationException {
        List headerValues = httpRequest.getHttpHeaders().getRequestHeader(this.sslClientCertHttpHeader);
        if (headerValues.isEmpty()) {
            return null;
        }
        if (headerValues.size() > 1) {
            throw new Rfc9440ViolationException("2.2", "client cert header must occur at most once");
        }
        return Rfc9440ClientCertificateLookup.parseCertificateFromHttpByteSequence((String)headerValues.get(0));
    }

    protected List<X509Certificate> getClientCertificateChainFromHeader(HttpRequest httpRequest) throws RfcViolationException, GeneralSecurityException {
        List chainHeaderValues = httpRequest.getHttpHeaders().getRequestHeader(this.sslCertChainHttpHeader);
        if (chainHeaderValues == null || chainHeaderValues.isEmpty()) {
            return Collections.emptyList();
        }
        ArrayList<String> encodedCerts = new ArrayList<String>();
        for (String chainHeaderValue : chainHeaderValues) {
            String[] listEntries = chainHeaderValue.split(",\\s*");
            encodedCerts.addAll(Arrays.asList(listEntries));
        }
        if (encodedCerts.size() > this.certificateChainLength) {
            throw new GeneralSecurityException("The amount of certificates in the chain header " + encodedCerts.size() + " is bigger than the configured limit of " + this.certificateChainLength + ".");
        }
        ArrayList<X509Certificate> parsedCertificates = new ArrayList<X509Certificate>();
        for (String encodedCert : encodedCerts) {
            parsedCertificates.add(Rfc9440ClientCertificateLookup.parseCertificateFromHttpByteSequence(encodedCert));
        }
        return parsedCertificates;
    }

    protected static X509Certificate parseCertificateFromHttpByteSequence(String byteSequence) throws RfcViolationException {
        X509Certificate certificate;
        byte[] certificateBytes;
        if (byteSequence.length() < 2 || !byteSequence.startsWith(":") || !byteSequence.endsWith(":")) {
            throw new Rfc8941ViolationException("3.3.5", "value is not encoded as byte sequence");
        }
        String base64EncodedByteSequence = byteSequence.substring(1, byteSequence.length() - 1);
        try {
            certificateBytes = Base64.decode((String)base64EncodedByteSequence);
        }
        catch (IOException e) {
            throw new Rfc9440ViolationException("2.1", "value does not contain base64 encoded content", e);
        }
        try (ByteArrayInputStream is = new ByteArrayInputStream(certificateBytes);){
            certificate = DerUtils.decodeCertificate((InputStream)is);
        }
        catch (Exception e) {
            throw new Rfc9440ViolationException("2.1", "value does not contain DER encoded certificate", e);
        }
        if (certificate == null) {
            throw new Rfc9440ViolationException("2.1", "value does not contain DER encoded certificate");
        }
        log.debugf("Parsed certificate : Subject DN=[%s]  SerialNumber=[%s]", (Object)certificate.getSubjectX500Principal(), (Object)certificate.getSerialNumber());
        return certificate;
    }

    public static class RfcViolationException
    extends Exception {
        public RfcViolationException(String rfc, String section, String details, Throwable cause) {
            super("Violation of RFC " + rfc + " (see section " + section + "): " + details, cause);
        }
    }

    public static class Rfc9440ViolationException
    extends RfcViolationException {
        public Rfc9440ViolationException(String section, String details) {
            this(section, details, null);
        }

        public Rfc9440ViolationException(String section, String details, Throwable cause) {
            super("9440", section, details, cause);
        }
    }

    public static class Rfc8941ViolationException
    extends RfcViolationException {
        public Rfc8941ViolationException(String section, String details) {
            this(section, details, null);
        }

        public Rfc8941ViolationException(String section, String details, Throwable cause) {
            super("8941", section, details, cause);
        }
    }
}

