import ts from 'typescript';
import { PublishDiagnosticsParams, RelativePattern, TextDocumentContentChangeEvent } from 'vscode-languageserver-protocol';
import { Document } from '../../lib/documents';
import { DocumentSnapshot } from './DocumentSnapshot';
import { GlobalSnapshotsManager, SnapshotManager } from './SnapshotManager';
import { ProjectService } from './serviceCache';
export interface LanguageServiceContainer {
    readonly tsconfigPath: string;
    readonly compilerOptions: ts.CompilerOptions;
    readonly configErrors: ts.Diagnostic[];
    readonly snapshotManager: SnapshotManager;
    getService(skipSynchronize?: boolean): ts.LanguageService;
    updateSnapshot(documentOrFilePath: Document | string): DocumentSnapshot;
    deleteSnapshot(filePath: string): void;
    invalidateModuleCache(filePath: string[]): void;
    scheduleProjectFileUpdate(watcherNewFiles: string[]): void;
    ensureProjectFileUpdates(newFile?: string): void;
    updateTsOrJsFile(fileName: string, changes?: TextDocumentContentChangeEvent[]): void;
    /**
     * Checks if a file is present in the project.
     * Unlike `fileBelongsToProject`, this doesn't run a file search on disk.
     */
    hasFile(filePath: string): boolean;
    /**
     * Careful, don't call often, or it will hurt performance.
     * Only works for TS versions that have ScriptKind.Deferred
     */
    fileBelongsToProject(filePath: string, isNew: boolean): boolean;
    onAutoImportProviderSettingsChanged(): void;
    onPackageJsonChange(packageJsonPath: string): void;
    getTsConfigSvelteOptions(): {
        namespace: string;
    };
    getResolvedProjectReferences(): TsConfigInfo[];
    openVirtualDocument(document: Document): void;
    isShimFiles(filePath: string): boolean;
    dispose(): void;
}
declare module 'typescript' {
    interface LanguageServiceHost {
        /**
         * @internal
         * This is needed for the languageService program to know that there is a new file
         * that might change the module resolution results
         */
        hasInvalidatedResolutions?: (sourceFile: string) => boolean;
        /**
         * @internal
         */
        getModuleResolutionCache?(): ts.ModuleResolutionCache;
        /** @internal */
        setCompilerHost?(host: ts.CompilerHost): void;
    }
    interface ResolvedModuleWithFailedLookupLocations {
        /** @internal */
        failedLookupLocations?: string[];
        /** @internal */
        affectingLocations?: string[];
        /** @internal */
        resolutionDiagnostics?: ts.Diagnostic[];
        /**
         * @internal
         * Used to issue a better diagnostic when an unresolvable module may
         * have been resolvable under different module resolution settings.
         */
        alternateResult?: string;
    }
}
export interface TsConfigInfo {
    parsedCommandLine: ts.ParsedCommandLine;
    snapshotManager: SnapshotManager;
    pendingProjectFileUpdate: boolean;
    configFilePath: string;
    extendedConfigPaths?: Set<string>;
}
/**
 * For testing only: Reset the cache for services.
 * Try to refactor this some day so that this file provides
 * a setup function which creates all this nicely instead.
 */
export declare function __resetCache(): void;
export interface LanguageServiceDocumentContext {
    isSvelteCheck: boolean;
    ambientTypesSource: string;
    transformOnTemplateError: boolean;
    createDocument: (fileName: string, content: string) => Document;
    globalSnapshotsManager: GlobalSnapshotsManager;
    notifyExceedSizeLimit: (() => void) | undefined;
    extendedConfigCache: Map<string, ts.ExtendedConfigCacheEntry>;
    onProjectReloaded: ((configFileNames: string[]) => void) | undefined;
    reportConfigError: ((diagnostics: PublishDiagnosticsParams) => void) | undefined;
    watchTsConfig: boolean;
    tsSystem: ts.System;
    projectService: ProjectService | undefined;
    watchDirectory: ((patterns: RelativePattern[]) => void) | undefined;
    nonRecursiveWatchPattern: string | undefined;
}
export declare function getService(path: string, workspaceUris: string[], docContext: LanguageServiceDocumentContext): Promise<LanguageServiceContainer>;
export declare function forAllServices(cb: (service: LanguageServiceContainer) => any): Promise<void>;
/**
 * @param tsconfigPath has to be absolute
 * @param docContext
 */
export declare function getServiceForTsconfig(tsconfigPath: string, workspacePath: string, docContext: LanguageServiceDocumentContext): Promise<LanguageServiceContainer>;
