"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addVueCommands = addVueCommands;
const language_core_1 = require("@vue/language-core");
const collectExtractProps_1 = require("./requests/collectExtractProps");
const getComponentDirectives_1 = require("./requests/getComponentDirectives");
const getComponentEvents_1 = require("./requests/getComponentEvents");
const getComponentNames_1 = require("./requests/getComponentNames");
const getComponentProps_1 = require("./requests/getComponentProps");
const getElementAttrs_1 = require("./requests/getElementAttrs");
const getElementNames_1 = require("./requests/getElementNames");
const getImportPathForFile_1 = require("./requests/getImportPathForFile");
const getPropertiesAtLocation_1 = require("./requests/getPropertiesAtLocation");
// https://github.com/JetBrains/intellij-plugins/blob/6435723ad88fa296b41144162ebe3b8513f4949b/Angular/src-js/angular-service/src/index.ts#L69
function addVueCommands(ts, info, project2Service) {
    const projectService = info.project.projectService;
    projectService.logger.info("Vue: called handler processing " + info.project.projectKind);
    const session = info.session;
    if (session == undefined) {
        projectService.logger.info("Vue: there is no session in info.");
        return;
    }
    if (session.addProtocolHandler == undefined) {
        // addProtocolHandler was introduced in TS 4.4 or 4.5 in 2021, see https://github.com/microsoft/TypeScript/issues/43893
        projectService.logger.info("Vue: there is no addProtocolHandler method.");
        return;
    }
    if (session.vueCommandsAdded) {
        return;
    }
    session.vueCommandsAdded = true;
    const isCaseSensitive = info.languageServiceHost.useCaseSensitiveFileNames?.() ?? false;
    let lastProjectVersion;
    let scriptInfos = new language_core_1.FileMap(isCaseSensitive);
    session.addProtocolHandler('vue:isProjectUpdated', () => {
        const projectVersion = info.project.getProjectVersion();
        if (projectVersion === lastProjectVersion) {
            return { response: 'no' };
        }
        lastProjectVersion = projectVersion;
        const [, [language, languageServiceHost]] = [...project2Service].find(([project]) => project.projectKind === ts.server.ProjectKind.Configured);
        const fileNames = languageServiceHost.getScriptFileNames();
        const infos = new language_core_1.FileMap(isCaseSensitive);
        let isAnyScriptVersionChanged = false;
        let isAnyScriptSnapshotChanged = false;
        for (const file of fileNames) {
            const scriptVersion = languageServiceHost.getScriptVersion(file);
            const scriptInfo = scriptInfos.get(file) ?? { version: "" };
            infos.set(file, scriptInfo);
            if (scriptInfo.version === scriptVersion) {
                continue;
            }
            scriptInfo.version = scriptVersion;
            isAnyScriptVersionChanged = true;
            const volarFile = language.scripts.get(file);
            const root = volarFile?.generated?.root;
            const serviceScript = volarFile?.generated?.languagePlugin.typescript?.getServiceScript(root);
            if (!serviceScript) {
                isAnyScriptSnapshotChanged = true;
                continue;
            }
            const { snapshot } = serviceScript.code;
            if (scriptInfo.snapshot !== snapshot) {
                scriptInfo.snapshot = snapshot;
                isAnyScriptSnapshotChanged = true;
            }
        }
        scriptInfos = infos;
        return { response: isAnyScriptSnapshotChanged || !isAnyScriptVersionChanged ? 'yes' : 'no' };
    });
    session.addProtocolHandler('vue:collectExtractProps', ({ arguments: args }) => {
        return {
            response: collectExtractProps_1.collectExtractProps.apply(getRequestContext(args[0]), args),
        };
    });
    session.addProtocolHandler('vue:getImportPathForFile', ({ arguments: args }) => {
        return {
            response: getImportPathForFile_1.getImportPathForFile.apply(getRequestContext(args[0]), args),
        };
    });
    session.addProtocolHandler('vue:getPropertiesAtLocation', ({ arguments: args }) => {
        return {
            response: getPropertiesAtLocation_1.getPropertiesAtLocation.apply(getRequestContext(args[0]), args),
        };
    });
    session.addProtocolHandler('vue:getComponentNames', ({ arguments: args }) => {
        return {
            response: getComponentNames_1.getComponentNames.apply(getRequestContext(args[0]), args),
        };
    });
    session.addProtocolHandler('vue:getComponentProps', ({ arguments: args }) => {
        return {
            response: getComponentProps_1.getComponentProps.apply(getRequestContext(args[0]), args),
        };
    });
    session.addProtocolHandler('vue:getComponentEvents', ({ arguments: args }) => {
        return {
            response: getComponentEvents_1.getComponentEvents.apply(getRequestContext(args[0]), args),
        };
    });
    session.addProtocolHandler('vue:getComponentDirectives', ({ arguments: args }) => {
        return {
            response: getComponentDirectives_1.getComponentDirectives.apply(getRequestContext(args[0]), args),
        };
    });
    session.addProtocolHandler('vue:getElementAttrs', ({ arguments: args }) => {
        return {
            response: getElementAttrs_1.getElementAttrs.apply(getRequestContext(args[0]), args),
        };
    });
    session.addProtocolHandler('vue:getElementNames', ({ arguments: args }) => {
        return {
            response: getElementNames_1.getElementNames.apply(getRequestContext(args[0]), args),
        };
    });
    projectService.logger.info('Vue specific commands are successfully added.');
    function getRequestContext(fileName) {
        const fileAndProject = info.session.getFileAndProject({
            file: fileName,
            projectFileName: undefined,
        });
        const service = project2Service.get(fileAndProject.project);
        if (!service) {
            throw 'No RequestContext';
        }
        return {
            typescript: ts,
            languageService: service[2],
            languageServiceHost: service[1],
            language: service[0],
            isTsPlugin: true,
            getFileId: (fileName) => fileName,
        };
    }
}
//# sourceMappingURL=commands.js.map