"use strict";
/// <reference types="@volar/typescript" />
Object.defineProperty(exports, "__esModule", { value: true });
exports.getReactiveReferences = getReactiveReferences;
const language_core_1 = require("@vue/language-core");
const analyzeCache = new WeakMap();
function getReactiveReferences(ts, language, languageService, sourceScript, fileName, position, leadingOffset = 0) {
    const serviceScript = sourceScript?.generated?.languagePlugin.typescript?.getServiceScript(sourceScript.generated.root);
    const map = serviceScript ? language.maps.get(serviceScript.code, sourceScript) : undefined;
    const toSourceRange = map
        ? (start, end) => {
            for (const [mappedStart, mappedEnd] of map.toSourceRange(start - leadingOffset, end - leadingOffset, false)) {
                return { start: mappedStart, end: mappedEnd };
            }
        }
        : (start, end) => ({ start, end });
    const sourceFile = languageService.getProgram().getSourceFile(fileName);
    if (!analyzeCache.has(sourceFile)) {
        analyzeCache.set(sourceFile, analyze(ts, sourceFile, toSourceRange));
    }
    const { signals, allValuePropertyAccess, allPropertyAccess, allFunctionCalls, } = analyzeCache.get(sourceFile);
    const info = findSignalByBindingRange(position) ?? findSignalByCallbackRange(position);
    if (!info) {
        return;
    }
    const dependents = info.binding ? findDependents(info.binding.ast, info.binding.accessTypes) : [];
    const dependencies = findDependencies(info);
    if ((!info.isDependent && !dependents.length) || (!info.isDependency && !dependencies.length)) {
        return;
    }
    const dependencyRanges = [];
    const dependentRanges = [];
    for (const dependency of dependencies) {
        let { ast } = dependency;
        if (ts.isBlock(ast) && ast.statements.length) {
            const sourceRange = toSourceRange(ast.statements[0].getStart(sourceFile), ast.statements[ast.statements.length - 1].end);
            if (sourceRange) {
                dependencyRanges.push({ start: sourceRange.start, end: sourceRange.end });
            }
        }
        else {
            dependencyRanges.push({ start: dependency.start, end: dependency.end });
        }
    }
    for (const { callback } of dependents) {
        if (!callback) {
            continue;
        }
        if (ts.isBlock(callback.ast) && callback.ast.statements.length) {
            const { statements } = callback.ast;
            const sourceRange = toSourceRange(statements[0].getStart(sourceFile), statements[statements.length - 1].end);
            if (sourceRange) {
                dependencyRanges.push({ start: sourceRange.start, end: sourceRange.end });
            }
        }
        else {
            dependentRanges.push({ start: callback.start, end: callback.end });
        }
    }
    return { dependencyRanges, dependentRanges };
    function findDependencies(signal, visited = new Set()) {
        if (visited.has(signal)) {
            return [];
        }
        visited.add(signal);
        const nodes = [];
        let hasDependency = signal.isDependency;
        if (signal.accessor) {
            const { requiredAccess } = signal.accessor;
            visit(signal.accessor, requiredAccess);
            signal.accessor.ast.forEachChild(child => {
                const childRange = toSourceRange(child.getStart(sourceFile), child.end);
                if (childRange) {
                    visit({
                        ...childRange,
                        ast: child,
                    }, requiredAccess);
                }
            });
        }
        if (!hasDependency) {
            return [];
        }
        return nodes;
        function visit(node, requiredAccess, parentIsPropertyAccess = false) {
            if (!requiredAccess) {
                if (!parentIsPropertyAccess && ts.isIdentifier(node.ast)) {
                    const definition = languageService.getDefinitionAtPosition(sourceFile.fileName, node.start);
                    for (const info of definition ?? []) {
                        if (info.fileName !== sourceFile.fileName) {
                            continue;
                        }
                        const signal = findSignalByBindingRange(info.textSpan.start);
                        if (!signal) {
                            continue;
                        }
                        if (signal.binding) {
                            nodes.push(signal.binding);
                            hasDependency ||= signal.isDependency;
                        }
                        if (signal.callback) {
                            nodes.push(signal.callback);
                        }
                        const deps = findDependencies(signal, visited);
                        nodes.push(...deps);
                        hasDependency ||= deps.length > 0;
                    }
                }
            }
            else if (ts.isPropertyAccessExpression(node.ast) || ts.isElementAccessExpression(node.ast)
                || ts.isCallExpression(node.ast)) {
                const definition = languageService.getDefinitionAtPosition(sourceFile.fileName, node.start);
                for (const info of definition ?? []) {
                    if (info.fileName !== sourceFile.fileName) {
                        continue;
                    }
                    const signal = findSignalByBindingRange(info.textSpan.start);
                    if (!signal) {
                        continue;
                    }
                    const oldSize = nodes.length;
                    if (signal.binding) {
                        for (const accessType of signal.binding.accessTypes) {
                            if (ts.isPropertyAccessExpression(node.ast)) {
                                if (accessType === 0 /* ReactiveAccessType.ValueProperty */ && node.ast.name.text === 'value') {
                                    nodes.push(signal.binding);
                                    hasDependency ||= signal.isDependency;
                                }
                                if (accessType === 1 /* ReactiveAccessType.AnyProperty */ && node.ast.name.text !== '') {
                                    nodes.push(signal.binding);
                                    hasDependency ||= signal.isDependency;
                                }
                            }
                            else if (ts.isElementAccessExpression(node.ast)) {
                                if (accessType === 1 /* ReactiveAccessType.AnyProperty */) {
                                    nodes.push(signal.binding);
                                    hasDependency ||= signal.isDependency;
                                }
                            }
                            else if (ts.isCallExpression(node.ast)) {
                                if (accessType === 2 /* ReactiveAccessType.Call */) {
                                    nodes.push(signal.binding);
                                    hasDependency ||= signal.isDependency;
                                }
                            }
                        }
                    }
                    const signalDetected = nodes.length > oldSize;
                    if (signalDetected) {
                        if (signal.callback) {
                            nodes.push(signal.callback);
                        }
                        const deps = findDependencies(signal, visited);
                        nodes.push(...deps);
                        hasDependency ||= deps.length > 0;
                    }
                }
            }
            node.ast.forEachChild(child => {
                const childRange = toSourceRange(child.getStart(sourceFile), child.end);
                if (childRange) {
                    visit({
                        ...childRange,
                        ast: child,
                    }, requiredAccess, ts.isPropertyAccessExpression(node.ast) || ts.isElementAccessExpression(node.ast));
                }
            });
        }
    }
    function findDependents(node, trackKinds, visited = new Set()) {
        return (0, language_core_1.collectBindingRanges)(ts, node, sourceFile)
            .map(range => {
            const sourceRange = toSourceRange(range.start, range.end);
            if (sourceRange) {
                return findDependentsWorker(sourceRange.start, trackKinds, visited);
            }
            return [];
        })
            .flat();
    }
    function findDependentsWorker(pos, accessTypes, visited = new Set()) {
        if (visited.has(pos)) {
            return [];
        }
        visited.add(pos);
        const references = languageService.findReferences(sourceFile.fileName, pos);
        if (!references) {
            return [];
        }
        const result = [];
        for (const reference of references) {
            for (const reference2 of reference.references) {
                if (reference2.fileName !== sourceFile.fileName) {
                    continue;
                }
                const effect = findSignalByAccessorRange(reference2.textSpan.start);
                if (effect?.accessor) {
                    let match = false;
                    if (effect.accessor.requiredAccess) {
                        for (const accessType of accessTypes) {
                            if (accessType === 1 /* ReactiveAccessType.AnyProperty */) {
                                match ||= allPropertyAccess.has(reference2.textSpan.start + reference2.textSpan.length);
                            }
                            else if (accessType === 0 /* ReactiveAccessType.ValueProperty */) {
                                match ||= allValuePropertyAccess.has(reference2.textSpan.start + reference2.textSpan.length);
                            }
                            else {
                                match ||= allFunctionCalls.has(reference2.textSpan.start + reference2.textSpan.length);
                            }
                        }
                    }
                    if (match) {
                        let hasDependent = effect.isDependent;
                        if (effect.binding) {
                            const dependents = findDependents(effect.binding.ast, effect.binding.accessTypes, visited);
                            result.push(...dependents);
                            hasDependent ||= dependents.length > 0;
                        }
                        if (hasDependent) {
                            result.push(effect);
                        }
                    }
                }
            }
        }
        return result;
    }
    function findSignalByBindingRange(position) {
        return signals.find(ref => ref.binding && ref.binding.start <= position
            && ref.binding.end >= position);
    }
    function findSignalByCallbackRange(position) {
        return signals.filter(ref => ref.callback && ref.callback.start <= position
            && ref.callback.end >= position).sort((a, b) => (a.callback.end - a.callback.start) - (b.callback.end - b.callback.start))[0];
    }
    function findSignalByAccessorRange(position) {
        return signals.filter(ref => ref.accessor && ref.accessor.start <= position
            && ref.accessor.end >= position).sort((a, b) => (a.accessor.end - a.accessor.start) - (b.accessor.end - b.accessor.start))[0];
    }
}
function analyze(ts, sourceFile, toSourceRange) {
    const signals = [];
    const allValuePropertyAccess = new Set();
    const allPropertyAccess = new Set();
    const allFunctionCalls = new Set();
    sourceFile.forEachChild(function visit(node) {
        if (ts.isVariableDeclaration(node)) {
            if (node.initializer && ts.isCallExpression(node.initializer)) {
                const call = node.initializer;
                if (ts.isIdentifier(call.expression)) {
                    const callName = call.expression.escapedText;
                    if (callName === 'ref' || callName === 'shallowRef' || callName === 'toRef' || callName === 'useTemplateRef'
                        || callName === 'defineModel') {
                        const nameRange = toSourceRange(node.name.getStart(sourceFile), node.name.end);
                        if (nameRange) {
                            signals.push({
                                isDependency: true,
                                isDependent: false,
                                binding: {
                                    ...nameRange,
                                    ast: node.name,
                                    accessTypes: [0 /* ReactiveAccessType.ValueProperty */],
                                },
                            });
                        }
                    }
                    else if (callName === 'reactive' || callName === 'shallowReactive' || callName === 'defineProps'
                        || callName === 'withDefaults') {
                        const nameRange = toSourceRange(node.name.getStart(sourceFile), node.name.end);
                        if (nameRange) {
                            signals.push({
                                isDependency: true,
                                isDependent: false,
                                binding: {
                                    ...nameRange,
                                    ast: node.name,
                                    accessTypes: [1 /* ReactiveAccessType.AnyProperty */],
                                },
                            });
                        }
                    }
                    // TODO: toRefs
                }
            }
        }
        else if (ts.isFunctionDeclaration(node)) {
            if (node.name && node.body) {
                const nameRange = toSourceRange(node.name.getStart(sourceFile), node.name.end);
                const bodyRange = toSourceRange(node.body.getStart(sourceFile), node.body.end);
                if (nameRange && bodyRange) {
                    signals.push({
                        isDependency: false,
                        isDependent: false,
                        binding: {
                            ...nameRange,
                            ast: node.name,
                            accessTypes: [2 /* ReactiveAccessType.Call */],
                        },
                        accessor: {
                            ...bodyRange,
                            ast: node.body,
                            requiredAccess: true,
                        },
                        callback: {
                            ...bodyRange,
                            ast: node.body,
                        },
                    });
                }
            }
        }
        else if (ts.isVariableStatement(node)) {
            for (const declaration of node.declarationList.declarations) {
                const name = declaration.name;
                const callback = declaration.initializer;
                if (callback && ts.isIdentifier(name) && (ts.isArrowFunction(callback) || ts.isFunctionExpression(callback))) {
                    const nameRange = toSourceRange(name.getStart(sourceFile), name.end);
                    const callbackRange = toSourceRange(callback.getStart(sourceFile), callback.end);
                    if (nameRange && callbackRange) {
                        signals.push({
                            isDependency: false,
                            isDependent: false,
                            binding: {
                                ...nameRange,
                                ast: name,
                                accessTypes: [2 /* ReactiveAccessType.Call */],
                            },
                            accessor: {
                                ...callbackRange,
                                ast: callback,
                                requiredAccess: true,
                            },
                            callback: {
                                ...callbackRange,
                                ast: callback,
                            },
                        });
                    }
                }
            }
        }
        else if (ts.isParameter(node)) {
            if (node.type && ts.isTypeReferenceNode(node.type)) {
                const typeName = node.type.typeName.getText(sourceFile);
                if (typeName.endsWith('Ref')) {
                    const nameRange = toSourceRange(node.name.getStart(sourceFile), node.name.end);
                    if (nameRange) {
                        signals.push({
                            isDependency: true,
                            isDependent: false,
                            binding: {
                                ...nameRange,
                                ast: node.name,
                                accessTypes: [0 /* ReactiveAccessType.ValueProperty */],
                            },
                        });
                    }
                }
            }
        }
        else if (ts.isCallExpression(node) && ts.isIdentifier(node.expression)) {
            const call = node;
            const callName = node.expression.escapedText;
            if ((callName === 'effect' || callName === 'watchEffect') && call.arguments.length) {
                const callback = call.arguments[0];
                if (ts.isArrowFunction(callback) || ts.isFunctionExpression(callback)) {
                    const callbackRange = toSourceRange(callback.getStart(sourceFile), callback.end);
                    if (callbackRange) {
                        signals.push({
                            isDependency: false,
                            isDependent: true,
                            accessor: {
                                ...callbackRange,
                                ast: callback.body,
                                requiredAccess: true,
                            },
                            callback: {
                                ...callbackRange,
                                ast: callback.body,
                            },
                        });
                    }
                }
            }
            if (callName === 'watch' && call.arguments.length >= 2) {
                const depsCallback = call.arguments[0];
                const effectCallback = call.arguments[1];
                if (ts.isArrowFunction(effectCallback) || ts.isFunctionExpression(effectCallback)) {
                    const depsRange = toSourceRange(depsCallback.getStart(sourceFile), depsCallback.end);
                    const effectRange = toSourceRange(effectCallback.getStart(sourceFile), effectCallback.end);
                    if (depsRange && effectRange) {
                        if (ts.isArrowFunction(depsCallback) || ts.isFunctionExpression(depsCallback)) {
                            signals.push({
                                isDependency: false,
                                isDependent: true,
                                accessor: {
                                    ...depsRange,
                                    ast: depsCallback.body,
                                    requiredAccess: true,
                                },
                                callback: {
                                    ...effectRange,
                                    ast: effectCallback.body,
                                },
                            });
                        }
                        else {
                            signals.push({
                                isDependency: false,
                                isDependent: true,
                                accessor: {
                                    ...depsRange,
                                    ast: depsCallback,
                                    requiredAccess: false,
                                },
                                callback: {
                                    ...effectRange,
                                    ast: effectCallback.body,
                                },
                            });
                        }
                    }
                }
            }
            else if ((0, language_core_1.hyphenateAttr)(callName).startsWith('use-')) {
                let binding;
                if (ts.isVariableDeclaration(call.parent)) {
                    const nameRange = toSourceRange(call.parent.name.getStart(sourceFile), call.parent.name.end);
                    if (nameRange) {
                        binding = {
                            ...nameRange,
                            ast: call.parent.name,
                            accessTypes: [1 /* ReactiveAccessType.AnyProperty */, 2 /* ReactiveAccessType.Call */],
                        };
                    }
                }
                const callRange = toSourceRange(call.getStart(sourceFile), call.end);
                if (callRange) {
                    signals.push({
                        isDependency: true,
                        isDependent: false,
                        binding,
                        accessor: {
                            ...callRange,
                            ast: call,
                            requiredAccess: false,
                        },
                    });
                }
            }
            else if ((callName === 'computed' || (0, language_core_1.hyphenateAttr)(callName).endsWith('-computed')) && call.arguments.length) {
                const arg = call.arguments[0];
                if (ts.isArrowFunction(arg) || ts.isFunctionExpression(arg)) {
                    let binding;
                    if (ts.isVariableDeclaration(call.parent)) {
                        const nameRange = toSourceRange(call.parent.name.getStart(sourceFile), call.parent.name.end);
                        if (nameRange) {
                            binding = {
                                ...nameRange,
                                ast: call.parent.name,
                                accessTypes: [0 /* ReactiveAccessType.ValueProperty */],
                            };
                        }
                    }
                    const argRange = toSourceRange(arg.getStart(sourceFile), arg.end);
                    if (argRange) {
                        signals.push({
                            isDependency: true,
                            isDependent: true,
                            binding,
                            accessor: {
                                ...argRange,
                                ast: arg.body,
                                requiredAccess: true,
                            },
                            callback: {
                                ...argRange,
                                ast: arg.body,
                            },
                        });
                    }
                }
                else if (ts.isIdentifier(arg)) {
                    let binding;
                    if (ts.isVariableDeclaration(call.parent)) {
                        const nameRange = toSourceRange(call.parent.name.getStart(sourceFile), call.parent.name.end);
                        if (nameRange) {
                            binding = {
                                ...nameRange,
                                ast: call.parent.name,
                                accessTypes: [0 /* ReactiveAccessType.ValueProperty */],
                            };
                        }
                    }
                    const argRange = toSourceRange(arg.getStart(sourceFile), arg.end);
                    if (argRange) {
                        signals.push({
                            isDependency: true,
                            isDependent: false,
                            binding,
                            accessor: {
                                ...argRange,
                                ast: arg,
                                requiredAccess: false,
                            },
                        });
                    }
                }
                else if (ts.isObjectLiteralExpression(arg)) {
                    for (const prop of arg.properties) {
                        if (prop.name?.getText(sourceFile) === 'get') {
                            let binding;
                            if (ts.isVariableDeclaration(call.parent)) {
                                const nameRange = toSourceRange(call.parent.name.getStart(sourceFile), call.parent.name.end);
                                if (nameRange) {
                                    binding = {
                                        ...nameRange,
                                        ast: call.parent.name,
                                        accessTypes: [0 /* ReactiveAccessType.ValueProperty */],
                                    };
                                }
                            }
                            if (ts.isPropertyAssignment(prop)) {
                                const callback = prop.initializer;
                                if (ts.isArrowFunction(callback) || ts.isFunctionExpression(callback)) {
                                    const callbackRange = toSourceRange(callback.getStart(sourceFile), callback.end);
                                    if (callbackRange) {
                                        signals.push({
                                            isDependency: true,
                                            isDependent: true,
                                            binding,
                                            accessor: {
                                                ...callbackRange,
                                                ast: callback.body,
                                                requiredAccess: true,
                                            },
                                            callback: {
                                                ...callbackRange,
                                                ast: callback.body,
                                            },
                                        });
                                    }
                                }
                            }
                            else if (ts.isMethodDeclaration(prop) && prop.body) {
                                const bodyRange = toSourceRange(prop.body.getStart(sourceFile), prop.body.end);
                                if (bodyRange) {
                                    signals.push({
                                        isDependency: true,
                                        isDependent: true,
                                        binding,
                                        accessor: {
                                            ...bodyRange,
                                            ast: prop.body,
                                            requiredAccess: true,
                                        },
                                        callback: {
                                            ...bodyRange,
                                            ast: prop.body,
                                        },
                                    });
                                }
                            }
                        }
                    }
                }
            }
        }
        node.forEachChild(visit);
    });
    sourceFile.forEachChild(function visit(node) {
        if (ts.isPropertyAccessExpression(node)) {
            const sourceRange = toSourceRange(node.expression.end, node.expression.end);
            if (sourceRange) {
                if (node.name.text === 'value') {
                    allValuePropertyAccess.add(sourceRange.end);
                    allPropertyAccess.add(sourceRange.end);
                }
                else if (node.name.text !== '') {
                    allPropertyAccess.add(sourceRange.end);
                }
            }
        }
        else if (ts.isElementAccessExpression(node)) {
            const sourceRange = toSourceRange(node.expression.end, node.expression.end);
            if (sourceRange) {
                allPropertyAccess.add(sourceRange.end);
            }
        }
        else if (ts.isCallExpression(node)) {
            const sourceRange = toSourceRange(node.expression.end, node.expression.end);
            if (sourceRange) {
                allFunctionCalls.add(sourceRange.end);
            }
        }
        node.forEachChild(visit);
    });
    return {
        signals,
        allValuePropertyAccess,
        allPropertyAccess,
        allFunctionCalls,
    };
}
//# sourceMappingURL=getReactiveReferences.js.map