import { EndOfStreamError } from './Errors.js';
import { AbstractStreamReader } from "./AbstractStreamReader.js";
export class WebStreamDefaultReader extends AbstractStreamReader {
    constructor(reader) {
        super();
        this.reader = reader;
        this.buffer = null; // Internal buffer to store excess data
        this.bufferOffset = 0; // Current position in the buffer
    }
    async readFromStream(buffer, offset, length) {
        let totalBytesRead = 0;
        // Serve from the internal buffer first
        if (this.buffer) {
            const remainingInBuffer = this.buffer.byteLength - this.bufferOffset;
            const toCopy = Math.min(remainingInBuffer, length);
            buffer.set(this.buffer.subarray(this.bufferOffset, this.bufferOffset + toCopy), offset);
            this.bufferOffset += toCopy;
            totalBytesRead += toCopy;
            length -= toCopy;
            offset += toCopy;
            // If the buffer is exhausted, clear it
            if (this.bufferOffset >= this.buffer.byteLength) {
                this.buffer = null;
                this.bufferOffset = 0;
            }
        }
        // Continue reading from the stream if more data is needed
        while (length > 0 && !this.endOfStream) {
            const result = await this.reader.read();
            if (result.done) {
                this.endOfStream = true;
                break;
            }
            if (result.value) {
                const chunk = result.value;
                // If the chunk is larger than the requested length, store the excess
                if (chunk.byteLength > length) {
                    buffer.set(chunk.subarray(0, length), offset);
                    this.buffer = chunk;
                    this.bufferOffset = length; // Keep track of the unconsumed part
                    totalBytesRead += length;
                    return totalBytesRead;
                }
                // Otherwise, consume the entire chunk
                buffer.set(chunk, offset);
                totalBytesRead += chunk.byteLength;
                length -= chunk.byteLength;
                offset += chunk.byteLength;
            }
        }
        if (totalBytesRead === 0 && this.endOfStream) {
            throw new EndOfStreamError();
        }
        return totalBytesRead;
    }
    abort() {
        this.interrupted = true;
        return this.reader.cancel();
    }
    async close() {
        await this.abort();
        this.reader.releaseLock();
    }
}
