package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.36.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.36.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,``P`!````D"(``#0````HP0```````#0`(``,
M`"@`&P`:``8````T````-````#0```"``0``@`$```0````$`````P```+0!
M``"T`0``M`$``!,````3````!`````$````!````````````````````M!$`
M`+01```$`````!````$`````(````"`````@``!D,```9#````4`````$```
M`0````!@````8````&````Q/```,3P``!``````0```!````U+T``-2]``#4
MO0``1`(``%@"```&`````!````(```#<O0``W+T``-R]```(`0``"`$```8`
M```$````!````,@!``#(`0``R`$``&P```!L````!`````0```!3Y71D[`$`
M`.P!``#L`0``*````"@````$````!````%#E=&00H0``$*$``!"A```4`0``
M%`$```0````$````4>5T9```````````````````````````!@```!````!2
MY71DU+T``-2]``#4O0``+`(``"P"```$`````0```"]L:6(O;&0M;&EN=7@N
M<V\N,@``!````!0````#````1TY5`'?=CR_:<*PTBL.8<+9A=.L)4!^$!```
M`!@````%````1TY5``$``<`$````"0````(``<`$`````0````0````0````
M`0```$=.50``````!`````0`````````+P```$0````(````"``````0$`0E
M0`2`*`$`!:"&`(2``0``"R@$"#B`$D&*```(``````````!$````````````
M``!%````````````````````1P```$@`````````20````````!*````````
M`$L```````````````````!,`````````$T``````````````$\```!0````
M`````%$``````````````%,`````````5````%4``````````````%8```!8
M````60```%H```!;````7````````````````````%T```!K"4/6+L6;MF_%
MWQ^S`8MO0_K7AVM_FGRM2^/`6>CM[.-'Q@$,.I<+P^4_+"'_Y`Y_J9R;X*<"
M"KOCDGS59-1\S]Y'+?NA;&!J&IY0H^#[23-BV^T5C"7,9Z_,9D-%U>R)!W#3
MV7%8'`````````````````````"U`0`````````````2````;P$`````````
M````$@```*\!`````````````!(```"+`@`````````````2````\@$`````
M````````$@```'("`````````````!(```#;`0`````````````2````S0$`
M````````````$@```"X"`````````````!(````!```````````````@````
M9P$`````````````$@```'T!`````````````!(```!V```````````````2
M````O0(`````````````$@```.(!`````````````!(````[`0``````````
M```2````F0$`````````````$@```+$``````````````!(````0````````
M```````@````HP(`````````````$@```)`"`````````````!(```"B````
M```````````2````+```````````````(````)L``````````````!(```"V
M```````````````2````E0(`````````````$@```/,``````````````!8`
M```&`0`````````````2````00(`````````````$@```"$!````````````
M`!(```"N`@`````````````2````O@``````````````$@```,8!````````
M`````!$```!V`0`````````````2````R```````````````$@```-T`````
M`````````!(```"/`P`````````````2````4`$`````````````$@```*$!
M`````````````!(```"*```````````````1````Y```````````````$@``
M`'T"`````````````!(```"H`0`````````````2````:P(`````````````
M$@```,8"`````````````!(````3`@`````````````2````X@(`````````
M````$@```.T``````````````!(```"+`0`````````````2````F@``````
M````````$@```#(!`````````````!(```!?`0`````````````2````_@$`
M````````````$@```%\"`````````````!(```#/```````````````2````
MB```````````````$0```$D!`````````````!(```"2```````````````2
M````4@(`````````````$0```!8!`````````````!(````?`@``````````
M```2````A`$`````````````$@```%4``````````````!(```"I````````
M```````2````D@$`````````````$@```&<``````````````"(```#=`@``
M```````````2````I`,```#`````````$``7``@#``"P.```$````!(`#@"8
M`P``<$L``-$````2``X`0P,``+!```!A````$@`.`+$#``"@3```EP$``!(`
M#@!B````4"```#T"```2``X`1@````1@```$````$0`0`"8#````8```!```
M`!$`$``9`P``\#H``#8````2``X`I@,```#`````````(``7`+P#``"@-P``
M-````!(`#@`$!```P#@``"<"```2``X`?0,``"!!```M````$@`.`%$#``#@
M-P``SP```!(`#@#9`P``+,`````````0`!@`+0,``&!)```!`@``$@`.`-X#
M``!00@``#@<``!(`#@#R`P``4$$``/D````2``X`<P,``$!.``"@`0``$@`.
M`!L$``"P/```_`,``!(`#@"J`P``D"(``#`````2``X`T`,``%!,``!"````
M$@`.`&<#```P.P``=`$``!(`#@#K`P``&,`````````0`!<`$00``!3````$
M````$0`7`,0#```8P````````!``&```7U]G;6]N7W-T87)T7U\`7TE435]D
M97)E9VES=&5R5$U#;&]N951A8FQE`%])5$U?<F5G:7-T97)434-L;VYE5&%B
M;&4`7TE/7W-T9&EN7W5S960`7U]L:6)C7W-T87)T7VUA:6X`7U]C>&%?9FEN
M86QI>F4`7U]R96=I<W1E<E]A=&9O<FL`7U]E;G9I<F]N`'-T<FYC;7``;'-T
M870V-`!A8V-E<W,`;'-E96LV-`!R96%D`&]P96YD:7(`<F5A9&1I<C8T`'-T
M<FQE;@!?7W-P<FEN=&9?8VAK`'5N;&EN:P!C;&]S961I<@!R;61I<@!03%]C
M=7)R96YT7V-O;G1E>'0`8F]O=%]$>6YA3&]A9&5R`%!E<FQ?;F5W6%,`4&5R
M;%]N97=84U]F;&%G<P!R96%D;&EN:P!097)L7W-V7W-E='!V`&UE;6-H<@!0
M97)L7W-V7W-E='!V;@!M96UM;W9E`')E86QL;V,`9V5T<&ED`&UE;6-P>0!S
M=')C:'(`<W1R9'5P`'-T<G1O:P!S=')C;7``<W1R<F-H<@!G971U:60`<W1R
M<W1R`&UK9&ER`%]?97)R;F]?;&]C871I;VX`<W1D97)R`%]?9G!R:6YT9E]C
M:&L`;W!E;C8T`%!E<FQ?9W9?9F5T8VAP=@!097)L7V=E=%]S=@!097)L7V-R
M;V%K7VYO8V]N=&5X=`!097)L7VUG7W-E=`!097)L7W-Y<U]I;FET,P!097)L
M7V%T9F]R:U]U;FQO8VL`4&5R;%]A=&9O<FM?;&]C:P!03%]D;U]U;F1U;7``
M4&5R;%]S869E<WES;6%L;&]C`'!E<FQ?<&%R<V4`<&5R;%]D97-T<G5C=`!P
M97)L7V9R964`4&5R;%]S>7-?=&5R;0!P97)L7V%L;&]C`'!E<FQ?8V]N<W1R
M=6-T`'!E<FQ?<G5N`%!E<FQ?8W)O86M?;65M;W)Y7W=R87``97AI=`!?7W-T
M86-K7V-H:U]F86EL`&QI8G!E<FPN<V\`;&EB8RYS;RXV`'!A<E]C=7)R96YT
M7V5X96,`<&%R7V)A<V5N86UE`%]F<%]H=P!84U]);G1E<FYA;'-?4$%27T)/
M3U0`<&%R7V5N=E]C;&5A;@!P87)?8W5R<F5N=%]E>&5C7W!R;V,`<&%R7V1I
M<FYA;64`<VAA7V9I;F%L`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`'!A
M<E]C;&5A;G5P`%]?9&%T85]S=&%R=`!S:&%?=7!D871E`'AS7VEN:70`7U]B
M<W-?<W1A<G0`<VAA7VEN:70`7V5N9`!P87)?;6MT;7!D:7(`7V5D871A`'!A
M<E]S971U<%]L:6)P871H`'!A<E]F:6YD<')O9P!M>5]P87)?<&P`<&%R7VEN
M:71?96YV`$=,24)#7S(N,2XS`$=,24)#7S(N,S0`1TQ)0D-?,BXT`$=,24)#
M7S(N,@!'3$E"0U\R+C,S`$=,24)#7S(N,RXR`$=,24)#7S(N,RXT`$=,24)#
M7S(N,0!'3$E"0U\R+C``+W5S<B]L:6(O<&5R;#4O-2XS-B]C;W)E7W!E<FPO
M0T]210`````"``(``@`!``$``0`#``0``0`!``(``@`%``$``0`!``(``@`!
M``$``@`"``$`!@`"``$``0`!``$``0`!``<``@`"``(``@`"``$``@`"``(`
M`0`"``(``0`!``@``@`"``8``@`"``$``0`$``(``@`"``$``0`!``(`"0`#
M``(`"@`"``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!``$``0````$`"0#^`@``$`````````!S'VD)```*`"@$```0
M````M)&6!@``"0`T!```$````!1I:0T```@`/P0``!`````2:6D-```'`$D$
M```0````LY&6!@``!@!3!```$````'(9:0D```4`7@0``!````!T&6D)```$
M`&H$```0````$6EI#0```P!V!```$````!!I:0T```(`@`0```````#4O0``
M"````-B]```(````)+\```@```#\OP``"`````3````(````%,````@```#T
MO@``!@$``/B^```&`@``_+X```8#````OP``!@0```2_```&!0``"+\```8&
M```,OP``!@<``!"_```&"```%+\```8)```8OP``!@H``!R_```&"P``(+\`
M``8,```HOP``!@T``"R_```&#@``,+\```8/```TOP``!A```#B_```&$0``
M/+\```82``!`OP``!A,``$2_```&%```2+\```85``!,OP``!A8``%"_```&
M%P``5+\```88``!8OP``!AD``%R_```&&@``8+\```X;``!DOP``!AP``&B_
M```&'0``;+\```8>``!POP``!A\``'2_```&(```>+\```8A``!\OP``!B(`
M`("_```&(P``A+\```8D``"(OP``!B4``(R_```&)@``D+\```8G``"4OP``
M!B@``)B_```&*0``G+\```8J``"@OP``!BL``*2_```&+```J+\```8M``"L
MOP``!BX``+"_```&+P``M+\```8P``"XOP``!C$``+R_```&,@``P+\```8S
M``#$OP``!C0``,B_```&-0``S+\```8V``#0OP``!C<``-2_```&.0``V+\`
M``8Z``#<OP``!CL``."_```&/```Y+\```8]``#HOP``!CX``.R_```&0```
M\+\```9!``#TOP``!D(``/B_```&0P``\+X```<_````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``#S#Q[[4X/L".BS`@``@</7G@``BX,T````A<!T`O_0@\0(6\,`````````
M``````#_LP0```#_HP@`````````_Z,,````:`````#IX/___XU,)`2#Y/#_
M<?Q5B>57B<]64^A9`@``@<-]G@``48/L+(M!!(E%U(M!"(E%T&6A%````(E%
MY#'`C4704(U%U%!1_Y,``0``BX,P````@\0,4%#_LX0```!GZ$(O``"+@_@`
M``"#Q!"`.``/A"4!``"+@WP```!EBP"`B)H"```#BP>-<`.#P`2)=<P]____
M/P^'AP$``(/L#,'@`E#_D^@```"+==2+/XG!B8-$`0``BP:#Q!")`8V#XZ+_
M_XE!!(V#,`$``(L`B4$(C8/FHO__B4$,@_\!#XY&`0``B4W(C48$C5$0C32^
MC;0F`````(L(@\`$@\($B4K\.<9U\8M-R(/'`X/L#(V#/'7__\<$N0````!J
M`%'_=<Q0_[,\`0``_Y,D````B<:#Q""%P`^$E@```(/L#/^S/`$``(U]X/^3
MN````(GZC8/#H?__Z!P.``"#Q!"%P`^$A0```(/L#/^S/`$``/^3'````(/$
M$/^3>````(M%Y&4K!10````/A;$```"-9?")\%E;7E]=C6'\P_^38````(F#
M/`$``(7`#X2$````@^P,4/^3C````(N#?````&6+$(/$$,:"0P8```#IJ_[_
M_X/L#/^S/`$``/^32````(G&@\00Z5'___^+@T`!``"%P'02@^P,4&?H*"D`
M`(/$$.E?____B?J-@WBA___H8PT``(F#0`$``(7`#X1$____Z]'_D\0```"_
M!````.G8_O__@^P,:@'_DQ0!``#HHRT``&:0D/,/'OLQ[5Z)X8/D\%!44N@8
M````@<-`G```:@!J`%%6_[,8`0``Z(7]___TBQPDPXL<),-FD&:09I!FD&:0
M9I#HY````('"#YP``(V*-`$``(V"-`$``#G(=!V+@EP```"%P'0358GE@^P4
M4?_0@\00R<.-="8`D,.-M"8`````Z*0```"!PL^;``!5B>53C8HT`0``C8(T
M`0``@^P$*<B)P\'H'\'[`@'8T?AT%(N2;````(72=`J#[`A04?_2@\00BUW\
MR<.-M"8`````9I#S#Q[[58GE4^A3____@<-WFP``@^P$@+LT`0```'4HBX,0
M`0``A<!T$H/L#/^S(`$``/^3$`$``(/$$.@P____QH,T`0```8M=_,G#C70F
M`/,/'OOI5____XL4),-FD)!55U93B<.!['P!``")1"00C4PD+&6A%````(F$
M)&P!```QP(UT)@"0BU2#'`_*B12!@\`!@_@0=>^+3"0PBW0D8(U4)#2+;"1D
MBT0D:(E,)`B-C"0T`0``BUPD+(E,)`SK"HVV`````(E,)`B+"HG'BT(8@\($
M,<@Q\(GNB?TQV(M<)`C1P(E"-#E4)`QUV8MT)!"+1"0LBQZ+5A"+;@B+?@R)
M5"0<C90"F7F"6HG8BTX$P<`%B?Z);"04`<*)Z(E,)`PQ^(ET)!@AR,')`C'X
MB=\!PHGHB7PD",'/`C'((=B+7"0P,>B-M!Z9>8):B=,!QL'#!0'>B<LQ^XG8
MBUPD-"'0P<H",<B-G!V9>8):BVPD.`'#B?#!P`6-C"F9>8):BVPD/`'#B?@Q
MT"'PP<X",?B-O"^9>8):BVPD0`'(B=G!P04!R(G1,?$AV3'1C90JF7F"6HML
M)$0!^8G'P<<%P<L"`?F)]S'?(<<Q]XVT+IEY@EJ+;"1(`?J)S\''!0'ZB<>)
MV,'/`C'X(<C!R0(QV(V<*YEY@EJ+;"1,`?")UL'&!0'PB<Z)^3'Q(=$Q^8V\
M+YEY@EJ+;"10`=F)P\'#!0'9B=.)\L'+`C':(<(Q\HVT+IEY@EJ+;"14`?J)
MS\''!0'ZB<>)V,'/`C'X(<C!R0(QV(V<*YEY@EJ+;"18`?")UL'&!0'PB<Z)
M^3'Q(=$Q^8V\+YEY@EJ+;"1<`=F)P\'#!0'9B=.)\L'+`C':(<(Q\HVT+IEY
M@EJ+;"1@`?J)S\''!0'ZB<>)V,'/`C'X(<C!R0(QV(V<*YEY@EJ+;"1D`?")
MUL'&!0'PB<Z)^3'Q(=$Q^8V\+YEY@EH!V8G#P<,%`=F)TXGRP<L",=HAPL'(
M`C'RB<6)V`'ZB<\QZ,''!2'(P<D"`?J+?"1H,=B-M#Z9>8):B=<!\,''!0'X
MB>\QSXG^BWPD;"'6P<H",>Z-O#N9>8):B<,!]\'#!8MT)'`!WXG+,=.-M#69
M>8):BVPD="'#P<@",<N-C"F9>8):BVPD>`'>B?O!PP4!WHG3,<,A^S'3P<\"
MC90JF7F"6@'+B?'!P04!RXG!,?DA\<'.`C'!`=&)VL'"!0'1BU0D?(V$$*'K
MV6Z)^C'R,=K!RP(!PHG(P<`%`<*+A"2`````C;P'H>O9;HGP,=@QR`'XB=?!
MQP4!^(G/BXPDA````,'/`HVT#J'KV6Z)V3'Y,='!R@(!\8G&P<8%`?&)UHN4
M)(@```"-G!.AZ]ENB?HQ\C'"`=J)R\'#!0':B<.+A"2,````P<L"C;P'H>O9
M;HGP,=@QR`'XB=?!QP4!^(G/BXPDD````,'/`HVT#J'KV6Z)V3'Y,='!R@(!
M\8G&P<8%`?&)UHN4))0```"-G!.AZ]ENB?HQ\C'"`=J)R\'#!0':B<.+A"28
M````P<L"C;P'H>O9;HGP,=@QR`'XB=?!QP4!^(G/BXPDG````,'/`HVT#J'K
MV6Z)V3'Y,=$!\8G&P<8%`?'!R@*)UHN4)*````"-G!.AZ]ENB?HQ\C'"`=J)
MR\'#!0':B<.+A"2D````P<L"C;P'H>O9;HGP,=@QR`'XB=?!QP4!^(G/BXPD
MJ````,'/`HVT#J'KV6Z)V3'Y,='!R@(!\8G&P<8%`?&)UHN4)*P```"-G!.A
MZ]ENB?HQ\C'"`=J)R\'#!0':B<.+A"2P````P<L"C;P'H>O9;HGP,=@QR`'X
MB=?!QP4!^(G/BXPDM````,'/`HVT#J'KV6Z)V3'Y,='!R@(!\8G&P<8%`?&)
MUHN4)+@```"-G!.AZ]ENB?HQ\C'"`=J)R\'#!0':B<.+A"2\````P<L"C;P'
MH>O9;HGP,=@QR`'XB=?!QP4!^(G/BXPDP````,'/`HVT#J'KV6Z)V3'Y,='!
MR@(!\8G&P<8%`?&)UHN4),0```"-G!.AZ]ENB?HQ\C'"`=J)R\'#!0':B<.+
MA"3(````P<L"B=V-G`>AZ]ENB?`QZ#'(`=B)T\')`L'#!0'8B>L)RXG?B>LA
MUR'+P<H""?N+O"3,````C;0^W+P;CXG'`?/!QP6)S@'[B<\AU@G7(<?!R`()
M_HN\)-````"-O#W<O!N/B=4!]XG>"<7!Q@4AW<'+`@'WB=8AQ@GNBZPDU```
M`(V,*=R\&X^)W8G#`<Z)^0GKP<$%(?L!SHG!(>D)V8N<)-@```"-G!K<O!N/
MB?(!R\'"!<'/`@'3B>H)^HG1B>HA\2'ZP<X""<J+C"3<````C8P(W+P;CXG8
M`='!P`6)^@'!"?*)^"':(?#!RP()T(N4).````"-E!7<O!N/`<*)R,'`!0'"
MB?`)V(G%B?`AS2'8">B+K"3D````C;POW+P;CXG=`?B)U\''!0'XB<^)V<'/
M`@G](?DAU0GIBZPDZ````(VT+MR\&X\!\8G&P<H"P<8%`?&)UHGZ"?*)U8GZ
M(<4A\@GJBZPD[````(V<*]R\&X\!VHG+P<,%`=J)PXGPP<L""=B)Q8GP(<TA
MV`GHBZPD\````(V\+]R\&X^)W0'XB=?!QP4!^(G/B=G!SP()_2'Y(=7!R@()
MZ8NL)/0```"-M"[<O!N/`?&)QL'&!0'QB=:)^@GRB=6)^B'%(?()ZHNL)/@`
M``"-G"O<O!N/`=J)R\'#!0':B<.)\,'+`@G8B<6)\"'-(=@)Z(NL)/P```"-
MO"_<O!N/B=T!^(G7P<<%`?B)SXG9P<\""?TA^2'5P<H"">F+K"0``0``C;0N
MW+P;CP'QB<;!Q@4!\8G6B?H)\HG5B?HAQ2'R">J+K"0$`0``C9PKW+P;CP':
MB<O!PP4!VHG#B?#!RP*)W8GS">LAZ"'+"=B+G"0(`0``C9P?W+P;CP'8B=/!
MR0+!PP4!V(GK"<N)WXGK(=<AR\'*`@G[B[PD#`$``(VT/MR\&X^)QP'SP<<%
MB<X!^XG/(=8)UR''P<@""?Z+O"00`0``C;P]W+P;CP'^B=_!QP4!_HG7"<>)
M_8G7(=TAQ\'+`@GOBZPD%`$``(V,*=R\&X\!^8GWP<<%`?F)QPG?B?V)QR'U
M(=\)[XNL)!@!``"-E"K<O!N/BVPD"`'ZB<_!S@+!QP4!^HN\)!P!``"-A#C6
MP6+*B=\Q]S'/P<D"`<>)T,'`!0''BX0D(`$``(V<`];!8LJ)\#'(,=`!V(G[
MP<,%`=B)TXN4)"0!``#!RP*-E!;6P6+*B<XQWC'^P<\"`=:)PL'"!0'6BY0D
M*`$``(V,$=;!8LJ)VC'Z,<+!R`(!RHGQP<$%`<J)P8N$)"P!``"-A`/6P6+*
MB?LQRS'S`<.)T,'`!0'#BX0D,`$``,'.`HV\!];!8LJ)R#'P,=`!^(G?P<<%
M`?B)UXN4)#0!``#!SP*-E!'6P6+*B?$Q^3'9P<L"`=&)PL'"!0'1BY0D.`$`
M`(VT%M;!8LJ)^C':,<+!R`(!\HG.P<8%`?*)QHN$)#P!``"-A`?6P6+*B=\Q
M]S'/P<D"`<>)T,'`!0''BX0D0`$``(V<`];!8LJ)\#'(,=`!V(G[P<,%`=B)
MTXN4)$0!``#!RP*-E!;6P6+*B<XQWC'^P<\"`=:)PL'"!0'6BY0D2`$``(V,
M$=;!8LJ)VC'Z,<+!R`(!RHGQP<$%`<J)P8N$)$P!``"-A`/6P6+*B?LQRS'S
MP<X"`<.)T,'`!0'#BX0D4`$``(V\!];!8LJ)R#'P,=`!^(G?P<<%`?B)UXN4
M)%0!``#!SP*-E!'6P6+*B?$Q^3'9P<L"`=&)PL'"!0'1BY0D6`$``(VT%M;!
M8LJ)^C':,<(!\HG.P<@"P<8%`?*)QHN$)%P!``"-A`?6P6+*B=\Q]S'/P<D"
M`<>)T,'`!0''BX0D8`$``(V<`];!8LJ)\#'(,=#!R@(!V(G[P<,%`=B+G"1D
M`0``C;0>UL%BRHG+,=,Q^\'/`@'SB<;!Q@4!\XNT)&@!``"-M#76P6+*B=T!
MSHG1P<4%,?DQP<'(`@'QBW0D$(TL*8M,)`QF#V[%BVPD%`'+C0PHBT0D&&8/
M;MMF#V[)9@]BPXT,.&8/;M%F#V+*9@]LP0\1!HM$)!P!T(E&$(N$)&P!``!E
M*P44````=0N!Q'P!``!;7E]=P^AU(```C70F`)!5Z"(@``"!Q1Z/``!75E.#
M[!R)5"0(A<`/A+(```")P8N%L````(L0A=(/A*`````/M@&$P'0$/#UU53'_
MBQJ%VW0]B50D#(G8B=.)3"0$B<:-M"8`````@^P$5_]T)`Q6_Y7T````@\00
MA<!U!H`\/CUT-(MS!(/#!(7V==J)\X/$'(G86UY?7<.-M@````")SP^V1P&#
MQP&$P'0$/#UU\2G/ZYB-=@"+5"0,B?")WHG#BT0D""G6C5P[`<'^`HDP@\0<
MB=A;7E]=PXUT)@"0@\0<,=N)V%M>7UW#C70F`%;H.A\``('&/HX``%.)PX/L
M?&6A%````(E$)'0QP(U$)!104_^6V````(/$$(7`=3>+5"0<@>(`\```@?H`
M0```=`B!^@"@``!U'X/L"&H"4_^6:````(/$$(7`#Y3`#[;`ZP:-="8`,<"+
M5"1L92L5%````'4&@\1T6U[#Z`X?``"-M"8`````C;0F`````%575E.-G"0`
M`/__@>P`$```@PPD`#G<=?*#["SHEQX``('%DXT``(E$)`R)QV6A%````(F$
M)!P``0`QP&H":@!J`%?_E0@!``"#P/^#TO^)QL=$)!0`````B=")UXM4)!3!
M^!^)P8G#B?`QR,'['RG(,=H/M\")1"00BT0D$#'(*<@9VBG&&=>#Q!"%_P^(
MQ@```(DT)(U<)!2)?"0$9I!J`/]T)`C_="0(BW0D&%;_E0@!``"#Q`QH"``!
M`%-6_Y58````@\00@_@'=FJ-1`/X.=AR8HT])`$``(UT)@"0\P]^`/,/?@PO
M9@_OP68/?L%F#W/0(&8/?L()RG4RBS0D*=B+?"0$F0'P$?J+C"0<``$`92L-
M%````'5)@<0L``$`6UY?7<.-M"8`````9I"#Z`$YV'.I@00D``#__XL$)(-4
M)`3_BU0D!#4``/__]]()T`^%1____[C_____NO_____KI^B6'0``C;8`````
M55=6Z#@=``"!QCR,``!3B<.![(P```!EH10```")1"1\,<"%VW0%@#L`=22+
M1"1\92L%%`````^%/`$``('$C````%M>7UW#C;0F`````)"#[`Q3_Y9T````
MB<>#Q!"%P'3)@^P,5_^6D````(/$$(7`#X2X````@'@3+HU0$W4/@'H!`'3<
MC;0F`````&:0@'@3+@^$M@```(E4)`R#[`Q3_Y:<````BU0D'(G%B10DB50D
M'/^6G````(U$!0*)!"3_EL````")Q5B-AB2A__]:BU0D%%)34&K_:@%5_Y;L
M````@\08C40D)%!5_Y9P````@\00@_C_=!"+1"0L)0#P```]`$```'1E@^P,
M5?^6H````(/$$(/L#%7_EF0```"#Q!#I.?___XVV`````(/L#%?_EK0```")
M'"3_EM````"#Q!#IX?[__XUT)@"0@'H!+@^%0/___X!Z`@`/A/_^___I,?__
M_XVT)@````")Z.B)_O__ZY_H$AP``&:05E/HF>[__X'#O8H``(/L!(V#$.+_
M_XNS?````%"-@RJA____LX````!09?\V_Y/\````6(V#P*+__UIJ`%!0C8-&
MH?___[-`````4&7_-O^3B````(/$)%M>PXUT)@"055=64X'L`!```(,,)`"#
M[##H3AL``(''3HH``(E4)!")QF6A%````(F$)"`0```QP&C_#P``C8=;H?__
MC6PD)%50_Y?<````@\00A<!^$(G#C4#_@'P<&P!U187`=3F+AWP```"#[`3_
M="005F7_,/^74````(/$$(N$)!P0``!E*P44````=4*!Q"P0``!;7E]=PXUT
M)@")PXVV`````(/L!%-J+U7_E_````"#Q!"%P'2KBX=\````4U569?\P_Y>H
M````@\00ZZ[HX!H``%7HDAH``('%CHD``%>)QU93B=.#[#AEH10```")1"0H
M,<`QP(`Z/0^4P`'#4_^5G````(/$$(E$)`2)^(U4)!CH*_K__X7`#X03`0``
M@^P,B<90_Y6<````BTPD%(/$$#G(#X/8````BT0D&,'@`HE$)`B+A;````")
M!"0/M@>$P`^$J0$``#P]#X2A`0``B?B-M"8`````#[90`8/``832=`6`^CUU
M\"GXBPPDBW0D"(/L#(M4)!`#,8U$$`)0_Y7`````B0:#Q!"%P'11BPPDBW0D
M"(L!C4\!BQ0P#[8'B`*$P'0=/#UU"NL7C70F`(3`=`\/M@&#P@&#P0&(`CP]
M=>W&`CTQP(VT)@````"0#[8,`XA,`@&#P`&$R77QBT0D'&4K!10````/A1,!
M``"#Q"Q;7E]=PXVT)@````"0,<"-M@`````/MA0#B!0&@\`!A-)U\NO'C;0F
M`````)"+A;````"+$(D$)(LRA?8/A,(````Q]HVT)@````!FD(GP@\8!BPRR
MA<EU](T$A0P```"+C3@!``"%R75CB50D#(/L#%#'A3@!```!````_Y7`````
MB<&#Q!"%P`^$7?___XT$M0````"#[`2)1"0,4(M4)!1248E,)!S_E>````"+
M1"00BTPD'(/$$(D(QT2Q!`````")="08Z7W^__^0@^P(4%+_E3@```")P8M$
M)!"#Q!")"(7)#X0!____C02U`````(E$)`CKPXUT)@"0,<#I<_[__[@(````
M,?;I4/___^B8&```C;0F`````)#H>!@```4_AP``@^P,C9`0XO__4HV0*J'_
M__^P@````%*+D'P```!E_S+_D/P```"#Q!S#C;0F`````(UT)@"055=64X'L
M`!```(,,)`"![``0``"##"0`@^P<Z,#J__^!P^2&``!EH10```")A"0,(```
M,<#_DQ0```"#[`B-DV:A__]24(V#:J'__U!H`1```&H!C;0D)Q```%;_D^P`
M``"#Q!QH`!```(U\)!)75O^3W````(/$$(7`>$Z#[`R)QHU``5#_D\````")
MQ8/$$(7`=#:#[`165U#_DY@```#&1#4``(/$$(N$)`P@``!E*P44````=0V!
MQ!P@``")Z%M>7UW#Z(47```Q[>O:D.AH%P``!2^&``#I(?___Y!55U93@>P`
M$```@PPD`('LG````.CGZ?__@<,+A@``B[PDL!```(NL)+00``!EH10```")
MA"2,$```,<"-1"0LB40D%(G"C8-XH?__Z*_V__^#[`AJ+XG&5_^3/````(/$
M$(7`=#*-@X&A__^)^N@=_/__B?B+E"2,$```92L5%`````^%F0$``('$G!``
M`%M>7UW#C70F`(/L#%7_DP0!``!:68V+CJ'__XE,)!114/^3U````(V+D*'_
M_XG%C8.`HO__@\00B40D&(7M=)>`?0``B4PD'`^$Z@```(UV`(7V=!:#[`A5
M5O^3#`$``(/$$(7`#X2G````@^P,5?^3G````(G"C40%_X/$$#G%<A'K(XUT
M)@"0Q@``@^@!.<5T!8`X+W3Q@^P,5?^3G````(G"@\00B50D$(/L#%?_DYP`
M``"+5"0@C40"`8/$$#W_#P``#X<,____@^P$5_]T)"15_W0D*&@`$```:@&-
MK"2H````5?^3[````(/$&/]T)!Q5_Y-P````@\00A<!U$(M$)#PE`/```#T`
M@```=$^#[`C_="04:@#_D]0```")Q8/$$(7`#X2I_O__@'T```^%&?___X7V
M=`6`/BYT$+H!````C:MVH?__Z5;___^`?@$`=+OKZ(VT)@````"0@^P(:@%5
M_Y-H````@\00A<!UGHGJC8.!H?__Z(#Z__^#[`Q5_Y,$`0``@\00Z5/^___H
M214``(VT)@````!FD%;HZA0``('&[H,``%.#[`R+7"08:B]3_Y94````A<"-
M4`&)\`]%VHE<)""#Q!1;7O^@!`$``(VT)@````"-=@!55U93@>P`$```@PPD
M`(/L'.BB%```@<6>@P``B[0D,!```&6A%````(F$)`P0```QP(7V#X2F````
M@#X`#X2=````@^P,5O^5G````(U(`8/$$('Y`!````^'X0```(U<)`R#^00/
M@Z0```"%R0^%W````(U$`_\YPW(.ZQ%FD,8``(/H`3G8=`6`."]T\8/L"&HO
M4_^55````(/$$(7`=#XYPP^"A@```(/L#%/_E00!``"#Q!"+E"0,$```92L5
M%`````^%I````('$'!```%M>7UW#C;0F`````(UV`(N$)`P0``!E*P44````
M=7^-A7:A__^)A"0P$```@<0<$```B>A;7E]=_Z`$`0``D(M4#OR)WXE4#`B)
MP<'I`O.EZ4[___^-M"8`````C78`Q@``Z7+___^-M"8`````D#'`Z7#___^-
MM"8`````9I`/MA:(5"0,]L$"#X04____#[=4#OYFB50,"ND%____Z(P3``"-
MM"8`````C70F`)!55U;H*!,``('&+((``%.#["QEH10```")1"0<,<"-7"08
MBZZP````C;Z2H?__B=J)^.C9\O__A<!T+HM4)!B+10"-!)"+4`2)$(72=.!F
MD(M0"(/`!(D0A=)U](G:B?CHJ_+__X7`==*+KK````"-OINA__^)VHGXZ)+R
M__^%P'0OBU0D&(M%`(T$D(M0!(D0A=)TX(UV`(M0"(/`!(D0A=)U](G:B?CH
M8_+__X7`==&+KK````"-OJ.A__^)VHGXZ$KR__^%P'0OBU0D&(M%`(T$D(M0
M!(D0A=)TX(UV`(M0"(/`!(D0A=)U](G:B?CH&_+__X7`==&+KK````"-OJRA
M__^)VHGXZ`+R__^%P'0OBU0D&(M%`(T$D(M0!(D0A=)TX(UV`(M0"(/`!(D0
MA=)U](G:B?CHT_'__X7`==&-AK.A__^+OK````")!"2+!"2)VNBV\?__A<!T
M+(M4)!B+!XT$D(M0!(D0A=)TX(M0"(/`!(D0A=)U](L$)(G:Z(KQ__^%P'74
MBZZP````C;[#H?__B=J)^.AQ\?__A<!T-HM4)!B+10"-!)"+4`2)$(72=."-
MM"8`````C78`BU`(@\`$B1"%TG7TB=J)^.@[\?__A<!URHV&>*'__XN^L```
M`(E$)`2+1"0$B=KH'/'__X7`=#.+5"08BP>-!)"+4`2)$(72=-^-M@````"+
M4`B#P`2)$(72=?2+1"0$B=KHZ?#__X7`=<V-AL^A__^+OK````")1"0,BT0D
M#(G:Z,KP__^%P'0QBU0D&(L'C020BU`$B1"%TG3?C70F`(M0"(/`!(D0A=)U
M](M$)`R)VNB9\/__A<!USXV&V:'__XN^L````(E$)`B+1"0(B=KH>O#__X7`
M=#&+5"08BP>-!)"+4`2)$(72=-^-="8`BU`(@\`$B1"%TG7TBT0D"(G:Z$GP
M__^%P'7/BZZP````C;[CH?__B=J)^.@P\/__A<!T-8M4)!B+10"-!)"+4`2)
M$(72=."-M"8`````9I"+4`B#P`2)$(72=?2)VHGXZ/OO__^%P'7+BZZP````
MC;Z!H?__B=J)^.CB[___A<!T+XM4)!B+10"-!)"+4`2)$(72=."-=@"+4`B#
MP`2)$(72=?2)VHGXZ+/O__^%P'71B=J-ANVA___HHN___X7`=`N)PHM$)`CH
M(_7__XG:C8;^H?__Z(;O__^%P'0-B<*-AA"B___H!?7__XG:C88;HO__Z&CO
M__^)PH7`="N+1"0$Z.GT__^+1"0<92L%%````'4SBP0D@\0LC98\HO__6UY?
M7>G']/__B=J-ABNB___H*N___X7`=,V)PHM$)`SHJ_3__^O`Z(0/``"-="8`
MZ&@/```%+WX``(/L'&6+%10```")5"0,,=*-5"0(C8#/H?__Z.?N__^%P'0K
M#[80@/HP#Y3`A-(/E,()T(/P`0^VP(M4)`QE*Q44````=0R#Q!S#C70F`#'`
MZ^?H'P\``(VT)@````"-M"8`````D%/HFN'__X'#OGT``(/L"/^3K````(/L
M#%#_DZ0```"#Q!"%P'0"BP"#Q`A;PXUV`%575NB(#@``@<:,?0``4X/L+&6A
M%````(E$)!PQP(U4)!B+;"1`C;X^HO__B?CH/>[__X7`=`>`.`")PW4BBT0D
M'&4K!10````/A:4```")ZH/$+(GX6UY?7>FA\___D(/L"%50_Y:\````@\00
MA<!T'HM$)!QE*P44````=76#Q"Q;7E]=PXVT)@````!FD(/L#%7_EIP```")
M1"0<B1PD_Y:<````BU0D'(U$`@*)!"3_EL````"#Q`R)PHV&CJ'__U-0C8:`
MHO__55!J_VH!4HE4)"S_ENP```"#Q""+5"0,BT0D'&4K!10````/A%W____H
MYPT``(VT)@````!55U93Z&?@__^!PXM\``"![`P!``"+A"0@`0``C7PD*(GZ
MB40D#(V+5:+__V8/;L&-BR:B__]EH10```")A"3\````,<"-@W:A__]F#V[)
MC8L)HO__QX0DJ`````````!F#V[0C8-BHO__9@]BPF8/;MB-@UJB___'A"3$
M`````````&8/;N!F#V++9@_6A"2@````9@]NP68/8L2-@VNB__^-BV:B___'
MA"28`````````&8/;,%F#V[HC8-XH?__#Q&$)+0```!F#V[!9@]BQ8E$)!AF
M#]:$))````#HC>S__X7`=`N`.`")Q@^%3@0``/^3K````(/L#%#_DZ0```"#
MQ!"%P`^$TP,``(LHA>T/A,D#``"#[`Q5_Y.<````C40``8/$$(/L#%#_D\``
M``")Q@^V10"#Q!")="0(A,!T/(V+=*+__XE\)`2)[XGUB<Z-="8`D(/L#`^V
MP(/'`5!6:O]J`56#Q0+_D^P````/M@>#Q""$P'7=BWPD!(VT)+````"-@Q"B
M___K#&:0BT8$@\8$A<!T-HGZZ,_K__^)Q87`=.F`.`!TY.B?[/__A<!TVX/L
M#%7_DP0!``")Q8/$$(7`=3B-M"8`````D(VT)*````"++H7M=".`?0``#X3Q
M`@``B>CH8NS__X7`#X7R````BVX$@\8$A>UUW8/L#%7_DYP```!:_W0D%(G&
M_Y.<````C;0&!`0``(DT)/^3P````(E$)!2)P8V#>:+___]T)!A0C8.0H?__
MB40D(%"-@WZB__]5B<U0:O]J`5'_D^P```"#Q"AHP`$``%7_DQ@```"#Q!"#
M^/]U$?^3$````(L`@_@1#X6E`@``@^P(5_]T)!#_D]@```")Q8/$$(7`='F#
M[`S_="00BT0D'/\PC8,HH___4#'VBX.4````:@'_,/^3+````(/$((N$)/P`
M``!E*P44````#X4U!```@<0,`0``B?!;7E]=PXVT)@````"#[`R#Q@15_Y,$
M`0``B<6#Q!"%P`^$U?[__^GY_O__C;0F`````&:0BT0D."4`\```/0!````/
MA7/___^+1"1`B40D$/^3K````#E$)!`/A5O___^+1"0X)?\!```]P`$```^%
M1____X/L#%;_D\````")^HG&C8-)HO__Z`?J__^#Q!"%P`^$+`(``(/L"%"+
M1"08_S!GZ.SR__^)1"0@@\00A<`/A`T"``"#[`S_="0<_Y.<````B<*#Q!"#
M^`-V1H/L"(V#AZ+__U"+1"0<C400_(E4)"A0_Y,,`0``@\00A<!U(HM4)!R#
M^@0/A-4"``"+1"00@'P0^R\/A,8"``"-M@````")^HV[SZ'__XGXZ''I__^%
MP`^$J0$```^V`(3`#X2>`0``/#`/A)8!``"-DSRB__^)^(V[P*+__^C3[O__
M_Y,4````5U"-@Z"B____="00_W0D$%!J_VH!5O^3[````(V#KJ+__X/$((E$
M)`SK-F:0_Y,0````@S@1=4#_DQ0```"#Q0&#[`Q755#_="0@_W0D(/]T)"QJ
M_VH!5O^3[````(/$,(/L"&C``0``5O^3&````(/$$(/X_W2U@^P,_W0D$/^3
M9````(M$)"B)\N@][O__B30D9^@T^O__@\00Z>S]__^-="8`,>WI(?W__Y"-
MM"20````ZPJ-M"8`````@\8$BP:%P`^$?0```(GZZ&SH__^%P'3H@#@`=..#
M[`Q0_Y,$`0``B<6#Q!"%P'30Z?+[__^#[`A0_W0D$(M$)!S_,(V#[*+__^EL
M_?__9I"#[`Q09^BV^?__@\00BX0D_````&4K!10````/A:@!``")M"0@`0``
M@<0,`0``B=A;7E]=_Z`$`0``@^P,C:M.HO__:@W_D\````")QH/$$(E$)`BX
M4P```.FE^___9I"+1"0,BP")1"00Z>3]__^0@^P(:@#_="0<_Y,H````@\00
MA<`/A%#^__^)1"0,Z`WI__^)1"00B50D%(72#X@W_O__@\#V:@"#TO]24(M,
M)!A1B4PD'/^3"`$``(/$#&H&C80DD@```%"+3"0848E,)!S_DU@```"#Q!"+
M3"0,@;PDB@````!#04,/A>K]__]F@;PDC@```$A%#X7:_?__:@"+1"04BU0D
M&(/`SH/2_U)048E,)!S_DP@!``"#Q`QJ*(V\)-L```!7BTPD&%'_DU@```"-
M@\"B___&A"0+`0```%"-@Y&B__]7_W0D(/]T)"!0:O]J`5;_D^P```"#Q##I
M!/[__XVT)@````!FD(M$)`R+4`2%T@^$,?W__X/L#%*)5"0<_Y.<````@\00
M@_@##X87_?__@^P(C8N,HO__48M4)!B-1`+\4/^3#`$``(M4)!R#Q!"%P`]%
M5"00B50D$.GG_/__Z-(&``!FD%575E/H5]G__X'#>W4``(/L'&6A%````(E$
M)`PQP(U\)`B)^HNS?````&6++@^V162(166-@[.A___H(>;__X7`=`F`.#$/
MA%0!``!GZ/[R__]EBRZ-@[^B__]J`VH!4%7_DTP```")Q8/$$(7`="J#[`2-
M@\&B__]0BT4,_S!E_S;_DU````"+10R+`(/$$/9`"D`/A38!``"-@\:B__]J
M`VH!4&7_-O^33````(G%@\00A<!T*HGZC8.!H?__Z)SE__^%P`^$Y````(/L
M!%"+10S_,&7_-O^34````(/$$(/L!(V#R*+__VH!4&7_-O^3(````(V3W:+_
M_X/$#%)09?\V_Y-0````98L&B?K&0&4`C8-XH?__Z$+E__^#Q!")@T`!``")
MQH7`=$.#[`AHP`$``%;_DQ@```"-4`&#Q!"#^@%V"8/X$0^%E0```(M$)`QE
M*P44````#X6;````@\0<6UY?7<.-M"8`````@^P,_[-$`0``9^AQ]___B8-`
M`0``B<:#Q!"%P'6?@^P,C8-\H___4/^3Y````&:0@'@!``^$J_[__^F=_O__
MD(N#1`$``(L0BT4,BP#H7NG__^D:____C;0F`````&:0@^P(4&7_-O^3R```
M`(/$$.FU_O__4(N#1`$``%;_,(V#H*/__U#_D^0```#HSP0``(VT)@````"-
MM"8`````D%=6Z&D$``"!QFUS``!3@^P<9:$4````B40D&#'`BWPD+%=GZ)CO
M__]:66HOB<-0_Y94````A<"-4`$/1=J)'"3_E@0!``"-5"08B<.-AL^A___H
M^>/__X/$$(7`="(/MA"$T@^4P(#Z,`^4P@C0=1"%_W0,@#\`=1^-M"8`````
MBT0D#&4K!10```!U/X/$$%M>7\.-="8`@^P$C89YHO__:@104_^6]````(/$
M$(7`=<Z+1"0,92L%%````'4-@\00B?A;7E_I9.;__^CO`P``C;0F`````(VT
M)@````"04^AJUO__@<..<@``@^P4:F#_D\````!F#V^#'.+__[GPX=+#,=N)
M2!")6!3'0!@`````QT!<``````\1`(/$&%O#C;0F`````(VT)@````!55U;H
M.`,``('&/'(``%.#[!R+7"0PBT0D.(M\)#3!X`,#0Q2+2QB-:QR)0Q2+1"0X
M@]$`P>@=`<B)0QB+0UR%P`^%^0```(-\)#@_#XX^`0``BT0D.(/H0(E$)`S!
MZ`:-<`&)1"0(P>8&`?Z-="8`\P]O!XG8@\=`#Q%%`/,/;T_0#Q%-$/,/;U?@
M#Q%5(/,/;U_P#Q%=,.B#UO__.?=USXM$)`B+5"0,P>`&*<*)5"0X@WPD.`1R
M5XL&B>F)0QR+1"0XBU0D.(M$!OR)1!7\C44$@^#\*<$IS@-,)#B#X?R#^01R
M$H/A_#'2BRP6B2P0@\($.<IR\XM$)#B)0UR#Q!Q;7E]=PXVT)@````!FD(M$
M)#B%P'3@#[8&B$,<]D0D.`)TTXM$)#B+3"0X#[=$!OYFB40-_NN_C;0F````
M`+I`````*<([5"0X#T]4)#B#[`0!Z%*)5"005U#_EI@```"+5"08BT-<`="#
MQ!")0UR#^$!UB8G8*50D.`'7Z)W5___IO_[__XVT)@````"0B?[I&____XVT
M)@````!FD%575NB8`0``@<:<<```4X/L'(M<)#2+?"0PBVL8BTL4#\V)Z(G-
MB40D"(G(#\W!Z`.);"0,C6L<@^`_C5`!QD0#'("-1!4`@_HX#X[@````N4``
M```IT70-,=+&!!``@\(!.<IR]8G8Z!/5__]F#^_`#Q%#'`\112`/$440#Q%%
M*/,/?DPD"(G89@_62U3H[=3__XL#P>@8B`>+`\'H$(A'`8L#B&<"BP.(1P,/
MMD,'B$<$#[=#!HA'!8M#!(AG!HM#!(A'!P^V0PN(1P@/MT,*B$<)BT,(B&<*
MBT,(B$<+#[9##XA'#`^W0PZ(1PV+0PR(9PZ+0PR(1P\/MD,3B$<0#[=#$HA'
M$8M#$(AG$HM#$(A'$XGPB5PD,(/$'%M>7UW_H&0```"-M"8`````9I"Y.```
M`#'M*=&#^01R0HU0!,<``````(/B_,=$"/P`````*=`!R(/@_(/X!`^"'___
M_X/@_#')B2P*@\$$.<%R]ND+____C;0F`````(UV`(7)#X3Y_O__Q@``]L$"
M#X3M_O__,=)FB50(_NGA_O__BS0DPXL\),.++"3#9I!FD/,/'OOH)`````7K
M;@``@^P,_[`@`0``_W0D'/]T)!S_="0<_Y!$````@\0<PXL$),-FD&:09I!F
MD&:09I!FD)#S#Q[[5E[HXO___P6I;@``@^P,_Y#,````````\P\>^U.#[`CH
M9]+__X'#BVX``(/$"%O#````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````P````$``@`E<R\E<P!$>6YA3&]A9&5R.CIB;V]T7T1Y;F%,;V%D
M97(`26YT97)N86QS.CI005(Z.D)/3U0`+W!R;V,O<V5L9B]E>&4`+W!R;V,O
M)6DO)7,`+@!005)?5$5-4`!005)?4%)/1TY!344`.@`O`%!%4DPU3$E"`%!%
M4DQ,24(`4$523#5/4%0`4$523$E/`%!!4E])3DE424%,25I%1`!005)?4U!!
M5TY%1`!005)?0TQ%04X`4$%27T1%0E5'`%!!4E]#04-(10!005)?1TQ/0D%,
M7T1%0E5'`%!!4E]'3$]"04Q?5$U01$E2`%!!4E]435!$25(`4$%27T=,3T)!
M3%]414U0`%!!4E]'3$]"04Q?0TQ%04X`,0!,1%],24)205)97U!!5$@`4UE3
M5$5-`"]T;7``5$5-4$1)4@!435``55-%4@!54T523D%-10`E,#)X`'!A<BT`
M)7,E<R5S)7,`<&%R;``N<&%R`"5S)7-C86-H92TE<R5S`"5S)7-T96UP+25U
M)7,`)7,E<W1E;7`M)74M)74E<P`8`'!E<FP`,`!005(Z.E!A8VME<CHZ5D52
M4TE/3@`Q+C`U-P`M90`M+0`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!S
M=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*```E<SH@<')I=F%T
M92!S=6)D:7)E8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E(&ET
M(&%N9"!R971R>2!Y;W5R(&]P97)A=&EO;BD*``!5;F%B;&4@=&\@8W)E871E
M(&-A8VAE(&1I<F5C=&]R>0`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!C
M86-H92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*`````&5V
M86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P
M.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?7W!A
M<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D8V%C
M:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M
M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P86-K*").(BD*)&-A
M8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S7#`P
M-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*;7D@
M)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE(#T@
M+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@)&-H
M=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@;V8@
M)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@)'!O
M<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H*"10
M05)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?34%'
M24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D8VAU
M;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R;V=N
M86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[
M4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL
M93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]
M($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;
M4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ
M9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P
M87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F
M)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M
M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N
M86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)
M3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R
M:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@
M9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@
M73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N
M90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@
M)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^
M)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U
M;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]
M"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE
M9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?96YV
M*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@/2`H
M"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I
M+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I
M;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>
M35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D[
M"F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S971?<&%R
M7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?
M5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S96-T
M:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT[
M"G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO=71S
M*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C(&1O
M;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA<W0@
M34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?1D@I
M.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD(&UA
M9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE(ETI
M.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P;W-I
M=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)
M3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A9"!?
M1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B7#!#
M04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C:&4@
M;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H92!M
M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@
M)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,15]O
M9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I.PIO
M=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T(BD[
M"G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2`D
M;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?1D@[
M"FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@
M("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H
M:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A
M9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@
M/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN
M72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO
M?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B
M=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L
M(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E
M9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I
M*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L
M(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D
M9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]
M("1F=6QL;F%M93L*?0IE;'-I9B`H("1F=6QL;F%M92`]?B!M?%XO/W-H;&EB
M+WP@86YD(&1E9FEN960@)$5.5GM005)?5$5-4'T@*2!["FUY("1F:6QE;F%M
M92`](%]S879E7V%S*"(D8F%S96YA;64D97AT(BP@)&)U9BP@,#<U-2D["F]U
M=',H(E-(3$E".B`D9FEL96YA;65<;B(I.PI]"F5L<V4@>PHD36]D=6QE0V%C
M:&5[)&9U;&QN86UE?2`]('L*8G5F(#T^("1B=68L"F-R8R`]/B`D8W)C+`IN
M86UE(#T^("1F=6QL;F%M92P*?3L*)')E<75I<F5?;&ES='LD9G5L;&YA;65]
M*RL["GT*<F5A9"!?1D@L("1B=68L(#0["GT*;&]C86P@0$E.0R`]("AS=6(@
M>PIM>2`H)'-E;&8L("1M;V1U;&4I(#T@0%\["G)E='5R;B!I9B!R968@)&UO
M9'5L92!O<B`A)&UO9'5L93L*;7D@)&EN9F\@/2`D36]D=6QE0V%C:&5[)&UO
M9'5L97T["G)E='5R;B!U;FQE<W,@)&EN9F\["F1E;&5T92`D<F5Q=6ER95]L
M:7-T>R1M;V1U;&5].PHD24Y#>R1M;V1U;&5](#T@(B]L;V%D97(O)&EN9F\O
M)&UO9'5L92(["FEF("@D14Y6>U!!4E]#3$5!3GT@86YD(&1E9FEN960H)DE/
M.CI&:6QE.CIN97<I*2!["FUY("1F:"`]($E/.CI&:6QE+3YN97=?=&UP9FEL
M92!O<B!D:64@(D-A;B=T(&-R96%T92!T96UP(&9I;&4Z("0A(CL*)&9H+3YB
M:6YM;V1E*"D["B1F:"T^<')I;G0H)&EN9F\M/GMB=69]*3L*)&9H+3YS965K
M*#`L(#`I.PIR971U<FX@)&9H.PI]"F5L<V4@>PIM>2`D9FEL96YA;64@/2!?
M<V%V95]A<R@B)&EN9F\M/GMC<F-]+G!M(BP@)&EN9F\M/GMB=69]*3L*)&EN
M9F\M/GMF:6QE?2`]("1F:6QE;F%M93L*;W!E;B!M>2`D9F@L("<\.G)A=R<L
M("1F:6QE;F%M92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)&9I;&5N86UE(CH@
M)"%=.PIR971U<FX@)&9H.PI]"F1I92`B0F]O='-T<F%P<&EN9R!F86EL960Z
M(&-A;B=T(&9I;F0@;6]D=6QE("1M;V1U;&4A(CL*?2P@0$E.0RD["G)E<75I
M<F4@6%-,;V%D97(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!#87)P
M.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["E!!4CHZ2&5A=GDZ
M.E]I;FET7V1Y;F%L;V%D97(H*3L*<F5Q=6ER92!)3SHZ1FEL93L*=VAI;&4@
M*&UY("1F:6QE;F%M92`]("AS;W)T(&ME>7,@)7)E<75I<F5?;&ES="E;,%TI
M('L*=6YL97-S("@D24Y#>R1F:6QE;F%M97T@;W(@)&9I;&5N86UE(#U^("]"
M4T1004XO*2!["FEF("@D9FEL96YA;64@/7X@+UPN<&UC/R0O:2D@>PIR97%U
M:7)E("1F:6QE;F%M93L*?0IE;'-E('L*9&\@)&9I;&5N86UE('5N;&5S<R`D
M9FEL96YA;64@/7X@+W-I=&5C=7-T;VUI>F5<+G!L)"\["GT*?0ID96QE=&4@
M)')E<75I<F5?;&ES='LD9FEL96YA;65].PI]"FEF("@D8G5F(&YE("102UI)
M4%]-04=)0RD@>PIO=71S*'%Q6TYO('II<"!F;W5N9"!A9G1E<B!&24Q%('-E
M8W1I;VX@:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL87-T($U!1TE#(#L*?0HD
M<W1A<G1?<&]S(#T@*'1E;&P@7T9(*2`M(#0[("`@("`@("`@("`@("`@(",@
M<W1A<G0@;V8@>FEP"GT*;7D@0'!A<E]A<F=S.PIM>2`H)&]U="P@)&)U;F1L
M92P@)&QO9V9H+"`D8V%C:&5?;F%M92D["F1E;&5T92`D14Y6>U!!4E]!4%!?
M4D554T5].R`C('-A;FET:7IE("A2155312!M87D@8F4@82!S96-U<FET>2!P
M<F]B;&5M*0II9B`H(21S=&%R=%]P;W,@;W(@*"1!4D=66S!=(&5Q("<M+7!A
M<BUO<'1I;VYS)R`F)B!S:&EF="DI('L*;7D@)61I<W1?8VUD(#T@<7<H"G`@
M("!B;&EB7W1O7W!A<@II("`@:6YS=&%L;%]P87(*=2`@('5N:6YS=&%L;%]P
M87(*<R`@('-I9VY?<&%R"G8@("!V97)I9GE?<&%R"BD["FEF("A`05)'5B!A
M;F0@)$%21U9;,%T@97$@)RTM<F5U<V4G*2!["G-H:69T($!!4D=6.PHD14Y6
M>U!!4E]!4%!?4D554T5](#T@<VAI9G0@0$%21U8["GT*96QS92![(",@;F]R
M;6%L('!A<FP@8F5H879I;W5R"FUY($!A9&1?=&]?:6YC.PIW:&EL92`H0$%2
M1U8I('L*)$%21U9;,%T@/7X@+UXM*%M!24U/0DQB<7!I=7-4=ETI*"XJ*2\@
M;W(@;&%S=#L*:68@*"0Q(&5Q("=))RD@>PIP=7-H($!A9&1?=&]?:6YC+"`D
M,CL*?0IE;'-I9B`H)#$@97$@)TTG*2!["F5V86P@(G5S92`D,B(["GT*96QS
M:68@*"0Q(&5Q("=!)RD@>PIU;G-H:69T($!P87)?87)G<RP@)#(["GT*96QS
M:68@*"0Q(&5Q("=/)RD@>PHD;W5T(#T@)#(["GT*96QS:68@*"0Q(&5Q("=B
M)RD@>PHD8G5N9&QE(#T@)W-I=&4G.PI]"F5L<VEF("@D,2!E<2`G0B<I('L*
M)&)U;F1L92`]("=A;&PG.PI]"F5L<VEF("@D,2!E<2`G<2<I('L*)'%U:65T
M(#T@,3L*?0IE;'-I9B`H)#$@97$@)TPG*2!["F]P96X@)&QO9V9H+"`B/CXB
M+"`D,B!O<B!D:64@<7%;0V%N)W0@;W!E;B!L;V<@9FEL92`B)#(B.B`D(5T[
M"GT*96QS:68@*"0Q(&5Q("=4)RD@>PHD8V%C:&5?;F%M92`]("0R.PI]"G-H
M:69T*$!!4D=6*3L*:68@*&UY("1C;60@/2`D9&ES=%]C;61[)#%]*2!["F1E
M;&5T92`D14Y6>R=005)?5$5-4"=].PII;FET7VEN8R@I.PIR97%U:7)E(%!!
M4CHZ1&ES=#L*)GLB4$%2.CI$:7-T.CHD8VUD(GTH*2!U;FQE<W,@0$%21U8[
M"B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"1?*2!F;W(@0$%21U8["F5X:70["GT*
M?0IU;G-H:69T($!)3D,L($!A9&1?=&]?:6YC.PI]"GT*:68@*"1O=70I('L*
M>PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q
M=6ER92!$:6=E<W0Z.E-(03L*?0IM>2`D<&%R(#T@<VAI9G0H0$%21U8I.PIM
M>2`D>FEP.PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)&9H+"`G/#IR
M87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!F:6YD('!A<B!F:6QE("(D<&%R
M(CH@)"%=.PIB;&5S<R@D9F@L("=)3SHZ1FEL92<I.PHD>FEP(#T@07)C:&EV
M93HZ6FEP+3YN97<["B@@)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@
M)'!A<BD@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I("D*;W(@9&EE('%Q6T5R
M<F]R(')E861I;F<@>FEP(&%R8VAI=F4@(B1P87(B73L*?0IM>2`E;65T85]P
M87(@/2!D;R!["FEF("@D>FEP(&%N9"!M>2`D;65T82`]("1Z:7`M/F-O;G1E
M;G1S*"=-151!+GEM;"<I*2!["B1M971A(#U^(',O+BI><&%R.B0O+VUS.PHD
M;65T82`]?B!S+UY<4RXJ+R]M<SL*)&UE=&$@/7X@+UX@("A;7CI=*RDZ("@N
M*RDD+VUG.PI]"GT["FEF("AD969I;F5D("1P87(I('L*;W!E;B!M>2`D<&@L
M("<\.G)A=R<L("1P87(@;W(@9&EE('%Q6T-A;B=T(')E860@<&%R(&9I;&4@
M(B1P87(B.B`D(5T["FUY("1B=68["G)E860@)'!H+"`D8G5F+"`T.PID:64@
M<7%;(B1P87(B(&ES(&YO="!A('!A<B!F:6QE72!U;FQE<W,@)&)U9B!E<2`D
M4$M:25!?34%'24,["F-L;W-E("1P:#L*?0I#<F5A=&50871H*"1O=70I(#L*
M;7D@)&9H(#T@24\Z.D9I;&4M/FYE=R@*)&]U="P*24\Z.D9I;&4Z.D]?0U)%
M050H*2!\($E/.CI&:6QE.CI/7U)$5U(H*2!\($E/.CI&:6QE.CI/7U1254Y#
M*"DL"C`W-S<L"BD@;W(@9&EE('%Q6T-A;B=T(&-R96%T92!F:6QE("(D;W5T
M(CH@)"%=.PHD9F@M/F)I;FUO9&4H*3L*<V5E:R!?1D@L(#`L(#`["FUY("1L
M;V%D97(["FEF("AD969I;F5D("1S=&%R=%]O9E]&24Q%7W-E8W1I;VXI('L*
M<F5A9"!?1D@L("1L;V%D97(L("1S=&%R=%]O9E]&24Q%7W-E8W1I;VX["GT@
M96QS92!["FQO8V%L("0O(#T@=6YD968["B1L;V%D97(@/2`\7T9(/CL*?0II
M9B`H(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&QO861E<B`]?B`O7B@_.B,A
M?%Q`<F5M*2\I('L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["E!!
M4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1L;V%D97(L("0P*3L*?0HD
M9F@M/G!R:6YT*"1L;V%D97(I"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG(&QO
M861E<B!T;R`B)&]U="(Z("0A73L*:68@*"1B=6YD;&4I('L*<F5Q=6ER92!0
M05(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["FEN
M:71?:6YC*"D["G)E<75I<F5?;6]D=6QE<R@I.PIM>2!`:6YC(#T@9W)E<"![
M("$O0E-$4$%.+R!]"F=R97`@>PHH)&)U;F1L92!N92`G<VET92<I(&]R"B@D
M7R!N92`D0V]N9FEG.CI#;VYF:6=[87)C:&QI8F5X<'T@86YD"B1?(&YE("1#
M;VYF:6<Z.D-O;F9I9WMP<FEV;&EB97AP?2D["GT@0$E.0SL*<WPO*WPO?&<L
M('-\+R1\?"!F;W)E86-H($!I;F,["FUY("5F:6QE<SL*)&9I;&5S>R1??2LK
M(&9O<B!`1'EN84QO861E<CHZ9&Q?<VAA<F5D7V]B:F5C=',L('9A;'5E<R`E
M24Y#.PIM>2`D;&EB7V5X="`]("1#;VYF:6<Z.D-O;F9I9WML:6)?97AT?3L@
M("`@("`@("`C(%A86"!L:6)?97AT('9S(&1L97AT(#\*;7D@)7=R:71T96X[
M"F9O<F5A8V@@;7D@)&ME>2`H<V]R="!K97ES("5F:6QE<RD@>PIM>2`H)&9I
M;&4L("1N86UE*3L*:68@*&1E9FEN960H;7D@)&9C(#T@)%!!4CHZ2&5A=GDZ
M.D9U;&Q#86-H97LD:V5Y?2DI('L**"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@
M)&9C*3L*?0IE;'-E('L*9F]R96%C:"!M>2`D9&ER("A`:6YC*2!["FEF("@D
M:V5Y(#U^(&U\7EQ1)&1I<EQ%+R@N*BDD?&DI('L**"1F:6QE+"`D;F%M92D@
M/2`H)&ME>2P@)#$I.PIL87-T.PI]"FEF("@D:V5Y(#U^(&U\7B]L;V%D97(O
M6UXO72LO*"XJ*21\*2!["FEF("AM>2`D<F5F(#T@)$UO9'5L94-A8VAE>R0Q
M?2D@>PHH)&9I;&4L("1N86UE*2`]("@D<F5F+"`D,2D["FQA<W0["GT*:68@
M*"UF("(D9&ER+R0Q(BD@>PHH)&9I;&4L("1N86UE*2`]("@B)&1I<B\D,2(L
M("0Q*3L*;&%S=#L*?0I]"GT*?0IN97AT('5N;&5S<R!D969I;F5D("1N86UE
M.PIN97AT(&EF("1W<FET=&5N>R1N86UE?2LK.PIN97AT(&EF("%R968H)&9I
M;&4I(&%N9"`D9FEL92`]?B`O7"Y<421L:6)?97AT7$4D+VD["F]U=',H<W!R
M:6YT9BAQ<5M086-K:6YG($9)3$4@(B5S(BXN+ETL(')E9B`D9FEL92`_("1F
M:6QE+3Y[;F%M97T@.B`D9FEL92DI.PIM>2`D8V]N=&5N=#L*:68@*')E9B@D
M9FEL92DI('L*)&-O;G1E;G0@/2`D9FEL92T^>V)U9GT["GT*96QS92!["FQO
M8V%L("0O(#T@=6YD968["F]P96X@;7D@)'1H+"`G/#IR87<G+"`D9FEL92!O
M<B!D:64@<7%;0V%N)W0@<F5A9"`B)&9I;&4B.B`D(5T["B1C;VYT96YT(#T@
M/"1T:#X["F-L;W-E("1T:#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YA<'!L
M>2A<)&-O;G1E;G0L("(\96UB961D960^+R1N86UE(BD*:68@(21%3E9[4$%2
M7U9%4D)!5$E-?2!A;F0@)&YA;64@/7X@+UPN*#\Z<&U\:7A\86PI)"]I.PI0
M05(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT+3YN97<M/F%P<&QY*%PD8V]N=&5N
M="P@)&9I;&4L("1N86UE*3L*?0HD9F@M/G!R:6YT*")&24Q%(BP*<&%C:R@G
M3B<L(&QE;F=T:"@D;F%M92D@*R`Y*2P*<W!R:6YT9B@B)3`X>"\E<R(L($%R
M8VAI=F4Z.EII<#HZ8V]M<'5T94-20S,R*"1C;VYT96YT*2P@)&YA;64I+`IP
M86-K*"=.)RP@;&5N9W1H*"1C;VYT96YT*2DL"B1C;VYT96YT*0IO<B!D:64@
M<7%;17)R;W(@=W)I=&EN9R!E;6)E9&1E9"!&24Q%('1O("(D;W5T(CH@)"%=
M.PIO=71S*'%Q6U=R:71T96X@87,@(B1N86UE(ETI.PI]"GT*:68@*"1Z:7`I
M('L*)'II<"T^=W)I=&54;T9I;&5(86YD;&4H)&9H*2`]/2!!<F-H:79E.CI:
M:7`Z.D%:7T]+"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('II<"!P87)T(&]F
M("(D;W5T(ET["GT*:68@*"1M971A7W!A<GMC;&5A;GTI('L*)&9H+3YP<FEN
M="@B7#!#3$5!3B(I.PI]"F5L<V4@>PII9B`H(61E9FEN960@)&-A8VAE7VYA
M;64I('L*;7D@)&-T>"`]($1I9V5S=#HZ4TA!+3YN97<H,2D["G-E96L@)&9H
M+"`P+"`P.PHD8W1X+3YA9&1F:6QE*"1F:"D["G-E96L@)&9H+"`P+"`R.PHD
M8V%C:&5?;F%M92`]("1C='@M/FAE>&1I9V5S=#L*?0HD8V%C:&5?;F%M92`]
M('-U8G-T<B`D8V%C:&5?;F%M92P@,"P@)&-A8VAE7VYA;65?<VEZ93L*)&-A
M8VAE7VYA;64@+CT@(EPP(B!X("@D8V%C:&5?;F%M95]S:7IE("T@;&5N9W1H
M("1C86-H95]N86UE*3L*)&9H+3YP<FEN="@D8V%C:&5?;F%M92P*(EPP0T%#
M2$4B*3L*?0IM>2`D;V9F<V5T(#T@)&9H+3YT96QL("T@;&5N9W1H*"1L;V%D
M97(I.PHD9F@M/G!R:6YT*'!A8VLH)TXG+"`D;V9F<V5T*2P*)%!!4E]-04=)
M0RD["B1F:"T^8VQO<V4@;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@=')A:6QE
M<B!O9B`B)&]U="(Z("0A73L*8VAM;V0@,#<U-2P@)&]U=#L*97AI=#L*?0I[
M"FQA<W0@=6YL97-S(&1E9FEN960@)'-T87)T7W!O<SL*7V9I>%]P<F]G;F%M
M92@I.PIR97%U:7)E(%!!4CL*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E
M<B@I.PI["G)E<75I<F4@1FEL93HZ1FEN9#L*<F5Q=6ER92!!<F-H:79E.CI:
M:7`["GT*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=SL@("`@("`@("`@("`@("`@
M("`@("`@("`@("`@(",@07)C:&EV93HZ6FEP(&]P97)A=&5S(&]N(&%N($E/
M.CI(86YD;&4*)&9H+3YF9&]P96XH9FEL96YO*%]&2"DL("=R)RD@;W(@9&EE
M('%Q6V9D;W!E;B@I(&9A:6QE9#H@)"%=.PI!<F-H:79E.CI::7`Z.G-E=$-H
M=6YK4VEZ92@M<R!?1D@I.PIM>2`D>FEP(#T@07)C:&EV93HZ6FEP+3YN97<[
M"B@D>FEP+3YR96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<')O9VYA;64I(#T]
M($%R8VAI=F4Z.EII<#HZ05I?3TLH*2D*;W(@9&EE('%Q6T5R<F]R(')E861I
M;F<@>FEP(&%R8VAI=F4@(B1P<F]G;F%M92)=.PI!<F-H:79E.CI::7`Z.G-E
M=$-H=6YK4VEZ92@V-"`J(#$P,C0I.PIP=7-H($!005(Z.DQI8D-A8VAE+"`D
M>FEP.PHD4$%2.CI,:6)#86-H97LD<')O9VYA;65](#T@)'II<#L*;W5T<R@B
M17AT<F%C=&EN9R!Z:7`N+BXB*3L*:68@*&1E9FEN960@)$5.5GM005)?5$5-
M4'TI('L@(R!S:&]U;&0@8F4@<V5T(&%T('1H:7,@<&]I;G0A"F9O<F5A8V@@
M;7D@)&UE;6)E<B`H("1Z:7`M/FUE;6)E<G,@*2!["FYE>'0@:68@)&UE;6)E
M<BT^:7-$:7)E8W1O<GD["FUY("1M96UB97)?;F%M92`]("1M96UB97(M/F9I
M;&5.86UE.PIN97AT('5N;&5S<R`D;65M8F5R7VYA;64@/7X@;7L*7@HO/W-H
M;&EB+PHH/SHD0V]N9FEG.CI#;VYF:6=[=F5R<VEO;GTO*3\**#\Z)$-O;F9I
M9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R
M86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A
M=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@
M)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I
M>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E
M(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@
M>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T
M7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E
M<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;
M17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE(ET[
M"F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*
M?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I
M92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@
M72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A
M;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.
M5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'
M5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I
M<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT
M*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I
M*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A
M=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O
M<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER
M92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T
M.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R
M<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER
M92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@
M1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C
M.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)
M3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S
M.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z
M.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER
M92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR
M97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T
M<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E
M(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L
M('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E
M<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I
M<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R
M97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I
M;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK
M*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P
M871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%
M35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*
M;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW
M("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I
M9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN
M960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z
M.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N
M86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).
M04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP9&ER
M(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R
M;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER.PII
M9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"([
M"GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@
M9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM>2`D
M;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS965K
M("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H
M*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L96YG
M=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E96L@
M)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H
M(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H+"`D
M8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C86-H
M95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@*"$D
M8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[4$%2
M7T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C86-H
M95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-4'T@
M/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!
M4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?
M<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`7SL*
M;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H+64@
M)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B.PIM
M>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H<')I
M;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q6T5R
M<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L("1T
M96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D
M9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L
M;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6
M>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK
M*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@
M*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%2
M7U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L
M(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II9B`H
M(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S>7-[
M7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R;V=N
M86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA;64D
M<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T=7)N
M.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?<V5P
M?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!4E]4
M14U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S+UQ1
M)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D
M96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER
M)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N86UE
M(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@
M(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O
M9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345]
M.PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@>PHD
M<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W
M9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@
M*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z
M(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN
M/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6>U!!
M4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E;G8@
M>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E='5R
M;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF;W(@
M*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@
M*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-4$1)
M4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D14Y6
M>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"04Q?
M)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[4$%2
M7T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@
M*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN
M="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*
M<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[
M)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I
M=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K
M86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF
M:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A<BYP
M;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C
M:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE.PID
M;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58
M251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]
M("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^
M+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ
M7U]%4E)/4CL*,3L*7U]%3D1?7PH``$-!0TA%``!P97)L>'-I+F,````!(T5G
MB:O-[_[<NIAV5#(0`1L#.Q`!```A````('___T`!``!`?___;`P``("!__\L
M`0``L(+__V0!``"PCO__J`$``)"/__\@`@``()#__VP"``"0D?__``,``!"3
M__^T`P``<)/__P0$``!`E/__E`0``)"6__\4!0``T);__S@%``"@E___T`4`
M`+"7___D!0``X)G__^0&```@FO__%`<``*";__^8!P``H)___]@'```0H/__
M]`<``$"@__\@"```0*'__Z0(``!0J/__-`L``&"J__\$#```0*O__[0,``"0
MJ___V`P``#"M__\H#0``T*[__V@-``#4KO__?`T``-BN__^0#0``X*[__Z0-
M```-K___R`T``""O___<#0``%``````````!>E(``7P(`1L,!`2(`0``$```
M`!P```!,@/__,`````!$!P@@````,````-A]__\@``````X(1@X,2@\+=`1X
M`#\:.RHR)")`````5````$2!___["P```$$."(4"00X,AP-!#A"&!$$.%(,%
M2`Z0`P/E"PH.%$'##A!!Q@X,0<<."$'%#@1!"P```'0```"8`````(W__]P`
M````00X(A0),#@R'`T$.$(8$00X4@P5##C`"20XT00XX1`X\00Y`20XP60H.
M%$/##A!!Q@X,0<<."$'%#@1'"W0*#A1#PPX00<8.#$''#@A!Q0X$1@M##A1%
MPPX00<8.#$''#@A!Q0X$`$@````0`0``:(W__X(`````00X(A@),#@R#`T4.
MB`%1#HP!00Z0`4D.@`%A#H@!0@Z,`4$.D`%)#H`!8`H.#$'##@A!Q@X$00L`
M``"0````7`$``*R-__]J`0```$$."(4"00X,AP-!#A"&!$$.%(,%1PP#E(`$
M3@T$0P[`@`1B#L2`!$(.R(`$0@[,@`1!#M"`!`)(#L"`!%<.Q(`$1`[(@`1$
M#LR`!$4.T(`$20[$@`1%#LB`!$$.S(`$00[0@`1)#L"`!`):"@X40<,.$$'&
M#@Q!QPX(0<4.!$H+````L````/`!``"(CO__?@$```!!#@B%`D$.#(<#00X0
MA@1,#A2#!4@.H`%L"@X40<,.$$'&#@Q!QPX(0<4.!$D+0PZL`4$.L`%+#J`!
M1PZL`4$.L`%)#J`!<0ZL`4$.L`%I#JP!1PZH`44.K`%!#K`!00ZT`4(.N`%"
M#KP!00[``4D.J`%%#JP!00ZP`4D.H`%8#JP!00ZP`4D.H`%##JP!00ZP`4D.
MH`%.#JP!00ZP`5(.H`$`3````*0"``!4C___6P````!!#@B&`D$.#(,#3@X0
M30X43`X800X<0PX@1PX<1PX80@X<00X@00XD3`XH00XL0PXP20X,0<,."$'&
M#@0```",````]`(``&2/___0`````$$."(4"00X,AP-!#A"&!$$.%(,%1@Z4
M($<.Q"!E#L@@2P[,($$.T"!)#L`@70[$($0.R"!!#LP@0P[0($D.P"!6"@X4
M0<,.$$'&#@Q!QPX(0<4.!$4+2P[$($$.R"!"#LP@00[0($D.P"!+#L0@00[(
M($$.S"!##M`@20[`(`!\````A`,``*2/__](`@```$$."(4"3`X,AP-##A"&
M!$$.%(,%10Y,5PY020Y`6@Y,0PY030Y``E0.3$L.4$L.0`)I"@X40<,.$$'&
M#@Q!QPX(0<4.!$D+`F0.3$$.4%4.0%(.1$4.2$4.3$$.4%4.0%<.2$$.3$$.
M4$\.0````"`````$!```=)'__S0`````30X01PX43`X800X<20X@20X$`)0`
M```H!```D)'__\\`````00X(A0)!#@R'`T$.$(8$00X4@P5&#I0@2@Z40$<.
ML$!C#KA`1PZ\0$$.P$!'#L1`10[(0$(.S$!(#M!`20ZT0$4.N$!%#KQ`00[`
M0$D.L$!'#KQ`1@[`0$L.L$!'#K1`00ZX0$$.O$!!#L!`3@ZP0%8*#A1#PPX0
M0<8.#$''#@A!Q0X$00L`$````,`$``#(D?__$`````````#\````U`0``,21
M__\G`@```$$."(4"00X,AP-!#A"&!$$.%(,%1@Z4($H.L"$"0`ZX(4(.O"%#
M#L`A20ZP(6T*#A1!PPX00<8.#$''#@A!Q0X$10M##KPA00[`(4<.O"%!#K@A
M2PZ\(4$.P"%7#K`A8`ZX(4$.O"%!#L`A20ZP(4L.O"%!#L`A3PZP(5T.O"%!
M#L`A2PZP(4<.O"%!#L`A40ZP(4X.M"%!#K@A1`Z\(4$.P"%$#L0A10[((4(.
MS"%(#M`A20ZX(40.O"%!#L`A20ZP(5<.N"%$#KPA0@[`(4L.L"%^#K@A0@Z\
M(4$.P"%)#K`A5`Z\(4$.P"%)#K`A+````-0%``#TDO__-@````!!#@B&`DP.
M#(,#0PX81@X<00X@5PX,0<,."$'&#@0`@`````0&```$D___=`$```!!#@B%
M`D$.#(<#00X0A@1!#A2#!48.E"!'#K`@=0Z\($$.P"!,#K`@?PZX($(.O"!!
M#L`@20ZP($\.O"!!#L`@20ZP(%H*#A1!PPX00<8.#$''#@A!Q0X$2PMC"@X4
M0\,.$$'&#@Q!QPX(0<4.!$<+````/````(@&````E/___`,```!!#@B%`D$.
M#(<#00X0A@1,#A2#!4,.0`.X`PH.%$?##A!!Q@X,0<<."$'%#@1%"Q@```#(
M!@``P)?__V$`````30X@`D8*#@1%"P`H````Y`8``!28__\M`````$$."(,"
M3@X020X<00X@20X020X(0<,.!````(`````0!P``&)C___D`````00X(A0)!
M#@R'`T$.$(8$3`X4@P5##D`"0@H.%$/##A!!Q@X,0<<."$'%#@1&"T,.2$$.
M3$$.4$D.0%0*#A1!PPX00<8.#$''#@A!Q0X$2@M##DQ!#E!G#D1)#DA!#DQ'
M#E!!#E1"#EA"#EQ!#F!-#D```(P"``"4!P``E)C__PX'````00X(A0)!#@R'
M`T$.$(8$00X4@P51#J`"`N8.K`)!#K`"20Z@`E4.K`)!#K`"30Z@`D,.K`)!
M#K`"3PZ@`F`.K`)'#K`"00ZT`D(.N`)"#KP"00[``D\.H`("00ZL`D$.L`)+
M#J`"?PZL`D$.L`)'#JP"1`ZP`F@.M`)!#K@"2PZ\`D<.P`)##L0"0@[(`D(.
MS`)!#M`"20ZH`D4.K`)!#K`"20Z@`ED.J`)!#JP"1`ZP`DL.H`)'#JP"1`ZP
M`D8.M`)'#K@"2@Z\`D(.P`))#J`"6@H.%$/##A!!Q@X,0<<."$'%#@1("T,.
MK`)$#K`"2PZ@`@)9#JP"00ZP`E@.H`)+#J@"00ZL`D8.L`)-#J`"2PZL`D0.
ML`)+#J`"2`ZH`D<.K`)-#K`"20Z@`@)J#J0"00ZH`DH.K`)$#K`"00ZT`D(.
MN`)"#KP"00[``D\.H`)?#JP"00ZP`D$.M`)!#K@"1`Z\`D0.P`)$#L0"0@[(
M`D(.S`)!#M`"20Z@`D,.J`)%#JP"00ZP`DD.H`)(#JP"1`ZP`ET.H`("00ZL
M`D$.L`)+#J`"3`H.J`)!#JP"1`ZP`D8.M`)-"T,.K`)!#K`"20Z@`F$*#A1#
MPPX00<8.#$''#@A!Q0X$1@M##JP"2`ZP`DL.H`)C#J@"0@ZL`D0.L`))#J`"
M9@ZD`D0.J`)!#JP"10ZP`DT.I`)"#J@"2`ZL`D4.L`)-#J`"9PZD`D\.J`)!
M#JP"00ZP`DT.I`)"#J@"2`ZL`D4.L`)5#K0"1PZX`D0.O`)$#L`"00[$`D(.
MR`)"#LP"00[0`DD.H`)@#JP"00ZP`DT.H`),#J@"1PZL`DD.L`)-#J`"````
MS````"0*```4G?__`0(```!!#@B%`D$.#(<#00X0A@1!#A2#!4X.,`)+#C1"
M#CA!#CQ!#D!+#C!'#C1'#CA%#CQ##D!.#C!2#C1"#CA!#CQ##D!+#C!<#C1!
M#CA%#CQ##D!)#C!##C1(#CA!#CQ##D!/#C1!#CA!#CQ##D!=#C!/#CA%#CQ!
M#D!,#C!B"@X40<,.$$'&#@Q!QPX(0<4.!$@+0PH./$8.0%$.,$<./$<.0$@+
M<PXX00X\0PY`20XP1@H.-$<..$(./$<.0$8+`&0```#T"@``5)[__]$`````
M00X(AP)!#@R&`TP.$(,$0PXL40XP1PXL00XH0@XL0PXP:PX@=@H.$$'##@Q!
MQ@X(0<<.!$4+0PXD2`XH00XL00XP20X@5`H.$$/##@Q!Q@X(0<<.!$4+1```
M`%P+``#,<O__/0(```!$#`$`21`%`G4`01`'`G5\1!`&`G5X$`,"=71,#P-U
M<`8#;P$*P0P!`$'#0<9!QT'%0PP$!$$+(````*0+``"$GO__0@````!!#@B#
M`DX.'$(.(&\."$'##@0`3````,@+``"PGO__EP$```!!#@B%`D$.#(<#00X0
MA@1,#A2#!4,.,`+P"@X40<,.$$'&#@Q!QPX(0<4.!$H+`D,.-$,..$4./$$.
M0%(.,``\````&`P```"@__^@`0```$$."(4"00X,AP-!#A"&!$P.%(,%0PXP
M`PL!"@X40<,.$$'&#@Q!QPX(0<4.!$\+$````%@,``!@H?__!``````````0
M````;`P``%"A__\$`````````!````"`#```0*'__P0`````````(````)0,
M```TH?__+0````!1#A!&#A1$#AA$#AQ$#B!)#@0`$````+@,```]H?__!```
M```````8````S`P``#RA__\9`````$4."$$.!$T.$```````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````"P(P``
M8",```$```#S`@```0```/X"```/````B@0```P`````(```#0```$Q0```9
M````U+T``!L````$````&@```-B]```<````!````/7^_V\T`@``!0```&@)
M```&````B`,```H```"M!```"P```!`````5``````````,```#DO@```@``
M``@````4````$0```!<```"L$0``$0```'0/```2````.`(``!,````(````
M&`````````#[__]O`0``"/[__V_4#@``____;P$```#P__]O%@X``/K__V\&
M`````````````````````````````````````````````````````````-R]
M`````````````$8@````````````````````````````````````````````
M``````````````````````!@20``````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````!0(`````````3````*4$%2+G!M"@````#(8@``1T-#.B`H1TY5
M*2`Q,BXR+C```"YS:'-T<G1A8@`N:6YT97)P`"YN;W1E+F=N=2YB=6EL9"UI
M9``N;F]T92YG;G4N<')O<&5R='D`+FYO=&4N04))+71A9P`N9VYU+FAA<V@`
M+F1Y;G-Y;0`N9'EN<W1R`"YG;G4N=F5R<VEO;@`N9VYU+G9E<G-I;VY?<@`N
M<F5L+F1Y;@`N<F5L+G!L=``N:6YI=``N=&5X=``N9FEN:0`N<F]D871A`"YE
M:%]F<F%M95]H9'(`+F5H7V9R86UE`"YI;FET7V%R<F%Y`"YF:6YI7V%R<F%Y
M`"YD>6YA;6EC`"YG;W0`+F1A=&$`+F)S<P`N8V]M;65N=```````````````
M````````````````````````````````````````````"P````$````"````
MM`$``+0!```3```````````````!`````````!,````'`````@```,@!``#(
M`0``)```````````````!``````````F````!P````(```#L`0``[`$``"@`
M``````````````0`````````.0````<````"````%`(``!0"```@````````
M```````$`````````$<```#V__]O`@```#0"```T`@``5`$```8`````````
M!`````0```!1````"P````(```"(`P``B`,``.`%```'`````0````0````0
M````60````,````"````:`D``&@)``"M!``````````````!`````````&$`
M``#___]O`@```!8.```6#@``O`````8``````````@````(```!N````_O__
M;P(```#4#@``U`X``*`````'`````0````0`````````?0````D````"````
M=`\``'0/```X`@``!@`````````$````"````(8````)````0@```*P1``"L
M$0``"`````8````6````!`````@```"/`````0````8`````(````"```"0`
M``````````````0`````````B@````$````&````,"```#`@```@````````
M```````0````!````)4````!````!@```%`@``!0(```^2\`````````````
M$`````````";`````0````8```!,4```3%```!@```````````````0`````
M````H0````$````"`````&````!@```000`````````````0`````````*D`
M```!`````@```!"A```0H0``%`$`````````````!`````````"W`````0``
M``(````DH@``)*(``.@,``````````````0`````````P0````X````#````
MU+T``-2]```$```````````````$````!````,T````/`````P```-B]``#8
MO0``!```````````````!`````0```#9````!@````,```#<O0``W+T```@!
M```'``````````0````(````X@````$````#````Y+X``.2^```<`0``````
M```````$````!````.<````!`````P````#`````P```&```````````````
M!`````````#M````"`````,````8P```&,```!0```````````````0`````
M````\@````$````P`````````!C````2```````````````!`````0````$`
F```#```````````````JP```^P```````````````0``````````
