#!/usr/bin/perl
#
# Netfilter_POM.pm, part of the patch-o-matic 'next generation' package
# (C) 2003-2004 by Harald Welte <laforge@netfilter.org>
# (C) 2004	by Jozsef Kadlecsik <kadlec@blackhole.kfki.hu>
#
# This code is subject to the GNU GPLv2
#
# Netfilter_POM.pm,v 1.18 2004/02/27 21:40:20 kadlec Exp
#
# The idea is to have the backend seperated from the frontend.  Thus,
# other frontends (like ncurses,...) could potentially be implemented on
# top of this.
#
package Netfilter_POM;

# we could export the public functions into caller namespace 
#require Exporter;
#BEGIN {
#	@ISA = qw(Exporter);
#}
#@EXPORT = qw();

use strict;

use File::Temp;
use File::Copy;
use File::Path;
use File::Basename;
#use Data::Dumper;

my $BIN_PATCH = "patch";
my $ERRMSG = "undefined error";
my %version;
my %last_words;
our %srcpath;

sub breaklink($)
{
	my ($file) = @_;
	my $nlinks = (stat($file))[3];
	if ($nlinks == 1) {
		return 1;
	}
	my $tmpnam = File::Temp::tmpnam();
	if (!File::Copy::copy($file, $tmpnam)) {
		$ERRMSG = "breaklink: unable to copy file $file to $tmpnam: $!";
		return 0;
	}
	if (!unlink($file)) {
		$ERRMSG = "breaklink: unable to unlink $file";
		return 0;
	}
	if (!File::Copy::move($tmpnam, $file)) {
		$ERRMSG = "breaklink: unable to move $tmpnam to $file: $!";
		return 0;
	}
	return 1;
}
	

# print the last error message
#
sub perror()
{
	print STDERR ($ERRMSG, "\n");
}

sub strerror()
{
	return $ERRMSG;
}

# count the number of hunks in a unified diff file
#
sub count_hunks($) {
	my ($file) = @_;
	my $hunk_count;

	open(INFILE, $file) || return -1;
	while (my $line = <INFILE>) {
		chomp($line);
		if ($line =~ /^@@/) {
			$hunk_count++;
		}
	}
	close(INFILE);

	return $hunk_count;
}

# get the kernel version of a specified kernel tree
#
sub kernelversion($) {
	my ($kerneldir) = @_;

	my ($version, $patchlevel, $sublevel);

	if (!open(MAKEFILE, "$kerneldir/Makefile")) {
		$ERRMSG = "No kernel makefile in $kerneldir!";
		return;
	}
	while (my $line = <MAKEFILE>) {
		chomp($line);
		if ($line =~ /^VERSION =\s*(\S+)/) {
			$version = $1;
		} elsif ($line =~ /^PATCHLEVEL =\s*(\S+)/) {
			$patchlevel = $1;
		} elsif ($line =~ /^SUBLEVEL =\s*(\S+)/) {
			$sublevel = $1;
		}
	}
	return ($version, $patchlevel, $sublevel);
}

# get the iptables version of a specified source tree
#
sub iptablesversion($) {
	my ($iptablesdir) = @_;

	my ($version, $patchlevel, $sublevel);

	if (!open(MAKEFILE, "$iptablesdir/Makefile")) {
		$ERRMSG = "No makefile in $iptablesdir!";
		return;
	}
	while (my $line = <MAKEFILE>) {
		chomp($line);
		if ($line =~ /^IPTABLES_VERSION =\s*(\S+)/) {
			$version = $1;
		} elsif ($line =~ /^NETFILTER_VERSION =\s*(\S+)/) {
			$version = $1;
		}
	}
	# don't use versioning like 1.2.3b!
	return (split(/\./, $version, 3));
}

# this should be taken from RPM or something like that
#
sub version_compare($$$) {
	my ($ver1, $op, $ver2) = @_;
	my (@ver1, @ver2);
	my ($ver, $min, $res);

	@ver1 = split(/\./, $ver1);
	@ver2 = split(/\./, $ver2);
	$min = $#ver1 < $#ver2 ? $#ver1 : $#ver2;
	foreach $ver (0..$min) {
		next if $ver1[$ver] == $ver2[$ver];
		eval "\$res = $ver1[$ver] $op $ver2[$ver]";
		return ($res ? ($ver+2) : 0);
	}
	eval "\$res = $ver1[$min] $op $ver2[$min]";
	return ( $res ? ($min+2) : 0);
}

# do we have Kconfig / Config.in applicable for kernel ?
#
sub check_kconfig($) {
	my ($patchlet) = @_;

	my $has_kconfig;
	my $has_config_in;

	for my $file (keys %{$$patchlet{ladds}{linux}{best}}) {
		my $basename = File::Basename::basename($file);
		$basename =~ s/\.ladd(_\d+)?$//;
		if ($basename eq 'Config.in') {
			$has_config_in = 1;
		} elsif ($basename eq 'Kconfig') {
			$has_kconfig = 1;
		}
	}

	if ($version{'linux'} =~ /^2\.4\.\d+/) {
		if ($has_kconfig && !$has_config_in) {
			$ERRMSG = "This patch is missing Config.in for 2.4.x";
			return 0;
		}
	} elsif ($version{'linux'} =~ /^2\.6\.\d+/) {
		if ($has_config_in && !$has_kconfig) {
			$ERRMSG = "This patch is missing Kconfig for 2.6.x";
			return 0;
		}
	} else {
		# FIXME: do a precise check when Kconfig went into 2.5
		$ERRMSG = sprintf("Kernel Version %s not supported by patch-o-matic", $version{'linux'});
		return 0;
	}

	return 1;
}


# are the requirements for a specific patchlet fulfilled?
#
sub requirements_fulfilled($) {
	my ($patchlet) = @_;

	foreach my $req (@{$$patchlet{info}{requires}}) {
		my ($prog, $op, $ver) = $req =~/(\S+)\s*(==|>=|>|<=|<)\s*(\S+)/;
		if (!defined($version{$prog})) {
			$ERRMSG = "don't know how to handle '$prog' requirement";
			return 0;
		}

		if (!version_compare($version{$prog}, $op, $ver)) {
			$ERRMSG = "requirement '$req' not fulfilled";
			return 0;
		}
	}

#print Dumper($patchlet);
	my($proj, $bingo, $match);
	foreach my $type (qw(patch files ladds)) {
		foreach $proj (keys %{$$patchlet{$type}}) {
			$bingo = -1;
			foreach my $ver (keys %{$$patchlet{$type}{$proj}}) {
				$match = !$ver ? 1
					: version_compare($ver, "==", $version{$proj});
				if ($match && $match > $bingo) {
					$bingo = $match;
					if ($type eq 'patch') {
						$$patchlet{$type}{$proj}{best} =
							$$patchlet{$type}{$proj}{$ver};
					} else {
						for my $item (keys %{$$patchlet{$type}{$proj}{$ver}}) {
							$$patchlet{$type}{$proj}{best}{$item} =
								$$patchlet{$type}{$proj}{$ver}{$item};
						}
					}
				}
			}
			if ($bingo < 0) {
				delete $$patchlet{$type}{$proj};
			}
		}
	}
	if (!(%{$$patchlet{patch}} || %{$$patchlet{files}} || %{$$patchlet{ladds}})) {
		$ERRMSG = "Can't find a patch version of $$patchlet{name} matching your " 
			   . ($proj eq 'linux' ? 'kernel' : $proj) . " version";
		return 0;
	}
	if (!check_kconfig($patchlet)) {
		# don't override $ERRMSG
		return 0;
	}
	
	return 1;
}

# recursively test if all dependencies are fulfilled
#
# return values:
# 	 1	dependencies fulfilled
# 	 0	dependencies not fulfilled
# 	-1	dependencies cannot be fulfilled [conflicting patchlets]
#
sub dependencies_fulfilled($$) {
	my ($plets, $plname) = @_;
	my $patchlet = $$plets{$plname};

	for my $dep (@{$$patchlet{info}{depends}}) {
		my $inverse = 0;
		if ($dep =~ /^!/) {
			$inverse = 1;
			$dep =~ s/^!//;
		}
		if (!defined($$plets{$dep})) {
			$ERRMSG = "$plname has dependency on $dep, but $dep is not known";
			return 0;
		}
		if (!apply($$plets{$dep}, !$inverse, 1)) {
			if (!$inverse) {
				$ERRMSG = "$dep not applied";
				return 0;
			} else {
				$ERRMSG = "present '$dep' conflicts with to-be-installed '$plname'";
				return -1;
			}
		}
		my $ret = dependencies_fulfilled($plets, $$plets{$dep});
		if ($ret <= 0) {
			# don't overwrite ERRMSG
			return $ret;
		}
	}
	return 1;
}

# apply_dependencies - recursively apply all dependencies
# plets: hash of all patchlets
# plname: name of patchlet subject to recursive dependency resolving
# force: forcibly try to apply dependent patches (to see .rej's)
sub apply_dependencies($$$$)
{
	my ($plets, $plname, $force, $test) = @_;
	my $patchlet = $$plets{$plname};
	for my $dep (@{$$patchlet{info}{depends}}) {
		# don't revert existing patches
		if ($dep =~ /^!/) {
			next;
		}
		if (!defined($$plets{$dep})) {
			$ERRMSG = "$plname has dependency on $dep, but $dep is not known";
			return 0;
		}
		if (!apply_dependencies($plets, $dep, $force, $test)) {
			# don't overwrite $ERRMSG
			return 0;
		}
		if (!$force) {
			# first test, then apply
			if (!apply($$plets{$dep}, 0, 1)) {
				$ERRMSG = "apply_dependencies: unable to apply dependent $dep: ".$ERRMSG;
				return 0;
			} 
		}
		if (!$test) {
			if (!apply($$plets{$dep}, 0, 0)) {
				$ERRMSG = "apply_dependencies: unable to apply dependent $dep: ".$ERRMSG;
				return 0;
			} else {
				print("apply_dependencies: successfully applied $dep\n");
			}
		}
	}

	return 1;
}

# recurse through subdirectories, pushing all filenames into a hash.
# differentiate between whole new files and line-adds (ladds)
#
sub recurse($$$$) {
	my ($pdir, $dir, $aref, $a2ref) = @_;
	if (!opendir(DIR, $dir)) {
		$ERRMSG = "can't open directory $dir: $!";
		return 0;
	}
	my @dents = grep {!/^\./} readdir(DIR);
	closedir(DIR);
	foreach my $dent (@dents) {
		my $fullpath = "$dir/$dent";
		if (-f $fullpath) {
			if ($fullpath =~ /.ladd(_\d+)?$/) {
				$$a2ref{$fullpath} = "$pdir/$fullpath";
			} else {
				$$aref{$fullpath} = "$pdir/$fullpath";
			}
		} elsif (-d $fullpath) {
			if ($dent ne 'CVS') {
				recurse($pdir, $fullpath, $aref, $a2ref);
			}
		}
	}
	return $aref;
}

# parse info file associated with patchlet
#
sub parse_patch_info($) {
	my ($info) = @_;
	my %pinfo;

	if (!open(INFILE, $info)) {
		$ERRMSG = "unable to open $info: $!";
		return;
	}

	while (my $line = <INFILE>) {
		chomp($line);
		if ($line =~ /^Title:\s+(.*)/) {
			$pinfo{title} = $1;
		} elsif ($line =~ /^Author:\s+(.*)/) {
			$pinfo{author} = $1;
		} elsif ($line =~ /^Status:\s+(.*)/) {
			$pinfo{status} = $1;
		} elsif ($line =~ /^Repository:\s+(.*)/) {
			$pinfo{repository} = $1;
		} elsif ($line =~ /^Requires:\s+(.*)/) {
			push(@{$pinfo{requires}}, $1);
		} elsif ($line =~ /^Depends:\s+(.*)/) {
			push(@{$pinfo{depends}}, $1);
		} elsif ($line =~ /^Recompile:\s+(kernel|netfilter|iptables)/) {
			$pinfo{recompile}{$1}++;
		} else {
			printf(STDERR "unknown config key '%s'\n",
				$line);
		}
	}
	close(INFILE);

	if (!defined($pinfo{repository})) {
		$ERRMSG = "missing repository definition!";
		print STDERR $ERRMSG;
		return;
	}
	
	return \%pinfo;
}

# parse a single patchlet specified as parameter
#
sub parse_patchlet($) {
	my ($patchdir) = @_;

	my %patchlet;

	$patchlet{basedir} = $patchdir;
	($patchlet{name} = $patchdir) =~ s,\./,,;
	# parse our info file
	$patchlet{info} = parse_patch_info($patchdir . '/info');
	if (!$patchlet{info}) {
		# don't override $ERRMSG
		return;
	}

	if (-f ($patchdir.'/help')) {
		$patchlet{help}{linux} = 'help';
	}
	
	# get list of source files that we'd need to copy
	if (!opendir(PDIR, $patchdir)) {
		$ERRMSG = "unable to open patchdir $patchdir: $!";
		return;
	}
	my @dents = readdir(PDIR);
	closedir(PDIR);

	# save the filename of the patch itself
	foreach my $pf (@dents) {
		if ($pf =~ /(([^-]+)(-([\d\.]*))?\.patch)$/) {
			$patchlet{patch}{$2}{$4} = $1;
		}
	}

	my @projects = grep {!/^\./ && -d "$patchdir/$_" } @dents;
	foreach my $pdir (@projects) {
		my $proj;
		my $ver;
		my $oldpwd = `pwd`;
		chomp($oldpwd);

		if ($pdir eq 'CVS') {
			next;
		}
		
		$pdir =~ /([^-]*)(-([\d\.]*))?/;
		($proj, $ver) = ($1, $3);
		chdir("$patchdir/$pdir");
		recurse($pdir, ".",
			\%{$patchlet{files}{$proj}{$ver}}, 
			\%{$patchlet{ladds}{$proj}{$ver}});
		chdir($oldpwd);
	}

	return \%patchlet;
}

sub parse_config($) {
	my ($config) = @_;
	my @confkeys;

	if (!open(INFILE, $config)) {
		$ERRMSG = "unable to open configfile $config: $!";
		return;
	}

	my $confkey;
	while (my $line = <INFILE>) {
		chomp($line);
		if ($line =~ /^Prev:\s+(.*)/) {
			$$confkey{prev} = $1;
		} elsif ($line =~ /^Type:\s+(.*)/) {
			$$confkey{type} = $1;
		} elsif ($line =~ /^Title:\s+(.*)/) {
			$$confkey{title} = $1;
		} elsif ($line =~ /^Depends:\s+(.*)/) {
			push(@{$$confkey{depends}}, $1);
		} elsif ($line =~ /^Config:\s+(.*)/) {
			my $tmp = $1;
			my %newh;
			$confkey = \%newh;
			$$confkey{config} = $tmp;
			push(@confkeys, $confkey);
		}
	}
	close(INFILE);

	return \@confkeys;
}

# Kconfig / Config.in code
#

sub push_helpfile($$$) {
	my ($outl, $file, $prefix) = @_;

	if (!open(HELP, $file)) {
		$ERRMSG = "push_helpfile: unable to open $file: $!";
		return 0;
	}
	while (my $line = <HELP>) {
		push(@$outl, $prefix.$line);
	}
	close(HELP);
	return 1;
}

# apply an old-style lineadd file
#
sub apply_lineadd($$$$$)
{
	my ($patchlet, $laddfile, $fname, $revert, $test) = @_;
	my @newlines;
	my $kconfigmode;
	my $configmode;
	my $lookingfor;

	if (!open(LADD, $laddfile)) {
		$ERRMSG = "unable to open ladd $laddfile";
		return 0;
	}

	my ($srcfile, $extn) = $fname =~ /(.*?)(\.ladd(_\d+)?)?$/;

	if ($srcfile =~ /Kconfig$/) {
		$kconfigmode = 1;
		$lookingfor = "endmenu\n";
	} elsif ($srcfile =~ /Configure\.help/) {
		$configmode = 1;
		$lookingfor = <LADD>;
	} else {
		$lookingfor = <LADD>;
	}

	if (!open(SRC, $srcfile)) {
		close(LADD);
		$ERRMSG = "unable to open ladd src $srcfile";
		return 0;
	}

	my $found = 0;
	SRCLINE: while (my $line = <SRC>) {
		push(@newlines, $line);
		if ($line eq $lookingfor) {
			$found = 1;
			if ($revert == 0) {
				my ($prev, $next, $last);
				if ($kconfigmode) {
					$prev = pop(@newlines);
				} elsif ($configmode) {
					while (($line = <SRC>) !~ /^\S/) {
						push(@newlines, $line);
					}
					$next = $line;
				}
				while (my $newline = <LADD>) {
					push(@newlines, $newline);
					$last = $newline;
				}
				# ugly kconfig/configure.help hacks
				if ($kconfigmode) {
					push(@newlines, "\t  help\n");
					push(@newlines, "\n");
					push(@newlines, $prev);
				} elsif ($configmode) {
					push(@newlines, "\n")
						unless $last =~ /^\s*$/;
					push(@newlines, $next);
				}
				# append rest of sourcefile
				while ($line = <SRC>) {
					push(@newlines, $line);
				}
			} else {
				if ($kconfigmode) {
					my @helplines;
					my $prev = pop(@newlines);
					
					while (my $l = <LADD>) {
						push(@helplines, $l);
					}
					push(@helplines, "\t  help\n");
					push(@helplines, "\n");

					while (my $l = pop(@helplines)) {
						my $m = pop(@newlines);
						if ($l ne $m) {
							$found = -1;
							last SRCLINE;
						}
					}
					push(@newlines, $prev);
				} else {
					while (my $newline = <LADD>) {
						my $srcline = <SRC>;
						if ($newline ne $srcline) {
							$found = -1;
							last SRCLINE;
						}
					}
				}
			}
		}
	}
	close(LADD);
	close(SRC);
		
	if ($found == 0) {
		$ERRMSG = "unable to find ladd slot in src $srcfile";
		return 0;
	} elsif (!$test && $found == -1) {
		$ERRMSG = "unable to find all to-be-removed lines in $srcfile";
		return 0;
	}

	if ($test == 0) {
		my $filenam = "$srcfile";
		if (!breaklink($filenam)) {
			return 0;
		}
		if (!open(SRC, '>'.$filenam)) {
			$ERRMSG = "unable to write to lad src $srcfile";
			return 0;
		}
		foreach my $line (@newlines) {
			print(SRC $line);
		}
		close(SRC);
	}

	return 1;
}

sub apply_newfiles($$$$)
{
	my ($patchlet, $proj, $revert, $test) = @_;

	my $projdir = $srcpath{$proj};

	my $test_found;
	my $test_notfound;

	for my $file (keys %{$$patchlet{files}{$proj}{best}}) {
		my $srcfile = $$patchlet{basedir}.'/'.$$patchlet{files}{$proj}{best}{$file};
		my $destdir = ($projdir . '/'.  File::Basename::dirname($file));
		my $destfile = ("$projdir/$file");
		if (!$test) {
			if (!$revert) {
				if (!-d $destdir) {
					if (!File::Path::mkpath($destdir)) {
						$ERRMSG = "unable to mkpath($destdir) while applying newfile: $!";
						return 0;
					}
				}
				File::Copy::copy($srcfile, $destfile);
			} else {
				if (!unlink($destfile)) {
					$ERRMSG = "unable to remove: $! while reverting newfile: $!";
					return 0;
				}
			}
		} else {
			if (-f $destfile) {
				$test_found++;
			} else {
				$test_notfound++;
			}
		}
	}

	if ($test) {
		if (!$revert && $test_found) {
			$ERRMSG = "newfile: $test_found files in our way, unable to apply";
			return 0;
		} elsif ($revert && $test_notfound) {
			$ERRMSG = "newfile: $test_notfound files missing, unable to revert";
			return 0;
		}
	}
		
	return 1;
}

sub apply_lineadds($$$$)
{
	my ($patchlet, $proj, $revert, $test) = @_;

	my $projdir = $srcpath{$proj};

	# apply the line-adds
	for my $file (keys %{$$patchlet{ladds}{$proj}{best}}) {
		my $basename = File::Basename::basename($file);
		my $destdir = "$projdir/$basename";
		if ($proj eq 'linux') {
			if ($version{'linux'} =~ /^2\.4\.\d+/ &&
			    $basename =~ /^Kconfig\.ladd/) {
			    	next;
			}
			if ($version{'linux'} =~ /^2\.6\.\d+/ &&
			    ($basename =~ /^Config\.in\.ladd/ ||
			     $basename =~ /^Configure\.help/)) {
			    	next;
			}
		}
		#if (!-d $destdir) {
		#	File::Path::mkpath($destdir) || 
		#		die("unable to mkpath($destdir): $!\n");
		#}
		if (!apply_lineadd($patchlet, $$patchlet{basedir}.'/'.$$patchlet{ladds}{$proj}{best}{$file},
				   $projdir.'/'.$file, $revert, $test)) {
			# don't override $ERRMSG
			return 0;
		} 
	}

	return 1;
}


# apply a given patchlet to a given kernel tree
#
# return value:
# 	normal (non-test) mode: 1 on success, 0 on failure
# 	test mode: 1 if test was successful (patch could be applied/reverted)
#
sub apply($$$)
{
	my ($patchlet, $revert, $test) = @_;
	my (@projects);

	@projects = keys %{{ keys %{$$patchlet{files}}, keys %{$$patchlet{patch}} }};

#print Dumper($patchlet);
	foreach my $proj (@projects) {
		my $projpath = $srcpath{$proj};
		# copy all new files
		if (!apply_newfiles($patchlet, $proj, $revert, $test)) {
			# don't override $ERRMSG
			return 0;
		}
		if (!apply_lineadds($patchlet, $proj, $revert, $test)) {
			# don't override $ERRMSG
			return 0;
		}

		if (defined($$patchlet{patch}{$proj}{best})) {
			# apply the patch itself
			my $options;
			if ($revert) {
				$options .= "-R ";
			}
			if ($test) {
				$options .= "--dry-run ";
			}
			my $patchfile = ($$patchlet{basedir}.'/'.$$patchlet{patch}{$proj}{best});
			my $cmd = sprintf("%s -f -p1 -d %s %s < %s", 
					  $BIN_PATCH, $projpath, 
					  $options,
					  $patchfile);

			my $missing_files;
			my $rejects;
			my $notempty;
			my $hunks = count_hunks($patchfile);
			open(PATCH, "$cmd|") || die("can't start patch '$cmd': $!\n");
			while (my $line = <PATCH>) {
				# FIXME: parse patch output
				chomp($line);
				if ($line =~ /No file to patch/) {
					$missing_files++;
				} elsif ($line =~ /FAILED at/) {
					$rejects++;
				} elsif ($line =~ /not empty after patch, as expected/) {
					$notempty++;
				}
			}
			close(PATCH);

			if ($test) {
				if ($missing_files != 0) {
					$ERRMSG = "cannot apply ($missing_files missing files)";
					return 0;
				#} elsif ($rejects*2 > $hunks) {
				} elsif ($rejects != 0) {
					$ERRMSG= "cannot apply ($rejects rejects out of $hunks hunks)";
					return 0;
				} else {
					# could be applied!
					#printf(" ALREADY APPLIED (%d rejects out of %d hunks)\n", $rejects, $hunks);
				}
			} else {
				if ($missing_files != 0) {
					$ERRMSG = "ERROR ($missing_files missing files)";
					return 0;
				} elsif ($rejects != 0) {
					$ERRMSG = "ERROR ($rejects rejects out of $hunks hunks)";
					return 0;
				}
			}
		}
	}
	if (!$test) {
		map { $last_words{$_}++ } keys %{$$patchlet{info}{recompile}};
	}
	
	return 1;
}

# iterate over all patchlet directories below the given base directory 
# and parse all patchlet definitions
#
sub parse_patchlets
{
	my ($patchdir) = @_;
	my %plets;
	
	my @patches;

	if (!opendir(INDIR, $patchdir)) {
		$ERRMSG = "Unable to open $patchdir: $!";
		return;
	}
	my @alldirs = grep {!/^\./ && -d "$patchdir/$_" } readdir(INDIR);
	closedir(INDIR);

	foreach my $patch (@alldirs) {
		if (-f "$patchdir/$patch/info") {
			$plets{$patch} = parse_patchlet("$patchdir/$patch");
			if (!defined($plets{$patch})) {
				return;
			}
		}
	}
	return \%plets;
}

sub init($$)
{
	my ($kp, $up) = @_;

	$srcpath{linux} = $kp;
	$srcpath{iptables} = $up;

	# get version information of all software packages we know of
	my @kver = kernelversion($kp);
	$version{linux} = sprintf("%d.%d.%d", $kver[0], $kver[1], $kver[2]);
	my @ipver = iptablesversion($up);
	$version{iptables} = sprintf("%d.%d.%d", $ipver[0], $ipver[1], $ipver[2]);
	return 1;
}

sub last_words()
{
	# print anything useful
	print <<TXT if $last_words{kernel};
Recompile the kernel image.
TXT
	if ($last_words{netfilter}) {	
		print <<TXT if !$last_words{kernel};
Recompile the kernel image (if there are non-modular netfilter modules).
Recompile the netfilter kernel modules.
TXT
		print <<TXT if $last_words{kernel};
Recompile the netfilter kernel modules.
TXT
	}
	print <<TXT if $last_words{iptables};
Recompile the iptables binaries.
TXT
}

return 1;

__END__

there are several diffent modes of operation:

=item1 isapplied

tests whether a given kernel tree does already contain a given patch. The only
case where this is true:
	1) all the newfiles do exist
	2) all the lineadds match and their newlines can be found
	3) 'patch -R --dry-run' runs cleanly with no rejected hunks
this is actually the same as 'revert+test' below.

=item1 apply + test

tests whether the given patchlet would apply cleanly to the given tree.  The
only case where this is true:
	1) all the newfiles don't exist
	2) all the lineadd searchlines can be found
	3) 'patch --dry-run' runs cleanly with no rejected hunks

=item1 apply

apply the given patch to the given kernel tree

=item1 revert + test

tests whether the given patchlet would revert cleanly in the given tree. The
only case where this is true:
	1) all the newfiles exist
	2) all the lineadds match and their newlines can be found
	3) 'patch -R --dry-run' runs cleanly with no rejected hunks

=item1 revert

reverts the given patch from the given kernel tree
