/* Connection state tracking for netfilter.  This is separated from,
   but required by, the NAT layer; it can also be used by an iptables
   extension. */

/* (C) 1999-2001 Paul `Rusty' Russell
 * (C) 2002-2005 Netfilter Core Team <coreteam@netfilter.org>
 * (C) 2003,2004 USAGI/WIDE Project <http://www.linux-ipv6.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * 23 Apr 2001: Harald Welte <laforge@gnumonks.org>
 *	- new API and handling of conntrack/nat helpers
 *	- now capable of multiple expectations for one master
 * 16 Jul 2002: Harald Welte <laforge@gnumonks.org>
 *	- add usage/reference counts to ip_conntrack_expect
 *	- export ip_conntrack[_expect]_{find_get,put} functions
 * 16 Dec 2003: Yasuyuki Kozakai @USAGI <yasuyuki.kozakai@toshiba.co.jp>
 *	- generalize L3 protocol denendent part.
 * 23 Mar 2004: Yasuyuki Kozakai @USAGI <yasuyuki.kozakai@toshiba.co.jp>
 *	- add support various size of conntrack structures.
 *
 * Derived from net/ipv4/netfilter/ip_conntrack_core.c
 */

#include <linux/config.h>
#include <linux/types.h>
#include <linux/netfilter.h>
#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/proc_fs.h>
#include <linux/vmalloc.h>
#include <linux/stddef.h>
#include <linux/slab.h>
#include <linux/random.h>
#include <linux/jhash.h>
#include <linux/err.h>
#include <linux/percpu.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/socket.h>

/* This rwlock protects the main hash table, protocol/helper/expected
   registrations, conntrack timers*/
#define ASSERT_READ_LOCK(x) MUST_BE_READ_LOCKED(&nf_conntrack_lock)
#define ASSERT_WRITE_LOCK(x) MUST_BE_WRITE_LOCKED(&nf_conntrack_lock)

#include <linux/netfilter/nf_conntrack.h>
#include <linux/netfilter/nf_conntrack_l3proto.h>
#include <linux/netfilter/nf_conntrack_protocol.h>
#include <linux/netfilter/nf_conntrack_helper.h>
#include <linux/netfilter/nf_conntrack_core.h>
#include <linux/netfilter_ipv4/listhelp.h>

#define NF_CONNTRACK_VERSION	"0.3.1"

#if 0
#define DEBUGP printk
#else
#define DEBUGP(format, args...)
#endif

DECLARE_RWLOCK(nf_conntrack_lock);

/* nf_conntrack_standalone needs this */
atomic_t nf_conntrack_count = ATOMIC_INIT(0);

void (*nf_conntrack_destroyed)(struct nf_conn *conntrack) = NULL;
LIST_HEAD(nf_conntrack_expect_list);
struct nf_conntrack_protocol **nf_ct_protos[PF_MAX];
struct nf_conntrack_l3proto *nf_ct_l3protos[PF_MAX];
static LIST_HEAD(helpers);
unsigned int nf_conntrack_htable_size = 0;
int nf_conntrack_max;
struct list_head *nf_conntrack_hash;
static kmem_cache_t *nf_conntrack_expect_cachep;
struct nf_conn nf_conntrack_untracked;
unsigned int nf_ct_log_invalid;
static LIST_HEAD(unconfirmed);
static int nf_conntrack_vmalloc;

DEFINE_PER_CPU(struct nf_conntrack_stat, nf_conntrack_stat);
EXPORT_PER_CPU_SYMBOL(nf_conntrack_stat);

/*
 * This scheme offers various size of "struct nf_conn" dependent on
 * features(helper, nat, ...)
 */

#define NF_CT_FEATURES_NAMELEN	256
static struct {
	/* name of slab cache. printed in /proc/slabinfo */
	char *name;

	/* size of slab cache */
	size_t size;

	/* slab cache pointer */
	kmem_cache_t *cachep;

	/* allocated slab cache + modules which uses this slab cache */
	int use;

	/* Initialization */
	int (*init_conntrack)(struct nf_conn *, u_int32_t);

} nf_ct_cache[NF_CT_F_NUM];

/* protect members of nf_ct_cache except of "use" */
DECLARE_RWLOCK(nf_ct_cache_lock);

/* This avoids calling kmem_cache_create() with same name simultaneously */
DECLARE_MUTEX(nf_ct_cache_mutex);

extern struct nf_conntrack_protocol nf_conntrack_generic_protocol;
struct nf_conntrack_protocol *
nf_ct_find_proto(u_int16_t l3proto, u_int8_t protocol)
{
	if (unlikely(nf_ct_protos[l3proto] == NULL))
		return &nf_conntrack_generic_protocol;

	return nf_ct_protos[l3proto][protocol];
}

void 
nf_ct_put(struct nf_conn *ct)
{
	NF_CT_ASSERT(ct);
	nf_conntrack_put(&ct->ct_general);
}

static int nf_conntrack_hash_rnd_initted;
static unsigned int nf_conntrack_hash_rnd;

static u_int32_t
hash_conntrack(const struct nf_conntrack_tuple *tuple)
{
	unsigned int a, b;
	a = jhash((void *)tuple->src.u3.all, sizeof(tuple->src.u3.all),
		  ((tuple->src.l3num) << 16) | tuple->dst.protonum);
	b = jhash((void *)tuple->dst.u3.all, sizeof(tuple->dst.u3.all),
			(tuple->src.u.all << 16) | tuple->dst.u.all);

	return jhash_2words(a, b, nf_conntrack_hash_rnd)
	       % nf_conntrack_htable_size;
}

struct nf_conn *
alloc_conntrack(u_int32_t features)
{
	struct nf_conn *conntrack = NULL;

	DEBUGP("alloc_conntrack: features=0x%x\n", features);

	READ_LOCK(&nf_ct_cache_lock);

	if (!nf_ct_cache[features].use) {
		DEBUGP("alloc_conntrack: not supported features = 0x%x\n",
			features);
		goto out;
	}

	conntrack = kmem_cache_alloc(nf_ct_cache[features].cachep, GFP_ATOMIC);
	if (conntrack == NULL) {
		DEBUGP("alloc_conntrack: Can't alloc conntrack from cache\n");
		goto out;
	}
	memset(conntrack, 0, nf_ct_cache[features].size);
	conntrack->features = features;
	if (nf_ct_cache[features].init_conntrack &&
	    nf_ct_cache[features].init_conntrack(conntrack, features) < 0) {
		DEBUGP("alloc_conntrack: failed to init\n");
		kmem_cache_free(nf_ct_cache[features].cachep, conntrack);
		conntrack = NULL;
		goto out;
	}

out:
	READ_UNLOCK(&nf_ct_cache_lock);
	return conntrack;
}

void
free_conntrack(struct nf_conn *conntrack)
{
	u_int32_t features = conntrack->features;
	NF_CT_ASSERT(features >= NF_CT_F_BASIC && features < NF_CT_F_NUM);
	DEBUGP("free_conntrack: features = 0x%x, conntrack=%p\n", features,
	       conntrack);
	kmem_cache_free(nf_ct_cache[features].cachep, conntrack);
}

/* Initialize "struct nf_conn" which has spaces for helper */
static int
init_conntrack_for_helper(struct nf_conn *conntrack, u_int32_t features)
{

	conntrack->help = (union nf_conntrack_help *)
		(((unsigned long)conntrack->data
		  + (__alignof__(union nf_conntrack_help) - 1))
		 & (~((unsigned long)(__alignof__(union nf_conntrack_help) -1))));
	return 0;
}

int nf_conntrack_register_cache(u_int32_t features, const char *name,
				size_t size,
				int (*init)(struct nf_conn *, u_int32_t))
{
	int ret = 0;
	char *cache_name;
	kmem_cache_t *cachep;

	DEBUGP("nf_conntrack_register_cache: features=0x%x, name=%s, size=%d\n",
	       features, name, size);

	if (features < NF_CT_F_BASIC || features >= NF_CT_F_NUM) {
		DEBUGP("nf_conntrack_register_cache: invalid features.: 0x%x\n",
			features);
		return -EINVAL;
	}

	down(&nf_ct_cache_mutex);

	WRITE_LOCK(&nf_ct_cache_lock);
	/* e.g: multiple helpers are loaded */
	if (nf_ct_cache[features].use > 0) {
		DEBUGP("nf_conntrack_register_cache: already resisterd.\n");
		if ((!strncmp(nf_ct_cache[features].name, name,
			      NF_CT_FEATURES_NAMELEN))
		    && nf_ct_cache[features].size == size
		    && nf_ct_cache[features].init_conntrack == init) {
			DEBUGP("nf_conntrack_register_cache: reusing.\n");
			nf_ct_cache[features].use++;
			ret = 0;
		} else
			ret = -EBUSY;

		WRITE_UNLOCK(&nf_ct_cache_lock);
		up(&nf_ct_cache_mutex);
		return ret;
	}
	WRITE_UNLOCK(&nf_ct_cache_lock);

	/*
	 * The memory space for name of slab cache must be alive until
	 * cache is destroyed.
	 */
	cache_name = kmalloc(sizeof(char)*NF_CT_FEATURES_NAMELEN, GFP_ATOMIC);
	if (cache_name == NULL) {
		DEBUGP("nf_conntrack_register_cache: can't alloc cache_name\n");
		ret = -ENOMEM;
		goto out_up_mutex;
	}

	if (strlcpy(cache_name, name, NF_CT_FEATURES_NAMELEN)
						>= NF_CT_FEATURES_NAMELEN) {
		printk("nf_conntrack_register_cache: name too long\n");
		ret = -EINVAL;
		goto out_free_name;
	}

	cachep = kmem_cache_create(cache_name, size, 0, 0,
				   NULL, NULL);
	if (!cachep) {
		printk("nf_conntrack_register_cache: Can't create slab cache "
		       "for the features = 0x%x\n", features);
		ret = -ENOMEM;
		goto out_free_name;
	}

	WRITE_LOCK(&nf_ct_cache_lock);
	nf_ct_cache[features].use = 1;
	nf_ct_cache[features].size = size;
	nf_ct_cache[features].init_conntrack = init;
	nf_ct_cache[features].cachep = cachep;
	nf_ct_cache[features].name = cache_name;
	WRITE_UNLOCK(&nf_ct_cache_lock);

	goto out_up_mutex;

out_free_name:
	kfree(cache_name);
out_up_mutex:
	up(&nf_ct_cache_mutex);
	return ret;
}

/* FIXME: In the current, only nf_conntrack_cleanup() can call this function. */
void nf_conntrack_unregister_cache(u_int32_t features)
{
	kmem_cache_t *cachep;
	char *name;

	/*
	 * This assures that kmem_cache_create() isn't called before destroying
	 * slab cache.
	 */
	DEBUGP("nf_conntrack_unregister_cache: 0x%04x\n", features);
	down(&nf_ct_cache_mutex);

	WRITE_LOCK(&nf_ct_cache_lock);
	if (--nf_ct_cache[features].use > 0) {
		WRITE_UNLOCK(&nf_ct_cache_lock);
		up(&nf_ct_cache_mutex);
		return;
	}
	cachep = nf_ct_cache[features].cachep;
	name = nf_ct_cache[features].name;
	nf_ct_cache[features].cachep = NULL;
	nf_ct_cache[features].name = NULL;
	nf_ct_cache[features].init_conntrack = NULL;
	nf_ct_cache[features].size = 0;
	WRITE_UNLOCK(&nf_ct_cache_lock);

	synchronize_net();

	kmem_cache_destroy(cachep);
	kfree(name);

	up(&nf_ct_cache_mutex);
}

int
nf_ct_get_tuple(const struct sk_buff *skb,
		unsigned int nhoff,
		unsigned int dataoff,
		u_int16_t l3num,
		u_int8_t protonum,
		struct nf_conntrack_tuple *tuple,
		const struct nf_conntrack_l3proto *l3proto,
		const struct nf_conntrack_protocol *protocol)
{
	NF_CT_TUPLE_U_BLANK(tuple);

	tuple->src.l3num = l3num;
	if (l3proto->pkt_to_tuple(skb, nhoff, tuple) == 0)
		return 0;

	tuple->dst.protonum = protonum;
	tuple->dst.dir = NF_CT_DIR_ORIGINAL;

	return protocol->pkt_to_tuple(skb, dataoff, tuple);
}

int
nf_ct_invert_tuple(struct nf_conntrack_tuple *inverse,
		   const struct nf_conntrack_tuple *orig,
		   const struct nf_conntrack_l3proto *l3proto,
		   const struct nf_conntrack_protocol *protocol)
{
	NF_CT_TUPLE_U_BLANK(inverse);

	inverse->src.l3num = orig->src.l3num;
	if (l3proto->invert_tuple(inverse, orig) == 0)
		return 0;

	inverse->dst.dir = !orig->dst.dir;

	inverse->dst.protonum = orig->dst.protonum;
	return protocol->invert_tuple(inverse, orig);
}

/* nf_conntrack_expect helper functions */
static void destroy_expect(struct nf_conntrack_expect *exp)
{
	nf_ct_put(exp->master);
	NF_CT_ASSERT(!timer_pending(&exp->timeout));
	kmem_cache_free(nf_conntrack_expect_cachep, exp);
	NF_CT_STAT_INC(expect_delete);
}

static void unlink_expect(struct nf_conntrack_expect *exp)
{
	MUST_BE_WRITE_LOCKED(&nf_conntrack_lock);
	list_del(&exp->list);
	/* Logically in destroy_expect, but we hold the lock here. */
	exp->master->expecting--;
}

static void expectation_timed_out(unsigned long ul_expect)
{
	struct nf_conntrack_expect *exp = (void *)ul_expect;

	WRITE_LOCK(&nf_conntrack_lock);
	unlink_expect(exp);
	WRITE_UNLOCK(&nf_conntrack_lock);
	destroy_expect(exp);
}

/* If an expectation for this connection is found, it gets delete from
 * global list then returned. */
static struct nf_conntrack_expect *
find_expectation(const struct nf_conntrack_tuple *tuple)
{
	struct nf_conntrack_expect *i;

	list_for_each_entry(i, &nf_conntrack_expect_list, list) {
	/* If master is not in hash table yet (ie. packet hasn't left
	   this machine yet), how can other end know about expected?
	   Hence these are not the droids you are looking for (if
	   master ct never got confirmed, we'd hold a reference to it
	   and weird things would happen to future packets). */
		if (nf_ct_tuple_mask_cmp(tuple, &i->tuple, &i->mask)
		    && is_confirmed(i->master)
		    && del_timer(&i->timeout)) {
			unlink_expect(i);
			return i;
		}
	}
	return NULL;
}

/* delete all expectations for this conntrack */
static void remove_expectations(struct nf_conn *ct)
{
	struct nf_conntrack_expect *i, *tmp;

	/* Optimization: most connection never expect any others. */
	if (ct->expecting == 0)
		return;

	list_for_each_entry_safe(i, tmp, &nf_conntrack_expect_list, list) {
		if (i->master == ct && del_timer(&i->timeout)) {
			unlink_expect(i);
			destroy_expect(i);
 		}
	}
}

static void
clean_from_lists(struct nf_conn *ct)
{
	unsigned int ho, hr;
	
	DEBUGP("clean_from_lists(%p)\n", ct);
	MUST_BE_WRITE_LOCKED(&nf_conntrack_lock);

	ho = hash_conntrack(&ct->tuplehash[NF_CT_DIR_ORIGINAL].tuple);
	hr = hash_conntrack(&ct->tuplehash[NF_CT_DIR_REPLY].tuple);
	LIST_DELETE(&nf_conntrack_hash[ho], &ct->tuplehash[NF_CT_DIR_ORIGINAL]);
	LIST_DELETE(&nf_conntrack_hash[hr], &ct->tuplehash[NF_CT_DIR_REPLY]);

	/* Destroy all pending expectations */
	remove_expectations(ct);
}

static void
destroy_conntrack(struct nf_conntrack *nfct)
{
	struct nf_conn *ct = (struct nf_conn *)nfct;
	struct nf_conntrack_l3proto *l3proto;
	struct nf_conntrack_protocol *proto;

	DEBUGP("destroy_conntrack(%p)\n", ct);
	NF_CT_ASSERT(atomic_read(&nfct->use) == 0);
	NF_CT_ASSERT(!timer_pending(&ct->timeout));

	/* To make sure we don't get any weird locking issues here:
	 * destroy_conntrack() MUST NOT be called with a write lock
	 * to nf_conntrack_lock!!! -HW */
	l3proto = nf_ct_find_l3proto(ct->tuplehash[NF_CT_DIR_REPLY].tuple.src.l3num);
	if (l3proto && l3proto->destroy)
		l3proto->destroy(ct);

	proto = nf_ct_find_proto(ct->tuplehash[NF_CT_DIR_REPLY].tuple.src.l3num,
				 ct->tuplehash[NF_CT_DIR_REPLY].tuple.dst.protonum);
	if (proto && proto->destroy)
		proto->destroy(ct);

	if (nf_conntrack_destroyed)
		nf_conntrack_destroyed(ct);

	WRITE_LOCK(&nf_conntrack_lock);
	/* Expectations will have been removed in clean_from_lists,
	 * except TFTP can create an expectation on the first packet,
	 * before connection is in the list, so we need to clean here,
	 * too. */
	remove_expectations(ct);

	/* We overload first tuple to link into unconfirmed list. */
	if (!is_confirmed(ct)) {
		BUG_ON(list_empty(&ct->tuplehash[NF_CT_DIR_ORIGINAL].list));
		list_del(&ct->tuplehash[NF_CT_DIR_ORIGINAL].list);
	}

	NF_CT_STAT_INC(delete);
	WRITE_UNLOCK(&nf_conntrack_lock);

	if (ct->master)
		nf_ct_put(ct->master);

	DEBUGP("destroy_conntrack: returning ct=%p to slab\n", ct);
	free_conntrack(ct);
	atomic_dec(&nf_conntrack_count);
}

static void death_by_timeout(unsigned long ul_conntrack)
{
	struct nf_conn *ct = (void *)ul_conntrack;

	WRITE_LOCK(&nf_conntrack_lock);
	/* Inside lock so preempt is disabled on module removal path.
	 * Otherwise we can get spurious warnings. */
	NF_CT_STAT_INC(delete_list);
	clean_from_lists(ct);
	WRITE_UNLOCK(&nf_conntrack_lock);
	nf_ct_put(ct);
}

static inline int
conntrack_tuple_cmp(const struct nf_conntrack_tuple_hash *i,
		    const struct nf_conntrack_tuple *tuple,
		    const struct nf_conn *ignored_conntrack)
{
	MUST_BE_READ_LOCKED(&nf_conntrack_lock);
	return tuplehash_to_ctrack(i) != ignored_conntrack
		&& nf_ct_tuple_equal(tuple, &i->tuple);
}

static struct nf_conntrack_tuple_hash *
__nf_conntrack_find(const struct nf_conntrack_tuple *tuple,
		    const struct nf_conn *ignored_conntrack)
{
	struct nf_conntrack_tuple_hash *h;
	unsigned int hash = hash_conntrack(tuple);

	MUST_BE_READ_LOCKED(&nf_conntrack_lock);
	list_for_each_entry(h, &nf_conntrack_hash[hash], list) {
		if (conntrack_tuple_cmp(h, tuple, ignored_conntrack)) {
			NF_CT_STAT_INC(found);
			return h;
		}
		NF_CT_STAT_INC(searched);
	}

	return NULL;
}

/* Find a connection corresponding to a tuple. */
struct nf_conntrack_tuple_hash *
nf_conntrack_find_get(const struct nf_conntrack_tuple *tuple,
		      const struct nf_conn *ignored_conntrack)
{
	struct nf_conntrack_tuple_hash *h;

	READ_LOCK(&nf_conntrack_lock);
	h = __nf_conntrack_find(tuple, ignored_conntrack);
	if (h)
		atomic_inc(&tuplehash_to_ctrack(h)->ct_general.use);
	READ_UNLOCK(&nf_conntrack_lock);

	return h;
}

/* Confirm a connection given skb; places it in hash table */
int
__nf_conntrack_confirm(struct sk_buff **pskb)
{
	unsigned int hash, repl_hash;
	struct nf_conn *ct;
	enum nf_conntrack_info ctinfo;

	ct = nf_ct_get(*pskb, &ctinfo);

	/* ipt_REJECT uses nf_conntrack_attach to attach related
	   ICMP/TCP RST packets in other direction.  Actual packet
	   which created connection will be NF_CT_NEW or for an
	   expected connection, NF_CT_RELATED. */
	if (NFCTINFO2DIR(ctinfo) != NF_CT_DIR_ORIGINAL)
		return NF_ACCEPT;

	hash = hash_conntrack(&ct->tuplehash[NF_CT_DIR_ORIGINAL].tuple);
	repl_hash = hash_conntrack(&ct->tuplehash[NF_CT_DIR_REPLY].tuple);

	/* We're not in hash table, and we refuse to set up related
	   connections for unconfirmed conns.  But packet copies and
	   REJECT will give spurious warnings here. */
	/* NF_CT_ASSERT(atomic_read(&ct->ct_general.use) == 1); */

	/* No external references means noone else could have
	   confirmed us. */
	NF_CT_ASSERT(!is_confirmed(ct));
	DEBUGP("Confirming conntrack %p\n", ct);

	WRITE_LOCK(&nf_conntrack_lock);

	/* See if there's one in the list already, including reverse:
	   NAT could have grabbed it without realizing, since we're
	   not in the hash.  If there is, we lost race. */
	if (!LIST_FIND(&nf_conntrack_hash[hash],
		       conntrack_tuple_cmp,
		       struct nf_conntrack_tuple_hash *,
		       &ct->tuplehash[NF_CT_DIR_ORIGINAL].tuple, NULL)
	    && !LIST_FIND(&nf_conntrack_hash[repl_hash],
			  conntrack_tuple_cmp,
			  struct nf_conntrack_tuple_hash *,
			  &ct->tuplehash[NF_CT_DIR_REPLY].tuple, NULL)) {
		/* Remove from unconfirmed list */
		list_del(&ct->tuplehash[NF_CT_DIR_ORIGINAL].list);

		list_prepend(&nf_conntrack_hash[hash],
			     &ct->tuplehash[NF_CT_DIR_ORIGINAL]);
		list_prepend(&nf_conntrack_hash[repl_hash],
			     &ct->tuplehash[NF_CT_DIR_REPLY]);
		/* Timer relative to confirmation time, not original
		   setting time, otherwise we'd get timer wrap in
		   weird delay cases. */
		ct->timeout.expires += jiffies;
		add_timer(&ct->timeout);
		atomic_inc(&ct->ct_general.use);
		set_bit(NF_S_CONFIRMED_BIT, &ct->status);
		NF_CT_STAT_INC(insert);
		WRITE_UNLOCK(&nf_conntrack_lock);
		return NF_ACCEPT;
	}

	NF_CT_STAT_INC(insert_failed);
	WRITE_UNLOCK(&nf_conntrack_lock);
	return NF_DROP;
}

/* Returns true if a connection correspondings to the tuple (required
   for NAT). */
int
nf_conntrack_tuple_taken(const struct nf_conntrack_tuple *tuple,
			 const struct nf_conn *ignored_conntrack)
{
	struct nf_conntrack_tuple_hash *h;

	READ_LOCK(&nf_conntrack_lock);
	h = __nf_conntrack_find(tuple, ignored_conntrack);
	READ_UNLOCK(&nf_conntrack_lock);

	return h != NULL;
}

/* There's a small race here where we may free a just-assured
   connection.  Too bad: we're in trouble anyway. */
static inline int unreplied(const struct nf_conntrack_tuple_hash *i)
{
	return !(test_bit(NF_S_ASSURED_BIT, &tuplehash_to_ctrack(i)->status));
}

static int early_drop(struct list_head *chain)
{
	/* Traverse backwards: gives us oldest, which is roughly LRU */
	struct nf_conntrack_tuple_hash *h;
	struct nf_conn *ct = NULL;
	int dropped = 0;

	READ_LOCK(&nf_conntrack_lock);
	h = LIST_FIND_B(chain, unreplied, struct nf_conntrack_tuple_hash *);
	if (h) {
		ct = tuplehash_to_ctrack(h);
		atomic_inc(&ct->ct_general.use);
	}
	READ_UNLOCK(&nf_conntrack_lock);

	if (!ct)
		return dropped;

	if (del_timer(&ct->timeout)) {
		death_by_timeout((unsigned long)ct);
		dropped = 1;
		NF_CT_STAT_INC(early_drop);
	}
	nf_ct_put(ct);
	return dropped;
}

static inline int helper_cmp(const struct nf_conntrack_helper *i,
			     const struct nf_conntrack_tuple *rtuple)
{
	return nf_ct_tuple_mask_cmp(rtuple, &i->tuple, &i->mask);
}

static struct nf_conntrack_helper *
nf_ct_find_helper(const struct nf_conntrack_tuple *tuple)
{
	return LIST_FIND(&helpers, helper_cmp,
			 struct nf_conntrack_helper *,
			 tuple);
}

/* Allocate a new conntrack: we return -ENOMEM if classification
   failed due to stress.  Otherwise it really is unclassifiable. */
static struct nf_conntrack_tuple_hash *
init_conntrack(const struct nf_conntrack_tuple *tuple,
	       struct nf_conntrack_l3proto *l3proto,
	       struct nf_conntrack_protocol *protocol,
	       struct sk_buff *skb,
	       unsigned int dataoff)
{
	struct nf_conn *conntrack;
	struct nf_conntrack_tuple repl_tuple;
	size_t hash;
	struct nf_conntrack_expect *exp;
	u_int32_t features = 0;
	int helper_used = 0;

	if (!nf_conntrack_hash_rnd_initted) {
		get_random_bytes(&nf_conntrack_hash_rnd, 4);
		nf_conntrack_hash_rnd_initted = 1;
	}

	hash = hash_conntrack(tuple);

	if (nf_conntrack_max
	    && atomic_read(&nf_conntrack_count) >= nf_conntrack_max) {
		/* Try dropping from this hash chain. */
		if (!early_drop(&nf_conntrack_hash[hash])) {
			if (net_ratelimit())
				printk(KERN_WARNING
				       "nf_conntrack: table full, dropping"
				       " packet.\n");
			return ERR_PTR(-ENOMEM);
		}
	}

	if (!nf_ct_invert_tuple(&repl_tuple, tuple, l3proto, protocol)) {
		DEBUGP("Can't invert tuple.\n");
		return NULL;
	}

	/*  find features needed by this conntrack. */
	features = l3proto->get_features(tuple);
	READ_LOCK(&nf_conntrack_lock);
	if (nf_ct_find_helper(&repl_tuple) != NULL) {
		features |= NF_CT_F_HELP;
		helper_used = 1;
	}
	READ_UNLOCK(&nf_conntrack_lock);

	conntrack = alloc_conntrack(features);
	if (!conntrack) {
		DEBUGP("Can't allocate conntrack.\n");
		return ERR_PTR(-ENOMEM);
	}

	atomic_set(&conntrack->ct_general.use, 1);
	conntrack->ct_general.destroy = destroy_conntrack;
	conntrack->tuplehash[NF_CT_DIR_ORIGINAL].tuple = *tuple;
	conntrack->tuplehash[NF_CT_DIR_REPLY].tuple = repl_tuple;
	if (!protocol->new(conntrack, skb, dataoff)) {
		free_conntrack(conntrack);
		DEBUGP("init conntrack: can't track with proto module\n");
		return NULL;
	}
	/* Don't set timer yet: wait for confirmation */
	init_timer(&conntrack->timeout);
	conntrack->timeout.data = (unsigned long)conntrack;
	conntrack->timeout.function = death_by_timeout;

	WRITE_LOCK(&nf_conntrack_lock);
	exp = find_expectation(tuple);

	if (exp) {
		DEBUGP("conntrack: expectation arrives ct=%p exp=%p\n",
			conntrack, exp);
		/* Welcome, Mr. Bond.  We've been expecting you... */
		__set_bit(NF_S_EXPECTED_BIT, &conntrack->status);
		conntrack->master = conntrack;
#if CONFIG_NF_CONNTRACK_MARK
		conntrack->mark = exp->master->mark;
#endif
		nf_conntrack_get(&conntrack->master->ct_general);
		NF_CT_STAT_INC(expect_new);
	} else {
		conntrack->helper = nf_ct_find_helper(&repl_tuple);

		NF_CT_STAT_INC(new);
        }

	/* Overload tuple linked list to put us in unconfirmed list. */
	list_add(&conntrack->tuplehash[NF_CT_DIR_ORIGINAL].list, &unconfirmed);

	atomic_inc(&nf_conntrack_count);
	WRITE_UNLOCK(&nf_conntrack_lock);

	if (exp) {
		if (exp->expectfn)
			exp->expectfn(conntrack, exp);
		destroy_expect(exp);
	}

	return &conntrack->tuplehash[NF_CT_DIR_ORIGINAL];
}

/* On success, returns conntrack ptr, sets skb->nfct and ctinfo */
static inline struct nf_conn *
resolve_normal_ct(struct sk_buff *skb,
		  unsigned int dataoff,
		  u_int16_t l3num,
		  u_int8_t protonum,
		  struct nf_conntrack_l3proto *l3proto,
		  struct nf_conntrack_protocol *proto,
		  int *set_reply,
		  enum nf_conntrack_info *ctinfo)
{
	struct nf_conntrack_tuple tuple;
	struct nf_conntrack_tuple_hash *h;
	struct nf_conn *ct;

	if (!nf_ct_get_tuple(skb, (unsigned int)(skb->nh.raw - skb->data),
			     dataoff, l3num, protonum, &tuple, l3proto,
			     proto)) {
		DEBUGP("resolve_normal_ct: Can't get tuple\n");
		return NULL;
	}

	/* look for tuple match */
	h = nf_conntrack_find_get(&tuple, NULL);
	if (!h) {
		h = init_conntrack(&tuple, l3proto, proto, skb, dataoff);
		if (!h)
			return NULL;
		if (IS_ERR(h))
			return (void *)h;
	}
	ct = tuplehash_to_ctrack(h);

	/* It exists; we have (non-exclusive) reference. */
	if (NF_CT_DIRECTION(h) == NF_CT_DIR_REPLY) {
		*ctinfo = NF_CT_ESTABLISHED + NF_CT_IS_REPLY;
		/* Please set reply bit if this packet OK */
		*set_reply = 1;
	} else {
		/* Once we've had two way comms, always ESTABLISHED. */
		if (test_bit(NF_S_SEEN_REPLY_BIT, &ct->status)) {
			DEBUGP("nf_conntrack_in: normal packet for %p\n", ct);
			*ctinfo = NF_CT_ESTABLISHED;
		} else if (test_bit(NF_S_EXPECTED_BIT, &ct->status)) {
			DEBUGP("nf_conntrack_in: related packet for %p\n", ct);
			*ctinfo = NF_CT_RELATED;
		} else {
			DEBUGP("nf_conntrack_in: new packet for %p\n", ct);
			*ctinfo = NF_CT_NEW;
		}
		*set_reply = 0;
	}
	skb->nfct = &ct->ct_general;
	skb->nfctinfo = *ctinfo;
	return ct;
}

unsigned int
nf_conntrack_in(int pf, unsigned int hooknum, struct sk_buff **pskb)
{
	struct nf_conn *ct;
	enum nf_conntrack_info ctinfo;
	struct nf_conntrack_l3proto *l3proto;
	struct nf_conntrack_protocol *proto;
	unsigned int dataoff;
	u_int8_t protonum;
	int set_reply;
	int ret;

	/* Previously seen (loopback or untracked)?  Ignore. */
	if ((*pskb)->nfct) {
		NF_CT_STAT_INC(ignore);
		return NF_ACCEPT;
	}

	l3proto = nf_ct_find_l3proto((u_int16_t)pf);
	if (l3proto->prepare(pskb, hooknum, &dataoff, &protonum, &ret) == 0) {
		DEBUGP("not prepared to track yet or error occured\n");
		return ret;
	}

	proto = nf_ct_find_proto((u_int16_t)pf, protonum);

	/* It may be an special packet, error, unclean...
	 * inverse of the return code tells to the netfilter
	 * core what to do with the packet. */
	if (proto->error != NULL &&
	    (ret = proto->error(*pskb, dataoff, &ctinfo, pf, hooknum)) <= 0) {
		NF_CT_STAT_INC(error);
		NF_CT_STAT_INC(invalid);
		return -ret;
	}

	ct = resolve_normal_ct(*pskb, dataoff, pf, protonum, l3proto, proto,
			       &set_reply, &ctinfo);
	if (!ct) {
		/* Not valid part of a connection */
		NF_CT_STAT_INC(invalid);
		return NF_ACCEPT;
	}

	if (IS_ERR(ct)) {
		/* Too stressed to deal. */
		NF_CT_STAT_INC(drop);
		return NF_DROP;
	}

	NF_CT_ASSERT((*pskb)->nfct);

	ret = proto->packet(ct, *pskb, dataoff, ctinfo, pf, hooknum);
	if (ret < 0) {
		/* Invalid: inverse of the return code tells
		 * the netfilter core what to do */
		DEBUGP("nf_conntrack_in: Can't track with proto module\n");
		nf_conntrack_put((*pskb)->nfct);
		(*pskb)->nfct = NULL;
		NF_CT_STAT_INC(invalid);
		return -ret;
	}

	if (set_reply)
		set_bit(NF_S_SEEN_REPLY_BIT, &ct->status);

	return ret;
}

int nf_ct_invert_tuplepr(struct nf_conntrack_tuple *inverse,
			 const struct nf_conntrack_tuple *orig)
{
	return nf_ct_invert_tuple(inverse, orig,
				  nf_ct_find_l3proto(orig->src.l3num),
				  nf_ct_find_proto(orig->src.l3num,
						   orig->dst.protonum));
}

/* Would two expected things clash? */
static inline int expect_clash(const struct nf_conntrack_expect *a,
			       const struct nf_conntrack_expect *b)
{
	/* Part covered by intersection of masks must be unequal,
	   otherwise they clash */
	struct nf_conntrack_tuple intersect_mask;
	int count;

	intersect_mask.src.l3num = a->mask.src.l3num & b->mask.src.l3num;
	intersect_mask.src.u.all = a->mask.src.u.all & b->mask.src.u.all;
	intersect_mask.dst.u.all = a->mask.dst.u.all & b->mask.dst.u.all;
	intersect_mask.dst.protonum = a->mask.dst.protonum
					& b->mask.dst.protonum;

	for (count = 0; count < NF_CT_TUPLE_L3SIZE; count++){
		intersect_mask.src.u3.all[count] =
			a->mask.src.u3.all[count] & b->mask.src.u3.all[count];
	}

	for (count = 0; count < NF_CT_TUPLE_L3SIZE; count++){
		intersect_mask.dst.u3.all[count] =
			a->mask.dst.u3.all[count] & b->mask.dst.u3.all[count];
	}

	return nf_ct_tuple_mask_cmp(&a->tuple, &b->tuple, &intersect_mask);
}

static inline int expect_matches(const struct nf_conntrack_expect *a,
				 const struct nf_conntrack_expect *b)
{
	return a->master == b->master
		&& nf_ct_tuple_equal(&a->tuple, &b->tuple)
		&& nf_ct_tuple_equal(&a->mask, &b->mask);
}

/* Generally a bad idea to call this: could have matched already. */
void nf_conntrack_unexpect_related(struct nf_conntrack_expect *exp)
{
	struct nf_conntrack_expect *i;

	WRITE_LOCK(&nf_conntrack_lock);
	/* choose the the oldest expectation to evict */
	list_for_each_entry_reverse(i, &nf_conntrack_expect_list, list) {
		if (expect_matches(i, exp) && del_timer(&i->timeout)) {
			unlink_expect(i);
			WRITE_UNLOCK(&nf_conntrack_lock);
			destroy_expect(i);
			return;
		}
	}
	WRITE_UNLOCK(&nf_conntrack_lock);
}

struct nf_conntrack_expect *nf_conntrack_expect_alloc(void)
{
	struct nf_conntrack_expect *new;

	new = kmem_cache_alloc(nf_conntrack_expect_cachep, GFP_ATOMIC);
	if (!new) {
		DEBUGP("expect_related: OOM allocating expect\n");
		return NULL;
	}
	new->master = NULL;
	return new;
}

void nf_conntrack_expect_free(struct nf_conntrack_expect *expect)
{
	kmem_cache_free(nf_conntrack_expect_cachep, expect);
}

static void nf_conntrack_expect_insert(struct nf_conntrack_expect *exp)
{
	atomic_inc(&exp->master->ct_general.use);
	exp->master->expecting++;
	list_add(&exp->list, &nf_conntrack_expect_list);

	if (exp->master->helper->timeout) {
		init_timer(&exp->timeout);
		exp->timeout.data = (unsigned long)exp;
		exp->timeout.function = expectation_timed_out;
		exp->timeout.expires
			= jiffies + exp->master->helper->timeout * HZ;
		add_timer(&exp->timeout);
	} else
		exp->timeout.function = NULL;

	NF_CT_STAT_INC(expect_create);
}

/* Race with expectations being used means we could have none to find; OK. */
static void evict_oldest_expect(struct nf_conn *master)
{
	struct nf_conntrack_expect *i;

	list_for_each_entry_reverse(i, &nf_conntrack_expect_list, list) {
		if (i->master == master) {
			if (del_timer(&i->timeout)) {
				unlink_expect(i);
				destroy_expect(i);
			}
			break;
		}
	}
}

static inline int refresh_timer(struct nf_conntrack_expect *i)
{
	if (!del_timer(&i->timeout))
		return 0;

	i->timeout.expires = jiffies + i->master->helper->timeout*HZ;
	add_timer(&i->timeout);
	return 1;
}

int nf_conntrack_expect_related(struct nf_conntrack_expect *expect)
{
	struct nf_conntrack_expect *i;
	int ret;

	DEBUGP("nf_conntrack_expect_related %p\n", related_to);
	DEBUGP("tuple: "); NF_CT_DUMP_TUPLE(&expect->tuple);
	DEBUGP("mask:  "); NF_CT_DUMP_TUPLE(&expect->mask);

	WRITE_LOCK(&nf_conntrack_lock);
	list_for_each_entry(i, &nf_conntrack_expect_list, list) {
		if (expect_matches(i, expect)) {
			/* Refresh timer: if it's dying, ignore.. */
			if (refresh_timer(i)) {
				ret = 0;
				/* We don't need the one they've given us. */
				nf_conntrack_expect_free(expect);
				goto out;
			}
		} else if (expect_clash(i, expect)) {
			ret = -EBUSY;
			goto out;
		}
	}
	/* Will be over limit? */
	if (expect->master->helper->max_expected && 
	    expect->master->expecting >= expect->master->helper->max_expected)
		evict_oldest_expect(expect->master);

	nf_conntrack_expect_insert(expect);
	ret = 0;
out:
	WRITE_UNLOCK(&nf_conntrack_lock);
	return ret;
}

/* Alter reply tuple (maybe alter helper).  This is for NAT, and is
   implicitly racy: see __nf_conntrack_confirm */
void nf_conntrack_alter_reply(struct nf_conn *conntrack,
			      const struct nf_conntrack_tuple *newreply)
{
	WRITE_LOCK(&nf_conntrack_lock);
	/* Should be unconfirmed, so not in hash table yet */
	NF_CT_ASSERT(!is_confirmed(conntrack));

	DEBUGP("Altering reply tuple of %p to ", conntrack);
	NF_CT_DUMP_TUPLE(newreply);

	conntrack->tuplehash[NF_CT_DIR_REPLY].tuple = *newreply;
	if (!conntrack->master && conntrack->expecting == 0)
		conntrack->helper = nf_ct_find_helper(newreply);
	WRITE_UNLOCK(&nf_conntrack_lock);
}

int nf_conntrack_helper_register(struct nf_conntrack_helper *me)
{
	int ret;
	BUG_ON(me->timeout == 0);

	ret = nf_conntrack_register_cache(NF_CT_F_HELP, "nf_conntrack:help",
					  sizeof(struct nf_conn)
					  + sizeof(union nf_conntrack_help)
					  + __alignof__(union nf_conntrack_help),
					  init_conntrack_for_helper);
	if (ret < 0) {
		printk(KERN_ERR "nf_conntrack_helper_reigster: Unable to create slab cache for conntracks\n");
		return ret;
	}
	WRITE_LOCK(&nf_conntrack_lock);
	list_prepend(&helpers, me);
	WRITE_UNLOCK(&nf_conntrack_lock);

	return 0;
}

static inline int unhelp(struct nf_conntrack_tuple_hash *i,
			 const struct nf_conntrack_helper *me)
{
	if (tuplehash_to_ctrack(i)->helper == me)
		tuplehash_to_ctrack(i)->helper = NULL;
	return 0;
}

void nf_conntrack_helper_unregister(struct nf_conntrack_helper *me)
{
	unsigned int i;
	struct nf_conntrack_expect *exp, *tmp;

	/* Need write lock here, to delete helper. */
	WRITE_LOCK(&nf_conntrack_lock);
	LIST_DELETE(&helpers, me);

	/* Get rid of expectations */
	list_for_each_entry_safe(exp, tmp, &nf_conntrack_expect_list, list) {
		if (exp->master->helper == me && del_timer(&exp->timeout)) {
			unlink_expect(exp);
			destroy_expect(exp);
		}
	}

	/* Get rid of expecteds, set helpers to NULL. */
	LIST_FIND_W(&unconfirmed, unhelp, struct nf_conntrack_tuple_hash*, me);
	for (i = 0; i < nf_conntrack_htable_size; i++)
		LIST_FIND_W(&nf_conntrack_hash[i], unhelp,
			    struct nf_conntrack_tuple_hash *, me);
	WRITE_UNLOCK(&nf_conntrack_lock);

	/* Someone could be still looking at the helper in a bh. */
	synchronize_net();
}

static inline void ct_add_counters(struct nf_conn *ct,
				   enum nf_conntrack_info ctinfo,
				   const struct sk_buff *skb)
{
#ifdef CONFIG_NF_CT_ACCT
	if (skb) {
		ct->counters[NFCTINFO2DIR(ctinfo)].packets++;
		/* XXX totlen should be used ? - YK */
		ct->counters[NFCTINFO2DIR(ctinfo)].bytes +=
			ntohs(skb->len - (unsigned int)(skb->nh.raw
							- skb->data));
	}
#endif
}

/* Refresh conntrack for this many jiffies and do accounting (if skb != NULL) */
void nf_ct_refresh_acct(struct nf_conn *ct,
			enum nf_conntrack_info ctinfo,
			const struct sk_buff *skb,
			unsigned long extra_jiffies)
{
	NF_CT_ASSERT(ct->timeout.data == (unsigned long)ct);

	/* If not in hash table, timer will not be active yet */
	if (!is_confirmed(ct)) {
		ct->timeout.expires = extra_jiffies;
		ct_add_counters(ct, ctinfo, skb);
	} else {
		WRITE_LOCK(&nf_conntrack_lock);
		/* Need del_timer for race avoidance (may already be dying). */
		if (del_timer(&ct->timeout)) {
			ct->timeout.expires = jiffies + extra_jiffies;
			add_timer(&ct->timeout);
		}
		ct_add_counters(ct, ctinfo, skb);
		WRITE_UNLOCK(&nf_conntrack_lock);
	}
}

/* Used by ipt_REJECT and ip6t_REJECT. */
void __nf_conntrack_attach(struct sk_buff *nskb, struct sk_buff *skb)
{
	struct nf_conn *ct;
	enum nf_conntrack_info ctinfo;

	/* This ICMP is in reverse direction to the packet which caused it */
	ct = nf_ct_get(skb, &ctinfo);
	if (NFCTINFO2DIR(ctinfo) == NF_CT_DIR_ORIGINAL)
		ctinfo = NF_CT_RELATED + NF_CT_IS_REPLY;
	else
		ctinfo = NF_CT_RELATED;

	/* Attach to new skbuff, and increment count */
	nskb->nfct = &ct->ct_general;
	nskb->nfctinfo = ctinfo;
	nf_conntrack_get(nskb->nfct);
}

static inline int
do_iter(const struct nf_conntrack_tuple_hash *i,
	int (*iter)(struct nf_conn *i, void *data),
	void *data)
{
	return iter(tuplehash_to_ctrack(i), data);
}

/* Bring out ya dead! */
static struct nf_conntrack_tuple_hash *
get_next_corpse(int (*iter)(struct nf_conn *i, void *data),
		void *data, unsigned int *bucket)
{
	struct nf_conntrack_tuple_hash *h = NULL;

	WRITE_LOCK(&nf_conntrack_lock);
	for (; *bucket < nf_conntrack_htable_size; (*bucket)++) {
		h = LIST_FIND_W(&nf_conntrack_hash[*bucket], do_iter,
				struct nf_conntrack_tuple_hash *, iter, data);
		if (h)
			break;
 	}
	if (!h)
		h = LIST_FIND_W(&unconfirmed, do_iter,
				struct nf_conntrack_tuple_hash *, iter, data);
	if (h)
		atomic_inc(&tuplehash_to_ctrack(h)->ct_general.use);
	WRITE_UNLOCK(&nf_conntrack_lock);

	return h;
}

void
nf_ct_iterate_cleanup(int (*iter)(struct nf_conn *i, void *data), void *data)
{
	struct nf_conntrack_tuple_hash *h;
	unsigned int bucket = 0;

	while ((h = get_next_corpse(iter, data, &bucket)) != NULL) {
		struct nf_conn *ct = tuplehash_to_ctrack(h);
		/* Time to push up daises... */
		if (del_timer(&ct->timeout))
			death_by_timeout((unsigned long)ct);
		/* ... else the timer will get him soon. */

		nf_ct_put(ct);
	}
}

static int kill_all(struct nf_conn *i, void *data)
{
	return 1;
}

static void free_conntrack_hash(void)
{
	if (nf_conntrack_vmalloc)
		vfree(nf_conntrack_hash);
	else
		free_pages((unsigned long)nf_conntrack_hash, 
			   get_order(sizeof(struct list_head)
				     * nf_conntrack_htable_size));
}

/* Mishearing the voices in his head, our hero wonders how he's
   supposed to kill the mall. */
void nf_conntrack_cleanup(void)
{
	int i;

	/* This makes sure all current packets have passed through
	   netfilter framework.  Roll on, two-stage module
	   delete... */
	synchronize_net();

 i_see_dead_people:
	nf_ct_iterate_cleanup(kill_all, NULL);
	if (atomic_read(&nf_conntrack_count) != 0) {
		schedule();
		goto i_see_dead_people;
	}

	for (i = 0; i < NF_CT_F_NUM; i++) {
		if (nf_ct_cache[i].use == 0)
			continue;

		NF_CT_ASSERT(nf_ct_cache[i].use == 1);
		nf_ct_cache[i].use = 1;
		nf_conntrack_unregister_cache(i);
	}
	kmem_cache_destroy(nf_conntrack_expect_cachep);
	free_conntrack_hash();
}

static int hashsize;
module_param(hashsize, int, 0400);

int __init nf_conntrack_init(void)
{
	unsigned int i;
	int ret;

	/* Idea from tcp.c: use 1/16384 of memory.  On i386: 32MB
	 * machine has 256 buckets.  >= 1GB machines have 8192 buckets. */
	if (hashsize) {
		nf_conntrack_htable_size = hashsize;
	} else {
		nf_conntrack_htable_size
			= (((num_physpages << PAGE_SHIFT) / 16384)
			   / sizeof(struct list_head));
		if (num_physpages > (1024 * 1024 * 1024 / PAGE_SIZE))
			nf_conntrack_htable_size = 8192;
		if (nf_conntrack_htable_size < 16)
			nf_conntrack_htable_size = 16;
	}
	nf_conntrack_max = 8 * nf_conntrack_htable_size;

	printk("nf_conntrack version %s (%u buckets, %d max)\n",
	       NF_CONNTRACK_VERSION, nf_conntrack_htable_size,
	       nf_conntrack_max);

	/* AK: the hash table is twice as big than needed because it
	   uses list_head.  it would be much nicer to caches to use a
	   single pointer list head here. */
	nf_conntrack_vmalloc = 0; 
	nf_conntrack_hash 
		=(void*)__get_free_pages(GFP_KERNEL, 
					 get_order(sizeof(struct list_head)
						   *nf_conntrack_htable_size));
	if (!nf_conntrack_hash) { 
		nf_conntrack_vmalloc = 1;
		printk(KERN_WARNING "nf_conntrack: falling back to vmalloc.\n");
		nf_conntrack_hash = vmalloc(sizeof(struct list_head)
					    * nf_conntrack_htable_size);
	}
	if (!nf_conntrack_hash) {
		printk(KERN_ERR "Unable to create nf_conntrack_hash\n");
		goto err_unreg_sockopt;
	}

	ret = nf_conntrack_register_cache(NF_CT_F_BASIC, "nf_conntrack:basic",
					  sizeof(struct nf_conn), NULL);
	if (ret < 0) {
		printk(KERN_ERR "Unable to create nf_conn slab cache\n");
		goto err_free_hash;
	}

	nf_conntrack_expect_cachep = kmem_cache_create("nf_conntrack_expect",
					sizeof(struct nf_conntrack_expect),
					0, 0, NULL, NULL);
	if (!nf_conntrack_expect_cachep) {
		printk(KERN_ERR "Unable to create nf_expect slab cache\n");
		goto err_free_conntrack_slab;
	}

	/* Don't NEED lock here, but good form anyway. */
	WRITE_LOCK(&nf_conntrack_lock);
        for (i = 0; i < PF_MAX; i++)
		nf_ct_l3protos[i] = &nf_conntrack_generic_l3proto;
        WRITE_UNLOCK(&nf_conntrack_lock);

	for (i = 0; i < nf_conntrack_htable_size; i++)
		INIT_LIST_HEAD(&nf_conntrack_hash[i]);

	/* Set up fake conntrack:
	    - to never be deleted, not in any hashes */
	atomic_set(&nf_conntrack_untracked.ct_general.use, 1);
	/*  - and look it like as a confirmed connection */
	set_bit(NF_S_CONFIRMED_BIT, &nf_conntrack_untracked.status);

	return ret;

err_free_conntrack_slab:
	nf_conntrack_unregister_cache(NF_CT_F_BASIC);
err_free_hash:
	vfree(nf_conntrack_hash);
err_unreg_sockopt:
	return -ENOMEM;
}
