/* Kernel module to match realm from routing. */
#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <net/route.h>

#include <linux/netfilter_ipv4/ipt_realm.h>
#include <linux/netfilter_ipv4/ip_tables.h>

MODULE_AUTHOR("Sampsa Ranta <sampsa@netsonic.fi>");
MODULE_LICENSE("GPL");

static int
match(const struct sk_buff *skb,
      const struct net_device *in,
      const struct net_device *out,
      const void *matchinfo,
      int offset,
      int *hotdrop)
{
	const struct ipt_realm_info *info = matchinfo;
	struct dst_entry *dst = skb->dst;
	u32 id;
    
	if(dst == NULL)
		return 0;
	id = dst->tclassid;

	return (info->id == (id & info->mask)) ^ info->invert;
}

static int check(const char *tablename,
                 const struct ipt_ip *ip,
                 void *matchinfo,
                 unsigned int matchsize,
                 unsigned int hook_mask)
{
	if (hook_mask
	    & ~((1 << NF_IP_POST_ROUTING) | (1 << NF_IP_FORWARD) |
	        (1 << NF_IP_LOCAL_OUT)| (1 << NF_IP_LOCAL_IN))) {
		printk("ipt_realm: only valid for POST_ROUTING, LOCAL_OUT, "
		       "LOCAL_IN or FORWARD.\n");
		return 0;
	}

	if (matchsize != IPT_ALIGN(sizeof(struct ipt_realm_info)))
		return 0;

	return 1;
}

static struct ipt_match realm_match = {
	.name = "realm",
	.match = match, 
	.checkentry = check,
	.me = THIS_MODULE
};

static int __init init(void)
{
	return ipt_register_match(&realm_match);
}

static void __exit fini(void)
{
	ipt_unregister_match(&realm_match);
}

module_init(init);
module_exit(fini);
