// imgtool_nvram.cpp : 
// imgtool_nvram - (c)2006 Jeremy Colalke - jeremy@bitsum.com
//
// Sets default nvram variables in a CFE image.
//
//

#include "stdafx.h"
#include <stdio.h>
#include <vector>

using namespace std;

#define APP_VERSION "0.01 alpha"
#define NVRAM_CFE_OFFSET 0x1000
#define FLASH_HEADER_SIZE 0x14
#define NVRAM_AREA_SIZE 0x1000

int find_variable_index(vector<string> *pvVariables, char *pszVariable)
{	
	for(unsigned int nI=0;nI<pvVariables->size();nI++)
	{
		string sVar=(*pvVariables)[nI];
		if(_strcmpi(sVar.c_str(),pszVariable)==0)
		{
			return nI;
		}
	}
	return -1;
}

int read_nvram(void *pvCfeImage, vector<string> *pvVariables, vector<string> *pvValues)
{
	printf("\n + Reading nvram ...");
	char *pszArray=(char *)pvCfeImage+NVRAM_CFE_OFFSET+FLASH_HEADER_SIZE;
	int nCount,nIndex;	
	for(nCount=0,nIndex=pvVariables->size();pszArray[nCount];nCount+=strlen(pszArray+nCount)+1)
	{
		char *pszValue=strstr(pszArray+nCount,"=");
		if(!pszValue)
		{
			printf(" ! ERROR: Can not find value for variable at image offset %d. Assuming bad image, aborting.", nCount);
			pvVariables->clear();
			pvValues->clear();
			return -1;
		}
		*pszValue++=0;		
		
		if(find_variable_index(pvVariables,pszArray+nCount)!=-1)
		{
			//
		}
		else
		{
			pvValues->push_back(pszValue);
			pvVariables->push_back(pszArray+nCount);				
			printf("\n    %s=%s", (*pvVariables)[nIndex].c_str(), (*pvValues)[nIndex].c_str());
			nIndex++;
		}		
		*(pszValue-1)='='; //for strlen
	}
	return nCount;
}

int write_nvram(void *pCfeImage, vector<string> *pvVariables, vector<string> *pvValues)
{
	printf("\n + Writing nvram ...");	
	char *pszArray=(char *)pCfeImage+NVRAM_CFE_OFFSET+FLASH_HEADER_SIZE;
	if(pvVariables->size()!=pvValues->size())
	{
		return -1;
	}
	memset(pszArray,0,NVRAM_AREA_SIZE-FLASH_HEADER_SIZE);
	char *pszTarget=pszArray;
	int nOffset=0;
	for(unsigned int nI=0;nI<pvVariables->size();nI++)
	{
		sprintf(pszTarget+nOffset,"%s=%s",(*pvVariables)[nI].c_str(),(*pvValues)[nI].c_str());
		printf("\n    %s", pszTarget+nOffset);
		nOffset+=strlen(pszTarget+nOffset)+1;
		if(nOffset>=NVRAM_AREA_SIZE)
		{
			printf("\n ! ERROR: NVRAM area exceeds max size of %d bytes.", NVRAM_AREA_SIZE);
			return -1;
		}
	}	
	return nOffset;
}

int embed_nvram(void *pvCFEImage, vector<string> *pvVariables, vector<string> *pvValues)
{
	// the read ignores subsequent instances of variables that
	// already exist, so by reading into the vectors that 
	// already contain any new nvram variables, we essentially
	// replace them.
	if(read_nvram(pvCFEImage,pvVariables,pvValues)<=0)
	{		
		printf("\n ! ERROR: Read");
		return -1;
	}
	printf("\n Embedding nvram ...");   // really no work here, all in read/write
	int nBytesWritten;
	if((nBytesWritten=write_nvram(pvCFEImage,pvVariables,pvValues))<=0)
	{
		printf("\n ! ERROR: Writes");
		return -1;
	}
	return nBytesWritten;
}

int _tmain(int argc, _TCHAR* argv[])
{
	printf(
		"\n imgtool_nvram %s - (c)2006 Jeremy Collake (jeremy@bitsum.com)"
		"\n Free for all the world. GPL License.\n", APP_VERSION);

	if(argc<2)
	{
usage:
		printf(
			"\n"
			"\n Usage: imgtool_nvram cfeimage.bin variable1=xxx variable2=yyy ..."
			"\n"
			"\n  cfeimage.bin is the input file, an pure image of the CFE."
			"\n"
			"\n  Any number of variables can be specified. If an existing variable"
			"\n  already exists, it will be replaced."
			"\n"
			"\n  Variables whose value includes spaces should be encapsulated in"
			"\n  quotation marks."
			"\n"
			"\n  If no variables are specified, the current variables are shown."			
			"\n\n");
		return 1;
	}

	vector<string> vVariables;
	vector<string> vValues;
	string sInputFile;
	for(int nI=1;nI<argc;nI++)
	{
		char *p1=new char[strlen(argv[nI])+1];
		strcpy(p1,argv[nI]);
		char *pszValue=strstr(p1,"=");		
		if(pszValue)
		{			
			*pszValue++=0;
			vVariables.push_back(p1);
			vValues.push_back(pszValue);
			printf("\n + Embedding %s=%s", p1, pszValue);			
		}
		else
		{
			if(sInputFile.length())
			{
				printf("\n ! ERROR: Variable without value or invalid parameters.");
				delete p1;
				goto usage;
			}
			sInputFile=p1;
			printf("\n + CFE image: %s", sInputFile.c_str());
		}
		delete p1;
	}

	FILE *fIn=fopen(sInputFile.c_str(),"r+b");
	if(!fIn)
	{
		printf("\n\n ! ERROR: Opening CFE image file for writing.\n");
		return 1;
	}
	fseek(fIn,0,SEEK_END);
	unsigned long nSize=ftell(fIn);	
	fseek(fIn,0,SEEK_SET);

	unsigned char *pImage=new unsigned char [nSize];
	if(fread(pImage,1,nSize,fIn)!=nSize)
	{
		printf("\n\n ! ERROR: Reading CFE image.\n");
		delete pImage;
		fclose(fIn);
		return 1;
	}
	fseek(fIn,0,SEEK_SET);

	if(embed_nvram((void *)pImage,&vVariables,&vValues)<=0)
	{
		printf("\n\n ! ERROR: NVRAM manipulation failed.\n");
		delete pImage;
		fclose(fIn);
		return 1;
	}	
	if(fwrite(pImage,1,nSize,fIn)!=nSize)
	{
		printf("\n\n ! ERROR: File write failed.\n");
		delete pImage;
		fclose(fIn);
		return 1;
	}
	delete pImage;
	fclose(fIn);
	printf("\n\n Completed successfully..\n");

	return 0;
}

