/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_auth_sql.pike,v 1.14.2.1 2004/02/24 22:49:57 vida Exp $
 */

#include <module.h>
#include <camas/globals.h>
inherit "module";

constant cvs_version="$Id: camas_auth_sql.pike,v 1.14.2.1 2004/02/24 22:49:57 vida Exp $";
constant module_type = MODULE_PROVIDER;
constant module_name = "CAMAS: Auth SQL";
constant module_doc  = "SQL Auth Module for CAMAS. This module provides the interface "
                       "between CAMAS and SQL backend to get login, name, ...";
constant module_unique = 1;
constant thread_safe=1;		// I think this module should be :)

private mapping(string:object) sql_cons;
private string table;

void create()
{
#ifdef CAMAS_DEBUG
  defvar("debug", 0, "Debug", TYPE_FLAG,
         "When on, debug messages will be logged in Caudium's debug logfile. "
         "This information is very useful to the developers when fixing bugs.");
#endif
  defvar("emaillogin",0,"Login using email address",TYPE_FLAG,
         "Use the email address to login instead of the imap login");

  defvar("dnsname","","The sitewide DNS domain name",TYPE_STRING,
         "The DNS domain name used for CAMAS webmail");

  string sqlhelp = "Specifies the default host to use for SQL-queries.\n"
         "This argument can also be used to specify which SQL-server to "
         "use by specifying an \"SQL-URL\":<ul>\n"
         "<pre>[<i>sqlserver</i>://][[<i>user</i>][:<i>password</i>]@]"
         "[<i>host</i>[:<i>port</i>]]/<i>database</i></pre></ul><br>\n"
         "Valid values for \"sqlserver\" depend on which "
         "sql-servers your pike has support for, but the following "
         "might exist: msql, mysql, odbc, oracle, postgres.<br />\n"
         "<i>Note</i>: You can specify several SQL servers, see Multiple server"
         " startegy. <br />Comma separated values.";
  defvar("sqlserver", "mysql://localhost", "SQL server host",
         TYPE_STRING, sqlhelp);
  defvar("strategy", "failover", "Multiple server strategy",
         TYPE_STRING_LIST, "If you defined several SQL server, this module"
         " will can use these strategies: <ul><ol><b>Failover</b>: In this mode "
         "the module will always query the first server in the list unless there "
         "is an error. In this case it will try the next server.</ol>"
         "<ol><b>Load balancing</b>: The module will query the server in a random "
         "order. If a server is not available, it will retry with another random choosen "
         "one</ol></ul>", ({ "failover", "load balancing" }));
  defvar("table", "CamasLogin", "Table name",
         TYPE_STRING, "The table where to put preferences. This table "
         "will be automatically created and fields will be automatically "
         "created if new preferences are added to CAMAS core.");
  defvar("loginquery", "SELECT imaplogin from $TABLE where login='$LOGIN'", 
         "SQL Query: Login",
         TYPE_STRING, "The SQL query to execute for fetching login sent to IMAP from "
         "login string entered by user. You can use the following replacements:<br />"
         "<ul><li><b>$TABLE</b>: The table name</li>"
         "<li><b>$LOGIN</b>: The login entered by the user</li>"
         "<li><b>$LOGIN1</b>: The first part of the login entered by the user if his login "
         "has a white space</li>"
         "<li><b>$LOGIN2</b>: The second part of the login entered by the user if his login "
         "has a white space</li>"
         "<li><b>$DOMAIN</b>: The domain name you enter in 'Sitewide DNS domain name'</li></ul>"
         "<br />SELECT id_user as imaplogin from $TABLE where prenom='$LOGIN1' and nom='$LOGIN2'");
  defvar("namequery", "SELECT surname, name from $TABLE where login='$LOGIN'",
         "SQL Query: Name",
         TYPE_STRING, "The SQL query to execute for fetching name and surname from login "
         "entered by user. You can use the following replacements:<br />"
         "<ul><li><b>$TABLE</b>: The table name</li>"
         "<li><b>$LOGIN</b>: The IMAP login</li>"
         "<li><b>$DOMAIN</b>: The domain name you enter in 'Sitewide DNS domain name'</li></ul>"
         "<br />SELECT prenom as surname, nom as name from $TABLE where id_user='$LOGIN'");
  defvar("emailquery", "SELECT email from $TABLE where login='$LOGIN'",
          "SQL Query: Email",
          TYPE_STRING, "The SQL query to execute for fetching email from login entered by user. "
          "If empty the email will be guess from login and the domain name. "
          "You can use the following replacements: <br />"
          "<ul><li><b>$TABLE</b>: The table name</li>"
          "<li><b>$LOGIN</b>: The IMAP login</li>"
          "<li><b>$DOMAIN</b>: The domain name you enter in 'Sitewide DNS domain name'</li></ul>"
          "<br />Example: SELECT email_from as email from $TABLE where id_user='$LOGIN'");
  defvar("sql_imapserver", "localhost", "IMAP Server:IMAP server", TYPE_STRING,
  	     "The default IMAP server if Get IMAP from LDAP is not set or if no "
         "IMAP server is found");
  defvar("sql_imapport", 143, "IMAP Server:IMAP server port", TYPE_INT,
         "The port of the IMAP server we connect to");
}

void connect(object conf)
{
  array(string) servers;
  mixed global_err;
#if constant(thread_create)
  object lock = conf->get_provider ("camas_main")->global_lock->lock();
#endif
  global_err = catch {
    sql_cons = ([ ]);
  };
#if constant(thread_create)
  destruct(lock);
#endif
  if(global_err)
    report_error("error in camas_auth_sql.pike: %s\n", describe_backtrace(global_err));
  if(stringp(QUERY(sqlserver)))
    servers = QUERY(sqlserver) / ",";
  foreach(servers, string server)
  {
    server = String.trim_all_whites(server);
#if constant(thread_create)
    object lock = conf->get_provider ("camas_main")->global_lock->lock();
#endif
    global_err = catch {
      mixed err = catch {
         sql_cons += ([ server: Sql.Sql(server) ]);
       };
       if(err)
       {
         if(sql_cons[server])
           destruct (sql_cons[server]);
         sql_cons[server] = 0;
         report_error("Camas Preferences: Fail to connect to %s: %s\n",
             server, describe_backtrace(err));
       }
       table = QUERY(table);
    };
#if constant(thread_create)
     destruct(lock);
#endif
     if(global_err)
       report_error("error in camas_auth_sql.pike: %s\n", describe_backtrace(global_err));
  }
  random_seed(getpid() + time());
}

void start(int cnf, object conf)
{
  connect(conf);
}

string status()
{
    string status = "";
  foreach(indices(sql_cons), string server)
  {
    object sql_con = sql_cons[server];
    if(!sql_con)
      status += "<strong><font color=\"red\">Not connected to database on " + server + 
        " <br /></font></strong>";
    else
      status += sprintf("Connected to %s<br />", sql_con->host_info());
  }
  return status;
}

string query_provides()
{
  return("camas_auth");
}

void stop()
{
  foreach(values(sql_cons), object sql_con) 
    if(sql_con)
      destruct(sql_con);
}

// returns a sql server
string get_server(int i)
{
  switch(QUERY(strategy))
  {
    case "load balancing":
      return indices(sql_cons)[random(sizeof(sql_cons))];
    case "failover": 
      if(i < sizeof(sql_cons))
        return indices(sql_cons)[i];
    default: 
      return indices(sql_cons)[0];
  }
}

// automatically reconnected from disconnected SQL connection
// if needed
void reconnect(object sql_con, string server)
{
  int err = catch(sql_con->list_dbs());
  if(!sql_con || err)
    sql_con = Sql.Sql(server);
}

mixed sql_auth(string login, string query)
{
  for(int i = 0; i < sizeof(sql_cons); i++)
  {
    string server = get_server(i);
    object sql_con = sql_cons[server];
    reconnect(sql_con, server);
    mixed err = catch {
      CDEBUG(sprintf("querying on host %O\n", server));
      if(!sql_con)
        throw (({ "Can't connect\n", backtrace() }));
      array logins = login / " ";
      if(sizeof(logins) == 1)
        logins += ({ "" });
      string|int domain = getdnsname();
      if(intp(domain))
        domain = "";
      query = replace(query,
          ({ 
             "$LOGIN",
             "$LOGIN1",
             "$LOGIN2",
             "$DOMAIN",
             "$TABLE",
           }),
           ({
             login,
             logins[0],
             logins[1],
             domain,
             table,
           })); 
      CDEBUG(sprintf("getlogin query=%s\n", query));
      sql_con->query(query);
    };
    if(err)
    {
      werror(sprintf("error querying server %s: %s\n", server, describe_backtrace(err)));
    }
    else
    {
      array res = sql_con->query(query);
      if(sizeof(res) > 0)
        return res[0]; 
    }
  }
  return 0; 
}

/*
 * What we provide here
 */

//
//! method: string|int getdnsname(void)
//!  Gets the sitewide email domain name that can be used
//!  for email login or to complete unqualified emails
//! returns:
//!  a string : the dns domain name<br />
//!  an int : an error code if there is an error, see notes.
//! note:
//!  error codes :<br />
//!  0 : Error (standart code or unknown error type)<br />
//!  1 : There is no domain name set, use whole email or imap login
//
string|int getdnsname() {
  if(QUERY(dnsname)=="")
    return 1;
  else
    return QUERY(dnsname);
}

//
//! method: int version(void)
//!  Give the CAMAS_AUTH api version
//!  supported by the module
//! returns:
//!  the version of the API
//! note:
//!  The base API is 1. But if we provide de v2, we
//!  *MUST* be backward compatible with v1.
//
int version()
{
  return 2;
}

//
//! method: string|int getlogin(string login)
//!  Return the imap login to be used to connect into IMAP
//!  server. Or a int to throw with an error
//! arg: string login
//!  The login typed/entered on the CAMAS login string.
//!  This can be an email, a part of an email or an imap id.
//! returns:
//!  a string : the imap login to be used.<br />
//!  an int : an error code if there is an error, see notes.
//! note:
//!  error codes :<br />
//!  0 : Error (standard code or for unknown error type)<br />
//!  1 : Bad password<br />
//!  2 : Access denied<br />
//!  3 : Account is locked<br />
//!  4 : Change your password<br />
//
string|int getlogin(string login)
{
  mixed res = sql_auth(login, QUERY(loginquery));
  if(!res)
    return 0;
  return res["imaplogin"];
}

//
//! method: int getemaillogin(void)
//!  Do the email is used instead of the imap login as
//!  login method ?
//! returns:
//!  a int : 1 = Yes, 0 = No
//
int getemaillogin()
{
  return (int)QUERY(emaillogin);
}

//
//! method: array|int getfullnames(string login)
//!  Get the name and the surname of the specifier imap login
//! arg: string login
//!  The imap login
//! returns:
//!  a array : the name and surname in the form :
//!  ({ "surname", "name", "email@domain.com", "imaplogin" })<br />
//!  an int : an error code if there is an error, see notes.
//! note:
//!  error codes :<br />
//!  0 : Error (standard error code or unknown error type)<br />
//!  1 : Empty set
//
array|int getfullnames(string login)
{
  array res = ({ "", "", login + "@" + getdnsname() , login });
  mixed res_name = sql_auth(login, QUERY(namequery));
  if(res_name)
    res = ({ res_name["surname"], res_name["name"] }) + res[2..3];
  res_name = sql_auth(login, QUERY(emailquery));
  if(res_name)
    res[2] = res_name["email"];
  return res;
}

//
//! method: mapping(string:string|int) getimapserver(string login)
//!  Return the imap informations to be used to connect into IMAP
//!  server. Or a int to throw with an error
//! arg: string login
//!  The login typed/entered on the CAMAS login string.
//!  This can be an email, a part of an email or an imap id.
//! returns:
//!  a mapping : maps configuration name to their values<br />
//!   usual names include domain, imapport, imapserver.
//!  an int : an error code if there is an error.
//
mapping(string:string|int)|int getimapserver(string login)
{
  CDEBUG("getimapserver from AUTH SQL\n");
  string|int imapsrv = QUERY(sql_imapserver);
  mapping res = ([ 
                 "imapserver":   imapsrv,
                 "imapport": QUERY(sql_imapport),
                ]);
  array loginsplit = login / "@";
  if(sizeof(loginsplit) == 2 && loginsplit[1])
    res += ([ "domain": loginsplit[1] ]);
  else if(stringp(getdnsname()))
    res += ([ "domain": getdnsname() ]);
  CDEBUG(sprintf("result=%O\n", res));
  return res;
}

//
//! method: mapping camas_provide()
//!  Return the capabilities of this CAMAS auth module.
//!  Used when multiple auth modules are installed in a Camas enviroment.
//! returns:
//!  A mapping with the following format :
//!  (["function":"auth" ])
//
mapping camas_provides() {
return ([ "function":"auth" ]);
}

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! When on, debug messages will be logged in Caudium's debug logfile. This information is very useful to the developers when fixing bugs.
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: emaillogin
//! Use the email address to login instead of the imap login
//!  type: TYPE_FLAG
//!  name: Login using email address
//
//! defvar: dnsname
//! The DNS domain name used for CAMAS webmail
//!  type: TYPE_STRING
//!  name: The sitewide DNS domain name
//
//! defvar: sqlserver
//!  type: TYPE_STRING
//!  name: SQL server host
//
//! defvar: strategy
//! If you defined several SQL server, this module will can use these strategies: <ul><ol><b>Failover</b>: In this mode the module will always query the first server in the list unless there is an error. In this case it will try the next server.</ol><ol><b>Load balancing</b>: The module will query the server in a random order. If a server is not available, it will retry with another random choosen one</ol></ul>
//!  type: TYPE_STRING_LIST
//!  name: Multiple server strategy
//
//! defvar: table
//! The table where to put preferences. This table will be automatically created and fields will be automatically created if new preferences are added to CAMAS core.
//!  type: TYPE_STRING
//!  name: Table name
//
//! defvar: loginquery
//! The SQL query to execute for fetching login sent to IMAP from login string entered by user. You can use the following replacements:<br /><ul><li><b>$TABLE</b>: The table name</li><li><b>$LOGIN</b>: The login entered by the user</li><li><b>$LOGIN1</b>: The first part of the login entered by the user if his login has a white space</li><li><b>$LOGIN2</b>: The second part of the login entered by the user if his login has a white space</li><li><b>$DOMAIN</b>: The domain name you enter in 'Sitewide DNS domain name'</li></ul><br />SELECT id_user as imaplogin from $TABLE where prenom='$LOGIN1' and nom='$LOGIN2'
//!  type: TYPE_STRING
//!  name: SQL Query: Login
//
//! defvar: namequery
//! The SQL query to execute for fetching name and surname from login entered by user. You can use the following replacements:<br /><ul><li><b>$TABLE</b>: The table name</li><li><b>$LOGIN</b>: The IMAP login</li><li><b>$DOMAIN</b>: The domain name you enter in 'Sitewide DNS domain name'</li></ul><br />SELECT prenom as surname, nom as name from $TABLE where id_user='$LOGIN'
//!  type: TYPE_STRING
//!  name: SQL Query: Name
//
//! defvar: emailquery
//! The SQL query to execute for fetching email from login entered by user. If empty the email will be guess from login and the domain name. You can use the following replacements: <br /><ul><li><b>$TABLE</b>: The table name</li><li><b>$LOGIN</b>: The IMAP login</li><li><b>$DOMAIN</b>: The domain name you enter in 'Sitewide DNS domain name'</li></ul><br />Example: SELECT email_from as email from $TABLE where id_user='$LOGIN'
//!  type: TYPE_STRING
//!  name: SQL Query: Email
//
//! defvar: sql_imapserver
//! The default IMAP server if Get IMAP from LDAP is not set or if no IMAP server is found
//!  type: TYPE_STRING
//!  name: IMAP Server:IMAP server
//
//! defvar: sql_imapport
//! The port of the IMAP server we connect to
//!  type: TYPE_INT
//!  name: IMAP Server:IMAP server port
//

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

