/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_logger.pike,v 1.24 2004/01/08 20:53:19 vida Exp $
 */

#include <module.h>
#include <camas/globals.h>
inherit "module";

inherit "caudiumlib";

//
//! module: CAMAS: Logger Module
//!  Logger Module for CAMAS.<br />
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_PROVIDER
//! cvs_version: $Id: camas_logger.pike,v 1.24 2004/01/08 20:53:19 vida Exp $
//

constant cvs_version = "$Id: camas_logger.pike,v 1.24 2004/01/08 20:53:19 vida Exp $";
constant module_type = MODULE_PROVIDER;
constant module_name = "CAMAS: Logger Module";
constant module_doc  = "Logger Module for CAMAS.<br />";

constant module_unique = 1;
// vida: is it always wright after the SQL stuff ?
constant thread_safe = 1;

#if constant(thread_create)
object log_lock = Thread.Mutex();
#endif

object logfile = 0;
mapping stats = ([ ]);

constant events = ({ "login", "send_mail", "smtp_errors", "imap_errors" });

string describe_what (array var, mixed path) {
  return "";
  // return (sizeof (var[VAR_VALUE]) ? replace (sprintf ("<pre>%O</pre>", var[VAR_VALUE]), "\n", "<br />") : "");
}

string describe_form_what (array var, mixed path) {
  array misc = var[VAR_MISC][3];
  string res = "<select name=\"" + path + "\" multiple size=\"" + sizeof (misc) + "\">";
  for (int i = 0; i < sizeof (misc); i++) {
    if (var[VAR_VALUE] && has_value (var[VAR_VALUE], misc[i]))
      res += "<option selected>" + misc[i] + "</option>";
    else
      res += "<option>" + misc[i] + "</option>";
  }
  res += "</select>&nbsp;<input type=\"submit\" value=\"Ok\">";
  return res;
}

array set_from_form_what (string val, int type, object o, mixed ... rest) {
  return (val / "\0");
}

void create () {
  defvar ("on", 0, "Logging", TYPE_FLAG,
          "Set to yes to turn logging on.");

  defvar ("method", "file", "Logging method", TYPE_STRING_LIST,
          "The logging method to use.",
          ({ "file", "syslog" }),
          lambda () { return !QUERY (on); });

  defvar ("logfile", "", "Log file name", TYPE_STRING,
          "The file name of the log file in the real file system."
          " If this is empty, logging is disabled.", 0,
          lambda () { return !QUERY (on) || (QUERY (method) != "file"); });

  defvar ("what", ({ }), "What to log", TYPE_CUSTOM,
          "The events to log. <br />"
          "Select from :<ul>"
          "<li>login: logins (in/out)</li>"
          "<li>send_mail: sent mails info</li>"
          "<li>smtp_errors: SMTP errors</li>"
          "<li>imap_errors: IMAP errors</li>"
          "</ul>",
          // function callbacks for the configuration interface
          ({ describe_what, describe_form_what, set_from_form_what,
             // extra argument
             events }),
          lambda () { return !QUERY (on); });
 #ifdef CAMAS_DEBUG
  // Debug
  defvar("debug",1,"Debug",TYPE_FLAG,"Debug the call / errors into Caudium "
         "error log ?");
 #endif
}

string query_provides () {
  return ("camas_logger");
}

void do_log (string sfile) {

  switch (QUERY (method)) {
  case "file":
    if (logfile) {
      string now = replace (ctime (time ()), "\n", "");
#if constant (thread_create)
      object lock = log_lock->lock ();
#endif
      mixed err = catch {
        logfile->write (now + " " + string_to_utf8(sfile));
      };
#if constant(thread_create)
      destruct (lock);
#endif
      if(err)
        report_error("error in camas_logger.pike: %s\n", describe_backtrace(err));
    }
    break;

  case "syslog":
    break;

  default:
    CDEBUG("CAMAS logger: error: unknow method");
    break;
  }
}

// log event for each users
void user_log(string event, mapping data, object sessobj)
{
  switch(event)
  {
  case "send_mail":
    sessobj->totalsentmaildata += sizeof(data["sentmaildata"]);
    sessobj->totalmsgsent += 1;
    sessobj->userstat_to += ({ data["to"]? data["to"] : "" });
    sessobj->userstat_cc += ({ data["cc"]? data["cc"] : "" });
    sessobj->userstat_subject += ({ data["subject"]? data["subject"]: "" });
    sessobj->userstat_replymessageid += ({ data["messageid"]? data["messageid"] : "" });
    sessobj->userstat_replydate += ({ data["date"]? data["date"]: "" }); 
    sessobj->userstat_date += ({ time() });
    break;
  default: do_log (sprintf ("Unknown user log event: %s\n", event));
    break;
  }
}

void log (string event, mapping data) {
  if (QUERY(method) == "file" && !logfile && sizeof (QUERY (logfile)) > 0) {
    logfile = Stdio.File ();
    if (!(logfile->open (QUERY (logfile), "wca")))
      logfile = 0;
  }

  if (QUERY(method) == "file" && !logfile)
    return;

  switch (event) {
  case "send_mail":
    stats->nomailsent++;
    stats->nobytessent += data->size;
    if (has_value (QUERY (what), "send_mail"))
      do_log (sprintf ("Sent mail from %s to %s (cc: %s), size %d\n",
                       data->from, data->to, data->cc, data->size)); // note we don't log bccs for now
    break;

  case "login":
    stats->nologins++;
    if (has_value (QUERY (what), "login"))
      do_log ("Login: " + data->login + "\n");
    break;

  case "loginfailed":
    stats->nofailedlogins++;
    if (has_value (QUERY (what), "login"))
      do_log ("Login failed: " + data->login + "\n");
    break;

  case "logout":
    stats->nologouts++;
    if (has_value (QUERY (what), "login"))
      do_log ("Logout: " + data->login + "\n");
    break;

  case "autologout":
    stats->noautologouts++;
    if (has_value (QUERY (what), "login"))
      do_log ("Auto logout: " + data->login + "\n");

    break;

  case "imap_errors":
    stats->noimapfailures++;
    if (has_value (QUERY (what), "imap_errors"))
      do_log ("IMAP Error, command: " + data->command
              + ", response: " + data->line + "\n- History:\n- "
              + replace (data->history, "\n", "\n- ") + "\n");
    break;

  case "smtpfail":
    stats->nosmtpfailures++;
    if (has_value (QUERY (what), "smtp_errors"))
      do_log ("SMTP Error, command: " + data->command
              + ", response: " + data->line + "\n");
    break;

  default:
    do_log (sprintf ("Unknown log action, event: %O, data: %O\n", event, data));
    break;
  }
}

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: on
//! Set to yes to turn logging on.
//!  type: TYPE_FLAG
//!  name: Logging
//
//! defvar: method
//! The logging method to use.
//!  type: TYPE_STRING_LIST
//!  name: Logging method
//
//! defvar: logfile
//! The file name of the log file in the real file system. If this is empty, logging is disabled.
//!  type: TYPE_STRING
//!  name: Log file name
//
//! defvar: what
//! The events to log. <br />Select from :<ul><li>login: logins (in/out)</li><li>send_mail: sent mails info</li><li>smtp_errors: SMTP errors</li><li>imap_errors: IMAP errors</li></ul>
//!  type: TYPE_CUSTOM
//!  name: What to log
//
//! defvar: debug
//! Debug the call / errors into Caudium error log ?
//!  type: TYPE_FLAG
//!  name: Debug
//
