/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * Based on IMHO  Stefan Wallstrm and Bosse Lincoln.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_imho.pike,v 1.1.2.3 2004/02/24 23:47:08 vida Exp $
 */
//
//! module: CAMAS: IMHO Tags
//!  This module gives backward IMHO compatibility for &lt;imho_..&gt;
//!  tags and containers.<br />
//!  So you can use old IMHO tags on old layouts...<br />
//!  <b>This module is automatically selected if you select "CAMAS: Main
//!  module".</b>
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_PARSER|MODULE_PROVIDER
//! cvs_version: $Id: camas_imho.pike,v 1.1.2.3 2004/02/24 23:47:08 vida Exp $
//
#include <module.h>
inherit "module";
inherit "caudiumlib";
#include <camas/globals.h>
#include <camas/msg.h>		// MSG() Language macros

constant cvs_version   = "$Id: camas_imho.pike,v 1.1.2.3 2004/02/24 23:47:08 vida Exp $";
constant module_type   = MODULE_PARSER|MODULE_PROVIDER;
constant module_name   = "CAMAS: IMHO Tags (DEPRECATED)";
constant module_doc    = "This module gives backward IMHO compatibility for "
                         "&lt;imho_..&gt; tag and containers.<br />"
                         "So you can use old IMHO tags on old layouts...";
constant module_unique = 1;
constant thread_safe = 1;

// Definition to give us an easy way to make the code and understand it :)
#define CAMAS_FEATURES  id->conf->get_provider("camas_features")

// Faster debugging
#define CAMAS_DEBUG
#ifdef CAMAS_DEBUG
# define CDEBUG(X)	if(QUERY(debug)) werror("CAMAS IMHO: "+X+"\n");
#else
# define CDEBUG(X)
#endif

void create()
{
#ifdef CAMAS_DEBUG
  defvar("debug",0,"Debug",TYPE_FLAG,"Debug the call / errors into Caudium "
         "error log ?");
#endif
  // Was in main module
  defvar("banner", "CAMAS", "Banner text", TYPE_STRING,
         "Banner text. This text is inserted on top of every page if you are "
         "using the default layout. Can be inserted using "
         "&lt;imho_banner&nbsp;/&gt; in a layout file.");
}

void start(int cnt, object conf)
{
  module_dependencies(conf,
                      ({ "tablify",	// <imho_servers>
                       }));
}

// What we provides
string query_provides ()
{
  return "camas_imho";
}

// Internal functions used only by this module

// Fonction used by tag_imho_main() tags and functions
#if 0
private string create_main (object id, int screen) {
  object camas_imhoscreens = id->conf->get_provider("camas_imhoscreens");
  if(objectp(camas_imhoscreens))
  {
    switch (screen) {

    case LOGINFAILEDIMAP:
    case LOGINFAILED:
    case LOGINPROMPT:
      return camas_imhoscreens->login(id, screen, QUERY(oldplug));
      break;

    case LOGOUT:
      return camas_imhoscreens->logout(id);
      break;

    case COMPOSE:
      return camas_imhoscreens->compose(id,feature(FEAT_ADDRESSBOOK),feature(FEAT_LDAP),sizeof(QUERY(ispell)),QUERY(ispelldict),QUERY(siteorg),feature(FEAT_MAILBOXES),feature(FEAT_SAVEMAILCOPY),feature(FEAT_DSN),feature(FEAT_ATTACHMENTS));
      break;

    case ATTACHMENTS:
      return camas_imhoscreens->attachments(id,sizeof(QUERY(uploaddir)));
      break;

    case ADDRESSBOOK:
      return camas_imhoscreens->addressbook(id,feature(FEAT_EDITADDRESSBOOK),QUERY(gabook));
      break;

    case ADDRESSBOOK2:
      return camas_imhoscreens->addressbook2(id,QUERY(gabook));
      break;

    case LDAPRESULT:
      return camas_imhoscreens->ldapresult(id,QUERY(ldapshowou));
      break;

    case LDAPRESULT2:
      return camas_imhoscreens->ldapresult2(id,QUERY(ldapshowou));
      break;

    case MAILFILTER:
      return camas_imhoscreens->mailfilter(id,feature(FEAT_EDITFILTERBOOK));
      break;

    case LDAPSEARCH:
      return camas_imhoscreens->ldapsearch(id);
      break;

    case IMPORTADDRESS:
      return camas_imhoscreens->importaddress(id);
      break;

    case EDITADDRESS:
      return camas_imhoscreens->editaddress(id,feature(FEAT_EXTENDEDABOOK));
      break;

    case EDITADDRESSFILTER:
      return camas_imhoscreens->editfilter(id);
      break;

    case MAILINDEX:
      return camas_imhoscreens->mailindex(id,QUERY(defaultmailpath),feature(FEAT_MAILBOXES),QUERY(deletemethod));
      break;

    case SEARCHMAIL:
      return camas_imhoscreens->searchmail(id);
      break;

    case FOLDERLIST:
      return camas_imhoscreens->folderlist(id,QUERY(foldersinfo));
      break;

    case MDNDIALOG:
      return camas_imhoscreens->mdndialog(id);
      break;

    case READMAIL:
      return camas_imhoscreens->readmail(id,QUERY(defaultmailpath),feature(FEAT_ADDRESSBOOK),feature(FEAT_SHOWTEXT),feature(FEAT_SHOWHTML),QUERY(deletemethod),feature(FEAT_MAILBOXES),QUERY(showhiddenheaders));
      break;

    case FILES:
      return camas_imhoscreens->files(id,QUERY(uploadquota));
      break;

    case SETUP:
      return camas_imhoscreens->setup(id,feature(FEAT_USERCANCHANGENAME),feature(FEAT_USERADDRESS),feature(FEAT_USERMAILPATH),feature(FEAT_USERLANGUAGE),QUERY(chgreplymsgprefix),feature(FEAT_USERHEADERS),feature(FEAT_SHOWHTML),feature(FEAT_SHOWTEXT),feature(FEAT_MAILBOXES),QUERY(deletemethod),feature(FEAT_USERDRAFTSFOLDER),feature(FEAT_USERSENTFOLDER),feature(FEAT_USERSENTSAVEATTACHMENTS),feature(FEAT_USERBCCCOPY),feature(FEAT_USERSETUPSHOWHIDDENHEADERS),feature(FEAT_ORGANIZATION),feature(FEAT_USERTRASHFOLDER));
      break;

    case DIALOGBOX:
      return camas_imhoscreens->dialog(id);
      break;

    case SPELLCHECK:
      return camas_imhoscreens->spellcheck(id);
      break;

    default:
      return "<font color=\"red\" size=\"+2\">Error: no such screen= " + screen + "</font>";
    }
  }
  else
    return "<font color=\"red\" size=\"+2\">Error: Screen module provider was not ready for screen "
           +screen+"...Please contact system administrator with this information "
           "</font>";
}

#endif

// Tags now

//
//! tag: imho_servers
//!  Gives a list of CAMAS Imap servers and domains served in a
//!  table. (need the tablify support).
//
string tag_imho_servers(string tag_name, mapping arg, object id, object file)
{
  object camas_module = CAMAS_MODULE;

  if(objectp(camas_module))
  {
    string tab = "<tablify nicer>";
    CDEBUG(tag_name);

    tab += "Domain\tServer\n";
    foreach (camas_module->imap_servers(), array arr) {
      tab+= arr[0] + "\t" + arr[1] + "\n";
    }
    tab += "</tablify>";
    return tab;

  }
  return "";
}

//
//! tag: imho_banner
//!  Return the contents of the CAMAS's defvar "banner".
//
string tag_imho_banner(string tag_name, mapping arg, object id, object file)
{
  CDEBUG(tag_name);
  return QUERY(banner);
}

//
//! tag: imho_title
//!  Return the symbolic name of the current screen eg mailindex, compose, etc..
//
string tag_imho_title(string tag_name, mapping arg, object id, object file)
{
  if (CSESSION) {
    CDEBUG(tag_name);

    if (screennames[CSESSION->status])
      return (MSG(screennames[CSESSION->status][1]));
    else
      return("");
  }
}

//
//! tag: imho_name
//!  Resturn the current username on the current session
//! attribute: first
//!  Will give the first part of the name before a blank space
//! attribute: second
//!  Will give the second part of the name after the blank space...
//
string tag_imho_name(string tag_name, mapping arg, object id, object file)
{
  if (CSESSION)
  {
    CDEBUG(tag_name);
    array username = CSESSION->name/" ";

    if (arg->first)
    {
      return username[0];
    }
    else
    {
      if (arg->second) {
        if (sizeof(username) > 1)
          return username[1];
        else
          return "";
      }
      return CSESSION->name;
    }
  }
  else
    return "";

}

//
//! tag: imho_frame
//!  Wrapper to HTML &lt;frame ... &gt; tag for CAMAS framed layout. It use
//!  internal data to set the src= arg with correct CAMAS arguments.
//! attribute: name
//!  Internal CAMAS' symbolic frame name used to generate HTML src parameter
//!  for standaart &lt;frame src=..&gt;
//
string tag_imho_frame(string tag_name, mapping arg, object id, object file)
{
  if (CSESSION) {
    CDEBUG(tag_name);
    string src = CSESSION->nexturl+arg->name + "-" + CAMAS.Tools.rand_string(20, id->supports->ssl_faulty);
    string ret = "<frame src=\"" + src + "\" ";
    foreach ((indices (arg) - ({ "/" })), string a)
    ret += " " + a + "=" + HTML_ENCODE_TAG_VALUE (arg[a]) + "";
    ret += ">";
    return ret;
  }
  return "";
}

//
//! tag: imho_main
//!  Main CAMAS tag for (allmost) all functions.
//! see_also: imho_mailindex
//!  Show the index of all mails for one mailbox
//! see_also: imho_compose
//!  Compose a mail
//! see_also: imho_readmail
//!  Read a mail
//! see_also: imho_folderlist
//!  List the folders
//! see_also: imho_mailfilter
//!  Access to mail filters
//! see_also: imho_files
//!  Access to files
//! see_also: imho_addressbook
//!  Access to the address book
//! see_also: imho_setup
//!  Setup the CAMAS preferences
//! attribute: thcolor
//!  To be documented
//! attribute: thbgcolor
//!  To be documented
//! attribute: tdbgcolor
//!  To be documented
//! attribute: checkcolor
//!  To be documented
//! attribute: checksize
//!  Size of the check box
//! attribute: arrowsize
//!  Size of the arrow
//! attribute: check
//!  Draw a checkbox (to be verified)
//! attribute: arrowup
//!  Draw a up arrow
//! attribute: arrowdown
//!  Draw a down arrow
//! attribute: folder
//!  To be documented
//! attribute: notopbuttons
//!  This will desactivate the top buttons when using &lt;imho_main&gt;
//!  or &lt;imho_mailindex&gt; on mail list.
//! attribute: nobottombuttons
//!  This will desactivate the bottom buttons when using &lt;imho_main&gt;
//!  or &lt;imho_mailindex&gt; on mail list.
//!
string tag_imho_main(string tag_name, mapping arg, object id, object file)
{
  object camas_module = CAMAS_MODULE;
  if (CSESSION && objectp(camas_module)) {
    string thcolor=arg->thcolor || "#ffffff";
    string thbgcolor=arg->thbgcolor || "#000070";
    string tdbgcolor=arg->tdbgcolor || "#fcfce0";
    string checkcolor=arg->checkcolor || "#000000";
    int checksize=(int)arg->checksize || 15;
    int arrowsize=(int)arg->arrowsize || 15;
    string check=arg->check||"";
    string arrowup=arg->arrowup||"";
    string arrowdown=arg->arrowdown||"";
    string folder=arg->folder||"";
    CAMAS.Tools.setup_state(id,arg);

    int screen = CSESSION->status;
    if (screen != DIALOGBOX) {
      switch(tag_name) {
      case "imho_mailindex":
        screen = MAILINDEX;
        break;
      case "imho_compose":
        screen = COMPOSE;
        break;
      case "imho_readmail":
        screen = READMAIL;
        break;
      case "imho_folderlist":
        screen = FOLDERLIST;
        break;
      case "imho_mailfilter":
        screen = MAILFILTER;
        break;
      case "imho_files":
        screen = FILES;
        break;
      case "imho_addressbook":
        screen = ADDRESSBOOK;
        break;
      case "imho_setup":
        screen = SETUP;
        break;
      }
    }

    string main = camas_module->create_main (id, screen);

    main = replace (main,
                    ({ "&imho_folder;", "&imho_thcolor;", "&imho_thbgcolor;",
                       "&imho_tdbgcolor;", "&imhocheckfg;", "&imhochecksrc;",
                       "&imhoarrowupsrc;", "&imhoarrowdownsrc;", "&imhochecksize;",
                       "&imhoarrowsize;" }),
                    ({ (sizeof (folder) ? folder : "internal-gopher-menu"), thcolor, thbgcolor,
                       tdbgcolor, checkcolor, check,
                       arrowup, arrowdown, (string)(checksize * 4), (string)(arrowsize * 4) }) );
    return main;
  }
  return "";
}

//
//! tag: imho_about
//!  Show version of CAMAS and eventualy a link to camas website.
//! attribute: nolink
//!  Do not add a link to camas website
//
string tag_imho_about(string tag_name, mapping arg, object id, object file)
{
  object camas_module = CAMAS_MODULE;
  if (objectp(camas_module))
  {
    CDEBUG(tag_name);
    return (arg->nolink ? "" : "<a target=\"_top\" href=\""
            + camas_module->server_url (id)
            + camas_module->QUERY(location) + "0/about\">" )
           + "CAMAS v" + camas_module->camas_version
           + (arg->nolink ? "" : "</a>");
  }
  return "";
}

//
//! tag: imho_buttons
//!  Returns the standart CAMAS buttons (eg compose mail, mail index,
//!  folderlist, files, address book, filter book, setup and logout).
//! attribute: bg
//!  Backgroup color of the table (to be verified).
//! attribute: fg
//!  Color of the text
//! attribute: magic_fg
//!  Color of the used when gtext is used.
//! attribute: scale
//!  Scale of the gtext image generated (see gtext tag).
//! attribute: xsize
//!  To be documented
//! attribute: breakline
//!  Add a &lt;br /&gt; after each buttons...
//
string tag_imho_buttons(string tag_name, mapping arg, object id, object file)
{
  object camas_module = CAMAS_MODULE;
  object camas_features = CAMAS_FEATURES;
  if (CSESSION
      && objectp (camas_module) && objectp(camas_features)) {
    CDEBUG(tag_name);
    string bg=arg->bg || "#c0c0c0";
    string fg=arg->fg || "#000000";
    string magic_fg=arg->magic_fg || "#303030";
    string scale=arg->scale || "0.4";
    string ret = "", folder="";
    CAMAS.Tools.setup_state(id,arg);

    // --oliv3 FIXME what's this ? the second branch of the if()
    // will never be reached as (id->misc->imho && id->misc->imho->sessobj)
    // will always be defined here.
    //if (id->misc->imho && id->misc->imho->sessobj) {
      folder = CAMAS.FolderTools.translate_frommbox(id, CSESSION->mailbox);
    //}
    /*
    else
      folder = MSG(M_INBOX);
    */

    string args="";
    if (arg->xsize)
      args += " xsize="+arg->xsize+" ";

    ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->camas->nexttarget+"?actioncompose=1\" target=\""+id->misc->camas->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+MSG(M_NEWMAIL)+"</imho_text>"+(arg->breakline?"<br />":"");
    ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->camas->nexttarget+"?actionindex=1\" target=\""+id->misc->camas->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+(camas_features->feature(FEAT_MAILBOXES)?(arg->noupdate?MSG(M_CURRMAILBOX):  replace(MSGA(M_MAILBOX,({"\0"})),"\0",folder)  ):MSG(M_INBOX))+"</imho_text>"+(arg->breakline?"<br />":"");
    if(camas_features->feature(FEAT_MAILBOXES))
      ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->camas->nexttarget+"?actionfolderlist=1\" target=\""+id->misc->camas->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+MSG(M_MAILBOXES)+"</imho_text>"+(arg->breakline?"<br />":"");
    if(camas_features->feature(FEAT_ATTACHMENTS) && sizeof(camas_module->QUERY(uploaddir)))
      ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->camas->nexttarget+"?actionfiles=1\" target=\""+id->misc->camas->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+MSG(M_FILES)+"</imho_text>"+(arg->breakline?"<br />":"");
    if(camas_features->feature(FEAT_ADDRESSBOOK))
      ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->camas->nexttarget+"?actionaddressbook=1\" target=\""+id->misc->camas->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+MSG(M_ADDRESSBOOKTITLE)+"</imho_text>"+(arg->breakline?"<br />":"");
    if(camas_features->feature(FEAT_MAILFILTER))
      ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->camas->nexttarget+"?actionfilterbook=1\" target=\""+id->misc->camas->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+MSG(M_FILTERBOOKTITLE)+"</imho_text>"+(arg->breakline?"<br />":"");
    if(camas_features->feature(FEAT_USEREDITSETUP))
      ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->camas->nexttarget+"?actionsetup=1\" target=\""+id->misc->camas->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+MSG(M_PREFS)+"</imho_text>"+(arg->breakline?"<br />":"");
    ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->camas->nexttarget+"?actionsetup=1\" target=\""+id->misc->camas->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+MSG(M_PREFS)+"</imho_text>"+(arg->breakline?"<br />":"");

    if(camas_module->QUERY(logout))
    {
      string logout_url = CSESSION->nexturl + "_top" + "-" + CAMAS.Tools.rand_string(20, id->supports->ssl_faulty) + "?actionlogout=1";
      ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\"";
      ret += ((camas_module->QUERY(jslogout) && id->supports->javascript) ?
              ("javascript:CAMAS_Confirm=0;top.location='" + logout_url + "';") : logout_url);
      ret += "\" target=\"_top\" magic magic_fg=" + magic_fg + " bevel=4 fg=" + fg;
      ret += " xspacing=10 yspacing=5>" + MSG (M_LOGOUT) + "</imho_text>";
      ret += (arg->breakline ? "<br />" : "");
    }
    return ret;
  }
  return "";
}

//
//! tag: imho_string
//!  Returns the string from the current language translation
//! attribute: no
//!  Number of the message (see camas/language.h file)
//
string tag_imho_string(string tag_name, mapping arg, object id, object file)
{
  if (CSESSION && arg["no"]) {
    CDEBUG(tag_name);
    return(MSG((int)arg["no"]));
  }
  return "";
}

//
//! tag: imho_submit
//!  To be documented (err... what's this tag for ?)
//! attribute: type
//!  To be documented
//! attribute: submit
//!  To be documented
//! attribute: class
//!  To be documented
//! attribute: button
//!  To be documented
//! attribute: value
//!  To be documented
//
string tag_imho_submit(string tag_name, mapping arg, object id, object file)
{
  if (CSESSION) {
    CDEBUG(tag_name);
    mapping args = ([ "type": "submit", "class": "button", "value": arg->value ]);
    // Allow all arguments. Lets e.g. javascript through.
    foreach ((indices (arg) - ({ "/" })), string a) {
      if (!(args[a]))
        args[a] = arg[a];
    }
    string tag = CAMAS.Tools.make_tag ("input", args);
    return tag[0..sizeof (tag)-2] + " />";
  }
  return "";
}

//
//! tag: imho_mailbox
//!  Returns the name of the current mailbox
//
string tag_imho_mailbox(string tag_name, mapping arg, object id, object file)
{
  if (CSESSION) 
  {
    CDEBUG(tag_name);
    string mbox = CAMAS.FolderTools.translate_frommbox(id, CSESSION->mailbox);
    return(HTML_ENCODE_STRING(mbox));
  }
  return "";
}

//
//! tag: imho_activemailboxbuttons
//!  Insert a list of user defined inboxes
//! attribute: bg
//!  Background color of the cells
//! attribute: fg
//!  Foreground color of the cells
//! attribute: magic_fg
//!  Magic color when gtext is used
//! attribute: scale
//!  Gtext scale
//! attribute: xsize
//!  To be documented
//! attribute: noupdate
//!  This set of button will not be updated for every action, so it don't
//!  display e.g. the name of the current mailbox.
string tag_imho_activemailboxbuttons(string tag_name, mapping arg, object id, object file)
{
  if (CSESSION) {
    CDEBUG(tag_name);
    string bg=arg->bg || "#c0c0c0";
    string fg=arg->fg || "#803000";
    string magic_fg=arg->magic_fg || "#402000";
    string scale=arg->scale || "0.3";
    string ret = "", folder;
    array mbox_parts;
    CAMAS.Tools.setup_state(id,arg);

    string args="";
    if (arg->xsize)
      args += " xsize="+arg->xsize+" ";

    string name = MSG(M_INBOX);
    ret += "<imho_text bg="+bg+args+" scale="+scale+" href=\""+id->misc->camas->nexttarget+"?actionindex=1&mbox=INBOX\" target=\""+id->misc->camas->target+"\" magic magic_fg="+magic_fg+" bevel=4 fg="+fg+" xspacing=10 yspacing=5>"+name+(arg->noupdate?"":(" "+CSESSION->cache["INBOX"]->messages))+"</imho_text>"+(arg->breakline?"<br />":"");
    return ret;
  }
  return "";
}

//
//! tag: imho_url
//!  Return the url that is used to reach internal objects like mailboxes...
//! attribute: href
//!  To be documented
//! attribute: mbox
//!  Give the link to reach mailbox named by this attribute
//! attribute: target
//!  Send it to framed named by this target.
string tag_imho_url(string tag_name, mapping arg, object id, object file)
{
  if (CSESSION) {
    CDEBUG(tag_name);
    string ret = "", action="", mbox="";
    string tag = "";
    string target = "";
    CAMAS.Tools.setup_state(id,arg);
    int mboxno = -1;
    foreach(indices(arg), string a) {
      switch(a) {
      case "href":
        action = arg[a];
        break;
      case "mbox":
	mboxno = CAMAS.IMHOTools.find_mailbox (CSESSION, arg[a]);
        if (mboxno == -1)
          mboxno = 0;
        break;
      case "target":
        break;
      case "/": // avoid: <imho_url href=\"checkactivemailboxes\" /> ==> / is not an attribute !
        break;
      default:
        ret += "&"+a+"="+arg[a];
        break;
      }
    }
    ret=id->misc->camas->nexttarget+"?action"+action+"=1"+ret;
    if (mboxno >= 0)
      ret+="&mbox="+HTTP_ENCODE_URL(CSESSION->mailboxes[mboxno][MB_FOLDERNAME_IDX]);
    return(ret);
  }
  return "";

}

//
//! tag: imho_countnewmails
//!  Return the number of new/unread mails in the current inbox or
//!  &lt;FALSE&gt; if the is NO new mail at all in the current mailbox
//!  If attribute is used then It will send a the correct string for
//!  each states.
//! attribute: nomessages
//!  The string given will be sent when there is no new message in the
//!  current mailbox
//! attribute: onemessage
//!  The string given will be sent when there is ONE new message in the
//!  current mailbox
//! attribute: message
//!  The string given will be sent when there is more than one new
//!  message in the current mailbox.
//! attribute: prefix
//!  Is the string to insert before the number of messages
//! attribute: prehave
//!  Is the string to insert before the number of message (backward
//!  backward compatibility with IMHO 0.97 templates).
//
string tag_imho_countnewmails(string tag_name, mapping arg, object id, object file)
{
  if (CSESSION) {
    CDEBUG(tag_name);
    mapping email;
    int cpt=0;
    int nbmessages=sizeof(CSESSION->mails);
    if (nbmessages) {
      foreach(CSESSION->mails, email)
      {
        if (sizeof(email->imap->FLAGS) == 0) cpt++;
        else
          if (search(email->imap->FLAGS,"\\Seen") == -1) cpt++;
      }
      nbmessages=cpt;
    }

    if (arg->nomessages && arg->onemessage && arg->messages && (arg->prefix||arg->prehave))
    {
      if (nbmessages==0)
        return arg->nomessages;

      // So there is at least one message here ;-)
      if (nbmessages==1)
        return sprintf("%s %d %s",(arg->prefix||arg->prehave), nbmessages, arg->onemessage);

      // So there is more than one message ;-)
      return sprintf("%s %d %s",(arg->prefix||arg->prehave), nbmessages, arg->messages);
    }
    else
    {
      return sprintf("%d%s", nbmessages,(nbmessages==0)?"<FALSE>":"<TRUE>");
    }
  }
  else
    return "";
}

//
//! tag: imho_address
//!  Returns the email address from preferences or the domain name from
//!  email address saved/entered in preferences
//! attribute: domain
//!  Give only the domain name
//
string tag_imho_address(string tag_name, mapping arg, object id, object file)
{
  if (CSESSION) {
    CDEBUG(tag_name);
    if (arg->domain)
      return sprintf("%s",((CSESSION->address)/"@")[1]);
    else
      return sprintf("%s",CSESSION->address);
  }
  else
    return "";
}

//
//! tag: imho_screen
//!  Will give the name of the current screen. Usefull for a help system
//!  or "unknown" when the screen name is not known...
//
string tag_imho_screen(string tag_name, mapping arg, object id, object file)
{
  if (CSESSION)
  {
    CDEBUG(tag_name);
    string screen="unknown";

    if (screennames[CSESSION->status])
      screen=screennames[CSESSION->status][0];
    return screen;
  }
  else
    return "";
}

//
//! tag: imho_dumpid
//!  Will return the mapping ID in human readable form. Usefull for debug
//
string tag_imho_dumpid(string tag_name, mapping arg, object id, object file)
{
  CDEBUG(tag_name);
  return (sprintf("<pre>Id:%O\n</pre>",mkmapping(indices(id),values(id))));
}

int get_totalsize (object id) {
  return (CSESSION ? (CSESSION->allfolderssize >> SIZE_SHIFT): 0);
}

//
//! tag: imho_totalsize
//!  Return the totalsize of all folders in kB
//
string tag_imho_totalsize(string tag_name, mapping arg, object id, object file)
{
  CDEBUG(tag_name);
  int size = get_totalsize (id);
  return (size) ? (string)size : "";
}

//
//! tag: imho_use
//!  Return the current usage in percent of the diskquota the user has
//
string tag_imho_use (string tag_name, mapping arg, object id, object file) {
  return "<!-- Used percentage is not available in IMHO Screens in this version of Camas, please upgrade to Camas tags --> -";
}


//
//! tag: imho_doctype
//!  Return the doctype used
//
string tag_imho_doctype(string tag_name, mapping arg, object id, object file)
{
  CDEBUG(tag_name);
  return "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" \"http://www.w3.org/TR/html4/loose.dtd\">\n";
}

//
//! tag: imho_logout
//!  This destroy and leave the camas session.<br />
//!  Use this tag in logout screen page in your templates.<br />
//!  This tags also set the name of the screen to "logout".
//
string tag_imho_logout (string tag_name, mapping args, object id, object file) {
  object camas_module = CAMAS_MODULE;

  if (CSESSION && objectp(camas_module)) {
    CDEBUG(tag_name);
    CSESSION->screen="logout";
    CSESSION->title=MSG(M_LOGGEDOUT);
    CAMAS.Tools.destroy_csession(id);
  }
  return "";
}

//
//! tag: imho_quota
//!  Return the diskquota the user can use.
//
string tag_imho_quota (string tag_name, mapping arg, object id, object file)
{
  return "<!-- Quotas are not available in IMHO Screens in this version of Camas, please upgrade to Camas tags --> -";
}

//
//! tag: imho_jslogout
//!  Little javascript devil to autoclose the CAMAS session when the
//!  user close the brower window :))
//
string tag_imho_jslogout (string tag_name, mapping arg, object id, object file) {
  object camas_module = CAMAS_MODULE;
  object camas_features = CAMAS_FEATURES;

  if (CSESSION && objectp(camas_module) && objectp(camas_features))
    CDEBUG(tag_name);
  {
    string res = "";
    if (id->supports->javascript && camas_features->QUERY (jslogout)) {
      res = "<script language=\"JavaScript\" type=\"text/javascript\"><!--\n";
      res += "var CAMAS_Width;\nvar CAMAS_Height;\nvar CAMAS_Confirm=1;\n\n";
      res += "function popup () {\n";
      res += "  if (CAMAS_Confirm) {\n";
      res += "    if (confirm ('" + MSG (M_ENDSESSION) + "')) {\n";

      string logout_url = "http://" + id->host
                          + (sizeof (id->prestate) ? ("/(" + indices (id->prestate) * "," + ")") : "")
                          + camas_module->QUERY (location) + "_top?actionlogout=1";
      if (QUERY (debug))
        write("js_logout: onunload () => '" + logout_url + "');\n");
      res += "      window.open ('" + logout_url + "');\n";

      res += "    }\n";
      res += "    else {\n";
      res += "      day = new Date ();\n";
      res += "      id = day.getTime ();\n";
      res += "      window.open (this.location, 'CAMAS' + id, 'toolbar=1,scrollbars=1,location=1,";
      res += "status=1,menubar=1,resizable=1,personalbar=1,width=CAMAS_Width,height=CAMAS_Height');\n";
      res += "    }\n";
      res += "  }\n";
      res += " } \n\n";


      res += "function set_size () {\n";
      if (id->supports->msie) {
        res += "  CAMAS_Width  = document.body.offsetWidth;\n";
        res += "  CAMAS_Height = document.body.offsetHeight;\n";
      }
      else {
        res += "  CAMAS_Width  = window.outerWidth;\n";
        res += "  CAMAS_Height = window.outerHeight;\n";
      }
      res += " }\n\n";

      res += "window.onload = set_size;\n";
      res += "window.onunload = popup;\n\n";

      res += "//-->\n</script>\n";
    }
    return res;
  }
  return "";
}

//
//! tag: jsnotify
//!  RXML code to handle notify window
//
string tag_imho_jsnotify (string tag_name, mapping arg, object id, object file) {
  CDEBUG(tag_name);
  string res = "";

  // Why use the RXML parser since we have there the whole CAMAS variables
  // to handle this kind of replacements ?
  // --Xavier
  if (id->misc->_xml_parser) {
    res += "<cset preparse=\"\" scope=\"var\" variable=\"url\"><imho_url href=\"checkactivemailboxes\" /></cset>\n";
    res += "<script language=\"JavaScript\" type=\"text/javascript\">\n";
    res += "  window.setTimeout('window.location=\"&var.url;\";',60000);\n";
    res += "</script>\n";
  }
  else {
    res += "<set variable=\"url\" eval=\"<imho_url href=\"checkactivemailboxes\" />\">\n";
    res += "<formoutput quote=\"$\">\n";
    res += "<script language=\"JavaScript\" type=\"text/javascript\">\n";
    res += "  window.setTimeout('window.location=\"$url:quote=none$\";',60000);\n";
    res += "</script>\n";
    res += "</formoutput>\n";
  }

  return res;
}


// Tags that this module provides
mapping query_tag_callers()
{
  return([
         "imho_servers":tag_imho_servers,
         "imho_banner":tag_imho_banner,
         "imho_title":tag_imho_title,
         "imho_name":tag_imho_name,
         "imho_frame":tag_imho_frame,
         "imho_about":tag_imho_about,
           // imho main la !
         "imho_main":tag_imho_main,
         "imho_buttons":tag_imho_buttons,
         "imho_mailindex":tag_imho_main,
         "imho_compose":tag_imho_main,
         "imho_readmail":tag_imho_main,
         "imho_folderlist":tag_imho_main,
         "imho_files":tag_imho_main,
         "imho_addressbook":tag_imho_main,
         "imho_mailfilter":tag_imho_main,
         "imho_setup":tag_imho_main,
           // imho main fini
         "imho_string":tag_imho_string,
         "imho_submit":tag_imho_submit,
         "imho_mailbox":tag_imho_mailbox,
         "imho_activemailboxbuttons":tag_imho_activemailboxbuttons,
         "imho_url":tag_imho_url,
         "imho_countnewmails":tag_imho_countnewmails,
         "imho_address":tag_imho_address,
         "imho_screen":tag_imho_screen,
         "imho_dumpid":tag_imho_dumpid,
         "imho_totalsize":tag_imho_totalsize,
         "imho_use":tag_imho_use,
         "imho_doctype":tag_imho_doctype,
         "imho_emailaddress":tag_imho_address,
         "imho_href":tag_imho_url,
         "imho_username":tag_imho_name,
         "imho_logout": tag_imho_logout,
         "imho_quota": tag_imho_quota,
         "imho_jslogout": tag_imho_jslogout,
         "imho_jsnotify": tag_imho_jsnotify,
         ]);
}

// Containers now

//
//! container: imho_text
//!  Container used intead of &lt;gtext&gt; and make normal text
//!  even we don't use gtext. Font face and foreground color will
//!  be preserved
//! attribute: href
//!  like a href=
//! attribute: fg
//!  foreground color
// todo: suppress the code remaining in camas.pike 'cause of imho_kludge...
//
string cont_imho_text(string tag_name, mapping arg, string content, object id)
{
  CDEBUG(tag_name);
  string ret = "";
  if (CSESSION) {
    if (arg->href) {
      ret += "<a";
      foreach(indices(arg), string a)
      ret += " "+a+"=\""+arg[a]+"\"";
      ret += ">";
    }
    if (arg->fg || CSESSION->face || CSESSION->size) {
      ret += "<font ";
      if (arg->fg)
        ret += " color="+arg->fg;
      if (CSESSION->face)
        ret += " face=\""+CSESSION->face+"\"";
      // **XB**
      if (CSESSION->size)
        ret += " size=\"" + CSESSION->size + "\"";
      ret += ">";
      if (arg->href)
        ret += "[&nbsp;"+replace(content," ","&nbsp;")+"&nbsp;]";
      else
        ret += content;
      ret += "</font>";
    }
    else {
      if (arg->href)
        ret += "[&nbsp;"+replace(content," ","&nbsp;")+"&nbsp;]";
      else
        ret += content;
    }
    if (arg->href)
      ret += "</a> ";
  } else {
    ret += "<gtext";
    foreach(indices(arg), string a) {
      if (a == "nfont" && CSESSION->face)
        ret += " "+a+"=\""+CSESSION->face+"\"";
      else
        ret += " "+a+"="+HTML_ENCODE_TAG_VALUE(arg[a])+"";
    }
    ret += ">"+content+"</gtext>";
  }
  return ret;
}

//
//! container: imho_gtext
//!  Container used for &lt;a&gt; with links to CAMAS actions
//! see also: imho_a
//! attribute: bg
//!  The background color
//! attribute: fg
//!  The foreground color
//! attribute: breakline
//!  Insert a &lt;br&gt; after each buttons
//! attribute: magic_fg
//!  The color of the highlighted button
//! attribute: noupdate
//!  This set of button will not be updated for every action, so don't
//!  don't display e.g. the name of the current mailbox.
//! attribute: scale
//!  The gtext scale.
//! attribute: xsize
//!  Specify the xsize of all buttons (usefull if placed in a column)
//
string cont_imho_href(string tag_name, mapping arg, string content, object id)
{
  if (CSESSION) {
    CDEBUG(tag_name);
    string ret = "", action="", mbox="", redirect_url="", target="", tag="";
    sscanf(tag_name,"imho_%s", tag);
    ret += "<"+tag;
    CAMAS.Tools.setup_state(id,arg);
    int mboxno = -1;
    foreach(indices(arg), string a) {
      switch(a) {
      case "href":
        action = arg[a];
        break;
      case "redirect_url":
        redirect_url = arg[a];
        break;
      case "mbox":
	mboxno = CAMAS.IMHOTools.find_mailbox (CSESSION, arg[a]);
        if (mboxno == -1)
          mboxno = 0;
        break;
      default:
        ret+=" "+a+"="+HTML_ENCODE_TAG_VALUE(arg[a])+"";
      }
    }
    ret+=" href=\""+id->misc->camas->nexttarget+"?action"+action+"=1";
    if (sizeof(redirect_url) > 1) ret += "&redirect_url="+redirect_url;
    if (mboxno >= 0)
      ret+="&mbox="+HTTP_ENCODE_URL(CSESSION->mailboxes[mboxno][MB_FOLDERNAME_IDX]);
    ret+="\">"+content+"</"+tag+">";
    return(ret);
  }
  return "";

}

//
//! container: imho_buttonoutput
//!  To be documented
//! attribute: buttons
//!  To be documented
//! attribute: noupdate
//!  To be documented
//! bugs:
//!  Fix this doc...
//
string cont_imho_buttonoutput(string tag_name, mapping arg, string content, object id)
{
  object camas_module = CAMAS_MODULE;
  object camas_features = CAMAS_FEATURES;

  if (CSESSION
      && objectp (camas_module) && objectp (camas_features)) {
    CDEBUG(tag_name);
    array vars = ({ });
    string folder = "";
    CAMAS.Tools.setup_state (id, arg);

    // same as above: the test will always be true below
    //if (id->misc->imho && id->misc->imho->sessobj) {
    //if (CSESSION) {
    folder = CAMAS.FolderTools.translate_frommbox(id, CSESSION->mailbox);
    //}
    /*
    else
      folder = MSG(M_INBOX);
    */

    //oliv3 FIXME next line: (args->button || "main") == "inboxes" ???
    if (lower_case (arg->buttons || "main") == "inboxes") {
      string name = MSG(M_INBOX);
      vars += ({
	([
	  "button-url": id->misc->camas->nexttarget + "?actionindex=1&mbox=INBOX",
	  "button-target": id->misc->camas->target,
	  "button-text": name + (arg->noupdate ? "" : (" " + CSESSION->cache["INBOX"]->messages))
	]) });
    }
    else {
      vars += ({
	([
	  "button-name":"compose",
	  "button-url":id->misc->camas->nexttarget+"?actioncompose=1",
	  "button-target":id->misc->camas->target,
	  "button-text":MSG(M_NEWMAIL)
	]),
	
	([
	  "button-name":"mailindex",
	  "button-url":id->misc->camas->nexttarget+"?actionindex=1",
	  "button-target":id->misc->camas->target,
	  "button-text":(camas_features->feature (FEAT_MAILBOXES)?(arg->noupdate?MSG(M_CURRMAILBOX): replace(MSGA(M_MAILBOX,({"\0"})),"\0",folder)  ):"INBOX")
	])
      });
      
      if(camas_features->feature(FEAT_MAILBOXES))
        vars += ({
	  ([
	    "button-name":"mailboxes",
	    "button-url":id->misc->camas->nexttarget+"?actionfolderlist=1",
	    "button-target":id->misc->camas->target,
	    "button-text":MSG(M_MAILBOXES)
	  ])
	});
      
      if(camas_features->feature(FEAT_ATTACHMENTS) && sizeof(camas_module->QUERY(uploaddir)))
        vars += ({
	  ([
	    "button-name":"files",
	    "button-url":id->misc->camas->nexttarget+"?actionfiles=1",
	    "button-target":id->misc->camas->target,
	    "button-text":MSG(M_FILES)
	  ])
	});
      
      if(camas_features->feature(FEAT_ADDRESSBOOK))
        vars += ({
	  ([
	    "button-name":"addressbook",
	    "button-url":id->misc->camas->nexttarget+"?actionaddressbook=1",
	    "button-target":id->misc->camas->target,
	    "button-text":MSG(M_ADDRESSBOOKTITLE)
	  ])
	});
      
      if(camas_features->feature(FEAT_MAILFILTER))
        vars += ({
	  ([
	    "button-name":"mailfilters",
	    "button-url":id->misc->camas->nexttarget+"?actionfilterbook=1",
	    "button-target":id->misc->camas->target,
	    "button-text":MSG(M_FILTERBOOKTITLE)
	  ])
	});

      if(camas_features->feature(FEAT_USEREDITSETUP))
        vars += ({
	  ([
	    "button-name":"preferences",
	    "button-url":id->misc->camas->nexttarget+"?actionsetup=1",
	    "button-target":id->misc->camas->target,
	    "button-text":MSG(M_PREFS)
	  ])
	});
      if(camas_features->QUERY(logout))
      {
        string logout_url = CSESSION->nexturl + "_top" + "-"
	  + CAMAS.Tools.rand_string(20, id->supports->ssl_faulty) + "?actionlogout=1";
        vars += ({
	  ([
	    "button-name":"logout",
	    "button-url":
	    ((camas_features->QUERY (jslogout) && id->supports->javascript) ?
	     ("javascript:CAMAS_Confirm=0;top.location='" + logout_url + "';") : logout_url),
	    "button-target":"_top",
	    "button-text":MSG(M_LOGOUT)
	  ])
	});
      }
    }

    return do_output_tag (arg, vars, content, id);
  }
  return "";
}

//
//! container: imho_jump
//!  Custom CAMAS logout button
//! attribute: redirect
//!  The url to be redirected to when quitting CAMAS
//! example: rxml
//!  To add an exit button to Caudium website
//!   {imho_jump redirect="http://caudium.net"}Exit to Caudium Website{/imho_jump}
//
string cont_imho_jump(string tag_name, mapping arg, string content, object id)
{
  CDEBUG(tag_name);
  object camas_module = CAMAS_MODULE;
  if(objectp(camas_module))
  {
    if (!arg->redirect)
      return "";
    else
    {
      string redir = "<a target=\"_top\" href=\"" ;
      redir += camas_module->server_url (id) ;
      redir += camas_module->QUERY (location);
      redir += "0/jump/" + HTTP_ENCODE_URL (arg->redirect) + "\">";
      redir += content + "</a>";
      return redir;
    }
  }
  return "";
}


// Containers that this module provides
mapping query_container_callers() {
  return ([
          "imho_text": cont_imho_text,
          "imho_a":cont_imho_href,
          "imho_gtext":cont_imho_href,
          "imho_buttonoutput":cont_imho_buttonoutput,
          "imho_jump":cont_imho_jump
          ]);
}

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! Debug the call / errors into Caudium error log ?
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: banner
//! Banner text. This text is inserted on top of every page if you are using the default layout. Can be inserted using &lt;imho_banner&#xa0;/&gt; in a layout file.
//!  type: TYPE_STRING
//!  name: Banner text
//
