(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** Various editors for Cameleon. *)

open Cam_plug

let exec com f =
  let s = com^" "^(Filename.quote f.f_name)^" &" in
  let n = Sys.command s in
  if n <> 0 then
    raise (Failure ("Error while executing the command "^s))
  else
    ();
  None

let editor com ?char f = (exec com f : Cam_plug.editor_app option)

(** Launch [gimp] on the given file. 
   @editor gimp
*)
let gimp = ()

(** Launch [gv] on the given file. 
   @editor gv
*)
let gv = ()

(** Launch [xfig] on the given file. 
   @editor xfig
*)
let xfig = ()

(** Launch [glade] on the given file. 
   @editor glade
*)
let glade = ()

(** Launch [epeire] (the one installed with Cameleon) 
   on the given Epeire configuration file. 
   @editor epeire
*)
let epeire_editor file = 
  let s = 
    "cd "^(Filename.dirname file.f_name)^" ; "^
    (Filename.concat Cam_installation.bin_dir "epeire")^" -c "^(Filename.quote file.f_name)^" &" in
  let n = Sys.command s in
  if n <> 0 then
    raise (Failure ("Error while executing the command "^s))
  else
    ();
  None

let epeire ?char f = (epeire_editor f : Cam_plug.editor_app option)

(** Launch a [gvim] serveur named "CAMELEON_GVIM" or connect to the existing one, if any.
   @editor gvim
   @author Alan Schmitt
*)
let gvim = ()

let simple_commands = ["gimp" ; "xfig" ; "gv" ; "glade"]

let _ = List.iter
    (fun com -> Cam_plug.add_editor com (editor com))
    simple_commands

let _ = Cam_plug.add_editor "epeire" epeire
let _ = Cam_plug.add_editor "gvim" (editor "gvim --servername CAMELEON_GVIM --remote-silent")

