(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** The class and functions to make the user edit the properties of a 
   gui element. *)

open Zog_types

(** Create a combo with given options. *)
let create_combo prop strings val_in_list can_empty =
  let c = GEdit.combo
      ~popdown_strings: strings
      ~use_arrows: `DEFAULT
      ~value_in_list: val_in_list
      ~ok_if_empty: can_empty
      ()
  in
  let we = c#entry in
  we#set_text prop.prop_value ;
  (c#coerce, c#entry)
  

(** Create a widget to edit the given property. 
   Return also the entry to get the value.*)
let entry_of_prop prop prop_values =
  match prop_values with
    Bool -> create_combo prop ["true" ; "false"] false false
  | PosInt -> 
      let entry = GEdit.entry () in
      entry#set_text prop.prop_value ;
      (entry#coerce, entry)
  | Float ->
      let entry = GEdit.entry () in
      entry#set_text prop.prop_value ;
      (entry#coerce, entry)
  | Code | Code_list ->
      let entry = GEdit.entry () in
      entry#set_text prop.prop_value ;
      (entry#coerce, entry)
  | Enum l -> 
      create_combo prop
	(List.map fst l)
	false false
  | Enum_list l -> 
      create_combo prop
	(List.map fst l)
	false false
  | Keysym ->
      create_combo prop
	("" :: (List.map fst Zog_keysyms.keys))
	false true




(** A box for a property. *)
class prop_box handler ele prop =
  let (_,prop_name,prop_values, typ) = 
    List.find (fun (k,_,_,_) -> k = prop.prop_kind) Zog_types.properties 
  in
  let wl_name = GMisc.label ~text: (prop_name^" : ") ~justify: `FILL () in
  let (w,entry) = entry_of_prop prop prop_values in
  let wl_type = GMisc.label 
      ~text: (match typ with "" -> " ("^Zog_messages.ocamlid^")" | _ -> " : "^typ)
      ~justify: `FILL () 
  in
  object (self)
    method box_name = wl_name#coerce
    method box_value = w
    method box_type = wl_type#coerce

    initializer
      let _ = entry#connect#changed
	  (fun () -> 
	    prop.prop_value <- entry#text ;
	    handler#property_changed ele prop)
      in
      ()
  end

(** A box to edit the properties of the given gui element. *)
class box data handler parent_opt ele =
  let vbox = GPack.vbox () in
  let hbox_name = GPack.hbox 
      ~packing: (vbox#pack ~expand: false ~padding: 4)
      ()
  in
  let wl_name = GMisc.label ~text: Zog_messages.name 
      ~packing: (hbox_name#pack ~expand: false ~padding: 3) ()
  in
  let we_name = GEdit.entry ~packing: (hbox_name#pack ~expand: true ~padding: 3) () in
  let wl_in_intf = GMisc.label ~text: "" ~packing: (hbox_name#pack ~expand: false ~padding: 3) () in 

  let f_set_intf_string () = 
    wl_in_intf#set_text 
      (if Zog_data.in_interface ele then 
	Zog_messages.in_interface
      else 
	Zog_messages.not_in_interface)
  in
  let _ = f_set_intf_string () in

  let _ = we_name#set_text ele.name in
  let table = GPack.table ~rows: (List.length ele.props) ~columns: 3 
      ~border_width: 10
      ~packing: (vbox#pack ~expand: true) () 
  in
  object (self)
    method box = vbox
	
    initializer
      let _ = we_name#connect#changed
	  (fun () -> 
	    ele.name <- we_name#text ;
	    f_set_intf_string ();
	    data#set_changed true ;
	    handler#ele_name_changed ele)
      in
      let filter l = 
	List.filter
	  (fun p -> not (List.mem p.prop_kind [ Padding ; Expand ; Fill ]))
	  l
      in
      let filter_tab_label l = List.filter (fun p -> p.prop_kind <> Tab_label) l in
      let lprops = 
	match parent_opt with 
	  None -> filter_tab_label (filter ele.props)
	| Some p_ele ->
	    match Zog_types.pack_method_of_ele p_ele ele with
	      Pack -> filter_tab_label ele.props
	    | Add | Add1 | Add2 | Add_with_viewport | No_pack | Set_submenu -> 
		filter_tab_label (filter ele.props)
	    | Insert_page ->
		filter ele.props
      in
      let _ = 
	List.fold_left
	  (fun acc -> fun p ->
	    let b = new prop_box handler ele p in
	    let _ = table#attach ~left: 0 ~top: acc ~fill: `X b#box_name#coerce in
	    let _ = table#attach ~left: 1 ~top: acc ~expand: `X b#box_value#coerce in
	    let _ = table#attach ~left: 2 ~top: acc ~fill: `X b#box_type#coerce in
	    acc + 1
	  )
	  0
	  lprops
      in
      ()

  end
  
