
(** Reading and writing zoggy files. *)

open Zog_types

(** {2 Reading} *)

let read_channel ?(gui=false) ic =
  try
    Zog_misc.line_number := 0;
    let lexbuf = Lexing.from_channel ic in
    Zog_parser.info Zog_lexer.token lexbuf 
  with
  | Parsing.Parse_error ->
      let s = "Parse error line "^(string_of_int !Zog_misc.line_number) in
      if gui then 
	GToolbox.message_box Zog_messages.error s
      else
	prerr_endline s;
      { 
	info_header = "" ;
	info_entities = [] ;
      }	


let read ?gui f =
  try
    let ic = open_in f in
    let info = read_channel ?gui ic in
    close_in ic;
    info
  with
    Sys_error s ->
      raise (Failure s)


(** {2 Writing} *)

let escape s =
  Toolhtml.escape_quotes (Toolhtml.escape_entities s)

let write_property oc p =
  Printf.fprintf oc " %s=\"%s\"" p#name (escape p#current)

let rec write_component oc c =
  Printf.fprintf oc "<component kind=\"%s\" name=\"%s\"" c#kind (escape c#name);
  List.iter (write_property oc) c#props ;
  Printf.fprintf oc ">";
  List.iter (write_component oc) c#children ;
  Printf.fprintf oc "</component>\n" 

let write_param oc s =
  Printf.fprintf oc "<param>\"%s\"</param>\n" (escape s)

let write_entity oc e =
  Printf.fprintf oc "<entity name=\"%s\">" (escape e.ent_name);
  List.iter (write_param oc) e.ent_params;
  (match e.ent_component with
    None -> ()
  | Some c -> write_component oc c);
  Printf.fprintf oc "</entity>\n"

let write_info oc info =
  Printf.fprintf oc "<header>\"%s\"</header>\n" 
    (escape info.info_header);
  Printf.fprintf oc "<entities>";
  List.iter (write_entity oc) info.info_entities ;
  Printf.fprintf oc "</entities>\n" 

let write f info =
  try
    let oc = open_out f in
    write_info oc info;
    close_out oc
  with
    Sys_error s ->
      raise (Failure s)
