/* KanjiPad - Japanese handwriting recognition front end
 * Copyright (C) 1997 Owen Taylor
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <ctype.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <netinet/in.h> /* for ntohl */
#include "jstroke.h"

#define MAX_STROKES 32
#define BUFLEN 1024

static char *stroke_dicts[MAX_STROKES];

void
load_database(char *fname)
{
  int fd;
  int i;

  fd = open(fname, 0, O_RDONLY);

  if (fd < 0)
    {
      fprintf(stderr,"Can't open %s\n", fname);
      exit(1);
    }


  for (i=0;i<MAX_STROKES;i++)
    stroke_dicts[i] = NULL;

  while (1)
    {
      int bytes_read;
      unsigned int nstrokes;
      unsigned int len;
      int buf[2];

      bytes_read = read(fd,&buf,2*sizeof(int));
      
      nstrokes = ntohl(buf[0]);
      len = ntohl(buf[1]);
      if ((bytes_read != 2*sizeof(int)) || (nstrokes > MAX_STROKES))
	{
	  fprintf(stderr, "Corrupt stroke database");
	  exit(1);
	}

      if (nstrokes == 0)
	break;

      stroke_dicts[nstrokes] = malloc(len);
      bytes_read = read(fd, stroke_dicts[nstrokes], len);

      if (bytes_read != len)
	{
	  fprintf(stderr, "Corrupt stroke database");
	  exit(1);
	}
    }
  
  close(fd);
}

/* From Ken Lunde's _Understanding Japanese Information Processing_
   O'Reilly, 1993 */

int
process_strokes (FILE *file)
{
  RawStroke strokes[MAX_STROKES];
  char *buffer = malloc(BUFLEN);
  int buflen = BUFLEN;
  int nstrokes = 0;

  /* Read in strokes from standard in, all points for each stroke
   *  strung together on one line, until we get a blank line
   */
  
  while (1)
    {
      char *p,*q;
      int len;

      if (!fgets(buffer, buflen, file))
	return 0;

      while ((strlen(buffer) == buflen - 1) && (buffer[buflen-2] != '\n'))
	{
	  buflen += BUFLEN;
	  buffer = realloc(buffer, buflen);
	  if (!fgets(buffer+buflen-BUFLEN-1, BUFLEN+1, file))
	    return 0;
	}
      
      len = 0;
      p = buffer;
      
      while (1) {
	while (isspace (*p)) p++;
	if (*p == 0)
	  break;
	strokes[nstrokes].m_x[len] = strtol (p, &q, 0);
	if (p == q)
	  break;
	p = q;
	  
	while (isspace (*p)) p++;
	if (*p == 0)
	  break;
	strokes[nstrokes].m_y[len] = strtol (p, &q, 0);
	if (p == q)
	  break;
	p = q;
	
	len++;
      }
      
      if (len == 0)
	break;
      
      strokes[nstrokes].m_len = len;
      nstrokes++;
      if (nstrokes == MAX_STROKES)
	break;
    }
  
  if (nstrokes != 0 && stroke_dicts[nstrokes])
    {
      int i;
      ListMem *top_picks;
      StrokeScorer *scorer = StrokeScorerCreate (stroke_dicts[nstrokes],
						 strokes, nstrokes);
      if (scorer)
	{
	  StrokeScorerProcess(scorer, -1);
	  top_picks = StrokeScorerTopPicks(scorer);
	  StrokeScorerDestroy(scorer);
	  
	  for (i=0;i<top_picks->m_argc;i++)
	    {
	      unsigned char c[2];
	      c[0] = top_picks->m_argv[i][0];
	      c[1] = top_picks->m_argv[i][1];
	      printf("%c%c",c[0],c[1]);
	    }
	  
	  free(top_picks);
	}
      printf("\n");
      fflush(stdout);
    }
  return 1;
}

int 
main(int argc, char **argv)
{
  load_database(argv[1]);

  while (process_strokes (stdin))
    ;

  return 0;
}
