
;;; Translation of Gears.c by Brian Paul.

(use-package :gl)

(defconstant M_PI 3.14159265)
(defvar display-callback)
(defvar idle-callback)
(defvar key-callback)
(defvar special-key-callback)
(defvar visible-callback)
(defvar reshape-callback)

(defvar window)
(defvar view_rotx 20.0)
(defvar view_roty 30.0)
(defvar view_rotz 0.0)
(defvar gear1)
(defvar gear2)
(defvar gear3)
(defvar angle 0.0)

(defvar limit 0)
(defvar count 1)


(defmacro glVertex (x y z)
 `(glVertex3f
   (coerce ,x 'single-float)
   (coerce ,y 'single-float)
   (coerce ,z 'single-float)))

(defmacro glNormal (x y z)
 `(glNormal3f
   (coerce ,x 'single-float)
   (coerce ,y 'single-float)
   (coerce ,z 'single-float)))

(defmacro glRotate (a x y z)
 `(glRotatef
   (coerce ,a 'single-float)
   (coerce ,x 'single-float)
   (coerce ,y 'single-float)
   (coerce ,z 'single-float)))

(defmacro glTranslate (x y z)
 `(glTranslatef
   (coerce ,x 'single-float)
   (coerce ,y 'single-float)
   (coerce ,z 'single-float)))

(defun gear (inner_radius outer_radius width teeth tooth_depth)
 (let ((r0 inner_radius)
       (r1 (/ (- outer_radius tooth_depth) 2.0))
       (r2 (/ (+ outer_radius tooth_depth) 2.0))
       (da (/ (* 2.0 M_PI) teeth 4.0)))
  
  (glShadeModel GL_FLAT)
  (glNormal 0.0 0.0 1.0)
  
  ;;/* draw front face */
  (glBegin GL_QUAD_STRIP)
  (dotimes (i teeth)
   (let* ((angle (/ (* i 2.0 M_PI) teeth)))
    (glVertex (* r0 (cos angle)) (* r0 (sin angle)) (* width 0.5))
    (glVertex (* r1 (cos angle)) (* r1 (sin angle)) (* width 0.5))
    (glVertex (* r0 (cos angle)) (* r0 (sin angle)) (* width 0.5))
    (glVertex (* r1 (cos (+ angle (* 3 da))))
	      (* r1 (sin (+ angle (* 3 da))))
	      (* width 0.5))))
  (glEnd)

  ;;/* draw front sides of teeth */
  (glBegin GL_QUADS)
  (let ((da (/ (* 2.0 M_PI) teeth 4.0)))
   (dotimes (i teeth)
    (let ((angle (/ (* i 2.0 M_PI) teeth)))
     (glVertex (* r1 (cos angle)) (* r1 (sin angle)) (* width 0.5))
     (glVertex (* r2 (cos (+ angle da))) (* r2 (sin (+ angle da)))
	       (* width 0.5))
     (glVertex (* r2 (cos (+ angle (* 2 da))))
	       (* r2 (sin (+ angle (* 2 da))))
	       (* width 0.5))
     (glVertex (* r1 (cos (+ angle (* 3 da))))
	       (* r1 (sin (+ angle (* 3 da))))
	       (* width 0.5)))))
  (glEnd)

  (glNormal 0.0 0.0 -1.0)

  ;;/* draw back face */
  (glBegin GL_QUAD_STRIP)
  (dotimes (i teeth)
   (let ((angle (/ (* i 2.0 M_PI) teeth)))
    (glVertex (* r1 (cos angle)) (* r1 (sin angle)) (* (- width) 0.5))
    (glVertex (* r0 (cos angle)) (* r0 (sin angle)) (* (- width) 0.5))
    (glVertex (* r1 (cos (+ angle (* 3 da))))
	      (* r1 (sin (+ angle (* 3 da))))
	      (* (- width) 0.5))
    (glVertex (* r0 (cos angle)) (* r0 (sin angle)) (* (- width) 0.5))))
  (glEnd)

  ;;/* draw back sides of teeth */
  (glBegin GL_QUADS)
  (let ((da (/ (* 2.0 M_PI) teeth 4.0)))
   (dotimes (i teeth)
    (let ((angle (/ (* i 2.0 M_PI) teeth)))
     (glVertex (* r1 (cos (+ angle (* 3 da))))
	       (* r1 (sin (+ angle (* 3 da))))
	       (* (- width) 0.5))
     (glVertex (* r2 (cos (+ angle (* 2 da))))
	       (* r2 (sin (+ angle (* 2 da))))
	       (* (- width) 0.5))
     (glVertex (* r2 (cos (+ angle da))) (* r2 (sin (+ angle da)))
	       (* (- width) 0.5))
     (glVertex (* r1 (cos angle)) (* r1 (sin angle)) (* (- width) 0.5)))))
  (glEnd)

  ;;/* draw outward faces of teeth */
  (glBegin GL_QUAD_STRIP)
  (dotimes (i teeth)
   (let ((angle (/ (* i 2.0 M_PI) teeth)))
    (glVertex (* r1 (cos angle)) (* r1 (sin angle)) (* width 0.5))
    (glVertex (* r1 (cos angle)) (* r1 (sin angle)) (* (- width) 0.5))
    (let* ((u (- (* r2 (cos (+ angle da))) (* r1 (cos angle))))
	   (v (- (* r2 (sin (+ angle da))) (* r1 (sin angle))))
	   (len (sqrt (+ (* u u) (* v v)))))
     (setq u (/ u len))
     (setq v (/ v len))
     (glNormal v (- u) 0.0)
     (glVertex (* r2 (cos (+ angle da)))
	       (* r2 (sin (+ angle da)))
	       (* width 0.5))
     (glVertex (* r2 (cos (+ angle da)))
	       (* r2 (sin (+ angle da))) (* (- width) 0.5))
     (glNormal (cos angle) (sin angle) 0.0)
     (glVertex (* r2 (cos (+ angle (* 2 da))))
	       (* r2 (sin (+ angle (* 2 da))))
	       (* width 0.5))
     (glVertex (* r2 (cos (+ angle (* 2 da))))
	       (* r2 (sin (+ angle (* 2 da))))
	       (* (- width) 0.5))
     (let ((u (- (* r1 (cos (+ angle (* 3 da))))
		 (* r2 (cos (+ angle (* 2 da))))))
	   (v (- (* r1 (sin (+ angle (* 3 da))))
		 (* r2 (sin (+ angle (* 2 da)))))))
      (glNormal v (- u) 0.0)
      (glVertex (* r1 (cos (+ angle (* 3 da))))
		(* r1 (sin (+ angle (* 3 da))))
		(* width 0.5))
      (glVertex (* r1 (cos (+ angle (* 3 da))))
		(* r1 (sin (+ angle (* 3 da))))
		(* (- width) 0.5))
      (glNormal (cos angle) (sin angle) 0.0)))))
  (glVertex (* r1 (cos 0)) (* r1 (sin 0)) (* width 0.5))
  (glVertex (* r1 (cos 0)) (* r1 (sin 0)) (* (- width) 0.5))
  (glEnd)
  
  (glShadeModel GL_SMOOTH)
  
  ;;/* draw inside radius cylinder */
  (glBegin GL_QUAD_STRIP)
  (dotimes (i teeth)
   (let ((angle (/ (* i 2.0 M_PI) teeth)))
    (glNormal (- (cos angle)) (- (sin angle)) 0.0)
    (glVertex (* r0 (cos angle)) (* r0 (sin angle)) (* (- width) 0.5))
    (glVertex (* r0 (cos angle)) (* r0 (sin angle)) (* width 0.5))))
  (glEnd)))


(ff:defun-c-callable display ()
 (format t "Callback DISPLAY.~%")
 ;; Redraw callback
 (glClear (+ GL_COLOR_BUFFER_BIT  GL_DEPTH_BUFFER_BIT))
 
 (glPushMatrix)
 (glRotate view_rotx 1.0 0.0 0.0)
 (glRotate view_roty 0.0 1.0 0.0)
 (glRotate view_rotz 0.0 0.0 1.0)

 (glPushMatrix)
 (glTranslate -3.0 -2.0 0.0)
 (glRotate angle 0.0 0.0 1.0)
 (glCallList gear1)
 (glPopMatrix)

 (glPushMatrix)
 (glTranslate 3.1 -2.0 0.0)
 (glRotate (- (* -2.0 angle) 9.0) 0.0 0.0 1.0)
 (glCallList gear2)
 (glPopMatrix)

 (glPushMatrix)
 (glTranslate -3.1 4.2 0.0)
 (glRotate (- (* -2.0 angle) 25.0) 0.0 0.0 1.0)
 (glCallList gear3)
 (glPopMatrix)

 (glPopMatrix)
 (glutSwapBuffers))

(ff:defun-c-callable idle ()
 (format t "Callback IDLE.~%")
 (incf angle 2.0)
 (glutPostRedisplay))

;;/* change view angle, exit upon ESC */
(ff:defun-c-callable key ((k :unsigned-byte) (x :fixnum) (y :fixnum))
 (format t "Callback KEY.  K:~s, X:~s, Y:~s~%" k x y)
 (case (character k)
   (#\z
    (incf view_rotz 5.0)
    (glutPostRedisplay))
   (#\Z
    (decf view_rotz 5.0)
    (glutPostRedisplay))
   (#\Escape
    (glutDestroyWindow window)
    (break))))

;;/* change view angle */
(ff:defun-c-callable special-key ((k :fixnum) (x :fixnum) (y :fixnum))
 (format t "Callback SPECIAL-KEY.  K:~s, X:~s, Y:~s~%" k x y)
 (case k
   (GLUT_KEY_UP
    (incf view_rotx 5.0))
   (GLUT_KEY_DOWN
    (decf view_rotx 5.0))
   (GLUT_KEY_LEFT
    (incf view_roty 5.0))
   (GLUT_KEY_RIGHT
    (decf view_roty 5.0))
   (otherwise
    (return-from special-key)))
 (glutPostRedisplay))

;;/* new window size or exposure */
(ff:defun-c-callable reshape ((width :fixnum) (height :fixnum))
 (format t "Callback RESHAPE. WIDTH:~s, HEIGHT:~s~%" width height)
 (let ((h (coerce (/ height width) 'double-float)))
  (glViewport 0 0 width height)
  (glMatrixMode GL_PROJECTION)
  (glLoadIdentity)
  (glFrustum (- h) h (- h) h 5d0 60d0)
  (glMatrixMode GL_MODELVIEW)
  (glLoadIdentity)
  (glTranslate 0.0 0.0 -40.0)))


(defun init ()
 (let ((pos (make-array 4 :element-type 'single-float
			:initial-contents '(5.0 5.0 10.0 0.0)))
       (red (make-array 4 :element-type 'single-float
			:initial-contents '(0.8 0.1 0.0 1.0)))
       (green (make-array 4 :element-type 'single-float
			  :initial-contents '(0.0 0.8 0.2 1.0)))
       (blue (make-array 4 :element-type 'single-float
			 :initial-contents '(0.2 0.2 1.0 1.0))))
  
  (glLightfv GL_LIGHT0 GL_POSITION pos)
  (glEnable GL_CULL_FACE)
  (glEnable GL_LIGHTING)
  (glEnable GL_LIGHT0)
  (glEnable GL_DEPTH_TEST)

  ;;/* make the gears */
  (setq gear1 (glGenLists 1))
  (glNewList gear1 GL_COMPILE)
  (glMaterialfv GL_FRONT GL_AMBIENT_AND_DIFFUSE red)
  (gear 1.0 4.0 1.0 20 0.7)
  (glEndList)

  (setq gear2 (glGenLists 1))
  (glNewList gear2 GL_COMPILE)
  (glMaterialfv GL_FRONT GL_AMBIENT_AND_DIFFUSE green)
  (gear 0.5 2.0 2.0 10 0.7)
  (glEndList)

  (setq gear3 (glGenLists 1))
  (glNewList gear3 GL_COMPILE)
  (glMaterialfv GL_FRONT GL_AMBIENT_AND_DIFFUSE blue)
  (gear 1.3 2.0 0.5 10 0.7)
  (glEndList)

  (glEnable GL_NORMALIZE)))

(ff:defun-c-callable visible ((vis :fixnum))
 (format t "Callback VISIBLE. VIS:~s~%" vis)
 (cond ((= vis GLUT_VISIBLE)
	(glutIdleFunc idle-callback))
       (t
	(glutIdleFunc (coerce 0 'integer)))))

;;; The *address* of callback functions is stored as variables.
(setq display-callback (ff:register-function 'display))
(setq idle-callback (ff:register-function 'idle))
(setq key-callback (ff:register-function 'key))
(setq special-key-callback (ff:register-function 'special-key))
(setq visible-callback (ff:register-function 'visible))
(setq reshape-callback (ff:register-function 'reshape))

(defun main ()
 (glutInitDisplayMode (+ GLUT_RGB GLUT_DEPTH GLUT_DOUBLE))
 (glutInitWindowPosition 0 0)
 (glutInitWindowSize 300 300)
 (setq window (glutCreateWindow "Gears"))
 (init)

 (glutDisplayFunc display-callback)
 (glutReshapeFunc reshape-callback)
 (glutKeyboardFunc key-callback)
 (glutSpecialFunc special-key-callback)
 (glutVisibilityFunc visible-callback)
 
 (glutMainLoop))
