;;;
;;; GLXDEMO.LSP
;;;
;;; Translation of "glxdemo.c" by Brian Paul
;;;

(defun redraw (display window)
 (format t "Redrawing~%")
 (gl:glclear GL:GL_COLOR_BUFFER_BIT)
 (gl:glcolor3f 1 1 0)
 (gl:glrectf -0.8 -0.8 0.8 0.8)
 (gl:glxswapbuffers display window))

;;; always keeps 2d geometry fullsize in window
(defun resize (width height)
 (format t "Resizing~%")
 (gl:glviewport 0 0 width height)
 (gl:glmatrixmode GL:GL_PROJECTION)
 (gl:glortho -1.0 1.0 -1.0 1.0 -1.0 1.0))

;;;
;;; Event loop:
;;; 1. "gobble" redraw events to prevent lag/queue overflow.
;;; 2. would have to check other events if there were more than one window
;;;    active...
;;; It seems that a single Resize occurs only once resize done.
;;; For some reason, several Expose occur during resize.  (Need to gobble).
;;; Also, since redraw may be slow, need to gobble.
;;;
(defun event-loop (display)
 ;; Event loop
 (let ((debug t)
       (event (xlib:make-xevent)))
  (when debug (format t "Event-loop.~%"))
  (loop
   ;; Wait for next event
   (when debug (format t "Wait..."))
   (xlib:xnextevent display event)
   (let ((event-type (xlib:xanyevent-type event))
	 (window (xlib:xanyevent-window event)))
    (when debug (format t "event-type:~a~%" event-type))
    (cond
      ;; Return on buttonpress event.
      ((eq event-type xlib:buttonpress) (return))
      ;; Expose
      ((eq event-type xlib:expose)
       ;; Gobble all other expose events
       (loop
	(unless (> (xlib:xeventsqueued display xlib:queuedalready) 0)
	 (return))
	(xlib:xnextevent display event)
	(let ((event-type (xlib:xanyevent-type event)))
	 (unless (eq event-type xlib:expose)
	  (xlib:xputbackevent display event)
	  (return))
	 (when debug (format t "Gobble event-type:~a~%" event-type))))
       (redraw display window))
      ;; Resize
      ((eq event-type xlib:configurenotify)
       (resize (xlib:xconfigureevent-width event)
	       (xlib:xconfigureevent-height event))))))))


;;;
;;; This call binds GL to an existing X window.
;;;

(defun bind-gl-to-window (display screen window)
 (let ((debug t))
  (when debug (format t "Bind-gl-to-current-window.~%"))
  ;;
  (when debug (format t "XGetWindowAttributes..."))
  (let* ((attr (xlib:make-xwindowattributes))
	 (foo (xlib:xgetwindowattributes display window attr))
	 (class (xlib:xwindowattributes-class attr))
	 (depth (xlib:xwindowattributes-depth attr))
	 (visual (xlib:xwindowattributes-visual attr))
	 (visual-class (xlib:visual-class visual)))
   (when debug
    (format t "screen:~a, " screen)
    (format t "class:~a, depth:~a, " class depth)
    (format t "visual-class:~a~%" visual-class))
   ;;
   (when debug (format t "XMatchVisualInfo..."))
   (let* ((visualinfo (xlib:make-xvisualinfo))
	  (num-visuals (xlib:xmatchvisualinfo display screen depth
					      visual-class visualinfo)))
    (unless (> num-visuals 0)
     (error "BIND-GL-TO-WINDOW: Could not get visual of class:~a, depth~a!"
	    visual-class depth))
    (when debug (format t "~a visuals found.~%" num-visuals))
    ;;
    (when debug (format t "glXCreateContext..."))
    (let ((glx-context (gl:glxcreatecontext display visualinfo
					    XLIB:NULL GL:GL_TRUE)))
     (when debug (format t "~%glXMakeCurrent..."))
     (gl:glxmakecurrent display window glx-context))))
  (when debug (format t "~%Done.~%"))))

 
(defun create-gl-simple-window (display width height)
 (let* ((screen (xlib:xdefaultscreen display))
        (root (xlib:xrootwindow display screen))
        (black-pixel (xlib:xblackpixel display screen))
        (white-pixel (xlib:xwhitepixel display screen))
        (window (xlib:xcreatesimplewindow display root 0 0 width height
					  1 black-pixel white-pixel)))
  ;; Enable events
  (xlib:xselectinput display window
		     (+ xlib:structurenotifymask
			xlib:exposuremask
			xlib:buttonpressmask))
  ;; Bind to GL
  (bind-gl-to-window display screen window)
  ;; Map window
  (xlib:xmapwindow display window)
  ;; Return window
  window))

(defun main ()
 (format t "Demo.~%")
 (let* ((display (xlib:xopendisplay ""))
	(window (create-gl-simple-window display 300 300)))
  (gl:glshademodel GL:GL_FLAT)
  (gl:glclearcolor 0.5 0.5 0.5 1.0)
  (event-loop display)))
