;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: odcl -*-
;;; $Id: persistence.lisp,v 1.8 2003/03/24 21:51:31 adam Exp $
;;;
;;; Copyright (c) 2001 - 2003 onShore Development, Inc.

(in-package :odcl)

;; persistent alist

(defclass persistent-alist (transaction-object)
  ((name :initarg :name)
   (path :initarg :path)
   (data :initarg :data
         :initform nil)))

(defun find-palist (name root)
  "Locate a persistent a-list."
  (let ((path (conjoin-palist-path root name)))
    (when (probe-file path)
      (when-bind (palist (make-instance 'persistent-alist
                                        :name name
                                        :path root
                                        :data (read-from-file path)))
        (when *default-editing-context*
          (ec-register palist *default-editing-context*)
          (ec-cache palist *default-editing-context*))
        palist))))

(defun conjoin-palist-path (root name)
  (extend-path root (string-downcase (symbol-name name))))

(defun palist-path (palist)
  (with-slots (path name)
    palist
    (conjoin-palist-path path name)))

(defun make-palist (name root)
  "Create a new persistent a-list at PATH."
  (let* ((palist (make-instance 'persistent-alist :name name :path root)))
    (when *default-editing-context*
      (ec-cache palist *default-editing-context*)
      (ec-insert palist *default-editing-context*))
    palist))

(defun destroy-palist (palist)
  "Destroy a filesystem database."
  (when (probe-file (palist-path palist))
    (if *default-editing-context*
        (ec-delete palist *default-editing-context*)
        (delete-file (palist-path palist))))
  t)

(defun get-palist (item alist &key (test #'eql))
  (cdr (assoc item (slot-value alist 'data) :test test)))

(defun (setf get-palist) (value item alist &key (test #'eql))
  (with-slots (data path)
    alist
    (update-alist item value data :test test)
    (if *default-editing-context*
        (ec-edit alist *default-editing-context*)
        (update-instance alist)))
  value)

;; Transaction Object support
(defclass persistent-alist-store (transactable-store)
  ())

(defmethod get-store-instance-by-key ((self persistent-alist-store) key)
  (find-palist (second key) (third key)))

(defmethod instance-key ((palist persistent-alist))
  (with-slots (name path)
    palist
    (list 'persistent-alist path name)))

(defmethod install-instance ((palist persistent-alist) &rest args)
  (declare (ignore args))
  (with-slots (data)
    palist
    (unless (probe-file (palist-path palist))
      (write-to-file (palist-path palist) data))
    palist))

(defmethod update-instance ((palist persistent-alist) &rest args)
  (declare (ignore args))
  (with-slots (data)
    palist
    (write-to-file (palist-path palist) data)
    palist))

(defmethod delete-instance ((palist persistent-alist) &rest args)
  (declare (ignore args))
  (without-context
   (destroy-palist palist)))

(defmethod odcl::commit-phase-1 ((self persistent-alist-store) transaction)
  (let (rollback)
    (dolist (action (reverse (slot-value transaction 'odcl::stack)))
      (when (eql self (odcl::tx-event-store action))
        (let ((data (odcl::tx-event-data action)))
          (ecase (odcl::tx-event-type action)
            (:rollback
             (push (lambda ()
                     (if data
                         (install-instance data)
                         (delete-instance self)))
                   rollback))
            (:edit
             (update-instance data))
            (:insert
             (install-instance data))
            (:delete
             (delete-instance data))))))
    (lambda (action)
      (ecase action
        (:commit-phase-2
         t)
        (:rollback
         (mapc #'funcall rollback))))))

(defmethod odcl::instance-snapshot ((self persistent-alist))
  (cons :snapshot (copy-alist (slot-value self 'data))))

(defmethod odcl::instance-restore ((self persistent-alist) snapshot)
  (destructuring-bind (tag . data)
      snapshot
    (unless (eql :snapshot tag)
      (error "invalid snapshot"))
    (setf (slot-value self 'data) data)
    self))
