/* sysinfo.c */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#ifdef __FreeBSD__
#include <kvm.h>
#include <sys/sysctl.h>
#endif	/* __FreeBSD__ */
#ifdef __linux__
#include <sys/utsname.h>
#endif	/* __linux__ */

#include "trivial.h"
#include "sysinfo.h"

static struct sysdata system_info = {"\0", 0, 0, 0, 0};

static int get_uname_info(struct sysdata *sd);
static int get_cpu_info(struct sysdata *sd);
static int get_vm(struct sysdata *sd);

#ifdef __FreeBSD__
static int get_uname_info(struct sysdata *sd)
{
	char *un=sd->uname;
	size_t max_uname=MAX_UNAME;
	if(sysctlbyname("kern.version", un, &max_uname, NULL, 0)==-1){
		printe("error getting os version using sysctlbyname()\n");
		return 1;
	}
	un[MAX_UNAME-1]='\0';
	strsep(&un, " ");
	set_bit(sd->valid, INFO_UNAME);
	return 0;
}
#endif	/* __FreeBSD__ */
#ifdef __linux__
static int get_uname_info(struct sysdata *sd)
{
	struct utsname buf;
	if(uname(&buf)==-1){
		printe("error getting os version using uname()\n");
		return 1;
	}
	strncpy(sd->uname, buf.release, MAX_UNAME);
	set_bit(sd->valid, INFO_UNAME);
	return 0;
}
#endif	/* __linux__ */

#ifdef __linux__
static int get_cpu_info(struct sysdata *sd)
{
	FILE *cpuinfo;
	char buf[200];
	char ch;
	int i, cpus;

	clear_bit(sd->valid, INFO_CPU);

#ifdef _SC_NPROCESSORS_ONLN
	if(((cpus=sysconf(_SC_NPROCESSORS_ONLN))!=-1) && (cpus>0)){
		sd->nrcpus=cpus;
		set_bit(sd->valid, INFO_CPU);
		return 0;
	}
	/* sysconf failed, let's try it ourselves */
#endif	/* _SC_NPROCESSORS_ONLN */

	if(!(cpuinfo=fopen("/proc/cpuinfo", "r"))){
		printsys("cannot open /proc/cpuinfo\n");
	}
	cpus=0;
	for(i=0; (ch=fgetc(cpuinfo))!=EOF;){
		if((ch!='\n') || (i==198)){
			buf[i++]=ch;
			continue;
		}
		i=0;
		if(!strncmp(buf, "processor", 9)){
			cpus++;
		}
	}
	sd->nrcpus=cpus;
	set_bit(sd->valid, INFO_CPU);
	return 0;
}
#endif	/* linux */

#ifdef __FreeBSD__
static int get_cpu_info(struct sysdata *sd)
{
	int buf;
	size_t max_buf=sizeof(int);

	clear_bit(sd->valid, INFO_CPU);
#ifdef _SC_NPROCESSORS_ONLN
	if(((buf=sysconf(_SC_NPROCESSORS_ONLN))!=-1) && (buf>0)){
		sd->nrcpus=buf;
		set_bit(sd->valid, INFO_CPU);
		return 0;
	}
	/* sysconf failed, let's try it ourselves */
#endif	/* _SC_NPROCESSORS_ONLN */

	if(sysctlbyname("hw.ncpu", &buf, &max_buf, NULL, 0)){
		printsys("failed sysctlbyname(\"hw.ncpu\")\n");
	}
	sd->nrcpus=buf;
	set_bit(sd->valid, INFO_CPU);
	return 0;
}
#endif	/* __FreeBSD__ */

static int get_vm(struct sysdata *sd)
{
#ifdef __linux__
	int res;
	FILE *meminfo;

	if(!(meminfo=fopen("/proc/meminfo", "r"))){
		printsys("cannot open /proc/meminfo\n");
	}
	for(;;){
		res=fscanf(meminfo, "MemTotal: %lu kB", &sd->ram);
		if(res==1){
			set_bit(sd->valid, INFO_RAM);
			break;
		}
		if(feof(meminfo)){
			clear_bit(sd->valid, INFO_RAM);
			break;
		}
		while(fgetc(meminfo)!='\n');
	}
	clearerr(meminfo);
	rewind(meminfo);
	for(;;){
		res=fscanf(meminfo, "SwapTotal: %lu kB", &sd->swap);
		if(res==1){
			set_bit(sd->valid, INFO_SWAP);
			break;
		}
		if(feof(meminfo)){
			clear_bit(sd->valid, INFO_SWAP);
			break;
		}
		while(fgetc(meminfo)!='\n');
	}
#endif	/* __linux__ */
#ifdef __FreeBSD__
	int buf;
	size_t max_buf=sizeof(int);
	kvm_t *kd;
	struct kvm_swap ks;

	clear_bit(sd->valid, INFO_SWAP);
   	kd=kvm_open(NULL, NULL, NULL, O_RDONLY, NULL);
	kvm_getswapinfo(kd, &ks, 1, 0);
	/* FIXME: unportable */
	sd->swap=ks.ksw_total*4;	/* pages to kilobytes */
	set_bit(sd->valid, INFO_SWAP);

	clear_bit(sd->valid, INFO_RAM);
	if(sysctlbyname("hw.physmem", &buf, &max_buf, NULL, 0)){
		printsys("failed sysctlbyname(\"hw.physmem\")\n");
	}
	sd->ram=buf/1024;	/* to kilobytes */
	set_bit(sd->valid, INFO_RAM);
#endif	/* __FreeBSD__ */

	return 0;
}

int get_uname(char **un)
{
	if(test_bit(system_info.valid, INFO_UNAME)){
		*un=system_info.uname;
		return 0;
	}
	get_uname_info(&system_info);
	if(test_bit(system_info.valid, INFO_UNAME)){
		*un=system_info.uname;
		return 0;
	}
	return 1;
}

inline int get_ram(unsigned long *num)
{
	if(test_bit(system_info.valid, INFO_RAM)){
		*num=system_info.ram;
		return 0;
	}
	get_vm(&system_info);
	if(test_bit(system_info.valid, INFO_RAM)){
		*num=system_info.ram;
		return 0;
	}
	return 1;
}

inline int get_swap(unsigned long *num)
{
	if(test_bit(system_info.valid, INFO_SWAP)){
		*num=system_info.swap;
		return 0;
	}
	get_vm(&system_info);
	if(test_bit(system_info.valid, INFO_SWAP)){
		*num=system_info.swap;
		return 0;
	}
	return 1;
}

int get_cpus(int *num)
{
	if(test_bit(system_info.valid, INFO_CPU)){
		*num=system_info.nrcpus;
		return 0;
	}
	get_cpu_info(&system_info);
	if(test_bit(system_info.valid, INFO_CPU)){
		*num=system_info.nrcpus;
		return 0;
	}
	return 1;
}
