/*  cssed (c) Iago Rubio, 2003-2004 - A tiny CSS editor.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */



#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <expat.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <gtk/gtk.h>

#ifndef WIN32
# include <sys/mman.h>
#endif

#include "cssedwindow.h"
#include "support.h"
#include "utils.h"
#include "plugin.h"
#include "debug.h"

/* old spat stuff needed for BSD like machines as MacOsX*/
#ifndef XML_STATUS_OK
   #define XML_STATUS_OK    1
   #define XML_STATUS_ERROR 0
#endif

enum {
	PARSER_STATE_IN_START,
	PARSER_STATE_IN_ROOT,
	PARSER_STATE_IN_WINDOW,
	PARSER_STATE_IN_DOCUMENT,
	PARSER_STATE_IN_OPENED_DOCS,
	PARSER_STATE_IN_HIGHLIGHTING,
	PARSER_STATE_IN_PLUGINS,
	PARSER_STATE_ERROR
};

/* protos */
static void
config_start_tag_cb (void* data, const char *el, const char **attr);

static void
config_end_tag_cb (void *data, const char *el);

void
parse_style_attributes( CssedConfigParserData* data, const char **attr);

/* parse the css definition, it doesn't validate against the DTD yet
	so ANY well formed XML will be parsed.
	Will return a plugins list to be loaded when window is created.
	FIXME: This must change to a separated function.
*/
GList*
parse_configuration_file (CssedWindow* window)
{
	struct stat _stat;
	int _retval;
	gchar* configfile;
	gchar* err ;
	CssedConfigParserData* parser_data;
	GList* plugins_list;
	XML_Parser p;
#ifdef WIN32
	gchar* buffer;
	GError* error = NULL;
	gchar* allocated;
#else
	void* map;
	int filedes;
	gchar* home;
#endif

	parser_data = g_malloc( sizeof(CssedConfigParserData));
	parser_data->window = window;
	parser_data->depth = 0;
	parser_data->state = PARSER_STATE_IN_START;
	parser_data->plugins_list = NULL;

#ifdef WIN32
	configfile = g_strdup("data\\cssed-cfg.xml");
	if( !g_file_test(configfile,G_FILE_TEST_IS_REGULAR| G_FILE_TEST_EXISTS) ){
		return NULL; // no configuration yet
	}
#else
	home = getenv("HOME");
	if( home != NULL ){
		configfile = g_strconcat(home,"/.cssed/cssed-cfg.xml",NULL);
		if( !g_file_test(configfile,G_FILE_TEST_IS_REGULAR| G_FILE_TEST_EXISTS) ){
			return NULL; // no configuration yet
		}
	}else{
		return NULL; // no home so no configuration file
	}
#endif
	DBGMSG("User's config exist: %s", configfile );
	_retval = stat( configfile , &_stat );
	if( _retval != 0 ){
		cssed_error_message( _("Cannot stat on configuration file!!!") , _("Error") );
  	}else{
#ifdef WIN32
		if( !g_file_get_contents (configfile,
									&buffer,
									NULL,
                                    &error) ){
			cssed_error_message( "Can't open configuration file" , "Error" );
			return FALSE;

		}

		allocated = g_strdup( buffer );

		p = XML_ParserCreate (NULL);
		if (!p)
		{
			cssed_error_message( "Can't allocate memory for XML parser !!!" , "Error" );
			return FALSE;
		}

		XML_SetElementHandler (p, config_start_tag_cb, config_end_tag_cb);
		XML_SetUserData( p, (void *) parser_data);

		if (XML_Parse (p, (char*) allocated, _stat.st_size, 1) == XML_STATUS_ERROR)
		{
			err = g_strdup_printf(
				"The configuration XML file is bogus.\nFile: %s\nParse error at line %d:\n%s\n",
				configfile,
				XML_GetCurrentLineNumber (p),
				XML_ErrorString (XML_GetErrorCode (p))
				);
			cssed_error_message	(err,"Configuration error" );
			g_free(err);
		}
		g_free( allocated );
#else
			if( S_ISREG( _stat.st_mode ) ){
	  		if( (filedes = open(configfile, O_RDONLY)) < 0){
				cssed_error_message( _("Cannot open configuration file!!!") , _("Error") );
	  		}else{
				if( (map = mmap(0,_stat.st_size, PROT_READ, MAP_SHARED, filedes, 0)) == MAP_FAILED ){
		  			cssed_error_message( _("Cannot map configuration file to memory!!!") , _("Error") );
		  			close(filedes);
					return NULL;
				}else{
					p = XML_ParserCreate (NULL);
					if (!p)
					{
						cssed_error_message( _("Cannot allocate memory for XML parser!!!") , _("Error") );
						munmap(map,_stat.st_size );
						close(filedes);
						return NULL;
					}

					XML_SetElementHandler (p, config_start_tag_cb, config_end_tag_cb);
					XML_SetUserData( p, (void *) parser_data);

					if (XML_Parse (p, (char*) map, _stat.st_size, 1) == XML_STATUS_ERROR)
					{
						err = g_strdup_printf(
							_("The configuration XML file is bogus.\nFile: %s\nParse error at line %d:\n%s\n"),
							configfile,
							XML_GetCurrentLineNumber (p),
						 	XML_ErrorString (XML_GetErrorCode (p))
							);
						cssed_error_message	(err, _("Configuration error") );
						g_free(err);
						return NULL;
					}
					XML_ParserFree(p);
				}
				munmap(map,_stat.st_size);
				close(filedes);
			}
			}
#endif
	}
	if( parser_data->plugins_list  != NULL ){
		DBGMSG( "config plugins: %d", g_list_length( parser_data->plugins_list ));
		plugins_list = g_list_copy(  parser_data->plugins_list  );
		DBGMSG( "config will return: %d - %p", g_list_length( parser_data->plugins_list ), plugins_list);
		g_list_free( parser_data->plugins_list );
		g_free( parser_data );
		g_free( configfile );
		return plugins_list;
	}else{
		g_free( parser_data );
		g_free( configfile );
		return NULL;
	}
}

static void
config_start_tag_cb (void* data, const char *el, const char **attr)
{
	gint i;
	gchar* err_depth;
	CssedWindow* window;
	CssedConfigParserData* tp_data;
	CssedConfig* cfg;

	tp_data = (CssedConfigParserData*) data;
	window = tp_data->window;
	cfg = cssed_window_get_config( window );
	// FIXME check parser state
	//if( data->state == PARSER_STATE_ERROR) return;

	switch( tp_data->depth ){
		case 0: //
				if( strcmp( el, "cssed-cfg")!=0){
					tp_data->state = PARSER_STATE_ERROR;
				}else{
					tp_data->state = PARSER_STATE_IN_ROOT;
				}
		break;
		case 1: //
			if( tp_data->state != PARSER_STATE_IN_ROOT ){
				tp_data->state = PARSER_STATE_ERROR;
			}
			if( strcmp( el, "window") == 0){
				tp_data->state = PARSER_STATE_IN_WINDOW;				
				for (i = 0; attr[i]; i += 2)
				{  
					if( strcmp( attr[i], "x")==0){	
						if( char_ptr_is_decimal_num(attr[i + 1]) ){
							cfg->window_x = atoi(attr[i + 1]);
						}
					}else if( strcmp( attr[i], "y")==0){	
						if( char_ptr_is_decimal_num(attr[i + 1]) ){
							cfg->window_y = atoi(attr[i + 1]);
						}
					}else if( strcmp( attr[i], "width")==0){
						if( char_ptr_is_decimal_num(attr[i + 1]) ){
							cfg->window_width = atoi(attr[i + 1]);
						}
					}else if( strcmp( attr[i], "height")==0){
						if( char_ptr_is_decimal_num(attr[i + 1]) ){
							cfg->window_height = atoi(attr[i + 1]);
						}
					}else{
						tp_data->state = PARSER_STATE_ERROR;
					}
				}
			}else if( strcmp( el, "document") == 0 ){
				tp_data->state = PARSER_STATE_IN_DOCUMENT;
			}else if( strcmp( el, "opened-docs") == 0 ){
				tp_data->state = PARSER_STATE_IN_OPENED_DOCS;
			}else if( strcmp( el, "highlighting") == 0 ){
				tp_data->state = PARSER_STATE_IN_HIGHLIGHTING;
			}else if( strcmp( el, "plugins") == 0 ){
				tp_data->state = PARSER_STATE_IN_PLUGINS;
			}else{
				tp_data->state = PARSER_STATE_ERROR;
			}
		break;
		case 2: // 
			switch( tp_data->state ){
				case PARSER_STATE_IN_WINDOW:
					if( strcmp( el, "notebook" ) == 0){
						if( strcmp( attr[0], "height") == 0 ){
							if( char_ptr_is_decimal_num(attr[1]) ){
								cfg->notebook_height = atoi(attr[1]);
							}
						}else{
							tp_data->state = PARSER_STATE_ERROR;
						}
					}else if( strcmp( el, "sidebar" ) == 0){
						if( strcmp( attr[0], "width") == 0 ){
							if( char_ptr_is_decimal_num(attr[1]) ){
								cfg->sidebar_width = atoi(attr[1]);
							}
						}else{
							tp_data->state = PARSER_STATE_ERROR;
						}
					}else if( strcmp( el, "treeview" ) == 0){	
						if( strcmp( attr[0], "height") == 0 ){
							if( char_ptr_is_decimal_num(attr[1]) ){
								cfg->treeview_height = atoi(attr[1]);
							}
						}else{
							tp_data->state = PARSER_STATE_ERROR;
						}
					}else{
						tp_data->state = PARSER_STATE_ERROR;
					}
					break;
				case PARSER_STATE_IN_DOCUMENT:
					if( strcmp( el, "linenumbers" ) == 0){
						if( strcmp( attr[0], "show") == 0 ){
							if( strcmp( attr[1], "true" ) == 0 ){
								cfg->linenumbers = TRUE;
							}else{
								cfg->linenumbers = FALSE;
							}
						}else{
							tp_data->state = PARSER_STATE_ERROR;
						}
					}else if( strcmp( el, "autocompletion" ) == 0){
						if( strcmp( attr[0], "enabled") == 0 ){
							if( strcmp( attr[1], "true" ) == 0 ){
								cfg->autocompletion = TRUE;
							}else{
								cfg->autocompletion = FALSE;
							}	
						}else{
							tp_data->state = PARSER_STATE_ERROR;
						}
					}else if( strcmp( el, "lineendings" ) == 0){	
						if( strcmp( attr[0], "show") == 0 ){
							if( strcmp( attr[1], "true" ) == 0 ){
								cfg->lineendings = TRUE;
							}else{
								cfg->lineendings = FALSE;
							}	
						}else{
							tp_data->state = PARSER_STATE_ERROR;
						}
					}else if( strcmp( el, "whitespaces" ) == 0){	
						if( strcmp( attr[0], "show") == 0 ){
							if( strcmp( attr[1], "true" ) == 0 ){
								cfg->whitespaces = TRUE;
							}else{
								cfg->whitespaces = FALSE;
							}	
						}else{
							tp_data->state = PARSER_STATE_ERROR;
						}
					}else if( strcmp( el, "folding" ) == 0){	
						if( strcmp( attr[0], "enabled") == 0 ){
							if( strcmp( attr[1], "true" ) == 0 ){
								cfg->folding = TRUE;
							}else{
								cfg->folding = FALSE;
							}	
						}else{
							tp_data->state = PARSER_STATE_ERROR;
						}	
					}else if( strcmp( el, "lineswraped" ) == 0){	
						if( strcmp( attr[0], "show") == 0 ){
							if( strcmp( attr[1], "true" ) == 0 ){
								cfg->lineswraped = TRUE;
							}else{
								cfg->lineswraped = FALSE;
							}	
						}else{
							tp_data->state = PARSER_STATE_ERROR;
						}							
					}else if( strcmp( el, "font" ) == 0){	
						for (i = 0; attr[i]; i += 2)
						{ 
							if( strcmp( attr[i], "name") == 0 ){
									cfg->font = g_strdup(attr[i+1]);
							}else if( strcmp( attr[i], "size") == 0 ){
									cfg->font_size = atoi(attr[i+1]);
							}else{
								tp_data->state = PARSER_STATE_ERROR;
							}	
						}
					}else{
						tp_data->state = PARSER_STATE_ERROR;
					}
					break;
				case PARSER_STATE_IN_OPENED_DOCS: // UNFINISHED FIXME 
					if( strcmp( el, "doc" ) == 0){
						if( strcmp( attr[0], "path") == 0 ){
							// value attr[1] true
						}else{
							tp_data->state = PARSER_STATE_ERROR;
						}
					}else {
						tp_data->state = PARSER_STATE_ERROR;
					}
					break;
				case PARSER_STATE_IN_PLUGINS: // UNFINISHED FIXME 
					if( strcmp( el, "plugin" ) == 0){
						if( strcmp( attr[0], "file") == 0 ){
							DBGMSG( "config read plugin found file:\n%s", attr[1] );
							tp_data->plugins_list = g_list_append( tp_data->plugins_list, g_strdup( attr[1] ) );
						}else{
							tp_data->state = PARSER_STATE_ERROR;
						}
					}else {
						tp_data->state = PARSER_STATE_ERROR;
					}
					break;					
				case PARSER_STATE_IN_HIGHLIGHTING:
					if( strcmp( el, "style" ) == 0){
						parse_style_attributes( tp_data, attr );
					}else{
						tp_data->state = PARSER_STATE_ERROR;
					}
					break;
				default:
					tp_data->state = PARSER_STATE_ERROR;
				break;				
			}
		break;
		default:
			// it must go out here
			err_depth = g_strdup_printf( 
							_("Bad XML configuration file\nDepth %d at element %s\nMax deptph must be 3"),
							tp_data->depth, el);
			cssed_error_message (err_depth, _("Configuration file error"));
			g_free(err_depth);
			tp_data->state = PARSER_STATE_ERROR;
		break;
	}
	tp_data->depth++;
}

static void
config_end_tag_cb (void *data, const char *el)
{
	CssedConfigParserData* pdata;
	
	pdata = (CssedConfigParserData*) data; 
	pdata->depth--;
}

void
parse_style_attributes( CssedConfigParserData* data, const char **attr)
{
	CssedWindow* window;
	CssedConfig* cfg;
	const gchar* value = NULL;
	long int fore = -1;
	long int back = -1;
	gboolean bold = FALSE;
	gboolean bold_found = FALSE;
	gint i;
	
	window = data->window;
	cfg = cssed_window_get_config( window );	
	
	for (i = 0; attr[i]; i += 2)
	{
		if( strcmp( attr[i], "name")==0){				
			value = attr[i + 1];			
		}else if( strcmp( attr[i], "fore")==0 ){			
			fore = strtol(attr[i + 1], NULL, 16);
			if( fore == LONG_MIN || fore == LONG_MAX ){
				fore = -1; // out of range
			}
		}else if( strcmp( attr[i], "back")==0 ){
			back = strtol(attr[i + 1], NULL, 16);
			if( back == LONG_MIN || back == LONG_MAX ){
				back = -1; // out of range
			}			
		}else if( strcmp( attr[i], "bold")==0 ){
			bold = (strcmp(attr[i + 1], "true" )==0)?TRUE:FALSE;
			bold_found = TRUE;
		}
	}	

	if( value == NULL){
		data->state = PARSER_STATE_ERROR;
		return;
	}
	
	if( strcmp( value,"DEFAULT") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->DEFAULT_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->DEFAULT_back_color = back;
		if( bold_found )
			cfg->DEFAULT_bold = bold;
	}else if( strcmp(value,"TAG") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->TAG_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->TAG_back_color = back;
		if( bold_found )
			cfg->TAG_bold = bold;
	}else if( strcmp(value,"PSEUDOCLASS") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->PSEUDOCLASS_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->PSEUDOCLASS_back_color = back;
		if( bold_found )
			cfg->PSEUDOCLASS_bold = bold;	
	}else if( strcmp(value,"UNKNOWN_PSEUDOCLASS") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->UNKNOWN_PSEUDOCLASS_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->UNKNOWN_PSEUDOCLASS_back_color = back;
		if( bold_found )
			cfg->UNKNOWN_PSEUDOCLASS_bold = bold;	
	}else if( strcmp(value,"OPERATOR") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->OPERATOR_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->OPERATOR_back_color = back;
		if( bold_found )
			cfg->OPERATOR_bold = bold;
	}else if( strcmp(value,"IDENTIFIER") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->IDENTIFIER_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->IDENTIFIER_back_color = back;
		if( bold_found )
			cfg->IDENTIFIER_bold = bold;
	}else if( strcmp(value,"UNKNOWN_IDENTIFIER") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->UNKNOWN_IDENTIFIER_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->UNKNOWN_IDENTIFIER_back_color = back;
		if( bold_found )
			cfg->UNKNOWN_IDENTIFIER_bold = bold;
	}else if( strcmp(value,"VALUE") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->VALUE_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->VALUE_back_color = back;
		if( bold_found )
			cfg->VALUE_bold = bold;
	}else if( strcmp(value,"COMMENT") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->COMMENT_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->COMMENT_back_color = back;
		if( bold_found )
			cfg->COMMENT_bold = bold;
	}else if( strcmp(value,"IMPORTANT") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->IMPORTANT_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->IMPORTANT_back_color = back;
		if( bold_found )
			cfg->IMPORTANT_bold = bold;
	}else if( strcmp(value,"DIRECTIVE") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->DIRECTIVE_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->DIRECTIVE_back_color = back;
		if( bold_found )
			cfg->DIRECTIVE_bold = bold;
	}else if( strcmp(value,"DOUBLESTRING") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->DOUBLESTRING_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->DOUBLESTRING_back_color = back;
		if( bold_found )
			cfg->DOUBLESTRING_bold = bold;
	}else if( strcmp(value,"SINGLESTRING") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->SINGLESTRING_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->SINGLESTRING_back_color = back;
		if( bold_found )
			cfg->SINGLESTRING_bold = bold;
	}else if( strcmp(value,"CLASS") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->CLASS_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->CLASS_back_color = back;
		if( bold_found )
			cfg->CLASS_bold = bold;
	}else if( strcmp(value,"ID") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->ID_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->ID_back_color = back;
		if( bold_found )
			cfg->ID_bold = bold;
	}else if( strcmp(value,"VALID_VALUE") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->VALID_VALUE_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->VALID_VALUE_back_color = back;
		if( bold_found )
			cfg->VALID_VALUE_bold = bold;
	}else if( strcmp(value,"FUNCTION") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->DEFAULT_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->DEFAULT_back_color = back;
		if( bold_found )
			cfg->DEFAULT_bold = bold;
	}else if( strcmp(value,"NUMBER") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->NUMBER_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->NUMBER_back_color = back;
		if( bold_found )
			cfg->NUMBER_bold = bold;
	}else if( strcmp(value,"UNIT") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->UNIT_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->UNIT_back_color = back;
		if( bold_found )
			cfg->UNIT_bold = bold;
	}else if( strcmp(value,"COLOR") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->COLOR_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->COLOR_back_color = back;
		if( bold_found )
			cfg->COLOR_bold = bold;
	}else if( strcmp(value,"HEXACOLOR") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->HEXACOLOR_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->HEXACOLOR_back_color = back;
		if( bold_found )
			cfg->HEXACOLOR_bold = bold;
	}else if( strcmp(value,"ATTR_MATCH") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->ATTR_MATCH_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->ATTR_MATCH_back_color = back;
		if( bold_found )
			cfg->ATTR_MATCH_bold = bold;
	}else if( strcmp(value,"LANGUAGE") == 0){
		if( fore >= 0 && fore <= 0xffffff )
			cfg->LANGUAGE_fore_color = fore;
		if( back >= 0 && back <= 0xffffff )
			cfg->LANGUAGE_back_color = back;
		if( bold_found )
			cfg->LANGUAGE_bold = bold;
	}else{
		data->state = PARSER_STATE_ERROR;
		return;
	}	
}

