/*
 * $Id: gt_gnutella.c,v 1.65 2003/12/31 18:51:22 hipnod Exp $
 *
 * Copyright (C) 2001-2003 giFT project (gift.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include "gt_gnutella.h"

#include "gt_conf.h"

#include "sha1.h"
#include "xml.h"

#include "gt_share.h"
#include "gt_accept.h"
#include "gt_ban.h"

#include "gt_node.h"
#include "gt_node_list.h"
#include "gt_netorg.h"

#include "gt_xfer_obj.h"
#include "gt_xfer.h"

#include "gt_search.h"
#include "gt_search_exec.h"

#include "gt_web_cache.h"

#include "gt_stats.h"

#include "gt_query_route.h"

#include "transfer/source.h"           /* gnutella_source_{add,remove,cmp} */

/*****************************************************************************/

/* node pointer for this machine */
GtNode         *GT_SELF;

/* giFT protocol pointer */
Protocol       *GT;

/* the static guid identifying this client -- this should remain the same
 * across restarts to give pushes some (probably very slim) chance to operate
 * across them */
gt_guid_t      *GT_SELF_GUID;

/*****************************************************************************/

static time_t   start_time;

/*****************************************************************************/

time_t gt_uptime (void)
{
	return time (NULL) - start_time;
}

/* The ip address is local if the address is local and
 * the source from which it was discovered was not local */
BOOL gt_is_local_ip (in_addr_t ip, in_addr_t src)
{
	if (ip == 0)
		return TRUE;

	if (net_match_host (ip, "LOCAL") &&
	    (src == 0 || !net_match_host (src, "LOCAL")))
	{
		return TRUE;
	}

	return FALSE;
}

/*****************************************************************************/

static char *fw_file (void)
{
	return gift_conf_path ("Gnutella/fwstatus");
}

static void save_fw_status (void)
{
	FILE  *f;

	if (!(f = fopen (fw_file (), "w")))
		return;

	/* 
	 * Store the last successful time of connect.
	 *
	 * But don't do that yet, store 1 or 0 as appropriate for now, 
	 * along with the port number.
	 */
	fprintf (f, "%lu %hu\n", 
	         (long)(GT_SELF->firewalled ? 0 : 1), 
	         GT_SELF->gt_port);

	fclose (f);
}

/* returns whether or not the node is firewalled */
static BOOL load_fw_status (in_port_t port)
{
	FILE      *f;
	char       buf[RW_BUFFER];
	time_t     last_time;
	in_port_t  last_port;

	if (!(f = fopen (fw_file (), "r")))
		return TRUE;

	if (fgets (buf, sizeof (buf) - 1, f) == NULL)
	{
		fclose (f);
		return TRUE;
	}

	fclose (f);

	/* Store the time of the last successful connect, so
	 * > 0 means _not_ firewalled */
	if (sscanf (buf, "%lu %hu", &last_time, &last_port) != 2)
		return TRUE;

	/* if we got a connection at some point with this port, 
	 * mark not firewalled */
	if (last_time > 0 && last_port == port)
		return FALSE;

	return TRUE;
}

/*****************************************************************************/

/* shutdown */
static void gnutella_destroy (Protocol *p)
{
	GT->DBGFN (GT, "entered");

	/* cleanup any network maintenance data */
	gt_netorg_cleanup ();

	/* save the node list to disk */
	gt_node_list_save ();

	/* cleanup any information about banned nodes */
	gt_ban_cleanup ();

	/* destroy query_router tables */
	gt_query_router_self_destroy ();

	/* cleanup remote search data structures */
	gt_search_cleanup ();

	/* cleanup local search data structures */
	gt_search_exec_cleanup ();

	/* cleanup XML structures */
	gt_xml_cleanup ();

	/* save firewalled status */
	save_fw_status ();

	/* free ourself */
	gt_node_free (GT_SELF);
	GT_SELF = NULL;

	/* free and disconnect all nodes */
	gt_node_remove_all ();

	/* destroy web cache information */
	gt_web_cache_cleanup ();

	/* free the client guid */
	free (GT_SELF_GUID);
	GT_SELF_GUID = NULL;

	/* free configuration information */
	gt_config_cleanup ();

	/* set time we were running back to 0 */
	start_time = 0;
}

/*****************************************************************************/

static unsigned char *gnutella_sha1_hash (const char *path, size_t *len)
{
	*len = SHA1_BINSIZE;
	return sha1_digest (path, 0);
}

static char *gnutella_sha1_dsp (unsigned char *hash, size_t len)
{
	return sha1_string (hash);
}

/*****************************************************************************/

static GtNodeClass read_class (void)
{
	char *klass;

	klass = gt_config_get_str ("main/class");

	if (klass && strstr (klass, "ultra"))
		return GT_NODE_ULTRA;

	return GT_NODE_LEAF;
}

static void setup_self (GtNode *node, TCPC *c, in_port_t port)
{
	/* load the firewalled status of this port */
	node->firewalled = load_fw_status (port);

	/* attach the connection to this node */
	gt_node_connect (node, c);
	node->gt_port = port;

	/* load the class for this node */
	node->klass = read_class();

	input_add (c->fd, c, INPUT_READ,
	           (InputCallback)gnutella_handle_incoming, FALSE);
}

static GtNode *bind_gnutella_port (in_port_t port)
{
	GtNode  *node;
	TCPC    *c;

	GT->DBGFN (GT, "entered");

	if (!(node = gt_node_new ()))
		return NULL;

	/* assume sane defaults in case the bind fails */
	node->gt_port    = 0;
	node->firewalled = TRUE;
	node->klass      = GT_NODE_LEAF;

	if (!port || !(c = tcp_bind (port, FALSE)))
	{
		GT->warn (GT, "Failed binding port %d, setting firewalled", port);
		return node;
	}

	GT->dbg (GT, "bound to port %d", port);

	/* setup what will become GT_SELF structure */
	setup_self (node, c, port);

	return node;
}

static void setup_listening_port (in_port_t port)
{
	GT_SELF = bind_gnutella_port (port);

	/*
	 * If running in local mode, let the user set firewalled status
	 * in GNUTELLA_LOCAL_FW. Not sure if this is a good idea, but it makes
	 * testing easier.
	 */
	if (GNUTELLA_LOCAL_MODE)
	{
		if (GNUTELLA_LOCAL_FW)
			GT_SELF->firewalled = TRUE;
		else
			GT_SELF->firewalled = FALSE;
	}
}

static gt_guid_t *get_client_id (char *conf_path)
{
	FILE      *f;
	gt_guid_t *client_id = NULL;
	char      *buf       = NULL;

	if ((f = fopen (gift_conf_path (conf_path), "r")))
	{
		while (file_read_line (f, &buf))
		{
			char *id;
			char *line;

			free (client_id);
			client_id = NULL;

			line = buf;
			id = string_sep_set (&line, "\r\n");

			if (string_isempty (id))
				continue;

			client_id = gt_guid_bin (id);
		}

		fclose (f);
	}

	/* create the id */
	if (!client_id)
	{
		client_id = gt_guid_new ();
		assert (client_id != NULL);
	}

	/* store the id in ~/.giFT/Gnutella/clientid */
	if (!(f = fopen (gift_conf_path (conf_path), "w")))
	{
		GIFT_ERROR (("clientid storage file: %s", GIFT_STRERROR ()));
		return client_id;
	}

	fprintf (f, "%s\n", gt_guid_str (client_id));
	fclose (f);

	return client_id;
}

static void too_old_error_msg (void)
{
	GIFT_ERROR (("\nYour version of the Gnutella plugin is more than 1 year\n"
	             "old.  In order to protect the Gnutella network from \n"
	             "older programs, this plugin has deactivated itself.\n\n"
	             "Please update the plugin with a new version from \n"
	             "http://www.giftproject.org/, or stop running the \n"
	             "plugin by runnning gift-setup or removing \"Gnutella\"\n"
	             "from the /main/plugins line in $HOME/.giFT/giftd.conf\n"
	             "manually.\n\n"
	             "Thanks, and sorry for the inconvenience.\n"));
}

static BOOL gnutella_start (Protocol *p)
{
	int port;

	p->DBGFN (p, "Starting Gnutella plugin");

	/* set the last time we were running to now */
	start_time = time (NULL);

	if (start_time - GT_RELEASE_DATE >= 365 * EDAYS)
	{
		too_old_error_msg ();
		return FALSE;
	}

	if (!gt_config_init ())
	{
		GIFT_ERROR (("Unable to load config file. Please copy it to "
		             "~/.giFT/Gnutella/Gnutella.conf"));
		return FALSE;
	}

	if (!gt_web_cache_init ())
	{
		GIFT_ERROR (("Unable to load gwebcaches file. Please copy it to "
		             "~/.giFT/Gnutella/gwebcaches"));
		return FALSE;
	}

	/* load any banned ip addresses */
	gt_ban_init ();

	port = gt_config_get_int ("main/port=6346");

	/* read the client id from ~/.giFT/Gnutella/clientid or create it */
	GT_SELF_GUID = get_client_id ("Gnutella/clientid");

	/* listen for connections */
	setup_listening_port (port);

	/* load the list of all previously contacted nodes */
	gt_node_list_load ();

	/* initialize query router tables */
	gt_query_router_self_init ();

	/* initialize the local search data structures */
	gt_search_exec_init ();

	/* initialize the remote search data structures */
	gt_search_init ();

	/* initialize support for xml metadata */
	gt_xml_init ();

	/* startup network maintenance */
	gt_netorg_init ();

	return TRUE;
}

/*
 * The entry-point for the giFT daemon
 */
BOOL Gnutella_init (Protocol *p)
{
	if (protocol_compat (p, LIBGIFTPROTO_MKVERSION (0, 11, 4)) != 0)
		return FALSE;

	p->version_str = STRDUP (GT_VERSION);

	GT = p;

	/* describe the hash algo */
	p->hash_handler (p, "SHA1", HASH_PRIMARY, gnutella_sha1_hash,
	                 gnutella_sha1_dsp);

	/* gt_gnutella.c: */
	p->start          = gnutella_start;
	p->destroy        = gnutella_destroy;

	/* gt_search.c: */
	p->search         = gnutella_search;
#if 0
	p->browse         = gnutella_browse;
#endif
	p->locate         = gnutella_locate;
	p->search_cancel  = gnutella_search_cancel;

	/* gt_xfer.c: */
	p->download_start = gnutella_download_start;
	p->download_stop  = gnutella_download_stop;
	p->upload_stop    = gnutella_upload_stop;
	p->chunk_suspend  = gnutella_chunk_suspend;
	p->chunk_resume   = gnutella_chunk_resume;

	/* transfer/source.c: */
	p->source_cmp     = gnutella_source_cmp;
	p->source_add     = gnutella_source_add;
	p->source_remove  = gnutella_source_remove;

#if 0
	p->upload_avail   = gnutella_upload_avail;
	p->user_cmp       = gnutella_user_cmp;
#endif

	/* gt_share.c: */
	p->share_new      = gnutella_share_new;
	p->share_free     = gnutella_share_free;
	p->share_add      = gnutella_share_add;
	p->share_remove   = gnutella_share_remove;
	p->share_sync     = gnutella_share_sync;

	/* gt_stats.c: */
	p->stats          = gnutella_stats;

	return TRUE;
}
