
#include "gt_gnutella.h"
#include "gt_urn.h"
#include "sha1.h"

/*****************************************************************************/

#define URN_PREFIX_LEN        (sizeof ("urn:")-1)
#define SHA1_PREFIX_LEN       (sizeof ("sha1:")-1)
#define BITPRINT_PREFIX_LEN   (sizeof ("bitprint:")-1)

/*****************************************************************************/

enum urn_types
{
	GT_URN_SHA1     = 0,
	GT_URN_BITPRINT = 1,   /* for now, collapse bitprint to sha1 */
};

/*****************************************************************************/

/* XXX */
static char *base32_alpha = "ABCDEFGHIJKLMNOPQRSTUVWXYZ234567";

/*****************************************************************************/

static uint32_t get_urn_type (gt_urn_t *urn)
{
	return net_get32 (urn, FALSE);
}

static void set_urn_type (gt_urn_t *urn, enum urn_types t)
{
	uint32_t tmp = t;

	net_put32 (urn, tmp);
}

static unsigned char *get_urn_data (gt_urn_t *urn)
{
	return urn + 4;
}

static void set_urn_data (gt_urn_t *urn, unsigned char *data, size_t len)
{
	memcpy (get_urn_data (urn), data, len);
}

/* 
 * WARNING: frees data if realloc failed.
 */
static gt_urn_t *urn_from_data (unsigned char *data, size_t len,
                                enum urn_types t)
{
	gt_urn_t *new_urn;

	if (!(new_urn = realloc (data, len + 4)))
	{
		free (data);
		return NULL;
	}

	memmove (get_urn_data (new_urn), new_urn, len);

	set_urn_type (new_urn, t);

	return new_urn;
}

/*****************************************************************************/

static size_t bin_length (enum urn_types t)
{
	switch (t)
	{
	 case GT_URN_BITPRINT:
	 case GT_URN_SHA1: return SHA1_BINSIZE;
	 default:          return 0;
	}
}

static gt_urn_t *sha1_urn_new (unsigned char *data)
{
	gt_urn_t *new_urn;

	if (!(new_urn = malloc (SHA1_BINSIZE + 4)))
		return NULL;

	/* put the identifier at the beginning */
	set_urn_type (new_urn, GT_URN_SHA1);
	
	/* copy the data */
	set_urn_data (new_urn, data, SHA1_BINSIZE);

	return new_urn;
}

gt_urn_t *gt_urn_new (char *urn_type, unsigned char *data)
{
	if (!strcasecmp (urn_type, "urn:sha1"))
		return sha1_urn_new (data);

	return NULL;
}

unsigned char *gt_urn_data (gt_urn_t *urn)
{
	if (!urn)
		return NULL;

	return get_urn_data (urn);
}

char *gt_urn_string (gt_urn_t *urn)
{
	unsigned char  *data;
	char           *str;
	char           *urn_str;
	int             prefix_len;
	int             len;

	/* 
	 * This is the same for bitprint and sha1 urns, because we convert
	 * to sha1 anyway.
	 */

	if (!(data = gt_urn_data (urn)))
		return NULL;

	if (!(str = sha1_string (data)))
		return NULL;

	if (!(urn_str = realloc (str, SHA1_STRLEN+1)))
	{
		free (str);
		return NULL;
	}

	prefix_len = URN_PREFIX_LEN + SHA1_PREFIX_LEN;

	memmove (urn_str + prefix_len, urn_str, SHA1_STRLEN);
	memcpy (urn_str, "urn:sha1:", prefix_len);

	len = prefix_len + SHA1_STRLEN;
	urn_str[len] = 0;

	return urn_str;
}

/*****************************************************************************/

static BOOL sha1_ensure_len (char *base32)
{
	char *s;

	s = base32;

	while (*s && strchr (base32_alpha, toupper (*s)))
		s++;

	if (s - base32 != 32)
		return FALSE;

	return TRUE;
}

gt_urn_t *sha1_urn_parse (char *base32)
{
	unsigned char *bin;
	gt_urn_t      *urn;

	/* make sure the hash is the right length */
	if (!sha1_ensure_len (base32))
		return NULL;

	if (!(bin = sha1_bin (base32)))
		return NULL;

	if (!(urn = urn_from_data (bin, SHA1_BINSIZE, GT_URN_SHA1)))
		return NULL;

	return urn;
}

/*
 * Bitprint urns are the format:
 *
 *      urn:bitprint:[32-character SHA1].[39-character TigerTree]
 *
 * We use the sha1 parsing and truncate the tigertree for now.
 */
gt_urn_t *bitprint_urn_parse (char *base32)
{
	return sha1_urn_parse (base32);
}

gt_urn_t *gt_urn_parse (char *str)
{
	if (strncasecmp ("urn:", str, URN_PREFIX_LEN) != 0)
		return NULL;

	str += URN_PREFIX_LEN;

	if (!strncasecmp (str, "sha1:", SHA1_PREFIX_LEN))
		return sha1_urn_parse (str + SHA1_PREFIX_LEN);

	if (!strncasecmp (str, "bitprint:", BITPRINT_PREFIX_LEN))
		return bitprint_urn_parse (str + BITPRINT_PREFIX_LEN);

	return NULL;
}

/*****************************************************************************/

int gt_urn_cmp (gt_urn_t *a, gt_urn_t *b)
{
	int ret;

	if (!a || !b)
		return -1;

	if ((ret = memcmp (a, b, 4)))
		return ret;

	return memcmp (a + 4, b + 4, bin_length (get_urn_type (a)));
}
