/*
 * $Id: tx_stack.c,v 1.5 2003/12/23 20:24:17 hipnod Exp $
 *
 * Copyright (C) 2003 giFT project (gift.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#include "gt_gnutella.h"

#include "tx_stack.h"
#include "tx_layer.h"

#include "io_buf.h"

#include "gt_packet.h"               /* gt_packet_log XXX */

/*****************************************************************************/

extern struct tx_layer_ops tx_link_ops;

/*****************************************************************************/

struct gt_tx_stack
{
	struct tx_layer  *layers;
	GtTxStackCleanup  cleanup;

	TCPC             *c;
	void             *udata;
};

static struct use_tx_layer
{
	const char          *name;
	struct tx_layer_ops *ops;
} tx_layers[] =
{
	{ "tx_link",   &tx_link_ops,   },
};

/*****************************************************************************/

static void foreach_tx_child (struct tx_layer *tx, 
                              void (*exec) (struct tx_layer *tx))
{
	struct tx_layer *next;

	while (tx != NULL)
	{
		/* grab the next element first so the callback can call free */
		next = tx->lower;

		exec (tx);

		tx = next;
	}
}

static struct tx_layer *tx_push_layer (struct tx_layer *below, 
                                       struct tx_layer *above)
{
	if (above)
		above->lower = below;

	if (below)
		below->upper = above;

	return above;
}

static void destroy_tx (struct tx_layer *tx)
{
	tx_layer_free (tx);
}

static void disable_tx (struct tx_layer *tx)
{
	tx_layer_disable (tx);
}

static void disable_all_tx_layers (struct tx_layer *layers)
{
	if (!layers)
		return;

	assert (layers->upper == NULL);

	foreach_tx_child (layers, disable_tx);
}

static void free_all_tx_layers (struct tx_layer *layers)
{
	if (!layers)
		return;

	disable_all_tx_layers (layers);
	foreach_tx_child (layers, destroy_tx);
}

static struct tx_layer *alloc_tx_layers (GtTxStack *stack, BOOL tx_deflated)
{
	struct tx_layer *new_layer;
	struct tx_layer *layer      = NULL;
	int i;

	for (i = 0; i < sizeof(tx_layers) / sizeof(tx_layers[0]); i++)
	{
		if (!(new_layer = tx_layer_new (stack, tx_layers[i].name, 
		                                tx_layers[i].ops)))
		{
			foreach_tx_child (layer, destroy_tx);
			return NULL;
		}

		layer = tx_push_layer (layer, new_layer);
	}

	return layer;
}

GtTxStack *gt_tx_stack_new (TCPC *c, BOOL tx_deflated)
{
	struct gt_tx_stack *stack;
	int                 size;

	if (!(stack = NEW (struct gt_tx_stack)))
		return NULL;

	if (!(stack->layers = alloc_tx_layers (stack, tx_deflated)))
	{
		free (stack);
		return NULL;
	}

	/* set the send buffer to a not too high value */
	size = 256;

	if (setsockopt (c->fd, SOL_SOCKET, SO_SNDBUF, &size, sizeof (size)) != 0)
		GT->DBGSOCK (GT, c, "Error setting sndbuf size: %s", GIFT_NETERROR());

	stack->c = c;

	return stack;
}

/*****************************************************************************/

void gt_tx_stack_free (GtTxStack *stack)
{
	if (!stack)
		return;

	free_all_tx_layers (stack->layers);
	FREE (stack);
}

static void cleanup_tx_notify (GtTxStack *stack)
{
	stack->cleanup (stack, stack->udata);
}

void gt_tx_stack_abort (GtTxStack *stack)
{
	cleanup_tx_notify (stack);
}

/*****************************************************************************/

BOOL gt_tx_stack_queue (GtTxStack *stack, uint8_t *data, size_t len)
{
	struct io_buf   *io_buf;
	struct tx_layer *tx;
	uint8_t         *ptr;

	if (!(io_buf = io_buf_new (len)))
		return FALSE;

	ptr = io_buf_write_ptr (io_buf);

	memcpy (ptr, data, len);
	io_buf_push (io_buf, len);

	tx = stack->layers;

	/* send the data on its way down the stack */
	tx->ops->send (tx, io_buf);

	return TRUE;
}

int gt_tx_stack_send (GtTxStack *stack, uint8_t *data, size_t len)
{
	int   ret;

	/* check if the file descriptor has an error */
	if (net_sock_error (stack->c->fd))
		return -1;

	ret = tcp_send (stack->c, data, len);

	/* XXX: we may not even be sending packets, so weshould add another
	 * function callback for logging purposes */
	if (ret == len)
	{
		GtPacket pkt;

		pkt.data = data;
		pkt.len  = len;

		gt_packet_log (&pkt, stack->c, TRUE);
	}

	return ret;
}

input_id gt_tx_stack_input_add (GtTxStack *stack, InputCallback cb, void *udata)
{
	return input_add (stack->c->fd, udata, INPUT_WRITE,
	                  cb, 0);
}

void gt_tx_stack_set_handler (GtTxStack *stack, GtTxStackCleanup cleanup, 
                              void *udata)
{
	stack->cleanup = cleanup;
	stack->udata   = udata;
}
