/*
 *  playlist-ogg.c
 *  mod_musicindex
 *
 *  $Id: playlist-flac.c,v 1.1 2004/07/06 17:14:42 boudinr Exp $
 *
 *  Created by Regis BOUDIN on Thu Jun 30 2004.
 *  Copyright (c) 2004 Regis BOUDIN
 *   
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1, or (at your option)
 *  any later version.
 *
 */

/**
 * @file 
 * Flac files management system.
 *
 * @author Regis Boudin
 * @version $Revision: 1.1 $
 * @date 2004
 *
 * This file contains everything needed to produce music entries from
 * Flac files.
 */

#include "playlist.h"
#include "playlist-flac.h"

/**
 * Fills in the information fields about FLAC data.
 *
 * This function reads the ogg tags (using libFLAC) from the flac file
 * and fills in the struct mu_ent fields accordingly. If the file is not an
 * ogg one, the function returns head.
 * 
 * @param pool Pool
 * @param head Head
 * @param in ogg file to parse (closed on normal exit)
 * @param conf MusicIndex configuration paramaters struct
 * @param names Names
 * @param r Apache request_rec struct to handle log writings (debugging)
 *
 * @return When possible, struct mu_ent correctly set up, file stream closed.
 */
mu_ent *make_flac_entry(apr_pool_t *pool, mu_ent *head,
	FILE *in, mu_config *conf, mu_ent_names *names, request_rec *r)
{
	mu_ent			*p = head;
	struct stat		filestat;
	FLAC__StreamMetadata streaminfo;
	FLAC__Metadata_SimpleIterator *iterator;
	
	if (FLAC__metadata_get_streaminfo(names->filename, &streaminfo) == 0)
		return head;
	
	p = new_ent(pool, head);
	p->filetype = FT_FLAC;
	
	fstat(fileno(in), &filestat);
	p->size = filestat.st_size;
	p->mtime = filestat.st_mtime;
	
	p->bitrate = (8 * filestat.st_size) / (streaminfo.data.stream_info.total_samples / streaminfo.data.stream_info.sample_rate);
	p->length = (int)(streaminfo.data.stream_info.total_samples / streaminfo.data.stream_info.sample_rate);

	iterator = FLAC__metadata_simple_iterator_new();
	
	fclose(in);
	
	if(iterator == NULL)
		return p;
	
	if(FLAC__metadata_simple_iterator_init(iterator, names->filename, true, true)) {
		
		FLAC__StreamMetadata *block = NULL;
		const FLAC__StreamMetadata_VorbisComment *vc = NULL;
		unsigned short i;
		do {
			if(FLAC__metadata_simple_iterator_get_block_type(iterator) != FLAC__METADATA_TYPE_VORBIS_COMMENT)
				continue;
				
			block = FLAC__metadata_simple_iterator_get_block(iterator);
			if(block == NULL)
				continue;
			
			vc = &block->data.vorbis_comment;
			for(i = 0; i < vc->num_comments; i++) {
				if (!strncasecmp((char *)vc->comments[i].entry, "album=", 6))
					p->album = ap_pstrndup(pool, (char *)vc->comments[i].entry+6, vc->comments[i].length-6);
				else if (!strncasecmp((char *)vc->comments[i].entry, "artist=", 7))
					p->artist = ap_pstrndup(pool, (char *)vc->comments[i].entry+7, vc->comments[i].length-7);
				else if (!strncasecmp((char *)vc->comments[i].entry, "title=", 6))
					p->title = ap_pstrndup(pool, (char *)vc->comments[i].entry+6, vc->comments[i].length-6);
				else if (!strncasecmp((char *)vc->comments[i].entry, "tracknumber=", 12))
					p->track = atoi((char *)vc->comments[i].entry+12);
				else if (!strncasecmp((char *)vc->comments[i].entry, "date=", 5))
					p->date = atoi((char *)vc->comments[i].entry+5);
			}
			FLAC__metadata_object_delete(block);
			
			break;
		} while (FLAC__metadata_simple_iterator_next(iterator));
	}
	FLAC__metadata_simple_iterator_delete(iterator);

	return p;
}
