/* 
**  mod_random.c 
*/

#include "httpd.h"
#include "http_config.h"
#include "http_protocol.h"
#include "ap_config.h"
#include "http_log.h"
#include "fnmatch.h"
#include <cpio.h>

#define WATCHPOINT printf("WATCHPOINT %s %d\n", __FILE__, __LINE__);

typedef struct {
	int enabled;
	array_header *urls;
	array_header *section_quotes;
	array_header *ads;
	table *handlers;
} random_conf;

typedef struct {
	int count;
	const char *name;
	array_header *text;
} message_bank;

module MODULE_VAR_EXPORT random_module;

message_bank * rn_get_bank(array_header *bank, const char *key) {
	int x;
	message_bank **members = (message_bank **) bank->elts;

	for(x = 0; x < bank->nelts; x++) {
		if(!strcmp(key, members[x]->name)) {
			return members[x];
		}
	}

	return NULL;
}

message_bank * rn_create_bank(pool *p, char *section) {
	message_bank *bank;

	if(!section) 
		return NULL;

	bank = ap_pcalloc (p, sizeof (message_bank));
	bank->name = ap_pstrdup (p, section);
	bank->text = ap_make_array (p, 5, sizeof (char *));

	return bank;
}

int rn_check_table(const char *a) {
	if (a == NULL) 
		return 0;
	if('1' == a[0])
		return 1;

	return 0;
}

int rn_table_find(const table * t, const char *key) {
	array_header *hdrs_arr = ap_table_elts(t);
	table_entry *elts = (table_entry *) hdrs_arr->elts;
	int i;

	if (t == NULL)
		return 0;
	if (key == NULL)
		return 0;

	for (i = 0; i < hdrs_arr->nelts; ++i) {
		if (!ap_fnmatch(elts[i].key, key, FNM_CASE_BLIND))
			if(rn_check_table(elts[i].val))
				return 1;
	}

	return 0;
}

char * rn_add_file(cmd_parms * cmd, char *file) {
	FILE *file_ptr;
	char buf[HUGE_STRING_LEN];
	char *content = NULL;

	if (!(file_ptr = ap_pfopen(cmd->temp_pool, file, "r"))) {
		ap_log_error(APLOG_MARK, APLOG_ERR, cmd->server,
			"Could not open random ad file: %s", file);
		return NULL;
	}
	while (fgets(buf, HUGE_STRING_LEN, file_ptr)) {
		if(content) {
			content = ap_pstrcat(cmd->temp_pool, content, buf, NULL);
		} else {
			content = ap_pstrcat(cmd->temp_pool, buf, NULL);
		}
	}
	ap_pfclose(cmd->temp_pool, file_ptr);

	return content;
}

static int random_fixup(request_rec * r) {
	int token = 0;
	random_conf *cfg = ap_get_module_config (r->per_dir_config, &random_module);
	int x = 0;
	char *type = NULL;
	message_bank **members = NULL;
	message_bank *bank = NULL;
	char **quotes = NULL;


	if(!cfg->enabled)
		return DECLINED;

	if(cfg->handlers) {
		if(r->handler) {
			type = ap_pstrdup(r->pool, r->handler);
		} else {
			type = ap_pstrdup(r->pool, r->content_type);
		}

		if(!rn_table_find(cfg->handlers, type)) {
			return DECLINED;
		}
	}

	if(cfg->section_quotes) {
		members = (message_bank **) cfg->section_quotes->elts;
		for(x = 0; x < cfg->section_quotes->nelts; x++) {
			bank = members[x];
			if(bank->text->nelts) {
				token = ((int)random() % bank->text->nelts);
				quotes = (char **) bank->text->elts;
				ap_table_setn(r->subprocess_env, bank->name, quotes[token]);
				ap_table_setn(r->notes, bank->name, quotes[token]);
			}
		}
	} 
	if(cfg->ads) {
		members = (message_bank **) cfg->ads->elts;
		for(x = 0; x < cfg->ads->nelts; x++) {
			bank = members[x];
			if(bank->text->nelts) {
				token = ((int)random() % bank->text->nelts);
				quotes = (char **) bank->text->elts;
				ap_table_setn(r->subprocess_env, bank->name, quotes[token]);
				ap_table_setn(r->notes, bank->name, quotes[token]);
			}
		}
	} 


	return DECLINED;
}

static int random_handler(request_rec * r) {
	int token = 0;

	random_conf *cfg = ap_get_module_config (r->per_dir_config, &random_module);
	char **members = NULL;

	/* We need to be sending a better warning */
	if(cfg->urls == NULL) 
		return HTTP_NOT_FOUND;

	members = (char **) cfg->urls->elts;

	if(!cfg->urls->nelts) 
		return HTTP_NOT_FOUND;

	token = ((int)random() % cfg->urls->nelts);
	ap_table_setn(r->headers_out, "Cache-Control", "no-cache");
	ap_table_setn (r->headers_out, "Location", members[token]);

	return HTTP_MOVED_TEMPORARILY;
}

static int random_page_handler(request_rec * r) {
	const char *string = NULL;

	if(!strcmp(r->handler,"random-quote-page")) {
		string = ap_table_get(r->notes, "RANDOM_QUOTE");
	} else {
		string = ap_table_get(r->notes, "RANDOM_AD");
	}

	if(!string)
		return HTTP_NOT_FOUND;

	r->content_type = "text/html";      
	ap_send_http_header(r);

	if (r->header_only)
		return OK;

	ap_rputs(string, r);
	return OK;
}

/* Dispatch list of content handlers */
static const handler_rec random_handlers[] = {
	{"random", random_handler},
	{"random-ad-page", random_page_handler},
	{"random-quote-page", random_page_handler},
	{NULL, NULL}
};

static void * mconfig_for_directory(pool *p, char *dir) {
	random_conf *cfg = ap_pcalloc (p, sizeof (random_conf));
	int timz;
	struct tm *t;

	cfg->urls = NULL;
	cfg->section_quotes = NULL;
	cfg->ads = NULL;

	cfg->handlers = NULL;

	t = ap_get_gmtoff (&timz);
	srandom (t->tm_sec);
	return (void *) cfg;
}	


static const char * add_random_url(cmd_parms * cmd, void *mconfig, char *param) {
	FILE *file_ptr;
	char buf[HUGE_STRING_LEN];
	random_conf *cfg = (random_conf *) mconfig;
	message_bank *bank;
	struct stat sbuf;

	if(cfg->urls == NULL)
		cfg->urls = ap_make_array (cmd->pool, 5, sizeof (char *));

	if(stat(param, &sbuf) == 0){
		if (!(file_ptr = ap_pfopen (cmd->pool, param, "r"))) {
			ap_log_error (APLOG_MARK, APLOG_ERR, cmd->server,
				"Could not open RandomFile: %s", param);
			return NULL;
		}
		while (fgets (buf, sizeof (buf), file_ptr)) {
			*(char **) ap_push_array (cfg->urls) = ap_pstrdup (cmd->pool, buf);
		}
		ap_pfclose (cmd->pool, file_ptr);
	} else {
		*(char **) ap_push_array (cfg->urls) = ap_pstrdup (cmd->pool, param);
	}


	return NULL;
}

static const char * add_random_quote(cmd_parms * cmd, void *mconfig, char *param, char *section) {
	FILE *file_ptr;
	char buf[HUGE_STRING_LEN];
	random_conf *cfg = (random_conf *) mconfig;
	message_bank *bank;
	struct stat sbuf;

	if(cfg->section_quotes == NULL)
		cfg->section_quotes = ap_make_array (cmd->pool, 5, sizeof (message_bank *));

	/* We default to RANDOM_QUOTE if nothing is specified */
	section = section ? section : "RANDOM_QUOTE";

	if((bank = rn_get_bank(cfg->section_quotes, section)) == NULL) {
		bank = rn_create_bank(cmd->pool, section);
		*(message_bank **) ap_push_array (cfg->section_quotes) = (message_bank *) bank;
	}

	if(stat(param, &sbuf) == 0){
		if (!(file_ptr = ap_pfopen (cmd->pool, param, "r"))) {
			ap_log_error (APLOG_MARK, APLOG_ERR, cmd->server,
				"Could not open RandomFile: %s", param);
			return NULL;
		}
		while (fgets (buf, sizeof (buf), file_ptr)) {
			*(char **) ap_push_array (bank->text) = ap_pstrdup (cmd->pool, buf);
		}
		ap_pfclose (cmd->pool, file_ptr);
	} else {
		*(char **) ap_push_array (bank->text) = ap_pstrdup (cmd->pool, param);
	}


	return NULL;
}

static const char * add_random_ad(cmd_parms * cmd, void *mconfig, char *param, char *section) {
	FILE *file_ptr;
	char buf[HUGE_STRING_LEN];
	random_conf *cfg = (random_conf *) mconfig;
	message_bank *bank;
	char *temp=NULL;
	struct stat sbuf;
	DIR *directory;
	struct dirent *file = NULL;

	if(cfg->ads == NULL)
		cfg->ads = ap_make_array (cmd->pool, 5, sizeof (message_bank *));

	/* We default to RANDOM_AD if nothing is specified */
	section = section ? section : "RANDOM_AD";

	if((bank = rn_get_bank(cfg->ads, section)) == NULL) {
		bank = rn_create_bank(cmd->pool, section);
		*(message_bank **) ap_push_array (cfg->ads) = (message_bank *) bank;
	}

	if(stat(param, &sbuf) == 0){
		if(S_ISDIR(sbuf.st_mode)) {
			if (!(directory = opendir(param))) {
				ap_log_error (APLOG_MARK, APLOG_ERR, cmd->server,
					"Could not open directory: %s", param);
				return NULL;
			}
			while(file = readdir(directory)) {
				if(stat(ap_pstrcat(cmd->temp_pool, param, "/", file->d_name, NULL), &sbuf) == 0){
					if(S_ISREG(sbuf.st_mode)) {
						temp = rn_add_file(cmd, ap_pstrcat(cmd->temp_pool, param, "/", file->d_name, NULL));
						if(temp) {
							*(char **) ap_push_array (bank->text) = ap_pstrdup (cmd->pool, temp);
						}	
					}
				}
			}
			closedir(directory);
		
		} else {
			temp = rn_add_file(cmd, param);
			if(temp) 
				*(char **) ap_push_array (bank->text) = ap_pstrdup (cmd->pool, temp);
		}
	} else {
		*(char **) ap_push_array (bank->text) = ap_pstrdup (cmd->pool, param);
	}


	return NULL;
}

static const char * add_handler(cmd_parms * cmd, void *mconfig, char *handler) {
	random_conf *cfg = (random_conf *) mconfig;

	if(!cfg->handlers)
		ap_make_table(cmd->pool, 3);

	ap_table_set(cfg->handlers, handler, "1");

	return NULL;
}

static const command_rec random_module_cmds[] = {
	{"RandomEngine", ap_set_flag_slot, 
		(void *) XtOffsetOf(random_conf, enabled), OR_ALL, FLAG,
		"Use this to turn on and off random quotes."},
	{"RandomURL", add_random_url, NULL, OR_ALL, TAKE1,
		"A filename with one URL per-line."},
	{"RandomQuote", add_random_quote, NULL, OR_ALL, TAKE12,
		"Takes either a double quoted string or a filename. An optional second parameter lets you adjust whate section the quote is added to."},
	{"RandomAd", add_random_ad, NULL, OR_ALL, TAKE12,
		"Takes either a double quoted string, a filename, or a directory name to read files from. An optional second parameter lets you adjust whate section the ad is added to."},
	{"RandomHandler", add_handler, NULL, OR_ALL, TAKE1,
		"Enable which handled types will be supplied with ads or quotes."},
	{NULL},
};

/* Dispatch list for API hooks */
module MODULE_VAR_EXPORT random_module = {
	STANDARD_MODULE_STUFF,
	NULL,				/* module initializer                  */
	mconfig_for_directory,	/* create per-dir    config structures */
	NULL,			/* merge  per-dir    config structures */
	NULL,				/* create per-server config structures */
	NULL,				/* merge  per-server config structures */
	random_module_cmds,		/* table of config file commands       */
	random_handlers,		/* [#8] MIME-typed-dispatched handlers */
	NULL,				/* [#1] URI to filename translation    */
	NULL,				/* [#4] validate user id from request  */
	NULL,				/* [#5] check if the user is ok _here_ */
	NULL,				/* [#3] check access by host address   */
	NULL,				/* [#6] determine MIME type            */
	random_fixup,	/* [#7] pre-run fixups                 */
	NULL,				/* [#9] log a transaction              */
	NULL,				/* [#2] header parser                  */
	NULL,				/* child_init                          */
	NULL,				/* child_exit                          */
	NULL				/* [#0] post read-request              */
#ifdef EAPI
	,NULL,                  /* EAPI: add_module                    */
	NULL,                  /* EAPI: remove_module                 */
	NULL,                  /* EAPI: rewrite_command               */
	NULL                   /* EAPI: new_connection                */
#endif
};
