/***************************************************************************
 $RCSfile: readerserver.c,v $
 -------------------
 cvs         : $Id: readerserver.c,v 1.15 2003/05/08 21:47:36 aquamaniac Exp $
 begin       : Sat Jan 11 2003
 copyright   : (C) 2003 by Martin Preuss
 email       : martin@libchipcard.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_CHIPCARD_DLL
#  define CHIPCARD_API __declspec (dllexport)
# else /* Not BUILDING_CHIPCARD_DLL */
#  define CHIPCARD_API __declspec (dllimport)
# endif /* Not BUILDING_CHIPCARD_DLL */
#else
# define CHIPCARD_API
#endif


#include "readerserver.h"
#include "readerservice.h"
#include "libchipcard.h"
#include <chameleon/debug.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>


READERSERVERDATA *ReaderServer_new(){
  READERSERVERDATA *rsd;

  rsd=(READERSERVERDATA *)malloc(sizeof(READERSERVERDATA));
  assert(rsd);
  memset(rsd,0,sizeof(READERSERVERDATA));
  return rsd;
}


void ReaderServer_free(READERSERVERDATA *rsd){
  if (rsd) {
    if (rsd->core)
      CTCore_free(rsd->core);
    free(rsd);
  }
}




/*___________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                          Wait-For-Reader-Status
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */

READERSERVERWAITDATA *ReaderServer__WaitRequest_new(){
  READERSERVERWAITDATA *wr;

  wr=(READERSERVERWAITDATA *)malloc(sizeof(READERSERVERWAITDATA));
  assert(wr);
  memset(wr,0,sizeof(READERSERVERWAITDATA));
  return wr;
}


void ReaderServer__WaitRequest_free(READERSERVERWAITDATA *wr){
  if (wr) {
    free(wr->typeName);
    free(wr);
  }
}


void ReaderServer__WaitRequest_add(READERSERVERWAITDATA *wr,
				   READERSERVERWAITDATA **head){
  READERSERVERWAITDATA *curr;

  DBG_ENTER;
  assert(wr);
  assert(head);

  curr=*head;
  if (!curr) {
    *head=wr;
  }
  else {
    /* find last */
    while(curr->next) {
      curr=curr->next;
    } /* while */
    curr->next=wr;
  }
  DBG_LEAVE;
}


void ReaderServer__WaitRequest_del(READERSERVERWAITDATA *wr,
				   READERSERVERWAITDATA **head){
  READERSERVERWAITDATA *curr;

  DBG_ENTER;
  assert(wr);
  assert(head);

  curr=*head;
  if (curr) {
    if (curr==wr) {
      *head=curr->next;
    }
    else {
      /* find predecessor */
      while(curr->next!=wr) {
	curr=curr->next;
      } /* while */
      if (curr)
	curr->next=wr->next;
    }
  }
  DBG_LEAVE;
}


READERSERVERWAITDATA *ReaderServer__WaitRequest_find(READERSERVERWAITDATA *wr,
						     int mlid, int reqid){
  DBG_ENTER;

  while(wr) {
    if (wr->msgLayerId==mlid &&
	wr->requestId==reqid)
      return wr;
    wr=wr->next;
  }
  DBG_LEAVE;
  return 0;
}




/*___________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                          Wait-For-Reader-Unlock
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


READERSERVEROPENDATA *ReaderServer__OpenRequest_new(){
  READERSERVEROPENDATA *wr;

  wr=(READERSERVEROPENDATA *)malloc(sizeof(READERSERVEROPENDATA));
  assert(wr);
  memset(wr,0,sizeof(READERSERVEROPENDATA));
  return wr;
}


void ReaderServer__OpenRequest_free(READERSERVEROPENDATA *wr){
  free(wr);
}


void ReaderServer__OpenRequest_add(READERSERVEROPENDATA *wr,
				   READERSERVEROPENDATA **head){
  READERSERVEROPENDATA *curr;

  DBG_ENTER;
  assert(wr);
  assert(head);

  curr=*head;
  if (!curr) {
    *head=wr;
  }
  else {
    /* find last */
    while(curr->next) {
      curr=curr->next;
    } /* while */
    curr->next=wr;
  }
  DBG_LEAVE;
}


void ReaderServer__OpenRequest_del(READERSERVEROPENDATA *wr,
				   READERSERVEROPENDATA **head){
  READERSERVEROPENDATA *curr;

  DBG_ENTER;
  assert(wr);
  assert(head);

  curr=*head;
  if (curr) {
    if (curr==wr) {
      *head=curr->next;
    }
    else {
      /* find predecessor */
      while(curr->next!=wr) {
	curr=curr->next;
      } /* while */
      if (curr)
	curr->next=wr->next;
    }
  }
  DBG_LEAVE;
}


READERSERVEROPENDATA *ReaderServer__OpenRequest_find(READERSERVEROPENDATA *wr,
						     int mlid, int reqid){
  DBG_ENTER;

  while(wr) {
    if (wr->msgLayerId==mlid &&
	wr->requestId==reqid)
      return wr;
    wr=wr->next;
  }
  DBG_LEAVE;
  return 0;
}








ERRORCODE ReaderServer_AddReader(CTSERVERDATA *sd, CTREADERDESCRIPTION *rd){
  ERRORCODE err;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);
  assert(rd);
  assert(rsd->core);
  err=CTCore_AddReader(rsd->core, rd);
  if (!Error_IsOk(err)) {
    DBG_ERROR("Could not add reader \"%s\"", rd->name);
  }
  DBG_LEAVE;
  return err;
}


ERRORCODE ReaderServer_AllocAllReaders(CTSERVERDATA *sd){
  CTREADERDESCRIPTION *rd;
  int badterms;
  int goodterms;
  READERSERVERDATA *rsd;

  badterms=0;
  goodterms=0;

  /* allocate all available readers */
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);
  assert(rsd->core);
  rd=rsd->core->readerDescriptions;
  while(rd) {
    int tid;
    CTREADERDESCRIPTION *descr;

    /* Allocate this terminal for the server, so that it's status will
     * be checked periodically */
    tid=CTCore_AllocTerminal(rsd->core,
			     rsd->coreClientId,
			     rd->id,
			     &descr);
    if (tid==-1) {
      DBG_ERROR("Could not allocate reader \"%s\"", rd->name);
      badterms++;
    }
    else {
      DBG_INFO("Allocated terminal \"%s\" for server", rd->name);
      goodterms++;
    }
    rd=rd->next;
  } /* while */
  if (!goodterms) {
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_NO_READER);
  }
  if (badterms) {
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTCORE_ERROR_TYPE),
		     CTCORE_ERROR_BAD_READER);
  }
  return 0;
}




/*_________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *           Handler for the various supported command messages
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


ERRORCODE ReaderServer__SendErrorMessage(CTSERVERDATA *sd,
					 IPCMESSAGELAYER *ml,
					 IPCMESSAGE *req,
					 ERRORCODE errcode) {
  ERRORCODE err;
  IPCMESSAGE *msg;
  READERSERVERDATA *rsd;
  int newerrcode;
  char errorbuffer[256];
  int requestid;
  int i;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  /* get request id */
  err=IPCMessage_FirstIntParameter(req, &i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  if (i>=0x10000) {
    /* open message, so no request id */
    requestid=0;
  }
  else {
    /* requestid */
    err=IPCMessage_IntParameter(req, 2, &requestid);
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      DBG_LEAVE;
      return err;
    }
  }

  /* create response */
  msg=CTService_Message_Create(CTSERVICE_MSGCODE_RP_ERROR,
			       CTSERVICE_MSGCODE_RP_ERROR_VERSION,
			       ++(sd->nextMessageId),
			       requestid,
			       300);
  if (!msg) {
    DBG_ERROR("Could not create message");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(IPCMESSAGE_ERROR_TYPE),
		     IPCMESSAGE_ERROR_NO_MESSAGE);
  }

  /* create new error code */
  if (Error_GetType(errcode)==Error_FindType(CTSERVICE_ERROR_TYPE))
    newerrcode=-Error_GetCode(errcode);
  else
    newerrcode=-CTSERVICE_ERROR_REMOTE;

  /* add error id */
  err=IPCMessage_AddIntParameter(msg, newerrcode);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* add error message */
  if (Error_ToString(errcode,
		     errorbuffer,
		     sizeof(errorbuffer))==0)
    strcpy(errorbuffer,"Unspecified error");
  err=IPCMessage_AddParameter(msg, errorbuffer, strlen(errorbuffer)+1);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* finalize message */
  err=IPCMessage_BuildMessage(msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* enqueue the message for sending */
  err=CTServer_SendResponse(sd,ml,msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  DBG_INFO("Message enqueued for sending");
  DBG_LEAVE;
  return 0;
}



ERRORCODE ReaderServer__HandlePing(CTSERVERDATA *sd,
				   IPCMESSAGELAYER *ml,
				   IPCMESSAGE *req) {
  ERRORCODE err;
  IPCMESSAGE *msg;
  int requestid;
  int i;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  /* check version */
  err=IPCMessage_IntParameter(req, 1, &i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  if ((i&0xff00)!=(READERSERVICE_MSGCODE_RQ_PING_VERSION&0xff00)) {
    DBG_ERROR("Bad message version");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_BAD_MESSAGE_VERSION);
  }

  /* get request id */
  err=IPCMessage_NextIntParameter(req, &requestid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* create response */
  msg=CTService_Message_Create(READERSERVICE_MSGCODE_RP_PING,
			       READERSERVICE_MSGCODE_RP_PING_VERSION,
			       ++(sd->nextMessageId),
			       requestid,
                               256);
  if (!msg) {
    DBG_ERROR("Could not create message");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(IPCMESSAGE_ERROR_TYPE),
		     IPCMESSAGE_ERROR_NO_MESSAGE);
  }

  /* finalize message */
  err=IPCMessage_BuildMessage(msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* enqueue the message for sending */
  err=CTServer_SendResponse(sd,ml,msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  DBG_INFO("Message enqueued for sending");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderServer__HandleAlloc(CTSERVERDATA *sd,
				IPCMESSAGELAYER *ml,
				IPCMESSAGE *req) {
  ERRORCODE err;
  IPCMESSAGE *msg;
  int requestid;
  int i;
  int tid;
  CTREADERDESCRIPTION *descr;
  const char *p;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  err=CTService_CheckMsgCodeAndVersion(req,
				       READERSERVICE_MSGCODE_RQ_ALLOC,
				       READERSERVICE_MSGCODE_RQ_ALLOC_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get request id */
  err=IPCMessage_NextIntParameter(req, &requestid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  /* skip reply id */
  err=IPCMessage_NextIntParameter(req, &i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get terminal id */
  err=IPCMessage_NextIntParameter(req, &tid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  /* now perform the command */
  tid=CTCore_AllocTerminal(rsd->core,
			   IPCMessageLayer_GetId(ml),
			   tid,
			   &descr);

  /* create response */
  msg=CTService_Message_Create(READERSERVICE_MSGCODE_RP_ALLOC,
			       READERSERVICE_MSGCODE_RP_ALLOC_VERSION,
			       ++(sd->nextMessageId),
			       requestid,
			       256);
  if (!msg) {
    DBG_ERROR("Could not create message");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(IPCMESSAGE_ERROR_TYPE),
		     IPCMESSAGE_ERROR_NO_MESSAGE);
  }

  /* add terminal id */
  err=IPCMessage_AddIntParameter(msg, tid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* add reader name */
  DBG_DEBUG("Adding reader name \"%s\"",descr->name);
  err=IPCMessage_AddParameter(msg, descr->name, strlen(descr->name)+1);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* add reader flags */
  DBG_DEBUG("Adding reader flags \"%04x\"",descr->readerFlags);
  err=IPCMessage_AddIntParameter(msg, descr->readerFlags);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* add reader type */
  p=descr->typeName;
  DBG_DEBUG("Adding reader type \"%s\"",p);
  err=IPCMessage_AddParameter(msg, p, strlen(p)+1);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* finalize message */
  err=IPCMessage_BuildMessage(msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* enqueue the message for sending */
  err=CTServer_SendResponse(sd,ml,msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  DBG_INFO("Message enqueued for sending");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderServer__HandleRelease(CTSERVERDATA *sd,
				  IPCMESSAGELAYER *ml,
				  IPCMESSAGE *req) {
  ERRORCODE err;
  IPCMESSAGE *msg;
  int requestid;
  int i;
  int tid;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  err=CTService_CheckMsgCodeAndVersion(req,
				       READERSERVICE_MSGCODE_RQ_RELEASE,
				       READERSERVICE_MSGCODE_RQ_RELEASE_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get request id */
  err=IPCMessage_NextIntParameter(req, &requestid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  /* skip reply id */
  err=IPCMessage_NextIntParameter(req, &i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get reader id */
  err=IPCMessage_NextIntParameter(req, &tid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  /* now perform the command */
  CTCore_ReleaseTerminal(rsd->core,
			 IPCMessageLayer_GetId(ml),
			 tid);
  /* create response */
  msg=CTService_Message_Create(READERSERVICE_MSGCODE_RP_RELEASE,
			       READERSERVICE_MSGCODE_RP_RELEASE_VERSION,
			       ++(sd->nextMessageId),
			       requestid,
			       256);
  if (!msg) {
    DBG_ERROR("Could not create message");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(IPCMESSAGE_ERROR_TYPE),
		     IPCMESSAGE_ERROR_NO_MESSAGE);
  }

  /* finalize message */
  err=IPCMessage_BuildMessage(msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* enqueue the message for sending */
  err=CTServer_SendResponse(sd,ml,msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  DBG_INFO("Message enqueued for sending");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderServer__HandleConnect(CTSERVERDATA *sd,
				      IPCMESSAGELAYER *ml,
				      IPCMESSAGE *req) {
  ERRORCODE err;
  int requestid;
  int i;
  int tid;
  READERSERVERDATA *rsd;
  int msgVersion;
  int cardId;
  int waitForIt;
  READERSERVEROPENDATA *wr;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  /* get msgversion (needed later) */
  err=IPCMessage_IntParameter(req, 1, &msgVersion);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  err=CTService_CheckMsgCodeAndVersion(req,
				       READERSERVICE_MSGCODE_RQ_CONNECT,
				       READERSERVICE_MSGCODE_RQ_CONNECT_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get request id */
  err=IPCMessage_NextIntParameter(req, &requestid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  /* skip reply id */
  err=IPCMessage_NextIntParameter(req, &i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get reader id */
  err=IPCMessage_NextIntParameter(req, &tid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get cardId and wait flag (if newer message) */
  if ((msgVersion&0xff)>0) {
    err=IPCMessage_NextIntParameter(req, &cardId);
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      DBG_LEAVE;
      return err;
    }
    err=IPCMessage_NextIntParameter(req, &waitForIt);
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      DBG_LEAVE;
      return err;
    }
  }
  else {
    /* use defaults otherwise */
    cardId=0;
    waitForIt=0;
  }

  DBG_ERROR("Connect request for card %d", cardId);

  wr=ReaderServer__OpenRequest_new();
  wr->msgLayerId=IPCMessageLayer_GetId(ml);
  wr->requestId=requestid;
  wr->readerNumber=tid;
  wr->readerId=CTCore_GetClientReaderId(rsd->core, wr->msgLayerId, tid);
  wr->cardId=cardId;
  ReaderServer__OpenRequest_add(wr, &(rsd->openRequests));
  DBG_INFO("Request enqueued");
  DBG_DEBUG("Will now check whether the reader is ready (waitForIt=%d)",
	    waitForIt);
  ReaderServer_CheckOpenResponses(sd, waitForIt==0);

  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderServer__HandleDisconnect(CTSERVERDATA *sd,
					 IPCMESSAGELAYER *ml,
					 IPCMESSAGE *req) {
  ERRORCODE err;
  IPCMESSAGE *msg;
  int requestid;
  int i;
  int tid;
  int result;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  err=CTService_CheckMsgCodeAndVersion(req,
				       READERSERVICE_MSGCODE_RQ_DISCONNECT,
				       READERSERVICE_MSGCODE_RQ_DISCONNECT_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get request id */
  err=IPCMessage_NextIntParameter(req, &requestid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  /* skip reply id */
  err=IPCMessage_NextIntParameter(req, &i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get reader id */
  err=IPCMessage_NextIntParameter(req, &tid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  /* now perform the command */
  err=CTCore_DisconnectTerminal(rsd->core,
				IPCMessageLayer_GetId(ml),
				tid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    result=CTSERVICE_ERROR_DRIVER;
  }
  else {
    result=CTSERVICE_SUCCESS;
  }

  /* create response */
  msg=CTService_Message_Create(READERSERVICE_MSGCODE_RP_DISCONNECT,
			       READERSERVICE_MSGCODE_RP_DISCONNECT_VERSION,
			       ++(sd->nextMessageId),
			       requestid,
			       256);
  if (!msg) {
    DBG_ERROR("Could not create message");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(IPCMESSAGE_ERROR_TYPE),
		     IPCMESSAGE_ERROR_NO_MESSAGE);
  }

  /* add result */
  err=IPCMessage_AddIntParameter(msg, result);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* finalize message */
  err=IPCMessage_BuildMessage(msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* enqueue the message for sending */
  err=CTServer_SendResponse(sd,ml,msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  DBG_INFO("Message enqueued for sending");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderServer__HandleCommand(CTSERVERDATA *sd,
				      IPCMESSAGELAYER *ml,
				      IPCMESSAGE *req) {
  ERRORCODE err;
  IPCMESSAGE *msg;
  int requestid;
  int i;
  int tid;
  int result;
  char *sendBuffer;
  int sendBufferLength;
  char recvBuffer[300];
  int recvBufferLength;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  err=CTService_CheckMsgCodeAndVersion(req,
				       READERSERVICE_MSGCODE_RQ_COMMAND,
				       READERSERVICE_MSGCODE_RQ_COMMAND_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get request id */
  err=IPCMessage_NextIntParameter(req, &requestid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  /* skip reply id */
  err=IPCMessage_NextIntParameter(req, &i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get reader id */
  err=IPCMessage_NextIntParameter(req, &tid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get sendBuffer */
  err=IPCMessage_NextParameter(req, &sendBuffer, &sendBufferLength);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* perform the command */
  recvBufferLength=sizeof(recvBuffer);
  err=CTCore_CommandTerminal(rsd->core,
			     IPCMessageLayer_GetId(ml),
			     tid,
			     sendBuffer,
			     sendBufferLength,
			     recvBuffer,
        		     &recvBufferLength);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    if (Error_GetType(err)==Error_FindType(CTCORE_ERROR_TYPE) &&
	Error_GetCode(err)==CTCORE_ERROR_CARD_REMOVED)
      result=CTSERVICE_ERROR_CARD_REMOVED;
    else
      result=CTSERVICE_ERROR_DRIVER;
  }
  else {
    result=CTSERVICE_SUCCESS;
  }

  /* create response */
  msg=CTService_Message_Create(READERSERVICE_MSGCODE_RP_COMMAND,
			       READERSERVICE_MSGCODE_RP_COMMAND_VERSION,
			       ++(sd->nextMessageId),
			       requestid,
			       1024);
  if (!msg) {
    DBG_ERROR("Could not create message");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(IPCMESSAGE_ERROR_TYPE),
		     IPCMESSAGE_ERROR_NO_MESSAGE);
  }

  /* add result */
  err=IPCMessage_AddIntParameter(msg, result);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  if (result==CTSERVICE_SUCCESS) {
    /* add recvBuffer */
    err=IPCMessage_AddParameter(msg, recvBuffer, recvBufferLength);
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      IPCMessage_free(msg);
      DBG_LEAVE;
      return err;
    }
  }

  /* finalize message */
  err=IPCMessage_BuildMessage(msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* enqueue the message for sending */
  err=CTServer_SendResponse(sd,ml,msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  DBG_INFO("Message enqueued for sending");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderServer__HandleFind(CTSERVERDATA *sd,
				   IPCMESSAGELAYER *ml,
				   IPCMESSAGE *req) {
  ERRORCODE err;
  IPCMESSAGE *msg;
  int requestid;
  int i;
  unsigned int readerFlags;
  unsigned int readerFlagsMask;
  int found;
  int idsFound[256];
  CTREADERDESCRIPTION *rd;
  READERSERVERDATA *rsd;
  char *typeName;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  err=CTService_CheckMsgCodeAndVersion(req,
				       READERSERVICE_MSGCODE_RQ_FIND,
				       READERSERVICE_MSGCODE_RQ_FIND_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get request id */
  err=IPCMessage_NextIntParameter(req, &requestid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  /* skip reply id */
  err=IPCMessage_NextIntParameter(req, &i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get reader type name  */
  err=IPCMessage_NextStringParameter(req, &typeName);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get readerFlags */
  err=IPCMessage_NextIntParameter(req, &readerFlags);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get readerFlagMask */
  err=IPCMessage_NextIntParameter(req, &readerFlagsMask);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* scan for matching readers */
  found=0;
  if (rsd->core->readerDescriptions) {
    rd=rsd->core->readerDescriptions;
    while(rd) {
      DBG_INFO("Checking: Flags %04x=%04x/%04x, "
	       "readerType %s=%s",
	       rd->readerFlags, readerFlags, readerFlagsMask,
	       rd->typeName, typeName);
      if ((typeName[0]==0 || (strcasecmp(typeName,rd->typeName)==0)) &&
	  (!((rd->readerFlags^readerFlags)&readerFlagsMask))){
	DBG_INFO("Found a matching reader (%d)",rd->id);
	idsFound[found++]=rd->id;
      }
      rd=rd->next;
    } /* while */
  } /* if readers */

  /* create response */
  msg=CTService_Message_Create(READERSERVICE_MSGCODE_RP_FIND,
			       READERSERVICE_MSGCODE_RP_FIND_VERSION,
			       ++(sd->nextMessageId),
			       requestid,
			       256+found*5);
  if (!msg) {
    DBG_ERROR("Could not create message");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(IPCMESSAGE_ERROR_TYPE),
		     IPCMESSAGE_ERROR_NO_MESSAGE);
  }

  /* add number of matching readers */
  err=IPCMessage_AddIntParameter(msg, found);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* add ids of matching readers */
  for (i=0; i<found; i++) {
    DBG_INFO("Adding reader %d", idsFound[i]);
    err=IPCMessage_AddIntParameter(msg, idsFound[i]);
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      IPCMessage_free(msg);
      DBG_LEAVE;
      return err;
    }
  } /* for */

  /* finalize message */
  err=IPCMessage_BuildMessage(msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* enqueue the message for sending */
  err=CTServer_SendResponse(sd,ml,msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  DBG_INFO("Message enqueued for sending");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderServer__HandleGetStatus(CTSERVERDATA *sd,
					IPCMESSAGELAYER *ml,
					IPCMESSAGE *req) {
  ERRORCODE err;
  IPCMESSAGE *msg;
  int requestid;
  int i;
  int tid;
  int result;
  unsigned int status;
  char atrbuffer[256];
  int atrlen;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  err=CTService_CheckMsgCodeAndVersion(req,
				       READERSERVICE_MSGCODE_RQ_STATUS,
				       READERSERVICE_MSGCODE_RQ_STATUS_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get request id */
  err=IPCMessage_NextIntParameter(req, &requestid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  /* skip reply id */
  err=IPCMessage_NextIntParameter(req, &i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get reader id */
  err=IPCMessage_NextIntParameter(req, &tid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  /* now perform the command */
  atrlen=sizeof(atrbuffer);
  err=CTCore_GetReaderStatus(rsd->core,
			     IPCMessageLayer_GetId(ml),
			     tid,
			     &status,
			     atrbuffer,
			     &atrlen);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    result=CTSERVICE_ERROR_DRIVER;
  }
  else {
    result=CTSERVICE_SUCCESS;
  }

  /* create response */
  msg=CTService_Message_Create(READERSERVICE_MSGCODE_RP_STATUS,
			       READERSERVICE_MSGCODE_RP_STATUS_VERSION,
			       ++(sd->nextMessageId),
			       requestid,
			       2048);
  if (!msg) {
    DBG_ERROR("Could not create message");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(IPCMESSAGE_ERROR_TYPE),
		     IPCMESSAGE_ERROR_NO_MESSAGE);
  }

  /* add result */
  err=IPCMessage_AddIntParameter(msg, result);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  if (result==CTSERVICE_SUCCESS) {
    /* add status */
    err=IPCMessage_AddIntParameter(msg, status);
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      IPCMessage_free(msg);
      DBG_LEAVE;
      return err;
    }
    /* add ATR */
    err=IPCMessage_AddParameter(msg, atrbuffer, atrlen);
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      IPCMessage_free(msg);
      DBG_LEAVE;
      return err;
    }
  }

  /* finalize message */
  err=IPCMessage_BuildMessage(msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* enqueue the message for sending */
  err=CTServer_SendResponse(sd,ml,msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  DBG_INFO("Message enqueued for sending");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderServer__HandleWait(CTSERVERDATA *sd,
				   IPCMESSAGELAYER *ml,
				   IPCMESSAGE *req) {
  ERRORCODE err;
  int requestid;
  int i;
  unsigned int readerFlags;
  unsigned int readerFlagsMask;
  unsigned int readerStatus;
  unsigned int readerStatusMask;
  unsigned int readerStatusDelta;
  int msgVersion;
  READERSERVERWAITDATA *wr;
  int mustChange;
  READERSERVERDATA *rsd;
  char *typeName;
  int typeNameSize;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  /* get msgversion (needed later) */
  err=IPCMessage_IntParameter(req, 1, &msgVersion);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  err=CTService_CheckMsgCodeAndVersion(req,
				       READERSERVICE_MSGCODE_RQ_WAIT,
				       READERSERVICE_MSGCODE_RQ_WAIT_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get request id */
  err=IPCMessage_NextIntParameter(req, &requestid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  /* skip reply id */
  err=IPCMessage_NextIntParameter(req, &i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get readerType name */
  err=IPCMessage_NextParameter(req, &typeName, &typeNameSize);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  if (typeNameSize)
    typeName[typeNameSize-1]=0;

  /* get readerFlags */
  err=IPCMessage_NextIntParameter(req, &readerFlags);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get readerFlagMask */
  err=IPCMessage_NextIntParameter(req, &readerFlagsMask);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get readerStatus */
  err=IPCMessage_NextIntParameter(req, &readerStatus);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get readerStatusMask */
  err=IPCMessage_NextIntParameter(req, &readerStatusMask);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get mustChange */
  err=IPCMessage_NextIntParameter(req, &mustChange);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get status delta mask (if newer message) */
  if ((msgVersion&0xff)>0) {
    err=IPCMessage_NextIntParameter(req, &readerStatusDelta);
    if (!Error_IsOk(err)) {
      DBG_ERROR_ERR(err);
      DBG_LEAVE;
      return err;
    }
  }
  else
    /* use default otherwise */
    readerStatusDelta=~0;

  wr=ReaderServer__WaitRequest_new();
  wr->msgLayerId=IPCMessageLayer_GetId(ml);
  wr->requestId=requestid;
  wr->typeName=strdup(typeName);
  wr->readerFlags=readerFlags;
  wr->readerFlagsMask=readerFlagsMask;
  wr->readerStatus=readerStatus;
  wr->readerStatusMask=readerStatusMask;
  wr->readerDeltaStatus=readerStatusDelta;
  wr->mustChange=mustChange;
  ReaderServer__WaitRequest_add(wr, &(rsd->waitRequests));
  DBG_INFO("Request enqueued");
  if (mustChange==0) {
    DBG_INFO("Will now check whether there are readers ready");
    ReaderServer_CheckWaitResponses(sd,0);
  }
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderServer__AddReaderDescription(IPCMESSAGE *msg,
					     CTREADERDESCRIPTION *rd) {
  ERRORCODE err;

  assert(msg);
  assert(rd);

  /* add terminal id */
  err=IPCMessage_AddIntParameter(msg, rd->id);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* add terminal flags */
  err=IPCMessage_AddIntParameter(msg, rd->readerFlags);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* add terminal type */
  err=IPCMessage_AddStringParameter(msg, rd->typeName);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* add terminal name */
  err=IPCMessage_AddParameter(msg,
			      rd->name,
			      strlen(rd->name)+1);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }
  return 0;
}


ERRORCODE ReaderServer__SendWaitResponse(CTSERVERDATA *sd,
					 READERSERVERWAITDATA *wr,
					 CTREADERTABLE *rt) {
  IPCMESSAGE *msg;
  IPCMESSAGELAYER *ml;
  unsigned int status;
  ERRORCODE err;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);
  assert(wr);
  assert(rt);

  DBG_DEBUG("Looking for message layer %d", wr->msgLayerId);
  ml=IPCServiceLayer_FindMessageLayer(sd->service,
				      wr->msgLayerId);
  if (ml) {
    DBG_DEBUG("Message layer %d found", wr->msgLayerId);
    if (IPCMessageLayer_GetStatus(ml)!=StateDisconnected) {
      /* create response */
      msg=CTService_Message_Create(READERSERVICE_MSGCODE_RP_WAIT,
				   READERSERVICE_MSGCODE_RP_WAIT_VERSION,
				   ++(sd->nextMessageId),
				   wr->requestId,
				   512);
      if (!msg) {
	DBG_ERROR("Could not create message");
	DBG_LEAVE;
	return Error_New(0,
			 ERROR_SEVERITY_ERR,
			 Error_FindType(IPCMESSAGE_ERROR_TYPE),
			 IPCMESSAGE_ERROR_NO_MESSAGE);
      }

      /* add terminal status */
      status=rt->lastStatus;
      if (rt->lockedByClient!=0 &&
	  rt->lockedByClient!=IPCMessageLayer_GetStatus(ml))
	status|=CTREADERSTATUS_LOCKED_BY_OTHER;
      err=IPCMessage_AddIntParameter(msg, status);
      if (!Error_IsOk(err)) {
	DBG_ERROR_ERR(err);
	IPCMessage_free(msg);
	DBG_LEAVE;
	return err;
      }

      assert(rt->descr);
      err=ReaderServer__AddReaderDescription(msg, rt->descr);
      if (!Error_IsOk(err)) {
	DBG_ERROR_ERR(err);
	IPCMessage_free(msg);
	DBG_LEAVE;
	return err;
      }

      /* add id of currently inserted card (only valid if there IS a card) */
      err=IPCMessage_AddIntParameter(msg, rt->cardId);
      if (!Error_IsOk(err)) {
	DBG_ERROR_ERR(err);
	IPCMessage_free(msg);
	DBG_LEAVE;
	return err;
      }

      /* finalize message */
      err=IPCMessage_BuildMessage(msg);
      if (!Error_IsOk(err)) {
	DBG_ERROR_ERR(err);
	IPCMessage_free(msg);
	DBG_LEAVE;
	return err;
      }

      if (1) {
	/* get peer address and port */
	IPCTRANSPORTLAYERTABLE *tlt;
	char addrbuffer[256];

	tlt=IPCMessageLayer_GetTransportLayer(ml);
	assert(tlt);
	assert(tlt->getPeerAddress);
	assert(tlt->getPeerPort);
	err=tlt->getPeerAddress(tlt, addrbuffer, sizeof(addrbuffer));
	if (!Error_IsOk(err)) {
	  DBG_ERROR_ERR(err);
	  return err;
	}
	if (strlen(addrbuffer)<1) {
	  /* no address, so use the id */
#ifdef HAVE_SNPRINTF
	  snprintf(addrbuffer,sizeof(addrbuffer),
		   "Id %d", IPCMessageLayer_GetId(ml));
#else
	  sprintf(addrbuffer,"Id %d", IPCMessageLayer_GetId(ml));
#endif
	}
        assert(rt->descr);
	DBG_NOTICE("Sending change notification for reader \"%s\""
		   "(%s) (Card %d)",
		   rt->descr->name, addrbuffer, rt->cardId);
      }

      /* enqueue the message for sending */
      err=CTServer_SendResponse(sd,ml,msg);
      if (!Error_IsOk(err)) {
	DBG_ERROR_ERR(err);
	IPCMessage_free(msg);
	DBG_LEAVE;
	return err;
      }

      DBG_INFO("Message enqueued for sending");
      DBG_LEAVE;
      return 0;
    }
  }

  /* message layer either not found or has a bad status */
  DBG_ERROR("Messagelayer %d not found",wr->msgLayerId);
  DBG_INFO("Removing wait request");
  ReaderServer__WaitRequest_del(wr,&(rsd->waitRequests));
  ReaderServer__WaitRequest_free(wr);
  return Error_New(0,
		   ERROR_SEVERITY_ERR,
		   Error_FindType(CTSERVICE_ERROR_TYPE),
		   CTSERVICE_ERROR_NO_MESSAGELAYER);
}


ERRORCODE ReaderServer__SendOpenResponse(CTSERVERDATA *sd,
					 READERSERVEROPENDATA *wr,
					 CTREADERTABLE *rt,
					 int result,
					 const char *atrbuffer,
					 unsigned int atrbufferlen) {
  IPCMESSAGE *msg;
  IPCMESSAGELAYER *ml;
  ERRORCODE err;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);
  assert(wr);
  assert(rt);

  DBG_DEBUG("Looking for message layer %d", wr->msgLayerId);
  ml=IPCServiceLayer_FindMessageLayer(sd->service,
				      wr->msgLayerId);
  if (ml) {
    DBG_DEBUG("Message layer %d found", wr->msgLayerId);
    if (IPCMessageLayer_GetStatus(ml)!=StateDisconnected) {
      /* create response */
      msg=CTService_Message_Create(READERSERVICE_MSGCODE_RP_CONNECT,
				   READERSERVICE_MSGCODE_RP_CONNECT_VERSION,
				   ++(sd->nextMessageId),
				   wr->requestId,
				   1024);
      if (!msg) {
	DBG_ERROR("Could not create message");
	DBG_LEAVE;
	ReaderServer__OpenRequest_del(wr,&(rsd->openRequests));
	ReaderServer__OpenRequest_free(wr);
	return Error_New(0,
			 ERROR_SEVERITY_ERR,
			 Error_FindType(IPCMESSAGE_ERROR_TYPE),
			 IPCMESSAGE_ERROR_NO_MESSAGE);
      }

      /* add result */
      err=IPCMessage_AddIntParameter(msg, result);
      if (!Error_IsOk(err)) {
	DBG_ERROR_ERR(err);
	IPCMessage_free(msg);
	ReaderServer__OpenRequest_del(wr,&(rsd->openRequests));
	ReaderServer__OpenRequest_free(wr);
	DBG_LEAVE;
	return err;
      }

      if (result==CTSERVICE_SUCCESS) {
	/* add ATR */
	err=IPCMessage_AddParameter(msg, atrbuffer, atrbufferlen);
	if (!Error_IsOk(err)) {
	  DBG_ERROR_ERR(err);
	  IPCMessage_free(msg);
	  ReaderServer__OpenRequest_del(wr,&(rsd->openRequests));
	  ReaderServer__OpenRequest_free(wr);
	  DBG_LEAVE;
	  return err;
	}
      }

      /* finalize message */
      err=IPCMessage_BuildMessage(msg);
      if (!Error_IsOk(err)) {
	DBG_ERROR_ERR(err);
	IPCMessage_free(msg);
	ReaderServer__OpenRequest_del(wr,&(rsd->openRequests));
	ReaderServer__OpenRequest_free(wr);
	DBG_LEAVE;
	return err;
      }

      if (1) {
	/* get peer address and port */
	IPCTRANSPORTLAYERTABLE *tlt;
	char addrbuffer[256];

	tlt=IPCMessageLayer_GetTransportLayer(ml);
	assert(tlt);
	assert(tlt->getPeerAddress);
	assert(tlt->getPeerPort);
	err=tlt->getPeerAddress(tlt, addrbuffer, sizeof(addrbuffer));
	if (!Error_IsOk(err)) {
	  DBG_ERROR_ERR(err);
	  ReaderServer__OpenRequest_del(wr,&(rsd->openRequests));
	  ReaderServer__OpenRequest_free(wr);
	  return err;
	}
	if (strlen(addrbuffer)<1) {
	  /* no address, so use the id */
#ifdef HAVE_SNPRINTF
	  snprintf(addrbuffer,sizeof(addrbuffer),
		   "Id %d", IPCMessageLayer_GetId(ml));
#else
	  sprintf(addrbuffer,"Id %d", IPCMessageLayer_GetId(ml));
#endif
	}
        assert(rt->descr);
	DBG_NOTICE("Sending open response for reader \"%s\", (Result=%d) (%s)",
		   rt->descr->name, result, addrbuffer);
      }

      /* enqueue the message for sending */
      err=CTServer_SendResponse(sd,ml,msg);
      if (!Error_IsOk(err)) {
	DBG_ERROR_ERR(err);
	IPCMessage_free(msg);
	ReaderServer__OpenRequest_del(wr,&(rsd->openRequests));
	ReaderServer__OpenRequest_free(wr);
	DBG_LEAVE;
	return err;
      }

      DBG_INFO("Message enqueued for sending");
      DBG_INFO("Removing open request");
      ReaderServer__OpenRequest_del(wr,&(rsd->openRequests));
      ReaderServer__OpenRequest_free(wr);
      DBG_LEAVE;
      return 0;
    }
  }

  /* message layer either not found or has a bad status */
  DBG_ERROR("Messagelayer %d not found",wr->msgLayerId);
  DBG_INFO("Removing open request");
  ReaderServer__OpenRequest_del(wr,&(rsd->openRequests));
  ReaderServer__OpenRequest_free(wr);
  return Error_New(0,
		   ERROR_SEVERITY_ERR,
		   Error_FindType(CTSERVICE_ERROR_TYPE),
		   CTSERVICE_ERROR_NO_MESSAGELAYER);
}





ERRORCODE ReaderServer__HandleStopWait(CTSERVERDATA *sd,
				       IPCMESSAGELAYER *ml,
				       IPCMESSAGE *req) {
  ERRORCODE err;
  IPCMESSAGE *msg;
  int requestid;
  int i;
  int result;
  int prevreq;
  READERSERVERWAITDATA *wr;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  err=CTService_CheckMsgCodeAndVersion(req,
				       READERSERVICE_MSGCODE_RQ_STOPWAIT,
				       READERSERVICE_MSGCODE_RQ_STOPWAIT_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get request id */
  err=IPCMessage_NextIntParameter(req, &requestid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  /* skip reply id */
  err=IPCMessage_NextIntParameter(req, &i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get previous request id */
  err=IPCMessage_NextIntParameter(req, &prevreq);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* now perform the command */
  wr=ReaderServer__WaitRequest_find(rsd->waitRequests,
				    IPCMessageLayer_GetId(ml),prevreq);
  if (wr) {
    result=CTSERVICE_SUCCESS;
    DBG_DEBUG("Dequeueing wait request");
    ReaderServer__WaitRequest_del(wr, &(rsd->waitRequests));
    ReaderServer__WaitRequest_free(wr);
  }
  else
    result=CTSERVICE_ERROR_NO_REQUEST;

  /* create response */
  msg=CTService_Message_Create(READERSERVICE_MSGCODE_RP_STOPWAIT,
			       READERSERVICE_MSGCODE_RP_STOPWAIT_VERSION,
			       ++(sd->nextMessageId),
			       requestid,
			       256);
  if (!msg) {
    DBG_ERROR("Could not create message");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(IPCMESSAGE_ERROR_TYPE),
		     IPCMESSAGE_ERROR_NO_MESSAGE);
  }

  /* add result */
  err=IPCMessage_AddIntParameter(msg, result);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* finalize message */
  err=IPCMessage_BuildMessage(msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* enqueue the message for sending */
  err=CTServer_SendResponse(sd,ml,msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  DBG_INFO("Message enqueued for sending");
  DBG_LEAVE;
  return 0;
}


ERRORCODE ReaderServer__HandleStopOpen(CTSERVERDATA *sd,
				       IPCMESSAGELAYER *ml,
				       IPCMESSAGE *req) {
  ERRORCODE err;
  IPCMESSAGE *msg;
  int requestid;
  int i;
  int result;
  int prevreq;
  READERSERVEROPENDATA *wr;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  err=CTService_CheckMsgCodeAndVersion(req,
				       READERSERVICE_MSGCODE_RQ_STOPOPEN,
				       READERSERVICE_MSGCODE_RQ_STOPOPEN_VERSION);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get request id */
  err=IPCMessage_NextIntParameter(req, &requestid);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }
  /* skip reply id */
  err=IPCMessage_NextIntParameter(req, &i);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* get previous request id */
  err=IPCMessage_NextIntParameter(req, &prevreq);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    DBG_LEAVE;
    return err;
  }

  /* now perform the command */
  wr=ReaderServer__OpenRequest_find(rsd->openRequests,
				    IPCMessageLayer_GetId(ml),prevreq);
  if (wr) {
    result=CTSERVICE_SUCCESS;
    DBG_DEBUG("Dequeueing open request");
    ReaderServer__OpenRequest_del(wr, &(rsd->openRequests));
    ReaderServer__OpenRequest_free(wr);
  }
  else
    result=CTSERVICE_ERROR_NO_REQUEST;

  /* create response */
  msg=CTService_Message_Create(READERSERVICE_MSGCODE_RP_STOPOPEN,
			       READERSERVICE_MSGCODE_RP_STOPOPEN_VERSION,
			       ++(sd->nextMessageId),
			       requestid,
			       256);
  if (!msg) {
    DBG_ERROR("Could not create message");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(IPCMESSAGE_ERROR_TYPE),
		     IPCMESSAGE_ERROR_NO_MESSAGE);
  }

  /* add result */
  err=IPCMessage_AddIntParameter(msg, result);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* finalize message */
  err=IPCMessage_BuildMessage(msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  /* enqueue the message for sending */
  err=CTServer_SendResponse(sd,ml,msg);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    IPCMessage_free(msg);
    DBG_LEAVE;
    return err;
  }

  DBG_INFO("Message enqueued for sending");
  DBG_LEAVE;
  return 0;
}





ERRORCODE ReaderServer_RequestHandler(CTSERVERDATA *sd,
				      IPCMESSAGELAYER *ml,
				      IPCMESSAGE *msg) {
  ERRORCODE err;
  int msgCode;
  IPCTRANSPORTLAYERTABLE *tlt;
  char addrbuffer[256];
  READERSERVERDATA *rsd;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  /* get peer address and port */
  tlt=IPCMessageLayer_GetTransportLayer(ml);
  assert(tlt);
  assert(tlt->getPeerAddress);
  assert(tlt->getPeerPort);
  err=tlt->getPeerAddress(tlt, addrbuffer, sizeof(addrbuffer));
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    return err;
  }
  if (strlen(addrbuffer)<1) {
    /* no address, so use the id */
#ifdef HAVE_SNPRINTF
    snprintf(addrbuffer,sizeof(addrbuffer),
	     "Id %d", IPCMessageLayer_GetId(ml));
#else
    sprintf(addrbuffer,"Id %d", IPCMessageLayer_GetId(ml));
#endif
  }

  err=IPCMessage_FirstIntParameter(msg, &msgCode);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    return err;
  }
  switch (msgCode) {
  case READERSERVICE_MSGCODE_RQ_PING:
    DBG_NOTICE("REQUEST: Ping (%s)",addrbuffer);
    err=ReaderServer__HandlePing(sd, ml, msg);
    break;
  case READERSERVICE_MSGCODE_RQ_ALLOC:
    DBG_NOTICE("REQUEST: Alloc (%s)",addrbuffer);
    err=ReaderServer__HandleAlloc(sd, ml, msg);
    break;
  case READERSERVICE_MSGCODE_RQ_RELEASE:
    DBG_NOTICE("REQUEST: Release (%s)",addrbuffer);
    err=ReaderServer__HandleRelease(sd, ml, msg);
    break;
  case READERSERVICE_MSGCODE_RQ_CONNECT:
    DBG_NOTICE("REQUEST: Connect (%s)",addrbuffer);
    err=ReaderServer__HandleConnect(sd, ml, msg);
    break;
  case READERSERVICE_MSGCODE_RQ_DISCONNECT:
    DBG_NOTICE("REQUEST: Disconnect (%s)",addrbuffer);
    err=ReaderServer__HandleDisconnect(sd, ml, msg);
    break;
  case READERSERVICE_MSGCODE_RQ_COMMAND:
    DBG_NOTICE("REQUEST: Command (%s)",addrbuffer);
    err=ReaderServer__HandleCommand(sd, ml, msg);
    break;
  case READERSERVICE_MSGCODE_RQ_FIND:
    DBG_NOTICE("REQUEST: Find (%s)",addrbuffer);
    err=ReaderServer__HandleFind(sd, ml, msg);
    break;
  case READERSERVICE_MSGCODE_RQ_STATUS:
    DBG_NOTICE("REQUEST: Status (%s)",addrbuffer);
    err=ReaderServer__HandleGetStatus(sd, ml, msg);
    break;
  case READERSERVICE_MSGCODE_RQ_WAIT:
    DBG_NOTICE("REQUEST: Wait (%s)",addrbuffer);
    err=ReaderServer__HandleWait(sd, ml, msg);
    break;
  case READERSERVICE_MSGCODE_RQ_STOPWAIT:
    DBG_NOTICE("REQUEST: StopWait (%s)",addrbuffer);
    err=ReaderServer__HandleStopWait(sd, ml, msg);
    break;
  case READERSERVICE_MSGCODE_RQ_STOPOPEN:
    DBG_NOTICE("REQUEST: StopOpen (%s)",addrbuffer);
    err=ReaderServer__HandleStopOpen(sd, ml, msg);
    break;
  default:
    DBG_ERROR("Message code %d not supported (%s)",
	      msgCode,
	      addrbuffer);
    err=Error_New(0,
		  ERROR_SEVERITY_ERR,
		  Error_FindType(CTSERVICE_ERROR_TYPE),
		  CTSERVICE_ERROR_BAD_MESSAGE_CODE);
    break;
  } /* switch */

  if (!Error_IsOk(err)) {
    ERRORCODE localerr;

    localerr=ReaderServer__SendErrorMessage(sd,
					    ml,
					    msg,
					    err);
    if (!Error_IsOk(localerr)) {
      DBG_ERROR_ERR(localerr);
    }
  }

  DBG_LEAVE;
  return err;
}


ERRORCODE ReaderServer_CheckWaitResponses(CTSERVERDATA *sd, int mustChange) {
  CTREADERTABLE *curr;
  READERSERVERWAITDATA *wr;
  READERSERVERWAITDATA *next;
  ERRORCODE err;
  unsigned int cstat;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);
  assert(rsd->core);

  curr=rsd->core->readers;
  while(curr) {
    DBG_DEBUG("Checking reader %08x", (unsigned int)curr);
    if (curr->deltaStatus || mustChange==0) {
      DBG_DEBUG(" status changed for %08x", (unsigned int)curr);
      wr=rsd->waitRequests;
      while(wr) {
	next=wr->next;
	cstat=curr->lastStatus;
	if (wr->msgLayerId!=0)
	  if (curr->lockedByClient!=0 &&
	      curr->lockedByClient!=wr->msgLayerId)
	    cstat|=CTREADERSTATUS_LOCKED_BY_OTHER;
	DBG_INFO("Checking: Flags %04x=%04x/%04x, "
		 "readerType %s=%s, Status %04x=%04x/%04x",
		 curr->descr->readerFlags, wr->readerFlags,
		 wr->readerFlagsMask,
		 curr->descr->typeName, wr->typeName,
		 cstat, wr->readerStatus,
		 wr->readerStatusMask);
	if (
	    (wr->typeName[0]==0 ||
	     (strcasecmp(wr->typeName,curr->descr->typeName)==0))
	    &&
	    (!((curr->descr->readerFlags^wr->readerFlags) &
	       wr->readerFlagsMask))
	    &&
	    (!((cstat^wr->readerStatus)&wr->readerStatusMask))
	    &&
	    ((curr->deltaStatus&wr->readerDeltaStatus) || wr->mustChange==0)
	   ) {
	  DBG_INFO("Sending status change notification (id=%d)",
		   wr->msgLayerId);
	  err=ReaderServer__SendWaitResponse(sd, wr, curr);
	  if (!Error_IsOk(err)) {
	    DBG_ERROR_ERR(err);
	  }
	}
	wr=next;
      } /* while request */
    }
    curr=curr->next;
  } /* while */

  /* set all must-change values in all waitRequests */
  wr=rsd->waitRequests;
  while(wr) {
    wr->mustChange=1;
    wr=wr->next;
  }

  return 0;
}


ERRORCODE ReaderServer_CheckOpenResponses(CTSERVERDATA *sd,
					  int ignoreLocks) {
  CTREADERTABLE *curr;
  READERSERVEROPENDATA *wr;
  READERSERVEROPENDATA *next;
  ERRORCODE err;
  READERSERVERDATA *rsd;
  int result;
  char atrbuffer[300];
  int atrbufferlen;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);
  assert(rsd->core);

  curr=rsd->core->readers;
  while(curr) {
    DBG_DEBUG("Checking reader %08x", (unsigned int)curr);
    if (curr->lockedByClient==0 || ignoreLocks) {
      /* reader is not locked by anyone, so check whether there is
       * someone waiting for this */
      wr=rsd->openRequests;
      while(wr) {
	next=wr->next;
	if (wr->readerId==curr->descr->id) {
	  /* have a request for this reader */
	  DBG_DEBUG(" Have a request for reader %s",
		    curr->descr->name);
	  atrbufferlen=0;
	  if (curr->lockedByClient==0 ||
	      curr->lockedByClient==wr->msgLayerId) {
	    DBG_DEBUG(" reader not locked");
	    /* client not locked by someone else */
	    if ((wr->cardId==curr->cardId || wr->cardId==0)) {
	      /* card is that one the client wanted, so connect */
	      atrbufferlen=sizeof(atrbuffer);
	      err=CTCore_ConnectTerminal(rsd->core,
					 wr->msgLayerId,
					 wr->readerNumber,
					 atrbuffer,
					 &atrbufferlen);
	      if (!Error_IsOk(err)) {
		DBG_ERROR_ERR(err);
		result=CTSERVICE_ERROR_DRIVER;
	      }
	      else {
		result=CTSERVICE_SUCCESS;
	      }
	    }
	    else
	      result=CTSERVICE_ERROR_CARD_REMOVED;
	  } /* if not locked */
	  else {
	    result=CTSERVICE_ERROR_CARD_LOCKED;
	  }

	  DBG_DEBUG("Sending open response (id=%d, result=%d)",
		    wr->msgLayerId, result);
	  err=ReaderServer__SendOpenResponse(sd,
					     wr,
					     curr,
					     result,
					     atrbuffer,
					     atrbufferlen);
	  if (!Error_IsOk(err)) {
	    DBG_ERROR_ERR(err);
	  }
	}

	wr=next;
      } /* while request */
    }
    curr=curr->next;
  } /* while reader */
  return 0;
}



ERRORCODE ReaderServer_ClientDown(CTSERVERDATA *sd,
				  IPCMESSAGELAYER *ml){
  int id;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  id=IPCMessageLayer_GetId(ml);
  if (id!=-1) {

    /* remove all standing wait requests for this client */
    if (rsd->waitRequests) {
      READERSERVERWAITDATA *wr;
      READERSERVERWAITDATA *next;

      DBG_INFO("Will remove all wait request for client %4x", id);
      wr=rsd->waitRequests;
      while(wr) {
	next=wr->next;
	if (wr->msgLayerId==id) {
	  ReaderServer__WaitRequest_del(wr,&(rsd->waitRequests));
	  ReaderServer__WaitRequest_free(wr);
	}
	wr=next;
      } /* while */
    }

    /* remove all standing open requests for this client */
    if (rsd->openRequests) {
      READERSERVEROPENDATA *wr;
      READERSERVEROPENDATA *next;
      DBG_INFO("Will remove all open request for client %4x", id);
      wr=rsd->openRequests;
      while(wr) {
	next=wr->next;
	if (wr->msgLayerId==id) {
	  ReaderServer__OpenRequest_del(wr,&(rsd->openRequests));
	  ReaderServer__OpenRequest_free(wr);
	}
	wr=next;
      } /* while */
    }

    /* unregister client from core */
    DBG_INFO("Unregistering client %4x from core (%4x)",
	     id,
	     rsd->coreClientId);
    CTCore_UnregisterClient(rsd->core, rsd->coreClientId);

  } /* if id ok */
  return 0;
}


ERRORCODE ReaderServer_ClientUp(CTSERVERDATA *sd,
				IPCMESSAGELAYER *ml){
  int i;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  i=CTCore_RegisterClient(rsd->core);
  if (i==-1) {
    DBG_ERROR("Could not register client");
    DBG_LEAVE;
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_NO_CLIENT);
  }
  IPCMessageLayer_SetId(ml,i);
  rsd->coreClientId=i;
  return 0;
}


ERRORCODE ReaderServer_Init(CTSERVERDATA *sd, CONFIGGROUP *root) {
  READERSERVERDATA *rsd;
  ERRORCODE err;
  CONFIGGROUP *drivers;

  DBG_INFO("Initializing reader server");
  assert(sd);

  /* read driver descriptions */
  drivers=Config_new();
  if (CTCore_ReadDriverDescriptions(CHIPCARD_DRIVERS, drivers)) {
    Config_free(drivers);
    DBG_ERROR("Error loading driver descriptions");
    return Error_New(0,
		     ERROR_SEVERITY_ERR,
		     Error_FindType(CTSERVICE_ERROR_TYPE),
		     CTSERVICE_ERROR_BAD_CONFIG);
  }

  /* initialize server */
  err=CTServer_Init(sd, root);
  if (!Error_IsOk(err)) {
    Config_free(drivers);
    DBG_ERROR_ERR(err);
    return err;
  }

  /* create new private data */
  rsd=ReaderServer_new();
  assert(rsd);

  /* ok, server is running, now setup CTCore */
  rsd->core=CTCore_new();
  err=CTCore_Init(rsd->core, drivers);
  if (!Error_IsOk(err)) {
    DBG_ERROR_ERR(err);
    ReaderServer_free(rsd);
    Config_free(drivers);
    return err;
  }

  /* set private data and callbacks */
  CTServer_SetPrivateData(sd, rsd);
  CTServer_SetRequestHandler(sd, ReaderServer_RequestHandler);
  CTServer_SetClientUpHandler(sd, ReaderServer_ClientUp);
  CTServer_SetClientDownHandler(sd, ReaderServer_ClientDown);


  /* register as a normal client to the core */
  rsd->coreClientId=CTCore_RegisterClient(rsd->core);
  assert(rsd->coreClientId!=-1);

  DBG_INFO("Reader server initialized");
  return 0;
}


ERRORCODE ReaderServer_Fini(CTSERVERDATA *sd){
  ERRORCODE err1, err2;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  DBG_INFO("Deinitializing reader server");
  assert(sd);
  rsd=CTServer_GetPrivateData(sd);
  assert(rsd);

  /* unregister from CTCore */
  CTCore_UnregisterClient(rsd->core, rsd->coreClientId);

  err1=CTServer_Fini(sd);
  if (!Error_IsOk(err1)) {
    DBG_ERROR_ERR(err1);
  }
  err2=CTCore_Fini(rsd->core);
  if (!Error_IsOk(err2)) {
    DBG_ERROR_ERR(err2);
  }

  CTServer_SetPrivateData(sd,0);
  ReaderServer_free(rsd);

  DBG_INFO("Reader server deinitialized.");
  DBG_LEAVE;
  if (!Error_IsOk(err1))
    return err1;
  if (!Error_IsOk(err2))
    return err2;

  return 0;
}


ERRORCODE ReaderServer_Work(CTSERVERDATA *sd,
			    int timeout,
			    int maxmsg){
  ERRORCODE err;
  READERSERVERDATA *rsd;

  DBG_ENTER;
  assert(sd);
  rsd=(READERSERVERDATA*)CTServer_GetPrivateData(sd);
  assert(rsd);

  err=CTServer_Work(sd, timeout, maxmsg);
  if (!Error_IsOk(err)) {
    DBG_DEBUG_ERR(err);
  }

  DBG_DEBUG("Checking the status of my terminals");
  CTCore_WalkTerminals(rsd->core);
  /* check for changes in the terminals and report to clients
   * which registered for those events.
   */
  ReaderServer_CheckWaitResponses(sd, 1);

  /* Check for readers which are unused and for which a connect request
   * exists. Thise readers will then connected */
  ReaderServer_CheckOpenResponses(sd, 1);

  DBG_LEAVE;
  return err;
}


